/*==============================================================*/
/* Data Model:     AEMO Electricity Data Model v5.6.0.1 Oracle  */
/* Created on:     2025/10/09 17:11:37                          */
/*==============================================================*/


/*==============================================================*/
/* Table: ADG_DETAIL                                            */
/*==============================================================*/
create table ADG_DETAIL (
   ADG_ID               VARCHAR2(20)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSION_DATETIME     DATE                  not null,
   ADG_TYPE             VARCHAR2(20),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table ADG_DETAIL is
'Table for tracking evolving Aggregate Dispatch Group attributes';

comment on column ADG_DETAIL.ADG_ID is
'Identifies the Aggregate Dispatch Group';

comment on column ADG_DETAIL.EFFECTIVEDATE is
'Effective calendar date of record';

comment on column ADG_DETAIL.VERSION_DATETIME is
'Date and time of the version of Dispatchable Unit details';

comment on column ADG_DETAIL.ADG_TYPE is
'Conformance Type for the Aggregate Dispatch Group. One of the following: CAP, MIXED, TARGET';

comment on column ADG_DETAIL.AUTHORISEDDATE is
'Date record authorised';

comment on column ADG_DETAIL.AUTHORISEDBY is
'User authorising record';

comment on column ADG_DETAIL.LASTCHANGED is
'Last date and time record changed';

alter table ADG_DETAIL
   add constraint ADG_DETAIL_PK primary key (ADG_ID, EFFECTIVEDATE, VERSION_DATETIME)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: AGGREGATE_DISPATCH_GROUP                              */
/*==============================================================*/
create table AGGREGATE_DISPATCH_GROUP (
   ADG_ID               VARCHAR2(20)          not null,
   COMMENTS             VARCHAR2(100),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table AGGREGATE_DISPATCH_GROUP is
'Entity allowing for compliance monitoring over grouped DUIDs';

comment on column AGGREGATE_DISPATCH_GROUP.ADG_ID is
'Aggregate Dispatch Group ID';

comment on column AGGREGATE_DISPATCH_GROUP.COMMENTS is
'A participant provided comment';

comment on column AGGREGATE_DISPATCH_GROUP.LASTCHANGED is
'Last date and time record changed';

alter table AGGREGATE_DISPATCH_GROUP
   add constraint AGGREGATE_DISPATCH_GROUP_PK primary key (ADG_ID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: ANCILLARY_RECOVERY_SPLIT                              */
/*==============================================================*/
create table ANCILLARY_RECOVERY_SPLIT (
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   SERVICE              VARCHAR2(10)          not null,
   PAYMENTTYPE          VARCHAR2(20)          not null,
   CUSTOMER_PORTION     NUMBER(8,5),
   LASTCHANGED          DATE,
   ACE_PORTION          NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table ANCILLARY_RECOVERY_SPLIT is
'ANCILLARY_RECOVERY_SPLIT holds the actual customer portion for each service and payment type. A single EFFECTIVEDATE/VERSIONNO combination applies to all services (i.e. the latest EFFECTIVEDATE/VERSIONNO is not retrieved for a single service, but applies to a data set).';

comment on column ANCILLARY_RECOVERY_SPLIT.EFFECTIVEDATE is
'Calendar settlement date record becomes effective.';

comment on column ANCILLARY_RECOVERY_SPLIT.VERSIONNO is
'Version number of the record for the given date.';

comment on column ANCILLARY_RECOVERY_SPLIT.SERVICE is
'Ancillary service name (e.g. AGC, FCASCOMP)';

comment on column ANCILLARY_RECOVERY_SPLIT.PAYMENTTYPE is
'A payment type associated with the service (can be ENABLING, AVAILABILITY, USAGE, or COMPENSATION).';

comment on column ANCILLARY_RECOVERY_SPLIT.CUSTOMER_PORTION is
'The percentage value of the recovery funded by market customers.';

comment on column ANCILLARY_RECOVERY_SPLIT.LASTCHANGED is
'Last date and time record changed';

comment on column ANCILLARY_RECOVERY_SPLIT.ACE_PORTION is
'The percentage value of the recovery funded using the ACE MWh Values. This field is only used for Settlement post IESS rule effective date.';

alter table ANCILLARY_RECOVERY_SPLIT
   add constraint ANCILLARY_RECOVERY_SPLIT_PK primary key (EFFECTIVEDATE, VERSIONNO, SERVICE, PAYMENTTYPE)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: ANCILLARY_RECOVERY_SPLIT_LCX                          */
/*==============================================================*/
create index ANCILLARY_RECOVERY_SPLIT_LCX on ANCILLARY_RECOVERY_SPLIT (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: APCCOMP                                               */
/*==============================================================*/
create table APCCOMP (
   APCID                VARCHAR2(10)          not null,
   REGIONID             VARCHAR2(10),
   STARTDATE            DATE,
   STARTPERIOD          NUMBER(3,0),
   ENDDATE              DATE,
   ENDPERIOD            NUMBER(3,0),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table APCCOMP is
'APCCOMP is to set out Administered Price Cap (APC) compensation periods for a participant.';

comment on column APCCOMP.APCID is
'APC event identifier.';

comment on column APCCOMP.REGIONID is
'Region';

comment on column APCCOMP.STARTDATE is
'Settlement start date';

comment on column APCCOMP.STARTPERIOD is
'Settlement start period (1-48)';

comment on column APCCOMP.ENDDATE is
'Settlement end date';

comment on column APCCOMP.ENDPERIOD is
'Settlement end period (1-48)';

comment on column APCCOMP.LASTCHANGED is
'Last date and time record changed';

alter table APCCOMP
   add constraint APCCOMP_PK primary key (APCID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: APCCOMP_LCX                                           */
/*==============================================================*/
create index APCCOMP_LCX on APCCOMP (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: APCCOMPAMOUNT                                         */
/*==============================================================*/
create table APCCOMPAMOUNT (
   APCID                VARCHAR2(10)          not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   PERIODID             NUMBER(6,0)           not null,
   AMOUNT               NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table APCCOMPAMOUNT is
'APCCOMPAMOUNT shows the Administered Price Cap (APC) compensation amount.';

comment on column APCCOMPAMOUNT.APCID is
'APC Identifier';

comment on column APCCOMPAMOUNT.PARTICIPANTID is
'Participant identifier';

comment on column APCCOMPAMOUNT.VERSIONNO is
'Version number';

comment on column APCCOMPAMOUNT.PERIODID is
'Offset from start date and period in APCCOMP table.';

comment on column APCCOMPAMOUNT.AMOUNT is
'Compensation audit.';

comment on column APCCOMPAMOUNT.LASTCHANGED is
'Last date and time record changed';

alter table APCCOMPAMOUNT
   add constraint APCCOMPAMOUNT_PK primary key (APCID, PARTICIPANTID, VERSIONNO, PERIODID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: APCCOMPAMOUNT_LCX                                     */
/*==============================================================*/
create index APCCOMPAMOUNT_LCX on APCCOMPAMOUNT (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: APCCOMPAMOUNTTRK                                      */
/*==============================================================*/
create table APCCOMPAMOUNTTRK (
   APCID                VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   AUTHORISEDBY         VARCHAR2(10),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table APCCOMPAMOUNTTRK is
'APCCOMPAMOUNTTRK sets out the relevant Administered Price Cap (APC) period for compensation purposes. Use the APCCOMPAMOUNTTRK table in conjunction with APCAMOUNT.';

comment on column APCCOMPAMOUNTTRK.APCID is
'APC Identifier';

comment on column APCCOMPAMOUNTTRK.VERSIONNO is
'Version number';

comment on column APCCOMPAMOUNTTRK.AUTHORISEDBY is
'Authorised by';

comment on column APCCOMPAMOUNTTRK.AUTHORISEDDATE is
'Authorised date';

comment on column APCCOMPAMOUNTTRK.LASTCHANGED is
'Last date and time record changed';

alter table APCCOMPAMOUNTTRK
   add constraint APCCOMPAMOUNTTRK_PK primary key (APCID, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: APCCOMPAMOUNTTRK_LCX                                  */
/*==============================================================*/
create index APCCOMPAMOUNTTRK_LCX on APCCOMPAMOUNTTRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: APEVENT                                               */
/*==============================================================*/
create table APEVENT (
   APEVENTID            NUMBER(22,0)          not null,
   EFFECTIVEFROMINTERVAL DATE,
   EFFECTIVETOINTERVAL  DATE,
   REASON               VARCHAR2(2000),
   STARTAUTHORISEDBY    VARCHAR2(15),
   STARTAUTHORISEDDATE  DATE,
   ENDAUTHORISEDBY      VARCHAR2(15),
   ENDAUTHORISEDDATE    DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table APEVENT is
'APEVENT is the driving data defining the existence and timeframes of an administered pricing event.';

comment on column APEVENT.APEVENTID is
'Unique identifier for this administered pricing event';

comment on column APEVENT.EFFECTIVEFROMINTERVAL is
'Date Time of the first Dispatch Interval to which the administered event applies';

comment on column APEVENT.EFFECTIVETOINTERVAL is
'Date Time of the final Dispatch Interval to which the administered event applies';

comment on column APEVENT.REASON is
'Description of the driver for the Event';

comment on column APEVENT.STARTAUTHORISEDBY is
'Authorising staff for start of AP event';

comment on column APEVENT.STARTAUTHORISEDDATE is
'Date-Time start authorised';

comment on column APEVENT.ENDAUTHORISEDBY is
'Authorising staff for end of AP event';

comment on column APEVENT.ENDAUTHORISEDDATE is
'Date Time end authorised';

comment on column APEVENT.LASTCHANGED is
'Date-Time the record was last modified';

alter table APEVENT
   add constraint APEVENT_PK primary key (APEVENTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: APEVENT_LCX                                           */
/*==============================================================*/
create index APEVENT_LCX on APEVENT (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: APEVENTREGION                                         */
/*==============================================================*/
create table APEVENTREGION (
   APEVENTID            NUMBER(22,0)          not null,
   REGIONID             VARCHAR2(10)          not null,
   LASTCHANGED          DATE,
   ENERGYAPFLAG         NUMBER(1,0),
   RAISE6SECAPFLAG      NUMBER(1,0),
   RAISE60SECAPFLAG     NUMBER(1,0),
   RAISE5MINAPFLAG      NUMBER(1,0),
   RAISEREGAPFLAG       NUMBER(1,0),
   LOWER6SECAPFLAG      NUMBER(1,0),
   LOWER60SECAPFLAG     NUMBER(1,0),
   LOWER5MINAPFLAG      NUMBER(1,0),
   LOWERREGAPFLAG       NUMBER(1,0),
   RAISE1SECAPFLAG      NUMBER(3,0),
   LOWER1SECAPFLAG      NUMBER(3,0)
)
   tablespace PSSTAB01;

comment on table APEVENTREGION is
'APEVENTREGION is the Region detail for an administered pricing event defined through APEVENT.';

comment on column APEVENTREGION.APEVENTID is
'Unique identifier for this administered pricing event';

comment on column APEVENTREGION.REGIONID is
'Date-Time of the first Dispatch Interval to which the administered event applies';

comment on column APEVENTREGION.LASTCHANGED is
'Date Time of the final Dispatch Interval to which the administered event applies';

comment on column APEVENTREGION.ENERGYAPFLAG is
'flag indicating if the apevent covers an energy AP';

comment on column APEVENTREGION.RAISE6SECAPFLAG is
'flag indicating if the apevent covers a raise6sec AP';

comment on column APEVENTREGION.RAISE60SECAPFLAG is
'flag indicating if the apevent covers a raise60sec AP';

comment on column APEVENTREGION.RAISE5MINAPFLAG is
'flag indicating if the apevent covers a raise5min AP';

comment on column APEVENTREGION.RAISEREGAPFLAG is
'flag indicating if the apevent covers a raisereg AP';

comment on column APEVENTREGION.LOWER6SECAPFLAG is
'flag indicating if the apevent covers a lower6sec AP';

comment on column APEVENTREGION.LOWER60SECAPFLAG is
'flag indicating if the apevent covers a lower60sec AP
flag indicating if the apevent covers a lower5min AP
flag indicating if the apevent covers a lowerreg AP
flag indicating if the apevent covers a lower60sec AP';

comment on column APEVENTREGION.LOWER5MINAPFLAG is
'flag indicating if the apevent covers a lower5min AP';

comment on column APEVENTREGION.LOWERREGAPFLAG is
'flag indicating if the apevent covers a lowerreg AP';

comment on column APEVENTREGION.RAISE1SECAPFLAG is
'Flag indicating if the APEvent covers a Raise1Sec AP';

comment on column APEVENTREGION.LOWER1SECAPFLAG is
'Flag indicating if the APEvent covers a Lower1Sec AP';

alter table APEVENTREGION
   add constraint APEVENTREGION_PK primary key (APEVENTID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: APEVENTREGION_LCX                                     */
/*==============================================================*/
create index APEVENTREGION_LCX on APEVENTREGION (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: AREA                                                  */
/*==============================================================*/
create table AREA (
   AREAID               VARCHAR2(10)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   AREA_NAME            VARCHAR2(20),
   AREA_DESCRIPTION     VARCHAR2(200),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table AREA is
'Table containing static metadata for the Areas, which are sub-regions used in load forecasting and rooftop PV forecasting. The latest metadata can be obtained for each AreaID using the most recent EffectiveDate and then VersionNo. If an AreaID is not mapped to an active RegionID in the corresponding REGION_AREA table, then that AreaID can be considered inactive.';

comment on column AREA.AREAID is
'Area identifier';

comment on column AREA.EFFECTIVEDATE is
'Calendar date from when this record set is effective.';

comment on column AREA.VERSIONNO is
'Version number for the same effectivedate.';

comment on column AREA.AREA_NAME is
'Area name';

comment on column AREA.AREA_DESCRIPTION is
'Area description';

comment on column AREA.LASTCHANGED is
'Last date and time record changed.';

alter table AREA
   add constraint AREA_PK primary key (AREAID, EFFECTIVEDATE, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: AUCTION                                               */
/*==============================================================*/
create table AUCTION (
   AUCTIONID            VARCHAR2(30)          not null,
   AUCTIONDATE          DATE,
   NOTIFYDATE           DATE,
   STARTDATE            DATE,
   ENDDATE              DATE,
   DESCRIPTION          VARCHAR2(100),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(30),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table AUCTION is
'AUCTION holds auction details. AUCTION is new in March 2003 to support SRA Inter-Temporal Linking.';

comment on column AUCTION.AUCTIONID is
'Unique id for each auction date';

comment on column AUCTION.AUCTIONDATE is
'Auction date';

comment on column AUCTION.STARTDATE is
'Open date for bidding';

comment on column AUCTION.ENDDATE is
'Close date for bidding';

comment on column AUCTION.DESCRIPTION is
'Description of an auction';

alter table AUCTION
   add constraint AUCTION_PK primary key (AUCTIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: AUCTION_LCX                                           */
/*==============================================================*/
create index AUCTION_LCX on AUCTION (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: AUCTION_CALENDAR                                      */
/*==============================================================*/
create table AUCTION_CALENDAR (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   QUARTER              NUMBER(1,0)           not null,
   STARTDATE            DATE,
   ENDDATE              DATE,
   NOTIFYDATE           DATE,
   PAYMENTDATE          DATE,
   RECONCILIATIONDATE   DATE,
   LASTCHANGED          DATE,
   PRELIMPURCHASESTMTDATE DATE,
   PRELIMPROCEEDSSTMTDATE DATE,
   FINALPURCHASESTMTDATE DATE,
   FINALPROCEEDSSTMTDATE DATE
)
   tablespace PSSTAB01;

comment on table AUCTION_CALENDAR is
'AUCTION_CALENDAR holds the definitions of each auction quarter in a contract year. AUCTION_CALENDAR supports the Settlement Residue Auction.';

comment on column AUCTION_CALENDAR.CONTRACTYEAR is
'SRA Contracted Year';

comment on column AUCTION_CALENDAR.QUARTER is
'SRA Contracted Quarter';

comment on column AUCTION_CALENDAR.STARTDATE is
'First day of SRA Contract Quarter expressed as Date';

comment on column AUCTION_CALENDAR.ENDDATE is
'Last day of SRA Contract Quarter expressed as Date';

comment on column AUCTION_CALENDAR.NOTIFYDATE is
'Default notification date';

comment on column AUCTION_CALENDAR.PAYMENTDATE is
'Date for payment by Participant';

comment on column AUCTION_CALENDAR.RECONCILIATIONDATE is
'Date of reconciliation for the quarter';

comment on column AUCTION_CALENDAR.LASTCHANGED is
'Last date and time record changed';

comment on column AUCTION_CALENDAR.PRELIMPURCHASESTMTDATE is
'The date the Prelim Purchase Statement is generated';

comment on column AUCTION_CALENDAR.PRELIMPROCEEDSSTMTDATE is
'The date the Prelim Proceeds Statement is generated';

comment on column AUCTION_CALENDAR.FINALPURCHASESTMTDATE is
'The date the Final Purchase Statement is generated';

comment on column AUCTION_CALENDAR.FINALPROCEEDSSTMTDATE is
'The date the Final Proceeds Statement is generated';

alter table AUCTION_CALENDAR
   add constraint AUCTION_CALENDAR_PK primary key (CONTRACTYEAR, QUARTER)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: AUCTION_CALENDAR_LCX                                  */
/*==============================================================*/
create index AUCTION_CALENDAR_LCX on AUCTION_CALENDAR (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: AUCTION_IC_ALLOCATIONS                                */
/*==============================================================*/
create table AUCTION_IC_ALLOCATIONS (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   QUARTER              NUMBER(1,0)           not null,
   VERSIONNO            NUMBER(3,0)           not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   FROMREGIONID         VARCHAR2(10)          not null,
   MAXIMUMUNITS         NUMBER(5,0),
   PROPORTION           NUMBER(8,5),
   AUCTIONFEE           NUMBER(17,5),
   CHANGEDATE           DATE,
   CHANGEDBY            VARCHAR2(15),
   LASTCHANGED          DATE,
   AUCTIONFEE_SALES     Number(18,8)
)
   tablespace PSSTAB01;

comment on table AUCTION_IC_ALLOCATIONS is
'AUCTION_IC_ALLOCATIONS supports the Settlement Residue Auction by providing the basis for setting up contracts for individual tranches. AUCTION_IC_ALLOCATIONS shows the default definitions for the total number of units and proportion applicable to each directional interconnector for a specified auction quarter.';

comment on column AUCTION_IC_ALLOCATIONS.CONTRACTYEAR is
'SRA Contracted Year';

comment on column AUCTION_IC_ALLOCATIONS.QUARTER is
'SRA Contracted Quarter';

comment on column AUCTION_IC_ALLOCATIONS.VERSIONNO is
'Version of data for other key data - a higher version for same key data takes precedence';

comment on column AUCTION_IC_ALLOCATIONS.INTERCONNECTORID is
'Contracted Interconnector Identifier';

comment on column AUCTION_IC_ALLOCATIONS.FROMREGIONID is
'Nominated source region for Interconnector';

comment on column AUCTION_IC_ALLOCATIONS.MAXIMUMUNITS is
'Number of units on the interconnector';

comment on column AUCTION_IC_ALLOCATIONS.PROPORTION is
'Percentage of the total residue for each Unit';

comment on column AUCTION_IC_ALLOCATIONS.AUCTIONFEE is
'Daily auction fee';

comment on column AUCTION_IC_ALLOCATIONS.CHANGEDATE is
'Authorisation date';

comment on column AUCTION_IC_ALLOCATIONS.CHANGEDBY is
'Name of person authorising this data set';

comment on column AUCTION_IC_ALLOCATIONS.LASTCHANGED is
'Last date and time record changed';

comment on column AUCTION_IC_ALLOCATIONS.AUCTIONFEE_SALES is
'Fees for Cancelled Units.';

alter table AUCTION_IC_ALLOCATIONS
   add constraint AUCTION_IC_ALLOCATIONS_PK primary key (CONTRACTYEAR, QUARTER, VERSIONNO, INTERCONNECTORID, FROMREGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: AUCTION_IC_ALLOCATIONS_LCX                            */
/*==============================================================*/
create index AUCTION_IC_ALLOCATIONS_LCX on AUCTION_IC_ALLOCATIONS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: AUCTION_REVENUE_ESTIMATE                              */
/*==============================================================*/
create table AUCTION_REVENUE_ESTIMATE (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   QUARTER              NUMBER(1,0)           not null,
   VALUATIONID          VARCHAR2(15)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   FROMREGIONID         VARCHAR2(10)          not null,
   MONTHNO              NUMBER(1,0)           not null,
   STARTDATE            DATE,
   ENDDATE              DATE,
   REVENUE              NUMBER(17,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table AUCTION_REVENUE_ESTIMATE is
'AUCTION_REVENUE_ESTIMATE supports the Settlement Residue Auction, by holding the evaluators estimates of revenue for each month of a given quarter.
Since reserve prices are no longer applicable from the end of 2001, zero is used as a default to avoid rewriting the system.';

comment on column AUCTION_REVENUE_ESTIMATE.CONTRACTYEAR is
'SRA Contracted Year';

comment on column AUCTION_REVENUE_ESTIMATE.QUARTER is
'SRA Contracted Quarter';

comment on column AUCTION_REVENUE_ESTIMATE.VALUATIONID is
'Identifier of the estimator';

comment on column AUCTION_REVENUE_ESTIMATE.VERSIONNO is
'Version of data for other key data - a higher version for same key data will take precedence';

comment on column AUCTION_REVENUE_ESTIMATE.INTERCONNECTORID is
'Contracted Interconnector';

comment on column AUCTION_REVENUE_ESTIMATE.FROMREGIONID is
'Nominated source region for Interconnector';

comment on column AUCTION_REVENUE_ESTIMATE.MONTHNO is
'Month number within quarter (1..3)';

comment on column AUCTION_REVENUE_ESTIMATE.STARTDATE is
'First day of month as date';

comment on column AUCTION_REVENUE_ESTIMATE.ENDDATE is
'Last day of month as date';

comment on column AUCTION_REVENUE_ESTIMATE.REVENUE is
'Estimated Revenue';

comment on column AUCTION_REVENUE_ESTIMATE.LASTCHANGED is
'Last date and time record changed';

alter table AUCTION_REVENUE_ESTIMATE
   add constraint AUCTION_REVENUE_ESTIMATE_PK primary key (CONTRACTYEAR, QUARTER, VALUATIONID, VERSIONNO, INTERCONNECTORID, FROMREGIONID, MONTHNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: AUCTION_REVENUE_ESTIMATE_LCX                          */
/*==============================================================*/
create index AUCTION_REVENUE_ESTIMATE_LCX on AUCTION_REVENUE_ESTIMATE (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: AUCTION_REVENUE_TRACK                                 */
/*==============================================================*/
create table AUCTION_REVENUE_TRACK (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   QUARTER              NUMBER(1,0)           not null,
   VALUATIONID          VARCHAR2(15)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   EFFECTIVEDATE        DATE,
   STATUS               VARCHAR2(10),
   DOCUMENTREF          VARCHAR2(30),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table AUCTION_REVENUE_TRACK is
'AUCTION_REVENUE_TRACK supports the Settlement Residue Auction, by holding the tracking information for each evaluators estimates for a given quarter. The status field is dynamic and is used for selection of estimates to be published.';

comment on column AUCTION_REVENUE_TRACK.CONTRACTYEAR is
'SRA Contracted Year';

comment on column AUCTION_REVENUE_TRACK.QUARTER is
'SRA Contracted Quarter';

comment on column AUCTION_REVENUE_TRACK.VALUATIONID is
'Identifier of the estimator';

comment on column AUCTION_REVENUE_TRACK.VERSIONNO is
'Version of data for other key data - a higher version for same key data takes precedence';

comment on column AUCTION_REVENUE_TRACK.EFFECTIVEDATE is
'Date from which the record change is applicable';

comment on column AUCTION_REVENUE_TRACK.STATUS is
'Internal use';

comment on column AUCTION_REVENUE_TRACK.DOCUMENTREF is
'Reference to methodology document';

comment on column AUCTION_REVENUE_TRACK.AUTHORISEDDATE is
'Date of authorisation for this record';

comment on column AUCTION_REVENUE_TRACK.AUTHORISEDBY is
'Name of person authorising this record';

comment on column AUCTION_REVENUE_TRACK.LASTCHANGED is
'Date and time this record was last changed';

alter table AUCTION_REVENUE_TRACK
   add constraint AUCTION_REVENUE_TRACK_PK primary key (CONTRACTYEAR, QUARTER, VALUATIONID, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: AUCTIONREVTRK_NDX_LCHD                                */
/*==============================================================*/
create index AUCTIONREVTRK_NDX_LCHD on AUCTION_REVENUE_TRACK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: AUCTION_RP_ESTIMATE                                   */
/*==============================================================*/
create table AUCTION_RP_ESTIMATE (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   QUARTER              NUMBER(1,0)           not null,
   VALUATIONID          VARCHAR2(15)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   FROMREGIONID         VARCHAR2(10)          not null,
   RPESTIMATE           NUMBER(17,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table AUCTION_RP_ESTIMATE is
'AUCTION_RP_ESTIMATE supports the Settlement Residue Auction, by holding the evaluators estimates of revenue prices for a given quarter.
Since reserve prices are no longer applicable from the end of 2001, zero is used as a default to avoid rewriting the system.';

comment on column AUCTION_RP_ESTIMATE.CONTRACTYEAR is
'SRA Contracted Year';

comment on column AUCTION_RP_ESTIMATE.QUARTER is
'SRA Contracted Quarter';

comment on column AUCTION_RP_ESTIMATE.VALUATIONID is
'Identifier of the estimator';

comment on column AUCTION_RP_ESTIMATE.VERSIONNO is
'Version of data for other key data - a higher version for same key data takes precedence';

comment on column AUCTION_RP_ESTIMATE.INTERCONNECTORID is
'Contracted Interconnector';

comment on column AUCTION_RP_ESTIMATE.FROMREGIONID is
'Nominated source region for Interconnector';

comment on column AUCTION_RP_ESTIMATE.RPESTIMATE is
'Estimate of reserve price';

comment on column AUCTION_RP_ESTIMATE.LASTCHANGED is
'Last date and time record was changed';

alter table AUCTION_RP_ESTIMATE
   add constraint AUCTION_RP_ESTIMATE_PK primary key (CONTRACTYEAR, QUARTER, VALUATIONID, VERSIONNO, INTERCONNECTORID, FROMREGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: AUCTION_RP_ESTIMATE_LCX                               */
/*==============================================================*/
create index AUCTION_RP_ESTIMATE_LCX on AUCTION_RP_ESTIMATE (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: AUCTION_TRANCHE                                       */
/*==============================================================*/
create table AUCTION_TRANCHE (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   QUARTER              NUMBER(1,0)           not null,
   VERSIONNO            NUMBER(3,0)           not null,
   TRANCHE              NUMBER(2,0)           not null,
   AUCTIONDATE          DATE,
   NOTIFYDATE           DATE,
   UNITALLOCATION       NUMBER(18,8),
   CHANGEDATE           DATE,
   CHANGEDBY            VARCHAR2(15),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table AUCTION_TRANCHE is
'AUCTION_TRANCHE supports the Settlement Residue Auction, by holding the default definitions for the percentage number of units allocated and dates applicable to each tranche for a specified auction quarter. This information provides the basis for setting up contracts for individual tranches.';

comment on column AUCTION_TRANCHE.CONTRACTYEAR is
'SRA Contracted Year';

comment on column AUCTION_TRANCHE.QUARTER is
'SRA Contracted Quarter';

comment on column AUCTION_TRANCHE.VERSIONNO is
'Version of data for other key data - a higher version for same key data will take precedence';

comment on column AUCTION_TRANCHE.TRANCHE is
'Label identifying the arbitrary segmented share of the Interconnector flow';

comment on column AUCTION_TRANCHE.AUCTIONDATE is
'Default date of the auction';

comment on column AUCTION_TRANCHE.NOTIFYDATE is
'Default date participants notified of details';

comment on column AUCTION_TRANCHE.UNITALLOCATION is
'Percentage of units allocated to the tranche';

comment on column AUCTION_TRANCHE.CHANGEDATE is
'Date of changing this record';

comment on column AUCTION_TRANCHE.CHANGEDBY is
'Name of person who changed this record';

comment on column AUCTION_TRANCHE.LASTCHANGED is
'Date and time record was last changed';

alter table AUCTION_TRANCHE
   add constraint AUCTION_TRANCHE_PK primary key (CONTRACTYEAR, QUARTER, VERSIONNO, TRANCHE)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: AUCTION_TRANCHE_LCX                                   */
/*==============================================================*/
create index AUCTION_TRANCHE_LCX on AUCTION_TRANCHE (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: AVERAGEPRICE30                                        */
/*==============================================================*/
create table AVERAGEPRICE30 (
   PERIODDATE           DATE                  not null,
   REGIONID             VARCHAR2(10)          not null,
   PERIODID             NUMBER(3,0)           not null,
   RRP                  NUMBER(15,5),
   PRICE_CONFIDENCE     VARCHAR2(20),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table AVERAGEPRICE30 is
'Reflects the 30-minute average price (the pre-5MS trading price).';

comment on column AVERAGEPRICE30.PERIODDATE is
'30-minute interval period, 1 to 48 from the start of the calendar day';

comment on column AVERAGEPRICE30.REGIONID is
'Region Identifier';

comment on column AVERAGEPRICE30.PERIODID is
'The 30-minute interval period, 1 to 48';

comment on column AVERAGEPRICE30.RRP is
'Regional reference price for this period';

comment on column AVERAGEPRICE30.PRICE_CONFIDENCE is
'Result of Manifestly Incorrect Inputs Price Status and OCD_Status - either "FIRM" or "NOT FIRM". Only FIRM if the Dispatch Interval is resolved for both MII and OCD';

comment on column AVERAGEPRICE30.LASTCHANGED is
'Last date and time record changed';

alter table AVERAGEPRICE30
   add constraint AVERAGEPRICE30_PK primary key (PERIODDATE, REGIONID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: BIDDAYOFFER                                           */
/*==============================================================*/
create table BIDDAYOFFER (
   DUID                 VARCHAR2(10)          not null,
   BIDTYPE              VARCHAR2(10)          not null,
   SETTLEMENTDATE       DATE                  not null,
   OFFERDATE            TIMESTAMP(3)          not null,
   DIRECTION            VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(22,0),
   PARTICIPANTID        VARCHAR2(10),
   DAILYENERGYCONSTRAINT NUMBER(12,6),
   REBIDEXPLANATION     VARCHAR2(500),
   PRICEBAND1           NUMBER(9,2),
   PRICEBAND2           NUMBER(9,2),
   PRICEBAND3           NUMBER(9,2),
   PRICEBAND4           NUMBER(9,2),
   PRICEBAND5           NUMBER(9,2),
   PRICEBAND6           NUMBER(9,2),
   PRICEBAND7           NUMBER(9,2),
   PRICEBAND8           NUMBER(9,2),
   PRICEBAND9           NUMBER(9,2),
   PRICEBAND10          NUMBER(9,2),
   MINIMUMLOAD          NUMBER(22,0),
   T1                   NUMBER(22,0),
   T2                   NUMBER(22,0),
   T3                   NUMBER(22,0),
   T4                   NUMBER(22,0),
   NORMALSTATUS         VARCHAR2(3),
   LASTCHANGED          DATE,
   MR_FACTOR            NUMBER(16,6),
   ENTRYTYPE            VARCHAR2(20),
   REBID_EVENT_TIME     VARCHAR2(20),
   REBID_AWARE_TIME     VARCHAR2(20),
   REBID_DECISION_TIME  VARCHAR2(20),
   REBID_CATEGORY       VARCHAR2(1),
   REFERENCE_ID         VARCHAR2(100)
)
   tablespace PSSTAB03;

comment on table BIDDAYOFFER is
'BIDDAYOFFER shows the Energy and Ancillary Service bid data for each Market Day. BIDDAYOFFER is the parent table to BIDOFFERPERIOD. BIDDAYOFFER is a child table to BIDOFFERFILETRK';

comment on column BIDDAYOFFER.DUID is
'Dispatchable unit identifier';

comment on column BIDDAYOFFER.BIDTYPE is
'Bid Type Identifier';

comment on column BIDDAYOFFER.SETTLEMENTDATE is
'Market date for applying the bid';

comment on column BIDDAYOFFER.OFFERDATE is
'Time this bid was processed and loaded';

comment on column BIDDAYOFFER.DIRECTION is
'The power flow direction to which this offer applies: GEN, LOAD or BIDIRECTIONAL';

comment on column BIDDAYOFFER.VERSIONNO is
'Version No. for given offer date';

comment on column BIDDAYOFFER.PARTICIPANTID is
'Unique participant identifier';

comment on column BIDDAYOFFER.DAILYENERGYCONSTRAINT is
'Maximum energy available from Energy Constrained Plant. (Energy Bids Only)';

comment on column BIDDAYOFFER.REBIDEXPLANATION is
'Explanation for all rebids and inflexibilities';

comment on column BIDDAYOFFER.PRICEBAND1 is
'Price for Availability Band 1';

comment on column BIDDAYOFFER.PRICEBAND2 is
'Price for Availability Band 2';

comment on column BIDDAYOFFER.PRICEBAND3 is
'Price for Availability Band 3';

comment on column BIDDAYOFFER.PRICEBAND4 is
'Price for Availability Band 4';

comment on column BIDDAYOFFER.PRICEBAND5 is
'Price for Availability Band 5';

comment on column BIDDAYOFFER.PRICEBAND6 is
'Price for Availability Band 6';

comment on column BIDDAYOFFER.PRICEBAND7 is
'Price for Availability Band 6';

comment on column BIDDAYOFFER.PRICEBAND8 is
'Price for Availability Band 8';

comment on column BIDDAYOFFER.PRICEBAND9 is
'Price for Availability Band 9';

comment on column BIDDAYOFFER.PRICEBAND10 is
'Price for Availability Band 10';

comment on column BIDDAYOFFER.MINIMUMLOAD is
'Minimum MW load fast start plant';

comment on column BIDDAYOFFER.T1 is
'Time to synchronise in minutes (Energy Bids Only)';

comment on column BIDDAYOFFER.T2 is
'Time to minimum load in minutes (Energy Bids Only)';

comment on column BIDDAYOFFER.T3 is
'Time at minimum load in minutes (Energy Bids Only)';

comment on column BIDDAYOFFER.T4 is
'Time to shutdown in minutes (Energy Bids Only)';

comment on column BIDDAYOFFER.NORMALSTATUS is
'not used; was ON/OFF for loads (Energy Bids Only)';

comment on column BIDDAYOFFER.LASTCHANGED is
'Last date and time record changed';

comment on column BIDDAYOFFER.MR_FACTOR is
'Mandatory Restriction Offer Factor';

comment on column BIDDAYOFFER.ENTRYTYPE is
'Daily if processed before BidCutOff of previous day, otherwise REBID';

comment on column BIDDAYOFFER.REBID_EVENT_TIME is
'The time of the event(s) or other occurrence(s) cited/adduced as the reason for the rebid. Required for rebids, not required for fixed load or low ramp rates. Expected in the format: HH:MM:SS e.g. 20:11:00';

comment on column BIDDAYOFFER.REBID_AWARE_TIME is
'Intended to support the Rebidding and Technical Parameters Guideline. The time at which the participant became aware of the event(s) / occurrence(s) that prompted the rebid.Not validated by AEMO';

comment on column BIDDAYOFFER.REBID_DECISION_TIME is
'Intended to support the Rebidding and Technical Parameters Guideline. The time at which the participant made the decision to rebid. Not validated by AEMO';

comment on column BIDDAYOFFER.REBID_CATEGORY is
'Intended to support the Rebidding and Technical Parameters Guideline. A provided rebid category. Not validated by AEMO';

comment on column BIDDAYOFFER.REFERENCE_ID is
'A participants unique Reference Id';

alter table BIDDAYOFFER
   add constraint BIDDAYOFFER_PK primary key (SETTLEMENTDATE, BIDTYPE, DUID, OFFERDATE, DIRECTION)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Index: BIDDAYOFFER_LCHD_IDX                                  */
/*==============================================================*/
create index BIDDAYOFFER_LCHD_IDX on BIDDAYOFFER (
   LASTCHANGED ASC
)
tablespace PSSIND03;

/*==============================================================*/
/* Index: BIDDAYOFFER_PART_IDX                                  */
/*==============================================================*/
create index BIDDAYOFFER_PART_IDX on BIDDAYOFFER (
   PARTICIPANTID ASC
)
tablespace PSSIND03;

/*==============================================================*/
/* Table: BIDDAYOFFER_D                                         */
/*==============================================================*/
create table BIDDAYOFFER_D (
   SETTLEMENTDATE       DATE                  not null,
   DUID                 VARCHAR2(10)          not null,
   BIDTYPE              VARCHAR2(10)          not null,
   DIRECTION            VARCHAR2(20)          not null,
   BIDSETTLEMENTDATE    DATE,
   OFFERDATE            DATE,
   VERSIONNO            NUMBER(22,0),
   PARTICIPANTID        VARCHAR2(10),
   DAILYENERGYCONSTRAINT NUMBER(12,6),
   REBIDEXPLANATION     VARCHAR2(500),
   PRICEBAND1           NUMBER(9,2),
   PRICEBAND2           NUMBER(9,2),
   PRICEBAND3           NUMBER(9,2),
   PRICEBAND4           NUMBER(9,2),
   PRICEBAND5           NUMBER(9,2),
   PRICEBAND6           NUMBER(9,2),
   PRICEBAND7           NUMBER(9,2),
   PRICEBAND8           NUMBER(9,2),
   PRICEBAND9           NUMBER(9,2),
   PRICEBAND10          NUMBER(9,2),
   MINIMUMLOAD          NUMBER(22,0),
   T1                   NUMBER(22,0),
   T2                   NUMBER(22,0),
   T3                   NUMBER(22,0),
   T4                   NUMBER(22,0),
   NORMALSTATUS         VARCHAR2(3),
   LASTCHANGED          DATE,
   MR_FACTOR            NUMBER(16,6),
   ENTRYTYPE            VARCHAR2(20)
)
   tablespace PSSTAB03;

comment on table BIDDAYOFFER_D is
'BIDDAYOFFER_D shows the public summary of the energy and FCAS offers applicable in the Dispatch for the
intervals identified. BIDDAYOFFER_D is the parent table to BIDPEROFFER_D.';

comment on column BIDDAYOFFER_D.SETTLEMENTDATE is
'Market date for which the bid applied';

comment on column BIDDAYOFFER_D.DUID is
'Dispatchable unit identifier';

comment on column BIDDAYOFFER_D.BIDTYPE is
'Bid Type Identifier';

comment on column BIDDAYOFFER_D.DIRECTION is
'The power flow direction to which this offer applies: GEN, LOAD or BIDIRECTIONAL';

comment on column BIDDAYOFFER_D.BIDSETTLEMENTDATE is
'Market date for which the bid was submitted.';

comment on column BIDDAYOFFER_D.OFFERDATE is
'Offer date and time';

comment on column BIDDAYOFFER_D.VERSIONNO is
'Version No. for given offer date';

comment on column BIDDAYOFFER_D.PARTICIPANTID is
'Unique participant identifier';

comment on column BIDDAYOFFER_D.DAILYENERGYCONSTRAINT is
'Maximum energy available from Energy Constrained Plant. (Energy Bids Only)';

comment on column BIDDAYOFFER_D.REBIDEXPLANATION is
'Explanation for all rebids and inflexibilities';

comment on column BIDDAYOFFER_D.PRICEBAND1 is
'Price for Availability Band 1';

comment on column BIDDAYOFFER_D.PRICEBAND2 is
'Price for Availability Band 2';

comment on column BIDDAYOFFER_D.PRICEBAND3 is
'Price for Availability Band 3';

comment on column BIDDAYOFFER_D.PRICEBAND4 is
'Price for Availability Band 4';

comment on column BIDDAYOFFER_D.PRICEBAND5 is
'Price for Availability Band 5';

comment on column BIDDAYOFFER_D.PRICEBAND6 is
'Price for Availability Band 6';

comment on column BIDDAYOFFER_D.PRICEBAND7 is
'Price for Availability Band 7';

comment on column BIDDAYOFFER_D.PRICEBAND8 is
'Price for Availability Band 8';

comment on column BIDDAYOFFER_D.PRICEBAND9 is
'Price for Availability Band 9';

comment on column BIDDAYOFFER_D.PRICEBAND10 is
'Price for Availability Band 10';

comment on column BIDDAYOFFER_D.MINIMUMLOAD is
'Minimum MW load fast start plant';

comment on column BIDDAYOFFER_D.T1 is
'Time to synchronise in minutes (Energy Bids Only)';

comment on column BIDDAYOFFER_D.T2 is
'Time to minimum load in minutes (Energy Bids Only)';

comment on column BIDDAYOFFER_D.T3 is
'Time at minimum load in minutes (Energy Bids Only)';

comment on column BIDDAYOFFER_D.T4 is
'Time to shutdown in minutes (Energy Bids Only)';

comment on column BIDDAYOFFER_D.NORMALSTATUS is
'ON/OFF for loads (Energy Bids Only)';

comment on column BIDDAYOFFER_D.LASTCHANGED is
'Last date and time record changed';

comment on column BIDDAYOFFER_D.MR_FACTOR is
'Mandatory Restriction Scaling Factor';

comment on column BIDDAYOFFER_D.ENTRYTYPE is
'Daily if processed before BidCutOff of previous day, otherwise REBID';

alter table BIDDAYOFFER_D
   add constraint BIDDAYOFFER_D_PK primary key (SETTLEMENTDATE, BIDTYPE, DUID, DIRECTION)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Index: BIDDAYOFFER_D_LCHD_IDX                                */
/*==============================================================*/
create index BIDDAYOFFER_D_LCHD_IDX on BIDDAYOFFER_D (
   LASTCHANGED ASC
)
tablespace PSSIND03;

/*==============================================================*/
/* Index: BIDDAYOFFER_D_PART_IDX                                */
/*==============================================================*/
create index BIDDAYOFFER_D_PART_IDX on BIDDAYOFFER_D (
   PARTICIPANTID ASC
)
tablespace PSSIND03;

/*==============================================================*/
/* Table: BIDDUIDDETAILS                                        */
/*==============================================================*/
create table BIDDUIDDETAILS (
   DUID                 VARCHAR2(10)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   BIDTYPE              VARCHAR2(10)          not null,
   MAXCAPACITY          NUMBER(22,0),
   MINENABLEMENTLEVEL   NUMBER(22,0),
   MAXENABLEMENTLEVEL   NUMBER(22,0),
   MAXLOWERANGLE        NUMBER(3,0),
   MAXUPPERANGLE        NUMBER(3,0),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BIDDUIDDETAILS is
'BIDDUIDDETAILS and the associated tracking object BIDDUIDDETAILSTRK define the registration data for each ancillary service a dispatchable unit is registered to provide. The registration data is required to validate a dispatchable unit bid submitted for that ancillary service.';

comment on column BIDDUIDDETAILS.DUID is
'Dispatchable unit identifier';

comment on column BIDDUIDDETAILS.EFFECTIVEDATE is
'Market date starting at 04:30 inclusive';

comment on column BIDDUIDDETAILS.VERSIONNO is
'Record version number';

comment on column BIDDUIDDETAILS.BIDTYPE is
'Bid Type Identifier';

comment on column BIDDUIDDETAILS.MAXCAPACITY is
'Maximum Capacity of this DUID for this BIDTYPE';

comment on column BIDDUIDDETAILS.MINENABLEMENTLEVEL is
'Minimum Energy Output (MW) at which this ancillary service becomes available (AS Only)';

comment on column BIDDUIDDETAILS.MAXENABLEMENTLEVEL is
'Maximum Energy Output (MW) at which this ancillary service can be supplied (AS Only)';

comment on column BIDDUIDDETAILS.MAXLOWERANGLE is
'Maximum Angle at the lower end of the ancillary service profile (Degrees)';

comment on column BIDDUIDDETAILS.MAXUPPERANGLE is
'Maximum Angle at the upper end of the ancillary service profile (Degrees)';

comment on column BIDDUIDDETAILS.LASTCHANGED is
'Last date and time record changed';

alter table BIDDUIDDETAILS
   add constraint BIDDUIDDETAILS_PK primary key (DUID, EFFECTIVEDATE, VERSIONNO, BIDTYPE)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BIDDUIDDETAILS_LCHD_IDX                               */
/*==============================================================*/
create index BIDDUIDDETAILS_LCHD_IDX on BIDDUIDDETAILS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BIDDUIDDETAILSTRK                                     */
/*==============================================================*/
create table BIDDUIDDETAILSTRK (
   DUID                 VARCHAR2(10)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BIDDUIDDETAILSTRK is
'BIDDUIDDETAILSTRK shows the tracking for the associated object BIDDUIDDETAILS. Together, BIDDUIDDETAILSTRK and BIDDUIDDETAILS define the registration data for each ancillary service a dispatchable unit is registered to provide. The registration data is required to validate a dispatchable unit bid submitted for that ancillary service.';

comment on column BIDDUIDDETAILSTRK.DUID is
'Dispatchable unit identifier';

comment on column BIDDUIDDETAILSTRK.EFFECTIVEDATE is
'Market date starting at 04:30 inclusive';

comment on column BIDDUIDDETAILSTRK.VERSIONNO is
'Record version number';

comment on column BIDDUIDDETAILSTRK.AUTHORISEDDATE is
'Date of record authorisation. A NULL value indicates the record is not authorised.';

comment on column BIDDUIDDETAILSTRK.AUTHORISEDBY is
'User that authorised record. A NULL value indicates the record is not authorised.';

comment on column BIDDUIDDETAILSTRK.LASTCHANGED is
'Last date and time record changed';

alter table BIDDUIDDETAILSTRK
   add constraint BIDDUIDDETAILSTRK_PK primary key (DUID, EFFECTIVEDATE, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BIDDUIDDETAILSTRK_LCHD_IDX                            */
/*==============================================================*/
create index BIDDUIDDETAILSTRK_LCHD_IDX on BIDDUIDDETAILSTRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BIDOFFERFILETRK                                       */
/*==============================================================*/
create table BIDOFFERFILETRK (
   PARTICIPANTID        VARCHAR2(10)          not null,
   OFFERDATE            TIMESTAMP(3)          not null,
   FILENAME             VARCHAR2(80)          not null,
   STATUS               VARCHAR2(10),
   LASTCHANGED          DATE,
   AUTHORISEDBY         VARCHAR2(20),
   AUTHORISEDDATE       DATE,
   TRANSACTION_ID       VARCHAR2(100),
   REFERENCE_ID         VARCHAR2(100),
   SUBMISSION_TIMESTAMP DATE,
   COMMENTS             VARCHAR2(1000),
   SUBMISSION_METHOD    VARCHAR2(20)
)
   tablespace PSSTAB01;

comment on table BIDOFFERFILETRK is
'BIDOFFERFILETRK shows an audit trail of all files submitted containing ENERGY/FCAS/MNSP bid, including corrupt bids and rebids.';

comment on column BIDOFFERFILETRK.PARTICIPANTID is
'Unique participant identifier';

comment on column BIDOFFERFILETRK.OFFERDATE is
'Time this bid was processed and loaded';

comment on column BIDOFFERFILETRK.FILENAME is
'Submitted file name';

comment on column BIDOFFERFILETRK.STATUS is
'Load status [SUCCESSFUL/CORRUPT]';

comment on column BIDOFFERFILETRK.LASTCHANGED is
'Last date and time record changed';

comment on column BIDOFFERFILETRK.AUTHORISEDBY is
'Participant agent who created the Offer';

comment on column BIDOFFERFILETRK.AUTHORISEDDATE is
'When the Offer was processed - synonymous with LastChanged';

comment on column BIDOFFERFILETRK.TRANSACTION_ID is
'A GUID used to identify the submission transaction in AEMOs systems';

comment on column BIDOFFERFILETRK.REFERENCE_ID is
'A participant provided reference, which is required to be unique per submission (for a PARTICIPANTID)';

comment on column BIDOFFERFILETRK.SUBMISSION_TIMESTAMP is
'The participant provided date/time for the submission';

comment on column BIDOFFERFILETRK.COMMENTS is
'A participant provided comment';

comment on column BIDOFFERFILETRK.SUBMISSION_METHOD is
'Method by which this submission was made typically FTP, API, WEB';

alter table BIDOFFERFILETRK
   add constraint BIDOFFERFILETRK_FILE_UK unique (FILENAME)
      using index tablespace PSSIND01;

alter table BIDOFFERFILETRK
   add constraint BIDOFFERFILETRK_PK primary key (PARTICIPANTID, OFFERDATE)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BIDOFFERFILETRK_LCHD_IDX                              */
/*==============================================================*/
create index BIDOFFERFILETRK_LCHD_IDX on BIDOFFERFILETRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BIDOFFERPERIOD                                        */
/*==============================================================*/
create table BIDOFFERPERIOD (
   DUID                 VARCHAR2(20)          not null,
   BIDTYPE              VARCHAR2(10)          not null,
   TRADINGDATE          DATE                  not null,
   OFFERDATETIME        TIMESTAMP(3)          not null,
   DIRECTION            VARCHAR2(20)          not null,
   PERIODID             NUMBER(3,0)           not null,
   MAXAVAIL             NUMBER(8,3),
   FIXEDLOAD            NUMBER(8,3),
   RAMPUPRATE           NUMBER(6),
   RAMPDOWNRATE         NUMBER(6),
   ENABLEMENTMIN        NUMBER(8,3),
   ENABLEMENTMAX        NUMBER(8,3),
   LOWBREAKPOINT        NUMBER(8,3),
   HIGHBREAKPOINT       NUMBER(8,3),
   BANDAVAIL1           NUMBER(8,3),
   BANDAVAIL2           NUMBER(8,3),
   BANDAVAIL3           NUMBER(8,3),
   BANDAVAIL4           NUMBER(8,3),
   BANDAVAIL5           NUMBER(8,3),
   BANDAVAIL6           NUMBER(8,3),
   BANDAVAIL7           NUMBER(8,3),
   BANDAVAIL8           NUMBER(8,3),
   BANDAVAIL9           NUMBER(8,3),
   BANDAVAIL10          NUMBER(8,3),
   PASAAVAILABILITY     NUMBER(8,3),
   ENERGYLIMIT          NUMBER(15,5),
   PERIODIDTO           NUMBER(3,0),
   RECALL_PERIOD        NUMBER(8,3)
)
   tablespace PSSTAB03;

comment on table BIDOFFERPERIOD is
'BIDOFFERPERIOD shows 5-minute period-based Energy and Ancillary Service bid data.BIDOFFERPERIOD is a child table of BIDDAYOFFER';

comment on column BIDOFFERPERIOD.DUID is
'Dispatchable Unit ID';

comment on column BIDOFFERPERIOD.BIDTYPE is
'The type of bid, one-of ENERGY, RAISE6SEC, RAISE60SEC, RAISE5MIN, RAISEREG, LOWER6SEC, LOWER60SEC, LOWER5MIN, LOWERREG';

comment on column BIDOFFERPERIOD.TRADINGDATE is
'The trading date this bid is for';

comment on column BIDOFFERPERIOD.OFFERDATETIME is
'Time this bid was processed and loaded';

comment on column BIDOFFERPERIOD.DIRECTION is
'The power flow direction to which this offer applies: GEN, LOAD or BIDIRECTIONAL';

comment on column BIDOFFERPERIOD.PERIODID is
'Period ID 1 to 288';

comment on column BIDOFFERPERIOD.MAXAVAIL is
'Maximum availability for this BidType in this period';

comment on column BIDOFFERPERIOD.FIXEDLOAD is
'Fixed unit output MW (Energy bids only) A null value means no fixed load so the unit is dispatched according to bid and market';

comment on column BIDOFFERPERIOD.RAMPUPRATE is
'MW/Min for raise (Energy bids only)';

comment on column BIDOFFERPERIOD.RAMPDOWNRATE is
'MW/Min for lower (Energy bids only)';

comment on column BIDOFFERPERIOD.ENABLEMENTMIN is
'Minimum Energy Output (MW) at which this ancillary service becomes available (AS Only)';

comment on column BIDOFFERPERIOD.ENABLEMENTMAX is
'Maximum Energy Output (MW) at which this ancillary service can be supplied (AS Only)';

comment on column BIDOFFERPERIOD.LOWBREAKPOINT is
'Minimum Energy Output (MW) at which the unit can provide the full availability (MAXAVAIL) for this ancillary service (AS Only)';

comment on column BIDOFFERPERIOD.HIGHBREAKPOINT is
'Maximum Energy Output (MW) at which the unit can provide the full availability (MAXAVAIL) for this ancillary service (AS Only)';

comment on column BIDOFFERPERIOD.BANDAVAIL1 is
'Availability at price band 1';

comment on column BIDOFFERPERIOD.BANDAVAIL2 is
'Availability at price band 2';

comment on column BIDOFFERPERIOD.BANDAVAIL3 is
'Availability at price band 3';

comment on column BIDOFFERPERIOD.BANDAVAIL4 is
'Availability at price band 4';

comment on column BIDOFFERPERIOD.BANDAVAIL5 is
'Availability at price band 5';

comment on column BIDOFFERPERIOD.BANDAVAIL6 is
'Availability at price band 6';

comment on column BIDOFFERPERIOD.BANDAVAIL7 is
'Availability at price band 7';

comment on column BIDOFFERPERIOD.BANDAVAIL8 is
'Availability at price band 8';

comment on column BIDOFFERPERIOD.BANDAVAIL9 is
'Availability at price band 9';

comment on column BIDOFFERPERIOD.BANDAVAIL10 is
'Availability at price band 10';

comment on column BIDOFFERPERIOD.PASAAVAILABILITY is
'Allows for future use for Energy bids, being the physical plant capability including any capability potentially available within 24 hours';

comment on column BIDOFFERPERIOD.ENERGYLIMIT is
'The Energy limit applying at the end of this dispatch interval in MWh. For GEN this is a lower energy limit. For LOAD this is an upper energy limit';

comment on column BIDOFFERPERIOD.PERIODIDTO is
'Period ID Ending';

comment on column BIDOFFERPERIOD.RECALL_PERIOD is
'The advance notice(in hours) that a Scheduled Resource requires to achieve the PASA Availability MW for this trading interval';

alter table BIDOFFERPERIOD
   add constraint BIDOFFERPERIOD_PK primary key (TRADINGDATE, BIDTYPE, DUID, OFFERDATETIME, DIRECTION, PERIODID)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: BIDPEROFFER                                           */
/*==============================================================*/
create table BIDPEROFFER (
   DUID                 VARCHAR2(10)          not null,
   BIDTYPE              VARCHAR2(10)          not null,
   SETTLEMENTDATE       DATE                  not null,
   OFFERDATE            DATE                  not null,
   PERIODID             NUMBER(22,0)          not null,
   VERSIONNO            NUMBER(22,0),
   MAXAVAIL             NUMBER(12,6),
   FIXEDLOAD            NUMBER(12,6),
   ROCUP                NUMBER(6,0),
   ROCDOWN              NUMBER(6,0),
   ENABLEMENTMIN        NUMBER(6,0),
   ENABLEMENTMAX        NUMBER(6,0),
   LOWBREAKPOINT        NUMBER(6,0),
   HIGHBREAKPOINT       NUMBER(6,0),
   BANDAVAIL1           NUMBER(22,0),
   BANDAVAIL2           NUMBER(22,0),
   BANDAVAIL3           NUMBER(22,0),
   BANDAVAIL4           NUMBER(22,0),
   BANDAVAIL5           NUMBER(22,0),
   BANDAVAIL6           NUMBER(22,0),
   BANDAVAIL7           NUMBER(22,0),
   BANDAVAIL8           NUMBER(22,0),
   BANDAVAIL9           NUMBER(22,0),
   BANDAVAIL10          NUMBER(22,0),
   LASTCHANGED          DATE,
   PASAAVAILABILITY     NUMBER(12,0),
   MR_CAPACITY          NUMBER(6,0)
)
   tablespace PSSTAB03;

comment on table BIDPEROFFER is
'BIDPEROFFER shows period-based Energy and Ancillary Service bid data. BIDPEROFFER is a child table of BIDDAYOFFER.';

comment on column BIDPEROFFER.DUID is
'Dispatchable Unit identifier';

comment on column BIDPEROFFER.BIDTYPE is
'Bid Type Identifier';

comment on column BIDPEROFFER.SETTLEMENTDATE is
'Market date starting at 04:05';

comment on column BIDPEROFFER.OFFERDATE is
'Offer date';

comment on column BIDPEROFFER.PERIODID is
'Period ID';

comment on column BIDPEROFFER.VERSIONNO is
'Version number of offer';

comment on column BIDPEROFFER.MAXAVAIL is
'Maximum availability for this BidType in this period';

comment on column BIDPEROFFER.FIXEDLOAD is
'Fixed unit output MW (Energy Bids Only)  A value of zero means no fixed load so the unit is dispatched according to bid and market (rather than zero fixed load)';

comment on column BIDPEROFFER.ROCUP is
'MW/min for raise (Energy Bids Only)';

comment on column BIDPEROFFER.ROCDOWN is
'MW/Min for lower (Energy Bids Only)';

comment on column BIDPEROFFER.ENABLEMENTMIN is
'Minimum Energy Output (MW) at which this ancillary service becomes available (AS Only)';

comment on column BIDPEROFFER.ENABLEMENTMAX is
'Maximum Energy Output (MW) at which this ancillary service can be supplied (AS Only)';

comment on column BIDPEROFFER.LOWBREAKPOINT is
'Minimum Energy Output (MW) at which the unit can provide the full availability (MAXAVAIL) for this ancillary service (AS Only)';

comment on column BIDPEROFFER.HIGHBREAKPOINT is
'Maximum Energy Output (MW) at which the unit can provide the full availability (MAXAVAIL) for this ancillary service (AS Only)';

comment on column BIDPEROFFER.BANDAVAIL1 is
'Availability at price band 1';

comment on column BIDPEROFFER.BANDAVAIL2 is
'Availability at price band 2';

comment on column BIDPEROFFER.BANDAVAIL3 is
'Availability at price band 3';

comment on column BIDPEROFFER.BANDAVAIL4 is
'Availability at price band 4';

comment on column BIDPEROFFER.BANDAVAIL5 is
'Availability at price band 5';

comment on column BIDPEROFFER.BANDAVAIL6 is
'Availability at price band 6';

comment on column BIDPEROFFER.BANDAVAIL7 is
'Availability at price band 7';

comment on column BIDPEROFFER.BANDAVAIL8 is
'Availability at price band 8';

comment on column BIDPEROFFER.BANDAVAIL9 is
'Availability at price band 9';

comment on column BIDPEROFFER.BANDAVAIL10 is
'Availability at price band 10';

comment on column BIDPEROFFER.LASTCHANGED is
'Last date and time record changed';

comment on column BIDPEROFFER.PASAAVAILABILITY is
'Allows for future use for energy bids, being the physical plant capability including any capability potentially available within 24 hours';

comment on column BIDPEROFFER.MR_CAPACITY is
'Mandatory Restriction Offer amount';

alter table BIDPEROFFER
   add constraint BIDPEROFFER_PK primary key (DUID, BIDTYPE, SETTLEMENTDATE, OFFERDATE, PERIODID)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Index: BIDPEROFFER_LCHD_IDX                                  */
/*==============================================================*/
create index BIDPEROFFER_LCHD_IDX on BIDPEROFFER (
   LASTCHANGED ASC
)
tablespace PSSIND03;

/*==============================================================*/
/* Table: BIDPEROFFER_D                                         */
/*==============================================================*/
create table BIDPEROFFER_D (
   SETTLEMENTDATE       DATE                  not null,
   DUID                 VARCHAR2(10)          not null,
   BIDTYPE              VARCHAR2(10)          not null,
   DIRECTION            VARCHAR2(20)          not null,
   INTERVAL_DATETIME    DATE                  not null,
   BIDSETTLEMENTDATE    DATE,
   OFFERDATE            DATE,
   PERIODID             NUMBER(22,0),
   VERSIONNO            NUMBER(22,0),
   MAXAVAIL             NUMBER(12,6),
   FIXEDLOAD            NUMBER(12,6),
   ROCUP                NUMBER(6,0),
   ROCDOWN              NUMBER(6,0),
   ENABLEMENTMIN        NUMBER(6,0),
   ENABLEMENTMAX        NUMBER(6,0),
   LOWBREAKPOINT        NUMBER(6,0),
   HIGHBREAKPOINT       NUMBER(6,0),
   BANDAVAIL1           NUMBER(22,0),
   BANDAVAIL2           NUMBER(22,0),
   BANDAVAIL3           NUMBER(22,0),
   BANDAVAIL4           NUMBER(22,0),
   BANDAVAIL5           NUMBER(22,0),
   BANDAVAIL6           NUMBER(22,0),
   BANDAVAIL7           NUMBER(22,0),
   BANDAVAIL8           NUMBER(22,0),
   BANDAVAIL9           NUMBER(22,0),
   BANDAVAIL10          NUMBER(22,0),
   LASTCHANGED          DATE,
   PASAAVAILABILITY     NUMBER(12,0),
   MR_CAPACITY          NUMBER(6,0),
   ENERGYLIMIT          NUMBER(15,5),
   RECALL_PERIOD        NUMBER(8,3)
)
   tablespace PSSTAB03;

comment on table BIDPEROFFER_D is
'BIDPEROFFER_D shows the public summary of the energy and FCAS offers applicable in the Dispatch for the
intervals identified. BIDPEROFFER_D is the child to BIDDAYOFFER_D.';

comment on column BIDPEROFFER_D.SETTLEMENTDATE is
'Market date for which the bid applied';

comment on column BIDPEROFFER_D.DUID is
'Dispatchable Unit identifier';

comment on column BIDPEROFFER_D.BIDTYPE is
'Bid Type Identifier';

comment on column BIDPEROFFER_D.DIRECTION is
'The power flow direction to which this offer applies: GEN, LOAD or BIDIRECTIONAL';

comment on column BIDPEROFFER_D.INTERVAL_DATETIME is
'Date and Time of the dispatch interval to which the offer applied';

comment on column BIDPEROFFER_D.BIDSETTLEMENTDATE is
'Market date for which the bid was submitted';

comment on column BIDPEROFFER_D.OFFERDATE is
'Offer date and time';

comment on column BIDPEROFFER_D.PERIODID is
'The trading interval period identifier (1-288)';

comment on column BIDPEROFFER_D.VERSIONNO is
'Version number of offer';

comment on column BIDPEROFFER_D.MAXAVAIL is
'Maximum availability for this BidType in this period';

comment on column BIDPEROFFER_D.FIXEDLOAD is
'Fixed unit output MW (Energy Bids Only).  A value of zero means no fixed load so the unit is dispatched according to bid and market (rather than zero fixed load)';

comment on column BIDPEROFFER_D.ROCUP is
'MW/min for raise (Energy Bids Only)';

comment on column BIDPEROFFER_D.ROCDOWN is
'MW/Min for lower (Energy Bids Only)';

comment on column BIDPEROFFER_D.ENABLEMENTMIN is
'Minimum Energy Output (MW) at which this ancillary service becomes available (AS Only)';

comment on column BIDPEROFFER_D.ENABLEMENTMAX is
'Maximum Energy Output (MW) at which this ancillary service can be supplied (AS Only)';

comment on column BIDPEROFFER_D.LOWBREAKPOINT is
'Minimum Energy Output (MW) at which the unit can provide the full availability (MAXAVAIL) for this ancillary service (AS Only)';

comment on column BIDPEROFFER_D.HIGHBREAKPOINT is
'Maximum Energy Output (MW) at which the unit can provide the full availability (MAXAVAIL) for this ancillary service (AS Only)';

comment on column BIDPEROFFER_D.BANDAVAIL1 is
'Availability at price band 1';

comment on column BIDPEROFFER_D.BANDAVAIL2 is
'Availability at price band 2';

comment on column BIDPEROFFER_D.BANDAVAIL3 is
'Availability at price band 3';

comment on column BIDPEROFFER_D.BANDAVAIL4 is
'Availability at price band 4';

comment on column BIDPEROFFER_D.BANDAVAIL5 is
'Availability at price band 5';

comment on column BIDPEROFFER_D.BANDAVAIL6 is
'Availability at price band 6';

comment on column BIDPEROFFER_D.BANDAVAIL7 is
'Availability at price band 7';

comment on column BIDPEROFFER_D.BANDAVAIL8 is
'Availability at price band 8';

comment on column BIDPEROFFER_D.BANDAVAIL9 is
'Availability at price band 9';

comment on column BIDPEROFFER_D.BANDAVAIL10 is
'Availability at price band 10';

comment on column BIDPEROFFER_D.LASTCHANGED is
'Last date and time record changed';

comment on column BIDPEROFFER_D.PASAAVAILABILITY is
'Allows for future use for energy bids, being the physical plant capability including any capability potentially available within 24 hours';

comment on column BIDPEROFFER_D.MR_CAPACITY is
'Mandatory Restriction Offer amount';

comment on column BIDPEROFFER_D.ENERGYLIMIT is
'The Energy limit applying at the end of this dispatch interval in MWh. For GEN this is a lower energy limit. For LOAD this is an upper energy limit';

comment on column BIDPEROFFER_D.RECALL_PERIOD is
'The advance notice (in hours) that a Scheduled Resource requires to achieve the PASA Availability MW for this trading interval.';

alter table BIDPEROFFER_D
   add constraint BIDPEROFFER_D_PK primary key (SETTLEMENTDATE, BIDTYPE, DUID, DIRECTION, INTERVAL_DATETIME)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Index: BIDPEROFFER_D_LCHD_IDX                                */
/*==============================================================*/
create index BIDPEROFFER_D_LCHD_IDX on BIDPEROFFER_D (
   LASTCHANGED ASC
)
tablespace PSSIND03;

/*==============================================================*/
/* Table: BIDTYPES                                              */
/*==============================================================*/
create table BIDTYPES (
   BIDTYPE              VARCHAR2(10)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   DESCRIPTION          VARCHAR2(64),
   NUMBEROFBANDS        NUMBER(3,0),
   NUMDAYSAHEADPRICELOCKED NUMBER(2,0),
   VALIDATIONRULE       VARCHAR2(10),
   LASTCHANGED          DATE,
   SPDALIAS             VARCHAR2(10)
)
   tablespace PSSTAB01;

comment on table BIDTYPES is
'BIDTYPES, together with the associated tracking data in BIDTYPESTRK, define a set of ancillary services with bidding parameters from a given date.
BIDTYPES is static data describing each type of bid quantity, the number of applicable bands, how many days ahead a price lock down becomes effective and the validation rule that applies.';

comment on column BIDTYPES.BIDTYPE is
'Bid Type Identifier';

comment on column BIDTYPES.EFFECTIVEDATE is
'Market date starting at 04:30 inclusive';

comment on column BIDTYPES.VERSIONNO is
'Record version number';

comment on column BIDTYPES.DESCRIPTION is
'Description of this Bid Type';

comment on column BIDTYPES.NUMBEROFBANDS is
'Number of active bands (1 to 10)';

comment on column BIDTYPES.NUMDAYSAHEADPRICELOCKED is
'Number of days prior to the Market Day when prices are locked from 12:30pm';

comment on column BIDTYPES.VALIDATIONRULE is
'ENERGY or AS validation rules to apply.';

comment on column BIDTYPES.LASTCHANGED is
'Last date and time record changed';

comment on column BIDTYPES.SPDALIAS is
'Alias for this BIDTYPE used in the SPD Solver';

alter table BIDTYPES
   add constraint BIDTYPES_PK primary key (BIDTYPE, EFFECTIVEDATE, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BIDTYPES_LCHD_IDX                                     */
/*==============================================================*/
create index BIDTYPES_LCHD_IDX on BIDTYPES (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BIDTYPESTRK                                           */
/*==============================================================*/
create table BIDTYPESTRK (
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BIDTYPESTRK is
'BIDTYPESTRK, together with the associated data in BIDTYPES, define a set of ancillary services with bidding parameters from a given date.';

comment on column BIDTYPESTRK.EFFECTIVEDATE is
'Market date starting at 04:30 inclusive';

comment on column BIDTYPESTRK.VERSIONNO is
'Record version number';

comment on column BIDTYPESTRK.AUTHORISEDDATE is
'Date of record authorisation. A NULL value indicates the record is not authorised.';

comment on column BIDTYPESTRK.AUTHORISEDBY is
'User that authorised record. A NULL value indicates the record is not authorised.';

comment on column BIDTYPESTRK.LASTCHANGED is
'Last date and time record changed';

alter table BIDTYPESTRK
   add constraint BIDTYPESTRK_PK primary key (EFFECTIVEDATE, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BIDTYPESTRK_LCHD_IDX                                  */
/*==============================================================*/
create index BIDTYPESTRK_LCHD_IDX on BIDTYPESTRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLADJUSTMENTS                                       */
/*==============================================================*/
create table BILLADJUSTMENTS (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0),
   PARTICIPANTID        VARCHAR2(10)          not null,
   PARTICIPANTTYPE      VARCHAR2(10),
   ADJCONTRACTYEAR      NUMBER(4,0)           not null,
   ADJWEEKNO            NUMBER(3,0)           not null,
   ADJBILLRUNNO         NUMBER(3,0)           not null,
   PREVAMOUNT           NUMBER(16,6),
   ADJAMOUNT            NUMBER(16,6),
   LASTCHANGED          DATE,
   LRS                  NUMBER(15,5),
   PRS                  NUMBER(15,5),
   OFS                  NUMBER(15,5),
   IRN                  NUMBER(15,5),
   IRP                  NUMBER(15,5),
   INTERESTAMOUNT       NUMBER(15,5)
)
   tablespace PSSTAB01;

comment on column BILLADJUSTMENTS.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLADJUSTMENTS.BILLRUNNO is
'The sequential number of a billing run';

comment on column BILLADJUSTMENTS.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLADJUSTMENTS.PARTICIPANTTYPE is
'Participant type Generator/Customer';

comment on column BILLADJUSTMENTS.ADJCONTRACTYEAR is
'The contract year of the new revised billing run for this adjustment';

comment on column BILLADJUSTMENTS.ADJWEEKNO is
'Week number of the new revised billing run for this adjustment';

comment on column BILLADJUSTMENTS.ADJBILLRUNNO is
'Billing run number of the new revised billing run for this adjustment';

comment on column BILLADJUSTMENTS.PREVAMOUNT is
'Prior account';

comment on column BILLADJUSTMENTS.ADJAMOUNT is
'The total bill figure for the new revised billing run';

comment on column BILLADJUSTMENTS.IRN is
'Interest rate applying to the new amount';

comment on column BILLADJUSTMENTS.IRP is
'Interest rate applying to the principal amount';

comment on column BILLADJUSTMENTS.INTERESTAMOUNT is
'The total interest payable for this adjustment';

alter table BILLADJUSTMENTS
   add constraint BILLADJUSTMENTS_PK primary key (CONTRACTYEAR, WEEKNO, ADJCONTRACTYEAR, ADJWEEKNO, ADJBILLRUNNO, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLADJUSTMENTS_NDX2                                  */
/*==============================================================*/
create index BILLADJUSTMENTS_NDX2 on BILLADJUSTMENTS (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLADJUSTMENTS_LCX                                   */
/*==============================================================*/
create index BILLADJUSTMENTS_LCX on BILLADJUSTMENTS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGAPCCOMPENSATION                                */
/*==============================================================*/
create table BILLINGAPCCOMPENSATION (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   REGIONID             VARCHAR2(10)          not null,
   APCCOMPENSATION      NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table BILLINGAPCCOMPENSATION is
'BILLINGAPCCOMPENSATION shows Administered Price Cap (APC) compensation amounts for the billing period. Data is for each participant by region.';

comment on column BILLINGAPCCOMPENSATION.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGAPCCOMPENSATION.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGAPCCOMPENSATION.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGAPCCOMPENSATION.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGAPCCOMPENSATION.REGIONID is
'Region Identifier';

comment on column BILLINGAPCCOMPENSATION.APCCOMPENSATION is
'APC Compensation';

comment on column BILLINGAPCCOMPENSATION.LASTCHANGED is
'Last changed date and time';

alter table BILLINGAPCCOMPENSATION
   add constraint BILLINGAPCCOMPENSATION_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGAPCCOMPENSATION_LCX                            */
/*==============================================================*/
create index BILLINGAPCCOMPENSATION_LCX on BILLINGAPCCOMPENSATION (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGAPCRECOVERY                                    */
/*==============================================================*/
create table BILLINGAPCRECOVERY (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   REGIONID             VARCHAR2(10)          not null,
   APCRECOVERY          NUMBER(15,0),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table BILLINGAPCRECOVERY is
'BILLINGAPCRECOVERY shows the Administered Price Cap (APC) Recovery for the billing period. Data is for each participant by region.';

comment on column BILLINGAPCRECOVERY.CONTRACTYEAR is
'Contract year';

comment on column BILLINGAPCRECOVERY.WEEKNO is
'Billing week';

comment on column BILLINGAPCRECOVERY.BILLRUNNO is
'Billing run number';

comment on column BILLINGAPCRECOVERY.PARTICIPANTID is
'Participant identifier';

comment on column BILLINGAPCRECOVERY.REGIONID is
'Region Identifier';

comment on column BILLINGAPCRECOVERY.APCRECOVERY is
'APC Recovery amount for week';

comment on column BILLINGAPCRECOVERY.LASTCHANGED is
'Last date and time record changed';

alter table BILLINGAPCRECOVERY
   add constraint BILLINGAPCRECOVERY_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGAPCRECOVERY_LCX                                */
/*==============================================================*/
create index BILLINGAPCRECOVERY_LCX on BILLINGAPCRECOVERY (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGASPAYMENTS                                     */
/*==============================================================*/
create table BILLINGASPAYMENTS (
   REGIONID             VARCHAR2(10),
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   CONNECTIONPOINTID    VARCHAR2(10)          not null,
   RAISE6SEC            NUMBER(15,5),
   LOWER6SEC            NUMBER(15,5),
   RAISE60SEC           NUMBER(15,5),
   LOWER60SEC           NUMBER(15,5),
   AGC                  NUMBER(15,5),
   FCASCOMP             NUMBER(15,5),
   LOADSHED             NUMBER(15,5),
   RGUL                 NUMBER(15,5),
   RGUU                 NUMBER(15,5),
   REACTIVEPOWER        NUMBER(15,5),
   SYSTEMRESTART        NUMBER(15,5),
   LASTCHANGED          DATE,
   LOWER5MIN            NUMBER(15,5),
   RAISE5MIN            NUMBER(15,5),
   LOWERREG             NUMBER(15,5),
   RAISEREG             NUMBER(15,5),
   AVAILABILITY_REACTIVE NUMBER(18,8),
   AVAILABILITY_REACTIVE_RBT NUMBER(18,8),
   RAISE1SEC            NUMBER(18,8),
   LOWER1SEC            NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLINGASPAYMENTS is
'BILLINGASPAYMENTS shows Ancillary Service payments for each billing period by each of the Ancillary Service types for each participants connection points.';

comment on column BILLINGASPAYMENTS.REGIONID is
'Region Identifier';

comment on column BILLINGASPAYMENTS.CONTRACTYEAR is
'Contract Year';

comment on column BILLINGASPAYMENTS.WEEKNO is
'Week No';

comment on column BILLINGASPAYMENTS.BILLRUNNO is
'Billing Run No.';

comment on column BILLINGASPAYMENTS.PARTICIPANTID is
'Participant Identifier';

comment on column BILLINGASPAYMENTS.CONNECTIONPOINTID is
'Connection point identifier';

comment on column BILLINGASPAYMENTS.RAISE6SEC is
'Raise 6 Sec Payments';

comment on column BILLINGASPAYMENTS.LOWER6SEC is
'Lower 6 Sec Payments';

comment on column BILLINGASPAYMENTS.RAISE60SEC is
'Raise 60 Sec Payments';

comment on column BILLINGASPAYMENTS.LOWER60SEC is
'Lower 60 Sec Payments';

comment on column BILLINGASPAYMENTS.AGC is
'AGC Payments';

comment on column BILLINGASPAYMENTS.FCASCOMP is
'Frequency Control Compensation Payments';

comment on column BILLINGASPAYMENTS.LOADSHED is
'Load Shed Payments';

comment on column BILLINGASPAYMENTS.RGUL is
'Rapid Generator unit Loading Payments';

comment on column BILLINGASPAYMENTS.RGUU is
'Rapid Generator Unit Unloading Payments';

comment on column BILLINGASPAYMENTS.REACTIVEPOWER is
'Reactive Power Payments';

comment on column BILLINGASPAYMENTS.SYSTEMRESTART is
'System Restart Payments';

comment on column BILLINGASPAYMENTS.LASTCHANGED is
'The latest date and time that a file was updated or inserted';

comment on column BILLINGASPAYMENTS.LOWER5MIN is
'Lower 5 Minute Payment';

comment on column BILLINGASPAYMENTS.RAISE5MIN is
'Raise 5 Minute Payment';

comment on column BILLINGASPAYMENTS.LOWERREG is
'Lower 5 Minute Regulation Payment';

comment on column BILLINGASPAYMENTS.RAISEREG is
'Raise 5 Minute Regulation Payment';

comment on column BILLINGASPAYMENTS.AVAILABILITY_REACTIVE is
'The total availability payment';

comment on column BILLINGASPAYMENTS.AVAILABILITY_REACTIVE_RBT is
'The total availability payment rebate';

comment on column BILLINGASPAYMENTS.RAISE1SEC is
'Payment amount for the very fast raise service';

comment on column BILLINGASPAYMENTS.LOWER1SEC is
'Payment amount for the very fast lower service';

alter table BILLINGASPAYMENTS
   add constraint BILLINGASPAYMENTS_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, CONNECTIONPOINTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGASPAYMENTS_LCX                                 */
/*==============================================================*/
create index BILLINGASPAYMENTS_LCX on BILLINGASPAYMENTS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGASRECOVERY                                     */
/*==============================================================*/
create table BILLINGASRECOVERY (
   REGIONID             VARCHAR2(10)          not null,
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   RAISE6SEC            NUMBER(15,5),
   LOWER6SEC            NUMBER(15,5),
   RAISE60SEC           NUMBER(15,5),
   LOWER60SEC           NUMBER(15,5),
   AGC                  NUMBER(15,5),
   FCASCOMP             NUMBER(15,5),
   LOADSHED             NUMBER(15,5),
   RGUL                 NUMBER(15,5),
   RGUU                 NUMBER(15,5),
   REACTIVEPOWER        NUMBER(15,5),
   SYSTEMRESTART        NUMBER(15,5),
   LASTCHANGED          DATE,
   RAISE6SEC_GEN        NUMBER(15,5),
   LOWER6SEC_GEN        NUMBER(15,5),
   RAISE60SEC_GEN       NUMBER(15,5),
   LOWER60SEC_GEN       NUMBER(15,5),
   AGC_GEN              NUMBER(15,5),
   FCASCOMP_GEN         NUMBER(15,5),
   LOADSHED_GEN         NUMBER(15,5),
   RGUL_GEN             NUMBER(15,5),
   RGUU_GEN             NUMBER(15,5),
   REACTIVEPOWER_GEN    NUMBER(15,5),
   SYSTEMRESTART_GEN    NUMBER(15,5),
   LOWER5MIN            NUMBER(15,5),
   RAISE5MIN            NUMBER(15,5),
   LOWERREG             NUMBER(18,8),
   RAISEREG             NUMBER(18,8),
   LOWER5MIN_GEN        NUMBER(16,6),
   RAISE5MIN_GEN        NUMBER(16,6),
   LOWERREG_GEN         NUMBER(16,6),
   RAISEREG_GEN         NUMBER(16,6),
   AVAILABILITY_REACTIVE NUMBER(18,8),
   AVAILABILITY_REACTIVE_RBT NUMBER(18,8),
   AVAILABILITY_REACTIVE_GEN NUMBER(18,8),
   AVAILABILITY_REACTIVE_RBT_GEN NUMBER(18,8),
   RAISE1SEC            NUMBER(18,8),
   LOWER1SEC            NUMBER(18,8),
   RAISE1SEC_GEN        NUMBER(18,8),
   LOWER1SEC_GEN        NUMBER(18,8),
   LOWERREG_ACE         NUMBER(18,8),
   RAISEREG_ACE         NUMBER(18,8),
   RAISE1SEC_ACE        NUMBER(18,8),
   RAISE1SEC_ASOE       NUMBER(18,8),
   LOWER1SEC_ACE        NUMBER(18,8),
   LOWER1SEC_ASOE       NUMBER(18,8),
   RAISE6SEC_ACE        NUMBER(18,8),
   RAISE6SEC_ASOE       NUMBER(18,8),
   LOWER6SEC_ACE        NUMBER(18,8),
   LOWER6SEC_ASOE       NUMBER(18,8),
   RAISE60SEC_ACE       NUMBER(18,8),
   RAISE60SEC_ASOE      NUMBER(18,8),
   LOWER60SEC_ACE       NUMBER(18,8),
   LOWER60SEC_ASOE      NUMBER(18,8),
   RAISE5MIN_ACE        NUMBER(18,8),
   RAISE5MIN_ASOE       NUMBER(18,8),
   LOWER5MIN_ACE        NUMBER(18,8),
   LOWER5MIN_ASOE       NUMBER(18,8),
   REACTIVEPOWER_ACE    NUMBER(18,8),
   REACTIVEPOWER_ASOE   NUMBER(18,8),
   LOADSHED_ACE         NUMBER(18,8),
   LOADSHED_ASOE        NUMBER(18,8),
   SYSTEMRESTART_ACE    NUMBER(18,8),
   SYSTEMRESTART_ASOE   NUMBER(18,8),
   AVAILABILITY_REACTIVE_ACE NUMBER(18,8),
   AVAILABILITY_REACTIVE_ASOE NUMBER(18,8),
   AVAILABILITY_REACTIVE_RBT_ACE NUMBER(18,8),
   AVAILABILITY_REACTIVE_RBT_ASOE NUMBER(18,8),
   LOWERREG_USED        NUMBER(18,8),
   LOWERREG_UNUSED      NUMBER(18,8),
   RAISEREG_USED        NUMBER(18,8),
   RAISEREG_UNUSED      NUMBER(18,8),
   LOWERREG_USED_ACE    NUMBER(18,8),
   LOWERREG_USED_ASOE   NUMBER(18,8),
   LOWERREG_USED_RESIDUAL NUMBER(18,8),
   RAISEREG_USED_ACE    NUMBER(18,8),
   RAISEREG_USED_ASOE   NUMBER(18,8),
   RAISEREG_USED_RESIDUAL NUMBER(18,8),
   LOWERREG_UNUSED_ACE  NUMBER(18,8),
   LOWERREG_UNUSED_ASOE NUMBER(18,8),
   LOWERREG_UNUSED_RESIDUAL NUMBER(18,8),
   RAISEREG_UNUSED_ACE  NUMBER(18,8),
   RAISEREG_UNUSED_ASOE NUMBER(18,8),
   RAISEREG_UNUSED_RESIDUAL NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLINGASRECOVERY is
'BILLINGASRECOVERY shows participant charges for Ancillary Services for the billing period. This view shows the billing amounts for Ancillary Service Recovery.';

comment on column BILLINGASRECOVERY.REGIONID is
'Region Identifier';

comment on column BILLINGASRECOVERY.CONTRACTYEAR is
'Contract Year';

comment on column BILLINGASRECOVERY.WEEKNO is
'Week No';

comment on column BILLINGASRECOVERY.BILLRUNNO is
'Billing Run No.';

comment on column BILLINGASRECOVERY.PARTICIPANTID is
'Participant Identifier';

comment on column BILLINGASRECOVERY.RAISE6SEC is
'Raise 6 Sec Recovery. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWER6SEC is
'Lower 6 Sec Recovery. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.RAISE60SEC is
'Raise 60 Sec Recovery. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWER60SEC is
'Lower 60 Sec Recovery. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.AGC is
'AGC Recovery - Not used since circa 2000';

comment on column BILLINGASRECOVERY.FCASCOMP is
'Frequency Control Compensation Recovery - Not used since circa 2000';

comment on column BILLINGASRECOVERY.LOADSHED is
'Load Shed Recovery. Post-IESS the value in this column only represent the Testing Payment Recovery from Customers. 0 if no testing payment exists.';

comment on column BILLINGASRECOVERY.RGUL is
'Rapid Generator Unit Loading Recovery - Not used since December 2001';

comment on column BILLINGASRECOVERY.RGUU is
'Rapid Generator Unit Unloading Recovery - Not used since December 2001';

comment on column BILLINGASRECOVERY.REACTIVEPOWER is
'Reactive Power Recovery. Post-IESS the value in this column only represent the Testing Payment Recovery from Customers. 0 if no testing payment exists.';

comment on column BILLINGASRECOVERY.SYSTEMRESTART is
'System Restart Recovery. Post-IESS the value in this column only represent the Testing Payment Recovery from Customers. 0 if no testing payment exists';

comment on column BILLINGASRECOVERY.LASTCHANGED is
'The latest date and time a file was updated/inserted';

comment on column BILLINGASRECOVERY.RAISE6SEC_GEN is
'Raise 6 Sec Recovery for Generator. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWER6SEC_GEN is
'Lower 6 Sec Recovery for Generator. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.RAISE60SEC_GEN is
'Raise 60 Sec Recovery for Generator. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWER60SEC_GEN is
'Lower 60 Sec Recovery for Generator. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.AGC_GEN is
'AGC Recovery for Generator';

comment on column BILLINGASRECOVERY.FCASCOMP_GEN is
'Frequency Control Compensation Recovery for Generator';

comment on column BILLINGASRECOVERY.LOADSHED_GEN is
'Load Shed Recovery for Generator. Post-IESS the value in this column only represent the Testing Payment Recovery from Generators. 0 if no testing payment exists.';

comment on column BILLINGASRECOVERY.RGUL_GEN is
'Rapid Generator unit Loading Recovery for. Generator - Not used since December 2001';

comment on column BILLINGASRECOVERY.RGUU_GEN is
'Rapid Generator Unit Unloading Recovery for Generator - Not used since December 2001';

comment on column BILLINGASRECOVERY.REACTIVEPOWER_GEN is
'Reactive Power Recovery for Generator. Post-IESS the value in this column only represent the Testing Payment Recovery from Generators. 0 if no testing payment exists.';

comment on column BILLINGASRECOVERY.SYSTEMRESTART_GEN is
'System Restart Recovery for Generator. Post-IESS the value in this column only represent the Testing Payment Recovery from Generators. 0 if no testing payment exists.';

comment on column BILLINGASRECOVERY.LOWER5MIN is
'Recovery amount for the Lower 5 Minute service attributable to customer connection points. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.RAISE5MIN is
'Recovery amount for the Raise 5 Minute service attributable to customer connection points. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWERREG is
'Post-IESS the amount in this column represent only the Lower Regulation FCAS MPF Recovery Amount from Customer and Generator Connection Point MPFs, no Residue Amounts are added to this column value. For Billing Weeks past FPP Rule Effective Date this column is not used. Always show 0.';

comment on column BILLINGASRECOVERY.RAISEREG is
'Post-IESS the amount in this column represent only the Raise Regulation FCAS MPF Recovery Amount from Customer and Generator Connection Point MPFs, no Residue Amounts are added to this column value. For Billing Weeks past FPP Rule Effective Date this column is not used. Always show 0.';

comment on column BILLINGASRECOVERY.LOWER5MIN_GEN is
'Recovery amount for the Lower 5 Minute service attributable to generator connection points. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.RAISE5MIN_GEN is
'Recovery amount for the Raise 5 Minute service attributable to generator connection points. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWERREG_GEN is
'Recovery amount for the Lower Regulation service attributable to generator connection points. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.RAISEREG_GEN is
'Recovery amount for the Raise Regulation service attributable to generator connection points. NULL for Billing Week post the IESS rule effective date. NULL for Billing Week post the IESS rule effective date.';

comment on column BILLINGASRECOVERY.AVAILABILITY_REACTIVE is
'The total availability payment recovery amount (customer).. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.AVAILABILITY_REACTIVE_RBT is
'The total availability payment rebate recovery amount (customer).. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.AVAILABILITY_REACTIVE_GEN is
'The total availability payment recovery amount (Generator).. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.AVAILABILITY_REACTIVE_RBT_GEN is
'The total availability payment rebate recovery amount (Generator).. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.RAISE1SEC is
'Customer recovery amount for the very fast raise service. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWER1SEC is
'Customer recovery amount for the very fast lower service. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.RAISE1SEC_GEN is
'Generator recovery amount for the very fast raise service. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWER1SEC_GEN is
'Generator recovery amount for the very fast lower service. NULL for Billing Week post the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWERREG_ACE is
'The Lower Regulation FCAS Residue Recovery Amount using ACE MWh values. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.RAISEREG_ACE is
'The Raise Regulation FCAS Residue Recovery Amount using ACE MWh values. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.RAISE1SEC_ACE is
'The Raise1Sec FCAS Recovery Amount for the Participant and Region from ACE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.RAISE1SEC_ASOE is
'The Raise1Sec FCAS Recovery Amount for the Participant and Region from ASOE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWER1SEC_ACE is
'The Lower1Sec FCAS Recovery Amount for the Participant and Region from ACE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWER1SEC_ASOE is
'The Lower1Sec FCAS Recovery Amount for the Participant and Region from ASOE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.RAISE6SEC_ACE is
'The Raise6Sec FCAS Recovery Amount for the Participant and Region from ACE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.RAISE6SEC_ASOE is
'The Raise6Sec FCAS Recovery Amount for the Participant and Region from ASOE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWER6SEC_ACE is
'The Lower6Sec FCAS Recovery Amount for the Participant and Region from ACE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWER6SEC_ASOE is
'The Lower6Sec FCAS Recovery Amount for the Participant and Region from ASOE MWh Portion. NULL for Billing Week prior to the IESS rule effective date value.';

comment on column BILLINGASRECOVERY.RAISE60SEC_ACE is
'The Raise60Sec FCAS Recovery Amount for the Participant and Region from ACE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.RAISE60SEC_ASOE is
'The Raise60Sec FCAS Recovery Amount for the Participant and Region from ASOE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWER60SEC_ACE is
'The Lower60Sec FCAS Recovery Amount for the Participant and Region from ACE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWER60SEC_ASOE is
'The Lower60Sec FCAS Recovery Amount for the Participant and Region from ASOE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.RAISE5MIN_ACE is
'The Raise5Min FCAS Recovery Amount for the Participant and Region from ACE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.RAISE5MIN_ASOE is
'The Raise5Min FCAS Recovery Amount for the Participant and Region from ASOE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWER5MIN_ACE is
'The Lower5Min FCAS Recovery Amount for the Participant and Region from ACE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWER5MIN_ASOE is
'The Lower5Min FCAS Recovery Amount for the Participant and Region from ASOE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.REACTIVEPOWER_ACE is
'The Reactive Power Ancillary Service Recovery Amount for for the Participant and Region from ACE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.REACTIVEPOWER_ASOE is
'The Reactive Power Ancillary Service Recovery Amount for for the Participant and Region from ASOE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOADSHED_ACE is
'The Load Shed Ancillary Service Recovery Amount for for the Participant and Region from ACE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOADSHED_ASOE is
'The Load Shed Ancillary Service Recovery Amount for for the Participant and Region from ASOE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.SYSTEMRESTART_ACE is
'The System Restart Ancillary Service Recovery Amount for for the Participant and Region from ACE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.SYSTEMRESTART_ASOE is
'The System Restart Ancillary Service Recovery Amount for for the Participant and Region from ASOE MWh Portion. NULL for Billing Week prior to the IESS rule effective date,';

comment on column BILLINGASRECOVERY.AVAILABILITY_REACTIVE_ACE is
'The Reactive Power Ancillary Service Availability Payment Recovery Amount for the Participant and Region from ACE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.AVAILABILITY_REACTIVE_ASOE is
'The Reactive Power Ancillary Service Availability Payment Recovery Amount for the Participant and Region from ASOE MWh Portion. For Pre-IESS Settlement dates this column will have NULL value. For Pre-IESS Settlement dates this column will have NULL value.';

comment on column BILLINGASRECOVERY.AVAILABILITY_REACTIVE_RBT_ACE is
'The Reactive Power Ancillary Service Availability Rebate Payment Recovery Amount for the Participant and Region from ACE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.AVAILABILITY_REACTIVE_RBT_ASOE is
'The Reactive Power Ancillary Service Availability Rebate Payment Recovery Amount for the Participant and Region from ASOE MWh Portion. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLINGASRECOVERY.LOWERREG_USED is
'The FCAS Lower Reg Service Used Amount for the Billing Week. This column will be NULL for Billing Week Prior to FPP Rule Effective Date';

comment on column BILLINGASRECOVERY.LOWERREG_UNUSED is
'The FCAS Lower Reg Service Unused Amount for the Billing Week. This column will be NULL for Billing Week Prior to FPP Rule Effective Date';

comment on column BILLINGASRECOVERY.RAISEREG_USED is
'The FCAS Raise Reg Service Used Amount for the Billing Week. This column will be NULL for Billing Week Prior to FPP Rule Effective Date';

comment on column BILLINGASRECOVERY.RAISEREG_UNUSED is
'The FCAS Raise Reg Service Unused Amount for the Billing Week. This column will be NULL for Billing Week Prior to FPP Rule Effective Date';

comment on column BILLINGASRECOVERY.LOWERREG_USED_ACE is
'The FCAS Lower Reg Service Used ACE Residual Amount for the Billing Week. This column will be NULL for Billing Week Prior to FPP Rule Effective Date';

comment on column BILLINGASRECOVERY.LOWERREG_USED_ASOE is
'The FCAS Lower Reg Service Used ASOE Amount for the Billing Week. This column will be NULL for Billing Week Prior to FPP Rule Effective Date';

comment on column BILLINGASRECOVERY.LOWERREG_USED_RESIDUAL is
'LOWERREG_USED_ACE + LOWERREG_USED_ASOE';

comment on column BILLINGASRECOVERY.RAISEREG_USED_ACE is
'The FCAS Raise Reg Service Used ACE Residual Amount for the Billing Week. This column will be NULL for Billing Week Prior to FPP Rule Effective Date';

comment on column BILLINGASRECOVERY.RAISEREG_USED_ASOE is
'The FCAS Raise Reg Service Used ASOE Residual Amount for the Billing Week. This column will be NULL for Billing Week Prior to FPP Rule Effective Date';

comment on column BILLINGASRECOVERY.RAISEREG_USED_RESIDUAL is
'RAISEREG_USED_ACE + RAISEREG_USED_ASOE';

comment on column BILLINGASRECOVERY.LOWERREG_UNUSED_ACE is
'The FCAS Lower Reg Service Unused ACE Residual Amount for the Billing Week. This column will be NULL for Billing Week Prior to FPP Rule Effective Date';

comment on column BILLINGASRECOVERY.LOWERREG_UNUSED_ASOE is
'The FCAS Lower Reg Service Unused ASOE Residual Amount for the Billing Week. This column will be NULL for Billing Week Prior to FPP Rule Effective Date';

comment on column BILLINGASRECOVERY.LOWERREG_UNUSED_RESIDUAL is
'LOWERREG_UNUSED_ACE + LOWERREG_UNUSED_ASOE';

comment on column BILLINGASRECOVERY.RAISEREG_UNUSED_ACE is
'The FCAS Raise Reg Service Unused ACE Residual Amount for the Billing Week. This column will be NULL for Billing Week Prior to FPP Rule Effective Date';

comment on column BILLINGASRECOVERY.RAISEREG_UNUSED_ASOE is
'The FCAS Raise Reg Service Unused ASOE Residual Amount for the Billing Week. This column will be NULL for Billing Week Prior to FPP Rule Effective Date';

comment on column BILLINGASRECOVERY.RAISEREG_UNUSED_RESIDUAL is
'RAISEREG_UNUSED_ACE + RAISEREG_UNUSED_ASOE';

alter table BILLINGASRECOVERY
   add constraint BILLINGASRECOVERY_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGASRECOVERY_LCX                                 */
/*==============================================================*/
create index BILLINGASRECOVERY_LCX on BILLINGASRECOVERY (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGCALENDAR                                       */
/*==============================================================*/
create table BILLINGCALENDAR (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   STARTDATE            DATE,
   ENDDATE              DATE,
   PRELIMINARYSTATEMENTDATE DATE,
   FINALSTATEMENTDATE   DATE,
   PAYMENTDATE          DATE,
   LASTCHANGED          DATE,
   REVISION1_STATEMENTDATE DATE,
   REVISION2_STATEMENTDATE DATE
)
   tablespace PSSTAB01;

comment on table BILLINGCALENDAR is
'BILLINGCALENDAR sets out the billing calendar for the year, with week number 1 starting on 1 January. BILLINGCALENDAR advises preliminary and final statement posting date and corresponding  settlement for each billing week.';

comment on column BILLINGCALENDAR.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGCALENDAR.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGCALENDAR.STARTDATE is
'Start Date of week';

comment on column BILLINGCALENDAR.ENDDATE is
'End Date of week';

comment on column BILLINGCALENDAR.PRELIMINARYSTATEMENTDATE is
'Preliminary Statement Date';

comment on column BILLINGCALENDAR.FINALSTATEMENTDATE is
'Final Statement Date';

comment on column BILLINGCALENDAR.PAYMENTDATE is
'Payment Date';

comment on column BILLINGCALENDAR.LASTCHANGED is
'Last date and time record changed';

comment on column BILLINGCALENDAR.REVISION1_STATEMENTDATE is
'Revision 1 Statement Date for the billing week.';

comment on column BILLINGCALENDAR.REVISION2_STATEMENTDATE is
'Revision 2 Statement Date for the billing week.';

alter table BILLINGCALENDAR
   add constraint BILLINGCALENDAR_PK primary key (CONTRACTYEAR, WEEKNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGCALENDAR_LCX                                   */
/*==============================================================*/
create index BILLINGCALENDAR_LCX on BILLINGCALENDAR (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGCPDATA                                         */
/*==============================================================*/
create table BILLINGCPDATA (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   CONNECTIONPOINTID    VARCHAR2(10)          not null,
   AGGREGATEENERGY      NUMBER(16,6),
   PURCHASES            NUMBER(16,6),
   LASTCHANGED          DATE,
   MDA                  VARCHAR2(10)          not null,
   AFE                  NUMBER(18,8),
   DME                  NUMBER(18,8),
   UFEA                 NUMBER(18,8),
   AGE                  NUMBER(18,8),
   SOLDENERGY           NUMBER(18,8),
   SALES                NUMBER(18,8),
   PURCHASEDENERGY      NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLINGCPDATA is
'BILLINGCPDATA shows energy quantity and $ value purchased per participant connection point. The table is populated for participants using Data Model 5.2. For participants using Data Model 5.3 or higher, BILLING_ENERGY_TRANSACTIONS AND BILLING_ENERGY_GENSET_DETAIL replace BILLINGCPDATA.';

comment on column BILLINGCPDATA.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGCPDATA.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGCPDATA.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGCPDATA.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGCPDATA.CONNECTIONPOINTID is
'Unique connection point identifier';

comment on column BILLINGCPDATA.AGGREGATEENERGY is
'Aggregate energy purchased/sold by customer, in MWh, plus UFEA. When GS commences, this includes the UFEA amount in the settlement runs.';

comment on column BILLINGCPDATA.PURCHASES is
'The Purchase column has the dollar value of the Energy Purchased rather than Aggregate Energy Dollar';

comment on column BILLINGCPDATA.LASTCHANGED is
'Last date and time record changed';

comment on column BILLINGCPDATA.MDA is
'relevant MDA for this connection point.';

comment on column BILLINGCPDATA.AFE is
'Adjusted Gross Energy for this Market Customer FRMP and TNI in the Billing run, excluding any UFEA component.';

comment on column BILLINGCPDATA.DME is
'Sum of ME- for all NMIs at this Market Customer FRMP and TNI in the Billing run.';

comment on column BILLINGCPDATA.UFEA is
'Share of UFE allocated to this FRMP and TNI in the Billing run.';

comment on column BILLINGCPDATA.AGE is
'Adjusted Gross Energy for this Market Customer FRMP and TNI in the trading interval. This will include the UFEA value once financial settlement of UFE commences with GS.';

comment on column BILLINGCPDATA.SOLDENERGY is
'Energy sold at the connection point by the participant in this billing run';

comment on column BILLINGCPDATA.SALES is
'The total cost of energy sold at the connection point by the participant in this billing run';

comment on column BILLINGCPDATA.PURCHASEDENERGY is
'The energy consumed at the connection point by the participant in this billing run';

alter table BILLINGCPDATA
   add constraint BILLINGCPDATA_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, CONNECTIONPOINTID, MDA)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGCPDATA_NDX2                                    */
/*==============================================================*/
create index BILLINGCPDATA_NDX2 on BILLINGCPDATA (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGCPDATA_LCX                                     */
/*==============================================================*/
create index BILLINGCPDATA_LCX on BILLINGCPDATA (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGCPSUM                                          */
/*==============================================================*/
create table BILLINGCPSUM (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   PARTICIPANTTYPE      VARCHAR2(10)          not null,
   PREVIOUSAMOUNT       NUMBER(16,6),
   ADJUSTEDAMOUNT       NUMBER(16,6),
   ADJUSTMENTWEEKNO     NUMBER(3,0),
   ADJUSTMENTRUNNO      NUMBER(3,0),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLINGCPSUM is
'BILLINGCPSUM shows adjustments for a billing run by participant.';

comment on column BILLINGCPSUM.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGCPSUM.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGCPSUM.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGCPSUM.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGCPSUM.PARTICIPANTTYPE is
'Participant type Generator/Customer';

comment on column BILLINGCPSUM.PREVIOUSAMOUNT is
'Previous amount billed';

comment on column BILLINGCPSUM.ADJUSTEDAMOUNT is
'Adjusted amount billed';

comment on column BILLINGCPSUM.ADJUSTMENTWEEKNO is
'Week no of adjustment';

comment on column BILLINGCPSUM.ADJUSTMENTRUNNO is
'Run no of adjustment';

comment on column BILLINGCPSUM.LASTCHANGED is
'Last date and time record changed';

alter table BILLINGCPSUM
   add constraint BILLINGCPSUM_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, PARTICIPANTTYPE)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGCPSUM_LCX                                      */
/*==============================================================*/
create index BILLINGCPSUM_LCX on BILLINGCPSUM (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGCPSUM_NDX2                                     */
/*==============================================================*/
create index BILLINGCPSUM_NDX2 on BILLINGCPSUM (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGCUSTEXCESSGEN                                  */
/*==============================================================*/
create table BILLINGCUSTEXCESSGEN (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   SETTLEMENTDATE       DATE                  not null,
   PERIODID             NUMBER(3,0)           not null,
   EXCESSGENPAYMENT     NUMBER(16,6),
   LASTCHANGED          DATE,
   REGIONID             VARCHAR2(10)          not null
)
   tablespace PSSTAB01;

comment on table BILLINGCUSTEXCESSGEN is
'BILLINGCUSTEXCESSGEN shows excess generation payments for each participant cutover.';

comment on column BILLINGCUSTEXCESSGEN.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGCUSTEXCESSGEN.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGCUSTEXCESSGEN.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGCUSTEXCESSGEN.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGCUSTEXCESSGEN.SETTLEMENTDATE is
'Calendar Settlement Date';

comment on column BILLINGCUSTEXCESSGEN.PERIODID is
'Half hourly trading period that excess generation is for';

comment on column BILLINGCUSTEXCESSGEN.EXCESSGENPAYMENT is
'Payment by Customer for Excess Generation';

comment on column BILLINGCUSTEXCESSGEN.LASTCHANGED is
'Last date and time record changed';

comment on column BILLINGCUSTEXCESSGEN.REGIONID is
'Region Identifier';

alter table BILLINGCUSTEXCESSGEN
   add constraint BILLINGCUSTEXCESSGEN_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, REGIONID, SETTLEMENTDATE, PERIODID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGCUSTEXCESSGEN_LCX                              */
/*==============================================================*/
create index BILLINGCUSTEXCESSGEN_LCX on BILLINGCUSTEXCESSGEN (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGDAYTRK                                         */
/*==============================================================*/
create table BILLINGDAYTRK (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLINGDAYTRK is
'BILLINGDAYTRK is key for matching settlement versions with billing runs. BILLINGDAYTRK displays the billrunnos per billing week, and the settlement version numbers per settlement day comprising the billrunno.';

comment on column BILLINGDAYTRK.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGDAYTRK.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGDAYTRK.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGDAYTRK.SETTLEMENTDATE is
'Calendar Settlement Date contained in the billing run.';

comment on column BILLINGDAYTRK.RUNNO is
'Settlement run number used for each settlement date in that billing run.';

comment on column BILLINGDAYTRK.LASTCHANGED is
'Last date and time record changed';

alter table BILLINGDAYTRK
   add constraint BILLINGDAYTRK_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, SETTLEMENTDATE)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGDAYTRK_LCX                                     */
/*==============================================================*/
create index BILLINGDAYTRK_LCX on BILLINGDAYTRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGEXCESSGEN                                      */
/*==============================================================*/
create table BILLINGEXCESSGEN (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   SETTLEMENTDATE       DATE                  not null,
   PERIODID             NUMBER(3,0)           not null,
   EXCESSENERGYCOST     NUMBER(15,5),
   LASTCHANGED          DATE,
   REGIONID             VARCHAR2(10)          not null
)
   tablespace PSSTAB01;

comment on table BILLINGEXCESSGEN is
'BILLINGEXCESSGEN shows the excess generation cost by period for each participant.';

comment on column BILLINGEXCESSGEN.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGEXCESSGEN.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGEXCESSGEN.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGEXCESSGEN.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGEXCESSGEN.SETTLEMENTDATE is
'Calendar settlement date record becomes effective';

comment on column BILLINGEXCESSGEN.PERIODID is
'Settlement interval within the settlement date (1-48) starting at 00:30';

comment on column BILLINGEXCESSGEN.EXCESSENERGYCOST is
'Cost of excess energy attributed to this customer';

comment on column BILLINGEXCESSGEN.LASTCHANGED is
'Last date and time record changed';

comment on column BILLINGEXCESSGEN.REGIONID is
'Region Identifier';

alter table BILLINGEXCESSGEN
   add constraint BILLINGEXCESSGEN_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, REGIONID, SETTLEMENTDATE, PERIODID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGEXCESSGEN_LCX                                  */
/*==============================================================*/
create index BILLINGEXCESSGEN_LCX on BILLINGEXCESSGEN (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGEXCESSGEN_NDX2                                 */
/*==============================================================*/
create index BILLINGEXCESSGEN_NDX2 on BILLINGEXCESSGEN (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGFEES                                           */
/*==============================================================*/
create table BILLINGFEES (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   MARKETFEEID          VARCHAR2(10)          not null,
   RATE                 NUMBER(15,5),
   ENERGY               NUMBER(16,6),
   VALUE                NUMBER(15,5),
   LASTCHANGED          DATE,
   PARTICIPANTCATEGORYID VARCHAR2(10)          not null
)
   tablespace PSSTAB01;

comment on table BILLINGFEES is
'BILLINGFEES presents pool fees applied to the statement, per billing run.';

comment on column BILLINGFEES.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGFEES.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGFEES.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGFEES.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGFEES.MARKETFEEID is
'Market fee identifier';

comment on column BILLINGFEES.RATE is
'Market fee rate';

comment on column BILLINGFEES.ENERGY is
'Energy, in MWh';

comment on column BILLINGFEES.VALUE is
'Fee in $';

comment on column BILLINGFEES.LASTCHANGED is
'Last date and time record changed';

comment on column BILLINGFEES.PARTICIPANTCATEGORYID is
'The participant category pertaining to the market fee recovery. Corresponds to the PARTICIPANTCATEGORYID column of the SETMARKETFEES table.';

alter table BILLINGFEES
   add constraint BILLINGFEES_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, MARKETFEEID, PARTICIPANTCATEGORYID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGFEES_LCX                                       */
/*==============================================================*/
create index BILLINGFEES_LCX on BILLINGFEES (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGFEES_NDX2                                      */
/*==============================================================*/
create index BILLINGFEES_NDX2 on BILLINGFEES (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGFINANCIALADJUSTMENTS                           */
/*==============================================================*/
create table BILLINGFINANCIALADJUSTMENTS (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   PARTICIPANTTYPE      VARCHAR2(10),
   ADJUSTMENTITEM       VARCHAR2(64)          not null,
   AMOUNT               NUMBER(15,5),
   VALUE                NUMBER(15,5),
   LASTCHANGED          DATE,
   FINANCIALCODE        NUMBER(10,0),
   BAS_CLASS            VARCHAR2(30)
)
   tablespace PSSTAB01;

comment on table BILLINGFINANCIALADJUSTMENTS is
'BILLINGFINANCIALADJUSTMENTS contains any manual adjustments included in the billing run.';

comment on column BILLINGFINANCIALADJUSTMENTS.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGFINANCIALADJUSTMENTS.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGFINANCIALADJUSTMENTS.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGFINANCIALADJUSTMENTS.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGFINANCIALADJUSTMENTS.PARTICIPANTTYPE is
'Not Used';

comment on column BILLINGFINANCIALADJUSTMENTS.ADJUSTMENTITEM is
'Description of the adjustment being made';

comment on column BILLINGFINANCIALADJUSTMENTS.AMOUNT is
'The amount of the manual adjustment line item';

comment on column BILLINGFINANCIALADJUSTMENTS.VALUE is
'Not Used';

comment on column BILLINGFINANCIALADJUSTMENTS.LASTCHANGED is
'Last date and time the record changed.';

comment on column BILLINGFINANCIALADJUSTMENTS.FINANCIALCODE is
'The GL financial code of the manual adjustment line item. Used internally by AEMO systems.';

comment on column BILLINGFINANCIALADJUSTMENTS.BAS_CLASS is
'The BAS classification of the manual adjustment line item.';

alter table BILLINGFINANCIALADJUSTMENTS
   add constraint BILLINGFINANCIALADJUSTMENTS_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, ADJUSTMENTITEM)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGFINANCIALADJUSTMEN_LCX                         */
/*==============================================================*/
create index BILLINGFINANCIALADJUSTMEN_LCX on BILLINGFINANCIALADJUSTMENTS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGGENDATA                                        */
/*==============================================================*/
create table BILLINGGENDATA (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   CONNECTIONPOINTID    VARCHAR2(10)          not null,
   STATIONID            VARCHAR2(10),
   DUID                 VARCHAR2(10),
   AGGREGATEENERGY      NUMBER(16,6),
   SALES                NUMBER(16,6),
   PURCHASES            NUMBER(16,6),
   LASTCHANGED          DATE,
   PURCHASEDENERGY      NUMBER(16,6),
   MDA                  VARCHAR2(10)
)
   tablespace PSSTAB01;

comment on table BILLINGGENDATA is
'BILLINGGENDATA shows the total energy sold and purchased per participant transmission connection point for a billing period. The table is populated for participants using Data Model 5.2. For participants using Data Model 5.3 or higher, BILLING_ENERGY_TRANSACTIONS AND BILLING_ENERGY_GENSET_DETAIL replace BILLINGGENDATA.';

comment on column BILLINGGENDATA.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGGENDATA.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGGENDATA.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGGENDATA.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGGENDATA.CONNECTIONPOINTID is
'Connection point identifier';

comment on column BILLINGGENDATA.STATIONID is
'not populated';

comment on column BILLINGGENDATA.DUID is
'not populated';

comment on column BILLINGGENDATA.AGGREGATEENERGY is
'Aggregate energy sold, in MWh';

comment on column BILLINGGENDATA.SALES is
'$ income';

comment on column BILLINGGENDATA.PURCHASES is
'$ outgoing';

comment on column BILLINGGENDATA.LASTCHANGED is
'Last date and time record changed';

comment on column BILLINGGENDATA.PURCHASEDENERGY is
'Amount of energy purchased in MWh';

comment on column BILLINGGENDATA.MDA is
'Metering Data Agent supplying data';

alter table BILLINGGENDATA
   add constraint BILLINGGENDATA_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, CONNECTIONPOINTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGGENDATA_LCX                                    */
/*==============================================================*/
create index BILLINGGENDATA_LCX on BILLINGGENDATA (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGGENDATA_NDX2                                   */
/*==============================================================*/
create index BILLINGGENDATA_NDX2 on BILLINGGENDATA (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGINTERRESIDUES                                  */
/*==============================================================*/
create table BILLINGINTERRESIDUES (
   ALLOCATION           NUMBER(6,3),
   TOTALSURPLUS         NUMBER(15,5),
   INTERCONNECTORID     VARCHAR2(10)          not null,
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   SURPLUSVALUE         NUMBER(15,6),
   LASTCHANGED          DATE,
   REGIONID             VARCHAR2(10)          not null
)
   tablespace PSSTAB01;

comment on table BILLINGINTERRESIDUES is
'BILLINGINTERRESIDUES shows interregion residues payable to NSP.';

comment on column BILLINGINTERRESIDUES.ALLOCATION is
'May not be necessary';

comment on column BILLINGINTERRESIDUES.TOTALSURPLUS is
'May not be necessary';

comment on column BILLINGINTERRESIDUES.INTERCONNECTORID is
'Unique identifier for an interconnector which joins two regions.';

comment on column BILLINGINTERRESIDUES.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGINTERRESIDUES.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGINTERRESIDUES.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGINTERRESIDUES.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGINTERRESIDUES.SURPLUSVALUE is
'Amount NSP is paid for Inter-Regional Residues';

comment on column BILLINGINTERRESIDUES.LASTCHANGED is
'Last date and time record changed';

comment on column BILLINGINTERRESIDUES.REGIONID is
'Region ID';

alter table BILLINGINTERRESIDUES
   add constraint BILLINGINTERRESIDUES_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, INTERCONNECTORID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGINTERRESIDUES_LCX                              */
/*==============================================================*/
create index BILLINGINTERRESIDUES_LCX on BILLINGINTERRESIDUES (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGINTERVENTION                                   */
/*==============================================================*/
create table BILLINGINTERVENTION (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   MARKETINTERVENTION   NUMBER(15,5),
   TOTALINTERVENTION    NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLINGINTERVENTION is
'BILLINGINTERVENTION shows billing intervention recovery details.';

comment on column BILLINGINTERVENTION.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGINTERVENTION.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGINTERVENTION.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGINTERVENTION.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGINTERVENTION.MARKETINTERVENTION is
'Intervention Amounts paid to Generator for Market Recovery for region';

comment on column BILLINGINTERVENTION.TOTALINTERVENTION is
'Total Intervention Amounts paid to Generator';

comment on column BILLINGINTERVENTION.LASTCHANGED is
'Last date and time record changed';

alter table BILLINGINTERVENTION
   add constraint BILLINGINTERVENTION_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGINTERVENTION_LCX                               */
/*==============================================================*/
create index BILLINGINTERVENTION_LCX on BILLINGINTERVENTION (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGINTERVENTIONREGION                             */
/*==============================================================*/
create table BILLINGINTERVENTIONREGION (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   REGIONID             VARCHAR2(10)          not null,
   REGIONINTERVENTION   NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLINGINTERVENTIONREGION is
'BILLINGINTERVENTIONREGION shows recovery charges for region intervention.';

comment on column BILLINGINTERVENTIONREGION.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGINTERVENTIONREGION.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGINTERVENTIONREGION.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGINTERVENTIONREGION.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGINTERVENTIONREGION.REGIONID is
'Region ID';

comment on column BILLINGINTERVENTIONREGION.REGIONINTERVENTION is
'Recovery amount for that region';

comment on column BILLINGINTERVENTIONREGION.LASTCHANGED is
'Last changed date';

alter table BILLINGINTERVENTIONREGION
   add constraint BILLINGINTERVENTIONREGION_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGINTERVENTIONREGION_LCX                         */
/*==============================================================*/
create index BILLINGINTERVENTIONREGION_LCX on BILLINGINTERVENTIONREGION (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGINTRARESIDUES                                  */
/*==============================================================*/
create table BILLINGINTRARESIDUES (
   ALLOCATION           NUMBER(6,3),
   TOTALSURPLUS         NUMBER(15,5),
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   SURPLUSVALUE         NUMBER(15,6),
   LASTCHANGED          DATE,
   REGIONID             VARCHAR2(10)          not null
)
   tablespace PSSTAB01;

comment on table BILLINGINTRARESIDUES is
'BILLINGINTRARESIDUES shows intra-region settlement residue details for each Transmission Network Service Provider participant by region.';

comment on column BILLINGINTRARESIDUES.ALLOCATION is
'TNSP allocation';

comment on column BILLINGINTRARESIDUES.TOTALSURPLUS is
'Total $ residue amount for the region';

comment on column BILLINGINTRARESIDUES.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGINTRARESIDUES.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGINTRARESIDUES.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGINTRARESIDUES.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGINTRARESIDUES.SURPLUSVALUE is
'Amount TNSP is paid for Intra-Regional Residues';

comment on column BILLINGINTRARESIDUES.LASTCHANGED is
'Last changed date';

comment on column BILLINGINTRARESIDUES.REGIONID is
'Region ID';

alter table BILLINGINTRARESIDUES
   add constraint BILLINGINTRARESIDUES_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGINTRARESIDUES_LCX                              */
/*==============================================================*/
create index BILLINGINTRARESIDUES_LCX on BILLINGINTRARESIDUES (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGIRAUCSURPLUS                                   */
/*==============================================================*/
create table BILLINGIRAUCSURPLUS (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(2,0)           not null,
   RESIDUEYEAR          NUMBER(4,0),
   QUARTER              NUMBER(2,0),
   BILLRUNNO            NUMBER(3,0)           not null,
   CONTRACTID           VARCHAR2(30)          not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   FROMREGIONID         VARCHAR2(10)          not null,
   TOTALRESIDUES        NUMBER(15,5),
   ADJUSTMENT           NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLINGIRAUCSURPLUS is
'BILLINGIRAUCSURPLUS supports the Settlements Residue Auction, by showing the weekly billing Interconnector Residue (IR) payments as calculated for each bill run for Network Service Providers (NSPs) from the amount not auctioned.';

comment on column BILLINGIRAUCSURPLUS.CONTRACTYEAR is
'SRA Contracted Year (calendar year)';

comment on column BILLINGIRAUCSURPLUS.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGIRAUCSURPLUS.RESIDUEYEAR is
'Year of the Residue Contract; may differ from the calendar year at week 1.';

comment on column BILLINGIRAUCSURPLUS.QUARTER is
'Residue Contract Quarter';

comment on column BILLINGIRAUCSURPLUS.BILLRUNNO is
'The sequential number of a billing run';

comment on column BILLINGIRAUCSURPLUS.CONTRACTID is
'SRA Contract unique identifier';

comment on column BILLINGIRAUCSURPLUS.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGIRAUCSURPLUS.INTERCONNECTORID is
'Contracted Interconnector';

comment on column BILLINGIRAUCSURPLUS.FROMREGIONID is
'Nominated source region for Interconnector';

comment on column BILLINGIRAUCSURPLUS.TOTALRESIDUES is
'Total residues allocated to participant';

comment on column BILLINGIRAUCSURPLUS.ADJUSTMENT is
'Adjustment allocated to participant';

comment on column BILLINGIRAUCSURPLUS.LASTCHANGED is
'Date and time this record was last modified';

alter table BILLINGIRAUCSURPLUS
   add constraint BILLINGAUCSURPLUS_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, CONTRACTID, INTERCONNECTORID, FROMREGIONID, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGIRAUCSURPLUS_IDX_LC                            */
/*==============================================================*/
create index BILLINGIRAUCSURPLUS_IDX_LC on BILLINGIRAUCSURPLUS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGIRAUCSURPLUSSUM                                */
/*==============================================================*/
create table BILLINGIRAUCSURPLUSSUM (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   RESIDUEYEAR          NUMBER(4,0)           not null,
   QUARTER              NUMBER(2,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   FROMREGIONID         VARCHAR2(10)          not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   TOTALSURPLUS         NUMBER(15,5),
   AUCTIONFEES          NUMBER(15,5),
   ACTUALPAYMENT        NUMBER(15,5),
   AUCTIONFEES_GST      NUMBER(15,5),
   LASTCHANGED          DATE,
   CSP_DEROGATION_AMOUNT NUMBER(18,8),
   UNADJUSTED_IRSR      NUMBER(18,8),
   NEGATIVE_RESIDUES    NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLINGIRAUCSURPLUSSUM is
'BILLINGIRAUCSURPLUSSUM contains Auction fees and Settlements Residue Auction distribution that may arise from unpurchased auction units that accrue to Transmission Network Service Providers.';

comment on column BILLINGIRAUCSURPLUSSUM.CONTRACTYEAR is
'Contracted Year (calendar year)';

comment on column BILLINGIRAUCSURPLUSSUM.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGIRAUCSURPLUSSUM.RESIDUEYEAR is
'Year of the Residue Contract; may differ from the calendar year at week 1.';

comment on column BILLINGIRAUCSURPLUSSUM.QUARTER is
'Residue Contract Quarter';

comment on column BILLINGIRAUCSURPLUSSUM.BILLRUNNO is
'The sequential number of a billing run';

comment on column BILLINGIRAUCSURPLUSSUM.INTERCONNECTORID is
'Contracted Interconnector';

comment on column BILLINGIRAUCSURPLUSSUM.FROMREGIONID is
'Nominated source region for Interconnector';

comment on column BILLINGIRAUCSURPLUSSUM.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGIRAUCSURPLUSSUM.TOTALSURPLUS is
'Total residue amount allocated to participant';

comment on column BILLINGIRAUCSURPLUSSUM.AUCTIONFEES is
'Total auction fees payable in this week (negative amount). If AUCTIONFEES + AUCTIONFEES_GST >= TOTALSURPLUS then ACTUALPAYMENT is zero';

comment on column BILLINGIRAUCSURPLUSSUM.ACTUALPAYMENT is
'Net payment to participant, including auction fees';

comment on column BILLINGIRAUCSURPLUSSUM.AUCTIONFEES_GST is
'The GST amount on the auction fees, always being zero.';

comment on column BILLINGIRAUCSURPLUSSUM.LASTCHANGED is
'Date and time this record was last modified';

comment on column BILLINGIRAUCSURPLUSSUM.CSP_DEROGATION_AMOUNT is
'The CSP derogation amount applied as an adjustment to SRA.';

comment on column BILLINGIRAUCSURPLUSSUM.UNADJUSTED_IRSR is
'The SRA amount unadjusted by CSP.';

comment on column BILLINGIRAUCSURPLUSSUM.NEGATIVE_RESIDUES is
'Negative residues in the billing week for this participant in the SRA Year/Quarter';

alter table BILLINGIRAUCSURPLUSSUM
   add constraint BILLINGIRAUCSURPLUSSUM_PK primary key (CONTRACTYEAR, WEEKNO, RESIDUEYEAR, QUARTER, BILLRUNNO, INTERCONNECTORID, FROMREGIONID, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGIRAUCSURPSUM_LCX                               */
/*==============================================================*/
create index BILLINGIRAUCSURPSUM_LCX on BILLINGIRAUCSURPLUSSUM (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGIRFM                                           */
/*==============================================================*/
create table BILLINGIRFM (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   IRFMPAYMENT          NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLINGIRFM is
'BILLINGIRFM shows billing amounts associated with Industrial Relations Forced Majeure events for each participant.';

comment on column BILLINGIRFM.CONTRACTYEAR is
'Settlement Year';

comment on column BILLINGIRFM.WEEKNO is
'Week number starting 1 Jan each year.';

comment on column BILLINGIRFM.BILLRUNNO is
'Unique bill run';

comment on column BILLINGIRFM.PARTICIPANTID is
'Participant Identifier';

comment on column BILLINGIRFM.IRFMPAYMENT is
'Industrial Relations Forced Majeure payment for the billing period.';

comment on column BILLINGIRFM.LASTCHANGED is
'Last changed.';

alter table BILLINGIRFM
   add constraint BILLINGIRFM_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGIRFM_LCX                                       */
/*==============================================================*/
create index BILLINGIRFM_LCX on BILLINGIRFM (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGIRNSPSURPLUS                                   */
/*==============================================================*/
create table BILLINGIRNSPSURPLUS (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(2,0)           not null,
   RESIDUEYEAR          NUMBER(4,0),
   QUARTER              NUMBER(2,0),
   BILLRUNNO            NUMBER(3,0)           not null,
   CONTRACTID           VARCHAR2(30)          not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   FROMREGIONID         VARCHAR2(10)          not null,
   TOTALRESIDUES        NUMBER(15,5),
   ADJUSTMENT           NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLINGIRNSPSURPLUS is
'BILLINGIRNSPSURPLUS supports the Settlements Residue Auction (SRA), by showing the weekly billing Interconnector Residue (IR) payments as calculated for each bill run for Transmission Network Service Providers (TNSP) from the amount paid by participants (i.e. derogated amounts).';

comment on column BILLINGIRNSPSURPLUS.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGIRNSPSURPLUS.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGIRNSPSURPLUS.RESIDUEYEAR is
'Year of the Residue Contract; may differ from the calendar year at week 1.';

comment on column BILLINGIRNSPSURPLUS.QUARTER is
'Residue Contract Quarter';

comment on column BILLINGIRNSPSURPLUS.BILLRUNNO is
'The sequential number of a billing run';

comment on column BILLINGIRNSPSURPLUS.CONTRACTID is
'SRA Contract unique identifier';

comment on column BILLINGIRNSPSURPLUS.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGIRNSPSURPLUS.INTERCONNECTORID is
'Contracted Interconnector';

comment on column BILLINGIRNSPSURPLUS.FROMREGIONID is
'Nominated source region for Interconnector';

comment on column BILLINGIRNSPSURPLUS.TOTALRESIDUES is
'Total residues allocated to participant';

comment on column BILLINGIRNSPSURPLUS.ADJUSTMENT is
'Adjustment allocated to participant';

comment on column BILLINGIRNSPSURPLUS.LASTCHANGED is
'Date and time this record was last modified';

alter table BILLINGIRNSPSURPLUS
   add constraint BILLINGNSPSURPLUS_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, CONTRACTID, INTERCONNECTORID, FROMREGIONID, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGIRNSPSURPLUS_LCX                               */
/*==============================================================*/
create index BILLINGIRNSPSURPLUS_LCX on BILLINGIRNSPSURPLUS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGIRNSPSURPLUSSUM                                */
/*==============================================================*/
create table BILLINGIRNSPSURPLUSSUM (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   RESIDUEYEAR          NUMBER(4,0)           not null,
   QUARTER              NUMBER(2,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   FROMREGIONID         VARCHAR2(10)          not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   TOTALSURPLUS         NUMBER(15,5),
   AUCTIONFEES          NUMBER(15,5),
   AUCTIONFEES_GST      NUMBER(15,5),
   LASTCHANGED          DATE,
   CSP_DEROGATION_AMOUNT NUMBER(18,8),
   UNADJUSTED_IRSR      NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLINGIRNSPSURPLUSSUM is
'BILLINGIRNSPSURPLUSSUM contains derogated payments made to TNSPs arising from the Settlements Residue Auction process.';

comment on column BILLINGIRNSPSURPLUSSUM.CONTRACTYEAR is
'SRA Contracted Year (calendar year)';

comment on column BILLINGIRNSPSURPLUSSUM.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGIRNSPSURPLUSSUM.RESIDUEYEAR is
'Year of the Residue Contract; may differ from the calendar year at week 1.';

comment on column BILLINGIRNSPSURPLUSSUM.QUARTER is
'SRA Contracted Quarter';

comment on column BILLINGIRNSPSURPLUSSUM.BILLRUNNO is
'The sequential number of a billing run';

comment on column BILLINGIRNSPSURPLUSSUM.INTERCONNECTORID is
'Contracted Interconnector';

comment on column BILLINGIRNSPSURPLUSSUM.FROMREGIONID is
'Nominated source region for Interconnector';

comment on column BILLINGIRNSPSURPLUSSUM.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGIRNSPSURPLUSSUM.TOTALSURPLUS is
'Total residue amount allocated to participant';

comment on column BILLINGIRNSPSURPLUSSUM.AUCTIONFEES is
'This field is 0.';

comment on column BILLINGIRNSPSURPLUSSUM.AUCTIONFEES_GST is
'The GST amount on the auction fees, always being zero.';

comment on column BILLINGIRNSPSURPLUSSUM.LASTCHANGED is
'Date and time this record was last modified';

comment on column BILLINGIRNSPSURPLUSSUM.CSP_DEROGATION_AMOUNT is
'The CSP derogation amount applied as an adjustment to SRA.';

comment on column BILLINGIRNSPSURPLUSSUM.UNADJUSTED_IRSR is
'The SRA amount unadjusted by CSP.';

alter table BILLINGIRNSPSURPLUSSUM
   add constraint BILLINGIRNSPSURPLUSSUM_PK primary key (CONTRACTYEAR, WEEKNO, RESIDUEYEAR, QUARTER, BILLRUNNO, INTERCONNECTORID, FROMREGIONID, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGIRNSPSURPSUM_LCX                               */
/*==============================================================*/
create index BILLINGIRNSPSURPSUM_LCX on BILLINGIRNSPSURPLUSSUM (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGIRPARTSURPLUS                                  */
/*==============================================================*/
create table BILLINGIRPARTSURPLUS (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(2,0)           not null,
   RESIDUEYEAR          NUMBER(4,0),
   QUARTER              NUMBER(2,0),
   BILLRUNNO            NUMBER(3,0)           not null,
   CONTRACTID           VARCHAR2(30)          not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   FROMREGIONID         VARCHAR2(10)          not null,
   TOTALRESIDUES        NUMBER(15,5),
   ADJUSTMENT           NUMBER(15,5),
   LASTCHANGED          DATE,
   ACTUALPAYMENT        NUMBER(15,5)
)
   tablespace PSSTAB01;

comment on table BILLINGIRPARTSURPLUS is
'BILLINGIRPARTSURPLUS supports the Settlements Residue Auction, by showing the weekly billing SRA distribution to Auction participants by Contract Identifier.';

comment on column BILLINGIRPARTSURPLUS.CONTRACTYEAR is
'SRA Contracted Year (calendar year)';

comment on column BILLINGIRPARTSURPLUS.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGIRPARTSURPLUS.RESIDUEYEAR is
'Year of the Residue Contract; may differ from the calendar year at week 1.';

comment on column BILLINGIRPARTSURPLUS.QUARTER is
'Residue Contract Quarter';

comment on column BILLINGIRPARTSURPLUS.BILLRUNNO is
'The sequential number of a billing run';

comment on column BILLINGIRPARTSURPLUS.CONTRACTID is
'SRA Contract unique identifier';

comment on column BILLINGIRPARTSURPLUS.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGIRPARTSURPLUS.INTERCONNECTORID is
'Contracted Interconnector';

comment on column BILLINGIRPARTSURPLUS.FROMREGIONID is
'Nominated source region for Interconnector';

comment on column BILLINGIRPARTSURPLUS.TOTALRESIDUES is
'Total residues allocated to participant';

comment on column BILLINGIRPARTSURPLUS.ADJUSTMENT is
'Adjustment allocated to participant';

comment on column BILLINGIRPARTSURPLUS.LASTCHANGED is
'Date and time this record was last modified';

comment on column BILLINGIRPARTSURPLUS.ACTUALPAYMENT is
'Net actual payment to participant, including auction fees';

alter table BILLINGIRPARTSURPLUS
   add constraint BILLINGPARTSURPLUS_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, CONTRACTID, INTERCONNECTORID, FROMREGIONID, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGIRPARTSURPLUS_LCX                              */
/*==============================================================*/
create index BILLINGIRPARTSURPLUS_LCX on BILLINGIRPARTSURPLUS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGIRPARTSURPLUSSUM                               */
/*==============================================================*/
create table BILLINGIRPARTSURPLUSSUM (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   RESIDUEYEAR          NUMBER(4,0)           not null,
   QUARTER              NUMBER(2,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   FROMREGIONID         VARCHAR2(10)          not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   TOTALSURPLUS         NUMBER(15,5),
   AUCTIONFEES          NUMBER(15,5),
   ACTUALPAYMENT        NUMBER(15,5),
   AUCTIONFEES_GST      NUMBER(15,5),
   LASTCHANGED          DATE,
   CSP_DEROGATION_AMOUNT NUMBER(18,8),
   UNADJUSTED_IRSR      NUMBER(18,8),
   AUCTIONFEES_TOTALGROSS_ADJ Number(18,8)
)
   tablespace PSSTAB01;

comment on table BILLINGIRPARTSURPLUSSUM is
'BILLINGIRPARTSURPLUSSUM supports the Settlements Residue Auction, by showing the weekly billing SRA distribution and associated fees to Auction participants.';

comment on column BILLINGIRPARTSURPLUSSUM.CONTRACTYEAR is
'SRA Contracted Year (calendar year)';

comment on column BILLINGIRPARTSURPLUSSUM.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGIRPARTSURPLUSSUM.RESIDUEYEAR is
'Year of the Residue Contract; may differ from the calendar year at week 1.';

comment on column BILLINGIRPARTSURPLUSSUM.QUARTER is
'Residue Contract Quarter';

comment on column BILLINGIRPARTSURPLUSSUM.BILLRUNNO is
'The sequential number of a billing run';

comment on column BILLINGIRPARTSURPLUSSUM.INTERCONNECTORID is
'Contracted Interconnector';

comment on column BILLINGIRPARTSURPLUSSUM.FROMREGIONID is
'Nominated source region for Interconnector';

comment on column BILLINGIRPARTSURPLUSSUM.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGIRPARTSURPLUSSUM.TOTALSURPLUS is
'Total residue amount allocated to participant';

comment on column BILLINGIRPARTSURPLUSSUM.AUCTIONFEES is
'Total auction fees payable in this week (negative amount). If AUCTIONFEES + AUCTIONFEES_GST >= TOTALSURPLUS then ACTUALPAYMENT is zero.';

comment on column BILLINGIRPARTSURPLUSSUM.ACTUALPAYMENT is
'Net payment to participant, including auction fees';

comment on column BILLINGIRPARTSURPLUSSUM.AUCTIONFEES_GST is
'The GST amount on the auction fees, always being zero.';

comment on column BILLINGIRPARTSURPLUSSUM.LASTCHANGED is
'Date and time this record was last modified';

comment on column BILLINGIRPARTSURPLUSSUM.CSP_DEROGATION_AMOUNT is
'The CSP derogation amount applied as an adjustment to SRA.';

comment on column BILLINGIRPARTSURPLUSSUM.UNADJUSTED_IRSR is
'The SRA amount unadjusted by CSP.';

comment on column BILLINGIRPARTSURPLUSSUM.AUCTIONFEES_TOTALGROSS_ADJ is
'The adjusted total Auction fees for the Directional Interconnector. Calculated as the amount of the total fees due from the SRA Auction Participant, pro-rated based on the total surplus for each Directional Interconnector the SRA Auction Participant contracted.';

alter table BILLINGIRPARTSURPLUSSUM
   add constraint BILLINGIRPARTSURPLUSSUM_PK primary key (CONTRACTYEAR, WEEKNO, RESIDUEYEAR, QUARTER, BILLRUNNO, INTERCONNECTORID, FROMREGIONID, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGIRPARTSURPSUM_LCX                              */
/*==============================================================*/
create index BILLINGIRPARTSURPSUM_LCX on BILLINGIRPARTSURPLUSSUM (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGIRPARTSURPLUSSUM_I01                           */
/*==============================================================*/
create index BILLINGIRPARTSURPLUSSUM_I01 on BILLINGIRPARTSURPLUSSUM (
   RESIDUEYEAR ASC,
   QUARTER ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGPRIORADJUSTMENTS                               */
/*==============================================================*/
create table BILLINGPRIORADJUSTMENTS (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   ADJCONTRACTYEAR      NUMBER(4,0)           not null,
   ADJWEEKNO            NUMBER(3,0)           not null,
   ADJBILLRUNNO         NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   PREVAMOUNT           NUMBER(15,5),
   ADJAMOUNT            NUMBER(15,5),
   IRN                  NUMBER(15,5),
   IRP                  NUMBER(15,5),
   INTERESTAMOUNT       NUMBER(15,5),
   LASTCHANGED          DATE,
   IRSR_PREVAMOUNT      NUMBER(15,5),
   IRSR_ADJAMOUNT       NUMBER(15,5),
   IRSR_INTERESTAMOUNT  NUMBER(15,5)
)
   tablespace PSSTAB01;

comment on table BILLINGPRIORADJUSTMENTS is
'BILLINGPRIORADJUSTMENTS sets out prior period adjustments and associated interest inserted in subsequent Final Statements arising from Revision Statement postings.';

comment on column BILLINGPRIORADJUSTMENTS.CONTRACTYEAR is
'Settlement year.';

comment on column BILLINGPRIORADJUSTMENTS.WEEKNO is
'Settlement week number.';

comment on column BILLINGPRIORADJUSTMENTS.BILLRUNNO is
'Billing run number.';

comment on column BILLINGPRIORADJUSTMENTS.ADJCONTRACTYEAR is
'ContractYear of the posted revision statement inserted to the Final Statement';

comment on column BILLINGPRIORADJUSTMENTS.ADJWEEKNO is
'WeekNo of the posted revision statement inserted to the Final Statement';

comment on column BILLINGPRIORADJUSTMENTS.ADJBILLRUNNO is
'Bill run number of the posted revision statement inserted to the Final Statement';

comment on column BILLINGPRIORADJUSTMENTS.PARTICIPANTID is
'Participant ID';

comment on column BILLINGPRIORADJUSTMENTS.PREVAMOUNT is
'Statement total of the previous posted revision statement inserted to the Final Statement.';

comment on column BILLINGPRIORADJUSTMENTS.ADJAMOUNT is
'Adjusted amount.';

comment on column BILLINGPRIORADJUSTMENTS.IRN is
'Interest rate applied to the revision adjustment';

comment on column BILLINGPRIORADJUSTMENTS.IRP is
'unused; always null';

comment on column BILLINGPRIORADJUSTMENTS.INTERESTAMOUNT is
'Interest amount.';

comment on column BILLINGPRIORADJUSTMENTS.LASTCHANGED is
'Last changed.';

comment on column BILLINGPRIORADJUSTMENTS.IRSR_PREVAMOUNT is
'unused; always null';

comment on column BILLINGPRIORADJUSTMENTS.IRSR_ADJAMOUNT is
'unused; always null';

comment on column BILLINGPRIORADJUSTMENTS.IRSR_INTERESTAMOUNT is
'unused; always null';

alter table BILLINGPRIORADJUSTMENTS
   add constraint BILLINGPRIORADJUSTMENTS_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, ADJCONTRACTYEAR, ADJWEEKNO, ADJBILLRUNNO, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGPRIORADJUSTMENTS_NDX2                          */
/*==============================================================*/
create index BILLINGPRIORADJUSTMENTS_NDX2 on BILLINGPRIORADJUSTMENTS (
   PARTICIPANTID ASC,
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGPRIORADJMNTS_NDX_LCHD                          */
/*==============================================================*/
create index BILLINGPRIORADJMNTS_NDX_LCHD on BILLINGPRIORADJUSTMENTS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGREALLOC                                        */
/*==============================================================*/
create table BILLINGREALLOC (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   COUNTERPARTY         VARCHAR2(10)          not null,
   VALUE                NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLINGREALLOC is
'BILLINGREALLOC shows reallocation contract values in each billing run, where participants have used reallocations.';

comment on column BILLINGREALLOC.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGREALLOC.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGREALLOC.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGREALLOC.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGREALLOC.COUNTERPARTY is
'Participant who is the counter party to this contract';

comment on column BILLINGREALLOC.VALUE is
'Value billed on this contract';

comment on column BILLINGREALLOC.LASTCHANGED is
'Last date and time record changed';

alter table BILLINGREALLOC
   add constraint BILLINGREALLOC_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, COUNTERPARTY)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGREALLOC_NDX2                                   */
/*==============================================================*/
create index BILLINGREALLOC_NDX2 on BILLINGREALLOC (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGREALLOC_LCX                                    */
/*==============================================================*/
create index BILLINGREALLOC_LCX on BILLINGREALLOC (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGREALLOC_DETAIL                                 */
/*==============================================================*/
create table BILLINGREALLOC_DETAIL (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   COUNTERPARTY         VARCHAR2(10)          not null,
   REALLOCATIONID       VARCHAR2(20)          not null,
   VALUE                NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLINGREALLOC_DETAIL is
'Billing Reallocation Data aggregated by REALLOCATIONID for each billing run over the billing week.';

comment on column BILLINGREALLOC_DETAIL.CONTRACTYEAR is
'BILLING CONTRACTYEAR';

comment on column BILLINGREALLOC_DETAIL.WEEKNO is
'BILLING WEEKNO';

comment on column BILLINGREALLOC_DETAIL.BILLRUNNO is
'BILLING RUN NO';

comment on column BILLINGREALLOC_DETAIL.PARTICIPANTID is
'REALLOCATION PARTICIPANTID';

comment on column BILLINGREALLOC_DETAIL.COUNTERPARTY is
'REALLOCATION COUNTERPARTY PARTICIPANTID';

comment on column BILLINGREALLOC_DETAIL.REALLOCATIONID is
'REALLOCATIONID';

comment on column BILLINGREALLOC_DETAIL.VALUE is
'REALLOCATION VALUE';

comment on column BILLINGREALLOC_DETAIL.LASTCHANGED is
'DATETIME WHEN RECORD SAVED';

alter table BILLINGREALLOC_DETAIL
   add constraint BILLINGREALLOC_DETAIL_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, COUNTERPARTY, REALLOCATIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGREALLOC_DETAIL_LCX                             */
/*==============================================================*/
create index BILLINGREALLOC_DETAIL_LCX on BILLINGREALLOC_DETAIL (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGREGIONEXPORTS                                  */
/*==============================================================*/
create table BILLINGREGIONEXPORTS (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   REGIONID             VARCHAR2(10)          not null,
   EXPORTTO             VARCHAR2(10)          not null,
   ENERGY               NUMBER(16,6),
   VALUE                NUMBER(15,5),
   SURPLUSENERGY        NUMBER(16,6),
   SURPLUSVALUE         NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLINGREGIONEXPORTS is
'BILLINGREGIONEXPORTS sets out the region summary table of overall energy exported to and from each region for each billing run.';

comment on column BILLINGREGIONEXPORTS.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGREGIONEXPORTS.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGREGIONEXPORTS.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGREGIONEXPORTS.REGIONID is
'Unique region identifier';

comment on column BILLINGREGIONEXPORTS.EXPORTTO is
'Region exported to';

comment on column BILLINGREGIONEXPORTS.ENERGY is
'MWh Energy value exported';

comment on column BILLINGREGIONEXPORTS.VALUE is
'$ Value of energy exported';

comment on column BILLINGREGIONEXPORTS.SURPLUSENERGY is
'This field is populated with 0';

comment on column BILLINGREGIONEXPORTS.SURPLUSVALUE is
'$ Interregional residue';

comment on column BILLINGREGIONEXPORTS.LASTCHANGED is
'Last date and time record changed';

alter table BILLINGREGIONEXPORTS
   add constraint BILLINGREGIONEXPORTS_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, REGIONID, EXPORTTO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGREGIONEXPORTS_LCX                              */
/*==============================================================*/
create index BILLINGREGIONEXPORTS_LCX on BILLINGREGIONEXPORTS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGREGIONFIGURES                                  */
/*==============================================================*/
create table BILLINGREGIONFIGURES (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   REGIONID             VARCHAR2(10)          not null,
   ENERGYOUT            NUMBER(16,6),
   VALUEOUT             NUMBER(16,6),
   ENERGYPURCHASED      NUMBER(16,6),
   VALUEPURCHASED       NUMBER(16,6),
   EXCESSGEN            NUMBER(16,6),
   RESERVETRADING       NUMBER(16,6),
   INTCOMPO             NUMBER(16,6),
   ADMINPRICECOMPO      NUMBER(16,6),
   SETTSURPLUS          NUMBER(16,6),
   ASPAYMENT            NUMBER(16,6),
   POOLFEES             NUMBER(16,6),
   LASTCHANGED          DATE,
   WDRSQ                NUMBER(18,8),
   WDRTA                NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLINGREGIONFIGURES is
'BILLINGREGIONFIGURES sets out additional summary region details including ancillary service amounts for each billing run.';

comment on column BILLINGREGIONFIGURES.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGREGIONFIGURES.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGREGIONFIGURES.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGREGIONFIGURES.REGIONID is
'Unique region identifier';

comment on column BILLINGREGIONFIGURES.ENERGYOUT is
'MWh Energy output in the region during the billing period';

comment on column BILLINGREGIONFIGURES.VALUEOUT is
'$ Value of energy output in region during billing period';

comment on column BILLINGREGIONFIGURES.ENERGYPURCHASED is
'MWh Amount of energy purchased in region during billing period';

comment on column BILLINGREGIONFIGURES.VALUEPURCHASED is
'$ Value of energy purchased during billing period';

comment on column BILLINGREGIONFIGURES.EXCESSGEN is
'This field is populated with 0';

comment on column BILLINGREGIONFIGURES.RESERVETRADING is
'This field is populated with 0';

comment on column BILLINGREGIONFIGURES.INTCOMPO is
'This field is populated with 0';

comment on column BILLINGREGIONFIGURES.ADMINPRICECOMPO is
'This field is populated with 0';

comment on column BILLINGREGIONFIGURES.SETTSURPLUS is
'Intraregional residues in $';

comment on column BILLINGREGIONFIGURES.ASPAYMENT is
'Ancillary service payments in $';

comment on column BILLINGREGIONFIGURES.POOLFEES is
'This field is populated with 0';

comment on column BILLINGREGIONFIGURES.LASTCHANGED is
'Last date and time record changed';

comment on column BILLINGREGIONFIGURES.WDRSQ is
'WDR Settlement Quantity Capped in MWh';

comment on column BILLINGREGIONFIGURES.WDRTA is
'WDR transaction amount in $';

alter table BILLINGREGIONFIGURES
   add constraint BILLINGREGIONFIGURES_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGREGIONFIGURES_LCX                              */
/*==============================================================*/
create index BILLINGREGIONFIGURES_LCX on BILLINGREGIONFIGURES (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGREGIONIMPORTS                                  */
/*==============================================================*/
create table BILLINGREGIONIMPORTS (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   REGIONID             VARCHAR2(10)          not null,
   IMPORTFROM           VARCHAR2(10)          not null,
   ENERGY               NUMBER(16,6),
   VALUE                NUMBER(15,5),
   SURPLUSENERGY        NUMBER(16,6),
   SURPLUSVALUE         NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLINGREGIONIMPORTS is
'BILLINGREGIONIMPORTS sets out the region summary table of overall energy imported to and from each region for each billing run.';

comment on column BILLINGREGIONIMPORTS.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGREGIONIMPORTS.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGREGIONIMPORTS.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGREGIONIMPORTS.REGIONID is
'Unique region identifier';

comment on column BILLINGREGIONIMPORTS.IMPORTFROM is
'Region energy imported from';

comment on column BILLINGREGIONIMPORTS.ENERGY is
'Amount of energy imported';

comment on column BILLINGREGIONIMPORTS.VALUE is
'Value of energy imported';

comment on column BILLINGREGIONIMPORTS.SURPLUSENERGY is
'Populated with 0';

comment on column BILLINGREGIONIMPORTS.SURPLUSVALUE is
'Interregional residue';

comment on column BILLINGREGIONIMPORTS.LASTCHANGED is
'Last date and time record changed';

alter table BILLINGREGIONIMPORTS
   add constraint BILLINGREGIONIMPORTS_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, REGIONID, IMPORTFROM)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGREGIONIMPORTS_LCX                              */
/*==============================================================*/
create index BILLINGREGIONIMPORTS_LCX on BILLINGREGIONIMPORTS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGRESERVERECOVERY                                */
/*==============================================================*/
create table BILLINGRESERVERECOVERY (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   MARKETRESERVE        NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLINGRESERVERECOVERY is
'BILLINGRESERVERECOVERY shows Market Reserve recovery details for each participant in a bill run.';

comment on column BILLINGRESERVERECOVERY.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGRESERVERECOVERY.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGRESERVERECOVERY.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGRESERVERECOVERY.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGRESERVERECOVERY.MARKETRESERVE is
'Amount Retailer pays for Reserve Trader Contracts with Market Recovery';

comment on column BILLINGRESERVERECOVERY.LASTCHANGED is
'Last date and time record changed';

alter table BILLINGRESERVERECOVERY
   add constraint BILLRESERVERECOVERY_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGRESERVERECOVERY_LCX                            */
/*==============================================================*/
create index BILLINGRESERVERECOVERY_LCX on BILLINGRESERVERECOVERY (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGRESERVEREGIONRECOVERY                          */
/*==============================================================*/
create table BILLINGRESERVEREGIONRECOVERY (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   REGIONID             VARCHAR2(10)          not null,
   REGIONRESERVE        NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLINGRESERVEREGIONRECOVERY is
'BILLINGRESERVEREGIONRECOVERY shows Billing Region Reserve region recovery details for each participant (by region).';

comment on column BILLINGRESERVEREGIONRECOVERY.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGRESERVEREGIONRECOVERY.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGRESERVEREGIONRECOVERY.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGRESERVEREGIONRECOVERY.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGRESERVEREGIONRECOVERY.REGIONID is
'Region Identifier for region recovery.';

comment on column BILLINGRESERVEREGIONRECOVERY.REGIONRESERVE is
'Amount Retailer pays for Reserve Trader Contracts with Region Recovery';

alter table BILLINGRESERVEREGIONRECOVERY
   add constraint BILLRESERVEREGIONRECOVERY_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGRESERVEREGIONRECOV_LCX                         */
/*==============================================================*/
create index BILLINGRESERVEREGIONRECOV_LCX on BILLINGRESERVEREGIONRECOVERY (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGRESERVETRADER                                  */
/*==============================================================*/
create table BILLINGRESERVETRADER (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   MARKETRESERVE        NUMBER(15,5),
   TOTALRESERVE         NUMBER(15,5),
   LASTCHANGED          DATE,
   TOTALCAPDIFFERENCE   NUMBER(15,5)
)
   tablespace PSSTAB01;

comment on table BILLINGRESERVETRADER is
'BILLINGRESERVETRADER shows Billing Market Reserve TRADER payment details to Generators.';

comment on column BILLINGRESERVETRADER.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGRESERVETRADER.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGRESERVETRADER.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGRESERVETRADER.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGRESERVETRADER.MARKETRESERVE is
'Reserve Trader Amounts paid to Generator for Market Recovery';

comment on column BILLINGRESERVETRADER.TOTALRESERVE is
'Total Reserve Trader Amounts paid to Generator';

comment on column BILLINGRESERVETRADER.LASTCHANGED is
'Last date and time record changed';

alter table BILLINGRESERVETRADER
   add constraint BILLINGRESERVETRADER_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGRESERVETRADER_LCX                              */
/*==============================================================*/
create index BILLINGRESERVETRADER_LCX on BILLINGRESERVETRADER (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGRESERVETRADERREGION                            */
/*==============================================================*/
create table BILLINGRESERVETRADERREGION (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   REGIONID             VARCHAR2(10)          not null,
   REGIONRESERVE        NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLINGRESERVETRADERREGION is
'BILLINGRESERVETRADERREGION shows Billing Region Reserve Trader payment details.';

comment on column BILLINGRESERVETRADERREGION.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGRESERVETRADERREGION.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGRESERVETRADERREGION.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGRESERVETRADERREGION.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGRESERVETRADERREGION.REGIONID is
'Region Identifier';

comment on column BILLINGRESERVETRADERREGION.REGIONRESERVE is
'Reserve Trader Amounts paid to Generator for Region Recovery';

comment on column BILLINGRESERVETRADERREGION.LASTCHANGED is
'Last date and time record changed';

alter table BILLINGRESERVETRADERREGION
   add constraint BILLINGRESERVETRADERREGION_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGRESERVETRADERREGIO_LCX                         */
/*==============================================================*/
create index BILLINGRESERVETRADERREGIO_LCX on BILLINGRESERVETRADERREGION (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGRUNTRK                                         */
/*==============================================================*/
create table BILLINGRUNTRK (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   STATUS               VARCHAR2(6),
   ADJ_CLEARED          VARCHAR2(1),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(10),
   POSTDATE             DATE,
   POSTBY               VARCHAR2(10),
   LASTCHANGED          DATE,
   RECEIPTPOSTDATE      DATE,
   RECEIPTPOSTBY        VARCHAR2(10),
   PAYMENTPOSTDATE      DATE,
   PAYMENTPOSTBY        VARCHAR2(10),
   SHORTFALL            NUMBER(16,6),
   MAKEUP               NUMBER(15,5)
)
   tablespace PSSTAB01;

comment on table BILLINGRUNTRK is
'BILLINGRUNTRK identifies the Statement type (i.e. Status of PRELIM, FINAL, REVISE) and date of the BillRunNo posted, per WeekNo. This provides a further extension of tracking data from the BILLINGDAYTRK table.';

comment on column BILLINGRUNTRK.CONTRACTYEAR is
'Year of the run';

comment on column BILLINGRUNTRK.WEEKNO is
'Week number of the run';

comment on column BILLINGRUNTRK.BILLRUNNO is
'Sequential run number';

comment on column BILLINGRUNTRK.STATUS is
'The billing run type, PRELIM, FINAL, REVISE or INTERIM';

comment on column BILLINGRUNTRK.ADJ_CLEARED is
'Flag';

comment on column BILLINGRUNTRK.AUTHORISEDDATE is
'null, since not used';

comment on column BILLINGRUNTRK.AUTHORISEDBY is
'null, since not used';

comment on column BILLINGRUNTRK.POSTDATE is
'When the results were posted';

comment on column BILLINGRUNTRK.POSTBY is
'Who posted the results';

comment on column BILLINGRUNTRK.LASTCHANGED is
'Last date and time record changed';

comment on column BILLINGRUNTRK.RECEIPTPOSTDATE is
'null, since not used';

comment on column BILLINGRUNTRK.RECEIPTPOSTBY is
'null, since not used';

comment on column BILLINGRUNTRK.PAYMENTPOSTDATE is
'When the payment was posted';

comment on column BILLINGRUNTRK.PAYMENTPOSTBY is
'Who posted the payment';

comment on column BILLINGRUNTRK.SHORTFALL is
'Payment shortfall amount';

comment on column BILLINGRUNTRK.MAKEUP is
'Not Used';

alter table BILLINGRUNTRK
   add constraint BILLINGRUNTRK_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGRUNTRK_LCX                                     */
/*==============================================================*/
create index BILLINGRUNTRK_LCX on BILLINGRUNTRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINGSMELTERREDUCTION                               */
/*==============================================================*/
create table BILLINGSMELTERREDUCTION (
   CONTRACTYEAR         NUMBER(22,0)          not null,
   WEEKNO               NUMBER(22,0)          not null,
   BILLRUNNO            NUMBER(22,0)          not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   RATE1                NUMBER(15,6),
   RA1                  NUMBER(15,6),
   RATE2                NUMBER(15,6),
   RA2                  NUMBER(15,6),
   TE                   NUMBER(15,6),
   PCSD                 NUMBER(15,6),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLINGSMELTERREDUCTION is
'BILLINGSMELTERREDUCTION shows the smelter reduction payment (only applies to  participants with Victorian customer connection points).';

comment on column BILLINGSMELTERREDUCTION.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINGSMELTERREDUCTION.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINGSMELTERREDUCTION.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINGSMELTERREDUCTION.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINGSMELTERREDUCTION.RATE1 is
'Rate in $/MWh';

comment on column BILLINGSMELTERREDUCTION.RA1 is
'Payment';

comment on column BILLINGSMELTERREDUCTION.RATE2 is
'Rate in $/MWh';

comment on column BILLINGSMELTERREDUCTION.RA2 is
'Payment';

comment on column BILLINGSMELTERREDUCTION.TE is
'Tabulated Energy';

comment on column BILLINGSMELTERREDUCTION.PCSD is
'Victorian Demand as defined by Code Chapter 9 definitions';

comment on column BILLINGSMELTERREDUCTION.LASTCHANGED is
'Last date and time record changed';

alter table BILLINGSMELTERREDUCTION
   add constraint BILLINGSMELTERREDUCTION_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGSMELTERREDUCT_NDX2                             */
/*==============================================================*/
create index BILLINGSMELTERREDUCT_NDX2 on BILLINGSMELTERREDUCTION (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINGSMELTERREDUCTION_LCX                           */
/*==============================================================*/
create index BILLINGSMELTERREDUCTION_LCX on BILLINGSMELTERREDUCTION (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_APC_COMPENSATION                              */
/*==============================================================*/
create table BILLING_APC_COMPENSATION (
   CONTRACTYEAR         NUMBER(4)             not null,
   WEEKNO               NUMBER(3)             not null,
   BILLRUNNO            NUMBER(3)             not null,
   APEVENTID            NUMBER(6)             not null,
   CLAIMID              NUMBER(6)             not null,
   PARTICIPANTID        VARCHAR2(20),
   COMPENSATION_AMOUNT  NUMBER(18,8),
   EVENT_TYPE           VARCHAR2(20),
   COMPENSATION_TYPE    VARCHAR2(20),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLING_APC_COMPENSATION is
'Billing result table for APC compensation payments.';

comment on column BILLING_APC_COMPENSATION.CONTRACTYEAR is
'Billing contract year';

comment on column BILLING_APC_COMPENSATION.WEEKNO is
'Billing week number';

comment on column BILLING_APC_COMPENSATION.BILLRUNNO is
'Billing run number';

comment on column BILLING_APC_COMPENSATION.APEVENTID is
'AP Event Id';

comment on column BILLING_APC_COMPENSATION.CLAIMID is
'AP Event Claim Id';

comment on column BILLING_APC_COMPENSATION.PARTICIPANTID is
'Participant identifier';

comment on column BILLING_APC_COMPENSATION.COMPENSATION_AMOUNT is
'Payment amount to the participant';

comment on column BILLING_APC_COMPENSATION.EVENT_TYPE is
'The Administered Price Event Type. Valid values: ENERGY, FCAS, BOTH';

comment on column BILLING_APC_COMPENSATION.COMPENSATION_TYPE is
'The Type of Administered Price Compensation Claim. Valid values: DIRECT_COST, OTHER_COST';

comment on column BILLING_APC_COMPENSATION.LASTCHANGED is
'The date and time of last changed record';

alter table BILLING_APC_COMPENSATION
   add constraint BILLING_APC_COMPENSATION_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, APEVENTID, CLAIMID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_APC_RECOVERY                                  */
/*==============================================================*/
create table BILLING_APC_RECOVERY (
   CONTRACTYEAR         NUMBER(4)             not null,
   WEEKNO               NUMBER(3)             not null,
   BILLRUNNO            NUMBER(3)             not null,
   APEVENTID            NUMBER(6)             not null,
   CLAIMID              NUMBER(6)             not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   RECOVERY_AMOUNT      NUMBER(18,8),
   ELIGIBILITY_START_INTERVAL DATE,
   ELIGIBILITY_END_INTERVAL DATE,
   PARTICIPANT_DEMAND   NUMBER(18,8),
   REGION_DEMAND        NUMBER(18,8),
   LASTCHANGED          DATE,
   PARTICIPANT_ACE_MWH  NUMBER(18,8),
   REGION_ACE_MWH       NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLING_APC_RECOVERY is
'Billing result table for recovery of APC compensation payments';

comment on column BILLING_APC_RECOVERY.CONTRACTYEAR is
'Billing contract year';

comment on column BILLING_APC_RECOVERY.WEEKNO is
'Billing week number';

comment on column BILLING_APC_RECOVERY.BILLRUNNO is
'Billing run number';

comment on column BILLING_APC_RECOVERY.APEVENTID is
'AP Event Id';

comment on column BILLING_APC_RECOVERY.CLAIMID is
'AP Event Claim Id';

comment on column BILLING_APC_RECOVERY.PARTICIPANTID is
'Participant identifier';

comment on column BILLING_APC_RECOVERY.REGIONID is
'Region Identifier';

comment on column BILLING_APC_RECOVERY.RECOVERY_AMOUNT is
'Recovery amount attributable to the participant in that region';

comment on column BILLING_APC_RECOVERY.ELIGIBILITY_START_INTERVAL is
'The starting half hourly interval for the eligibility period for recovery of APC Payment';

comment on column BILLING_APC_RECOVERY.ELIGIBILITY_END_INTERVAL is
'The ending half hourly interval for the eligibility period for recovery of APC Payment';

comment on column BILLING_APC_RECOVERY.PARTICIPANT_DEMAND is
'The participant demand in the cost recovery region';

comment on column BILLING_APC_RECOVERY.REGION_DEMAND is
'The sum of demand of all participants in the cost recovery region (Region Sum)';

comment on column BILLING_APC_RECOVERY.LASTCHANGED is
'The date and time of last changed record';

comment on column BILLING_APC_RECOVERY.PARTICIPANT_ACE_MWH is
'The ACE MWh value of the participant from the Eligibility Interval used for the APC Recovery Calculation. If the Billing Week is prior to the IESS rule effective date, then value is Null.';

comment on column BILLING_APC_RECOVERY.REGION_ACE_MWH is
'The ACE MWh value of the Region from the Eligibility Interval used for the APC Recovery Calculation. This is the sum of the ACE MWh of all the participants in that recovery. If the Billing Week is prior to the IESS rule effective date, then value is Null.';

alter table BILLING_APC_RECOVERY
   add constraint BILLING_APC_RECOVERY_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, APEVENTID, CLAIMID, PARTICIPANTID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_CO2E_PUBLICATION                              */
/*==============================================================*/
create table BILLING_CO2E_PUBLICATION (
   CONTRACTYEAR         NUMBER(4)             not null,
   WEEKNO               NUMBER(3)             not null,
   BILLRUNNO            NUMBER(3)             not null,
   SETTLEMENTDATE       DATE                  not null,
   REGIONID             VARCHAR(20)           not null,
   SENTOUTENERGY        NUMBER(18,8),
   GENERATOREMISSIONS   NUMBER(18,8),
   INTENSITYINDEX       NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLING_CO2E_PUBLICATION is
'Carbon Dioxide Intensity Index publication table';

comment on column BILLING_CO2E_PUBLICATION.CONTRACTYEAR is
'Billing contract year';

comment on column BILLING_CO2E_PUBLICATION.WEEKNO is
'Billing week no';

comment on column BILLING_CO2E_PUBLICATION.BILLRUNNO is
'Billing run no';

comment on column BILLING_CO2E_PUBLICATION.SETTLEMENTDATE is
'Settlement Date (Calendar)';

comment on column BILLING_CO2E_PUBLICATION.REGIONID is
'Region identifier';

comment on column BILLING_CO2E_PUBLICATION.SENTOUTENERGY is
'Total sent out energy for region (MWh)';

comment on column BILLING_CO2E_PUBLICATION.GENERATOREMISSIONS is
'Total generator emissions for region (Co2-e)';

comment on column BILLING_CO2E_PUBLICATION.INTENSITYINDEX is
'Carbon Dioxide Intensity index for region (CO2-e/MWh)';

alter table BILLING_CO2E_PUBLICATION
   add constraint BILLING_CO2E_PUBLICATION_PK primary key (CONTRACTYEAR, WEEKNO, SETTLEMENTDATE, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_CO2E_PUBLICATION_TRK                          */
/*==============================================================*/
create table BILLING_CO2E_PUBLICATION_TRK (
   CONTRACTYEAR         NUMBER(4)             not null,
   WEEKNO               NUMBER(3)             not null,
   BILLRUNNO            NUMBER(3),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLING_CO2E_PUBLICATION_TRK is
'Carbon Dioxide Intensity Index publication tracking table';

comment on column BILLING_CO2E_PUBLICATION_TRK.CONTRACTYEAR is
'Billing contract year';

comment on column BILLING_CO2E_PUBLICATION_TRK.WEEKNO is
'Billing week no';

comment on column BILLING_CO2E_PUBLICATION_TRK.BILLRUNNO is
'Billing run no';

comment on column BILLING_CO2E_PUBLICATION_TRK.LASTCHANGED is
'Last changed date time';

alter table BILLING_CO2E_PUBLICATION_TRK
   add constraint BILLING_CO2E_PUBLICATIO_TRK_PK primary key (CONTRACTYEAR, WEEKNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_CSP_DEROGATION_AMOUNT                         */
/*==============================================================*/
create table BILLING_CSP_DEROGATION_AMOUNT (
   CONTRACTYEAR         NUMBER(4)             not null,
   WEEKNO               NUMBER(3)             not null,
   BILLRUNNO            NUMBER(3)             not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   AMOUNT_ID            VARCHAR2(20)          not null,
   DEROGATION_AMOUNT    NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLING_CSP_DEROGATION_AMOUNT is
'CSP derogation amounts with respect to participant allocated payment';

comment on column BILLING_CSP_DEROGATION_AMOUNT.CONTRACTYEAR is
'Billing contract year';

comment on column BILLING_CSP_DEROGATION_AMOUNT.WEEKNO is
'Billing week number';

comment on column BILLING_CSP_DEROGATION_AMOUNT.BILLRUNNO is
'Billing run number';

comment on column BILLING_CSP_DEROGATION_AMOUNT.PARTICIPANTID is
'The participant allocated the payment amount for the derogation';

comment on column BILLING_CSP_DEROGATION_AMOUNT.AMOUNT_ID is
'Amount identifier represented as a string, from "ta1" through to "ta6" (or "ta8" for a lymmco derogation result)';

comment on column BILLING_CSP_DEROGATION_AMOUNT.DEROGATION_AMOUNT is
'Derogation amount associated with the amount identifier';

comment on column BILLING_CSP_DEROGATION_AMOUNT.LASTCHANGED is
'Last changed date for the record';

alter table BILLING_CSP_DEROGATION_AMOUNT
   add constraint BILLING_CSP_DEROGATN_AMNT_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, AMOUNT_ID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLING_CSP_DEROGATN_AMNT_NDX1                        */
/*==============================================================*/
create index BILLING_CSP_DEROGATN_AMNT_NDX1 on BILLING_CSP_DEROGATION_AMOUNT (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_DAILY_ENERGY_SUMMARY                          */
/*==============================================================*/
create table BILLING_DAILY_ENERGY_SUMMARY (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   SETTLEMENTDATE       DATE                  not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   CUSTOMER_ENERGY_PURCHASED NUMBER(18,8),
   GENERATOR_ENERGY_SOLD NUMBER(18,8),
   GENERATOR_ENERGY_PURCHASED NUMBER(18,8),
   ACE_MWH              NUMBER(18,8),
   ASOE_MWH             NUMBER(18,8),
   ACE_AMOUNT           NUMBER(18,8),
   ASOE_AMOUNT          NUMBER(18,8),
   CE_MWH               NUMBER(18,8),
   UFEA_MWH             NUMBER(18,8),
   TOTAL_MWH            NUMBER(18,8),
   TOTAL_AMOUNT         NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLING_DAILY_ENERGY_SUMMARY is
'Billing result table containing daily summary data for customer and generator energy amounts';

comment on column BILLING_DAILY_ENERGY_SUMMARY.CONTRACTYEAR is
'Billing Contract Year';

comment on column BILLING_DAILY_ENERGY_SUMMARY.WEEKNO is
'Billing Week number';

comment on column BILLING_DAILY_ENERGY_SUMMARY.BILLRUNNO is
'Billing Run number';

comment on column BILLING_DAILY_ENERGY_SUMMARY.SETTLEMENTDATE is
'settlement date';

comment on column BILLING_DAILY_ENERGY_SUMMARY.PARTICIPANTID is
'participant identifier';

comment on column BILLING_DAILY_ENERGY_SUMMARY.REGIONID is
'Unique Region Identifier';

comment on column BILLING_DAILY_ENERGY_SUMMARY.CUSTOMER_ENERGY_PURCHASED is
'Customer energy amount purchased on this settlement day by the participant in the region. NULL for Billing Week post the IESS rule effective date.';

comment on column BILLING_DAILY_ENERGY_SUMMARY.GENERATOR_ENERGY_SOLD is
'Generator energy amount sold on this settlement day by the participant in the region. NULL for Billing Week post the IESS rule effective date.';

comment on column BILLING_DAILY_ENERGY_SUMMARY.GENERATOR_ENERGY_PURCHASED is
'Generator energy amount purchased on this settlement day by the participant in the region. NULL for Billing Week post the IESS rule effective date.';

comment on column BILLING_DAILY_ENERGY_SUMMARY.ACE_MWH is
'The Sum of ACE MWh value for the Participant and region for the Settlement Date. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLING_DAILY_ENERGY_SUMMARY.ASOE_MWH is
'The Sum of ASOE MWh value for the Participant and region for the Settlement Date. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLING_DAILY_ENERGY_SUMMARY.ACE_AMOUNT is
'The Sum of ACE Amount for the Participant and region for the Settlement Date. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLING_DAILY_ENERGY_SUMMARY.ASOE_AMOUNT is
'The Sum of ASOE Amount for the Participant and region for the Settlement Date. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLING_DAILY_ENERGY_SUMMARY.CE_MWH is
'The Sum of CE MWh value for the Participant and region for the Settlement Date. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLING_DAILY_ENERGY_SUMMARY.UFEA_MWH is
'The Sum of UFEA MWh value for the Participant and region for the Settlement Date. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLING_DAILY_ENERGY_SUMMARY.TOTAL_MWH is
'The Sum of Total MWh value for the Participant and region for the Settlement Date. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLING_DAILY_ENERGY_SUMMARY.TOTAL_AMOUNT is
'The Sum of Total Amount for the Participant and region for the Settlement Date. NULL for Billing Week prior to the IESS rule effective date';

alter table BILLING_DAILY_ENERGY_SUMMARY
   add constraint BILLING_DAILY_ENRGY_SUMMARY_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, SETTLEMENTDATE, PARTICIPANTID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_DIRECTION_RECONCILIATN                        */
/*==============================================================*/
create table BILLING_DIRECTION_RECONCILIATN (
   CONTRACTYEAR         NUMBER(4)             not null,
   WEEKNO               NUMBER(3)             not null,
   BILLRUNNO            NUMBER(3)             not null,
   DIRECTION_ID         VARCHAR2(20)          not null,
   DIRECTION_DESC       VARCHAR2(200),
   DIRECTION_START_DATE DATE,
   DIRECTION_END_DATE   DATE,
   COMPENSATION_AMOUNT  NUMBER(16,6),
   INDEPENDENT_EXPERT_FEE NUMBER(16,6),
   INTEREST_AMOUNT      NUMBER(16,6),
   CRA                  NUMBER(16,6),
   NEM_FEE_ID           VARCHAR2(20),
   NEM_FIXED_FEE_AMOUNT NUMBER(16,6),
   MKT_CUSTOMER_PERC    NUMBER(16,6),
   GENERATOR_PERC       NUMBER(16,6),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLING_DIRECTION_RECONCILIATN is
'Billing reconciliation result table for both provisional and final directions using the FPP methodology (prior to 1st July 2011)';

comment on column BILLING_DIRECTION_RECONCILIATN.CONTRACTYEAR is
'Billing contract year';

comment on column BILLING_DIRECTION_RECONCILIATN.WEEKNO is
'Billing week no';

comment on column BILLING_DIRECTION_RECONCILIATN.BILLRUNNO is
'Billing run no';

comment on column BILLING_DIRECTION_RECONCILIATN.DIRECTION_ID is
'Direction identifier';

comment on column BILLING_DIRECTION_RECONCILIATN.DIRECTION_DESC is
'Direction description';

comment on column BILLING_DIRECTION_RECONCILIATN.DIRECTION_START_DATE is
'Direction start date time';

comment on column BILLING_DIRECTION_RECONCILIATN.DIRECTION_END_DATE is
'Direction end date time';

comment on column BILLING_DIRECTION_RECONCILIATN.COMPENSATION_AMOUNT is
'Direction compensation amount';

comment on column BILLING_DIRECTION_RECONCILIATN.INDEPENDENT_EXPERT_FEE is
'Independent expert fee charged on calculating direction compensation amount';

comment on column BILLING_DIRECTION_RECONCILIATN.INTEREST_AMOUNT is
'Interest occurred on direction compensation amount';

comment on column BILLING_DIRECTION_RECONCILIATN.CRA is
'Direction compensation recovery amount';

comment on column BILLING_DIRECTION_RECONCILIATN.NEM_FEE_ID is
'Fixed settlement fee identifier for direction purpose';

comment on column BILLING_DIRECTION_RECONCILIATN.NEM_FIXED_FEE_AMOUNT is
'Fixed settlement fee for participants between direction start and end date';

comment on column BILLING_DIRECTION_RECONCILIATN.MKT_CUSTOMER_PERC is
'Direction compensation recovery amount percentage breakdown among customers';

comment on column BILLING_DIRECTION_RECONCILIATN.GENERATOR_PERC is
'Direction compensation recovery amount percentage breakdown among generators';

comment on column BILLING_DIRECTION_RECONCILIATN.LASTCHANGED is
'Last changed date time';

alter table BILLING_DIRECTION_RECONCILIATN
   add constraint BILLING_DIRECTION_RCNCLTN_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, DIRECTION_ID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLING_DIRECTION_RCNCLTN_NDX1                        */
/*==============================================================*/
create index BILLING_DIRECTION_RCNCLTN_NDX1 on BILLING_DIRECTION_RECONCILIATN (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_DIRECTION_RECON_OTHER                         */
/*==============================================================*/
create table BILLING_DIRECTION_RECON_OTHER (
   CONTRACTYEAR         NUMBER(4)             not null,
   WEEKNO               NUMBER(3)             not null,
   BILLRUNNO            NUMBER(3)             not null,
   DIRECTION_ID         VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   DIRECTION_DESC       VARCHAR2(200),
   DIRECTION_TYPE_ID    VARCHAR2(20),
   DIRECTION_START_DATE DATE,
   DIRECTION_END_DATE   DATE,
   DIRECTION_START_INTERVAL DATE,
   DIRECTION_END_INTERVAL DATE,
   COMPENSATION_AMOUNT  NUMBER(18,8),
   INTEREST_AMOUNT      NUMBER(18,8),
   INDEPENDENT_EXPERT_FEE NUMBER(18,8),
   CRA                  NUMBER(18,8),
   REGIONAL_CUSTOMER_ENERGY NUMBER(18,8),
   REGIONAL_GENERATOR_ENERGY NUMBER(18,8),
   REGIONAL_BENEFIT_FACTOR NUMBER(18,8),
   REGION_ACE_MWH       NUMBER(18,8),
   REGION_ASOE_MWH      NUMBER(18,8),
   DIRECTION_SERVICE_ID VARCHAR2(20)
)
   tablespace PSSTAB01;

comment on table BILLING_DIRECTION_RECON_OTHER is
'Billing reconciliation result table for both provisional and final directions';

comment on column BILLING_DIRECTION_RECON_OTHER.CONTRACTYEAR is
'Billing contract year';

comment on column BILLING_DIRECTION_RECON_OTHER.WEEKNO is
'Billing week no';

comment on column BILLING_DIRECTION_RECON_OTHER.BILLRUNNO is
'Billing run no';

comment on column BILLING_DIRECTION_RECON_OTHER.DIRECTION_ID is
'Direction identifier';

comment on column BILLING_DIRECTION_RECON_OTHER.REGIONID is
'Region Identifier';

comment on column BILLING_DIRECTION_RECON_OTHER.DIRECTION_DESC is
'Direction description';

comment on column BILLING_DIRECTION_RECON_OTHER.DIRECTION_TYPE_ID is
'The service for which the direction occurred (ENERGY, ANCILLARY, NON_ENERGY_NON_AS, etc)';

comment on column BILLING_DIRECTION_RECON_OTHER.DIRECTION_START_DATE is
'Settlement day on which the direction starts';

comment on column BILLING_DIRECTION_RECON_OTHER.DIRECTION_END_DATE is
'Settlement day on which the direction ends.  The same value for all regions';

comment on column BILLING_DIRECTION_RECON_OTHER.DIRECTION_START_INTERVAL is
'Dispatch interval in which the direction starts.  The same value for all regions';

comment on column BILLING_DIRECTION_RECON_OTHER.DIRECTION_END_INTERVAL is
'Dispatch interval in which the direction ends.  The same value for all regions';

comment on column BILLING_DIRECTION_RECON_OTHER.COMPENSATION_AMOUNT is
'The final compensation amount for the direction.  The same value for all regions';

comment on column BILLING_DIRECTION_RECON_OTHER.INTEREST_AMOUNT is
'The interest amount calculated on the final compensation amount for the direction.  The same value for all regions';

comment on column BILLING_DIRECTION_RECON_OTHER.INDEPENDENT_EXPERT_FEE is
'The independent expert fee amount for the direction.  The same value for all regions';

comment on column BILLING_DIRECTION_RECON_OTHER.CRA is
'The total recovery amount for the direction.  The same value for all regions';

comment on column BILLING_DIRECTION_RECON_OTHER.REGIONAL_CUSTOMER_ENERGY is
'The total customer energy for this region, over the duration of the direction. NULL for Billing Week post the IESS rule effective date.';

comment on column BILLING_DIRECTION_RECON_OTHER.REGIONAL_GENERATOR_ENERGY is
'The total generator energy for this region, over the duration of the direction. NULL for Billing Week post the IESS rule effective date.';

comment on column BILLING_DIRECTION_RECON_OTHER.REGIONAL_BENEFIT_FACTOR is
'The regional benefit factor allocated to this region for the direction';

comment on column BILLING_DIRECTION_RECON_OTHER.REGION_ACE_MWH is
'The Sum of ACE MWh value for the Region over the duration of the direction. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLING_DIRECTION_RECON_OTHER.REGION_ASOE_MWH is
'The Sum of ASOE MWh value for the Region over the duration of the direction. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLING_DIRECTION_RECON_OTHER.DIRECTION_SERVICE_ID is
'The Direction Service ID associated with the Direction Type ID. Eg For FCAS Direction Type, Direction Service could be any contingency service.';

alter table BILLING_DIRECTION_RECON_OTHER
   add constraint BILLING_DIRECTION_REC_OTHER_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, DIRECTION_ID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_DIR_FINAL_AMOUNT                              */
/*==============================================================*/
create table BILLING_DIR_FINAL_AMOUNT (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   DIRECTION_ID         VARCHAR2(20)          not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   COMPENSATION_TYPE    VARCHAR2(40)          not null,
   PROVISIONAL_AMOUNT   NUMBER(18,8),
   FINAL_AMOUNT         NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table BILLING_DIR_FINAL_AMOUNT is
'The Billing Final Directions Payment Amount for Directed/Affected/Eligible participants';

comment on column BILLING_DIR_FINAL_AMOUNT.CONTRACTYEAR is
'The Billing Contract Year';

comment on column BILLING_DIR_FINAL_AMOUNT.WEEKNO is
'The Billing WeekNo';

comment on column BILLING_DIR_FINAL_AMOUNT.BILLRUNNO is
'The Billing RunNo';

comment on column BILLING_DIR_FINAL_AMOUNT.DIRECTION_ID is
'The Direction Unique Identifier';

comment on column BILLING_DIR_FINAL_AMOUNT.PARTICIPANTID is
'The Direction Payment Participant ID';

comment on column BILLING_DIR_FINAL_AMOUNT.COMPENSATION_TYPE is
'The Direction Payment Type, Directed_Comp, Affected_Comp, Eligible_Comp.';

comment on column BILLING_DIR_FINAL_AMOUNT.PROVISIONAL_AMOUNT is
'The Direction Provisional Payment Amount';

comment on column BILLING_DIR_FINAL_AMOUNT.FINAL_AMOUNT is
'The Direction Final Payment Amount';

comment on column BILLING_DIR_FINAL_AMOUNT.LASTCHANGED is
'The Last datetime record is updated';

alter table BILLING_DIR_FINAL_AMOUNT
   add constraint BILLING_DIR_FINAL_AMOUNT_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, DIRECTION_ID, PARTICIPANTID, COMPENSATION_TYPE)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: BILLING_DIR_FINAL_RECOVERY                            */
/*==============================================================*/
create table BILLING_DIR_FINAL_RECOVERY (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   DIRECTION_ID         VARCHAR2(20)          not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   CRA_AMOUNT           NUMBER(18,8),
   PROVISIONAL_AMOUNT   NUMBER(18,8),
   FINAL_AMOUNT         NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table BILLING_DIR_FINAL_RECOVERY is
'The Billing Final Directions Recovery Amount for the participants';

comment on column BILLING_DIR_FINAL_RECOVERY.CONTRACTYEAR is
'The Billing Contract Year';

comment on column BILLING_DIR_FINAL_RECOVERY.WEEKNO is
'The Billing WeekNo';

comment on column BILLING_DIR_FINAL_RECOVERY.BILLRUNNO is
'The Billing RunNo';

comment on column BILLING_DIR_FINAL_RECOVERY.DIRECTION_ID is
'The Direction Unique Identifier';

comment on column BILLING_DIR_FINAL_RECOVERY.PARTICIPANTID is
'The Direction Payment Participant ID';

comment on column BILLING_DIR_FINAL_RECOVERY.CRA_AMOUNT is
'The Direction Compensation Recovery Amount';

comment on column BILLING_DIR_FINAL_RECOVERY.PROVISIONAL_AMOUNT is
'The Provisional Recovery Amount';

comment on column BILLING_DIR_FINAL_RECOVERY.FINAL_AMOUNT is
'The Final Recovery Amount';

comment on column BILLING_DIR_FINAL_RECOVERY.LASTCHANGED is
'The Last datetime record is updated';

alter table BILLING_DIR_FINAL_RECOVERY
   add constraint BILLING_DIR_FINAL_RECOVERY_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, DIRECTION_ID, PARTICIPANTID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: BILLING_DIR_PROV_AMOUNT                               */
/*==============================================================*/
create table BILLING_DIR_PROV_AMOUNT (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   DIRECTION_ID         VARCHAR2(20)          not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   COMPENSATION_TYPE    VARCHAR2(40)          not null,
   COMPENSATION_AMOUNT  NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table BILLING_DIR_PROV_AMOUNT is
'The Billing Provisional Directions Payment Amount for Directed/Affected/Eligible participants';

comment on column BILLING_DIR_PROV_AMOUNT.CONTRACTYEAR is
'The Billing Contract Year';

comment on column BILLING_DIR_PROV_AMOUNT.WEEKNO is
'The Billing WeekNo';

comment on column BILLING_DIR_PROV_AMOUNT.BILLRUNNO is
'The Billing RunNo';

comment on column BILLING_DIR_PROV_AMOUNT.DIRECTION_ID is
'The Direction Unique Identifier';

comment on column BILLING_DIR_PROV_AMOUNT.PARTICIPANTID is
'The Direction Payment Participant ID';

comment on column BILLING_DIR_PROV_AMOUNT.COMPENSATION_TYPE is
'The Direction Payment Type, Directed_Comp, Affected_Comp, Eligible_Comp';

comment on column BILLING_DIR_PROV_AMOUNT.COMPENSATION_AMOUNT is
'The Direction Payment Amount';

comment on column BILLING_DIR_PROV_AMOUNT.LASTCHANGED is
'The Last datetime record is updated';

alter table BILLING_DIR_PROV_AMOUNT
   add constraint BILLING_DIR_PROV_AMOUNT_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, DIRECTION_ID, PARTICIPANTID, COMPENSATION_TYPE)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: BILLING_DIR_PROV_RECOVERY                             */
/*==============================================================*/
create table BILLING_DIR_PROV_RECOVERY (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   DIRECTION_ID         VARCHAR2(20)          not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   CRA_AMOUNT           NUMBER(18,8),
   RECOVERY_AMOUNT      NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table BILLING_DIR_PROV_RECOVERY is
'The Billing Provisional Directions Recovery Amount for the participants';

comment on column BILLING_DIR_PROV_RECOVERY.CONTRACTYEAR is
'The Billing Contract Year';

comment on column BILLING_DIR_PROV_RECOVERY.WEEKNO is
'The Billing WeekNo';

comment on column BILLING_DIR_PROV_RECOVERY.BILLRUNNO is
'The Billing RunNo';

comment on column BILLING_DIR_PROV_RECOVERY.DIRECTION_ID is
'The Direction Unique Identifier';

comment on column BILLING_DIR_PROV_RECOVERY.PARTICIPANTID is
'The Direction Payment Participant ID';

comment on column BILLING_DIR_PROV_RECOVERY.CRA_AMOUNT is
'The Direction Compensation Recovery Amount';

comment on column BILLING_DIR_PROV_RECOVERY.RECOVERY_AMOUNT is
'The Direction Recovery Amount';

comment on column BILLING_DIR_PROV_RECOVERY.LASTCHANGED is
'The Last datetime record is updated';

alter table BILLING_DIR_PROV_RECOVERY
   add constraint BILLING_DIR_PROV_RECOVERY_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, DIRECTION_ID, PARTICIPANTID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: BILLING_DIR_RECOVERY_DETAIL                           */
/*==============================================================*/
create table BILLING_DIR_RECOVERY_DETAIL (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   DIRECTION_ID         VARCHAR2(20)          not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   PARTICIPANTCATEGORYID VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   RECOVERY_AMOUNT      NUMBER(18,8),
   RECOVERY_ENERGY      NUMBER(18,8),
   REGION_ENERGY        NUMBER(18,8),
   EXCLUDED_ENERGY      NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table BILLING_DIR_RECOVERY_DETAIL is
'The Billing Directions Recovery Details for the participants';

comment on column BILLING_DIR_RECOVERY_DETAIL.CONTRACTYEAR is
'The Billing Contract Year';

comment on column BILLING_DIR_RECOVERY_DETAIL.WEEKNO is
'The Billing WeekNo';

comment on column BILLING_DIR_RECOVERY_DETAIL.BILLRUNNO is
'The Billing RunNo';

comment on column BILLING_DIR_RECOVERY_DETAIL.DIRECTION_ID is
'The Direction Unique Identifier';

comment on column BILLING_DIR_RECOVERY_DETAIL.PARTICIPANTID is
'The Direction Payment Participant ID';

comment on column BILLING_DIR_RECOVERY_DETAIL.PARTICIPANTCATEGORYID is
'The Participant Category for recovery Customer/Generator /SmallGen';

comment on column BILLING_DIR_RECOVERY_DETAIL.REGIONID is
'The Region ID for the recovery';

comment on column BILLING_DIR_RECOVERY_DETAIL.RECOVERY_AMOUNT is
'The Direction Recovery Amount';

comment on column BILLING_DIR_RECOVERY_DETAIL.RECOVERY_ENERGY is
'The Energy Value used for the Recovery';

comment on column BILLING_DIR_RECOVERY_DETAIL.REGION_ENERGY is
'The total Energy at the Region ID';

comment on column BILLING_DIR_RECOVERY_DETAIL.EXCLUDED_ENERGY is
'The Energy Value (Scheduled Loads) that is excluded';

comment on column BILLING_DIR_RECOVERY_DETAIL.LASTCHANGED is
'The Last datetime record is updated';

alter table BILLING_DIR_RECOVERY_DETAIL
   add constraint BILLING_DIR_RECOVERY_DETAIL_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, DIRECTION_ID, PARTICIPANTID, PARTICIPANTCATEGORYID, REGIONID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: BILLING_EFTSHORTFALL_AMOUNT                           */
/*==============================================================*/
create table BILLING_EFTSHORTFALL_AMOUNT (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   SHORTFALL_AMOUNT     NUMBER(18,8),
   SHORTFALL            NUMBER(18,8),
   SHORTFALL_COMPANY_ID VARCHAR2(20),
   COMPANY_SHORTFALL_AMOUNT NUMBER(18,8),
   PARTICIPANT_NET_ENERGY NUMBER(18,8),
   COMPANY_NET_ENERGY   NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLING_EFTSHORTFALL_AMOUNT is
'The billing shortfall run amounts';

comment on column BILLING_EFTSHORTFALL_AMOUNT.CONTRACTYEAR is
'The shortfall affected billing contract year';

comment on column BILLING_EFTSHORTFALL_AMOUNT.WEEKNO is
'The shortfall affected billing week no';

comment on column BILLING_EFTSHORTFALL_AMOUNT.BILLRUNNO is
'The shortfall affected billing week run no';

comment on column BILLING_EFTSHORTFALL_AMOUNT.PARTICIPANTID is
'The participant affected by the shortfall calculation';

comment on column BILLING_EFTSHORTFALL_AMOUNT.SHORTFALL_AMOUNT is
'The Participant shortfall amount';

comment on column BILLING_EFTSHORTFALL_AMOUNT.SHORTFALL is
'The market shortfall amount';

comment on column BILLING_EFTSHORTFALL_AMOUNT.SHORTFALL_COMPANY_ID is
'The Company ID associated with the Participant ID used in the shortfall calculation';

comment on column BILLING_EFTSHORTFALL_AMOUNT.COMPANY_SHORTFALL_AMOUNT is
'The shortfall amount for the Company ID associated with the Participant ID used in the shortfall calculation';

comment on column BILLING_EFTSHORTFALL_AMOUNT.PARTICIPANT_NET_ENERGY is
'The participant NET energy used in shortfall calculation';

comment on column BILLING_EFTSHORTFALL_AMOUNT.COMPANY_NET_ENERGY is
'The NET energy for the Company ID associated with the Participant ID used in the shortfall calculation';

alter table BILLING_EFTSHORTFALL_AMOUNT
   add constraint BILLING_EFTSHORTFALL_AMT_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_EFTSHORTFALL_DETAIL                           */
/*==============================================================*/
create table BILLING_EFTSHORTFALL_DETAIL (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   TRANSACTION_TYPE     VARCHAR2(40)          not null,
   AMOUNT               NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLING_EFTSHORTFALL_DETAIL is
'The Billing Shortfall Run Amount details';

comment on column BILLING_EFTSHORTFALL_DETAIL.CONTRACTYEAR is
'The shortfall affected billing contract year';

comment on column BILLING_EFTSHORTFALL_DETAIL.WEEKNO is
'The shortfall affected billing week no';

comment on column BILLING_EFTSHORTFALL_DETAIL.BILLRUNNO is
'The shortfall affected billing week run no';

comment on column BILLING_EFTSHORTFALL_DETAIL.PARTICIPANTID is
'The participant affected by the shortfall calculation';

comment on column BILLING_EFTSHORTFALL_DETAIL.TRANSACTION_TYPE is
'The transaction type details associated with the shortfall calculation';

comment on column BILLING_EFTSHORTFALL_DETAIL.AMOUNT is
'The amount for each transaction type';

alter table BILLING_EFTSHORTFALL_DETAIL
   add constraint BILLING_EFTSHORTFALL_DETL_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, TRANSACTION_TYPE)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_ENERGY_GENSET_DETAIL                          */
/*==============================================================*/
create table BILLING_ENERGY_GENSET_DETAIL (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(4,0)           not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   STATIONID            VARCHAR2(20)          not null,
   DUID                 VARCHAR2(20)          not null,
   GENSETID             VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   CONNECTIONPOINTID    VARCHAR2(20)          not null,
   METERID              VARCHAR2(20)          not null,
   CE_MWH               NUMBER(18,8),
   UFEA_MWH             NUMBER(18,8),
   ACE_MWH              NUMBER(18,8),
   ASOE_MWH             NUMBER(18,8),
   TOTAL_MWH            NUMBER(18,8),
   DME_MWH              NUMBER(18,8),
   ACE_AMOUNT           NUMBER(18,8),
   ASOE_AMOUNT          NUMBER(18,8),
   TOTAL_AMOUNT         NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table BILLING_ENERGY_GENSET_DETAIL is
'The Billing Energy Genset report contains the Genset Energy detail summary for the Billing Week data';

comment on column BILLING_ENERGY_GENSET_DETAIL.CONTRACTYEAR is
'The Billing Contract Year';

comment on column BILLING_ENERGY_GENSET_DETAIL.WEEKNO is
'The Billing Week No';

comment on column BILLING_ENERGY_GENSET_DETAIL.BILLRUNNO is
'The Billing Run No';

comment on column BILLING_ENERGY_GENSET_DETAIL.PARTICIPANTID is
'The Participant Id Identifier';

comment on column BILLING_ENERGY_GENSET_DETAIL.STATIONID is
'The StationId identifier associated with the GensetId';

comment on column BILLING_ENERGY_GENSET_DETAIL.DUID is
'The DUID for the meter associated with the GensetId';

comment on column BILLING_ENERGY_GENSET_DETAIL.GENSETID is
'The GensetId for the Meter Id received';

comment on column BILLING_ENERGY_GENSET_DETAIL.REGIONID is
'The Region Id for the Connection Point associated with the DUID';

comment on column BILLING_ENERGY_GENSET_DETAIL.CONNECTIONPOINTID is
'The Connection Point associated with the DUID';

comment on column BILLING_ENERGY_GENSET_DETAIL.METERID is
'The Meter ID Identifier (NMI)';

comment on column BILLING_ENERGY_GENSET_DETAIL.CE_MWH is
'The Consumed Energy for the Meter Id . Energy received in the meter reads (DLF Adjusted) in that Billing Week';

comment on column BILLING_ENERGY_GENSET_DETAIL.UFEA_MWH is
'The UFEA Energy MWh Consumed for that Connection Point for the Participant Id in that Billing Week';

comment on column BILLING_ENERGY_GENSET_DETAIL.ACE_MWH is
'The Adjusted Consumed Energy MWh Consumed for that Connection Point for the Participant Id in that Billing Week';

comment on column BILLING_ENERGY_GENSET_DETAIL.ASOE_MWH is
'The Adjusted Sent Out Energy MWh Consumed for that Connection Point for the Participant Id in that Billing Week';

comment on column BILLING_ENERGY_GENSET_DETAIL.TOTAL_MWH is
'The Total MWh(ACE_MWh + ASOE_MWh) for that Connection Point for the Participant Id in that Billing Week';

comment on column BILLING_ENERGY_GENSET_DETAIL.DME_MWH is
'The DME MWh for that Connection Point for the Participant Id in that Billing Week. This is the MWh value that is used for the UFEA Allocation';

comment on column BILLING_ENERGY_GENSET_DETAIL.ACE_AMOUNT is
'The Adjusted Consumed Energy Dollar Amount for that Connection Point for the Participant Id in that Billing Week';

comment on column BILLING_ENERGY_GENSET_DETAIL.ASOE_AMOUNT is
'The Adjusted Sent Out Energy Dollar Amount for that Connection Point for the Participant Id in that Billing Week';

comment on column BILLING_ENERGY_GENSET_DETAIL.TOTAL_AMOUNT is
'The Total Amount(ACE_Amount + ASOE_Amount) for that Connection Point for the Participant Id in that Billing Week';

comment on column BILLING_ENERGY_GENSET_DETAIL.LASTCHANGED is
'The Last changed date time for the record';

alter table BILLING_ENERGY_GENSET_DETAIL
   add constraint BILL_ENERGY_GENSET_DETAIL_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, STATIONID, DUID, GENSETID, REGIONID, CONNECTIONPOINTID, METERID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: BILLING_ENERGY_TRANSACTIONS                           */
/*==============================================================*/
create table BILLING_ENERGY_TRANSACTIONS (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(4,0)           not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   CONNECTIONPOINTID    VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   CE_MWH               NUMBER(18,8),
   UFEA_MWH             NUMBER(18,8),
   ACE_MWH              NUMBER(18,8),
   ASOE_MWH             NUMBER(18,8),
   ACE_AMOUNT           NUMBER(18,8),
   ASOE_AMOUNT          NUMBER(18,8),
   TOTAL_MWH            NUMBER(18,8),
   TOTAL_AMOUNT         NUMBER(18,8),
   DME_MWH              NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table BILLING_ENERGY_TRANSACTIONS is
'The Billing Energy Transactions is the summary of the Settlement Energy Transactions that has the ACE and ASOE MWh and Dollar values that is used for the Statement';

comment on column BILLING_ENERGY_TRANSACTIONS.CONTRACTYEAR is
'The Billing Contract Year';

comment on column BILLING_ENERGY_TRANSACTIONS.WEEKNO is
'The Billing WeekNo';

comment on column BILLING_ENERGY_TRANSACTIONS.BILLRUNNO is
'The Billing RunNo';

comment on column BILLING_ENERGY_TRANSACTIONS.PARTICIPANTID is
'The Participant Id Identifier';

comment on column BILLING_ENERGY_TRANSACTIONS.CONNECTIONPOINTID is
'The ConnectionPoint Id for the Billing Aggregation for the Participant Id.';

comment on column BILLING_ENERGY_TRANSACTIONS.REGIONID is
'The Region Id Identifier';

comment on column BILLING_ENERGY_TRANSACTIONS.CE_MWH is
'The Consumed Energy MWh Consumed for that Connection Point for the Participant Id in that Billing Week';

comment on column BILLING_ENERGY_TRANSACTIONS.UFEA_MWH is
'The UFEA Energy MWh Consumed for that Connection Point for the Participant Id in that Billing Week';

comment on column BILLING_ENERGY_TRANSACTIONS.ACE_MWH is
'The Adjusted Consumed Energy MWh Consumed for that Connection Point for the Participant Id in that Billing Week';

comment on column BILLING_ENERGY_TRANSACTIONS.ASOE_MWH is
'The Adjusted Sent Out Energy MWh Consumed for that Connection Point for the Participant Id in that Billing Week';

comment on column BILLING_ENERGY_TRANSACTIONS.ACE_AMOUNT is
'The Adjusted Consumed Energy Dollar Amount for that Connection Point for the Participant Id in that Billing Week';

comment on column BILLING_ENERGY_TRANSACTIONS.ASOE_AMOUNT is
'The Adjusted Sent Out Energy Dollar Amount for that Connection Point for the Participant Id in that Billing Week';

comment on column BILLING_ENERGY_TRANSACTIONS.TOTAL_MWH is
'The Total MWh(ACE_MWh + ASOE_MWh) for that Connection Point for the Participant Id in that Billing Week';

comment on column BILLING_ENERGY_TRANSACTIONS.TOTAL_AMOUNT is
'The Total Amount(ACE_Amount + ASOE_Amount) for that Connection Point for the Participant Id in that Billing Week';

comment on column BILLING_ENERGY_TRANSACTIONS.DME_MWH is
'The DME MWh for that Connection Point for the Participant Id in that Billing Week. This is the MWh value that is used for the UFEA Allocation.';

comment on column BILLING_ENERGY_TRANSACTIONS.LASTCHANGED is
'The Last Changed date time for the record';

alter table BILLING_ENERGY_TRANSACTIONS
   add constraint BILLING_ENERGY_TRANSACTIONS_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, CONNECTIONPOINTID, REGIONID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: BILLING_ENERGY_TRAN_SAPS                              */
/*==============================================================*/
create table BILLING_ENERGY_TRAN_SAPS (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   TNI                  VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20),
   CONSUMED_ENERGY_MWH  NUMBER(18,8),
   SENTOUT_ENERGY_MWH   NUMBER(18,8),
   CONSUMED_ENERGY_COST NUMBER(18,8),
   SENTOUT_ENERGY_COST  NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table BILLING_ENERGY_TRAN_SAPS is
'The SAP Billing Transaction Details for the Participants';

comment on column BILLING_ENERGY_TRAN_SAPS.CONTRACTYEAR is
'The Billing Contract Year';

comment on column BILLING_ENERGY_TRAN_SAPS.WEEKNO is
'The Billing WeekNo';

comment on column BILLING_ENERGY_TRAN_SAPS.BILLRUNNO is
'The Billing RunNo';

comment on column BILLING_ENERGY_TRAN_SAPS.PARTICIPANTID is
'The SAP Participant ID';

comment on column BILLING_ENERGY_TRAN_SAPS.TNI is
'The SAPS Connection Point ID';

comment on column BILLING_ENERGY_TRAN_SAPS.REGIONID is
'The Region ID associated with the TNI';

comment on column BILLING_ENERGY_TRAN_SAPS.CONSUMED_ENERGY_MWH is
'The Energy MWh Consumed for that TNI for the Participant Id in that Billing Week';

comment on column BILLING_ENERGY_TRAN_SAPS.SENTOUT_ENERGY_MWH is
'The Energy MWh Sent Out for the TNI for the Participant Id in that Billing Week';

comment on column BILLING_ENERGY_TRAN_SAPS.CONSUMED_ENERGY_COST is
'The Cost of the Consumed Energy';

comment on column BILLING_ENERGY_TRAN_SAPS.SENTOUT_ENERGY_COST is
'The Cost of the Sent Out Energy';

comment on column BILLING_ENERGY_TRAN_SAPS.LASTCHANGED is
'The Last datetime record is updated';

alter table BILLING_ENERGY_TRAN_SAPS
   add constraint BILLING_ENERGY_TRAN_SAPS_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, TNI)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: BILLING_FCAS_REG_AMT                                  */
/*==============================================================*/
create table BILLING_FCAS_REG_AMT (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(4,0)           not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   UNITID               VARCHAR2(20)          not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   CONNECTIONPOINTID    VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   BIDTYPE              VARCHAR2(10),
   FPP_AMOUNT           NUMBER(18,8),
   USED_AMOUNT          NUMBER(18,8),
   UNUSED_AMOUNT        NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table BILLING_FCAS_REG_AMT is
'This report show the summary of the Billing run FCAS Regulation Amounts incl default amounts.';

comment on column BILLING_FCAS_REG_AMT.CONTRACTYEAR is
'The Billing Contract Year';

comment on column BILLING_FCAS_REG_AMT.WEEKNO is
'The Billing WeekNo';

comment on column BILLING_FCAS_REG_AMT.BILLRUNNO is
'The Billing RunNo';

comment on column BILLING_FCAS_REG_AMT.PARTICIPANTID is
'The Participant Id Identifier';

comment on column BILLING_FCAS_REG_AMT.UNITID is
'The UnitId for which a FPP Factor is assigned';

comment on column BILLING_FCAS_REG_AMT.CONSTRAINTID is
'The FCAS Reg Constraint Id';

comment on column BILLING_FCAS_REG_AMT.CONNECTIONPOINTID is
'The ConnectionPointId from Settlement Results for the UnitId';

comment on column BILLING_FCAS_REG_AMT.REGIONID is
'The Region Id Identifier';

comment on column BILLING_FCAS_REG_AMT.BIDTYPE is
'The BidType for the Constraint Id';

comment on column BILLING_FCAS_REG_AMT.FPP_AMOUNT is
'The FPP Amount for the Participant and the Unit';

comment on column BILLING_FCAS_REG_AMT.USED_AMOUNT is
'The FCAS Regulation Used Amount for the Participant and the Unit';

comment on column BILLING_FCAS_REG_AMT.UNUSED_AMOUNT is
'The FCAS Regulation Unused Amount for the Participant and the Unit';

comment on column BILLING_FCAS_REG_AMT.LASTCHANGED is
'The Lastchanged datetime of the record.';

alter table BILLING_FCAS_REG_AMT
   add constraint BILLING_FCAS_REG_AMT_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, UNITID, CONSTRAINTID, CONNECTIONPOINTID, REGIONID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: BILLING_FCAS_REG_RESIDAMT                             */
/*==============================================================*/
create table BILLING_FCAS_REG_RESIDAMT (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(4,0)           not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   BIDTYPE              VARCHAR2(10),
   ACE_MWH              NUMBER(18,8),
   ASOE_MWH             NUMBER(18,8),
   RESIDUAL_MWH         NUMBER(18,8),
   FPP_ACE_AMOUNT       NUMBER(18,8),
   FPP_ASOE_AMOUNT      NUMBER(18,8),
   FPP_RESIDUAL_AMOUNT  NUMBER(18,8),
   USED_ACE_AMOUNT      NUMBER(18,8),
   USED_ASOE_AMOUNT     NUMBER(18,8),
   USED_RESIDUAL_AMOUNT NUMBER(18,8),
   UNUSED_ACE_AMOUNT    NUMBER(18,8),
   UNUSED_ASOE_AMOUNT   NUMBER(18,8),
   UNUSED_RESIDUAL_AMOUNT NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table BILLING_FCAS_REG_RESIDAMT is
'This report show the summary of the Billing run FCAS Regulation Residual Amounts incl default amounts';

comment on column BILLING_FCAS_REG_RESIDAMT.CONTRACTYEAR is
'The Billing Contract Year';

comment on column BILLING_FCAS_REG_RESIDAMT.WEEKNO is
'The Billing WeekNo';

comment on column BILLING_FCAS_REG_RESIDAMT.BILLRUNNO is
'The Billing RunNo';

comment on column BILLING_FCAS_REG_RESIDAMT.PARTICIPANTID is
'The Participant Id Identifier';

comment on column BILLING_FCAS_REG_RESIDAMT.CONSTRAINTID is
'The FCAS Reg Constraint Id';

comment on column BILLING_FCAS_REG_RESIDAMT.REGIONID is
'The Region Id Identifier';

comment on column BILLING_FCAS_REG_RESIDAMT.BIDTYPE is
'The BidType for the Constraint Id';

comment on column BILLING_FCAS_REG_RESIDAMT.ACE_MWH is
'The ACE MWh used for the Residual Calculation for the Participant';

comment on column BILLING_FCAS_REG_RESIDAMT.ASOE_MWH is
'The ASOE MWh used for the Residual Calculation of the Participant';

comment on column BILLING_FCAS_REG_RESIDAMT.RESIDUAL_MWH is
'Sum of ACE_MWH + ASOE_MWH';

comment on column BILLING_FCAS_REG_RESIDAMT.FPP_ACE_AMOUNT is
'The FPP ACE Portion Amount for the Billing Week';

comment on column BILLING_FCAS_REG_RESIDAMT.FPP_ASOE_AMOUNT is
'The FPP ASOE Portion Amount for the Billing Week';

comment on column BILLING_FCAS_REG_RESIDAMT.FPP_RESIDUAL_AMOUNT is
'The FPP Residual Amount for the Billing Week';

comment on column BILLING_FCAS_REG_RESIDAMT.USED_ACE_AMOUNT is
'The Used ACE Portion Amount for the Billing Week';

comment on column BILLING_FCAS_REG_RESIDAMT.USED_ASOE_AMOUNT is
'The Used ASOE Portion Amount for the Billing Week';

comment on column BILLING_FCAS_REG_RESIDAMT.USED_RESIDUAL_AMOUNT is
'The Used Residual Amount for the Billing Week';

comment on column BILLING_FCAS_REG_RESIDAMT.UNUSED_ACE_AMOUNT is
'The Unused ACE Portion Amount for the Billing Week';

comment on column BILLING_FCAS_REG_RESIDAMT.UNUSED_ASOE_AMOUNT is
'The Unused ASOE Portion Amount for the Billing Week';

comment on column BILLING_FCAS_REG_RESIDAMT.UNUSED_RESIDUAL_AMOUNT is
'The Unused Residual Amount for the Billing Week';

comment on column BILLING_FCAS_REG_RESIDAMT.LASTCHANGED is
'The Lastchanged datetime of the record.';

alter table BILLING_FCAS_REG_RESIDAMT
   add constraint BILLING_FCAS_REG_RESIDAMT_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, CONSTRAINTID, REGIONID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: BILLING_FPP                                           */
/*==============================================================*/
create table BILLING_FPP (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(4,0)           not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   LOWERREG_AMOUNT      NUMBER(18,8),
   LOWERREG_ACE_AMOUNT  NUMBER(18,8),
   LOWERREG_ASOE_AMOUNT NUMBER(18,8),
   LOWERREG_RESIDUAL_AMOUNT NUMBER(18,8),
   RAISEREG_AMOUNT      NUMBER(18,8),
   RAISEREG_ACE_AMOUNT  NUMBER(18,8),
   RAISEREG_ASOE_AMOUNT NUMBER(18,8),
   RAISEREG_RESIDUAL_AMOUNT NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table BILLING_FPP is
'This report show the summary of the Billing run FPP Amounts for the week';

comment on column BILLING_FPP.CONTRACTYEAR is
'The Billing Contract Year';

comment on column BILLING_FPP.WEEKNO is
'The Billing WeekNo';

comment on column BILLING_FPP.BILLRUNNO is
'The Billing RunNo';

comment on column BILLING_FPP.PARTICIPANTID is
'The Participant Id Identifier';

comment on column BILLING_FPP.REGIONID is
'The Region Id Identifier';

comment on column BILLING_FPP.LOWERREG_AMOUNT is
'The Lower Reg Service FPP Amount for the Participant and the Region for the Billing Week';

comment on column BILLING_FPP.LOWERREG_ACE_AMOUNT is
'The Lower Reg Service FPP ACE Residual Amount for the Participant and the Region for the Billing Week';

comment on column BILLING_FPP.LOWERREG_ASOE_AMOUNT is
'The Lower Reg Service FPP ASOE Residual Amount for the Participant and the Region for the Billing Week';

comment on column BILLING_FPP.LOWERREG_RESIDUAL_AMOUNT is
'Sum of LOWERREG_ACE_AMOUNT + LOWERREG_ASOE_AMOUNT';

comment on column BILLING_FPP.RAISEREG_AMOUNT is
'The Raise Reg Service FPP Amount for the Participant and the Region for the Billing Week';

comment on column BILLING_FPP.RAISEREG_ACE_AMOUNT is
'The Raise Reg Service FPP ACE Residual Amount for the Participant and the Region for the Billing Week';

comment on column BILLING_FPP.RAISEREG_ASOE_AMOUNT is
'The Raise Reg Service FPP ASOE Residual Amount for the Participant and the Region for the Billing Week';

comment on column BILLING_FPP.RAISEREG_RESIDUAL_AMOUNT is
'RAISEREG_ACE_AMOUNT + RAISEREG_ASOE_AMOUNT';

comment on column BILLING_FPP.LASTCHANGED is
'The Lastchanged datetime of the record.';

alter table BILLING_FPP
   add constraint BILLING_FPP_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, REGIONID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: BILLING_GST_DETAIL                                    */
/*==============================================================*/
create table BILLING_GST_DETAIL (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   BAS_CLASS            VARCHAR2(30)          not null,
   TRANSACTION_TYPE     VARCHAR2(30)          not null,
   GST_EXCLUSIVE_AMOUNT NUMBER(15,5),
   GST_AMOUNT           NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLING_GST_DETAIL is
'BILLING_GST_DETAIL shows the BAS class, GST_Exclusive and GST amount (if any) attributable to a participant for each transaction type.';

comment on column BILLING_GST_DETAIL.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLING_GST_DETAIL.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLING_GST_DETAIL.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLING_GST_DETAIL.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLING_GST_DETAIL.BAS_CLASS is
'The BAS classification that the transaction type belongs to.';

comment on column BILLING_GST_DETAIL.TRANSACTION_TYPE is
'The transaction type (e.g. CUSTOMER_ENERGY_PURCHASES)';

comment on column BILLING_GST_DETAIL.GST_EXCLUSIVE_AMOUNT is
'The GST exclusive amount paid by/to the participant to/by AEMO for this transaction type.';

comment on column BILLING_GST_DETAIL.GST_AMOUNT is
'The GST amount for this transaction type.';

comment on column BILLING_GST_DETAIL.LASTCHANGED is
'Last date and time record changed';

alter table BILLING_GST_DETAIL
   add constraint BILLING_GST_DETAIL_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, TRANSACTION_TYPE, BAS_CLASS)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLING_GST_DETAIL_LCX                                */
/*==============================================================*/
create index BILLING_GST_DETAIL_LCX on BILLING_GST_DETAIL (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_GST_SUMMARY                                   */
/*==============================================================*/
create table BILLING_GST_SUMMARY (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   BAS_CLASS            VARCHAR2(30)          not null,
   GST_EXCLUSIVE_AMOUNT NUMBER(15,5),
   GST_AMOUNT           NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLING_GST_SUMMARY is
'BILLING_GST_SUMMARY shows the GST_Exclusive and GST amount (if any)  attributable to a participant for each BAS class.';

comment on column BILLING_GST_SUMMARY.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLING_GST_SUMMARY.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLING_GST_SUMMARY.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLING_GST_SUMMARY.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLING_GST_SUMMARY.BAS_CLASS is
'The BAS classification';

comment on column BILLING_GST_SUMMARY.GST_EXCLUSIVE_AMOUNT is
'The GST exclusive amount paid by/to the participant to/by AEMO for this BAS classification.';

comment on column BILLING_GST_SUMMARY.GST_AMOUNT is
'The GST amount for this BAS classification.';

comment on column BILLING_GST_SUMMARY.LASTCHANGED is
'Last date and time record changed';

alter table BILLING_GST_SUMMARY
   add constraint BILLING_GST_SUMMARY_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, BAS_CLASS)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLING_GST_SUMMARY_LCX                               */
/*==============================================================*/
create index BILLING_GST_SUMMARY_LCX on BILLING_GST_SUMMARY (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_MR_PAYMENT                                    */
/*==============================================================*/
create table BILLING_MR_PAYMENT (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   MR_DATE              DATE                  not null,
   REGIONID             VARCHAR2(10)          not null,
   PARTICIPANTID        VARCHAR2(10),
   DUID                 VARCHAR2(10)          not null,
   MR_AMOUNT            NUMBER(16,6),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLING_MR_PAYMENT is
'BILLING_MR_PAYMENT shows aggregate payments on a dispatchable unit/MR Event basis for accepted MR capacity';

comment on column BILLING_MR_PAYMENT.CONTRACTYEAR is
'Billing Contract Year';

comment on column BILLING_MR_PAYMENT.WEEKNO is
'Billing Week number';

comment on column BILLING_MR_PAYMENT.BILLRUNNO is
'Billing Run number';

comment on column BILLING_MR_PAYMENT.MR_DATE is
'Trading Date of Mandatory Restriction event; Mandatory Restriction imposition date';

comment on column BILLING_MR_PAYMENT.REGIONID is
'Unique Region Identifier';

comment on column BILLING_MR_PAYMENT.PARTICIPANTID is
'Unique Participant identifier';

comment on column BILLING_MR_PAYMENT.DUID is
'Unique identifier for DUID / MNSP LinkID';

comment on column BILLING_MR_PAYMENT.MR_AMOUNT is
'Payment amount by AEMO';

comment on column BILLING_MR_PAYMENT.LASTCHANGED is
'Date/Time record inserted/modified';

alter table BILLING_MR_PAYMENT
   add constraint BILLING_MR_PAYMENT_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, MR_DATE, REGIONID, DUID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLING_MR_PAYMENT_LCX                                */
/*==============================================================*/
create index BILLING_MR_PAYMENT_LCX on BILLING_MR_PAYMENT (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_MR_RECOVERY                                   */
/*==============================================================*/
create table BILLING_MR_RECOVERY (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   MR_DATE              DATE                  not null,
   REGIONID             VARCHAR2(10)          not null,
   PARTICIPANTID        VARCHAR2(10),
   DUID                 VARCHAR2(10)          not null,
   MR_AMOUNT            NUMBER(16,6),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLING_MR_RECOVERY is
'BILLING_MR_RECOVERY shows aggregate recovery charges on a dispatchable unit / MR Event basis for spot market income from dispatch of MR capacity.';

comment on column BILLING_MR_RECOVERY.CONTRACTYEAR is
'Billing Contract Year';

comment on column BILLING_MR_RECOVERY.WEEKNO is
'Billing Week number';

comment on column BILLING_MR_RECOVERY.BILLRUNNO is
'Billing Run number';

comment on column BILLING_MR_RECOVERY.MR_DATE is
'Trading Date of Mandatory Restriction event; Mandatory Restriction imposition date';

comment on column BILLING_MR_RECOVERY.REGIONID is
'Unique Region Identifier';

comment on column BILLING_MR_RECOVERY.PARTICIPANTID is
'Unique Participant identifier';

comment on column BILLING_MR_RECOVERY.DUID is
'Unique identifier for DUID / MNSP LinkID';

comment on column BILLING_MR_RECOVERY.MR_AMOUNT is
'Payment amount to AEMO';

comment on column BILLING_MR_RECOVERY.LASTCHANGED is
'Date/Time record inserted/modified';

alter table BILLING_MR_RECOVERY
   add constraint BILLING_MR_RECOVERY_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, MR_DATE, REGIONID, DUID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLING_MR_RECOVERY_LCX                               */
/*==============================================================*/
create index BILLING_MR_RECOVERY_LCX on BILLING_MR_RECOVERY (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_MR_SHORTFALL                                  */
/*==============================================================*/
create table BILLING_MR_SHORTFALL (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   MR_DATE              DATE                  not null,
   REGIONID             VARCHAR2(10)          not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   AGE                  NUMBER(16,6),
   RSA                  NUMBER(16,6),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLING_MR_SHORTFALL is
'BILLING_MR_SHORTFALL shows aggregate MR shortfall payments (or recovery charges) to each participant in the region for the MR event.';

comment on column BILLING_MR_SHORTFALL.CONTRACTYEAR is
'Billing Contract Year';

comment on column BILLING_MR_SHORTFALL.WEEKNO is
'Billing Week number';

comment on column BILLING_MR_SHORTFALL.BILLRUNNO is
'Billing Run number';

comment on column BILLING_MR_SHORTFALL.MR_DATE is
'Trading Date of Mandatory Restriction event; Mandatory Restriction imposition date';

comment on column BILLING_MR_SHORTFALL.REGIONID is
'Unique Region Identifier';

comment on column BILLING_MR_SHORTFALL.PARTICIPANTID is
'Unique Participant Identifier';

comment on column BILLING_MR_SHORTFALL.AGE is
'The adjusted gross energy for the market customer in the restricted region for the duration of the mandatory restriction event (MWh)';

comment on column BILLING_MR_SHORTFALL.RSA is
'Restriction Shortfall amount payable to AEMO for a mandatory restriction period';

comment on column BILLING_MR_SHORTFALL.LASTCHANGED is
'Date/Time record inserted/modified';

alter table BILLING_MR_SHORTFALL
   add constraint BILLING_MR_SHORTFALL_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, MR_DATE, REGIONID, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLING_MR_SHORTFALL_LCX                              */
/*==============================================================*/
create index BILLING_MR_SHORTFALL_LCX on BILLING_MR_SHORTFALL (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_MR_SUMMARY                                    */
/*==============================================================*/
create table BILLING_MR_SUMMARY (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   MR_DATE              DATE                  not null,
   REGIONID             VARCHAR2(10)          not null,
   TOTAL_PAYMENTS       NUMBER(16,6),
   TOTAL_RECOVERY       NUMBER(16,6),
   TOTAL_RSA            NUMBER(16,6),
   AAGE                 NUMBER(16,6),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLING_MR_SUMMARY is
'BILLING_MR_SUMMARY shows aggregate payment/recovery and shortfall figures for an MR Event.';

comment on column BILLING_MR_SUMMARY.CONTRACTYEAR is
'Billing Contract Year';

comment on column BILLING_MR_SUMMARY.WEEKNO is
'Billing Week number';

comment on column BILLING_MR_SUMMARY.BILLRUNNO is
'Billing Run number';

comment on column BILLING_MR_SUMMARY.MR_DATE is
'Trading Date of Mandatory Restriction event; Mandatory Restriction imposition date';

comment on column BILLING_MR_SUMMARY.REGIONID is
'Unique Region Identifier';

comment on column BILLING_MR_SUMMARY.TOTAL_PAYMENTS is
'Total payments by AEMO';

comment on column BILLING_MR_SUMMARY.TOTAL_RECOVERY is
'Total payments to AEMO';

comment on column BILLING_MR_SUMMARY.TOTAL_RSA is
'Total Restriction Shortfall Amount';

comment on column BILLING_MR_SUMMARY.AAGE is
'The aggregate of then adjusted gross energy of all the market customer in the restricted region for the duration of the mandatory restriction period (MWh)';

comment on column BILLING_MR_SUMMARY.LASTCHANGED is
'Date/Time record inserted/modified';

alter table BILLING_MR_SUMMARY
   add constraint BILLING_MR_SUMMARY_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, MR_DATE, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLING_MR_SUMMARY_LCX                                */
/*==============================================================*/
create index BILLING_MR_SUMMARY_LCX on BILLING_MR_SUMMARY (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_NMAS_MANUAL_PAYMENT                           */
/*==============================================================*/
create table BILLING_NMAS_MANUAL_PAYMENT (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(4,0)           not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   CONTRACTID           VARCHAR2(20)          not null,
   DUID                 VARCHAR2(20)          not null,
   SERVICETYPE          VARCHAR2(40)          not null,
   PAYMENTTYPE          VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(10),
   PAYMENTAMOUNT        NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table BILLING_NMAS_MANUAL_PAYMENT is
'This report contains the NMAS Manual Weekly Payments that are uploaded by the Settlement Business Team. Amounts in this table are not calculated by the Settlement System.';

comment on column BILLING_NMAS_MANUAL_PAYMENT.CONTRACTYEAR is
'The Billing Contract Year.';

comment on column BILLING_NMAS_MANUAL_PAYMENT.WEEKNO is
'The Billing WeekNo';

comment on column BILLING_NMAS_MANUAL_PAYMENT.BILLRUNNO is
'The Billing RunNo';

comment on column BILLING_NMAS_MANUAL_PAYMENT.PARTICIPANTID is
'The Contract Participant Id.';

comment on column BILLING_NMAS_MANUAL_PAYMENT.CONTRACTID is
'The NMAS System Security Contract ID.';

comment on column BILLING_NMAS_MANUAL_PAYMENT.DUID is
'The DUID associated with the Contract Payment.';

comment on column BILLING_NMAS_MANUAL_PAYMENT.SERVICETYPE is
'The NMAS System Security Service Types (INERTIA, SYSTEM STRENGTH, TYPE1, TYPE2 and so on).';

comment on column BILLING_NMAS_MANUAL_PAYMENT.PAYMENTTYPE is
'The Payment Type associated with the Service like Availability, Usage, Enablement, Energy Revenue, Test, ADHOC and so on.';

comment on column BILLING_NMAS_MANUAL_PAYMENT.REGIONID is
'The Contract Region Id.';

comment on column BILLING_NMAS_MANUAL_PAYMENT.PAYMENTAMOUNT is
'The NMAS Contract Manual Payment for the Payment Type.';

comment on column BILLING_NMAS_MANUAL_PAYMENT.LASTCHANGED is
'The last changed date time of the record.';

alter table BILLING_NMAS_MANUAL_PAYMENT
   add constraint BILLING_NMAS_MANUAL_PAYMENT_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, CONTRACTID, DUID, SERVICETYPE, PAYMENTTYPE)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: BILLING_NMAS_MANUAL_RECOVERY                          */
/*==============================================================*/
create table BILLING_NMAS_MANUAL_RECOVERY (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(4,0)           not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   CONTRACTID           VARCHAR2(20)          not null,
   SERVICETYPE          VARCHAR2(40)          not null,
   PAYMENTTYPE          VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(10)          not null,
   PAYMENTAMOUNT        NUMBER(18,8),
   RECOVERYSTARTDATETIME DATE,
   RECOVERYENDDATETIME  DATE,
   RECOVERYAMOUNT_ACE   NUMBER(18,8),
   RECOVERYAMOUNT_ASOE  NUMBER(18,8),
   PARTICIPANT_ACE_MWH  NUMBER(18,8),
   PARTICIPANT_ASOE_MWH NUMBER(18,8),
   REGION_ACE_MWH       NUMBER(18,8),
   REGION_ASOE_MWH      NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table BILLING_NMAS_MANUAL_RECOVERY is
'This report shows the summary of the Billing NMAS Recovery Amounts. This table will have recovery data for manual payments for System Security Services.';

comment on column BILLING_NMAS_MANUAL_RECOVERY.CONTRACTYEAR is
'The Billing Contract Year.';

comment on column BILLING_NMAS_MANUAL_RECOVERY.WEEKNO is
'The Billing WeekNo';

comment on column BILLING_NMAS_MANUAL_RECOVERY.BILLRUNNO is
'The Billing RunNo';

comment on column BILLING_NMAS_MANUAL_RECOVERY.PARTICIPANTID is
'The Contract Participant Id.';

comment on column BILLING_NMAS_MANUAL_RECOVERY.CONTRACTID is
'The NMAS System Security Contract ID.';

comment on column BILLING_NMAS_MANUAL_RECOVERY.SERVICETYPE is
'The NMAS System Security Service Types (INERTIA, SYSTEM STRENGTH, TYPE1, TYPE2 and so on).';

comment on column BILLING_NMAS_MANUAL_RECOVERY.PAYMENTTYPE is
'The Payment Type associated with the Service like Availability, Usage, Enablement, Energy Revenue, Test, ADHOC and so on.';

comment on column BILLING_NMAS_MANUAL_RECOVERY.REGIONID is
'Region Identifier';

comment on column BILLING_NMAS_MANUAL_RECOVERY.PAYMENTAMOUNT is
'The NMAS Contract Manual Payment for the Payment Type.';

comment on column BILLING_NMAS_MANUAL_RECOVERY.RECOVERYSTARTDATETIME is
'The Recovery Start Date and Time for the Payment Calculation.';

comment on column BILLING_NMAS_MANUAL_RECOVERY.RECOVERYENDDATETIME is
'The Recovery End Date and Time for the Payment Calculation.';

comment on column BILLING_NMAS_MANUAL_RECOVERY.RECOVERYAMOUNT_ACE is
'Recovery Amount on ACE portion ($).';

comment on column BILLING_NMAS_MANUAL_RECOVERY.RECOVERYAMOUNT_ASOE is
'Recovery Amount on ASOE portion ($).';

comment on column BILLING_NMAS_MANUAL_RECOVERY.PARTICIPANT_ACE_MWH is
'Participant Consumed Energy in MWh.';

comment on column BILLING_NMAS_MANUAL_RECOVERY.PARTICIPANT_ASOE_MWH is
'Participant Sent Out Energy in MWh.';

comment on column BILLING_NMAS_MANUAL_RECOVERY.REGION_ACE_MWH is
'Region Consumed Energy in MWh.';

comment on column BILLING_NMAS_MANUAL_RECOVERY.REGION_ASOE_MWH is
'Region Sent Out Energy in MWh.';

comment on column BILLING_NMAS_MANUAL_RECOVERY.LASTCHANGED is
'The last changed date time of the record.';

alter table BILLING_NMAS_MANUAL_RECOVERY
   add constraint BILLING_NMAS_MANUAL_RECOVER_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, CONTRACTID, SERVICETYPE, PAYMENTTYPE, REGIONID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: BILLING_NMAS_TST_PAYMENTS                             */
/*==============================================================*/
create table BILLING_NMAS_TST_PAYMENTS (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR(20)           not null,
   SERVICE              VARCHAR(10)           not null,
   CONTRACTID           VARCHAR(10)           not null,
   PAYMENT_AMOUNT       NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLING_NMAS_TST_PAYMENTS is
'BILLING_NMAS_TEST_PAYMENTS publish the NSCAS/SRAS Testing Payments data for a posted billing week. This Table may also be used for NSCAS and Type 1 transitional services procured by AEMO under the ISF framework during 2025 and prior to the implementation of all system changes. During this time descriptions in these tables may not be correct.';

comment on column BILLING_NMAS_TST_PAYMENTS.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1 January';

comment on column BILLING_NMAS_TST_PAYMENTS.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1 January';

comment on column BILLING_NMAS_TST_PAYMENTS.BILLRUNNO is
'The current Billing RunNo for the week';

comment on column BILLING_NMAS_TST_PAYMENTS.PARTICIPANTID is
'The Participant from whom the amount is recovered';

comment on column BILLING_NMAS_TST_PAYMENTS.SERVICE is
'The type of NSCAS service. Current value values are:
- REACTIVE
- LOADSHED';

comment on column BILLING_NMAS_TST_PAYMENTS.CONTRACTID is
'The NMAS Contract Id';

comment on column BILLING_NMAS_TST_PAYMENTS.PAYMENT_AMOUNT is
'The Testing Payment Amount to recover';

alter table BILLING_NMAS_TST_PAYMENTS
   add constraint PK_BILLING_NMAS_TST_PAYMENTS primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, SERVICE, CONTRACTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_NMAS_TST_RECOVERY                             */
/*==============================================================*/
create table BILLING_NMAS_TST_RECOVERY (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR(20)           not null,
   SERVICE              VARCHAR(10)           not null,
   CONTRACTID           VARCHAR(10)           not null,
   REGIONID             VARCHAR(10)           not null,
   RBF                  NUMBER(18,8),
   TEST_PAYMENT         NUMBER(18,8),
   RECOVERY_START_DATE  DATE,
   RECOVERY_END_DATE    DATE,
   PARTICIPANT_ENERGY   NUMBER(18,8),
   REGION_ENERGY        NUMBER(18,8),
   NEM_ENERGY           NUMBER(18,8),
   CUSTOMER_PROPORTION  NUMBER(18,8),
   GENERATOR_PROPORTION NUMBER(18,8),
   PARTICIPANT_GENERATION NUMBER(18,8),
   NEM_GENERATION       NUMBER(18,8),
   RECOVERY_AMOUNT      NUMBER(18,8),
   LASTCHANGED          DATE,
   PARTICIPANT_ACE_MWH  NUMBER(18,8),
   REGION_ACE_MWH       NUMBER(18,8),
   ACE_PORTION          NUMBER(18,8),
   ASOE_PORTION         NUMBER(18,8),
   PARTICIPANT_ASOE_MWH NUMBER(18,8),
   REGION_ASOE_MWH      NUMBER(18,8),
   RECOVERYAMOUNT_ACE   NUMBER(18,8),
   RECOVERYAMOUNT_ASOE  NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLING_NMAS_TST_RECOVERY is
'BILLING_NMAS_TEST_RECOVERY sets out the recovery of NMAS testing payments This Table may also be used for NSCAS and Type 1 transitional services procured by AEMO under the ISF framework during 2025 and prior to the implementation of all system changes. During this time descriptions in these tables may not be correct.';

comment on column BILLING_NMAS_TST_RECOVERY.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1 January';

comment on column BILLING_NMAS_TST_RECOVERY.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1 January';

comment on column BILLING_NMAS_TST_RECOVERY.BILLRUNNO is
'The current Billing RunNo for the week';

comment on column BILLING_NMAS_TST_RECOVERY.PARTICIPANTID is
'The Participant from whom the amount is recovered';

comment on column BILLING_NMAS_TST_RECOVERY.SERVICE is
'The type of NSCAS service. Current value values are:
- REACTIVE
- LOADSHED
- RESTART';

comment on column BILLING_NMAS_TST_RECOVERY.CONTRACTID is
'The NMAS Contract Id';

comment on column BILLING_NMAS_TST_RECOVERY.REGIONID is
'The region from where the amount is recovered';

comment on column BILLING_NMAS_TST_RECOVERY.RBF is
'The Benefitting Factor for the RegionId';

comment on column BILLING_NMAS_TST_RECOVERY.TEST_PAYMENT is
'The total Testing Payment Amount to recover from all benefitting regions';

comment on column BILLING_NMAS_TST_RECOVERY.RECOVERY_START_DATE is
'The Recovery Start Date for the Testing Payment Calculation';

comment on column BILLING_NMAS_TST_RECOVERY.RECOVERY_END_DATE is
'The Recovery End Date for the Testing Payment Calculation';

comment on column BILLING_NMAS_TST_RECOVERY.PARTICIPANT_ENERGY is
'The Participant energy in MWh for the recovery period';

comment on column BILLING_NMAS_TST_RECOVERY.REGION_ENERGY is
'The RegionId energy in MWh for the recovery period';

comment on column BILLING_NMAS_TST_RECOVERY.NEM_ENERGY is
'The NEM energy in MWh for the recovery period';

comment on column BILLING_NMAS_TST_RECOVERY.CUSTOMER_PROPORTION is
'The Customer Proportion for recovery amount in Percent';

comment on column BILLING_NMAS_TST_RECOVERY.GENERATOR_PROPORTION is
'The Generator Proportion for recovery amount in Percent (100-Customer Portion)';

comment on column BILLING_NMAS_TST_RECOVERY.PARTICIPANT_GENERATION is
'The Participant Generation for the recovery period';

comment on column BILLING_NMAS_TST_RECOVERY.NEM_GENERATION is
'The NEM Generation for the recovery period';

comment on column BILLING_NMAS_TST_RECOVERY.RECOVERY_AMOUNT is
'The Total recovery amount for the billing week, being the sum of the customer and generator proportions for the PARTICIPANTID in REGIONID and sum of RecoveryAmount_ACE and RecoveryAmount_ASOE.';

comment on column BILLING_NMAS_TST_RECOVERY.LASTCHANGED is
'The Last Updated date and time';

comment on column BILLING_NMAS_TST_RECOVERY.PARTICIPANT_ACE_MWH is
'The Participant ACE MWh Value used in the Recovery of the Testing Payment Amount if the service is recovered from ACE. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLING_NMAS_TST_RECOVERY.REGION_ACE_MWH is
'The Region ACE MWh Value used in the Recovery of the Testing Payment Amount if the service is recovered from ACE. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLING_NMAS_TST_RECOVERY.ACE_PORTION is
'The Portion of ACE MWh Value used in the Recovery Calculation. . NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLING_NMAS_TST_RECOVERY.ASOE_PORTION is
'The Portion of ASOE MWh Value used in the Recovery Calculation (100 - ACE_Portion). . NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLING_NMAS_TST_RECOVERY.PARTICIPANT_ASOE_MWH is
'The Participant ASOE MWh Value used in the Recovery of the Testing Payment Amount if the service is recovered from ASOE. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLING_NMAS_TST_RECOVERY.REGION_ASOE_MWH is
'The Region ASOE MWh Value used in the Recovery of the Testing Payment Amount if the service is recovered from ASOE. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLING_NMAS_TST_RECOVERY.RECOVERYAMOUNT_ACE is
'The Participant Recovery Amount based on ACE MWh Value if the service is recovered from ACE . NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLING_NMAS_TST_RECOVERY.RECOVERYAMOUNT_ASOE is
'The Participant Recovery Amount based on ASOE MWh Value if the service is recovered from ASOE . NULL for Billing Week prior to the IESS rule effective date';

alter table BILLING_NMAS_TST_RECOVERY
   add constraint PK_BILLING_NMAS_TST_RECOVERY primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, SERVICE, CONTRACTID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLING_NMAS_TST_RECOVERY_LCX                         */
/*==============================================================*/
create index BILLING_NMAS_TST_RECOVERY_LCX on BILLING_NMAS_TST_RECOVERY (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_NMAS_TST_RECVRY_RBF                           */
/*==============================================================*/
create table BILLING_NMAS_TST_RECVRY_RBF (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   SERVICE              VARCHAR(10)           not null,
   CONTRACTID           VARCHAR(10)           not null,
   REGIONID             VARCHAR(10)           not null,
   RBF                  NUMBER(18,8),
   PAYMENT_AMOUNT       NUMBER(18,8),
   RECOVERY_AMOUNT      NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLING_NMAS_TST_RECVRY_RBF is
'BILLING_NMAS_TEST_RECVRY_RBF sets out the NSCAS/SRAS Testing Payment recovery data for the posted billing week.';

comment on column BILLING_NMAS_TST_RECVRY_RBF.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1 January';

comment on column BILLING_NMAS_TST_RECVRY_RBF.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1 January';

comment on column BILLING_NMAS_TST_RECVRY_RBF.BILLRUNNO is
'The current Billing RunNo for the week';

comment on column BILLING_NMAS_TST_RECVRY_RBF.SERVICE is
'The type of NSCAS service. Current value values are:
- REACTIVE
- LOADSHED';

comment on column BILLING_NMAS_TST_RECVRY_RBF.CONTRACTID is
'The NMAS Contract Id';

comment on column BILLING_NMAS_TST_RECVRY_RBF.REGIONID is
'The region from where the amount is recovered';

comment on column BILLING_NMAS_TST_RECVRY_RBF.RBF is
'The Benefitting Factor for the RegionId';

comment on column BILLING_NMAS_TST_RECVRY_RBF.PAYMENT_AMOUNT is
'The total Testing Payment Amount to recover from all benefitting regions';

comment on column BILLING_NMAS_TST_RECVRY_RBF.RECOVERY_AMOUNT is
'The Testing Payment amount to recover from RegionId';

comment on column BILLING_NMAS_TST_RECVRY_RBF.LASTCHANGED is
'The Last Updated date and time';

alter table BILLING_NMAS_TST_RECVRY_RBF
   add constraint PK_BILLING_NMAS_TST_RECVRY_RBF primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, SERVICE, CONTRACTID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLING_NMAS_TST_RCVRY_RBF_LCX                        */
/*==============================================================*/
create index BILLING_NMAS_TST_RCVRY_RBF_LCX on BILLING_NMAS_TST_RECVRY_RBF (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_NMAS_TST_RECVRY_TRK                           */
/*==============================================================*/
create table BILLING_NMAS_TST_RECVRY_TRK (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   RECOVERY_CONTRACTYEAR NUMBER(4,0)           not null,
   RECOVERY_WEEKNO      NUMBER(3,0)           not null,
   RECOVERY_BILLRUNNO   NUMBER(3,0)           not null
)
   tablespace PSSTAB01;

comment on table BILLING_NMAS_TST_RECVRY_TRK is
'BILLING_NMAS_TEST_RECVRY_TRK tracks the energy data used to allocate the test payment recovery over the recovery period. This Table may also be used for NSCAS and Type 1 transitional services procured by AEMO under the ISF framework during 2025 and prior to the implementation of all system changes. During this time descriptions in these tables may not be correct.';

comment on column BILLING_NMAS_TST_RECVRY_TRK.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1 January';

comment on column BILLING_NMAS_TST_RECVRY_TRK.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1 January';

comment on column BILLING_NMAS_TST_RECVRY_TRK.BILLRUNNO is
'The current Billing RunNo for the week';

comment on column BILLING_NMAS_TST_RECVRY_TRK.RECOVERY_CONTRACTYEAR is
'AEMO Contract Year for energy data used in recovery calculation';

comment on column BILLING_NMAS_TST_RECVRY_TRK.RECOVERY_WEEKNO is
'Week no for energy data used in recovery calculation';

comment on column BILLING_NMAS_TST_RECVRY_TRK.RECOVERY_BILLRUNNO is
'Billing RunNo for energy data used in recovery calculation';

alter table BILLING_NMAS_TST_RECVRY_TRK
   add constraint PK_BILLING_NMAS_TST_RECVRY_TRK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, RECOVERY_CONTRACTYEAR, RECOVERY_WEEKNO, RECOVERY_BILLRUNNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_RES_TRADER_PAYMENT                            */
/*==============================================================*/
create table BILLING_RES_TRADER_PAYMENT (
   CONTRACTYEAR         NUMBER(4)             not null,
   WEEKNO               NUMBER(3)             not null,
   BILLRUNNO            NUMBER(3)             not null,
   CONTRACTID           VARCHAR2(20)          not null,
   PAYMENT_TYPE         VARCHAR2(40)          not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   PAYMENT_AMOUNT       NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLING_RES_TRADER_PAYMENT is
'Billing result table for reserve trader contract payments';

comment on column BILLING_RES_TRADER_PAYMENT.CONTRACTYEAR is
'Billing contract year';

comment on column BILLING_RES_TRADER_PAYMENT.WEEKNO is
'Billing week number';

comment on column BILLING_RES_TRADER_PAYMENT.BILLRUNNO is
'Billing run number';

comment on column BILLING_RES_TRADER_PAYMENT.CONTRACTID is
'Reserve trader contract identifier';

comment on column BILLING_RES_TRADER_PAYMENT.PAYMENT_TYPE is
'Payment type for the reserve trader contract payment amount';

comment on column BILLING_RES_TRADER_PAYMENT.PARTICIPANTID is
'Participant identifier associated with the contract';

comment on column BILLING_RES_TRADER_PAYMENT.PAYMENT_AMOUNT is
'Payment amount to the participant';

alter table BILLING_RES_TRADER_PAYMENT
   add constraint BILLING_RES_TRADER_PAYMENT_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, CONTRACTID, PAYMENT_TYPE, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_RES_TRADER_RECOVERY                           */
/*==============================================================*/
create table BILLING_RES_TRADER_RECOVERY (
   CONTRACTYEAR         NUMBER(4)             not null,
   WEEKNO               NUMBER(3)             not null,
   BILLRUNNO            NUMBER(3)             not null,
   REGIONID             VARCHAR2(20)          not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   RECOVERY_AMOUNT      NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLING_RES_TRADER_RECOVERY is
'Billing result table for reserve trader contract recovery';

comment on column BILLING_RES_TRADER_RECOVERY.CONTRACTYEAR is
'Billing contract year';

comment on column BILLING_RES_TRADER_RECOVERY.WEEKNO is
'Billing week number';

comment on column BILLING_RES_TRADER_RECOVERY.BILLRUNNO is
'Billing run number';

comment on column BILLING_RES_TRADER_RECOVERY.REGIONID is
'Region id for the aggregated recovery amount';

comment on column BILLING_RES_TRADER_RECOVERY.PARTICIPANTID is
'Participant identifier';

comment on column BILLING_RES_TRADER_RECOVERY.RECOVERY_AMOUNT is
'Payment amount to be recovered from the participant';

alter table BILLING_RES_TRADER_RECOVERY
   add constraint BILLING_RES_TRADER_RECOVERY_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, REGIONID, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_SECDEPOSIT_APPLICATION                        */
/*==============================================================*/
create table BILLING_SECDEPOSIT_APPLICATION (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   APPLICATION_AMOUNT   NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLING_SECDEPOSIT_APPLICATION is
'The security deposit application details';

comment on column BILLING_SECDEPOSIT_APPLICATION.CONTRACTYEAR is
'The billing contract year where (security deposit application) SDA is applied';

comment on column BILLING_SECDEPOSIT_APPLICATION.WEEKNO is
'The billing week no. where the SDA is applied';

comment on column BILLING_SECDEPOSIT_APPLICATION.BILLRUNNO is
'The billing run no. where the SDA is applied';

comment on column BILLING_SECDEPOSIT_APPLICATION.PARTICIPANTID is
'The Participant ID lodging the SDA';

comment on column BILLING_SECDEPOSIT_APPLICATION.APPLICATION_AMOUNT is
'The SDA application amount';

alter table BILLING_SECDEPOSIT_APPLICATION
   add constraint BILLING_SECDEPOSIT_APPL_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_SECDEP_INTEREST_PAY                           */
/*==============================================================*/
create table BILLING_SECDEP_INTEREST_PAY (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   SECURITY_DEPOSIT_ID  VARCHAR2(20)          not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   INTEREST_AMOUNT      NUMBER(18,8),
   INTEREST_CALC_TYPE   VARCHAR2(20),
   INTEREST_ACCT_ID     VARCHAR2(20),
   INTEREST_RATE        NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLING_SECDEP_INTEREST_PAY is
'The interest amount for security deposit calculated by billing, based on whether it is a fixed/floating rate';

comment on column BILLING_SECDEP_INTEREST_PAY.CONTRACTYEAR is
'The billing contract year the SDA application is processed and interest calculated';

comment on column BILLING_SECDEP_INTEREST_PAY.WEEKNO is
'The billing week no. the SDA application is processed and interest calculated';

comment on column BILLING_SECDEP_INTEREST_PAY.BILLRUNNO is
'The billing run no. the SDA application is processed and interest calculated';

comment on column BILLING_SECDEP_INTEREST_PAY.SECURITY_DEPOSIT_ID is
'The security deposit ID for which billing has calculated the Interest amount';

comment on column BILLING_SECDEP_INTEREST_PAY.PARTICIPANTID is
'The participant ID of the security deposit for whom the interest is paid';

comment on column BILLING_SECDEP_INTEREST_PAY.INTEREST_AMOUNT is
'The security deposit interest amount calculated by billing';

comment on column BILLING_SECDEP_INTEREST_PAY.INTEREST_CALC_TYPE is
'FIXED or DAILY';

comment on column BILLING_SECDEP_INTEREST_PAY.INTEREST_ACCT_ID is
'The interest account ID used by billing for calculating the interest. 
NULL if INTEREST_CALC_TYPE = FIXED';

comment on column BILLING_SECDEP_INTEREST_PAY.INTEREST_RATE is
'The STATIC Interest Rate used by Billing for calculating the interest. This is NULL if INTEREST_CALC_TYPE <> FIXED';

alter table BILLING_SECDEP_INTEREST_PAY
   add constraint BILLING_SECDEP_INTEREST_PAY_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, SECURITY_DEPOSIT_ID, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_SECDEP_INTEREST_RATE                          */
/*==============================================================*/
create table BILLING_SECDEP_INTEREST_RATE (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   INTEREST_ACCT_ID     VARCHAR2(20)          not null,
   EFFECTIVEDATE        DATE                  not null,
   INTEREST_RATE        NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLING_SECDEP_INTEREST_RATE is
'The DAILY interest rates used by billing when calculating the interest amount';

comment on column BILLING_SECDEP_INTEREST_RATE.CONTRACTYEAR is
'The billing contract year the SDA application is processed and interest calculated';

comment on column BILLING_SECDEP_INTEREST_RATE.WEEKNO is
'The billing week no. the SDA application is processed and interest calculated';

comment on column BILLING_SECDEP_INTEREST_RATE.BILLRUNNO is
'The billing run no. the SDA application is processed and interest calculated';

comment on column BILLING_SECDEP_INTEREST_RATE.INTEREST_ACCT_ID is
'The interest account ID used by security deposit interest calculation';

comment on column BILLING_SECDEP_INTEREST_RATE.EFFECTIVEDATE is
'The effective date of the new interest change';

comment on column BILLING_SECDEP_INTEREST_RATE.INTEREST_RATE is
'The interest rate to apply from the effective date';

alter table BILLING_SECDEP_INTEREST_RATE
   add constraint BILL_SECDEP_INTEREST_RATE_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, INTEREST_ACCT_ID, EFFECTIVEDATE)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_SUBST_DEMAND                                  */
/*==============================================================*/
create table BILLING_SUBST_DEMAND (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   SETTLEMENTDATE       DATE                  not null,
   TNI                  VARCHAR2(20)          not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20),
   SUBSTITUTEDEMAND     NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLING_SUBST_DEMAND is
'Demand Values Substituted in Billing Calculation';

comment on column BILLING_SUBST_DEMAND.CONTRACTYEAR is
'Billing contract year';

comment on column BILLING_SUBST_DEMAND.WEEKNO is
'Billing week number';

comment on column BILLING_SUBST_DEMAND.BILLRUNNO is
'Billing run number';

comment on column BILLING_SUBST_DEMAND.SETTLEMENTDATE is
'Settlement Date';

comment on column BILLING_SUBST_DEMAND.TNI is
'Unique identifier for the connection point';

comment on column BILLING_SUBST_DEMAND.PARTICIPANTID is
'Unique identifier for the participant';

comment on column BILLING_SUBST_DEMAND.REGIONID is
'Unique identifier for the region to which the TNI belongs to on this settlement date';

comment on column BILLING_SUBST_DEMAND.SUBSTITUTEDEMAND is
'Substitute metered quantity for non-energy recovery in MWh for the TNI and participant in the trading interval. A negative value indicates net consumption and a positive value indicates net generation';

alter table BILLING_SUBST_DEMAND
   add constraint BILLING_SUBST_DEMAND_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, SETTLEMENTDATE, TNI, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_SUBST_RUN_VERSION                             */
/*==============================================================*/
create table BILLING_SUBST_RUN_VERSION (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   REFERENCESETTLEMENTDATE DATE                  not null,
   REFERENCESETTLEMENTRUNNO NUMBER(3,0)           not null
)
   tablespace PSSTAB01;

comment on table BILLING_SUBST_RUN_VERSION is
'Details of settlement runs used as input in the substitute demand calculation';

comment on column BILLING_SUBST_RUN_VERSION.CONTRACTYEAR is
'Billing contract year';

comment on column BILLING_SUBST_RUN_VERSION.WEEKNO is
'Billing week number';

comment on column BILLING_SUBST_RUN_VERSION.BILLRUNNO is
'Billing run number';

comment on column BILLING_SUBST_RUN_VERSION.REFERENCESETTLEMENTDATE is
'Settlement Date';

comment on column BILLING_SUBST_RUN_VERSION.REFERENCESETTLEMENTRUNNO is
'The settlement run number matching the settlement date for a settlement run included in the reference period';

alter table BILLING_SUBST_RUN_VERSION
   add constraint BILLING_SUBST_RUN_VERSION_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, REFERENCESETTLEMENTDATE, REFERENCESETTLEMENTRUNNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_WDR                                           */
/*==============================================================*/
create table BILLING_WDR (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(20)          not null,
   WDR_CREDIT_AMOUNT    NUMBER(18,8),
   WDR_DEBIT_AMOUNT     NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLING_WDR is
'Billing WDR Transaction Weekly Summary';

comment on column BILLING_WDR.CONTRACTYEAR is
'Contract year of the Billing run';

comment on column BILLING_WDR.WEEKNO is
'Week number of the Billing run';

comment on column BILLING_WDR.BILLRUNNO is
'Billing run number identifier';

comment on column BILLING_WDR.PARTICIPANTID is
'DRSP or FRMP Participant Identifier';

comment on column BILLING_WDR.WDR_CREDIT_AMOUNT is
'WDR credit transaction amount';

comment on column BILLING_WDR.WDR_DEBIT_AMOUNT is
'WDR debit transaction amount';

alter table BILLING_WDR
   add constraint BILLING_WDR_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLING_WDR_DETAIL                                    */
/*==============================================================*/
create table BILLING_WDR_DETAIL (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   WDRRRPERIOD          VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   FRMP                 VARCHAR2(20)          not null,
   DRSP                 VARCHAR2(20)          not null,
   WDRSQ                NUMBER(18,8),
   WDRRR                NUMBER(18,8),
   WDRTA                NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table BILLING_WDR_DETAIL is
'Billing WDR transaction detail summary';

comment on column BILLING_WDR_DETAIL.CONTRACTYEAR is
'Contract year of the Billing run';

comment on column BILLING_WDR_DETAIL.WEEKNO is
'Week number of the Billing run';

comment on column BILLING_WDR_DETAIL.BILLRUNNO is
'Billing run number identifier';

comment on column BILLING_WDR_DETAIL.WDRRRPERIOD is
'Unique identifier for the period to which the WDRRR applies. For quarter-based periods, this will be equal to YYYY[Q]NN, for example, 2020Q3 for 2020 Quarter 3.';

comment on column BILLING_WDR_DETAIL.REGIONID is
'Region identifier';

comment on column BILLING_WDR_DETAIL.FRMP is
'Financial Responsible Market Participant Identifier';

comment on column BILLING_WDR_DETAIL.DRSP is
'Demand Response Service Provider Identifier';

comment on column BILLING_WDR_DETAIL.WDRSQ is
'WDR Settlement Quantity capped in MWh';

comment on column BILLING_WDR_DETAIL.WDRRR is
'WDR reimbursement rate in $/MWh';

comment on column BILLING_WDR_DETAIL.WDRTA is
'WDR transaction amount in $ for demand response';

alter table BILLING_WDR_DETAIL
   add constraint BILLING_WDR_DETAIL_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, WDRRRPERIOD, REGIONID, FRMP, DRSP)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINTERVENTIONRECOVERY                              */
/*==============================================================*/
create table BILLINTERVENTIONRECOVERY (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   MARKETINTERVENTION   NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLINTERVENTIONRECOVERY is
'BILLINTERVENTIONRECOVERY shows billing market intervention recovery details for each participant.';

comment on column BILLINTERVENTIONRECOVERY.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINTERVENTIONRECOVERY.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINTERVENTIONRECOVERY.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINTERVENTIONRECOVERY.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINTERVENTIONRECOVERY.MARKETINTERVENTION is
'Amount Retailer pays for Intervention with Market Recovery';

comment on column BILLINTERVENTIONRECOVERY.LASTCHANGED is
'Last date and time record changed';

alter table BILLINTERVENTIONRECOVERY
   add constraint BILLINTERVENTIONRECOVERY_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINTERVENTIONRECOVERY_LCX                          */
/*==============================================================*/
create index BILLINTERVENTIONRECOVERY_LCX on BILLINTERVENTIONRECOVERY (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLINTERVENTIONREGIONRECOVERY                        */
/*==============================================================*/
create table BILLINTERVENTIONREGIONRECOVERY (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   REGIONID             VARCHAR2(10)          not null,
   REGIONINTERVENTION   NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLINTERVENTIONREGIONRECOVERY is
'BILLINTERVENTIONREGIONRECOVERY shows billing region intervention recovery details for each participant by region.';

comment on column BILLINTERVENTIONREGIONRECOVERY.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLINTERVENTIONREGIONRECOVERY.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLINTERVENTIONREGIONRECOVERY.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLINTERVENTIONREGIONRECOVERY.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLINTERVENTIONREGIONRECOVERY.REGIONID is
'Region Identifier';

comment on column BILLINTERVENTIONREGIONRECOVERY.REGIONINTERVENTION is
'Amount retailer pays for intervention with Region Recovery';

comment on column BILLINTERVENTIONREGIONRECOVERY.LASTCHANGED is
'Last date and time record changed';

alter table BILLINTERVENTIONREGIONRECOVERY
   add constraint BILLINTERVENTIONREGIONRECOV_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLINTERVENTIONREGIONREC_LCX                         */
/*==============================================================*/
create index BILLINTERVENTIONREGIONREC_LCX on BILLINTERVENTIONREGIONRECOVERY (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLRESERVETRADERPAYMENT                              */
/*==============================================================*/
create table BILLRESERVETRADERPAYMENT (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(20),
   CONTRACTID           VARCHAR2(20)          not null,
   PAYMENT_ID           NUMBER(3,0)           not null,
   PAYMENT_TYPE         VARCHAR2(40),
   PAYMENT_AMOUNT       NUMBER(18,8)
)
   tablespace PSSTAB02;

comment on table BILLRESERVETRADERPAYMENT is
'Details of the RERT Usage and Availability Payments made to the participant.';

comment on column BILLRESERVETRADERPAYMENT.CONTRACTYEAR is
'Billing contract year';

comment on column BILLRESERVETRADERPAYMENT.WEEKNO is
'Billing week number';

comment on column BILLRESERVETRADERPAYMENT.BILLRUNNO is
'Billing posted run number';

comment on column BILLRESERVETRADERPAYMENT.PARTICIPANTID is
'Participant identifier.';

comment on column BILLRESERVETRADERPAYMENT.CONTRACTID is
'RERT payment contract ID';

comment on column BILLRESERVETRADERPAYMENT.PAYMENT_ID is
'RERT payment number';

comment on column BILLRESERVETRADERPAYMENT.PAYMENT_TYPE is
'Description for the reserve trader contract payment amount.';

comment on column BILLRESERVETRADERPAYMENT.PAYMENT_AMOUNT is
'RERT payment amount for the payment type';

alter table BILLRESERVETRADERPAYMENT
   add constraint BILLRESERVETRADERPAYMENT_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, CONTRACTID, PAYMENT_ID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: BILLRESERVETRADERRECOVERY                             */
/*==============================================================*/
create table BILLRESERVETRADERRECOVERY (
   CONTRACTYEAR         NUMBER(4,0)           not null,
   WEEKNO               NUMBER(3,0)           not null,
   BILLRUNNO            NUMBER(3,0)           not null,
   PUBLICATION_ID       VARCHAR2(40)          not null,
   PAYMENT_ID           NUMBER(3,0)           not null,
   PAYMENT_AMOUNT       NUMBER(18,8),
   PARTICIPANTID        VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   PARTICIPANT_DEMAND   NUMBER(18,8),
   REGION_DEMAND        NUMBER(18,8),
   ELIGIBILITY_START_INTERVAL DATE,
   ELIGIBILITY_END_INTERVAL DATE,
   RECOVERY_AMOUNT      NUMBER(18,8),
   EXCLUDED_ENERGY      NUMBER(18,8),
   PARTICIPANT_ACE_MWH  NUMBER(18,8),
   REGION_ACE_MWH       NUMBER(18,8)
)
   tablespace PSSTAB02;

comment on table BILLRESERVETRADERRECOVERY is
'Provides details of the RERT Recovery Amount for the Market Customers.';

comment on column BILLRESERVETRADERRECOVERY.CONTRACTYEAR is
'Billing contract year';

comment on column BILLRESERVETRADERRECOVERY.WEEKNO is
'Billing week number';

comment on column BILLRESERVETRADERRECOVERY.BILLRUNNO is
'Billing posted run number';

comment on column BILLRESERVETRADERRECOVERY.PUBLICATION_ID is
'Unique Publication Identifier for RERT Payment';

comment on column BILLRESERVETRADERRECOVERY.PAYMENT_ID is
'RERT payment number';

comment on column BILLRESERVETRADERRECOVERY.PAYMENT_AMOUNT is
'RERT payment amount';

comment on column BILLRESERVETRADERRECOVERY.PARTICIPANTID is
'Participant identifier.';

comment on column BILLRESERVETRADERRECOVERY.REGIONID is
'Region from which the amount is recovered';

comment on column BILLRESERVETRADERRECOVERY.PARTICIPANT_DEMAND is
'Participant Demand Value used for RERT Recovery. NULL for Billing Week post the IESS rule effective date.';

comment on column BILLRESERVETRADERRECOVERY.REGION_DEMAND is
'Region Demand Value used for RERT Recovery. NULL for Billing Week post the IESS rule effective date.';

comment on column BILLRESERVETRADERRECOVERY.ELIGIBILITY_START_INTERVAL is
'Starting Period of RERT Recovery for Usage Payments';

comment on column BILLRESERVETRADERRECOVERY.ELIGIBILITY_END_INTERVAL is
'Ending Period of RERT Recovery for Usage Payments';

comment on column BILLRESERVETRADERRECOVERY.RECOVERY_AMOUNT is
'Recovery Amount applicable for each Market Customer';

comment on column BILLRESERVETRADERRECOVERY.EXCLUDED_ENERGY is
'The Energy Value (Scheduled Loads) that is excluded';

comment on column BILLRESERVETRADERRECOVERY.PARTICIPANT_ACE_MWH is
'The Participant ACE MWh Value used in the Recovery of the RERT Amount. NULL for Billing Week prior to the IESS rule effective date';

comment on column BILLRESERVETRADERRECOVERY.REGION_ACE_MWH is
'The Region ACE MWh Value used in the Recovery of the RERT Amount. NULL for Billing Week prior to the IESS rule effective date';

alter table BILLRESERVETRADERRECOVERY
   add constraint BILLRESERVETRADERRECOVERY_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PUBLICATION_ID, PARTICIPANTID, REGIONID, PAYMENT_ID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: BILLSMELTERRATE                                       */
/*==============================================================*/
create table BILLSMELTERRATE (
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   CONTRACTYEAR         NUMBER(22,0)          not null,
   RAR1                 NUMBER(6,2),
   RAR2                 NUMBER(6,2),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(10),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table BILLSMELTERRATE is
'BILLSMELTERRATE is standing data, setting out the rates used in smelter reduction calculations.';

comment on column BILLSMELTERRATE.EFFECTIVEDATE is
'Calendar settlement date record becomes effective';

comment on column BILLSMELTERRATE.VERSIONNO is
'Version no of the record for the given effective date';

comment on column BILLSMELTERRATE.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLSMELTERRATE.RAR1 is
'Smelter rate 1';

comment on column BILLSMELTERRATE.RAR2 is
'Smelter rate 2';

comment on column BILLSMELTERRATE.AUTHORISEDDATE is
'Authorised date';

comment on column BILLSMELTERRATE.AUTHORISEDBY is
'Who authorised';

comment on column BILLSMELTERRATE.LASTCHANGED is
'Last date and time record changed';

alter table BILLSMELTERRATE
   add constraint BILLSMELTERRATE_PK primary key (EFFECTIVEDATE, VERSIONNO, CONTRACTYEAR)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLSMELTERRATE_LCX                                   */
/*==============================================================*/
create index BILLSMELTERRATE_LCX on BILLSMELTERRATE (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: BILLWHITEHOLE                                         */
/*==============================================================*/
create table BILLWHITEHOLE (
   CONTRACTYEAR         NUMBER(22,0)          not null,
   WEEKNO               NUMBER(22,0)          not null,
   BILLRUNNO            NUMBER(22,0)          not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   NL                   NUMBER(15,6),
   PARTICIPANTDEMAND    NUMBER(15,6),
   REGIONDEMAND         NUMBER(15,6),
   WHITEHOLEPAYMENT     NUMBER(15,6),
   LASTCHANGED          DATE,
   INTERCONNECTORID     VARCHAR2(10)          not null
)
   tablespace PSSTAB01;

comment on table BILLWHITEHOLE is
'BILLWHITEHOLE shows white hole payments based on participant vs region demand.';

comment on column BILLWHITEHOLE.CONTRACTYEAR is
'AEMO Contract Year number starting in week containing 1st January';

comment on column BILLWHITEHOLE.WEEKNO is
'Week no within the contract year. Week no 1 is the week containing 1st January';

comment on column BILLWHITEHOLE.BILLRUNNO is
'Unique run no within a given contract year and week no';

comment on column BILLWHITEHOLE.PARTICIPANTID is
'Unique participant identifier';

comment on column BILLWHITEHOLE.NL is
'Sum of billing week (RRP * interconnector flow)';

comment on column BILLWHITEHOLE.PARTICIPANTDEMAND is
'The sum of all customer purchases in MWh';

comment on column BILLWHITEHOLE.REGIONDEMAND is
'Sum of all region purchases in MWh';

comment on column BILLWHITEHOLE.WHITEHOLEPAYMENT is
'Payment in $';

comment on column BILLWHITEHOLE.LASTCHANGED is
'The latest date and time that a file was updated or inserted';

comment on column BILLWHITEHOLE.INTERCONNECTORID is
'Interconnector ID';

alter table BILLWHITEHOLE
   add constraint BILLWHITEHOLE_PK primary key (CONTRACTYEAR, WEEKNO, BILLRUNNO, PARTICIPANTID, INTERCONNECTORID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: BILLWHITEHOLE_LCX                                     */
/*==============================================================*/
create index BILLWHITEHOLE_LCX on BILLWHITEHOLE (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: CONNECTIONPOINT                                       */
/*==============================================================*/
create table CONNECTIONPOINT (
   CONNECTIONPOINTID    VARCHAR2(10)          not null,
   CONNECTIONPOINTNAME  VARCHAR2(80),
   CONNECTIONPOINTTYPE  VARCHAR2(20),
   ADDRESS1             VARCHAR2(80),
   ADDRESS2             VARCHAR2(80),
   ADDRESS3             VARCHAR2(80),
   ADDRESS4             VARCHAR2(80),
   CITY                 VARCHAR2(40),
   STATE                VARCHAR2(10),
   POSTCODE             VARCHAR2(10),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table CONNECTIONPOINT is
'CONNECTIONPOINT shows all valid connection points and their type. Transmission loss factors are available for all connection points in TRANSMISSIONLOSSFACTOR.';

comment on column CONNECTIONPOINT.CONNECTIONPOINTID is
'Connection Point Identifier';

comment on column CONNECTIONPOINT.CONNECTIONPOINTNAME is
'Connection point full description';

comment on column CONNECTIONPOINT.CONNECTIONPOINTTYPE is
'Connection point type. transmission, distribution, station, genunit, or interconnector';

comment on column CONNECTIONPOINT.ADDRESS1 is
'Connection point location';

comment on column CONNECTIONPOINT.ADDRESS2 is
'Connection point location';

comment on column CONNECTIONPOINT.ADDRESS3 is
'Connection point location';

comment on column CONNECTIONPOINT.ADDRESS4 is
'Not Used';

comment on column CONNECTIONPOINT.CITY is
'City';

comment on column CONNECTIONPOINT.STATE is
'State of Australia';

comment on column CONNECTIONPOINT.POSTCODE is
'Post Code';

comment on column CONNECTIONPOINT.LASTCHANGED is
'Last date and time record changed';

alter table CONNECTIONPOINT
   add constraint CONNECTIONPOINT_PK primary key (CONNECTIONPOINTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: CONNECTIONPOINT_LCX                                   */
/*==============================================================*/
create index CONNECTIONPOINT_LCX on CONNECTIONPOINT (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: CONNECTIONPOINTDETAILS                                */
/*==============================================================*/
create table CONNECTIONPOINTDETAILS (
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   CONNECTIONPOINTID    VARCHAR2(10)          not null,
   REGIONID             VARCHAR2(10),
   TRANSMISSIONCPTID    VARCHAR2(10),
   METERDATAPROVIDER    VARCHAR2(10),
   TRANSMISSIONLOSSFACTOR NUMBER(7,5),
   DISTRIBUTIONLOSSFACTOR NUMBER(7,5),
   NETWORKSERVICEPROVIDER VARCHAR2(10),
   FINRESPORGAN         VARCHAR2(10),
   NATIONALMETERINSTALLID NUMBER(7,5),
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE,
   INUSE                VARCHAR2(1),
   LNSP                 VARCHAR2(10),
   MDA                  VARCHAR2(10),
   ROLR                 VARCHAR2(10),
   RP                   VARCHAR2(10),
   AGGREGATEDDATA       VARCHAR2(1),
   VALID_TODATE         DATE,
   LR                   VARCHAR2(10)
)
   tablespace PSSTAB01;

comment on table CONNECTIONPOINTDETAILS is
'CONNECTIONPOINTDETAILS is obsolete, since it was never populated by Participants accessing AEMO''s Oracle Interface.
CONNECTIONPOINTDETAILS was designed to show relevant details for each connection point including the responsible party, loss factor and relevant MDAs.';

comment on column CONNECTIONPOINTDETAILS.EFFECTIVEDATE is
'Effective date of record';

comment on column CONNECTIONPOINTDETAILS.VERSIONNO is
'Version no of record for given effective date';

comment on column CONNECTIONPOINTDETAILS.CONNECTIONPOINTID is
'Connection point identifier';

comment on column CONNECTIONPOINTDETAILS.REGIONID is
'Region Identifier';

comment on column CONNECTIONPOINTDETAILS.TRANSMISSIONCPTID is
'Associated transmission connection point id for a distribution connection point';

comment on column CONNECTIONPOINTDETAILS.METERDATAPROVIDER is
'The MDA providing meter data for this connection point';

comment on column CONNECTIONPOINTDETAILS.TRANSMISSIONLOSSFACTOR is
'The transmission level loss factor for this connection point';

comment on column CONNECTIONPOINTDETAILS.DISTRIBUTIONLOSSFACTOR is
'The distribution level loss factor for a distribution connection point';

comment on column CONNECTIONPOINTDETAILS.NETWORKSERVICEPROVIDER is
'The Network Service Provider';

comment on column CONNECTIONPOINTDETAILS.FINRESPORGAN is
'Financially responsible organisation';

comment on column CONNECTIONPOINTDETAILS.NATIONALMETERINSTALLID is
'National Meter Id';

comment on column CONNECTIONPOINTDETAILS.AUTHORISEDBY is
'User authorising record';

comment on column CONNECTIONPOINTDETAILS.AUTHORISEDDATE is
'Date record authorised';

comment on column CONNECTIONPOINTDETAILS.LASTCHANGED is
'Last date and time record changed';

comment on column CONNECTIONPOINTDETAILS.INUSE is
'Status flag.';

comment on column CONNECTIONPOINTDETAILS.LNSP is
'Local Electricity Network Service Provider';

comment on column CONNECTIONPOINTDETAILS.MDA is
'Metering Data Agent for connection point.';

comment on column CONNECTIONPOINTDETAILS.ROLR is
'Retailer of last resort.';

comment on column CONNECTIONPOINTDETAILS.RP is
'Responsible party.';

comment on column CONNECTIONPOINTDETAILS.AGGREGATEDDATA is
'Aggregate flag.';

comment on column CONNECTIONPOINTDETAILS.VALID_TODATE is
'Date of validity.';

comment on column CONNECTIONPOINTDETAILS.LR is
'Local Retailer';

alter table CONNECTIONPOINTDETAILS
   add constraint CONNECTIONPOINTDETAILS_PK primary key (EFFECTIVEDATE, VERSIONNO, CONNECTIONPOINTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: CONNECTIONPOINTDETAILS_LCX                            */
/*==============================================================*/
create index CONNECTIONPOINTDETAILS_LCX on CONNECTIONPOINTDETAILS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: CONNECTIONPOINTDETAI_NDX2                             */
/*==============================================================*/
create index CONNECTIONPOINTDETAI_NDX2 on CONNECTIONPOINTDETAILS (
   METERDATAPROVIDER ASC,
   NETWORKSERVICEPROVIDER ASC,
   FINRESPORGAN ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: CONNECTIONPOINTDETAI_NDX3                             */
/*==============================================================*/
create index CONNECTIONPOINTDETAI_NDX3 on CONNECTIONPOINTDETAILS (
   CONNECTIONPOINTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: CONNECTIONPOINTOPERATINGSTA                           */
/*==============================================================*/
create table CONNECTIONPOINTOPERATINGSTA (
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   CONNECTIONPOINTID    VARCHAR2(10)          not null,
   OPERATINGSTATUS      VARCHAR2(16),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table CONNECTIONPOINTOPERATINGSTA is
'CONNECTIONPOINTOPERATINGSTA shows whether a connection point is active or not.';

comment on column CONNECTIONPOINTOPERATINGSTA.EFFECTIVEDATE is
'Effective date of record';

comment on column CONNECTIONPOINTOPERATINGSTA.CONNECTIONPOINTID is
'Connection point identifier';

comment on column CONNECTIONPOINTOPERATINGSTA.OPERATINGSTATUS is
'Active or inactive indicator';

comment on column CONNECTIONPOINTOPERATINGSTA.AUTHORISEDDATE is
'Date record authorised';

comment on column CONNECTIONPOINTOPERATINGSTA.AUTHORISEDBY is
'User authorising record';

comment on column CONNECTIONPOINTOPERATINGSTA.LASTCHANGED is
'Last date and time record changed';

alter table CONNECTIONPOINTOPERATINGSTA
   add constraint CPOPSTATUS_PK primary key (EFFECTIVEDATE, VERSIONNO, CONNECTIONPOINTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: CONNECTIONPOINTOPERA_NDX2                             */
/*==============================================================*/
create index CONNECTIONPOINTOPERA_NDX2 on CONNECTIONPOINTOPERATINGSTA (
   CONNECTIONPOINTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: CONNECTIONPOINTOPERATINGS_LCX                         */
/*==============================================================*/
create index CONNECTIONPOINTOPERATINGS_LCX on CONNECTIONPOINTOPERATINGSTA (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: CONSTRAINTRELAXATION_OCD                              */
/*==============================================================*/
create table CONSTRAINTRELAXATION_OCD (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0)           not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   RHS                  NUMBER(16,6),
   LASTCHANGED          DATE,
   VERSIONNO            NUMBER(3,0)          default 1  not null
)
   tablespace PSSTAB01;

comment on table CONSTRAINTRELAXATION_OCD is
'CONSTRAINTRELAXATION_OCD contains details of interconnector constraints and unit ancillary service constraints relaxed in the over-constrained dispatch (OCD) re-run for this interval (if there was one).
Note: INTERVENTION is not included in CONSTRAINTRELAXATION_OCD, since the relaxation of the same constraint is the same amount in both intervened and non-intervened cases.';

comment on column CONSTRAINTRELAXATION_OCD.SETTLEMENTDATE is
'End date and time of the dispatch interval';

comment on column CONSTRAINTRELAXATION_OCD.RUNNO is
'Dispatch run no';

comment on column CONSTRAINTRELAXATION_OCD.CONSTRAINTID is
'Constraint identifier';

comment on column CONSTRAINTRELAXATION_OCD.RHS is
'Relaxed RHS used in attempt to avoid constraint violation';

comment on column CONSTRAINTRELAXATION_OCD.LASTCHANGED is
'Last date and time record changed';

comment on column CONSTRAINTRELAXATION_OCD.VERSIONNO is
'Version Number';

alter table CONSTRAINTRELAXATION_OCD
   add constraint PK_CONSTRAINTRELAXATION_OCD primary key (SETTLEMENTDATE, RUNNO, CONSTRAINTID, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: CONSTRAINTRELAX_OCD_LCX                               */
/*==============================================================*/
create index CONSTRAINTRELAX_OCD_LCX on CONSTRAINTRELAXATION_OCD (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: CONTRACTAGC                                           */
/*==============================================================*/
create table CONTRACTAGC (
   CONTRACTID           VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   STARTDATE            DATE,
   ENDDATE              DATE,
   PARTICIPANTID        VARCHAR2(10),
   DUID                 VARCHAR2(10),
   CRR                  NUMBER(4,0),
   CRL                  NUMBER(4,0),
   RLPRICE              NUMBER(10,2),
   CCPRICE              NUMBER(10,2),
   BS                   NUMBER(10,2),
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table CONTRACTAGC is
'CONTRACTAGC shows Automatic Generation Control (AGC) contract details for each dispatchable unit. There is a separate contract for each unit.';

comment on column CONTRACTAGC.CONTRACTID is
'Contract Identifier';

comment on column CONTRACTAGC.VERSIONNO is
'Contract Version No';

comment on column CONTRACTAGC.STARTDATE is
'Starting Date of Contract';

comment on column CONTRACTAGC.ENDDATE is
'End date of contract';

comment on column CONTRACTAGC.PARTICIPANTID is
'Unique participant identifier';

comment on column CONTRACTAGC.DUID is
'Dispatchable Unit ID';

comment on column CONTRACTAGC.CRR is
'Control Range Raise 5 Min MW';

comment on column CONTRACTAGC.CRL is
'Control Range Lower 5 Min MW';

comment on column CONTRACTAGC.RLPRICE is
'Enabling Price in $';

comment on column CONTRACTAGC.CCPRICE is
'Compensation Cap in $';

comment on column CONTRACTAGC.BS is
'Block Size';

comment on column CONTRACTAGC.AUTHORISEDBY is
'User Name';

comment on column CONTRACTAGC.AUTHORISEDDATE is
'Date Contract was Authorised';

comment on column CONTRACTAGC.LASTCHANGED is
'Last date and time record changed';

alter table CONTRACTAGC
   add constraint CONTRACTAGC_PK primary key (CONTRACTID, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTAGC_NDX2                                      */
/*==============================================================*/
create index CONTRACTAGC_NDX2 on CONTRACTAGC (
   PARTICIPANTID ASC,
   CONTRACTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTAGC_LCX                                       */
/*==============================================================*/
create index CONTRACTAGC_LCX on CONTRACTAGC (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: CONTRACTGOVERNOR                                      */
/*==============================================================*/
create table CONTRACTGOVERNOR (
   CONTRACTID           VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   STARTDATE            DATE,
   ENDDATE              DATE,
   PARTICIPANTID        VARCHAR2(10),
   DUID                 VARCHAR2(10),
   CCPRICE              NUMBER(10,2),
   LOWER60SECBREAKPOINT NUMBER(9,6),
   LOWER60SECMAX        NUMBER(9,6),
   LOWER6SECBREAKPOINT  NUMBER(9,6),
   LOWER6SECMAX         NUMBER(9,6),
   RAISE60SECBREAKPOINT NUMBER(9,6),
   RAISE60SECCAPACITY   NUMBER(9,6),
   RAISE60SECMAX        NUMBER(9,6),
   RAISE6SECBREAKPOINT  NUMBER(9,6),
   RAISE6SECCAPACITY    NUMBER(9,6),
   RAISE6SECMAX         NUMBER(9,6),
   PRICE6SECRAISEMANDATORY NUMBER(16,6),
   QUANT6SECRAISEMANDATORY NUMBER(16,6),
   PRICE6SECRAISECONTRACT NUMBER(16,6),
   QUANT6SECRAISECONTRACT NUMBER(16,6),
   PRICE60SECRAISEMANDATORY NUMBER(16,6),
   QUANT60SECRAISEMANDATORY NUMBER(16,6),
   PRICE60SECRAISECONTRACT NUMBER(16,6),
   QUANT60SECRAISECONTRACT NUMBER(16,6),
   PRICE6SECLOWERMANDATORY NUMBER(16,6),
   QUANT6SECLOWERMANDATORY NUMBER(16,6),
   PRICE6SECLOWERCONTRACT NUMBER(16,6),
   QUANT6SECLOWERCONTRACT NUMBER(16,6),
   PRICE60SECLOWERMANDATORY NUMBER(16,6),
   QUANT60SECLOWERMANDATORY NUMBER(16,6),
   PRICE60SECLOWERCONTRACT NUMBER(16,6),
   QUANT60SECLOWERCONTRACT NUMBER(16,6),
   DEADBANDUP           NUMBER(4,2),
   DEADBANDDOWN         NUMBER(4,2),
   DROOP6SECRAISEBREAKPOINT NUMBER(9,6),
   DROOP6SECRAISECAPACITY NUMBER(9,6),
   DROOP6SECRAISEMAX    NUMBER(9,6),
   DROOP60SECRAISEBREAKPOINT NUMBER(9,6),
   DROOP60SECRAISECAPACITY NUMBER(9,6),
   DROOP60SECRAISEMAX   NUMBER(9,6),
   DROOP6SECLOWERBREAKPOINT NUMBER(9,6),
   DROOP6SECLOWERMAX    NUMBER(9,6),
   DROOP60SECLOWERBREAKPOINT NUMBER(9,6),
   DROOP60SECLOWERMAX   NUMBER(9,6),
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table CONTRACTGOVERNOR is
'CONTRACTGOVERNOR became unused when Ancillary Services Review was implemented in 2001. For more details, see Change Notice 126.
CONTRACTGOVERNOR shows Governor contract details used in the settlement and dispatch of this service. Note services are dispatched as 6 and 60 raise and lower Frequency Control Ancillary Services (FCAS). Lower and raise 6 and 60 second fields are used in dispatch of services. Deadband and Droop details are used in settlements.';

comment on column CONTRACTGOVERNOR.CONTRACTID is
'Contract Identifier';

comment on column CONTRACTGOVERNOR.VERSIONNO is
'Contract Version No.';

comment on column CONTRACTGOVERNOR.STARTDATE is
'Starting Date of Contract';

comment on column CONTRACTGOVERNOR.ENDDATE is
'Termination Date of Contract';

comment on column CONTRACTGOVERNOR.PARTICIPANTID is
'Unique participant identifier';

comment on column CONTRACTGOVERNOR.DUID is
'Dispatchable Unit ID';

comment on column CONTRACTGOVERNOR.CCPRICE is
'Compensation Cap';

comment on column CONTRACTGOVERNOR.LOWER60SECBREAKPOINT is
'Limit Equation Lower 60 Second Breakpoint MW';

comment on column CONTRACTGOVERNOR.LOWER60SECMAX is
'Limit Equation Lower 60 Second Maximum MW';

comment on column CONTRACTGOVERNOR.LOWER6SECBREAKPOINT is
'Limit Equation Lower 6 Second Breakpoint MW';

comment on column CONTRACTGOVERNOR.LOWER6SECMAX is
'Limit Equation Lower 6 Second Maximum MW';

comment on column CONTRACTGOVERNOR.RAISE60SECBREAKPOINT is
'Limit Equation Raise 60 Second Breakpoint MW';

comment on column CONTRACTGOVERNOR.RAISE60SECCAPACITY is
'Limit Equation Raise 60 Second Capacity MW';

comment on column CONTRACTGOVERNOR.RAISE60SECMAX is
'Limit Equation Raise 60 Second Maximum MW';

comment on column CONTRACTGOVERNOR.RAISE6SECBREAKPOINT is
'Limit Equation Raise 6 Second Breakpoint MW';

comment on column CONTRACTGOVERNOR.RAISE6SECCAPACITY is
'Limit Equation Raise 6 Second Capacity MW';

comment on column CONTRACTGOVERNOR.RAISE6SECMAX is
'Limit Equation Raise 6 Second Maximum MW';

comment on column CONTRACTGOVERNOR.PRICE6SECRAISEMANDATORY is
'Not used';

comment on column CONTRACTGOVERNOR.QUANT6SECRAISEMANDATORY is
'Not used';

comment on column CONTRACTGOVERNOR.PRICE6SECRAISECONTRACT is
'Contract Price for 6 Second Raise';

comment on column CONTRACTGOVERNOR.QUANT6SECRAISECONTRACT is
'Contract Quantity for 6 Second Raise';

comment on column CONTRACTGOVERNOR.PRICE60SECRAISEMANDATORY is
'Not used';

comment on column CONTRACTGOVERNOR.QUANT60SECRAISEMANDATORY is
'Not used';

comment on column CONTRACTGOVERNOR.PRICE60SECRAISECONTRACT is
'Contract Price for 60 Second Raise';

comment on column CONTRACTGOVERNOR.QUANT60SECRAISECONTRACT is
'Contract Quantity for 60 Second Raise';

comment on column CONTRACTGOVERNOR.PRICE6SECLOWERMANDATORY is
'Not used';

comment on column CONTRACTGOVERNOR.QUANT6SECLOWERMANDATORY is
'Not used';

comment on column CONTRACTGOVERNOR.PRICE6SECLOWERCONTRACT is
'Contract Price for 6 Second Lower';

comment on column CONTRACTGOVERNOR.QUANT6SECLOWERCONTRACT is
'Contract Quantity for 6 Second Lower';

comment on column CONTRACTGOVERNOR.PRICE60SECLOWERMANDATORY is
'Not used';

comment on column CONTRACTGOVERNOR.QUANT60SECLOWERMANDATORY is
'Not used';

comment on column CONTRACTGOVERNOR.PRICE60SECLOWERCONTRACT is
'Contract Price for 60 Second Lower';

comment on column CONTRACTGOVERNOR.QUANT60SECLOWERCONTRACT is
'Contract Quantity for 60 Second Lower';

comment on column CONTRACTGOVERNOR.DEADBANDUP is
'Raise Deadband';

comment on column CONTRACTGOVERNOR.DEADBANDDOWN is
'Lower Deadband';

comment on column CONTRACTGOVERNOR.DROOP6SECRAISEBREAKPOINT is
'Droop Equation Raise 6 Second Breakpoint';

comment on column CONTRACTGOVERNOR.DROOP6SECRAISECAPACITY is
'Droop Equation Raise 6 Second Capacity';

comment on column CONTRACTGOVERNOR.DROOP6SECRAISEMAX is
'Droop Equation Raise 6 Second Maximum';

comment on column CONTRACTGOVERNOR.DROOP60SECRAISEBREAKPOINT is
'Droop Equation Raise 60 Second Breakpoint';

comment on column CONTRACTGOVERNOR.DROOP60SECRAISECAPACITY is
'Droop Equation Raise 60 Second Capacity';

comment on column CONTRACTGOVERNOR.DROOP60SECRAISEMAX is
'Droop Equation Raise 60 Second Maximum';

comment on column CONTRACTGOVERNOR.DROOP6SECLOWERBREAKPOINT is
'Droop Equation Lower 6 Second Breakpoint';

comment on column CONTRACTGOVERNOR.DROOP6SECLOWERMAX is
'Droop Equation Lower 6 Second Maximum';

comment on column CONTRACTGOVERNOR.DROOP60SECLOWERBREAKPOINT is
'Droop Equation Lower 60 Second Breakpoint';

comment on column CONTRACTGOVERNOR.DROOP60SECLOWERMAX is
'Droop Equation Lower 60 Second Maximum';

comment on column CONTRACTGOVERNOR.AUTHORISEDBY is
'User Name';

comment on column CONTRACTGOVERNOR.AUTHORISEDDATE is
'Date Contract was authorised';

comment on column CONTRACTGOVERNOR.LASTCHANGED is
'Last date and time record changed';

alter table CONTRACTGOVERNOR
   add constraint CONTRACTGOVERNOR_PK primary key (CONTRACTID, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTGOVERNOR_NDX2                                 */
/*==============================================================*/
create index CONTRACTGOVERNOR_NDX2 on CONTRACTGOVERNOR (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTGOVERNOR_LCX                                  */
/*==============================================================*/
create index CONTRACTGOVERNOR_LCX on CONTRACTGOVERNOR (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: CONTRACTLOADSHED                                      */
/*==============================================================*/
create table CONTRACTLOADSHED (
   CONTRACTID           VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   STARTDATE            DATE,
   ENDDATE              DATE,
   PARTICIPANTID        VARCHAR2(10),
   DUID                 VARCHAR2(10),
   LSEPRICE             NUMBER(6,2),
   MCPPRICE             NUMBER(12,2),
   TENDEREDPRICE        NUMBER(6,2),
   LSCR                 NUMBER(6,2),
   ILSCALINGFACTOR      NUMBER(15,5),
   LOWER60SECBREAKPOINT NUMBER(9,6),
   LOWER60SECMAX        NUMBER(9,6),
   LOWER6SECBREAKPOINT  NUMBER(9,6),
   LOWER6SECMAX         NUMBER(9,6),
   RAISE60SECBREAKPOINT NUMBER(9,6),
   RAISE60SECCAPACITY   NUMBER(9,6),
   RAISE60SECMAX        NUMBER(9,6),
   RAISE6SECBREAKPOINT  NUMBER(9,6),
   RAISE6SECCAPACITY    NUMBER(9,6),
   RAISE6SECMAX         NUMBER(9,6),
   PRICE6SECRAISEMANDATORY NUMBER(16,6),
   QUANT6SECRAISEMANDATORY NUMBER(9,6),
   PRICE6SECRAISECONTRACT NUMBER(16,6),
   QUANT6SECRAISECONTRACT NUMBER(9,6),
   PRICE60SECRAISEMANDATORY NUMBER(16,6),
   QUANT60SECRAISEMANDATORY NUMBER(9,6),
   PRICE60SECRAISECONTRACT NUMBER(16,6),
   QUANT60SECRAISECONTRACT NUMBER(9,6),
   PRICE6SECLOWERMANDATORY NUMBER(16,6),
   QUANT6SECLOWERMANDATORY NUMBER(9,6),
   PRICE6SECLOWERCONTRACT NUMBER(16,6),
   QUANT6SECLOWERCONTRACT NUMBER(9,6),
   PRICE60SECLOWERMANDATORY NUMBER(16,6),
   QUANT60SECLOWERMANDATORY NUMBER(9,6),
   PRICE60SECLOWERCONTRACT NUMBER(16,6),
   QUANT60SECLOWERCONTRACT NUMBER(9,6),
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE,
   DEFAULT_TESTINGPAYMENT_AMOUNT NUMBER(18,8),
   SERVICE_START_DATE   DATE
)
   tablespace PSSTAB01;

comment on table CONTRACTLOADSHED is
'CONTRACTLOADSHED shows Governor contract details used in the settlement and dispatch of this service. Note: services are dispatched as 6 and 60 raise Frequency Control Ancillary Services (FCAS). Mandatory requirements and breakpoint details are not used for load shed. This Table may also be used for NSCAS and Type 1 transitional services procured by AEMO under the ISF framework during 2025 and prior to the implementation of all system changes. During this time descriptions in these tables may not be correct.';

comment on column CONTRACTLOADSHED.CONTRACTID is
'Contract Identifier';

comment on column CONTRACTLOADSHED.VERSIONNO is
'Contract Version No.';

comment on column CONTRACTLOADSHED.STARTDATE is
'Starting Date of Contract';

comment on column CONTRACTLOADSHED.ENDDATE is
'Termination Date of Contract';

comment on column CONTRACTLOADSHED.PARTICIPANTID is
'Unique participant identifier';

comment on column CONTRACTLOADSHED.DUID is
'Dispatchable Unit ID';

comment on column CONTRACTLOADSHED.LSEPRICE is
'The load shed enabling price for this contract';

comment on column CONTRACTLOADSHED.MCPPRICE is
'Minimum Compensation price';

comment on column CONTRACTLOADSHED.TENDEREDPRICE is
'Price Tendered for Compensation per Trading interval - Not used since 13/12/1998';

comment on column CONTRACTLOADSHED.LSCR is
'Load Shed Control Range';

comment on column CONTRACTLOADSHED.ILSCALINGFACTOR is
'SPD scaling factor for load shed vs dispatched, (1 = dispatched)';

comment on column CONTRACTLOADSHED.LOWER60SECBREAKPOINT is
'Not used';

comment on column CONTRACTLOADSHED.LOWER60SECMAX is
'Not used';

comment on column CONTRACTLOADSHED.LOWER6SECBREAKPOINT is
'Not used';

comment on column CONTRACTLOADSHED.LOWER6SECMAX is
'Not used';

comment on column CONTRACTLOADSHED.RAISE60SECBREAKPOINT is
'Not used';

comment on column CONTRACTLOADSHED.RAISE60SECCAPACITY is
'Not used';

comment on column CONTRACTLOADSHED.RAISE60SECMAX is
'Maximum 60 second raise';

comment on column CONTRACTLOADSHED.RAISE6SECBREAKPOINT is
'Not used';

comment on column CONTRACTLOADSHED.RAISE6SECCAPACITY is
'Not used';

comment on column CONTRACTLOADSHED.RAISE6SECMAX is
'Limit Equation Raise 6 Second Maximum MW';

comment on column CONTRACTLOADSHED.PRICE6SECRAISEMANDATORY is
'Not used';

comment on column CONTRACTLOADSHED.QUANT6SECRAISEMANDATORY is
'Not used';

comment on column CONTRACTLOADSHED.PRICE6SECRAISECONTRACT is
'Contract Price for 6 Second Raise';

comment on column CONTRACTLOADSHED.QUANT6SECRAISECONTRACT is
'Contract Quantity for 6 Second Raise';

comment on column CONTRACTLOADSHED.PRICE60SECRAISEMANDATORY is
'Not used';

comment on column CONTRACTLOADSHED.QUANT60SECRAISEMANDATORY is
'Not used';

comment on column CONTRACTLOADSHED.PRICE60SECRAISECONTRACT is
'Not used';

comment on column CONTRACTLOADSHED.QUANT60SECRAISECONTRACT is
'Not used';

comment on column CONTRACTLOADSHED.PRICE6SECLOWERMANDATORY is
'Not used';

comment on column CONTRACTLOADSHED.QUANT6SECLOWERMANDATORY is
'Not used';

comment on column CONTRACTLOADSHED.PRICE6SECLOWERCONTRACT is
'Not used';

comment on column CONTRACTLOADSHED.QUANT6SECLOWERCONTRACT is
'Not used';

comment on column CONTRACTLOADSHED.PRICE60SECLOWERMANDATORY is
'Not used';

comment on column CONTRACTLOADSHED.QUANT60SECLOWERMANDATORY is
'Not used';

comment on column CONTRACTLOADSHED.PRICE60SECLOWERCONTRACT is
'Not used';

comment on column CONTRACTLOADSHED.QUANT60SECLOWERCONTRACT is
'Not used';

comment on column CONTRACTLOADSHED.AUTHORISEDBY is
'User Name';

comment on column CONTRACTLOADSHED.AUTHORISEDDATE is
'Date Contract was Authorised';

comment on column CONTRACTLOADSHED.LASTCHANGED is
'Last date and time record changed';

comment on column CONTRACTLOADSHED.DEFAULT_TESTINGPAYMENT_AMOUNT is
'The NMAS default payment amount';

comment on column CONTRACTLOADSHED.SERVICE_START_DATE is
'The NMAS Testing Service Start Date';

alter table CONTRACTLOADSHED
   add constraint CONTRACTLOADSHED_PK primary key (CONTRACTID, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTLOADSHED_NDX2                                 */
/*==============================================================*/
create index CONTRACTLOADSHED_NDX2 on CONTRACTLOADSHED (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTLOADSHED_LCX                                  */
/*==============================================================*/
create index CONTRACTLOADSHED_LCX on CONTRACTLOADSHED (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: CONTRACTREACTIVEPOWER                                 */
/*==============================================================*/
create table CONTRACTREACTIVEPOWER (
   CONTRACTID           VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   STARTDATE            DATE,
   ENDDATE              DATE,
   PARTICIPANTID        VARCHAR2(10),
   DUID                 VARCHAR2(10),
   SYNCCOMPENSATION     VARCHAR2(1),
   MVARAPRICE           NUMBER(10,2),
   MVAREPRICE           NUMBER(10,2),
   MVARGPRICE           NUMBER(10,2),
   CCPRICE              NUMBER(10,2),
   MTA                  NUMBER(10,2),
   MTG                  NUMBER(10,2),
   MMCA                 NUMBER(10,2),
   MMCG                 NUMBER(10,2),
   EU                   NUMBER(10,2),
   PP                   NUMBER(10,2),
   BS                   NUMBER(10,2),
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE,
   DEFAULT_TESTINGPAYMENT_AMOUNT NUMBER(18,8),
   SERVICE_START_DATE   DATE,
   AVAILABILITY_MWH_THRESHOLD NUMBER(18,8),
   MVAR_THRESHOLD       NUMBER(18,8),
   REBATE_CAP           NUMBER(18,8),
   REBATE_AMOUNT_PER_MVAR NUMBER(18,8),
   ISREBATEAPPLICABLE   NUMBER(1,0)
)
   tablespace PSSTAB01;

comment on table CONTRACTREACTIVEPOWER is
'CONTRACTREACTIVEPOWER shows Reactive Power contract details used in the settlement and dispatch of this service. This Table may also be used for NSCAS and Type 1 transitional services procured by AEMO under the ISF framework during 2025 and prior to the implementation of all system changes. During this time descriptions in these tables may not be correct.';

comment on column CONTRACTREACTIVEPOWER.CONTRACTID is
'Contract Identifier';

comment on column CONTRACTREACTIVEPOWER.VERSIONNO is
'Contract Version No.';

comment on column CONTRACTREACTIVEPOWER.STARTDATE is
'Starting Date of Contract';

comment on column CONTRACTREACTIVEPOWER.ENDDATE is
'Termination Date of Contract';

comment on column CONTRACTREACTIVEPOWER.PARTICIPANTID is
'Unique participant identifier';

comment on column CONTRACTREACTIVEPOWER.DUID is
'Dispatchable Unit ID';

comment on column CONTRACTREACTIVEPOWER.SYNCCOMPENSATION is
'Sync Compensation Flag - Y for SYNCCOMP';

comment on column CONTRACTREACTIVEPOWER.MVARAPRICE is
'Availability price per MVAr of RP absorption capability';

comment on column CONTRACTREACTIVEPOWER.MVAREPRICE is
'Enabling price';

comment on column CONTRACTREACTIVEPOWER.MVARGPRICE is
'Availability price per MVAr of RP generation capability';

comment on column CONTRACTREACTIVEPOWER.CCPRICE is
'Compensation Cap';

comment on column CONTRACTREACTIVEPOWER.MTA is
'Reactive Power Absorption Capability (MVAr)';

comment on column CONTRACTREACTIVEPOWER.MTG is
'Reactive Power Generation Capability (MVAr)';

comment on column CONTRACTREACTIVEPOWER.MMCA is
'Minimum Capability for MVAr Absorption required by Code';

comment on column CONTRACTREACTIVEPOWER.MMCG is
'Minimum Capability for MVAr Generation required by Code';

comment on column CONTRACTREACTIVEPOWER.EU is
'Estimated Power consumption of unit when operating on SYNCCOMP';

comment on column CONTRACTREACTIVEPOWER.PP is
'Estimated Price for supply';

comment on column CONTRACTREACTIVEPOWER.BS is
'Block Size of Unit';

comment on column CONTRACTREACTIVEPOWER.AUTHORISEDBY is
'User Name';

comment on column CONTRACTREACTIVEPOWER.AUTHORISEDDATE is
'Date Contract was Authorised';

comment on column CONTRACTREACTIVEPOWER.LASTCHANGED is
'Last date and time record changed';

comment on column CONTRACTREACTIVEPOWER.DEFAULT_TESTINGPAYMENT_AMOUNT is
'The NMAS default payment amount';

comment on column CONTRACTREACTIVEPOWER.SERVICE_START_DATE is
'The NMAS Testing Service Start Date';

comment on column CONTRACTREACTIVEPOWER.AVAILABILITY_MWH_THRESHOLD is
'The MWh the unit must produce in a trading interval to be eligible for an excess-to-gen availability payment';

comment on column CONTRACTREACTIVEPOWER.MVAR_THRESHOLD is
'The threshold value for MegaVar (MVAr) to check whether the service is fully available.';

comment on column CONTRACTREACTIVEPOWER.REBATE_CAP is
'The maximum capped amount for the rebate payment.';

comment on column CONTRACTREACTIVEPOWER.REBATE_AMOUNT_PER_MVAR is
'The per MVAR rebate amount used to calculate the rebate payment.';

comment on column CONTRACTREACTIVEPOWER.ISREBATEAPPLICABLE is
'Used to check whether the contract is eligible for rebate. For new NSCAS contracts to apply new payment methodology this flag is 1.';

alter table CONTRACTREACTIVEPOWER
   add constraint CONTRACTREACTIVEPOWER_PK primary key (CONTRACTID, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTREACTIVEPOWE_NDX2                             */
/*==============================================================*/
create index CONTRACTREACTIVEPOWE_NDX2 on CONTRACTREACTIVEPOWER (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTREACTIVEPOWER_LCX                             */
/*==============================================================*/
create index CONTRACTREACTIVEPOWER_LCX on CONTRACTREACTIVEPOWER (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: CONTRACTRESERVEFLAG                                   */
/*==============================================================*/
create table CONTRACTRESERVEFLAG (
   CONTRACTID           VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   PERIODID             NUMBER(3,0)           not null,
   RCF                  CHAR(1),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table CONTRACTRESERVEFLAG is
'CONTRACTRESERVEFLAG has never been or will be used. It was to show a period by period flag for regional or market recovery of reserve trading contract amounts.';

comment on column CONTRACTRESERVEFLAG.CONTRACTID is
'Reserve Trader Contract Identifier';

comment on column CONTRACTRESERVEFLAG.VERSIONNO is
'Reserve Trader Contract Version';

comment on column CONTRACTRESERVEFLAG.PERIODID is
'Calendar settlement date period identifier, i.e. period 1 is 00:30';

comment on column CONTRACTRESERVEFLAG.RCF is
'Reserve Recovery Flag';

comment on column CONTRACTRESERVEFLAG.LASTCHANGED is
'Last date and time record changed';

alter table CONTRACTRESERVEFLAG
   add constraint CONTRACTRESERVEFLAG_PK primary key (CONTRACTID, VERSIONNO, PERIODID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTRESERVEFLAG_LCX                               */
/*==============================================================*/
create index CONTRACTRESERVEFLAG_LCX on CONTRACTRESERVEFLAG (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: CONTRACTRESERVETHRESHOLD                              */
/*==============================================================*/
create table CONTRACTRESERVETHRESHOLD (
   CONTRACTID           VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   CRA                  NUMBER(16,6),
   CRE                  NUMBER(16,6),
   CRU                  NUMBER(16,6),
   CTA                  NUMBER(16,6),
   CTE                  NUMBER(16,6),
   CTU                  NUMBER(16,6),
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table CONTRACTRESERVETHRESHOLD is
'CONTRACTRESERVETHRESHOLD shows reserve contract threshold details for enabling, usage and availability thresholds and rates for reserve trader contracts.';

comment on column CONTRACTRESERVETHRESHOLD.CONTRACTID is
'Contract Identifier';

comment on column CONTRACTRESERVETHRESHOLD.VERSIONNO is
'Contract Version';

comment on column CONTRACTRESERVETHRESHOLD.CRA is
'Availability Rate $';

comment on column CONTRACTRESERVETHRESHOLD.CRE is
'Enabling Rate $';

comment on column CONTRACTRESERVETHRESHOLD.CRU is
'Usage Rate $';

comment on column CONTRACTRESERVETHRESHOLD.CTA is
'Availability Threshold MW/h';

comment on column CONTRACTRESERVETHRESHOLD.CTE is
'Enabling Threshold MW/h';

comment on column CONTRACTRESERVETHRESHOLD.CTU is
'Usage Threshold MW/h';

comment on column CONTRACTRESERVETHRESHOLD.AUTHORISEDBY is
'User name';

comment on column CONTRACTRESERVETHRESHOLD.AUTHORISEDDATE is
'Date contract was authorised';

comment on column CONTRACTRESERVETHRESHOLD.LASTCHANGED is
'Last date and time record changed';

alter table CONTRACTRESERVETHRESHOLD
   add constraint CONTRACTRESERVETHRESHOLD_PK primary key (CONTRACTID, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTRESERVETHRESHOLD_LCX                          */
/*==============================================================*/
create index CONTRACTRESERVETHRESHOLD_LCX on CONTRACTRESERVETHRESHOLD (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: CONTRACTRESERVETRADER                                 */
/*==============================================================*/
create table CONTRACTRESERVETRADER (
   CONTRACTID           VARCHAR2(10)          not null,
   DUID                 VARCHAR2(10),
   STARTDATE            DATE,
   ENDDATE              DATE,
   STARTPERIOD          NUMBER(3,0),
   ENDPERIOD            NUMBER(3,0),
   DEREGISTRATIONDATE   DATE,
   DEREGISTRATIONPERIOD NUMBER(3,0),
   PARTICIPANTID        VARCHAR2(10),
   LASTCHANGED          DATE,
   REGIONID             VARCHAR2(10)
)
   tablespace PSSTAB01;

comment on table CONTRACTRESERVETRADER is
'CONTRACTRESERVETRADER shows reserve trader contract details. Version numbers do not apply as contracts exist for specified purposes.';

comment on column CONTRACTRESERVETRADER.CONTRACTID is
'Reserve Trader Contract Identifier';

comment on column CONTRACTRESERVETRADER.DUID is
'Dispatchable Unit ID';

comment on column CONTRACTRESERVETRADER.STARTDATE is
'Starting Date of Contract';

comment on column CONTRACTRESERVETRADER.ENDDATE is
'Terminate Date of contract';

comment on column CONTRACTRESERVETRADER.STARTPERIOD is
'Starting period of contract';

comment on column CONTRACTRESERVETRADER.ENDPERIOD is
'Terminate period of contract based on calendar date.';

comment on column CONTRACTRESERVETRADER.DEREGISTRATIONDATE is
'De-registration date of contract; Not Used';

comment on column CONTRACTRESERVETRADER.DEREGISTRATIONPERIOD is
'De-registration period of contract; Not Used';

comment on column CONTRACTRESERVETRADER.PARTICIPANTID is
'Unique participant identifier';

comment on column CONTRACTRESERVETRADER.LASTCHANGED is
'Last date and time record changed';

comment on column CONTRACTRESERVETRADER.REGIONID is
'Region Identifier';

alter table CONTRACTRESERVETRADER
   add constraint CONTRACTRESERVETRADER_PK primary key (CONTRACTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTRESERVETRADER_LCX                             */
/*==============================================================*/
create index CONTRACTRESERVETRADER_LCX on CONTRACTRESERVETRADER (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: CONTRACTRESTARTSERVICES                               */
/*==============================================================*/
create table CONTRACTRESTARTSERVICES (
   CONTRACTID           VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   STARTDATE            DATE,
   ENDDATE              DATE,
   PARTICIPANTID        VARCHAR2(10),
   RESTARTTYPE          NUMBER(1,0),
   RCPRICE              NUMBER(6,2),
   TRIPTOHOUSELEVEL     NUMBER(5,0),
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE,
   DEFAULT_TESTINGPAYMENT_AMOUNT NUMBER(18,8),
   SERVICE_START_DATE   DATE
)
   tablespace PSSTAB01;

comment on table CONTRACTRESTARTSERVICES is
'CONTRACTRESTARTSERVICES shows Restart Services contract details used in the settlement and dispatch of this service.';

comment on column CONTRACTRESTARTSERVICES.CONTRACTID is
'Contract Identifier';

comment on column CONTRACTRESTARTSERVICES.VERSIONNO is
'Contract Version No.';

comment on column CONTRACTRESTARTSERVICES.STARTDATE is
'Starting Date of Contract';

comment on column CONTRACTRESTARTSERVICES.ENDDATE is
'Termination Date of Contract';

comment on column CONTRACTRESTARTSERVICES.PARTICIPANTID is
'Unique participant identifier';

comment on column CONTRACTRESTARTSERVICES.RESTARTTYPE is
'Restart Type - 0 = BlackStart, 1 = Combination, 2 = Trip To House';

comment on column CONTRACTRESTARTSERVICES.RCPRICE is
'Availability Price';

comment on column CONTRACTRESTARTSERVICES.TRIPTOHOUSELEVEL is
'Trip To House Level';

comment on column CONTRACTRESTARTSERVICES.AUTHORISEDBY is
'User Name';

comment on column CONTRACTRESTARTSERVICES.AUTHORISEDDATE is
'Date Contract was Authorised';

comment on column CONTRACTRESTARTSERVICES.LASTCHANGED is
'Last date and time record changed';

comment on column CONTRACTRESTARTSERVICES.DEFAULT_TESTINGPAYMENT_AMOUNT is
'The NMAS default payment amount';

comment on column CONTRACTRESTARTSERVICES.SERVICE_START_DATE is
'The NMAS Testing Service Start Date';

alter table CONTRACTRESTARTSERVICES
   add constraint CONTRACTRESTARTSERVICES_PK primary key (CONTRACTID, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTRESTARTSERVI_NDX2                             */
/*==============================================================*/
create index CONTRACTRESTARTSERVI_NDX2 on CONTRACTRESTARTSERVICES (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTRESTARTSERVICES_LCX                           */
/*==============================================================*/
create index CONTRACTRESTARTSERVICES_LCX on CONTRACTRESTARTSERVICES (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: CONTRACTRESTARTUNITS                                  */
/*==============================================================*/
create table CONTRACTRESTARTUNITS (
   CONTRACTID           VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   DUID                 VARCHAR2(10)          not null,
   LASTCHANGED          DATE,
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE
)
   tablespace PSSTAB01;

comment on table CONTRACTRESTARTUNITS is
'CONTRACTRESTARTUNITS shows Restart units provided under a system restart contract. A service can have multiple units.';

comment on column CONTRACTRESTARTUNITS.CONTRACTID is
'Contract Identifier';

comment on column CONTRACTRESTARTUNITS.VERSIONNO is
'Version No of contract';

comment on column CONTRACTRESTARTUNITS.DUID is
'Dispatchable Unit identifier';

comment on column CONTRACTRESTARTUNITS.LASTCHANGED is
'Last date and time record changed';

alter table CONTRACTRESTARTUNITS
   add constraint CONTRACTRESTARTUNITS_PK primary key (CONTRACTID, VERSIONNO, DUID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTRESTARTUNITS_NDX2                             */
/*==============================================================*/
create index CONTRACTRESTARTUNITS_NDX2 on CONTRACTRESTARTUNITS (
   CONTRACTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTRESTARTUNITS_LCX                              */
/*==============================================================*/
create index CONTRACTRESTARTUNITS_LCX on CONTRACTRESTARTUNITS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: CONTRACTUNITLOADING                                   */
/*==============================================================*/
create table CONTRACTUNITLOADING (
   CONTRACTID           VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   STARTDATE            DATE,
   ENDDATE              DATE,
   PARTICIPANTID        VARCHAR2(10),
   DUID                 VARCHAR2(10),
   RPRICE               NUMBER(10,2),
   SUPRICE              NUMBER(10,2),
   CCPRICE              NUMBER(10,2),
   ACR                  NUMBER(10,2),
   BS                   NUMBER(10,2),
   PP                   NUMBER(10,2),
   EU                   NUMBER(10,2),
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table CONTRACTUNITLOADING is
'CONTRACTUNITLOADING became unused when Ancillary Services Review was implemented in 2001. For more details, see Change Notice 126.
CONTRACTUNITLOADING shows Unit Loading contract details used in the settlement and dispatch of this service.';

comment on column CONTRACTUNITLOADING.CONTRACTID is
'Contract Identifier';

comment on column CONTRACTUNITLOADING.VERSIONNO is
'Contract Version No.';

comment on column CONTRACTUNITLOADING.STARTDATE is
'Starting Date of Contract';

comment on column CONTRACTUNITLOADING.ENDDATE is
'Termination Date of Contract';

comment on column CONTRACTUNITLOADING.PARTICIPANTID is
'Unique participant identifier';

comment on column CONTRACTUNITLOADING.DUID is
'Dispatchable Unit ID';

comment on column CONTRACTUNITLOADING.RPRICE is
'Enabling Price';

comment on column CONTRACTUNITLOADING.SUPRICE is
'Usage Price';

comment on column CONTRACTUNITLOADING.CCPRICE is
'Compensation Cap';

comment on column CONTRACTUNITLOADING.ACR is
'Available Control Range';

comment on column CONTRACTUNITLOADING.BS is
'Block Size of Unit';

comment on column CONTRACTUNITLOADING.PP is
'Estimated Price for supply';

comment on column CONTRACTUNITLOADING.EU is
'Estimated Power consumption of unit when enabled for RGUL';

comment on column CONTRACTUNITLOADING.AUTHORISEDBY is
'User Name';

comment on column CONTRACTUNITLOADING.AUTHORISEDDATE is
'Date Contract was authorised';

comment on column CONTRACTUNITLOADING.LASTCHANGED is
'Last date and time record changed';

alter table CONTRACTUNITLOADING
   add constraint CONTRACTUNITLOADING_PK primary key (CONTRACTID, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTUNITLOADING_NDX2                              */
/*==============================================================*/
create index CONTRACTUNITLOADING_NDX2 on CONTRACTUNITLOADING (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTUNITLOADING_LCX                               */
/*==============================================================*/
create index CONTRACTUNITLOADING_LCX on CONTRACTUNITLOADING (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: CONTRACTUNITUNLOADING                                 */
/*==============================================================*/
create table CONTRACTUNITUNLOADING (
   CONTRACTID           VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   STARTDATE            DATE,
   ENDDATE              DATE,
   PARTICIPANTID        VARCHAR2(10),
   DUID                 VARCHAR2(10),
   RPRICE               NUMBER(10,2),
   SUPRICE              NUMBER(10,2),
   CCPRICE              NUMBER(10,2),
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table CONTRACTUNITUNLOADING is
'CONTRACTUNITUNLOADING shows Ancillary Service contract data for rapid generator unit unloading.';

comment on column CONTRACTUNITUNLOADING.CONTRACTID is
'Contract Identifier';

comment on column CONTRACTUNITUNLOADING.VERSIONNO is
'Contract Version No.';

comment on column CONTRACTUNITUNLOADING.STARTDATE is
'Starting Date of Contract';

comment on column CONTRACTUNITUNLOADING.ENDDATE is
'Termination Date of Contract';

comment on column CONTRACTUNITUNLOADING.PARTICIPANTID is
'Unique participant identifier';

comment on column CONTRACTUNITUNLOADING.DUID is
'Dispatchable Unit ID';

comment on column CONTRACTUNITUNLOADING.RPRICE is
'Enabling Price';

comment on column CONTRACTUNITUNLOADING.SUPRICE is
'Usage Price';

comment on column CONTRACTUNITUNLOADING.CCPRICE is
'Compensation Cap';

comment on column CONTRACTUNITUNLOADING.AUTHORISEDBY is
'User Name';

comment on column CONTRACTUNITUNLOADING.AUTHORISEDDATE is
'Date Contract was Authorised';

comment on column CONTRACTUNITUNLOADING.LASTCHANGED is
'Last date and time record changed';

alter table CONTRACTUNITUNLOADING
   add constraint CONTRACTUNITUNLOADING_PK primary key (CONTRACTID, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTUNITUNLOADIN_NDX2                             */
/*==============================================================*/
create index CONTRACTUNITUNLOADIN_NDX2 on CONTRACTUNITUNLOADING (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: CONTRACTUNITUNLOADING_LCX                             */
/*==============================================================*/
create index CONTRACTUNITUNLOADING_LCX on CONTRACTUNITUNLOADING (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DAYOFFER                                              */
/*==============================================================*/
create table DAYOFFER (
   SETTLEMENTDATE       DATE                  not null,
   DUID                 VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   OFFERDATE            DATE                  not null,
   SELFCOMMITFLAG       VARCHAR2(1),
   DAILYENERGYCONSTRAINT NUMBER(12,6),
   ENTRYTYPE            VARCHAR2(20),
   CONTINGENCYPRICE     NUMBER(9,2),
   REBIDEXPLANATION     VARCHAR2(64),
   BANDQUANTISATIONID   NUMBER(2,0),
   PRICEBAND1           NUMBER(9,2),
   PRICEBAND2           NUMBER(9,2),
   PRICEBAND3           NUMBER(9,2),
   PRICEBAND4           NUMBER(9,2),
   PRICEBAND5           NUMBER(9,2),
   PRICEBAND6           NUMBER(9,2),
   PRICEBAND7           NUMBER(9,2),
   PRICEBAND8           NUMBER(9,2),
   PRICEBAND9           NUMBER(9,2),
   PRICEBAND10          NUMBER(9,2),
   MAXRAMPUP            NUMBER(9,2),
   MAXRAMPDOWN          NUMBER(9,2),
   MINIMUMLOAD          NUMBER(6,0),
   T1                   NUMBER(6,0),
   T2                   NUMBER(6,0),
   T3                   NUMBER(6,0),
   T4                   NUMBER(6,0),
   NORMALSTATUS         VARCHAR2(3),
   LASTCHANGED          DATE,
   MR_FACTOR            NUMBER(16,6)
)
   tablespace PSSTAB01;

comment on table DAYOFFER is
'DAYOFFER sets out the participants daily components of participant bid containing details applying for the whole day (such as prices, daily energy constraint and fast start profiles).
To retrieve full bid details, read in conjunction with PEROFFER.';

comment on column DAYOFFER.SETTLEMENTDATE is
'Market date starting at 04:00am';

comment on column DAYOFFER.DUID is
'Dispatchable unit identifier';

comment on column DAYOFFER.VERSIONNO is
'Version no. for given offer date';

comment on column DAYOFFER.OFFERDATE is
'Offer date of data';

comment on column DAYOFFER.SELFCOMMITFLAG is
'Not used';

comment on column DAYOFFER.DAILYENERGYCONSTRAINT is
'Maximum energy available from Energy Constrained Plant.';

comment on column DAYOFFER.ENTRYTYPE is
'Daily or Rebid';

comment on column DAYOFFER.CONTINGENCYPRICE is
'Not used';

comment on column DAYOFFER.REBIDEXPLANATION is
'Explanation for all rebids and inflexibilities';

comment on column DAYOFFER.BANDQUANTISATIONID is
'Not used';

comment on column DAYOFFER.PRICEBAND1 is
'Price for Availability Band 1';

comment on column DAYOFFER.PRICEBAND2 is
'Price for Availability Band 2';

comment on column DAYOFFER.PRICEBAND3 is
'Price for Availability Band 3';

comment on column DAYOFFER.PRICEBAND4 is
'Price for Availability Band 4';

comment on column DAYOFFER.PRICEBAND5 is
'Price for Availability Band 5';

comment on column DAYOFFER.PRICEBAND6 is
'Price for Availability Band 6';

comment on column DAYOFFER.PRICEBAND7 is
'Price for Availability Band 7';

comment on column DAYOFFER.PRICEBAND8 is
'Price for Availability Band 8';

comment on column DAYOFFER.PRICEBAND9 is
'Price for Availability Band 9';

comment on column DAYOFFER.PRICEBAND10 is
'Price for Availability Band 10';

comment on column DAYOFFER.MAXRAMPUP is
'Not used';

comment on column DAYOFFER.MAXRAMPDOWN is
'Not used';

comment on column DAYOFFER.MINIMUMLOAD is
'Minimum MW load fast start plant in MW';

comment on column DAYOFFER.T1 is
'Time to synchronise in minutes';

comment on column DAYOFFER.T2 is
'Time to minimum load in minutes';

comment on column DAYOFFER.T3 is
'Time at minimum load in minutes';

comment on column DAYOFFER.T4 is
'Time to shutdown in minutes';

comment on column DAYOFFER.NORMALSTATUS is
'ON/OFF for loads';

comment on column DAYOFFER.LASTCHANGED is
'Last date and time record changed';

comment on column DAYOFFER.MR_FACTOR is
'Mandatory Restriction Price Scaling Factor';

alter table DAYOFFER
   add constraint DAYOFFER_PK primary key (SETTLEMENTDATE, DUID, OFFERDATE, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: DAYOFFER_NDX2                                         */
/*==============================================================*/
create index DAYOFFER_NDX2 on DAYOFFER (
   DUID ASC,
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: DAYOFFER_LCX                                          */
/*==============================================================*/
create index DAYOFFER_LCX on DAYOFFER (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DAYOFFER_D                                            */
/*==============================================================*/
create table DAYOFFER_D (
   SETTLEMENTDATE       DATE                  not null,
   DUID                 VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   OFFERDATE            DATE                  not null,
   SELFCOMMITFLAG       VARCHAR2(1),
   DAILYENERGYCONSTRAINT NUMBER(12,6),
   ENTRYTYPE            VARCHAR2(20),
   CONTINGENCYPRICE     NUMBER(9,2),
   REBIDEXPLANATION     VARCHAR2(64),
   BANDQUANTISATIONID   NUMBER(2,0),
   PRICEBAND1           NUMBER(9,2),
   PRICEBAND2           NUMBER(9,2),
   PRICEBAND3           NUMBER(9,2),
   PRICEBAND4           NUMBER(9,2),
   PRICEBAND5           NUMBER(9,2),
   PRICEBAND6           NUMBER(9,2),
   PRICEBAND7           NUMBER(9,2),
   PRICEBAND8           NUMBER(9,2),
   PRICEBAND9           NUMBER(9,2),
   PRICEBAND10          NUMBER(9,2),
   MAXRAMPUP            NUMBER(9,2),
   MAXRAMPDOWN          NUMBER(9,2),
   MINIMUMLOAD          NUMBER(6,0),
   T1                   NUMBER(6,0),
   T2                   NUMBER(6,0),
   T3                   NUMBER(6,0),
   T4                   NUMBER(6,0),
   NORMALSTATUS         VARCHAR2(3),
   LASTCHANGED          DATE,
   MR_FACTOR            NUMBER(6,0)
)
   tablespace PSSTAB02;

comment on table DAYOFFER_D is
'DAYOFFER_D sets out the participants daily components of participant bid containing just the latest details (such as prices, daily energy constraint and fast start profiles).
To retrieve latest bid details, read in conjunction with PEROFFER_D.';

comment on column DAYOFFER_D.SETTLEMENTDATE is
'Market date starting at 04:00am';

comment on column DAYOFFER_D.DUID is
'Dispatchable unit identifier';

comment on column DAYOFFER_D.VERSIONNO is
'Version no. for given offer date';

comment on column DAYOFFER_D.OFFERDATE is
'Offer date of data';

comment on column DAYOFFER_D.SELFCOMMITFLAG is
'Not used';

comment on column DAYOFFER_D.DAILYENERGYCONSTRAINT is
'Maximum energy available from Energy Constrained Plant.';

comment on column DAYOFFER_D.ENTRYTYPE is
'Daily or Rebid';

comment on column DAYOFFER_D.CONTINGENCYPRICE is
'Not used';

comment on column DAYOFFER_D.REBIDEXPLANATION is
'Explanation for all rebids and inflexibilities';

comment on column DAYOFFER_D.BANDQUANTISATIONID is
'Not used';

comment on column DAYOFFER_D.PRICEBAND1 is
'Price for Availability Band 1';

comment on column DAYOFFER_D.PRICEBAND2 is
'Price for Availability Band 2';

comment on column DAYOFFER_D.PRICEBAND3 is
'Price for Availability Band 3';

comment on column DAYOFFER_D.PRICEBAND4 is
'Price for Availability Band 4';

comment on column DAYOFFER_D.PRICEBAND5 is
'Price for Availability Band 5';

comment on column DAYOFFER_D.PRICEBAND6 is
'Price for Availability Band 6';

comment on column DAYOFFER_D.PRICEBAND7 is
'Price for Availability Band 7';

comment on column DAYOFFER_D.PRICEBAND8 is
'Price for Availability Band 8';

comment on column DAYOFFER_D.PRICEBAND9 is
'Price for Availability Band 9';

comment on column DAYOFFER_D.PRICEBAND10 is
'Price for Availability Band 10';

comment on column DAYOFFER_D.MAXRAMPUP is
'Not used';

comment on column DAYOFFER_D.MAXRAMPDOWN is
'Not used';

comment on column DAYOFFER_D.MINIMUMLOAD is
'Minimum MW load fast start plant in MW';

comment on column DAYOFFER_D.T1 is
'Time to synchronise in minutes';

comment on column DAYOFFER_D.T2 is
'Time to minimum load in minutes';

comment on column DAYOFFER_D.T3 is
'Time at minimum load in minutes';

comment on column DAYOFFER_D.T4 is
'Time to shutdown in minutes';

comment on column DAYOFFER_D.NORMALSTATUS is
'ON/OFF for loads';

comment on column DAYOFFER_D.LASTCHANGED is
'Last date and time record changed';

comment on column DAYOFFER_D.MR_FACTOR is
'Mandatory Restriction Price Scaling Factor';

alter table DAYOFFER_D
   add constraint DAYOFFER_D_PK primary key (SETTLEMENTDATE, DUID, OFFERDATE, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: DAYOFFER_D_NDX2                                       */
/*==============================================================*/
create index DAYOFFER_D_NDX2 on DAYOFFER_D (
   DUID ASC,
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: DAYOFFER_D_LCX                                        */
/*==============================================================*/
create index DAYOFFER_D_LCX on DAYOFFER_D (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DAYTRACK                                              */
/*==============================================================*/
create table DAYTRACK (
   SETTLEMENTDATE       DATE                  not null,
   REGIONID             VARCHAR2(10),
   EXANTERUNSTATUS      VARCHAR2(15),
   EXANTERUNNO          NUMBER(3,0),
   EXPOSTRUNSTATUS      VARCHAR2(15),
   EXPOSTRUNNO          NUMBER(3,0)           not null,
   LASTCHANGED          DATE,
   SETTLEMENTINTERVALLENGTH NUMBER(3,0),
   METER_CASEID         NUMBER(5,0),
   METER_RUNTYPE        VARCHAR2(10)
)
   tablespace PSSTAB01;

comment on table DAYTRACK is
'DAYTRACK identifies the actual settlement run processed for each settlement day. Settlement run is in the column EXPOSTRUNNO. Generally, the number of the settlement run used in the latest statement is the maximum number.';

comment on column DAYTRACK.SETTLEMENTDATE is
'Calendar Settlement Date';

comment on column DAYTRACK.REGIONID is
'Not Used';

comment on column DAYTRACK.EXANTERUNSTATUS is
'This is to determine whether the Settlement Run is a valid run or not.';

comment on column DAYTRACK.EXANTERUNNO is
'Not Used';

comment on column DAYTRACK.EXPOSTRUNSTATUS is
'Estimate for DAILY/INITIAL and Actual for PRELIM/FINAL and REVISE.';

comment on column DAYTRACK.EXPOSTRUNNO is
'Settlement Run No';

comment on column DAYTRACK.LASTCHANGED is
'Last date and time record changed';

comment on column DAYTRACK.SETTLEMENTINTERVALLENGTH is
'Length of a settlement interval, in minutes (was 30 minutes, will be 5 minutes starting the commencement of 5MS rule change date).';

comment on column DAYTRACK.METER_CASEID is
'The Metering Case Id used for the Settlement Run. For Estimate Daily Run this will be 0';

comment on column DAYTRACK.METER_RUNTYPE is
'The Type of Settlement Run(ESTIMATE/PRELIM/FINAL/REVISE)';

alter table DAYTRACK
   add constraint DAYTRACK_PK primary key (SETTLEMENTDATE, EXPOSTRUNNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: DAYTRACK_LCX                                          */
/*==============================================================*/
create index DAYTRACK_LCX on DAYTRACK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DEFAULTDAYOFFER                                       */
/*==============================================================*/
create table DEFAULTDAYOFFER (
   SETTLEMENTDATE       DATE                  not null,
   DUID                 VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   SELFCOMMITFLAG       VARCHAR2(1),
   DAILYENERGYCONSTRAINT NUMBER(12,6),
   ENTRYTYPE            VARCHAR2(20),
   CONTINGENCYPRICE     NUMBER(9,2),
   REBIDEXPLANATION     VARCHAR2(64),
   BANDQUANTISATIONID   NUMBER(2,0),
   PRICEBAND1           NUMBER(9,2),
   PRICEBAND2           NUMBER(9,2),
   PRICEBAND3           NUMBER(9,2),
   PRICEBAND4           NUMBER(9,2),
   PRICEBAND5           NUMBER(9,2),
   PRICEBAND6           NUMBER(9,2),
   PRICEBAND7           NUMBER(9,2),
   PRICEBAND8           NUMBER(9,2),
   PRICEBAND9           NUMBER(9,2),
   PRICEBAND10          NUMBER(9,2),
   MAXRAMPUP            NUMBER(9,2),
   MAXRAMPDOWN          NUMBER(9,2),
   MINIMUMLOAD          NUMBER(6,0),
   T1                   NUMBER(6,0),
   T2                   NUMBER(6,0),
   T3                   NUMBER(6,0),
   T4                   NUMBER(6,0),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table DEFAULTDAYOFFER is
'DEFAULTDAYOFFER shows day-based details of participants default bids unit for the same day.';

comment on column DEFAULTDAYOFFER.SETTLEMENTDATE is
'Market date starting at 04:30';

comment on column DEFAULTDAYOFFER.DUID is
'Dispatchable unit Identifier';

comment on column DEFAULTDAYOFFER.VERSIONNO is
'Version No for given offer date';

comment on column DEFAULTDAYOFFER.SELFCOMMITFLAG is
'Not used';

comment on column DEFAULTDAYOFFER.DAILYENERGYCONSTRAINT is
'Maximum energy available from Energy Constrained Plant.';

comment on column DEFAULTDAYOFFER.ENTRYTYPE is
'Daily or Rebid';

comment on column DEFAULTDAYOFFER.CONTINGENCYPRICE is
'Not used';

comment on column DEFAULTDAYOFFER.REBIDEXPLANATION is
'Explanation for all rebids and inflexibilities';

comment on column DEFAULTDAYOFFER.BANDQUANTISATIONID is
'Not used';

comment on column DEFAULTDAYOFFER.PRICEBAND1 is
'Price for Availability Band 1';

comment on column DEFAULTDAYOFFER.PRICEBAND2 is
'Price for Availability Band 2';

comment on column DEFAULTDAYOFFER.PRICEBAND3 is
'Price for Availability Band 3';

comment on column DEFAULTDAYOFFER.PRICEBAND4 is
'Price for Availability Band 4';

comment on column DEFAULTDAYOFFER.PRICEBAND5 is
'Price for Availability Band 5';

comment on column DEFAULTDAYOFFER.PRICEBAND6 is
'Price for Availability Band 6';

comment on column DEFAULTDAYOFFER.PRICEBAND7 is
'Price for Availability Band 7';

comment on column DEFAULTDAYOFFER.PRICEBAND8 is
'Price for Availability Band 8';

comment on column DEFAULTDAYOFFER.PRICEBAND9 is
'Price for Availability Band 9';

comment on column DEFAULTDAYOFFER.PRICEBAND10 is
'Price for Availability Band 10';

comment on column DEFAULTDAYOFFER.MAXRAMPUP is
'Not used';

comment on column DEFAULTDAYOFFER.MAXRAMPDOWN is
'Not used';

comment on column DEFAULTDAYOFFER.MINIMUMLOAD is
'Minimum stable load';

comment on column DEFAULTDAYOFFER.T1 is
'Time to synchronise in minutes';

comment on column DEFAULTDAYOFFER.T2 is
'Time to minimum load in minutes';

comment on column DEFAULTDAYOFFER.T3 is
'Time at minimum load in minutes';

comment on column DEFAULTDAYOFFER.T4 is
'Time to shut down in minutes';

comment on column DEFAULTDAYOFFER.LASTCHANGED is
'Last date and time record changed';

alter table DEFAULTDAYOFFER
   add constraint DEFDAYOFFER_PK primary key (SETTLEMENTDATE, DUID, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: DEFAULTDAYOFFER_LCX                                   */
/*==============================================================*/
create index DEFAULTDAYOFFER_LCX on DEFAULTDAYOFFER (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DEFAULTOFFERTRK                                       */
/*==============================================================*/
create table DEFAULTOFFERTRK (
   DUID                 VARCHAR2(10)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   FILENAME             VARCHAR2(40),
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table DEFAULTOFFERTRK is
'DEFAULTOFFERTRK shows the file names of default offers submitted for each unit.';

comment on column DEFAULTOFFERTRK.DUID is
'Dispatchable Unit Identifier';

comment on column DEFAULTOFFERTRK.EFFECTIVEDATE is
'Market date default offer file is effective';

comment on column DEFAULTOFFERTRK.VERSIONNO is
'Version no of file for this date';

comment on column DEFAULTOFFERTRK.FILENAME is
'Load File identifier';

comment on column DEFAULTOFFERTRK.AUTHORISEDBY is
'User authorising record';

comment on column DEFAULTOFFERTRK.AUTHORISEDDATE is
'Date record authorised';

comment on column DEFAULTOFFERTRK.LASTCHANGED is
'Last date and time record changed';

alter table DEFAULTOFFERTRK
   add constraint DEFOFFERTRK_PK primary key (DUID, EFFECTIVEDATE, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: DEFAULTOFFERTRK_LCX                                   */
/*==============================================================*/
create index DEFAULTOFFERTRK_LCX on DEFAULTOFFERTRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DEFAULTPEROFFER                                       */
/*==============================================================*/
create table DEFAULTPEROFFER (
   SETTLEMENTDATE       DATE                  not null,
   DUID                 VARCHAR2(10)          not null,
   PERIODID             NUMBER(3,0)           not null,
   VERSIONNO            NUMBER(3,0)           not null,
   SELFDISPATCH         NUMBER(9,6),
   MAXAVAIL             NUMBER(12,6),
   FIXEDLOAD            NUMBER(9,6),
   ROCUP                NUMBER(6,0),
   ROCDOWN              NUMBER(6,0),
   LASTCHANGED          DATE,
   BANDAVAIL1           NUMBER(6,0),
   BANDAVAIL2           NUMBER(6,0),
   BANDAVAIL3           NUMBER(6,0),
   BANDAVAIL4           NUMBER(6,0),
   BANDAVAIL5           NUMBER(6,0),
   BANDAVAIL6           NUMBER(6,0),
   BANDAVAIL7           NUMBER(6,0),
   BANDAVAIL8           NUMBER(6,0),
   BANDAVAIL9           NUMBER(6,0),
   BANDAVAIL10          NUMBER(6,0),
   PASAAVAILABILITY     NUMBER(12,0)
)
   tablespace PSSTAB01;

comment on table DEFAULTPEROFFER is
'DEFAULTPEROFFER shows half hourly period-based data in the default bid for each Dispatchable Unit, such as period availability, rate of change and band quantities.';

comment on column DEFAULTPEROFFER.SETTLEMENTDATE is
'Market date starting at 04:30';

comment on column DEFAULTPEROFFER.DUID is
'Dispatchable Unit Identifier';

comment on column DEFAULTPEROFFER.PERIODID is
'Market data. Trading Interval number';

comment on column DEFAULTPEROFFER.VERSIONNO is
'Version no of the offer file.';

comment on column DEFAULTPEROFFER.SELFDISPATCH is
'Not used';

comment on column DEFAULTPEROFFER.MAXAVAIL is
'Maximum planned availability MW';

comment on column DEFAULTPEROFFER.FIXEDLOAD is
'Fixed unit output MW. A value of zero means no fixed load so the unit is dispatched according to bid and market (rather than zero fixed load)';

comment on column DEFAULTPEROFFER.ROCUP is
'Rate of change up MW/min';

comment on column DEFAULTPEROFFER.ROCDOWN is
'Rate of change down MW/min';

comment on column DEFAULTPEROFFER.LASTCHANGED is
'Last date and time record changed';

comment on column DEFAULTPEROFFER.BANDAVAIL1 is
'Availability at price band 1';

comment on column DEFAULTPEROFFER.BANDAVAIL2 is
'Availability at price band 2';

comment on column DEFAULTPEROFFER.BANDAVAIL3 is
'Availability at price band 3';

comment on column DEFAULTPEROFFER.BANDAVAIL4 is
'Availability at price band 4';

comment on column DEFAULTPEROFFER.BANDAVAIL5 is
'Availability at price band 5';

comment on column DEFAULTPEROFFER.BANDAVAIL6 is
'Availability at price band 6';

comment on column DEFAULTPEROFFER.BANDAVAIL7 is
'Availability at price band 7';

comment on column DEFAULTPEROFFER.BANDAVAIL8 is
'Availability at price band 8';

comment on column DEFAULTPEROFFER.BANDAVAIL9 is
'Availability at price band 9';

comment on column DEFAULTPEROFFER.BANDAVAIL10 is
'Availability at price band 10';

comment on column DEFAULTPEROFFER.PASAAVAILABILITY is
'The physical plant capability including any capability potentially available within 24 hours.';

alter table DEFAULTPEROFFER
   add constraint DEFPEROFFER_PK primary key (DUID, SETTLEMENTDATE, VERSIONNO, PERIODID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: DEFAULTPEROFFER_LCX                                   */
/*==============================================================*/
create index DEFAULTPEROFFER_LCX on DEFAULTPEROFFER (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DELTAMW                                               */
/*==============================================================*/
create table DELTAMW (
   SETTLEMENTDATE       DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   REGIONID             VARCHAR2(10)          not null,
   PERIODID             NUMBER(2,0)           not null,
   DELTAMW              NUMBER(6,0),
   LOWER5MIN            NUMBER(6,0),
   LOWER60SEC           NUMBER(6,0),
   LOWER6SEC            NUMBER(6,0),
   RAISE5MIN            NUMBER(6,0),
   RAISE60SEC           NUMBER(6,0),
   RAISE6SEC            NUMBER(6,0),
   LASTCHANGED          DATE,
   RAISEREG             NUMBER(6,0),
   LOWERREG             NUMBER(6,0)
)
   tablespace PSSTAB02;

comment on table DELTAMW is
'DELTAMW sets out the Frequency Control Ancillary Services (FCAS) requirement to be provided locally within each region and each half-hour period in a market day. Two fields specify Frequency Controlled Ancillary Services requirements to be provided locally for the new regulation ancillary services.';

comment on column DELTAMW.SETTLEMENTDATE is
'Market date starting at 04:30';

comment on column DELTAMW.VERSIONNO is
'Version No of record for this date';

comment on column DELTAMW.REGIONID is
'Differentiates this region from all other regions';

comment on column DELTAMW.PERIODID is
'Market trading interval from 1 to 48 starting at 04:30';

comment on column DELTAMW.DELTAMW is
'Not Used';

comment on column DELTAMW.LOWER5MIN is
'Lower 5 min local share requirement';

comment on column DELTAMW.LOWER60SEC is
'Lower 60 sec local share requirement';

comment on column DELTAMW.LOWER6SEC is
'Lower 6 sec local share requirement';

comment on column DELTAMW.RAISE5MIN is
'Raise 5 minute local share requirement';

comment on column DELTAMW.RAISE60SEC is
'Raise 60 sec local share requirement';

comment on column DELTAMW.RAISE6SEC is
'Raise 6 sec local share requirement';

comment on column DELTAMW.LASTCHANGED is
'Last date and time record changed';

comment on column DELTAMW.RAISEREG is
'Raise Regulation local share requirement';

comment on column DELTAMW.LOWERREG is
'Lower Regulation local share requirement';

alter table DELTAMW
   add constraint DELTAMW_PK primary key (SETTLEMENTDATE, VERSIONNO, REGIONID, PERIODID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: DELTAMW_LCX                                           */
/*==============================================================*/
create index DELTAMW_LCX on DELTAMW (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: DEMANDOPERATIONALACTUAL                               */
/*==============================================================*/
create table DEMANDOPERATIONALACTUAL (
   INTERVAL_DATETIME    date                  not null,
   REGIONID             Varchar2(20)          not null,
   OPERATIONAL_DEMAND   number(10,0),
   LASTCHANGED          date,
   OPERATIONAL_DEMAND_ADJUSTMENT NUMBER(10,0),
   WDR_ESTIMATE         NUMBER(10)
)
   tablespace PSSTAB02;

comment on table DEMANDOPERATIONALACTUAL is
'Shows Actual Operational Demand for a particular date time interval.';

comment on column DEMANDOPERATIONALACTUAL.INTERVAL_DATETIME is
'Date time interval for operational demand value';

comment on column DEMANDOPERATIONALACTUAL.REGIONID is
'Region identifier';

comment on column DEMANDOPERATIONALACTUAL.OPERATIONAL_DEMAND is
'Average 30-minute measured operational demand MW value (unadjusted)';

comment on column DEMANDOPERATIONALACTUAL.LASTCHANGED is
'Last date and time record changed';

comment on column DEMANDOPERATIONALACTUAL.OPERATIONAL_DEMAND_ADJUSTMENT is
'Adjustment value containing the estimated amount of activated RERT and involuntary load shedding that occurred as a result of a NER 4.8.9 instruction for load shedding from AEMO.';

comment on column DEMANDOPERATIONALACTUAL.WDR_ESTIMATE is
'Estimated average 30-minute MW amount of Wholesale Demand Response that occurred';

alter table DEMANDOPERATIONALACTUAL
   add constraint DEMANDOPERATIONALACTUAL_PK primary key (INTERVAL_DATETIME, REGIONID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: DEMANDOPERATIONALFORECAST                             */
/*==============================================================*/
create table DEMANDOPERATIONALFORECAST (
   INTERVAL_DATETIME    date                  not null,
   REGIONID             Varchar2(20)          not null,
   LOAD_DATE            date,
   OPERATIONAL_DEMAND_POE10 number(15,2),
   OPERATIONAL_DEMAND_POE50 number(15,2),
   OPERATIONAL_DEMAND_POE90 number(15,2),
   LASTCHANGED          date
)
   tablespace PSSTAB02;

comment on table DEMANDOPERATIONALFORECAST is
'Shows Forecast Operational Demand for a particular date time interval.';

comment on column DEMANDOPERATIONALFORECAST.INTERVAL_DATETIME is
'Forecast for a particular date time interval';

comment on column DEMANDOPERATIONALFORECAST.REGIONID is
'Region identifier';

comment on column DEMANDOPERATIONALFORECAST.LOAD_DATE is
'Date time this forecast was produced';

comment on column DEMANDOPERATIONALFORECAST.OPERATIONAL_DEMAND_POE10 is
'10% probability of exceedance operational demand forecast value';

comment on column DEMANDOPERATIONALFORECAST.OPERATIONAL_DEMAND_POE50 is
'50% probability of exceedance operational demand forecast value';

comment on column DEMANDOPERATIONALFORECAST.OPERATIONAL_DEMAND_POE90 is
'90% probability of exceedance operational demand forecast value';

comment on column DEMANDOPERATIONALFORECAST.LASTCHANGED is
'Last date and time record changed';

alter table DEMANDOPERATIONALFORECAST
   add constraint DEMANDOPERATIONALFORECAST_PK primary key (INTERVAL_DATETIME, REGIONID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: DISPATCHABLEUNIT                                      */
/*==============================================================*/
create table DISPATCHABLEUNIT (
   DUID                 VARCHAR2(10)          not null,
   DUNAME               VARCHAR2(20),
   UNITTYPE             VARCHAR2(20),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table DISPATCHABLEUNIT is
'DISPATCHABLEUNIT sets out the unit name and type of each dispatchable unit in the market.';

comment on column DISPATCHABLEUNIT.DUID is
'Dispatchable Unit Identifier';

comment on column DISPATCHABLEUNIT.DUNAME is
'Dispatchable Unit full description';

comment on column DISPATCHABLEUNIT.UNITTYPE is
'Identifies LOAD, GENERATOR or BIDIRECTIONAL';

comment on column DISPATCHABLEUNIT.LASTCHANGED is
'Last date and time record changed';

alter table DISPATCHABLEUNIT
   add constraint DISPATCHABLEUNIT_PK primary key (DUID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: DISPATCHABLEUNIT_LCX                                  */
/*==============================================================*/
create index DISPATCHABLEUNIT_LCX on DISPATCHABLEUNIT (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DISPATCHBIDTRK                                        */
/*==============================================================*/
create table DISPATCHBIDTRK (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0)           not null,
   OFFEREFFECTIVEDATE   DATE                  not null,
   OFFERVERSIONNO       NUMBER(3,0)           not null,
   DUID                 VARCHAR2(10)          not null,
   BIDTYPE              VARCHAR2(10),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table DISPATCHBIDTRK is
'DISPATCHBIDTRK shows the bid tracking, including the bid version used in each dispatch run for each unit. DISPATCHBIDTRK is the audit trail of the bid actually used in each dispatch.';

comment on column DISPATCHBIDTRK.SETTLEMENTDATE is
'Market date starting at 04:05';

comment on column DISPATCHBIDTRK.RUNNO is
'Dispatch run no from 1 to 288 (as per bid)';

comment on column DISPATCHBIDTRK.OFFEREFFECTIVEDATE is
'Effective date of offer used';

comment on column DISPATCHBIDTRK.OFFERVERSIONNO is
'Version no of offer used';

comment on column DISPATCHBIDTRK.DUID is
'Dispatchable unit identifier';

comment on column DISPATCHBIDTRK.BIDTYPE is
'Bid type (daily, default or rebid)';

comment on column DISPATCHBIDTRK.LASTCHANGED is
'Last date and time record changed';

alter table DISPATCHBIDTRK
   add constraint DISPATCHBIDTRK_PK primary key (SETTLEMENTDATE, RUNNO, OFFEREFFECTIVEDATE, OFFERVERSIONNO, DUID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: DISPATCHBIDTRK_NDX2                                   */
/*==============================================================*/
create index DISPATCHBIDTRK_NDX2 on DISPATCHBIDTRK (
   DUID ASC,
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: DISPATCHBIDTRK_LCX                                    */
/*==============================================================*/
create index DISPATCHBIDTRK_LCX on DISPATCHBIDTRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DISPATCHBLOCKEDCONSTRAINT                             */
/*==============================================================*/
create table DISPATCHBLOCKEDCONSTRAINT (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0)           not null,
   CONSTRAINTID         VARCHAR2(20)          not null
)
   tablespace PSSTAB01;

comment on table DISPATCHBLOCKEDCONSTRAINT is
'DISPATCH Blocked Constraints lists any constraints that were blocked in a dispatch run. If no constraints are blocked, there will be no rows for that dispatch run.';

comment on column DISPATCHBLOCKEDCONSTRAINT.SETTLEMENTDATE is
'Dispatch Interval';

comment on column DISPATCHBLOCKEDCONSTRAINT.RUNNO is
'Dispatch run no; always 1';

comment on column DISPATCHBLOCKEDCONSTRAINT.CONSTRAINTID is
'Generic Constraint identifier (synonymous with GenConID)';

alter table DISPATCHBLOCKEDCONSTRAINT
   add constraint DISPATCHBLOCKEDCONSTRAINT_PK primary key (SETTLEMENTDATE, RUNNO, CONSTRAINTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: DISPATCHCASESOLUTION                                  */
/*==============================================================*/
create table DISPATCHCASESOLUTION (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0)           not null,
   INTERVENTION         NUMBER(2,0)           not null,
   CASESUBTYPE          VARCHAR2(3),
   SOLUTIONSTATUS       NUMBER(2,0),
   SPDVERSION           VARCHAR2(20),
   NONPHYSICALLOSSES    NUMBER(1,0),
   TOTALOBJECTIVE       NUMBER(27,10),
   TOTALAREAGENVIOLATION NUMBER(15,5),
   TOTALINTERCONNECTORVIOLATION NUMBER(15,5),
   TOTALGENERICVIOLATION NUMBER(15,5),
   TOTALRAMPRATEVIOLATION NUMBER(15,5),
   TOTALUNITMWCAPACITYVIOLATION NUMBER(15,5),
   TOTAL5MINVIOLATION   NUMBER(15,5),
   TOTALREGVIOLATION    NUMBER(15,5),
   TOTAL6SECVIOLATION   NUMBER(15,5),
   TOTAL60SECVIOLATION  NUMBER(15,5),
   TOTALASPROFILEVIOLATION NUMBER(15,5),
   TOTALFASTSTARTVIOLATION NUMBER(15,5),
   TOTALENERGYOFFERVIOLATION NUMBER(15,5),
   LASTCHANGED          DATE,
   SWITCHRUNINITIALSTATUS NUMBER(1,0),
   SWITCHRUNBESTSTATUS  NUMBER(1,0),
   SWITCHRUNBESTSTATUS_INT NUMBER(1,0)
)
   tablespace PSSTAB02;

comment on table DISPATCHCASESOLUTION is
'DISPATCHCASESOLUTION shows information relating to the complete dispatch run. The fields in DISPATCHCASESOLUTION provide an overview of the dispatch run results allowing immediate identification of conditions such as energy or FCAS deficiencies.';

comment on column DISPATCHCASESOLUTION.SETTLEMENTDATE is
'Date and time of the dispatch interval (e.g. five minute dispatch interval ending 28/09/2000 16:35)';

comment on column DISPATCHCASESOLUTION.RUNNO is
'Dispatch run no; always 1';

comment on column DISPATCHCASESOLUTION.INTERVENTION is
'Intervention flag - refer to package documentation for definition and practical query examples';

comment on column DISPATCHCASESOLUTION.CASESUBTYPE is
'Overconstrained dispatch indicator: 
* OCD = detecting over-constrained dispatch
* null = no special condition';

comment on column DISPATCHCASESOLUTION.SOLUTIONSTATUS is
'If non-zero indicated one of the following conditions:
* 1 = Supply Scarcity, Excess generation or constraint violations
* X = Model failure';

comment on column DISPATCHCASESOLUTION.SPDVERSION is
'Current version of SPD';

comment on column DISPATCHCASESOLUTION.NONPHYSICALLOSSES is
'Non-Physical Losses algorithm invoked occurred during this run';

comment on column DISPATCHCASESOLUTION.TOTALOBJECTIVE is
'The Objective function from the LP';

comment on column DISPATCHCASESOLUTION.TOTALAREAGENVIOLATION is
'Total Region Demand violations';

comment on column DISPATCHCASESOLUTION.TOTALINTERCONNECTORVIOLATION is
'Total interconnector violations';

comment on column DISPATCHCASESOLUTION.TOTALGENERICVIOLATION is
'Total generic constraint violations';

comment on column DISPATCHCASESOLUTION.TOTALRAMPRATEVIOLATION is
'Total ramp rate violations';

comment on column DISPATCHCASESOLUTION.TOTALUNITMWCAPACITYVIOLATION is
'Total unit capacity violations';

comment on column DISPATCHCASESOLUTION.TOTAL5MINVIOLATION is
'Total of 5 minute ancillary service region violations';

comment on column DISPATCHCASESOLUTION.TOTALREGVIOLATION is
'Total of Regulation ancillary service region violations';

comment on column DISPATCHCASESOLUTION.TOTAL6SECVIOLATION is
'Total of 6 second ancillary service region violations';

comment on column DISPATCHCASESOLUTION.TOTAL60SECVIOLATION is
'Total of 60 second ancillary service region violations';

comment on column DISPATCHCASESOLUTION.TOTALASPROFILEVIOLATION is
'Total of ancillary service trader profile violations';

comment on column DISPATCHCASESOLUTION.TOTALFASTSTARTVIOLATION is
'Total of fast start trader profile violations';

comment on column DISPATCHCASESOLUTION.TOTALENERGYOFFERVIOLATION is
'Total of unit summated offer band violations';

comment on column DISPATCHCASESOLUTION.LASTCHANGED is
'Last date and time record changed';

comment on column DISPATCHCASESOLUTION.SWITCHRUNINITIALSTATUS is
'Flag indicating the SCADA status for FCAS Interconnector dead-band. "0" if SCADA Status or requesting Constraint not invoked. "1" if SCADA Status AND requesting Constraint is invoked';

comment on column DISPATCHCASESOLUTION.SWITCHRUNBESTSTATUS is
'Flag indicating which Switch run was used for the Solution  from PeriodSolution';

comment on column DISPATCHCASESOLUTION.SWITCHRUNBESTSTATUS_INT is
'Flag indicating which Switch run was used for the Intervention Physical Solution - from PeriodSolution';

alter table DISPATCHCASESOLUTION
   add constraint DISPATCHCASESOLUTION_PK primary key (SETTLEMENTDATE, RUNNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: DISPATCHCASESOLUTION_LCX                              */
/*==============================================================*/
create index DISPATCHCASESOLUTION_LCX on DISPATCHCASESOLUTION (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: DISPATCHCASESOLUTION_BNC                              */
/*==============================================================*/
create table DISPATCHCASESOLUTION_BNC (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0)           not null,
   INTERVENTION         NUMBER(2,0)           not null,
   CASESUBTYPE          VARCHAR2(3),
   SOLUTIONSTATUS       NUMBER(2,0),
   SPDVERSION           NUMBER(10,3),
   STARTPERIOD          VARCHAR2(20),
   NONPHYSICALLOSSES    NUMBER(1,0),
   TOTALOBJECTIVE       NUMBER(27,10),
   TOTALAREAGENVIOLATION NUMBER(15,5),
   TOTALINTERCONNECTORVIOLATION NUMBER(15,5),
   TOTALGENERICVIOLATION NUMBER(15,5),
   TOTALRAMPRATEVIOLATION NUMBER(15,5),
   TOTALUNITMWCAPACITYVIOLATION NUMBER(15,5),
   TOTAL5MINVIOLATION   NUMBER(15,5),
   TOTALREGVIOLATION    NUMBER(15,5),
   TOTAL6SECVIOLATION   NUMBER(15,5),
   TOTAL60SECVIOLATION  NUMBER(15,5),
   TOTALENERGYCONSTRVIOLATION NUMBER(15,5),
   TOTALENERGYOFFERVIOLATION NUMBER(15,5),
   TOTALASPROFILEVIOLATION NUMBER(15,5),
   TOTALFASTSTARTVIOLATION NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table DISPATCHCASESOLUTION_BNC is
'DISPATCHCASESOLUTION_BNC was discontinued on 30 September 2009. Prior: DISPATCHCASESOLUTION_BNC is the key data to indicate when a binding intra-regional network constraints (BNC) re-run actually occurred.';

comment on column DISPATCHCASESOLUTION_BNC.SETTLEMENTDATE is
'End date and time of the dispatch interval';

comment on column DISPATCHCASESOLUTION_BNC.RUNNO is
'Dispatch run no; always 1';

comment on column DISPATCHCASESOLUTION_BNC.INTERVENTION is
'Manual intervention flag';

comment on column DISPATCHCASESOLUTION_BNC.CASESUBTYPE is
'always BNC';

comment on column DISPATCHCASESOLUTION_BNC.SOLUTIONSTATUS is
'If non-zero indicated one of the following conditions:
* 1 = Supply Scarcity, Excess generation or constraint violations
* X = Model failure';

comment on column DISPATCHCASESOLUTION_BNC.SPDVERSION is
'Current version of SPD';

comment on column DISPATCHCASESOLUTION_BNC.STARTPERIOD is
'Period identifier of first interval of the case (yyyymmddppp)';

comment on column DISPATCHCASESOLUTION_BNC.NONPHYSICALLOSSES is
'Non-Physical Losses algorithm invoked occurred during this run';

comment on column DISPATCHCASESOLUTION_BNC.TOTALOBJECTIVE is
'The Objective function from the LP';

comment on column DISPATCHCASESOLUTION_BNC.TOTALAREAGENVIOLATION is
'Total Region Demand violations';

comment on column DISPATCHCASESOLUTION_BNC.TOTALINTERCONNECTORVIOLATION is
'Total interconnector violations';

comment on column DISPATCHCASESOLUTION_BNC.TOTALGENERICVIOLATION is
'Total generic constraint violations';

comment on column DISPATCHCASESOLUTION_BNC.TOTALRAMPRATEVIOLATION is
'Total ramp rate violations';

comment on column DISPATCHCASESOLUTION_BNC.TOTALUNITMWCAPACITYVIOLATION is
'Total unit capacity violations';

comment on column DISPATCHCASESOLUTION_BNC.TOTAL5MINVIOLATION is
'Total of 5 minute ancillary service region violations';

comment on column DISPATCHCASESOLUTION_BNC.TOTALREGVIOLATION is
'Total of Regulation ancillary service region violations';

comment on column DISPATCHCASESOLUTION_BNC.TOTAL6SECVIOLATION is
'Total of 6 second ancillary service region violations';

comment on column DISPATCHCASESOLUTION_BNC.TOTAL60SECVIOLATION is
'Total of 60 second ancillary service region violations';

comment on column DISPATCHCASESOLUTION_BNC.TOTALENERGYOFFERVIOLATION is
'Total of unit summated offer band violations';

comment on column DISPATCHCASESOLUTION_BNC.TOTALASPROFILEVIOLATION is
'Total of ancillary service trader profile violations';

comment on column DISPATCHCASESOLUTION_BNC.TOTALFASTSTARTVIOLATION is
'Total of fast start trader profile violations';

comment on column DISPATCHCASESOLUTION_BNC.LASTCHANGED is
'Last date and time record changed';

alter table DISPATCHCASESOLUTION_BNC
   add constraint PK_DISPATCHCASESOLUTION_BNC primary key (SETTLEMENTDATE, RUNNO, INTERVENTION)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: DISPATCHCASESOLUTION_BNC_LCX                          */
/*==============================================================*/
create index DISPATCHCASESOLUTION_BNC_LCX on DISPATCHCASESOLUTION_BNC (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: DISPATCHCASE_OCD                                      */
/*==============================================================*/
create table DISPATCHCASE_OCD (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0)           not null,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table DISPATCHCASE_OCD is
'DISPATCHCASE_OCD shows the key data to indicate when an over-constrained dispatch (OCD) re-run actually occurred. One record per over-constrained dispatch interval.';

comment on column DISPATCHCASE_OCD.SETTLEMENTDATE is
'End date and time of the dispatch interval';

comment on column DISPATCHCASE_OCD.RUNNO is
'Dispatch run no; always 1';

comment on column DISPATCHCASE_OCD.LASTCHANGED is
'Last date and time record changed';

alter table DISPATCHCASE_OCD
   add constraint DISPATCHCASE_OCD_PK primary key (SETTLEMENTDATE, RUNNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: DISPATCHCASE_OCD_LCX                                  */
/*==============================================================*/
create index DISPATCHCASE_OCD_LCX on DISPATCHCASE_OCD (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DISPATCHCONSTRAINT                                    */
/*==============================================================*/
create table DISPATCHCONSTRAINT (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0)           not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   DISPATCHINTERVAL     NUMBER(22,0)          not null,
   INTERVENTION         NUMBER(2,0)           not null,
   RHS                  NUMBER(15,5),
   MARGINALVALUE        NUMBER(15,5),
   VIOLATIONDEGREE      NUMBER(15,5),
   LASTCHANGED          DATE,
   DUID                 VARCHAR2(20),
   GENCONID_EFFECTIVEDATE DATE,
   GENCONID_VERSIONNO   NUMBER(22,0),
   LHS                  number(15,5)
)
   tablespace PSSTAB03;

comment on table DISPATCHCONSTRAINT is
'DISPATCHCONSTRAINT sets out details of all binding and interregion constraints in each dispatch run. Note: invoked constraints can be established from GENCONSETINVOKE. Binding constraints show as marginal value >$0. Interconnector constraints are listed so RHS (SCADA calculated limits) can be reported.';

comment on column DISPATCHCONSTRAINT.SETTLEMENTDATE is
'Market date starting at 04:05';

comment on column DISPATCHCONSTRAINT.RUNNO is
'Dispatch run no; always 1';

comment on column DISPATCHCONSTRAINT.CONSTRAINTID is
'Generic Constraint identifier (synonymous with GenConID)';

comment on column DISPATCHCONSTRAINT.DISPATCHINTERVAL is
'Dispatch period identifier, from 001 to 288 in format YYYYMMDDPPP.';

comment on column DISPATCHCONSTRAINT.INTERVENTION is
'Manual Intervention flag, which, if set (1), causes predispatch to solve twice.';

comment on column DISPATCHCONSTRAINT.RHS is
'Right hand Side value as used in dispatch.';

comment on column DISPATCHCONSTRAINT.MARGINALVALUE is
'$ Value of binding constraint';

comment on column DISPATCHCONSTRAINT.VIOLATIONDEGREE is
'Degree of violation in MW';

comment on column DISPATCHCONSTRAINT.LASTCHANGED is
'Last date and time record changed';

comment on column DISPATCHCONSTRAINT.DUID is
'DUID to which the Constraint is confidential. Null denotes non-confidential';

comment on column DISPATCHCONSTRAINT.GENCONID_EFFECTIVEDATE is
'Effective date of the Generic Constraint (ConstraintID). This field is used to track the version of this generic constraint applied in this dispatch interval';

comment on column DISPATCHCONSTRAINT.GENCONID_VERSIONNO is
'Version number of the Generic Constraint (ConstraintID). This field is used to track the version of this generic constraint applied in this dispatch interval';

comment on column DISPATCHCONSTRAINT.LHS is
'Aggregation of the constraints LHS term solution values';

alter table DISPATCHCONSTRAINT
   add constraint PK_DISPATCHCONSTRAINT primary key (SETTLEMENTDATE, RUNNO, CONSTRAINTID, DISPATCHINTERVAL, INTERVENTION)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Index: DISPATCHCONSTRAINT_NDX2                               */
/*==============================================================*/
create index DISPATCHCONSTRAINT_NDX2 on DISPATCHCONSTRAINT (
   SETTLEMENTDATE ASC
)
tablespace PSSIND03;

/*==============================================================*/
/* Index: DISPATCHCONSTRAINT_LCX                                */
/*==============================================================*/
create index DISPATCHCONSTRAINT_LCX on DISPATCHCONSTRAINT (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: DISPATCHINTERCONNECTORRES                             */
/*==============================================================*/
create table DISPATCHINTERCONNECTORRES (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0)           not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   DISPATCHINTERVAL     NUMBER(22,0)          not null,
   INTERVENTION         NUMBER(2,0)           not null,
   METEREDMWFLOW        NUMBER(15,5),
   MWFLOW               NUMBER(15,5),
   MWLOSSES             NUMBER(15,5),
   MARGINALVALUE        NUMBER(15,5),
   VIOLATIONDEGREE      NUMBER(15,5),
   LASTCHANGED          DATE,
   EXPORTLIMIT          NUMBER(15,5),
   IMPORTLIMIT          NUMBER(15,5),
   MARGINALLOSS         NUMBER(15,5),
   EXPORTGENCONID       VARCHAR2(20),
   IMPORTGENCONID       VARCHAR2(20),
   FCASEXPORTLIMIT      NUMBER(15,5),
   FCASIMPORTLIMIT      NUMBER(15,5),
   LOCAL_PRICE_ADJUSTMENT_EXPORT NUMBER(10,2),
   LOCALLY_CONSTRAINED_EXPORT NUMBER(1,0),
   LOCAL_PRICE_ADJUSTMENT_IMPORT NUMBER(10,2),
   LOCALLY_CONSTRAINED_IMPORT NUMBER(1,0)
)
   tablespace PSSTAB03;

comment on table DISPATCHINTERCONNECTORRES is
'DISPATCHINTERCONNECTORRES sets out MW flow and losses on each interconnector for each dispatch period, including fields for the Frequency Controlled Ancillary Services export and import limits and extra reporting of the generic constraints set the energy import and export limits.';

comment on column DISPATCHINTERCONNECTORRES.SETTLEMENTDATE is
'Market date starting at 04:05';

comment on column DISPATCHINTERCONNECTORRES.RUNNO is
'Dispatch run no; always 1';

comment on column DISPATCHINTERCONNECTORRES.INTERCONNECTORID is
'Interconnector identifier';

comment on column DISPATCHINTERCONNECTORRES.DISPATCHINTERVAL is
'Dispatch period identifier, from 001 to 288 in format YYYYMMDDPPP.';

comment on column DISPATCHINTERCONNECTORRES.INTERVENTION is
'Intervention case or not';

comment on column DISPATCHINTERCONNECTORRES.METEREDMWFLOW is
'Metered MW Flow from SCADA.';

comment on column DISPATCHINTERCONNECTORRES.MWFLOW is
'Target MW Flow for next 5 mins.';

comment on column DISPATCHINTERCONNECTORRES.MWLOSSES is
'Calculated MW Losses';

comment on column DISPATCHINTERCONNECTORRES.MARGINALVALUE is
'Shadow price resulting from thermal or reserve sharing constraints on Interconnector import/export (0 unless binding) - NEMDE Solution InterconnectorSolution element "Price" attribute';

comment on column DISPATCHINTERCONNECTORRES.VIOLATIONDEGREE is
'Degree of violation on interconnector constraints';

comment on column DISPATCHINTERCONNECTORRES.LASTCHANGED is
'Last changed.';

comment on column DISPATCHINTERCONNECTORRES.EXPORTLIMIT is
'Calculated export limit applying to energy only.';

comment on column DISPATCHINTERCONNECTORRES.IMPORTLIMIT is
'Calculated import limit applying to energy only.';

comment on column DISPATCHINTERCONNECTORRES.MARGINALLOSS is
'Marginal loss factor. Use this to adjust prices between regions.';

comment on column DISPATCHINTERCONNECTORRES.EXPORTGENCONID is
'Generic Constraint setting the export limit';

comment on column DISPATCHINTERCONNECTORRES.IMPORTGENCONID is
'Generic Constraint setting the import limit';

comment on column DISPATCHINTERCONNECTORRES.FCASEXPORTLIMIT is
'Calculated export limit applying to energy + FCAS.';

comment on column DISPATCHINTERCONNECTORRES.FCASIMPORTLIMIT is
'Calculated import limit applying to energy + FCAS.';

comment on column DISPATCHINTERCONNECTORRES.LOCAL_PRICE_ADJUSTMENT_EXPORT is
'Aggregate Constraint contribution cost of this Interconnector: Sum(MarginalValue x Factor) for all relevant Constraints, for Export (Factor >= 0)';

comment on column DISPATCHINTERCONNECTORRES.LOCALLY_CONSTRAINED_EXPORT is
'Key for Local_Price_Adjustment_Export: 2 = at least one Outage Constraint; 1 = at least 1 System Normal Constraint (and no Outage Constraint); 0 = No System Normal or Outage Constraints';

comment on column DISPATCHINTERCONNECTORRES.LOCAL_PRICE_ADJUSTMENT_IMPORT is
'Aggregate Constraint contribution cost of this Interconnector: Sum(MarginalValue x Factor) for all relevant Constraints, for Import (Factor >= 0)';

comment on column DISPATCHINTERCONNECTORRES.LOCALLY_CONSTRAINED_IMPORT is
'Key for Local_Price_Adjustment_Import: 2 = at least one Outage Constraint; 1 = at least 1 System Normal Constraint (and no Outage Constraint); 0 = No System Normal or Outage Constraints';

alter table DISPATCHINTERCONNECTORRES
   add constraint PK_DISPATCHINTERCONNECTORRES primary key (SETTLEMENTDATE, RUNNO, INTERCONNECTORID, DISPATCHINTERVAL, INTERVENTION)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: DISPATCHINTERCONNECTORRES_LCX                         */
/*==============================================================*/
create index DISPATCHINTERCONNECTORRES_LCX on DISPATCHINTERCONNECTORRES (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: DISPATCHLOAD                                          */
/*==============================================================*/
create table DISPATCHLOAD (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0)           not null,
   DUID                 VARCHAR2(10)          not null,
   TRADETYPE            NUMBER(2,0),
   DISPATCHINTERVAL     NUMBER(22,0),
   INTERVENTION         NUMBER(2,0)           not null,
   CONNECTIONPOINTID    VARCHAR2(12),
   DISPATCHMODE         NUMBER(2,0),
   AGCSTATUS            NUMBER(2,0),
   INITIALMW            NUMBER(15,5),
   TOTALCLEARED         NUMBER(15,5),
   RAMPDOWNRATE         NUMBER(15,5),
   RAMPUPRATE           NUMBER(15,5),
   LOWER5MIN            NUMBER(15,5),
   LOWER60SEC           NUMBER(15,5),
   LOWER6SEC            NUMBER(15,5),
   RAISE5MIN            NUMBER(15,5),
   RAISE60SEC           NUMBER(15,5),
   RAISE6SEC            NUMBER(15,5),
   DOWNEPF              NUMBER(15,5),
   UPEPF                NUMBER(15,5),
   MARGINAL5MINVALUE    NUMBER(15,5),
   MARGINAL60SECVALUE   NUMBER(15,5),
   MARGINAL6SECVALUE    NUMBER(15,5),
   MARGINALVALUE        NUMBER(15,5),
   VIOLATION5MINDEGREE  NUMBER(15,5),
   VIOLATION60SECDEGREE NUMBER(15,5),
   VIOLATION6SECDEGREE  NUMBER(15,5),
   VIOLATIONDEGREE      NUMBER(15,5),
   LASTCHANGED          DATE,
   LOWERREG             NUMBER(15,5),
   RAISEREG             NUMBER(15,5),
   AVAILABILITY         NUMBER(15,5),
   RAISE6SECFLAGS       NUMBER(3,0),
   RAISE60SECFLAGS      NUMBER(3,0),
   RAISE5MINFLAGS       NUMBER(3,0),
   RAISEREGFLAGS        NUMBER(3,0),
   LOWER6SECFLAGS       NUMBER(3,0),
   LOWER60SECFLAGS      NUMBER(3,0),
   LOWER5MINFLAGS       NUMBER(3,0),
   LOWERREGFLAGS        NUMBER(3,0),
   RAISEREGAVAILABILITY NUMBER(15,5),
   RAISEREGENABLEMENTMAX NUMBER(15,5),
   RAISEREGENABLEMENTMIN NUMBER(15,5),
   LOWERREGAVAILABILITY NUMBER(15,5),
   LOWERREGENABLEMENTMAX NUMBER(15,5),
   LOWERREGENABLEMENTMIN NUMBER(15,5),
   RAISE6SECACTUALAVAILABILITY NUMBER(16,6),
   RAISE60SECACTUALAVAILABILITY NUMBER(16,6),
   RAISE5MINACTUALAVAILABILITY NUMBER(16,6),
   RAISEREGACTUALAVAILABILITY NUMBER(16,6),
   LOWER6SECACTUALAVAILABILITY NUMBER(16,6),
   LOWER60SECACTUALAVAILABILITY NUMBER(16,6),
   LOWER5MINACTUALAVAILABILITY NUMBER(16,6),
   LOWERREGACTUALAVAILABILITY NUMBER(16,6),
   SEMIDISPATCHCAP      NUMBER(3,0),
   DISPATCHMODETIME     NUMBER(4,0),
   CONFORMANCE_MODE     NUMBER(6,0),
   UIGF                 NUMBER(15,5),
   RAISE1SEC            NUMBER(15,5),
   RAISE1SECFLAGS       NUMBER(3,0),
   LOWER1SEC            NUMBER(15,5),
   LOWER1SECFLAGS       NUMBER(3,0),
   RAISE1SECACTUALAVAILABILITY NUMBER(16,6),
   LOWER1SECACTUALAVAILABILITY NUMBER(16,6),
   INITIAL_ENERGY_STORAGE NUMBER(15,5),
   ENERGY_STORAGE       NUMBER(15,5),
   MIN_AVAILABILITY     NUMBER(15,5),
   ELEMENT_CAP          NUMBER(5,0)
)
   tablespace PSSTAB03;

comment on table DISPATCHLOAD is
'DISPATCHLOAD set out the current SCADA MW and target MW for each dispatchable unit, including relevant Frequency Control Ancillary Services (FCAS) enabling targets for each five minutes and additional fields to handle the new Ancillary Services functionality. Fast Start Plant status is indicated by dispatch mode.';

comment on column DISPATCHLOAD.SETTLEMENTDATE is
'Market date and time starting at 04:05';

comment on column DISPATCHLOAD.RUNNO is
'Dispatch run no; always 1';

comment on column DISPATCHLOAD.DUID is
'Dispatchable unit identifier';

comment on column DISPATCHLOAD.TRADETYPE is
'Not used';

comment on column DISPATCHLOAD.DISPATCHINTERVAL is
'Dispatch period identifier, from 001 to 288 in format YYYYMMDDPPP.';

comment on column DISPATCHLOAD.INTERVENTION is
'Intervention flag if intervention run';

comment on column DISPATCHLOAD.CONNECTIONPOINTID is
'Connection point identifier for DUID';

comment on column DISPATCHLOAD.DISPATCHMODE is
'Dispatch mode for fast start plant (0 to 4).';

comment on column DISPATCHLOAD.AGCSTATUS is
'AGC Status from EMS
* 1 = on
* 0 = off';

comment on column DISPATCHLOAD.INITIALMW is
'Initial MW at start of period. Negative values when Bi-directional Unit start from importing power, otherwise positive.';

comment on column DISPATCHLOAD.TOTALCLEARED is
'Target MW for end of period. Negative values when Bi-directional Unit is importing power, otherwise positive.';

comment on column DISPATCHLOAD.RAMPDOWNRATE is
'Ramp down rate used in dispatch (lesser of bid or telemetered rate).';

comment on column DISPATCHLOAD.RAMPUPRATE is
'Ramp up rate (lesser of bid or telemetered rate).';

comment on column DISPATCHLOAD.LOWER5MIN is
'Lower 5 min reserve target';

comment on column DISPATCHLOAD.LOWER60SEC is
'Lower 60 sec reserve target';

comment on column DISPATCHLOAD.LOWER6SEC is
'Lower 6 sec reserve target';

comment on column DISPATCHLOAD.RAISE5MIN is
'Raise 5 min reserve target';

comment on column DISPATCHLOAD.RAISE60SEC is
'Raise 60 sec reserve target';

comment on column DISPATCHLOAD.RAISE6SEC is
'Raise 6 sec reserve target';

comment on column DISPATCHLOAD.DOWNEPF is
'Not Used';

comment on column DISPATCHLOAD.UPEPF is
'Not Used';

comment on column DISPATCHLOAD.MARGINAL5MINVALUE is
'Marginal $ value for 5 min';

comment on column DISPATCHLOAD.MARGINAL60SECVALUE is
'Marginal $ value for 60 seconds';

comment on column DISPATCHLOAD.MARGINAL6SECVALUE is
'Marginal $ value for 6 seconds';

comment on column DISPATCHLOAD.MARGINALVALUE is
'Marginal $ value for energy';

comment on column DISPATCHLOAD.VIOLATION5MINDEGREE is
'Violation MW 5 min';

comment on column DISPATCHLOAD.VIOLATION60SECDEGREE is
'Violation MW 60 seconds';

comment on column DISPATCHLOAD.VIOLATION6SECDEGREE is
'Violation MW 6 seconds';

comment on column DISPATCHLOAD.VIOLATIONDEGREE is
'Violation MW energy';

comment on column DISPATCHLOAD.LASTCHANGED is
'Last date and time record changed';

comment on column DISPATCHLOAD.LOWERREG is
'Lower Regulation reserve target';

comment on column DISPATCHLOAD.RAISEREG is
'Raise Regulation reserve target';

comment on column DISPATCHLOAD.AVAILABILITY is
'For Scheduled units, this is the MAXAVAIL bid availability For Semi-scheduled units, this is the lower of MAXAVAIL bid availability and UIGF';

comment on column DISPATCHLOAD.RAISE6SECFLAGS is
'Raise 6sec status flag  - see';

comment on column DISPATCHLOAD.RAISE60SECFLAGS is
'Raise 60sec status flag  - see';

comment on column DISPATCHLOAD.RAISEREGFLAGS is
'Raise Reg status flag  - see';

comment on column DISPATCHLOAD.LOWER6SECFLAGS is
'Lower 6sec status flag  - see';

comment on column DISPATCHLOAD.LOWER60SECFLAGS is
'Lower 60sec status flag';

comment on column DISPATCHLOAD.LOWER5MINFLAGS is
'Lower 5min status flag';

comment on column DISPATCHLOAD.LOWERREGFLAGS is
'Lower Reg status flag  - see';

comment on column DISPATCHLOAD.RAISEREGAVAILABILITY is
'RaiseReg availability - minimum of bid and telemetered value';

comment on column DISPATCHLOAD.RAISEREGENABLEMENTMAX is
'RaiseReg enablement max point - minimum of bid and telemetered value';

comment on column DISPATCHLOAD.RAISEREGENABLEMENTMIN is
'RaiseReg Enablement Min point - maximum of bid and telemetered value';

comment on column DISPATCHLOAD.LOWERREGAVAILABILITY is
'Lower Reg availability - minimum of bid and telemetered value';

comment on column DISPATCHLOAD.LOWERREGENABLEMENTMAX is
'Lower Reg enablement Max point - minimum of bid and telemetered value';

comment on column DISPATCHLOAD.LOWERREGENABLEMENTMIN is
'Lower Reg Enablement Min point - maximum of bid and telemetered value';

comment on column DISPATCHLOAD.RAISE6SECACTUALAVAILABILITY is
'trapezium adjusted raise 6sec availability';

comment on column DISPATCHLOAD.RAISE60SECACTUALAVAILABILITY is
'trapezium adjusted raise 60sec availability';

comment on column DISPATCHLOAD.RAISE5MINACTUALAVAILABILITY is
'trapezium adjusted raise 5min availability';

comment on column DISPATCHLOAD.RAISEREGACTUALAVAILABILITY is
'trapezium adjusted raise reg availability';

comment on column DISPATCHLOAD.LOWER6SECACTUALAVAILABILITY is
'trapezium adjusted lower 6sec availability';

comment on column DISPATCHLOAD.LOWER60SECACTUALAVAILABILITY is
'trapezium adjusted lower 60sec availability';

comment on column DISPATCHLOAD.LOWER5MINACTUALAVAILABILITY is
'trapezium adjusted lower 5min availability';

comment on column DISPATCHLOAD.LOWERREGACTUALAVAILABILITY is
'trapezium adjusted lower reg availability';

comment on column DISPATCHLOAD.SEMIDISPATCHCAP is
'Boolean representation flagging if the Target is Capped';

comment on column DISPATCHLOAD.DISPATCHMODETIME is
'Minutes for which the unit has been in the current DISPATCHMODE. From NEMDE TRADERSOLUTION element FSTARGETMODETIME attribute.';

comment on column DISPATCHLOAD.CONFORMANCE_MODE is
'Mode specific to units within an aggregate. 0 - no monitoring, 1 - aggregate monitoring, 2 - individual monitoring due to constraint';

comment on column DISPATCHLOAD.UIGF is
'For Semi-Scheduled units. Unconstrained Intermittent Generation Forecast value provided to NEMDE';

comment on column DISPATCHLOAD.RAISE1SEC is
'Dispatched Raise1Sec - TraderSolution element R1Target attribute';

comment on column DISPATCHLOAD.RAISE1SECFLAGS is
'TraderSolution element R1Flags attribute';

comment on column DISPATCHLOAD.LOWER1SEC is
'Dispatched Lower1Sec - TraderSolution element L1Target attribute';

comment on column DISPATCHLOAD.LOWER1SECFLAGS is
'TraderSolution element L1Flags attribute';

comment on column DISPATCHLOAD.RAISE1SECACTUALAVAILABILITY is
'Trapezium adjusted Raise 1Sec Availability';

comment on column DISPATCHLOAD.LOWER1SECACTUALAVAILABILITY is
'Trapezium adjusted Lower 1Sec Availability';

comment on column DISPATCHLOAD.INITIAL_ENERGY_STORAGE is
'The energy storage at the start of this dispatch interval(MWh)';

comment on column DISPATCHLOAD.ENERGY_STORAGE is
'The projected energy storage based on cleared energy and regulation FCAS dispatch(MWh)';

comment on column DISPATCHLOAD.MIN_AVAILABILITY is
'BDU only. Load side availability (BidOfferPeriod.MAXAVAIL where DIRECTION = LOAD)';

comment on column DISPATCHLOAD.ELEMENT_CAP is
'Cap on the number of turbines or inverters at a DUID.';

alter table DISPATCHLOAD
   add constraint PK_DISPATCHLOAD primary key (SETTLEMENTDATE, RUNNO, DUID, INTERVENTION)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Index: DISPATCHLOAD_LCX                                      */
/*==============================================================*/
create index DISPATCHLOAD_LCX on DISPATCHLOAD (
   LASTCHANGED ASC
)
tablespace PSSIND03;

/*==============================================================*/
/* Index: DISPATCHLOAD_NDX2                                     */
/*==============================================================*/
create index DISPATCHLOAD_NDX2 on DISPATCHLOAD (
   DUID ASC,
   LASTCHANGED ASC
)
tablespace PSSIND03;

/*==============================================================*/
/* Table: DISPATCHLOAD_BNC                                      */
/*==============================================================*/
create table DISPATCHLOAD_BNC (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0)           not null,
   DUID                 VARCHAR2(10)          not null,
   INTERVENTION         NUMBER(2,0)           not null,
   CONNECTIONPOINTID    VARCHAR2(12),
   DISPATCHMODE         NUMBER(2,0),
   TOTALCLEARED         NUMBER(15,5),
   RAISEREG             NUMBER(15,5),
   RAISE5MIN            NUMBER(15,5),
   RAISE60SEC           NUMBER(15,5),
   RAISE6SEC            NUMBER(15,5),
   LOWERREG             NUMBER(15,5),
   LOWER5MIN            NUMBER(15,5),
   LOWER60SEC           NUMBER(15,5),
   LOWER6SEC            NUMBER(15,5),
   RAISEREGFLAGS        NUMBER(3,0),
   RAISE5MINFLAGS       NUMBER(3,0),
   RAISE60SECFLAGS      NUMBER(3,0),
   RAISE6SECFLAGS       NUMBER(3,0),
   LOWERREGFLAGS        NUMBER(3,0),
   LOWER5MINFLAGS       NUMBER(3,0),
   LOWER60SECFLAGS      NUMBER(3,0),
   LOWER6SECFLAGS       NUMBER(3,0),
   LASTCHANGED          DATE
)
   tablespace PSSTAB03;

comment on table DISPATCHLOAD_BNC is
'DISPATCHLOAD_BNC was discontinued on 30 September 2009. Prior: DISPATCHLOAD_BNC gives binding intra-regional network constraints (BNC) re-run dispatch results for all scheduled generating units. DISPATCHLOAD_BNC has a similar structure to DISPATCHLOAD but does not repeat input type data (e.g. InitialMW, AGCStatus) since these values are available from DISPATCHLOAD.';

comment on column DISPATCHLOAD_BNC.SETTLEMENTDATE is
'End date and time of the dispatch interval';

comment on column DISPATCHLOAD_BNC.RUNNO is
'Dispatch run no; always 1';

comment on column DISPATCHLOAD_BNC.DUID is
'Dispatchable unit identifier';

comment on column DISPATCHLOAD_BNC.INTERVENTION is
'Intervention flag if intervention run';

comment on column DISPATCHLOAD_BNC.CONNECTIONPOINTID is
'Connection point identifier for DUID';

comment on column DISPATCHLOAD_BNC.DISPATCHMODE is
'Dispatch mode for fast start plant (0 to 4).';

comment on column DISPATCHLOAD_BNC.TOTALCLEARED is
'Target MW for end of period';

comment on column DISPATCHLOAD_BNC.RAISEREG is
'Raise Regulation reserve target';

comment on column DISPATCHLOAD_BNC.RAISE5MIN is
'Raise 5 min reserve target';

comment on column DISPATCHLOAD_BNC.RAISE60SEC is
'Raise 60 sec reserve target';

comment on column DISPATCHLOAD_BNC.RAISE6SEC is
'Raise 6 sec reserve target';

comment on column DISPATCHLOAD_BNC.LOWERREG is
'Lower Regulation reserve target';

comment on column DISPATCHLOAD_BNC.LOWER5MIN is
'Lower 5 min reserve target';

comment on column DISPATCHLOAD_BNC.LOWER60SEC is
'Lower 60 sec reserve target';

comment on column DISPATCHLOAD_BNC.LOWER6SEC is
'Lower 6 sec reserve target';

comment on column DISPATCHLOAD_BNC.RAISEREGFLAGS is
'Raise Reg status flag';

comment on column DISPATCHLOAD_BNC.RAISE5MINFLAGS is
'Raise 5min status flag';

comment on column DISPATCHLOAD_BNC.RAISE60SECFLAGS is
'Raise 60sec status flag';

comment on column DISPATCHLOAD_BNC.RAISE6SECFLAGS is
'Raise 6sec status flag';

comment on column DISPATCHLOAD_BNC.LOWERREGFLAGS is
'Lower Reg status flag';

comment on column DISPATCHLOAD_BNC.LOWER5MINFLAGS is
'Lower 5min status flag';

comment on column DISPATCHLOAD_BNC.LOWER60SECFLAGS is
'Lower 60sec status flag';

comment on column DISPATCHLOAD_BNC.LOWER6SECFLAGS is
'Lower 6sec status flag';

comment on column DISPATCHLOAD_BNC.LASTCHANGED is
'Last date and time record changed';

alter table DISPATCHLOAD_BNC
   add constraint PK_DISPATCHLOAD_BNC primary key (SETTLEMENTDATE, RUNNO, DUID, INTERVENTION)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: DISPATCHLOAD_BNC_LCX                                  */
/*==============================================================*/
create index DISPATCHLOAD_BNC_LCX on DISPATCHLOAD_BNC (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Index: DISPATCHLOAD_BNC_NDX2                                 */
/*==============================================================*/
create index DISPATCHLOAD_BNC_NDX2 on DISPATCHLOAD_BNC (
   DUID ASC,
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: DISPATCHOFFERTRK                                      */
/*==============================================================*/
create table DISPATCHOFFERTRK (
   SETTLEMENTDATE       DATE                  not null,
   DUID                 VARCHAR2(10)          not null,
   BIDTYPE              VARCHAR2(10)          not null,
   BIDSETTLEMENTDATE    DATE,
   BIDOFFERDATE         TIMESTAMP(3),
   LASTCHANGED          DATE
)
   tablespace PSSTAB03;

comment on table DISPATCHOFFERTRK is
'DISPATCHOFFERTRK is the energy and ancillary service bid tracking table for the Dispatch process. The table identifies which bids from BIDDAYOFFER and BIDOFFERPERIOD were applied for a given unit and bid type for each dispatch interval.';

comment on column DISPATCHOFFERTRK.SETTLEMENTDATE is
'Date and time of the dispatch interval (e.g. five minute dispatch interval ending 28/09/2000 16:35)';

comment on column DISPATCHOFFERTRK.DUID is
'Dispatchable unit identifier';

comment on column DISPATCHOFFERTRK.BIDTYPE is
'Bid type Identifier - the ancillary service to which the bid applies';

comment on column DISPATCHOFFERTRK.BIDSETTLEMENTDATE is
'Settlement date of bid applied';

comment on column DISPATCHOFFERTRK.BIDOFFERDATE is
'Time this bid was processed and loaded';

comment on column DISPATCHOFFERTRK.LASTCHANGED is
'Last date and time record changed';

alter table DISPATCHOFFERTRK
   add constraint DISPATCHOFFERTRK_PK primary key (SETTLEMENTDATE, DUID, BIDTYPE)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Index: DISPATCHOFFERTRK_LCHD_IDX                             */
/*==============================================================*/
create index DISPATCHOFFERTRK_LCHD_IDX on DISPATCHOFFERTRK (
   LASTCHANGED ASC
)
tablespace PSSIND03;

/*==============================================================*/
/* Index: DISPATCHOFFERTRK_NDX2                                 */
/*==============================================================*/
create index DISPATCHOFFERTRK_NDX2 on DISPATCHOFFERTRK (
   DUID ASC,
   LASTCHANGED ASC
)
tablespace PSSIND03;

/*==============================================================*/
/* Table: DISPATCHPRICE                                         */
/*==============================================================*/
create table DISPATCHPRICE (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0)           not null,
   REGIONID             VARCHAR2(10)          not null,
   DISPATCHINTERVAL     VARCHAR2(22)          not null,
   INTERVENTION         NUMBER(2,0)           not null,
   RRP                  NUMBER(15,5),
   EEP                  NUMBER(15,5),
   ROP                  NUMBER(15,5),
   APCFLAG              NUMBER(3,0),
   MARKETSUSPENDEDFLAG  NUMBER(3,0),
   LASTCHANGED          DATE,
   RAISE6SECRRP         NUMBER(15,5),
   RAISE6SECROP         NUMBER(15,5),
   RAISE6SECAPCFLAG     NUMBER(3,0),
   RAISE60SECRRP        NUMBER(15,5),
   RAISE60SECROP        NUMBER(15,5),
   RAISE60SECAPCFLAG    NUMBER(3,0),
   RAISE5MINRRP         NUMBER(15,5),
   RAISE5MINROP         NUMBER(15,5),
   RAISE5MINAPCFLAG     NUMBER(3,0),
   RAISEREGRRP          NUMBER(15,5),
   RAISEREGROP          NUMBER(15,5),
   RAISEREGAPCFLAG      NUMBER(3,0),
   LOWER6SECRRP         NUMBER(15,5),
   LOWER6SECROP         NUMBER(15,5),
   LOWER6SECAPCFLAG     NUMBER(3,0),
   LOWER60SECRRP        NUMBER(15,5),
   LOWER60SECROP        NUMBER(15,5),
   LOWER60SECAPCFLAG    NUMBER(3,0),
   LOWER5MINRRP         NUMBER(15,5),
   LOWER5MINROP         NUMBER(15,5),
   LOWER5MINAPCFLAG     NUMBER(3,0),
   LOWERREGRRP          NUMBER(15,5),
   LOWERREGROP          NUMBER(15,5),
   LOWERREGAPCFLAG      NUMBER(3,0),
   PRICE_STATUS         VARCHAR2(20),
   PRE_AP_ENERGY_PRICE  NUMBER(15,5),
   PRE_AP_RAISE6_PRICE  NUMBER(15,5),
   PRE_AP_RAISE60_PRICE NUMBER(15,5),
   PRE_AP_RAISE5MIN_PRICE NUMBER(15,5),
   PRE_AP_RAISEREG_PRICE NUMBER(15,5),
   PRE_AP_LOWER6_PRICE  NUMBER(15,5),
   PRE_AP_LOWER60_PRICE NUMBER(15,5),
   PRE_AP_LOWER5MIN_PRICE NUMBER(15,5),
   PRE_AP_LOWERREG_PRICE NUMBER(15,5),
   CUMUL_PRE_AP_ENERGY_PRICE NUMBER(15,5),
   CUMUL_PRE_AP_RAISE6_PRICE NUMBER(15,5),
   CUMUL_PRE_AP_RAISE60_PRICE NUMBER(15,5),
   CUMUL_PRE_AP_RAISE5MIN_PRICE NUMBER(15,5),
   CUMUL_PRE_AP_RAISEREG_PRICE NUMBER(15,5),
   CUMUL_PRE_AP_LOWER6_PRICE NUMBER(15,5),
   CUMUL_PRE_AP_LOWER60_PRICE NUMBER(15,5),
   CUMUL_PRE_AP_LOWER5MIN_PRICE NUMBER(15,5),
   CUMUL_PRE_AP_LOWERREG_PRICE NUMBER(15,5),
   OCD_STATUS           VARCHAR2(14),
   MII_STATUS           VARCHAR2(21),
   RAISE1SECRRP         NUMBER(15,5),
   RAISE1SECROP         NUMBER(15,5),
   RAISE1SECAPCFLAG     NUMBER(3,0),
   LOWER1SECRRP         NUMBER(15,5),
   LOWER1SECROP         NUMBER(15,5),
   LOWER1SECAPCFLAG     NUMBER(3,0),
   PRE_AP_RAISE1_PRICE  NUMBER(15,5),
   PRE_AP_LOWER1_PRICE  NUMBER(15,5),
   CUMUL_PRE_AP_RAISE1_PRICE NUMBER(15,5),
   CUMUL_PRE_AP_LOWER1_PRICE NUMBER(15,5)
)
   tablespace PSSTAB03;

comment on table DISPATCHPRICE is
'DISPATCHPRICE records 5 minute dispatch prices for energy and FCAS, including whether an intervention has occurred, or price override (e.g. for Administered Price Cap). DISPATCHPRICE updates when price adjustments occur, in which case the new price is written to the RRP field, and the old price to the ROP field as an audit trail.';

comment on column DISPATCHPRICE.SETTLEMENTDATE is
'Market date and time starting at 04:05';

comment on column DISPATCHPRICE.RUNNO is
'Dispatch run no; always 1';

comment on column DISPATCHPRICE.REGIONID is
'Region Identifier';

comment on column DISPATCHPRICE.DISPATCHINTERVAL is
'Dispatch interval identifier 001 to 288 in format YYYYMMDDPPP';

comment on column DISPATCHPRICE.INTERVENTION is
'Manual intervention flag';

comment on column DISPATCHPRICE.RRP is
'Regional Reference Price for this dispatch period. RRP is the price used to settle the market';

comment on column DISPATCHPRICE.EEP is
'Excess energy price - no longer used';

comment on column DISPATCHPRICE.ROP is
'Regional Override Price, being the original price prior to any price scaling, price capping or VoLL override being applied. The APC flag allows the determination of whether capping, scaling or override occurred';

comment on column DISPATCHPRICE.APCFLAG is
'APC Active flag (see note)';

comment on column DISPATCHPRICE.MARKETSUSPENDEDFLAG is
'Market suspended flag';

comment on column DISPATCHPRICE.LASTCHANGED is
'Last date and time record changed';

comment on column DISPATCHPRICE.PRICE_STATUS is
'Status of regional prices for this dispatch interval "NOT FIRM" or "FIRM"';

comment on column DISPATCHPRICE.PRE_AP_ENERGY_PRICE is
'Price before ap capping or scaling - for rolling sum price monitoring';

comment on column DISPATCHPRICE.PRE_AP_RAISE6_PRICE is
'Price before ap capping or scaling - for rolling sum price monitoring';

comment on column DISPATCHPRICE.PRE_AP_RAISE60_PRICE is
'Price before ap capping or scaling - for rolling sum price monitoring';

comment on column DISPATCHPRICE.PRE_AP_RAISE5MIN_PRICE is
'Price before ap capping or scaling - for rolling sum price monitoring';

comment on column DISPATCHPRICE.PRE_AP_RAISEREG_PRICE is
'Price before ap capping or scaling - for rolling sum price monitoring';

comment on column DISPATCHPRICE.PRE_AP_LOWER6_PRICE is
'Price before ap capping or scaling - for rolling sum price monitoring';

comment on column DISPATCHPRICE.PRE_AP_LOWER60_PRICE is
'Price before ap capping or scaling - for rolling sum price monitoring';

comment on column DISPATCHPRICE.PRE_AP_LOWER5MIN_PRICE is
'Price before ap capping or scaling - for rolling sum price monitoring';

comment on column DISPATCHPRICE.PRE_AP_LOWERREG_PRICE is
'Price before ap capping or scaling - for rolling sum price monitoring';

comment on column DISPATCHPRICE.CUMUL_PRE_AP_ENERGY_PRICE is
'Cumulative price that triggers administered pricing event if above the threshold';

comment on column DISPATCHPRICE.CUMUL_PRE_AP_RAISE6_PRICE is
'Cumulative price that triggers administered pricing event if above the threshold';

comment on column DISPATCHPRICE.CUMUL_PRE_AP_RAISE60_PRICE is
'Cumulative price that triggers administered pricing event if above the threshold';

comment on column DISPATCHPRICE.CUMUL_PRE_AP_RAISE5MIN_PRICE is
'Cumulative price that triggers administered pricing event if above the threshold';

comment on column DISPATCHPRICE.CUMUL_PRE_AP_RAISEREG_PRICE is
'Cumulative price that triggers administered pricing event if above the threshold';

comment on column DISPATCHPRICE.CUMUL_PRE_AP_LOWER6_PRICE is
'Cumulative price that triggers administered pricing event if above the threshold';

comment on column DISPATCHPRICE.CUMUL_PRE_AP_LOWER60_PRICE is
'Cumulative price that triggers administered pricing event if above the threshold';

comment on column DISPATCHPRICE.CUMUL_PRE_AP_LOWER5MIN_PRICE is
'Cumulative price that triggers administered pricing event if above the threshold';

comment on column DISPATCHPRICE.CUMUL_PRE_AP_LOWERREG_PRICE is
'Cumulative price that triggers administered pricing event if above the threshold';

comment on column DISPATCHPRICE.OCD_STATUS is
'Communicates the current OCD status for this dispatch interval.  Values of: ''NOT_OCD'', ''OCD_UNRESOLVED'', ''OCD_RESOLVED''.';

comment on column DISPATCHPRICE.MII_STATUS is
'Communicates the current MII status for this dispatch interval.  Values of: ''NOT_MII'', ''MII_SUBJECT_TO_REVIEW'', ''MII_PRICE_REJECTED'', ''MII_PRICE_ACCEPTED''.';

comment on column DISPATCHPRICE.RAISE1SECRRP is
'Regional Raise 1Sec Price - R1Price attribute after capping/flooring';

comment on column DISPATCHPRICE.RAISE1SECROP is
'Raise1Sec Regional Original Price - uncapped/unfloored and unscaled';

comment on column DISPATCHPRICE.RAISE1SECAPCFLAG is
'BitFlag field for Price adjustments - "1" = Voll_Override; "4" = Floor_VoLL; "8" = Manual_Override; "16" = Price_Scaled';

comment on column DISPATCHPRICE.LOWER1SECRRP is
'Regional Lower 1Sec Price - RegionSolution element L1Price attribute';

comment on column DISPATCHPRICE.LOWER1SECROP is
'Lower1Sec Regional Original Price - uncapped/unfloored and unscaled';

comment on column DISPATCHPRICE.LOWER1SECAPCFLAG is
'BitFlag field for Price adjustments - "1" = Voll_Override; "4" = Floor_VoLL; "8" = Manual_Override; "16" = Price_Scaled';

comment on column DISPATCHPRICE.PRE_AP_RAISE1_PRICE is
'Price before AP capping or scaling - for Rolling Sum Price monitoring';

comment on column DISPATCHPRICE.PRE_AP_LOWER1_PRICE is
'Price before AP capping or scaling - for Rolling Sum Price monitoring';

comment on column DISPATCHPRICE.CUMUL_PRE_AP_RAISE1_PRICE is
'Cumulative price that triggers administered pricing event if above the threshold';

comment on column DISPATCHPRICE.CUMUL_PRE_AP_LOWER1_PRICE is
'Cumulative price that triggers administered pricing event if above the threshold';

alter table DISPATCHPRICE
   add constraint PK_DISPATCHPRICE primary key (SETTLEMENTDATE, RUNNO, REGIONID, DISPATCHINTERVAL, INTERVENTION)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: DISPATCHPRICE_LCX                                     */
/*==============================================================*/
create index DISPATCHPRICE_LCX on DISPATCHPRICE (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: DISPATCHREGIONSUM                                     */
/*==============================================================*/
create table DISPATCHREGIONSUM (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0)           not null,
   REGIONID             VARCHAR2(10)          not null,
   DISPATCHINTERVAL     NUMBER(22,0)          not null,
   INTERVENTION         NUMBER(2,0)           not null,
   TOTALDEMAND          NUMBER(15,5),
   AVAILABLEGENERATION  NUMBER(15,5),
   AVAILABLELOAD        NUMBER(15,5),
   DEMANDFORECAST       NUMBER(15,5),
   DISPATCHABLEGENERATION NUMBER(15,5),
   DISPATCHABLELOAD     NUMBER(15,5),
   NETINTERCHANGE       NUMBER(15,5),
   EXCESSGENERATION     NUMBER(15,5),
   LOWER5MINDISPATCH    NUMBER(15,5),
   LOWER5MINIMPORT      NUMBER(15,5),
   LOWER5MINLOCALDISPATCH NUMBER(15,5),
   LOWER5MINLOCALPRICE  NUMBER(15,5),
   LOWER5MINLOCALREQ    NUMBER(15,5),
   LOWER5MINPRICE       NUMBER(15,5),
   LOWER5MINREQ         NUMBER(15,5),
   LOWER5MINSUPPLYPRICE NUMBER(15,5),
   LOWER60SECDISPATCH   NUMBER(15,5),
   LOWER60SECIMPORT     NUMBER(15,5),
   LOWER60SECLOCALDISPATCH NUMBER(15,5),
   LOWER60SECLOCALPRICE NUMBER(15,5),
   LOWER60SECLOCALREQ   NUMBER(15,5),
   LOWER60SECPRICE      NUMBER(15,5),
   LOWER60SECREQ        NUMBER(15,5),
   LOWER60SECSUPPLYPRICE NUMBER(15,5),
   LOWER6SECDISPATCH    NUMBER(15,5),
   LOWER6SECIMPORT      NUMBER(15,5),
   LOWER6SECLOCALDISPATCH NUMBER(15,5),
   LOWER6SECLOCALPRICE  NUMBER(15,5),
   LOWER6SECLOCALREQ    NUMBER(15,5),
   LOWER6SECPRICE       NUMBER(15,5),
   LOWER6SECREQ         NUMBER(15,5),
   LOWER6SECSUPPLYPRICE NUMBER(15,5),
   RAISE5MINDISPATCH    NUMBER(15,5),
   RAISE5MINIMPORT      NUMBER(15,5),
   RAISE5MINLOCALDISPATCH NUMBER(15,5),
   RAISE5MINLOCALPRICE  NUMBER(15,5),
   RAISE5MINLOCALREQ    NUMBER(15,5),
   RAISE5MINPRICE       NUMBER(15,5),
   RAISE5MINREQ         NUMBER(15,5),
   RAISE5MINSUPPLYPRICE NUMBER(15,5),
   RAISE60SECDISPATCH   NUMBER(15,5),
   RAISE60SECIMPORT     NUMBER(15,5),
   RAISE60SECLOCALDISPATCH NUMBER(15,5),
   RAISE60SECLOCALPRICE NUMBER(15,5),
   RAISE60SECLOCALREQ   NUMBER(15,5),
   RAISE60SECPRICE      NUMBER(15,5),
   RAISE60SECREQ        NUMBER(15,5),
   RAISE60SECSUPPLYPRICE NUMBER(15,5),
   RAISE6SECDISPATCH    NUMBER(15,5),
   RAISE6SECIMPORT      NUMBER(15,5),
   RAISE6SECLOCALDISPATCH NUMBER(15,5),
   RAISE6SECLOCALPRICE  NUMBER(15,5),
   RAISE6SECLOCALREQ    NUMBER(15,5),
   RAISE6SECPRICE       NUMBER(15,5),
   RAISE6SECREQ         NUMBER(15,5),
   RAISE6SECSUPPLYPRICE NUMBER(15,5),
   AGGEGATEDISPATCHERROR NUMBER(15,5),
   AGGREGATEDISPATCHERROR NUMBER(15,5),
   LASTCHANGED          DATE,
   INITIALSUPPLY        NUMBER(15,5),
   CLEAREDSUPPLY        NUMBER(15,5),
   LOWERREGIMPORT       NUMBER(15,5),
   LOWERREGLOCALDISPATCH NUMBER(15,5),
   LOWERREGLOCALREQ     NUMBER(15,5),
   LOWERREGREQ          NUMBER(15,5),
   RAISEREGIMPORT       NUMBER(15,5),
   RAISEREGLOCALDISPATCH NUMBER(15,5),
   RAISEREGLOCALREQ     NUMBER(15,5),
   RAISEREGREQ          NUMBER(15,5),
   RAISE5MINLOCALVIOLATION NUMBER(15,5),
   RAISEREGLOCALVIOLATION NUMBER(15,5),
   RAISE60SECLOCALVIOLATION NUMBER(15,5),
   RAISE6SECLOCALVIOLATION NUMBER(15,5),
   LOWER5MINLOCALVIOLATION NUMBER(15,5),
   LOWERREGLOCALVIOLATION NUMBER(15,5),
   LOWER60SECLOCALVIOLATION NUMBER(15,5),
   LOWER6SECLOCALVIOLATION NUMBER(15,5),
   RAISE5MINVIOLATION   NUMBER(15,5),
   RAISEREGVIOLATION    NUMBER(15,5),
   RAISE60SECVIOLATION  NUMBER(15,5),
   RAISE6SECVIOLATION   NUMBER(15,5),
   LOWER5MINVIOLATION   NUMBER(15,5),
   LOWERREGVIOLATION    NUMBER(15,5),
   LOWER60SECVIOLATION  NUMBER(15,5),
   LOWER6SECVIOLATION   NUMBER(15,5),
   RAISE6SECACTUALAVAILABILITY NUMBER(16,6),
   RAISE60SECACTUALAVAILABILITY NUMBER(16,6),
   RAISE5MINACTUALAVAILABILITY NUMBER(16,6),
   RAISEREGACTUALAVAILABILITY NUMBER(16,6),
   LOWER6SECACTUALAVAILABILITY NUMBER(16,6),
   LOWER60SECACTUALAVAILABILITY NUMBER(16,6),
   LOWER5MINACTUALAVAILABILITY NUMBER(16,6),
   LOWERREGACTUALAVAILABILITY NUMBER(16,6),
   LORSURPLUS           NUMBER(16,6),
   LRCSURPLUS           NUMBER(16,6),
   TOTALINTERMITTENTGENERATION NUMBER(15,5),
   DEMAND_AND_NONSCHEDGEN NUMBER(15,5),
   UIGF                 NUMBER(15,5),
   SEMISCHEDULE_CLEAREDMW NUMBER(15,5),
   SEMISCHEDULE_COMPLIANCEMW NUMBER(15,5),
   SS_SOLAR_UIGF        Number(15,5),
   SS_WIND_UIGF         Number (15,5),
   SS_SOLAR_CLEAREDMW   Number(15,5),
   SS_WIND_CLEAREDMW    Number(15,5),
   SS_SOLAR_COMPLIANCEMW Number(15,5),
   SS_WIND_COMPLIANCEMW Number(15,5),
   WDR_INITIALMW        NUMBER(15,5),
   WDR_AVAILABLE        NUMBER(15,5),
   WDR_DISPATCHED       NUMBER(15,5),
   SS_SOLAR_AVAILABILITY NUMBER(15,5),
   SS_WIND_AVAILABILITY NUMBER(15,5),
   RAISE1SECLOCALDISPATCH NUMBER(15,5),
   LOWER1SECLOCALDISPATCH NUMBER(15,5),
   RAISE1SECACTUALAVAILABILITY NUMBER(16,6),
   LOWER1SECACTUALAVAILABILITY NUMBER(16,6),
   BDU_ENERGY_STORAGE   NUMBER(15,5),
   BDU_MIN_AVAIL        NUMBER(15,5),
   BDU_MAX_AVAIL        NUMBER(15,5),
   BDU_CLEAREDMW_GEN    NUMBER(15,5),
   BDU_CLEAREDMW_LOAD   NUMBER(15,5),
   BDU_INITIAL_ENERGY_STORAGE NUMBER(15,5),
   DECGEN_INITIAL_ENERGY_STORAGE NUMBER(15,5)
)
   tablespace PSSTAB03;

comment on table DISPATCHREGIONSUM is
'DISPATCHREGIONSUM sets out the 5-minute solution for each dispatch run for each region, including the Frequency Control Ancillary Services (FCAS) services provided. Additional fields are for the Raise Regulation and Lower Regulation Ancillary Services plus improvements to demand calculations.';

comment on column DISPATCHREGIONSUM.SETTLEMENTDATE is
'Market date and time starting at 04:05';

comment on column DISPATCHREGIONSUM.RUNNO is
'Dispatch run no; always 1';

comment on column DISPATCHREGIONSUM.REGIONID is
'Region Identifier';

comment on column DISPATCHREGIONSUM.DISPATCHINTERVAL is
'Dispatch period identifier, from 001 to 288 in format YYYYMMDDPPP.';

comment on column DISPATCHREGIONSUM.INTERVENTION is
'Manual Intervention flag';

comment on column DISPATCHREGIONSUM.TOTALDEMAND is
'Demand (less loads)';

comment on column DISPATCHREGIONSUM.AVAILABLEGENERATION is
'Aggregate generation bid available in region';

comment on column DISPATCHREGIONSUM.AVAILABLELOAD is
'Aggregate load bid available in region';

comment on column DISPATCHREGIONSUM.DEMANDFORECAST is
'5 minute forecast adjust';

comment on column DISPATCHREGIONSUM.DISPATCHABLEGENERATION is
'Dispatched Generation';

comment on column DISPATCHREGIONSUM.DISPATCHABLELOAD is
'Dispatched Load (add to total demand to get inherent region demand).';

comment on column DISPATCHREGIONSUM.NETINTERCHANGE is
'Net interconnector flow from the regional reference node';

comment on column DISPATCHREGIONSUM.EXCESSGENERATION is
'MW quantity of excess';

comment on column DISPATCHREGIONSUM.LOWER5MINDISPATCH is
'Not used since Dec 2003. Lower 5 min MW dispatch';

comment on column DISPATCHREGIONSUM.LOWER5MINIMPORT is
'Not used since Dec 2003. Lower 5 min MW imported';

comment on column DISPATCHREGIONSUM.LOWER5MINLOCALDISPATCH is
'Lower 5 min local dispatch';

comment on column DISPATCHREGIONSUM.LOWER5MINLOCALPRICE is
'Not used since Dec 2003. Local price of lower 5 min';

comment on column DISPATCHREGIONSUM.LOWER5MINLOCALREQ is
'Not used since Dec 2003. Lower 5 min local requirement';

comment on column DISPATCHREGIONSUM.LOWER5MINPRICE is
'Not used since Dec 2003. Regional price of lower 5 min';

comment on column DISPATCHREGIONSUM.LOWER5MINREQ is
'Not used since Dec 2003. Lower 5 min total requirement';

comment on column DISPATCHREGIONSUM.LOWER5MINSUPPLYPRICE is
'Not used since Dec 2003. Supply price of lower 5 min';

comment on column DISPATCHREGIONSUM.LOWER60SECDISPATCH is
'Not used since Dec 2003. Lower 60 sec MW dispatch';

comment on column DISPATCHREGIONSUM.LOWER60SECIMPORT is
'Not used since Dec 2003. Lower 60 sec MW imported';

comment on column DISPATCHREGIONSUM.LOWER60SECLOCALDISPATCH is
'Lower 60 sec local dispatch';

comment on column DISPATCHREGIONSUM.LOWER60SECLOCALPRICE is
'Not used since Dec 2003. Local price of lower 60 sec';

comment on column DISPATCHREGIONSUM.LOWER60SECLOCALREQ is
'Not used since Dec 2003. Lower 60 sec local requirement';

comment on column DISPATCHREGIONSUM.LOWER60SECPRICE is
'Not used since Dec 2003. Regional price of lower 60 sec';

comment on column DISPATCHREGIONSUM.LOWER60SECREQ is
'Not used since Dec 2003. Lower 60 sec total requirement';

comment on column DISPATCHREGIONSUM.LOWER60SECSUPPLYPRICE is
'Not used since Dec 2003. Supply price of lower 60 sec';

comment on column DISPATCHREGIONSUM.LOWER6SECDISPATCH is
'Not used since Dec 2003. Lower 6 sec MW dispatch';

comment on column DISPATCHREGIONSUM.LOWER6SECIMPORT is
'Not used since Dec 2003. Lower 6 sec MW imported';

comment on column DISPATCHREGIONSUM.LOWER6SECLOCALDISPATCH is
'Lower 6 sec local dispatch';

comment on column DISPATCHREGIONSUM.LOWER6SECLOCALPRICE is
'Not used since Dec 2003. Local price of lower 6 sec';

comment on column DISPATCHREGIONSUM.LOWER6SECLOCALREQ is
'Not used since Dec 2003. Lower 6 sec local requirement';

comment on column DISPATCHREGIONSUM.LOWER6SECPRICE is
'Not used since Dec 2003. Regional price of lower 6 sec';

comment on column DISPATCHREGIONSUM.LOWER6SECREQ is
'Not used since Dec 2003. Lower 6 sec total requirement';

comment on column DISPATCHREGIONSUM.LOWER6SECSUPPLYPRICE is
'Not used since Dec 2003. Supply price of lower 6 sec';

comment on column DISPATCHREGIONSUM.RAISE5MINDISPATCH is
'Not used since Dec 2003. Raise 5 min MW dispatch';

comment on column DISPATCHREGIONSUM.RAISE5MINIMPORT is
'Not used since Dec 2003. Raise 5 min MW imported';

comment on column DISPATCHREGIONSUM.RAISE5MINLOCALDISPATCH is
'Raise 5 min local dispatch';

comment on column DISPATCHREGIONSUM.RAISE5MINLOCALPRICE is
'Not used since Dec 2003. Raise price of lower 5 min';

comment on column DISPATCHREGIONSUM.RAISE5MINLOCALREQ is
'Not used since Dec 2003. Raise 5 min local requirement';

comment on column DISPATCHREGIONSUM.RAISE5MINPRICE is
'Not used since Dec 2003. Regional price of raise 5 min';

comment on column DISPATCHREGIONSUM.RAISE5MINREQ is
'Not used since Dec 2003. Raise 5 min total requirement';

comment on column DISPATCHREGIONSUM.RAISE5MINSUPPLYPRICE is
'Not used since Dec 2003. Supply price of raise 5 min';

comment on column DISPATCHREGIONSUM.RAISE60SECDISPATCH is
'Not used since Dec 2003. Raise 60 sec MW dispatch';

comment on column DISPATCHREGIONSUM.RAISE60SECIMPORT is
'Not used since Dec 2003. Raise 60 sec MW imported';

comment on column DISPATCHREGIONSUM.RAISE60SECLOCALDISPATCH is
'Raise 60 sec local dispatch';

comment on column DISPATCHREGIONSUM.RAISE60SECLOCALPRICE is
'Not used since Dec 2003. Local price of raise 60 sec';

comment on column DISPATCHREGIONSUM.RAISE60SECLOCALREQ is
'Not used since Dec 2003. Raise 60 sec local requirement';

comment on column DISPATCHREGIONSUM.RAISE60SECPRICE is
'Not used since Dec 2003. Regional price of raise 60 sec';

comment on column DISPATCHREGIONSUM.RAISE60SECREQ is
'Not used since Dec 2003. Raise 60 sec total requirement';

comment on column DISPATCHREGIONSUM.RAISE60SECSUPPLYPRICE is
'Not used since Dec 2003. Supply price of raise 60 sec';

comment on column DISPATCHREGIONSUM.RAISE6SECDISPATCH is
'Not used since Dec 2003. Raise 6 sec MW dispatch';

comment on column DISPATCHREGIONSUM.RAISE6SECIMPORT is
'Not used since Dec 2003. Raise 6 sec MW imported';

comment on column DISPATCHREGIONSUM.RAISE6SECLOCALDISPATCH is
'Raise 6 sec local dispatch';

comment on column DISPATCHREGIONSUM.RAISE6SECLOCALPRICE is
'Not used since Dec 2003. Local price of raise 6 sec';

comment on column DISPATCHREGIONSUM.RAISE6SECLOCALREQ is
'Not used since Dec 2003. Raise 6 sec local requirement';

comment on column DISPATCHREGIONSUM.RAISE6SECPRICE is
'Not used since Dec 2003. Regional price of raise 6 sec';

comment on column DISPATCHREGIONSUM.RAISE6SECREQ is
'Not used since Dec 2003. Raise 6 sec total requirement';

comment on column DISPATCHREGIONSUM.RAISE6SECSUPPLYPRICE is
'Not used since Dec 2003. Supply price of raise 6 sec';

comment on column DISPATCHREGIONSUM.AGGEGATEDISPATCHERROR is
'Calculated dispatch error';

comment on column DISPATCHREGIONSUM.AGGREGATEDISPATCHERROR is
'Calculated dispatch error';

comment on column DISPATCHREGIONSUM.LASTCHANGED is
'Last date and time record changed';

comment on column DISPATCHREGIONSUM.INITIALSUPPLY is
'Sum of initial generation and import for region';

comment on column DISPATCHREGIONSUM.CLEAREDSUPPLY is
'Sum of cleared generation and import for region';

comment on column DISPATCHREGIONSUM.LOWERREGIMPORT is
'Not used since Dec 2003. Lower Regulation MW imported';

comment on column DISPATCHREGIONSUM.LOWERREGLOCALDISPATCH is
'Lower Regulation local dispatch';

comment on column DISPATCHREGIONSUM.LOWERREGLOCALREQ is
'Not used since Dec 2003. Lower Regulation local requirement';

comment on column DISPATCHREGIONSUM.LOWERREGREQ is
'Not used since Dec 2003. Lower Regulation total requirement';

comment on column DISPATCHREGIONSUM.RAISEREGIMPORT is
'Not used since Dec 2003. Raise Regulation MW imported';

comment on column DISPATCHREGIONSUM.RAISEREGLOCALDISPATCH is
'Raise Regulation local dispatch';

comment on column DISPATCHREGIONSUM.RAISEREGLOCALREQ is
'Not used since Dec 2003. Raise Regulation local requirement';

comment on column DISPATCHREGIONSUM.RAISEREGREQ is
'Not used since Dec 2003. Raise Regulation total requirement';

comment on column DISPATCHREGIONSUM.RAISE5MINLOCALVIOLATION is
'Not used since Dec 2003. Violation (MW) of Raise 5 min local requirement';

comment on column DISPATCHREGIONSUM.RAISEREGLOCALVIOLATION is
'Not used since Dec 2003. Violation (MW) of Raise Reg local requirement';

comment on column DISPATCHREGIONSUM.RAISE60SECLOCALVIOLATION is
'Not used since Dec 2003. Violation (MW) of Raise 60 sec local requirement';

comment on column DISPATCHREGIONSUM.RAISE6SECLOCALVIOLATION is
'Not used since Dec 2003. Violation (MW) of Raise 6 sec local requirement';

comment on column DISPATCHREGIONSUM.LOWER5MINLOCALVIOLATION is
'Not used since Dec 2003. Violation (MW) of Lower 5 min local requirement';

comment on column DISPATCHREGIONSUM.LOWERREGLOCALVIOLATION is
'Not used since Dec 2003. Violation (MW) of Lower Reg local requirement';

comment on column DISPATCHREGIONSUM.LOWER60SECLOCALVIOLATION is
'Not used since Dec 2003. Violation (MW) of Lower 60 sec local requirement';

comment on column DISPATCHREGIONSUM.LOWER6SECLOCALVIOLATION is
'Not used since Dec 2003. Violation (MW) of Lower 6 sec local requirement';

comment on column DISPATCHREGIONSUM.RAISE5MINVIOLATION is
'Not used since Dec 2003. Violation (MW) of Raise 5 min requirement';

comment on column DISPATCHREGIONSUM.RAISEREGVIOLATION is
'Not used since Dec 2003. Violation (MW) of Raise Reg requirement';

comment on column DISPATCHREGIONSUM.RAISE60SECVIOLATION is
'Not used since Dec 2003. Violation (MW) of Raise 60 seconds requirement';

comment on column DISPATCHREGIONSUM.RAISE6SECVIOLATION is
'Not used since Dec 2003. Violation (MW) of Raise 6 seconds requirement';

comment on column DISPATCHREGIONSUM.LOWER5MINVIOLATION is
'Not used since Dec 2003. Violation (MW) of Lower 5 min requirement';

comment on column DISPATCHREGIONSUM.LOWERREGVIOLATION is
'Not used since Dec 2003. Violation (MW) of Lower Reg requirement';

comment on column DISPATCHREGIONSUM.LOWER60SECVIOLATION is
'Not used since Dec 2003. Violation (MW) of Lower 60 seconds requirement';

comment on column DISPATCHREGIONSUM.LOWER6SECVIOLATION is
'Not used since Dec 2003. Violation (MW) of Lower 6 seconds requirement';

comment on column DISPATCHREGIONSUM.RAISE6SECACTUALAVAILABILITY is
'trapezium adjusted raise 6sec availability';

comment on column DISPATCHREGIONSUM.RAISE60SECACTUALAVAILABILITY is
'trapezium adjusted raise 60sec availability';

comment on column DISPATCHREGIONSUM.RAISE5MINACTUALAVAILABILITY is
'trapezium adjusted raise 5min availability';

comment on column DISPATCHREGIONSUM.RAISEREGACTUALAVAILABILITY is
'trapezium adjusted raise reg availability';

comment on column DISPATCHREGIONSUM.LOWER6SECACTUALAVAILABILITY is
'trapezium adjusted lower 6sec availability';

comment on column DISPATCHREGIONSUM.LOWER60SECACTUALAVAILABILITY is
'trapezium adjusted lower 60sec availability';

comment on column DISPATCHREGIONSUM.LOWER5MINACTUALAVAILABILITY is
'trapezium adjusted lower 5min availability';

comment on column DISPATCHREGIONSUM.LOWERREGACTUALAVAILABILITY is
'trapezium adjusted lower reg availability';

comment on column DISPATCHREGIONSUM.LORSURPLUS is
'Not in use after 17 Feb 2006. Total short term generation capacity reserve used in assessing lack of reserve condition';

comment on column DISPATCHREGIONSUM.LRCSURPLUS is
'Not in use after 17 Feb 2006. Total short term generation capacity reserve above the stated low reserve condition requirement';

comment on column DISPATCHREGIONSUM.TOTALINTERMITTENTGENERATION is
'Allowance made for non-scheduled generation in the demand forecast (MW).';

comment on column DISPATCHREGIONSUM.DEMAND_AND_NONSCHEDGEN is
'Sum of Cleared Scheduled generation, imported generation (at the region boundary) and allowances made for non-scheduled generation (MW).';

comment on column DISPATCHREGIONSUM.UIGF is
'Regional aggregated Unconstrained Intermittent Generation Forecast of Semi-scheduled generation (MW).';

comment on column DISPATCHREGIONSUM.SEMISCHEDULE_CLEAREDMW is
'Regional aggregated Semi-Schedule generator Cleared MW';

comment on column DISPATCHREGIONSUM.SEMISCHEDULE_COMPLIANCEMW is
'Regional aggregated Semi-Schedule generator Cleared MW where Semi-Dispatch cap is enforced';

comment on column DISPATCHREGIONSUM.SS_SOLAR_UIGF is
'Regional aggregated Unconstrained Intermittent Generation Forecast of Semi-scheduled generation (MW) where the primary fuel source is solar';

comment on column DISPATCHREGIONSUM.SS_WIND_UIGF is
'Regional aggregated Unconstrained Intermittent Generation Forecast of Semi-scheduled generation (MW) where the primary fuel source is wind';

comment on column DISPATCHREGIONSUM.SS_SOLAR_CLEAREDMW is
'Regional aggregated Semi-Schedule generator Cleared MW where the primary fuel source is solar';

comment on column DISPATCHREGIONSUM.SS_WIND_CLEAREDMW is
'Regional aggregated Semi-Schedule generator Cleared MW where the primary fuel source is wind';

comment on column DISPATCHREGIONSUM.SS_SOLAR_COMPLIANCEMW is
'Regional aggregated Semi-Schedule generator Cleared MW where Semi-Dispatch cap is enforced and the primary fuel source is solar';

comment on column DISPATCHREGIONSUM.SS_WIND_COMPLIANCEMW is
'Regional aggregated Semi-Schedule generator Cleared MW where Semi-Dispatch cap is enforced and the primary fuel source is wind';

comment on column DISPATCHREGIONSUM.WDR_INITIALMW is
'Regional aggregated MW value at start of interval for Wholesale Demand Response (WDR) units';

comment on column DISPATCHREGIONSUM.WDR_AVAILABLE is
'Regional aggregated available MW for Wholesale Demand Response (WDR) units';

comment on column DISPATCHREGIONSUM.WDR_DISPATCHED is
'Regional aggregated dispatched MW for Wholesale Demand Response (WDR) units';

comment on column DISPATCHREGIONSUM.SS_SOLAR_AVAILABILITY is
'For Semi-Scheduled units. Aggregate Energy Availability from Solar units in that region';

comment on column DISPATCHREGIONSUM.SS_WIND_AVAILABILITY is
'For Semi-Scheduled units. Aggregate Energy Availability from Wind units in that region';

comment on column DISPATCHREGIONSUM.RAISE1SECLOCALDISPATCH is
'Total Raise1Sec Dispatched in Region - RegionSolution element R1Dispatch attribute';

comment on column DISPATCHREGIONSUM.LOWER1SECLOCALDISPATCH is
'Total Lower1Sec Dispatched in Region - RegionSolution element L1Dispatch attribute';

comment on column DISPATCHREGIONSUM.RAISE1SECACTUALAVAILABILITY is
'Trapezium adjusted Raise1Sec availability (summated from UnitSolution)';

comment on column DISPATCHREGIONSUM.LOWER1SECACTUALAVAILABILITY is
'Trapezium adjusted Lower1Sec availability (summated from UnitSolution)';

comment on column DISPATCHREGIONSUM.BDU_ENERGY_STORAGE is
'Regional aggregated energy storage where the DUID type is BDU (MWh)';

comment on column DISPATCHREGIONSUM.BDU_MIN_AVAIL is
'Total available load side BDU summated for region (MW)';

comment on column DISPATCHREGIONSUM.BDU_MAX_AVAIL is
'Total available generation side BDU summated for region (MW)';

comment on column DISPATCHREGIONSUM.BDU_CLEAREDMW_GEN is
'Regional aggregated cleared MW where the DUID type is BDU. Net of export (Generation)';

comment on column DISPATCHREGIONSUM.BDU_CLEAREDMW_LOAD is
'Regional aggregated cleared MW where the DUID type is BDU. Net of import (Load)';

comment on column DISPATCHREGIONSUM.BDU_INITIAL_ENERGY_STORAGE is
'Energy Storage for BDU at the start of the interval(MWh) - Region Aggregated';

comment on column DISPATCHREGIONSUM.DECGEN_INITIAL_ENERGY_STORAGE is
'Energy storage for Daily Energy Constrained Scheduled Generating Units at the start of the interval(MWh) - Region Aggregated';

alter table DISPATCHREGIONSUM
   add constraint PK_DISPATCHREGIONSUM primary key (SETTLEMENTDATE, RUNNO, REGIONID, DISPATCHINTERVAL, INTERVENTION)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: DISPATCHREGIONSUM_LCX                                 */
/*==============================================================*/
create index DISPATCHREGIONSUM_LCX on DISPATCHREGIONSUM (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: DISPATCHTRK                                           */
/*==============================================================*/
create table DISPATCHTRK (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0)           not null,
   REASON               VARCHAR2(64),
   SPDRUNNO             NUMBER(3,0),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table DISPATCHTRK is
'DISPATCHTRK is no longer used. DISPATCHTRK was  the cross-reference between each dispatch run and SPD case run. DISPATCHTRK may be available on the InfoServer but not replicated to participant databases as it contains data duplicated in other tables.';

comment on column DISPATCHTRK.SETTLEMENTDATE is
'Market date and time starting at 04:05';

comment on column DISPATCHTRK.RUNNO is
'Dispatch run no, normally 1.';

comment on column DISPATCHTRK.REASON is
'Reason code (if rerun)';

comment on column DISPATCHTRK.SPDRUNNO is
'Case identifier for LP Solver';

comment on column DISPATCHTRK.LASTCHANGED is
'Last date and time record changed';

alter table DISPATCHTRK
   add constraint DISPATCHTRK_PK primary key (SETTLEMENTDATE, RUNNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: DISPATCHTRK_LCX                                       */
/*==============================================================*/
create index DISPATCHTRK_LCX on DISPATCHTRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DISPATCH_CONSTRAINT_FCAS_OCD                          */
/*==============================================================*/
create table DISPATCH_CONSTRAINT_FCAS_OCD (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3)             not null,
   INTERVENTION         NUMBER(2)             not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(3)             not null,
   LASTCHANGED          DATE,
   RHS                  NUMBER(15,5),
   MARGINALVALUE        NUMBER(15,5),
   VIOLATIONDEGREE      NUMBER(15,5)
)
   tablespace PSSTAB01;

comment on table DISPATCH_CONSTRAINT_FCAS_OCD is
'FCAS constraint solution from OCD re-run.';

comment on column DISPATCH_CONSTRAINT_FCAS_OCD.SETTLEMENTDATE is
'Dispatch interval that the prices were loaded to';

comment on column DISPATCH_CONSTRAINT_FCAS_OCD.RUNNO is
'Dispatch run no; always 1';

comment on column DISPATCH_CONSTRAINT_FCAS_OCD.INTERVENTION is
'Intervention 0/1';

comment on column DISPATCH_CONSTRAINT_FCAS_OCD.CONSTRAINTID is
'ConstraintID/GenconID';

comment on column DISPATCH_CONSTRAINT_FCAS_OCD.VERSIONNO is
'VersionNo';

comment on column DISPATCH_CONSTRAINT_FCAS_OCD.LASTCHANGED is
'The datetime that the record was last changed';

comment on column DISPATCH_CONSTRAINT_FCAS_OCD.RHS is
'RHS from OCD re-run';

comment on column DISPATCH_CONSTRAINT_FCAS_OCD.MARGINALVALUE is
'marginalvalue from OCD re-run';

comment on column DISPATCH_CONSTRAINT_FCAS_OCD.VIOLATIONDEGREE is
'The violation degree of this constraint in the solution result';

alter table DISPATCH_CONSTRAINT_FCAS_OCD
   add constraint DISPATCH_CONSTRNT_FCAS_OCD_PK primary key (SETTLEMENTDATE, RUNNO, INTERVENTION, CONSTRAINTID, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: DISPATCH_CONSTRNT_FCASOCD_LCX                         */
/*==============================================================*/
create index DISPATCH_CONSTRNT_FCASOCD_LCX on DISPATCH_CONSTRAINT_FCAS_OCD (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DISPATCH_FCAS_REQ                                     */
/*==============================================================*/
create table DISPATCH_FCAS_REQ (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0)           not null,
   INTERVENTION         NUMBER(2,0)           not null,
   GENCONID             VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(10)          not null,
   BIDTYPE              VARCHAR2(10)          not null,
   GENCONEFFECTIVEDATE  DATE,
   GENCONVERSIONNO      NUMBER(3,0),
   MARGINALVALUE        NUMBER(16,6),
   LASTCHANGED          DATE,
   BASE_COST            NUMBER(18,8),
   ADJUSTED_COST        NUMBER(18,8),
   ESTIMATED_CMPF       NUMBER(18,8),
   ESTIMATED_CRMPF      NUMBER(18,8),
   RECOVERY_FACTOR_CMPF NUMBER(18,8),
   RECOVERY_FACTOR_CRMPF NUMBER(18,8)
)
   tablespace PSSTAB03;

comment on table DISPATCH_FCAS_REQ is
'DISPATCH_FCAS_REQ shows Dispatch Constraint tracking for Regional FCAS recovery.';

comment on column DISPATCH_FCAS_REQ.SETTLEMENTDATE is
'Settlement date and time of Dispatch Interval';

comment on column DISPATCH_FCAS_REQ.RUNNO is
'Dispatch run no; always 1';

comment on column DISPATCH_FCAS_REQ.INTERVENTION is
'Intervention Flag';

comment on column DISPATCH_FCAS_REQ.GENCONID is
'Generic Constraint ID - Join to table GenConData';

comment on column DISPATCH_FCAS_REQ.BIDTYPE is
'DUID offered type';

comment on column DISPATCH_FCAS_REQ.GENCONEFFECTIVEDATE is
'Generic Constraint EffectiveDate - Join to table GenConData';

comment on column DISPATCH_FCAS_REQ.GENCONVERSIONNO is
'Generic Constraint Version number - Join to table GenConData';

comment on column DISPATCH_FCAS_REQ.LASTCHANGED is
'Date record is changed';

comment on column DISPATCH_FCAS_REQ.BASE_COST is
'The base cost of the constraint for this service, before the regulation/contingency split';

comment on column DISPATCH_FCAS_REQ.ADJUSTED_COST is
'The adjusted cost of the constraint for this service, before the regulation/contingency split';

comment on column DISPATCH_FCAS_REQ.ESTIMATED_CMPF is
'An estimated value for the constraint CMPF, based on dispatched data';

comment on column DISPATCH_FCAS_REQ.ESTIMATED_CRMPF is
'An estimated value for the constraint CRMPF, based on dispatched data';

comment on column DISPATCH_FCAS_REQ.RECOVERY_FACTOR_CMPF is
'Estimated recovery factor for CMPF based recovery';

comment on column DISPATCH_FCAS_REQ.RECOVERY_FACTOR_CRMPF is
'Estimated recovery factor for CRMPF based recovery';

alter table DISPATCH_FCAS_REQ
   add constraint DISPATCH_FCAS_REQ_PK primary key (SETTLEMENTDATE, RUNNO, INTERVENTION, GENCONID, REGIONID, BIDTYPE)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: DISPATCH_FCAS_REQ_LCX                                 */
/*==============================================================*/
create index DISPATCH_FCAS_REQ_LCX on DISPATCH_FCAS_REQ (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: DISPATCH_FCAS_REQ_CONSTRAINT                          */
/*==============================================================*/
create table DISPATCH_FCAS_REQ_CONSTRAINT (
   RUN_DATETIME         DATE                  not null,
   RUNNO                NUMBER(5)             not null,
   INTERVAL_DATETIME    DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   BIDTYPE              VARCHAR2(10)          not null,
   LHS                  NUMBER(15,5),
   RHS                  NUMBER(15,5),
   MARGINALVALUE        NUMBER(15,5),
   RRP                  NUMBER(15,5),
   REGIONAL_ENABLEMENT  NUMBER(15,5),
   CONSTRAINT_ENABLEMENT NUMBER(15,5),
   REGION_BASE_COST     NUMBER(18,8),
   BASE_COST            NUMBER(18,8),
   ADJUSTED_COST        NUMBER(18,8),
   P_REGULATION         NUMBER(18,8)
)
   tablespace PSSTAB02;

comment on table DISPATCH_FCAS_REQ_CONSTRAINT is
'The constraint level FCAS cost / price details for constraint FCAS processor runs. This enhanced output table format is established for the constraint FCAS processor release required for the Frequency Performance Payments (FPP) release. This enhanced output is hierarchical in nature, with the parent *_FCAS_REQ_RUN table holding the details about the triggering case run and time, and the child *_FCAS_REQ_CONSTRAINT table holding the constraint level details of FCAS costs / prices.';

comment on column DISPATCH_FCAS_REQ_CONSTRAINT.RUN_DATETIME is
'The run date and time of the dispatch case that triggers the constraint FCAS processor run';

comment on column DISPATCH_FCAS_REQ_CONSTRAINT.RUNNO is
'The dispatch case run number that has triggers the constraint FCAS processor run';

comment on column DISPATCH_FCAS_REQ_CONSTRAINT.INTERVAL_DATETIME is
'The trading interval date and time of the dispatch interval that was processed by the constraint FCAS processor';

comment on column DISPATCH_FCAS_REQ_CONSTRAINT.CONSTRAINTID is
'ConstraintID join to table GenConData';

comment on column DISPATCH_FCAS_REQ_CONSTRAINT.REGIONID is
'Region identifier';

comment on column DISPATCH_FCAS_REQ_CONSTRAINT.BIDTYPE is
'DUID offered type';

comment on column DISPATCH_FCAS_REQ_CONSTRAINT.LHS is
'Constraints summed LHS - aggregate LHS Solution values from the physical run from the DISPATCHCONSTRAINT table';

comment on column DISPATCH_FCAS_REQ_CONSTRAINT.RHS is
'Constraints RHS value used in the solution - may be either dynamic (calculated) or static from the physical run from the DISPATCHCONSTRAINT table';

comment on column DISPATCH_FCAS_REQ_CONSTRAINT.MARGINALVALUE is
'Shadow price of constraint from the DISPATCHCONSTRAINT table from the physical run.';

comment on column DISPATCH_FCAS_REQ_CONSTRAINT.RRP is
'Bid type prices for the region coming from the pricing run of the DISPATCHREGIONSUM table';

comment on column DISPATCH_FCAS_REQ_CONSTRAINT.REGIONAL_ENABLEMENT is
'The dispatched MW for the bid type inside the region from the physical run of the DISPATCHREGIONSUM table';

comment on column DISPATCH_FCAS_REQ_CONSTRAINT.CONSTRAINT_ENABLEMENT is
'MW enabled for this bid type within the constraint';

comment on column DISPATCH_FCAS_REQ_CONSTRAINT.REGION_BASE_COST is
'The regional payment allocated to the constraint for the interval pro-rated based on marginal value ratios over the binding constraints for that service in that region (this is an intermediate calculation to get to the base cost)';

comment on column DISPATCH_FCAS_REQ_CONSTRAINT.BASE_COST is
'The base cost of the constraint, before the regulation/contingency split';

comment on column DISPATCH_FCAS_REQ_CONSTRAINT.ADJUSTED_COST is
'The adjusted cost of the constraint for this service, after the regulation/contingency split';

comment on column DISPATCH_FCAS_REQ_CONSTRAINT.P_REGULATION is
'The adjusted marginal value of the constraint for FPP recovery (blank for constraints without REG terms)';

alter table DISPATCH_FCAS_REQ_CONSTRAINT
   add constraint DISPATCH_FCASREQ_CONSTRAINT_PK primary key (RUN_DATETIME, RUNNO, INTERVAL_DATETIME, CONSTRAINTID, REGIONID, BIDTYPE)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: DISPATCH_FCAS_REQ_RUN                                 */
/*==============================================================*/
create table DISPATCH_FCAS_REQ_RUN (
   RUN_DATETIME         DATE                  not null,
   RUNNO                NUMBER(5)             not null,
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table DISPATCH_FCAS_REQ_RUN is
'The constraint FCAS processor run details. This enhanced output table format is established for the constraint FCAS processor release required for the Frequency Performance Payments (FPP) release. This enhanced output is hierarchical in nature, with the parent *_FCAS_REQ_RUN table holding the details about the triggering case run and time, and the child *_FCAS_REQ_CONSTRAINT table holding the constraint level details of FCAS costs / prices.';

comment on column DISPATCH_FCAS_REQ_RUN.RUN_DATETIME is
'The run date and time of the dispatch case that triggers the constraint FCAS processor run';

comment on column DISPATCH_FCAS_REQ_RUN.RUNNO is
'The dispatch case run number that has triggers the constraint FCAS processor run';

comment on column DISPATCH_FCAS_REQ_RUN.LASTCHANGED is
'The last time the constraint FCAS processor was executed for this case run time.';

alter table DISPATCH_FCAS_REQ_RUN
   add constraint DISPATCH_FCAS_REQ_RUN_PK primary key (RUN_DATETIME, RUNNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: DISPATCH_INTERCONNECTION                              */
/*==============================================================*/
create table DISPATCH_INTERCONNECTION (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0)           not null,
   INTERVENTION         NUMBER(2,0)           not null,
   FROM_REGIONID        VARCHAR2(20)          not null,
   TO_REGIONID          VARCHAR2(20)          not null,
   DISPATCHINTERVAL     NUMBER(22,0),
   IRLF                 NUMBER(15,5),
   MWFLOW               NUMBER(16,6),
   METEREDMWFLOW        NUMBER(16,6),
   FROM_REGION_MW_LOSSES NUMBER(16,6),
   TO_REGION_MW_LOSSES  NUMBER(16,6),
   LASTCHANGED          DATE
)
   tablespace PSSTAB03;

comment on table DISPATCH_INTERCONNECTION is
'Inter-regional flow information common to or aggregated for regulated (i.e. not MNSP) Interconnectors spanning the From-Region and To-Region - NB only the physical run is calculated''';

comment on column DISPATCH_INTERCONNECTION.SETTLEMENTDATE is
'Market date starting at 04:05';

comment on column DISPATCH_INTERCONNECTION.RUNNO is
'Dispatch run no; always 1';

comment on column DISPATCH_INTERCONNECTION.INTERVENTION is
'Intervention case or not';

comment on column DISPATCH_INTERCONNECTION.FROM_REGIONID is
'Nominated RegionID from which the energy flows';

comment on column DISPATCH_INTERCONNECTION.TO_REGIONID is
'Nominated RegionID to which the energy flows';

comment on column DISPATCH_INTERCONNECTION.DISPATCHINTERVAL is
'Dispatch period identifier, from 001 to 288 in format YYYYMMDDPPP';

comment on column DISPATCH_INTERCONNECTION.IRLF is
'Inter-Regional Loss Factor. Calculated based on the MWFLOW and the nominal From and To Region losses.';

comment on column DISPATCH_INTERCONNECTION.MWFLOW is
'Summed MW flow of the parallel regulated Interconnectors';

comment on column DISPATCH_INTERCONNECTION.METEREDMWFLOW is
'Summed Metered MW flow of the parallel regulated Interconnectors';

comment on column DISPATCH_INTERCONNECTION.FROM_REGION_MW_LOSSES is
'Losses across the Interconnection attributable to the nominal From Region';

comment on column DISPATCH_INTERCONNECTION.TO_REGION_MW_LOSSES is
'Losses across the Interconnection attributable to the nominal To Region';

comment on column DISPATCH_INTERCONNECTION.LASTCHANGED is
'The datetime that the record was last changed';

alter table DISPATCH_INTERCONNECTION
   add constraint DISPATCH_INTERCONNECTION_PK primary key (SETTLEMENTDATE, RUNNO, FROM_REGIONID, TO_REGIONID, INTERVENTION)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: DISPATCH_LOCAL_PRICE                                  */
/*==============================================================*/
create table DISPATCH_LOCAL_PRICE (
   SETTLEMENTDATE       DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   LOCAL_PRICE_ADJUSTMENT NUMBER(10,2),
   LOCALLY_CONSTRAINED  NUMBER(1,0)
)
   tablespace PSSTAB03;

comment on table DISPATCH_LOCAL_PRICE is
'Sets out local pricing offsets associated with each DUID connection point for each dispatch period. Note that from 2014 Mid year release only records with non-zero Local_Price_Adjustment values are issued';

comment on column DISPATCH_LOCAL_PRICE.SETTLEMENTDATE is
'Market date time starting at 04:05';

comment on column DISPATCH_LOCAL_PRICE.DUID is
'Dispatchable unit identifier';

comment on column DISPATCH_LOCAL_PRICE.LOCAL_PRICE_ADJUSTMENT is
'Aggregate Constraint contribution cost of this unit: Sum(MarginalValue x Factor) for all relevant Constraints';

comment on column DISPATCH_LOCAL_PRICE.LOCALLY_CONSTRAINED is
'Key for Local_Price_Adjustment: 2 = at least one Outage Constraint; 1 = at least 1 System Normal Constraint (and no Outage Constraint); 0 = No System Normal or Outage Constraints';

alter table DISPATCH_LOCAL_PRICE
   add constraint DISPATCH_LOCAL_PRICE_PK primary key (SETTLEMENTDATE, DUID)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: DISPATCH_MNSPBIDTRK                                   */
/*==============================================================*/
create table DISPATCH_MNSPBIDTRK (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   LINKID               VARCHAR2(10)          not null,
   OFFERSETTLEMENTDATE  DATE,
   OFFEREFFECTIVEDATE   TIMESTAMP(3),
   OFFERVERSIONNO       NUMBER(3,0),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table DISPATCH_MNSPBIDTRK is
'DISPATCH_MNSPBIDTRK shows the MNSP bid tracking, including the bid version used in each dispatch run for each MNSP Interconnector Link. The table identifies which bids from MNSP_DAYOFFER and MNSP_BIDOFFERPERIOD were applied.';

comment on column DISPATCH_MNSPBIDTRK.SETTLEMENTDATE is
'Market date starting at 04:05';

comment on column DISPATCH_MNSPBIDTRK.RUNNO is
'Dispatch run no; always 1';

comment on column DISPATCH_MNSPBIDTRK.PARTICIPANTID is
'Participant that owns unit during effective record period';

comment on column DISPATCH_MNSPBIDTRK.LINKID is
'Identifier for each of the two MNSP Interconnector Links. Each link pertains to the direction from and to.';

comment on column DISPATCH_MNSPBIDTRK.OFFERSETTLEMENTDATE is
'Offer date for bid';

comment on column DISPATCH_MNSPBIDTRK.OFFEREFFECTIVEDATE is
'Time this bid was processed and loaded';

comment on column DISPATCH_MNSPBIDTRK.OFFERVERSIONNO is
'VersionNo of the bid/offer used';

comment on column DISPATCH_MNSPBIDTRK.LASTCHANGED is
'Record creation timestamp';

alter table DISPATCH_MNSPBIDTRK
   add constraint DISPATCH_MNSPBIDTRK_PK primary key (SETTLEMENTDATE, RUNNO, PARTICIPANTID, LINKID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: DISPATCH_MNSPBIDTRK_LCX                               */
/*==============================================================*/
create index DISPATCH_MNSPBIDTRK_LCX on DISPATCH_MNSPBIDTRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DISPATCH_MR_SCHEDULE_TRK                              */
/*==============================================================*/
create table DISPATCH_MR_SCHEDULE_TRK (
   SETTLEMENTDATE       DATE                  not null,
   REGIONID             VARCHAR2(10)          not null,
   MR_DATE              DATE,
   VERSION_DATETIME     DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table DISPATCH_MR_SCHEDULE_TRK is
'DISPATCH_MR_SCHEDULE_TRK records the Mandatory Restrictions Acceptance Schedule applied to this dispatch interval for this region.
DISPATCH_MR_SCHEDULE_TRK is populated by the Dispatch process and records the MR Offer Stack applied in each dispatch interval. DISPATCH_MR_SCHEDULE_TRK is used by Settlements to calculate payments according to the correct MR offer stack.';

comment on column DISPATCH_MR_SCHEDULE_TRK.SETTLEMENTDATE is
'Date Time of the Dispatch Interval';

comment on column DISPATCH_MR_SCHEDULE_TRK.REGIONID is
'Unique RegionID; Key reference to MR_Event_Schedule';

comment on column DISPATCH_MR_SCHEDULE_TRK.MR_DATE is
'Mandatory Restriction date; Key reference to MR_Event_Schedule table';

comment on column DISPATCH_MR_SCHEDULE_TRK.VERSION_DATETIME is
'Date Time the MR  acceptance stack was created; Key reference to MR_Event_Schedule table';

comment on column DISPATCH_MR_SCHEDULE_TRK.LASTCHANGED is
'Date and  time the record was last inserted/modified';

alter table DISPATCH_MR_SCHEDULE_TRK
   add constraint DISPATCH_MR_SCHEDULE_TRK_PK primary key (SETTLEMENTDATE, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: DISPATCH_MR_SCHEDULE_TRK_LCX                          */
/*==============================================================*/
create index DISPATCH_MR_SCHEDULE_TRK_LCX on DISPATCH_MR_SCHEDULE_TRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DISPATCH_PRICE_REVISION                               */
/*==============================================================*/
create table DISPATCH_PRICE_REVISION (
   SETTLEMENTDATE       DATE                  not null,
   RUNNO                NUMBER(3,0)           not null,
   INTERVENTION         NUMBER(2,0)           not null,
   REGIONID             VARCHAR2(10)          not null,
   BIDTYPE              VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3)             not null,
   RRP_NEW              NUMBER(15,5),
   RRP_OLD              NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table DISPATCH_PRICE_REVISION is
'An audit trail of price changes on the DISPATCHPRICE table (i.e. for 5 minute dispatch prices for energy and FCAS).';

comment on column DISPATCH_PRICE_REVISION.SETTLEMENTDATE is
'Market date and time starting at 04:05';

comment on column DISPATCH_PRICE_REVISION.RUNNO is
'Dispatch run no; always 1';

comment on column DISPATCH_PRICE_REVISION.INTERVENTION is
'Manual intervention flag; always 0';

comment on column DISPATCH_PRICE_REVISION.REGIONID is
'Affected Region Identifier';

comment on column DISPATCH_PRICE_REVISION.BIDTYPE is
'Affected Bid Type Identifier';

comment on column DISPATCH_PRICE_REVISION.VERSIONNO is
'Version No of price revision for this settlement date';

comment on column DISPATCH_PRICE_REVISION.RRP_NEW is
'New RRP in DISPATCHPRICE table';

comment on column DISPATCH_PRICE_REVISION.RRP_OLD is
'Old RRP from DISPATCHPRICE table';

comment on column DISPATCH_PRICE_REVISION.LASTCHANGED is
'The datetime the record was last changed';

alter table DISPATCH_PRICE_REVISION
   add constraint DISPATCH_PRICE_REVISION_PK primary key (SETTLEMENTDATE, RUNNO, INTERVENTION, REGIONID, BIDTYPE, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: DISPATCH_PRICE_REVISION_LCX                           */
/*==============================================================*/
create index DISPATCH_PRICE_REVISION_LCX on DISPATCH_PRICE_REVISION (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DISPATCH_ROOFTOP_PV_FCST_TRK                          */
/*==============================================================*/
create table DISPATCH_ROOFTOP_PV_FCST_TRK (
   RUN_DATETIME         DATE                  not null,
   AREAID               VARCHAR2(10)          not null,
   FORECAST_RUN_DATETIME DATE,
   PROVIDERID           VARCHAR2(20),
   FORECAST_PRIORITY    NUMBER(10,0),
   OFFERDATETIME        DATE
)
   tablespace PSSTAB02;

comment on table DISPATCH_ROOFTOP_PV_FCST_TRK is
'Uniquely tracks which Rooftop PV forecast run (from ROOFTOP_PV_FCST_P5_RUN) was used for the Area in which Dispatch run.';

comment on column DISPATCH_ROOFTOP_PV_FCST_TRK.RUN_DATETIME is
'Datetime (interval ending) of the Dispatch run.';

comment on column DISPATCH_ROOFTOP_PV_FCST_TRK.AREAID is
'Area identifier aligning with the load forecasting areas, tracks to ROOFTOP_PV_FCST_P5_RUN.AREAID.';

comment on column DISPATCH_ROOFTOP_PV_FCST_TRK.FORECAST_RUN_DATETIME is
'Datetime (interval ending) when this forecast run is valid. It aligns with run_datetime, unless a forecast run is missed, in this case the previous run is used. Tracks to ROOFTOP_PV_FCST_P5_RUN.FORECAST_RUN_DATETIME.';

comment on column DISPATCH_ROOFTOP_PV_FCST_TRK.PROVIDERID is
'Provider identifier of the forecast run used for the DS run, tracks to ROOFTOP_PV_FCST_P5_RUN.PROVIDERID.';

comment on column DISPATCH_ROOFTOP_PV_FCST_TRK.FORECAST_PRIORITY is
'Priority of the forecast run used for the DS run, tracks to ROOFTOP_PV_FCST_P5_RUN.FORECAST_PRIORITY.';

comment on column DISPATCH_ROOFTOP_PV_FCST_TRK.OFFERDATETIME is
'Submission datetime of the forecast run used for the DS run, tracks to ROOFTOP_PV_FCST_P5_RUN.OFFERDATETIME.';

alter table DISPATCH_ROOFTOP_PV_FCST_TRK
   add constraint DISPATCH_ROOFTOPPV_FCST_TRK_PK primary key (RUN_DATETIME, AREAID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: DISPATCH_UNIT_CONFORMANCE                             */
/*==============================================================*/
create table DISPATCH_UNIT_CONFORMANCE (
   INTERVAL_DATETIME    DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   TOTALCLEARED         NUMBER(16,6),
   ACTUALMW             NUMBER(16,6),
   ROC                  NUMBER(16,6),
   AVAILABILITY         NUMBER(16,6),
   LOWERREG             NUMBER(16,6),
   RAISEREG             NUMBER(16,6),
   STRIGLM              NUMBER(16,6),
   LTRIGLM              NUMBER(16,6),
   MWERROR              NUMBER(16,6),
   MAX_MWERROR          NUMBER(16,6),
   LECOUNT              NUMBER(6),
   SECOUNT              NUMBER(6),
   STATUS               VARCHAR2(20),
   PARTICIPANT_STATUS_ACTION VARCHAR2(100),
   OPERATING_MODE       VARCHAR2(20),
   LASTCHANGED          DATE,
   ADG_ID               VARCHAR2(20),
   SEMIDISPATCHCAP      NUMBER(3,0),
   CONFORMANCE_MODE     NUMBER(6,0)
)
   tablespace PSSTAB01;

comment on table DISPATCH_UNIT_CONFORMANCE is
'DISPATCH_UNIT_CONFORMANCE details the conformance of a scheduled units operation with respect to a cleared target on dispatch interval basis.
Data is confidential';

comment on column DISPATCH_UNIT_CONFORMANCE.INTERVAL_DATETIME is
'Dispatch Interval that the conformance data applies to';

comment on column DISPATCH_UNIT_CONFORMANCE.DUID is
'Dispatchable Unit Identifier, or Aggregate Dispatch Group identifier';

comment on column DISPATCH_UNIT_CONFORMANCE.TOTALCLEARED is
'Dispatch Target - MW';

comment on column DISPATCH_UNIT_CONFORMANCE.ACTUALMW is
'Unit output measured at the conclusion of the dispatch interval - MW (MWB)';

comment on column DISPATCH_UNIT_CONFORMANCE.ROC is
'Rate of change in direction of error MW per minute';

comment on column DISPATCH_UNIT_CONFORMANCE.AVAILABILITY is
'Offered unit capacity - MW (MWO)';

comment on column DISPATCH_UNIT_CONFORMANCE.LOWERREG is
'Lower Regulation FCAS enabled - MW (FCL)';

comment on column DISPATCH_UNIT_CONFORMANCE.RAISEREG is
'Raise Regulation FCAS enabled - MW (FCR)';

comment on column DISPATCH_UNIT_CONFORMANCE.STRIGLM is
'Calculated small trigger error limit in MW';

comment on column DISPATCH_UNIT_CONFORMANCE.LTRIGLM is
'Calculated large trigger error limit in MW';

comment on column DISPATCH_UNIT_CONFORMANCE.MWERROR is
'Calculated actual error';

comment on column DISPATCH_UNIT_CONFORMANCE.MAX_MWERROR is
'Max of mwerror while that unit was not in a normal state';

comment on column DISPATCH_UNIT_CONFORMANCE.LECOUNT is
'Large trigger error count. Reset when mwerror changes sign';

comment on column DISPATCH_UNIT_CONFORMANCE.SECOUNT is
'Small trigger error count.  Reset when mwerror changes sign';

comment on column DISPATCH_UNIT_CONFORMANCE.STATUS is
'Unit conformance status.
NORMAL
OFF-TARGET
NOT-RESPONDING
NC-PENDING
NON-CONFORMING
SUSPENDED';

comment on column DISPATCH_UNIT_CONFORMANCE.PARTICIPANT_STATUS_ACTION is
'Participant action required in response to current STATUS';

comment on column DISPATCH_UNIT_CONFORMANCE.OPERATING_MODE is
'conformance operating mode
MANUAL
AUTO';

comment on column DISPATCH_UNIT_CONFORMANCE.LASTCHANGED is
'Last date and time record changed';

comment on column DISPATCH_UNIT_CONFORMANCE.ADG_ID is
'Aggregate Dispatch Group to which this dispatch unit belongs';

comment on column DISPATCH_UNIT_CONFORMANCE.SEMIDISPATCHCAP is
'Boolean representation flagging if the Target is capped';

comment on column DISPATCH_UNIT_CONFORMANCE.CONFORMANCE_MODE is
'For an individual unit in an aggregate dispatch group (where DUID <> ADG_ID), Mode specific to that unit. 0 - no monitoring, 1 - aggregate monitoring, 2 - individual monitoring due to constraint. For the aggregate dispatch group (where DUID = ADG_ID), 0 - no aggregate monitoring, 1 - aggregate monitoring';

alter table DISPATCH_UNIT_CONFORMANCE
   add constraint PK_DISPATCH_UNIT_CONFORMANCE primary key (INTERVAL_DATETIME, DUID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: DISPATCH_UNIT_CONFORMANCE_LCX                         */
/*==============================================================*/
create index DISPATCH_UNIT_CONFORMANCE_LCX on DISPATCH_UNIT_CONFORMANCE (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DISPATCH_UNIT_SCADA                                   */
/*==============================================================*/
create table DISPATCH_UNIT_SCADA (
   SETTLEMENTDATE       Date                  not null,
   DUID                 varchar2(20)          not null,
   SCADAVALUE           Number(16,6),
   LASTCHANGED          DATE
)
   tablespace PSSTAB03;

comment on table DISPATCH_UNIT_SCADA is
'Dispatchable unit MW from SCADA at the start of the dispatch interval. The table includes all scheduled and semi-scheduled (and non-scheduled units where SCADA is available)';

comment on column DISPATCH_UNIT_SCADA.SETTLEMENTDATE is
'Date Time of the Dispatch Interval';

comment on column DISPATCH_UNIT_SCADA.DUID is
'Dispatchable Unit Identifier';

comment on column DISPATCH_UNIT_SCADA.SCADAVALUE is
'Instantaneous MW reading from SCADA at the start of the Dispatch interval';

comment on column DISPATCH_UNIT_SCADA.LASTCHANGED is
'Last date and time record changed';

alter table DISPATCH_UNIT_SCADA
   add constraint DISPATCH_UNIT_SCADA_PK primary key (SETTLEMENTDATE, DUID)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: DUALLOC                                               */
/*==============================================================*/
create table DUALLOC (
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   DUID                 VARCHAR2(10)          not null,
   GENSETID             VARCHAR2(20)          not null,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table DUALLOC is
'DUALLOC cross references dispatch unit identifier to genset ID for each participant.';

comment on column DUALLOC.EFFECTIVEDATE is
'Effective calendar date of record';

comment on column DUALLOC.VERSIONNO is
'Version no of record';

comment on column DUALLOC.DUID is
'Dispatchable Unit identifier';

comment on column DUALLOC.GENSETID is
'Physical unit identifier';

comment on column DUALLOC.LASTCHANGED is
'Last date and time record changed';

alter table DUALLOC
   add constraint DUALLOC_PK primary key (DUID, EFFECTIVEDATE, VERSIONNO, GENSETID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: DUALLOC_NDX2                                          */
/*==============================================================*/
create index DUALLOC_NDX2 on DUALLOC (
   DUID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: DUALLOC_LCX                                           */
/*==============================================================*/
create index DUALLOC_LCX on DUALLOC (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DUDETAIL                                              */
/*==============================================================*/
create table DUDETAIL (
   EFFECTIVEDATE        DATE                  not null,
   DUID                 VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   CONNECTIONPOINTID    VARCHAR2(10),
   VOLTLEVEL            VARCHAR2(10),
   REGISTEREDCAPACITY   NUMBER(6,0),
   AGCCAPABILITY        VARCHAR2(1),
   DISPATCHTYPE         VARCHAR2(20),
   MAXCAPACITY          NUMBER(6,0),
   STARTTYPE            VARCHAR2(20),
   NORMALLYONFLAG       VARCHAR2(1),
   PHYSICALDETAILSFLAG  VARCHAR2(1),
   SPINNINGRESERVEFLAG  VARCHAR2(1),
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE,
   INTERMITTENTFLAG     VARCHAR(1),
   SEMISCHEDULE_FLAG    VARCHAR2(1),
   MAXRATEOFCHANGEUP    Number(6,0),
   MAXRATEOFCHANGEDOWN  Number(6,0),
   DISPATCHSUBTYPE      VARCHAR2(20),
   ADG_ID               VARCHAR2(20),
   MINCAPACITY          NUMBER(6,0),
   REGISTEREDMINCAPACITY NUMBER(6,0),
   MAXRATEOFCHANGEUP_LOAD NUMBER(6,0),
   MAXRATEOFCHANGEDOWN_LOAD NUMBER(6,0),
   MAXSTORAGECAPACITY   NUMBER(15,5),
   STORAGEIMPORTEFFICIENCYFACTOR NUMBER(15,5),
   STORAGEEXPORTEFFICIENCYFACTOR NUMBER(15,5),
   MIN_RAMP_RATE_UP     NUMBER(6,0),
   MIN_RAMP_RATE_DOWN   NUMBER(6,0),
   LOAD_MIN_RAMP_RATE_UP NUMBER(6,0),
   LOAD_MIN_RAMP_RATE_DOWN NUMBER(6,0),
   AGGREGATED           NUMBER(1,0)
)
   tablespace PSSTAB01;

comment on table DUDETAIL is
'DUDETAIL sets out a records specific details for each unit including start type and whether normally on or off load. Much of this data is information only and is not used in dispatch or settlements.';

comment on column DUDETAIL.EFFECTIVEDATE is
'Effective calendar date of record';

comment on column DUDETAIL.DUID is
'Dispatchable Unit Identifier';

comment on column DUDETAIL.VERSIONNO is
'version of Dispatchable Unit details for this effective date';

comment on column DUDETAIL.CONNECTIONPOINTID is
'Country wide - Unique id of a connection point';

comment on column DUDETAIL.VOLTLEVEL is
'Voltage Level';

comment on column DUDETAIL.REGISTEREDCAPACITY is
'Registered capacity for normal operations';

comment on column DUDETAIL.AGCCAPABILITY is
'AGC Capability flag';

comment on column DUDETAIL.DISPATCHTYPE is
'Identifies LOAD, GENERATOR or BIDIRECTIONAL.';

comment on column DUDETAIL.MAXCAPACITY is
'Maximum Capacity as used for bid validation';

comment on column DUDETAIL.STARTTYPE is
'Identify unit as Fast or Slow';

comment on column DUDETAIL.NORMALLYONFLAG is
'For a dispatchable load indicates that the load is normally on or off.';

comment on column DUDETAIL.PHYSICALDETAILSFLAG is
'Indicates that the physical details for this unit are to be recorded';

comment on column DUDETAIL.SPINNINGRESERVEFLAG is
'Indicates spinning reserve capability';

comment on column DUDETAIL.AUTHORISEDBY is
'User authorising record';

comment on column DUDETAIL.AUTHORISEDDATE is
'Date record authorised';

comment on column DUDETAIL.LASTCHANGED is
'Last date and time record changed';

comment on column DUDETAIL.INTERMITTENTFLAG is
'Indicate whether a unit is intermittent (e.g. a wind farm)';

comment on column DUDETAIL.SEMISCHEDULE_FLAG is
'Indicates if the DUID is a Semi-Scheduled Unit';

comment on column DUDETAIL.MAXRATEOFCHANGEUP is
'Maximum ramp up rate for Unit (Mw/min)';

comment on column DUDETAIL.MAXRATEOFCHANGEDOWN is
'Maximum ramp down rate for Unit (Mw/min)';

comment on column DUDETAIL.DISPATCHSUBTYPE is
'Additional information for DISPATCHTYPE. For DISPATCHTYPE = LOAD, subtype value is WDR for wholesale demand response units. For DISPATCHTYPE = LOAD, subtype value is NULL for Scheduled Loads. For DISPATCHTYPE = GENERATOR type, the subtype value is NULL.';

comment on column DUDETAIL.ADG_ID is
'Aggregate Dispatch Group to which this dispatch unit belongs';

comment on column DUDETAIL.MINCAPACITY is
'Minimum capacity only for load side of BDU, otherwise 0 (MW)';

comment on column DUDETAIL.REGISTEREDMINCAPACITY is
'Registered minimum capacity only for load side of BDU, otherwise 0 (MW)';

comment on column DUDETAIL.MAXRATEOFCHANGEUP_LOAD is
'Raise Ramp rate applied to BDU Load component (MW/min)';

comment on column DUDETAIL.MAXRATEOFCHANGEDOWN_LOAD is
'Lower Ramp rate applied to BDU Load component (MW/min)';

comment on column DUDETAIL.MAXSTORAGECAPACITY is
'The rated storage capacity (MWh), information only';

comment on column DUDETAIL.STORAGEIMPORTEFFICIENCYFACTOR is
'The storage energy import conversion efficiency. Number from 0 to 1 where 1 is lossless. Calculated as (increase in stored energy / increase in imported energy)';

comment on column DUDETAIL.STORAGEEXPORTEFFICIENCYFACTOR is
'The storage energy export conversion efficiency. Number from 0 to 1 where 1 is lossless. Calculated as (decrease in exported energy / decrease in stored energy)';

comment on column DUDETAIL.MIN_RAMP_RATE_UP is
'Calculated Minimum Ramp Rate Up value accepted for Energy Offers or Bids with explanation for energy imports (all DUID types and BDU Generation side) (MW/min)';

comment on column DUDETAIL.MIN_RAMP_RATE_DOWN is
'Calculated Minimum Ramp Rate Down value accepted for Energy Offers or Bids with explanation for energy imports (all DUID types and BDU Generation side) (MW/min)';

comment on column DUDETAIL.LOAD_MIN_RAMP_RATE_UP is
'Calculated Minimum Ramp Rate Up value accepted for Energy Offers or Bids on BDU Load component with explanation for energy imports (MW/min)';

comment on column DUDETAIL.LOAD_MIN_RAMP_RATE_DOWN is
'Calculated Minimum Ramp Rate Down value accepted for Energy Offers or Bids on BDU Load component with explanation for energy imports (MW/min)';

comment on column DUDETAIL.AGGREGATED is
'Identifies if a unit is aggregated. This flag was initially added in GENUNITS_UNIT table which is now deprecated with IESS release.';

alter table DUDETAIL
   add constraint DUDETAIL_PK primary key (DUID, EFFECTIVEDATE, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: DUDETAIL_LCX                                          */
/*==============================================================*/
create index DUDETAIL_LCX on DUDETAIL (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: DUDETAILSUMMARY                                       */
/*==============================================================*/
create table DUDETAILSUMMARY (
   DUID                 VARCHAR2(10)          not null,
   START_DATE           DATE                  not null,
   END_DATE             DATE                  not null,
   DISPATCHTYPE         VARCHAR2(20),
   CONNECTIONPOINTID    VARCHAR2(10),
   REGIONID             VARCHAR2(10),
   STATIONID            VARCHAR2(10),
   PARTICIPANTID        VARCHAR2(10),
   LASTCHANGED          DATE,
   TRANSMISSIONLOSSFACTOR NUMBER(15,5),
   STARTTYPE            VARCHAR2(20),
   DISTRIBUTIONLOSSFACTOR NUMBER(15,5),
   MINIMUM_ENERGY_PRICE NUMBER(9,2),
   MAXIMUM_ENERGY_PRICE NUMBER(9,2),
   SCHEDULE_TYPE        VARCHAR2(20),
   MIN_RAMP_RATE_UP     number(6,0),
   MIN_RAMP_RATE_DOWN   number(6,0),
   MAX_RAMP_RATE_UP     number(6,0),
   MAX_RAMP_RATE_DOWN   number(6,0),
   IS_AGGREGATED        NUMBER(1,0),
   DISPATCHSUBTYPE      VARCHAR2(20),
   ADG_ID               VARCHAR2(20),
   LOAD_MINIMUM_ENERGY_PRICE NUMBER(9,2),
   LOAD_MAXIMUM_ENERGY_PRICE NUMBER(9,2),
   LOAD_MIN_RAMP_RATE_UP NUMBER(6,0),
   LOAD_MIN_RAMP_RATE_DOWN NUMBER(6,0),
   LOAD_MAX_RAMP_RATE_UP NUMBER(6,0),
   LOAD_MAX_RAMP_RATE_DOWN NUMBER(6,0),
   SECONDARY_TLF        NUMBER(18,8)
)
   tablespace PSSTAB01;

comment on table DUDETAILSUMMARY is
'DUDETAILSUMMARY sets out a single summary unit table so reducing the need for participants to use the various dispatchable unit detail and owner tables to establish generating unit specific details.';

comment on column DUDETAILSUMMARY.DUID is
'Dispatchable Unit Identifier';

comment on column DUDETAILSUMMARY.START_DATE is
'Start date for effective record';

comment on column DUDETAILSUMMARY.END_DATE is
'End date for effective record';

comment on column DUDETAILSUMMARY.DISPATCHTYPE is
'Identifies LOAD, GENERATOR or BIDIRECTIONAL. This will likely expand to more generic models as new technology types are integrated into the NEM';

comment on column DUDETAILSUMMARY.CONNECTIONPOINTID is
'Country wide - Unique id of a connection point';

comment on column DUDETAILSUMMARY.REGIONID is
'Region identifier that unit is in';

comment on column DUDETAILSUMMARY.STATIONID is
'Station that unit is in';

comment on column DUDETAILSUMMARY.PARTICIPANTID is
'Participant that owns unit during effective record period';

comment on column DUDETAILSUMMARY.LASTCHANGED is
'Last date and time record changed';

comment on column DUDETAILSUMMARY.TRANSMISSIONLOSSFACTOR is
'Used in Bidding, Dispatch and Settlements. For Bidding and Dispatch, where the DUID is a BDU with DISPATCHTYPE of BIDIRECTIONAL, the TLF for the load component of the BDU. For Settlements, where dual TLFs apply, the primary TLF is applied to all energy (load and generation) when the Net Energy Flow of the ConnectionPointID in the interval is negative (net load).';

comment on column DUDETAILSUMMARY.STARTTYPE is
'Unit start type. At this time restricted to Fast, Slow or Non Dispatched';

comment on column DUDETAILSUMMARY.DISTRIBUTIONLOSSFACTOR is
'The distribution loss factor to the currently assigned connection point';

comment on column DUDETAILSUMMARY.MINIMUM_ENERGY_PRICE is
'Floored Offer/Bid Energy Price adjusted for TLF, DLF and MPF';

comment on column DUDETAILSUMMARY.MAXIMUM_ENERGY_PRICE is
'Capped Offer/Bid Energy Price adjusted for TLF, DLF and VoLL';

comment on column DUDETAILSUMMARY.SCHEDULE_TYPE is
'Scheduled status of the unit:
    ''SCHEDULED''
    ''NON-SCHEDULED''
    ''SEMI-SCHEDULED''';

comment on column DUDETAILSUMMARY.MIN_RAMP_RATE_UP is
'MW/Min. Calculated Minimum Ramp Rate Up value accepted for Energy Offers or Bids with explanation';

comment on column DUDETAILSUMMARY.MIN_RAMP_RATE_DOWN is
'MW/Min. Calculated Minimum Ramp Rate Down value accepted for Energy Offers or Bids with explanation';

comment on column DUDETAILSUMMARY.MAX_RAMP_RATE_UP is
'Maximum ramp up rate for Unit (Mw/min) - from DUDetail table';

comment on column DUDETAILSUMMARY.MAX_RAMP_RATE_DOWN is
'Maximum ramp down rate for Unit (Mw/min) - from DUDetail table';

comment on column DUDETAILSUMMARY.IS_AGGREGATED is
'Whether the DUID is classified as an "Aggregated Unit" under the rules. This impacts the Minimum Ramp Rate calculation';

comment on column DUDETAILSUMMARY.DISPATCHSUBTYPE is
'Additional information for DISPATCHTYPE. For DISPATCHTYPE = LOAD, subtype value is WDR for wholesale demand response units For DISPATCHTYPE = LOAD, subtype value is NULL for Scheduled Loads. For DISPATCHTYPE = GENERATOR type, subtype value is NULL.';

comment on column DUDETAILSUMMARY.ADG_ID is
'Aggregate Dispatch Group. Group into which the DUID is aggregated for Conformance. Null if DUID not aggregated for Conformance';

comment on column DUDETAILSUMMARY.LOAD_MINIMUM_ENERGY_PRICE is
'BDU only. Floored Offer/Bid Energy Price adjusted for TLF, DLF and MPF for energy imports';

comment on column DUDETAILSUMMARY.LOAD_MAXIMUM_ENERGY_PRICE is
'BDU only. Capped Offer/Bid Energy Price adjusted for TLF, DLF and VoLL for energy imports';

comment on column DUDETAILSUMMARY.LOAD_MIN_RAMP_RATE_UP is
'BDU only. MW/Min. Calculated Minimum Ramp Rate Up value accepted for Energy Offers or Bids with explanation for energy imports';

comment on column DUDETAILSUMMARY.LOAD_MIN_RAMP_RATE_DOWN is
'BDU only. MW/Min. Calculated Minimum Ramp Rate Down value accepted for Energy Offers or Bids with explanation for energy imports';

comment on column DUDETAILSUMMARY.LOAD_MAX_RAMP_RATE_UP is
'BDU only. MW/Min. Registered Maximum Ramp Rate Up value accepted for Energy Offers or Bids for energy imports';

comment on column DUDETAILSUMMARY.LOAD_MAX_RAMP_RATE_DOWN is
'BDU only. MW/Min. Registered Maximum Ramp Rate Down value accepted for Energy Offers or Bids for energy imports';

comment on column DUDETAILSUMMARY.SECONDARY_TLF is
'Used in Bidding, Dispatch and Settlements, only populated where Dual TLFs apply. For Bidding and Dispatch, the TLF for the generation component of a BDU, when null the TRANSMISSIONLOSSFACTOR is used for both the load and generation components. For Settlements, the secondary TLF is applied to all energy (load and generation) when the Net Energy Flow of the ConnectionPointID in the interval is positive (net generation).';

alter table DUDETAILSUMMARY
   add constraint DUDETAILSUMMARY_PK primary key (DUID, START_DATE)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: DUDETAILSUMMARY_LCX                                   */
/*==============================================================*/
create index DUDETAILSUMMARY_LCX on DUDETAILSUMMARY (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: EMSMASTER                                             */
/*==============================================================*/
create table EMSMASTER (
   SPD_ID               VARCHAR(21)           not null,
   SPD_TYPE             VARCHAR(1)            not null,
   DESCRIPTION          VARCHAR(255),
   GROUPING_ID          VARCHAR(20),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table EMSMASTER is
'EMSMASTER provides a description of the SCADA measurements that are associated with the SPD_ID points utilised in generic equation RHS terms';

comment on column EMSMASTER.SPD_ID is
'ID defining data source';

comment on column EMSMASTER.SPD_TYPE is
'ID describing type of data source';

comment on column EMSMASTER.DESCRIPTION is
'The detailed description of the SCADA point associated with the SPD_ID';

comment on column EMSMASTER.GROUPING_ID is
'The Grouping associated with the SPD ID - most often a RegionID';

comment on column EMSMASTER.LASTCHANGED is
'Last date and time record changed';

alter table EMSMASTER
   add constraint EMSMASTER_PK primary key (SPD_ID, SPD_TYPE)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: EMSMASTER_LCX                                         */
/*==============================================================*/
create index EMSMASTER_LCX on EMSMASTER (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: FCAS_REGU_USAGE_FACTORS                               */
/*==============================================================*/
create table FCAS_REGU_USAGE_FACTORS (
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   REGIONID             VARCHAR2(20)          not null,
   BIDTYPE              VARCHAR2(20)          not null,
   PERIODID             NUMBER(3,0)           not null,
   USAGE_FACTOR         NUMBER(8,3),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table FCAS_REGU_USAGE_FACTORS is
'Stores the proportion of enabled regulation FCAS dispatch that is typically consumed for frequency regulation. Used to calculate the projected state of charge for energy storage systems.';

comment on column FCAS_REGU_USAGE_FACTORS.EFFECTIVEDATE is
'The effective date for this regulation FCAS usage factor';

comment on column FCAS_REGU_USAGE_FACTORS.VERSIONNO is
'Version with respect to effective date';

comment on column FCAS_REGU_USAGE_FACTORS.REGIONID is
'Unique RegionID';

comment on column FCAS_REGU_USAGE_FACTORS.BIDTYPE is
'The type of regulation FCAS service [RAISEREG,LOWERREG]';

comment on column FCAS_REGU_USAGE_FACTORS.PERIODID is
'The Period ID (1 - 48) within the calendar day to which this usage factor applies';

comment on column FCAS_REGU_USAGE_FACTORS.USAGE_FACTOR is
'The proportion of cleared regulation FCAS that is assumed to be used within a dispatch interval. Expressed as a fractional amount between 0 and 1';

comment on column FCAS_REGU_USAGE_FACTORS.LASTCHANGED is
'The last time the data has been changed/updated';

alter table FCAS_REGU_USAGE_FACTORS
   add constraint FCAS_REGU_USAGE_FACTORS_PK primary key (EFFECTIVEDATE, VERSIONNO, REGIONID, BIDTYPE, PERIODID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: FCAS_REGU_USAGE_FACTORS_TRK                           */
/*==============================================================*/
create table FCAS_REGU_USAGE_FACTORS_TRK (
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table FCAS_REGU_USAGE_FACTORS_TRK is
'Stores the proportion of enabled regulation FCAS dispatch that is typically consumed for frequency regulation. Used to calculate the projected state of charge for energy storage systems.';

comment on column FCAS_REGU_USAGE_FACTORS_TRK.EFFECTIVEDATE is
'The effective date for this regulation FCAS usage factor';

comment on column FCAS_REGU_USAGE_FACTORS_TRK.VERSIONNO is
'Version of the date with respect to effective date';

comment on column FCAS_REGU_USAGE_FACTORS_TRK.AUTHORISEDDATE is
'The date time that this set of usage factors was authorised';

comment on column FCAS_REGU_USAGE_FACTORS_TRK.LASTCHANGED is
'The last time the data has been changed/updated';

alter table FCAS_REGU_USAGE_FACTORS_TRK
   add constraint FCAS_REGU_USAGE_FACTORS_TRK_PK primary key (EFFECTIVEDATE, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: FORCEMAJEURE                                          */
/*==============================================================*/
create table FORCEMAJEURE (
   FMID                 VARCHAR2(10)          not null,
   STARTDATE            DATE,
   STARTPERIOD          NUMBER(3,0),
   ENDDATE              DATE,
   ENDPERIOD            NUMBER(3,0),
   APCSTARTDATE         DATE,
   STARTAUTHORISEDBY    VARCHAR2(15),
   ENDAUTHORISEDBY      VARCHAR2(15),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table FORCEMAJEURE is
'FORCEMAJEURE used to set out the start and end dates / periods of any force majeure event. FORCEMAJEURE is not used.';

comment on column FORCEMAJEURE.FMID is
'Force Majeure Identifier';

comment on column FORCEMAJEURE.STARTDATE is
'Start Date for this event';

comment on column FORCEMAJEURE.STARTPERIOD is
'Start Trading Interval for event';

comment on column FORCEMAJEURE.ENDDATE is
'End Date for this event';

comment on column FORCEMAJEURE.ENDPERIOD is
'End Trading Interval for this event';

comment on column FORCEMAJEURE.APCSTARTDATE is
'APC Start Date';

comment on column FORCEMAJEURE.STARTAUTHORISEDBY is
'User authorising start';

comment on column FORCEMAJEURE.ENDAUTHORISEDBY is
'User authorising end of event';

comment on column FORCEMAJEURE.LASTCHANGED is
'Last date and time record changed';

alter table FORCEMAJEURE
   add constraint FORCEMAJEURE_PK primary key (FMID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: FORCEMAJEURE_LCX                                      */
/*==============================================================*/
create index FORCEMAJEURE_LCX on FORCEMAJEURE (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: FORCEMAJEUREREGION                                    */
/*==============================================================*/
create table FORCEMAJEUREREGION (
   FMID                 VARCHAR2(10)          not null,
   REGIONID             VARCHAR2(10)          not null,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table FORCEMAJEUREREGION is
'FORCEMAJEUREREGION used to set out regions impacted by a force majeure event. This table is not used.';

comment on column FORCEMAJEUREREGION.FMID is
'Force Majeure ID';

comment on column FORCEMAJEUREREGION.REGIONID is
'Differentiates this region from all other regions';

comment on column FORCEMAJEUREREGION.LASTCHANGED is
'Last date and time record changed';

alter table FORCEMAJEUREREGION
   add constraint FORCEMAJEUREREGION_PK primary key (FMID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: FORCEMAJEUREREGION_LCX                                */
/*==============================================================*/
create index FORCEMAJEUREREGION_LCX on FORCEMAJEUREREGION (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: FPP_CONSTRAINT_FREQ_MEASURE                           */
/*==============================================================*/
create table FPP_CONSTRAINT_FREQ_MEASURE (
   INTERVAL_DATETIME    DATE                  not null,
   MEASUREMENT_DATETIME DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(5)             not null,
   BIDTYPE              VARCHAR2(10),
   FM_RAISE_HZ          NUMBER(18,8),
   FM_LOWER_HZ          NUMBER(18,8),
   USED_IN_RCR_FLAG     NUMBER(5),
   CORRELATION_FLAG     NUMBER(5)
)
   tablespace PSSTAB03;

comment on table FPP_CONSTRAINT_FREQ_MEASURE is
'This report delivers the weighted 4 second frequency measure data for each constraint';

comment on column FPP_CONSTRAINT_FREQ_MEASURE.INTERVAL_DATETIME is
'Date and time of the trading interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_CONSTRAINT_FREQ_MEASURE.MEASUREMENT_DATETIME is
'Date and time of the SCADA data (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_CONSTRAINT_FREQ_MEASURE.CONSTRAINTID is
'Constraint ID (binding constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_CONSTRAINT_FREQ_MEASURE.VERSIONNO is
'Version (FPP run number from the FPP database)';

comment on column FPP_CONSTRAINT_FREQ_MEASURE.BIDTYPE is
'Bid type (the bid type saved in relation to constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_CONSTRAINT_FREQ_MEASURE.FM_RAISE_HZ is
'Calculated 4 second Frequency Measure in Hz for that constraint from the FPP database. Frequency Measure data is split across these two raise and lower columns in the following ways: >0 = Allocated to the FM_RAISE_HZ column <0 = Allocated to the FM_LOWER_HZ column 0 = To fill any gaps where the alternative column is not applicable (or no deviation from 50 Hz)';

comment on column FPP_CONSTRAINT_FREQ_MEASURE.FM_LOWER_HZ is
'Calculated 4 second Frequency Measure in Hz for that constraint from the FPP database. Frequency Measure data is split across these two raise and lower columns in the following ways: >0 = Allocated to the FM_RAISE_HZ column <0 = Allocated to the FM_LOWER_HZ column 0 = To fill any gaps where the alternative column is not applicable (or no deviation from 50 Hz)';

comment on column FPP_CONSTRAINT_FREQ_MEASURE.USED_IN_RCR_FLAG is
'Flag to indicate the result of the Frequency Measure alignment check between Mainland and Tasmania for global constraints. Supported values are: 0 = Not used in RCR calculation as the signs for the frequency measures between Mainland and Tasmania do not align 1 = Used in the RCR calculation as the signs for the frequency measures between Mainland and Tasmania do align in the case of global constraints. For non-global constraints this flag is set to 1';

comment on column FPP_CONSTRAINT_FREQ_MEASURE.CORRELATION_FLAG is
'Flag to indicate the result of the Frequency Measure correlation check between regions in the same constraint. Supported values are: 0 = Frequency measures in this constraint are not correlated (e.g. system separation between two regions) 1 = Frequency measures in this constraint are correlated';

alter table FPP_CONSTRAINT_FREQ_MEASURE
   add constraint FPP_CONSTRAINT_FREQ_MEASURE_PK primary key (INTERVAL_DATETIME, MEASUREMENT_DATETIME, CONSTRAINTID, VERSIONNO)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: FPP_CONTRIBUTION_FACTOR                               */
/*==============================================================*/
create table FPP_CONTRIBUTION_FACTOR (
   INTERVAL_DATETIME    DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   FPP_UNITID           VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(5)             not null,
   BIDTYPE              VARCHAR2(10),
   CONTRIBUTION_FACTOR  NUMBER(18,8),
   NEGATIVE_CONTRIBUTION_FACTOR NUMBER(18,8),
   DEFAULT_CONTRIBUTION_FACTOR NUMBER(18,8),
   CF_REASON_FLAG       NUMBER(5),
   CF_ABS_POSITIVE_PERF_TOTAL NUMBER(18,8),
   CF_ABS_NEGATIVE_PERF_TOTAL NUMBER(18,8),
   NCF_ABS_NEGATIVE_PERF_TOTAL NUMBER(18,8),
   PARTICIPANTID        VARCHAR2(20),
   SETTLEMENTS_UNITID   VARCHAR2(50)
)
   tablespace PSSTAB02;

comment on table FPP_CONTRIBUTION_FACTOR is
'This report delivers the calculated contribution factor value for each 5 minute trading interval for each constraint and FPP unit';

comment on column FPP_CONTRIBUTION_FACTOR.INTERVAL_DATETIME is
'Date and time of the trading interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_CONTRIBUTION_FACTOR.CONSTRAINTID is
'Constraint ID (binding constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_CONTRIBUTION_FACTOR.FPP_UNITID is
'FPP Unit ID (registered DUID/ TNI)';

comment on column FPP_CONTRIBUTION_FACTOR.VERSIONNO is
'Version (FPP run number from the FPP database)';

comment on column FPP_CONTRIBUTION_FACTOR.BIDTYPE is
'Bid type (the bid type saved in relation to constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_CONTRIBUTION_FACTOR.CONTRIBUTION_FACTOR is
'Contribution Factor (the calculated contribution factor for the FPP unit and constraint ID for that trading interval) - for further details please see the FPP procedure document';

comment on column FPP_CONTRIBUTION_FACTOR.NEGATIVE_CONTRIBUTION_FACTOR is
'Negative Contribution Factor (the calculated negative contribution factor for the FPP unit and constraint ID for that trading interval) - for further details please see the FPP procedure document';

comment on column FPP_CONTRIBUTION_FACTOR.DEFAULT_CONTRIBUTION_FACTOR is
'The Default Contribution Factor (the calculated default contribution factor based on historical performance for the FPP unit and constraint ID for that trading interval) that is effective for this trading interval, which joins back to FPP_FORECAST_DEFAULT_CF - for further details please see the FPP procedure document';

comment on column FPP_CONTRIBUTION_FACTOR.CF_REASON_FLAG is
'The reason flag showing the decision matrix for the contribution factor (CF) Supported values are: 0 = CF is calculated based on good input data 1 = CF is 0 because it is not primary in the group 2 = CF is not for the DUID but for the whole group 4 = CF is calculated based on substitute performance 8 = CF is 0 because FM is unreliable. 16 = CF is 0 because more than 50 percent input is bad or not available.';

comment on column FPP_CONTRIBUTION_FACTOR.CF_ABS_POSITIVE_PERF_TOTAL is
'The sum of absolute positive performance in MWHz for the combination of constraint / bid type (raise or lower). This is used as the denominator in normalising contribution factors (CF) where a units performance is positive. For further details please see the FPP procedure document. >0 = Performance was calculated for the units NULL = Performance for the units was unavailable';

comment on column FPP_CONTRIBUTION_FACTOR.CF_ABS_NEGATIVE_PERF_TOTAL is
'The sum of absolute negative performance in MWHz for the combination of constraint / bid type (raise or lower). This is used as the denominator in normalising contribution factors (CF) where a units performance is negative. For further details please see the FPP procedure document. >0 = Performance was calculated for the units NULL = Performance for the units was unavailable';

comment on column FPP_CONTRIBUTION_FACTOR.NCF_ABS_NEGATIVE_PERF_TOTAL is
'The sum of absolute negative performance in MWHz for the combination of constraint / bid type (raise or lower). This is used as the denominator in normalising negative contribution factors (NCF). For further details please see the FPP procedure document. >0 = Performance was calculated for the units NULL = Performance for the units was unavailable 0 = When NCF is zero (i.e. CF is positive), then this total will be represented as zero';

comment on column FPP_CONTRIBUTION_FACTOR.PARTICIPANTID is
'Participant ID';

comment on column FPP_CONTRIBUTION_FACTOR.SETTLEMENTS_UNITID is
'The Settlements Unit ID (registered DUID / TNI) Note that this SETTLEMENTS_UNITID is what is sent and used by the Settlements system, and may be different from the FPP_UNITID for non-scheduled loads where the FPP_UNITID may be a descriptive key, whereas what will be sent to Settlements as the SETTLEMENTS_UNITID will be the TNI code.';

alter table FPP_CONTRIBUTION_FACTOR
   add constraint FPP_CONTRIBUTION_FACTOR_PK primary key (INTERVAL_DATETIME, CONSTRAINTID, FPP_UNITID, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: FPP_EST_COST                                          */
/*==============================================================*/
create table FPP_EST_COST (
   INTERVAL_DATETIME    DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   FPP_UNITID           VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(10)            not null,
   BIDTYPE              VARCHAR2(10),
   RELEVANT_REGIONS     VARCHAR2(200),
   FPP                  NUMBER(18,8),
   USED_FCAS            NUMBER(18,8),
   UNUSED_FCAS          NUMBER(18,8),
   PARTICIPANTID        VARCHAR2(20)
)
   tablespace PSSTAB02;

comment on table FPP_EST_COST is
'This report delivers the estimated cost for each FPP unit for each constraint for each 5 minute trading interval';

comment on column FPP_EST_COST.INTERVAL_DATETIME is
'Date and time of the trading interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_EST_COST.CONSTRAINTID is
'Constraint ID (binding constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_EST_COST.FPP_UNITID is
'FPP Unit ID (registered DUID / TNI)';

comment on column FPP_EST_COST.VERSIONNO is
'The version number. In most cases this version will be the FPP run number from the FPP database, however there are some cases (like a new pricing run of the constraint FCAS processor received by the FPP system) where the version number listed here will be the financial estimate run number from the FPP database (this number is a different sequence from the FPP run number because there is no recalculation of performance or contribution, just changes to pricing / p regulation hence only the financial estimation is performed).';

comment on column FPP_EST_COST.BIDTYPE is
'Bid type (the bid type saved in relation to constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_EST_COST.RELEVANT_REGIONS is
'Relevant regions (a comma separated list of the relevant regions for the constraint from FCAS data)';

comment on column FPP_EST_COST.FPP is
'FPP in AUD (the financial estimate of frequency performance payment calculated for the constraint / bid type / unit). This value can be either positive (credit) or negative (debit). For details on the calculation, please see FPP procedure and supporting documentation.';

comment on column FPP_EST_COST.USED_FCAS is
'Used recovery FCAS in AUD (the financial estimate of the recovery of used FCAS calculated for the constraint / bid type / unit). This value will be either 0 (nil), or a negative value (debit) only. For details on the calculation, please see FPP procedure and supporting documentation.';

comment on column FPP_EST_COST.UNUSED_FCAS is
'Unused recovery FCAS in AUD (the financial estimate of the recovery of unused FCAS calculated for the constraint / bid type / unit). This value will be either 0 (nil), or a negative value (debit) only. For details on the calculation, please see FPP procedure and supporting documentation.';

comment on column FPP_EST_COST.PARTICIPANTID is
'Participant ID';

alter table FPP_EST_COST
   add constraint FPP_EST_COST_PK primary key (INTERVAL_DATETIME, CONSTRAINTID, FPP_UNITID, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: FPP_EST_PERF_COST_RATE                                */
/*==============================================================*/
create table FPP_EST_PERF_COST_RATE (
   INTERVAL_DATETIME    DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(10)            not null,
   BIDTYPE              VARCHAR2(10),
   RELEVANT_REGIONS     VARCHAR2(200),
   FPP_PAYMENT_RATE     NUMBER(18,8),
   FPP_RECOVERY_RATE    NUMBER(18,8),
   USED_FCAS_RATE       NUMBER(18,8),
   UNUSED_FCAS_RATE     NUMBER(18,8)
)
   tablespace PSSTAB02;

comment on table FPP_EST_PERF_COST_RATE is
'This report delivers the estimated performance cost rate for each constraint for each 5 minute trading interval';

comment on column FPP_EST_PERF_COST_RATE.INTERVAL_DATETIME is
'Date and time of the trading interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_EST_PERF_COST_RATE.CONSTRAINTID is
'Constraint ID (binding constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_EST_PERF_COST_RATE.VERSIONNO is
'The version number. In most cases this version will be the FPP run number from the FPP database, however there are some cases (like a new pricing run of the constraint FCAS processor received by the FPP system) where the version number listed here will be the financial estimate run number from the FPP database (this number is a different sequence from the FPP run number because there is no recalculation of performance or contribution, just changes to pricing / p regulation hence only the financial estimation is performed).';

comment on column FPP_EST_PERF_COST_RATE.BIDTYPE is
'Bid type (the bid type saved in relation to constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_EST_PERF_COST_RATE.RELEVANT_REGIONS is
'Relevant regions (a comma separated list of the relevant regions for the constraint from FCAS data)';

comment on column FPP_EST_PERF_COST_RATE.FPP_PAYMENT_RATE is
'The payment rate for FPP in AUD / MWHz (the denominator used is the sum of positive performance for the constraint calculated by contribution factor calculation). This value will be either 0 (nil), or a positive value (credit) only. For details on the calculation, please see FPP procedure and supporting documentation.';

comment on column FPP_EST_PERF_COST_RATE.FPP_RECOVERY_RATE is
'The recovery rate for FPP in AUD / MWHz (the denominator used is the absolute sum of negative performance for the constraint calculated by the contribution factor calculation). This value will be either 0 (nil), or a negative value (debit) only. For details on the calculation, please see FPP procedure and supporting documentation.';

comment on column FPP_EST_PERF_COST_RATE.USED_FCAS_RATE is
'The rate for used FCAS in AUD / MWHz (the denominator used is the absolute sum of negative performance for the constraint calculated by the negative contribution factor calculation). This value will be either 0 (nil), or a negative value (debit) only. For details on the calculation, please see FPP procedure and supporting documentation.';

comment on column FPP_EST_PERF_COST_RATE.UNUSED_FCAS_RATE is
'The rate for unused FCAS in AUD / MWHz (the denominator used is the absolute sum of negative performance for the constraint calculated by the default contribution factor calculation). This value will be either 0 (nil), or a negative value (debit) only. For details on the calculation, please see FPP procedure and supporting documentation.';

alter table FPP_EST_PERF_COST_RATE
   add constraint FPP_EST_PERF_COST_RATE_PK primary key (INTERVAL_DATETIME, CONSTRAINTID, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: FPP_EST_RESIDUAL_COST_RATE                            */
/*==============================================================*/
create table FPP_EST_RESIDUAL_COST_RATE (
   INTERVAL_DATETIME    DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(10)            not null,
   BIDTYPE              VARCHAR2(10),
   RELEVANT_REGIONS     VARCHAR2(200),
   FPP                  NUMBER(18,8),
   USED_FCAS            NUMBER(18,8),
   UNUSED_FCAS          NUMBER(18,8)
)
   tablespace PSSTAB02;

comment on table FPP_EST_RESIDUAL_COST_RATE is
'This report delivers the estimated residual cost rate for each constraint for each 5 minute trading interval';

comment on column FPP_EST_RESIDUAL_COST_RATE.INTERVAL_DATETIME is
'Date and time of the trading interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_EST_RESIDUAL_COST_RATE.CONSTRAINTID is
'Constraint ID (binding constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_EST_RESIDUAL_COST_RATE.VERSIONNO is
'The version number. In most cases this version will be the FPP run number from the FPP database, however there are some cases (like a new pricing run of the constraint FCAS processor received by the FPP system) where the version number listed here will be the financial estimate run number from the FPP database (this number is a different sequence from the FPP run number because there is no recalculation of performance or contribution, just changes to pricing / p regulation hence only the financial estimation is performed).';

comment on column FPP_EST_RESIDUAL_COST_RATE.BIDTYPE is
'Bid type (the bid type saved in relation to constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_EST_RESIDUAL_COST_RATE.RELEVANT_REGIONS is
'Relevant regions (a comma separated list of the relevant regions for the constraint from FCAS data)';

comment on column FPP_EST_RESIDUAL_COST_RATE.FPP is
'FPP in AUD/MWh (the financial estimate of frequency performance payment calculated). This value can be either positive (credit) or negative (debit). For details on the calculation, please see FPP procedure and supporting documentation.';

comment on column FPP_EST_RESIDUAL_COST_RATE.USED_FCAS is
'Used recovery FCAS in AUD/MWh (the financial estimate of the recovery of used FCAS calculated). This value will be either 0 (nil), or a negative value (debit) only. For details on the calculation, please see FPP procedure and supporting documentation.';

comment on column FPP_EST_RESIDUAL_COST_RATE.UNUSED_FCAS is
'Unused recovery FCAS in AUD/MWh (the financial estimate of the recovery of unused FCAS calculated). This value will be either 0 (nil), or a negative value (debit) only. For details on the calculation, please see FPP procedure and supporting documentation.';

alter table FPP_EST_RESIDUAL_COST_RATE
   add constraint FPP_EST_RESIDUAL_COST_RATE_PK primary key (INTERVAL_DATETIME, CONSTRAINTID, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: FPP_FCAS_SUMMARY                                      */
/*==============================================================*/
create table FPP_FCAS_SUMMARY (
   RUN_DATETIME         DATE                  not null,
   RUNNO                NUMBER(5)             not null,
   INTERVAL_DATETIME    DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(10)            not null,
   BIDTYPE              VARCHAR2(10),
   RELEVANT_REGIONS     VARCHAR2(200),
   REGULATION_MW        NUMBER(18,8),
   CONSTRAINT_MARGINAL_VALUE NUMBER(18,8),
   P_REGULATION         NUMBER(18,8),
   BASE_COST            NUMBER(18,8),
   TSFCAS               NUMBER(18,8),
   TOTAL_FPP            NUMBER(18,8),
   RCR                  NUMBER(18,5),
   USAGE_VALUE          NUMBER(18,8)
)
   tablespace PSSTAB02;

comment on table FPP_FCAS_SUMMARY is
'This report delivers a summary of FCAS requirements as used by the FPP calculation (i.e. only RAISEREG / LOWERREG bid types)';

comment on column FPP_FCAS_SUMMARY.RUN_DATETIME is
'The run date and time of the dispatch case that was the trigger for the constraint FCAS processor execution';

comment on column FPP_FCAS_SUMMARY.RUNNO is
'The dispatch case run number that was the trigger for the constraint FCAS processor execution';

comment on column FPP_FCAS_SUMMARY.INTERVAL_DATETIME is
'Date and time of the trading interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_FCAS_SUMMARY.CONSTRAINTID is
'Constraint ID (binding constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_FCAS_SUMMARY.VERSIONNO is
'The version number. In most cases this version will be the FPP run number from the FPP database, however there are some cases (like a new pricing run of the constraint FCAS processor received by the FPP system) where the version number listed here will be the financial estimate run number from the FPP database (this number is a different sequence from the FPP run number because there is no recalculation of performance or contribution, just changes to pricing / p regulation hence only the financial estimation is performed).';

comment on column FPP_FCAS_SUMMARY.BIDTYPE is
'Bid type (the bid type saved in relation to constraint ID from FCAS data used in FPP calculations, i.e. RAISEREG or LOWERREG)';

comment on column FPP_FCAS_SUMMARY.RELEVANT_REGIONS is
'Relevant regions (a comma separated list of the relevant regions for the constraint from FCAS data)';

comment on column FPP_FCAS_SUMMARY.REGULATION_MW is
'Enabled regulation MW used in the FPP calculation (from FPP database)';

comment on column FPP_FCAS_SUMMARY.CONSTRAINT_MARGINAL_VALUE is
'Marginal value in AUD/MW per hour related to the constraint (from FCAS data used for FPP calculations)';

comment on column FPP_FCAS_SUMMARY.P_REGULATION is
'P regulation value in AUD/MW per hour related to the constraint (from FCAS data used for FPP calculations also known as adjusted marginal value)';

comment on column FPP_FCAS_SUMMARY.BASE_COST is
'Base cost in AUD related to the constraint (from FCAS data used for FPP calculations)';

comment on column FPP_FCAS_SUMMARY.TSFCAS is
'TSFCAS in AUD related to the constraint (FCAS recovery amount related to the constraint also known as adjusted cost)';

comment on column FPP_FCAS_SUMMARY.TOTAL_FPP is
'Total amount of FPP in AUD changing hands related to the constraint (note that this is not the sum of FPP)';

comment on column FPP_FCAS_SUMMARY.RCR is
'RCR MW (the calculated requirement for corrective response from FPP database). Note that this is a join back to the FPP_RCR table.';

comment on column FPP_FCAS_SUMMARY.USAGE_VALUE is
'Usage (calculation of the proportion of regulation FCAS that was calculated to be used). Note that this is a join back to the FPP_USAGE table.';

alter table FPP_FCAS_SUMMARY
   add constraint FPP_FCAS_SUMMARY_PK primary key (RUN_DATETIME, RUNNO, INTERVAL_DATETIME, CONSTRAINTID, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: FPP_FORECAST_DEFAULT_CF                               */
/*==============================================================*/
create table FPP_FORECAST_DEFAULT_CF (
   FPP_UNITID           VARCHAR2(20)          not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   EFFECTIVE_START_DATETIME DATE                  not null,
   EFFECTIVE_END_DATETIME DATE                  not null,
   VERSIONNO            NUMBER(10)            not null,
   BIDTYPE              VARCHAR2(10),
   REGIONID             VARCHAR2(20),
   DEFAULT_CONTRIBUTION_FACTOR NUMBER(18,8),
   DCF_REASON_FLAG      NUMBER(5),
   DCF_ABS_NEGATIVE_PERF_TOTAL NUMBER(18,8),
   SETTLEMENTS_UNITID   VARCHAR2(50)
)
   tablespace PSSTAB02;

comment on table FPP_FORECAST_DEFAULT_CF is
'This report delivers the forecast default contribution factors (DCF) effective for a billing period (aligned to the settlement week)';

comment on column FPP_FORECAST_DEFAULT_CF.FPP_UNITID is
'FPP Unit ID (registered DUID/ TNI)';

comment on column FPP_FORECAST_DEFAULT_CF.CONSTRAINTID is
'Constraint ID (binding constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_FORECAST_DEFAULT_CF.EFFECTIVE_START_DATETIME is
'Effective period start date and time (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time) of the effective period for this default contribution factor related to the combination of FPP unit ID / constraint. This is the billing period over which these default contribution factors will be effective / applied. In most cases this will align to the settlement week, however there are some cases (like a new constraint) where the effective start date will be prorated to align with the change.';

comment on column FPP_FORECAST_DEFAULT_CF.EFFECTIVE_END_DATETIME is
'Effective period end date and time (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time) of the effective period for this default contribution factor related to the combination of FPP unit ID / constraint. This is the billing period over which these default contribution factors will be effective / applied. Effective end date will align with the end of a settlement week.';

comment on column FPP_FORECAST_DEFAULT_CF.VERSIONNO is
'The version number. In most cases this version will be the historical performance calculation run number from the FPP database (which is different from the FPP run number), however there are some cases (like a new constraint) where the version number listed here will be the FPP run number from the FPP database (this will be where the effective start date time will be prorated to align with the detection of this change).';

comment on column FPP_FORECAST_DEFAULT_CF.BIDTYPE is
'Bid type (the bid type saved in relation to constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_FORECAST_DEFAULT_CF.REGIONID is
'Region ID of the frequency deviation / frequency measure';

comment on column FPP_FORECAST_DEFAULT_CF.DEFAULT_CONTRIBUTION_FACTOR is
'Calculated default contribution factor from the historical performance period. For further details please see the FPP procedure document.';

comment on column FPP_FORECAST_DEFAULT_CF.DCF_REASON_FLAG is
'The reason flag showing the decision matrix for the default contribution factor (DCF)';

comment on column FPP_FORECAST_DEFAULT_CF.DCF_ABS_NEGATIVE_PERF_TOTAL is
'The sum of absolute negative performance in MWHz for the combination of constraint (raise or lower). This is used as the denominator in normalising default contribution factors (DCF) as the historical performance is always negative for DCF. For further details please see the FPP procedure document. >0 = Performance was calculated for the units NULL = Performance for the units was unavailable';

comment on column FPP_FORECAST_DEFAULT_CF.SETTLEMENTS_UNITID is
'The Settlements Unit ID (registered DUID / TNI) Note that this SETTLEMENTS_UNITID is what is sent and used by the Settlements system, and may be different from the FPP_UNITID for non-scheduled loads where the FPP_UNITID may be a descriptive key, whereas what will be sent to Settlements as the SETTLEMENTS_UNITID will be the TNI code.';

alter table FPP_FORECAST_DEFAULT_CF
   add constraint FPP_FORECAST_DEFAULT_CF_PK primary key (FPP_UNITID, CONSTRAINTID, EFFECTIVE_START_DATETIME, EFFECTIVE_END_DATETIME, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: FPP_FORECAST_RESIDUAL_DCF                             */
/*==============================================================*/
create table FPP_FORECAST_RESIDUAL_DCF (
   CONSTRAINTID         VARCHAR2(20)          not null,
   EFFECTIVE_START_DATETIME DATE                  not null,
   EFFECTIVE_END_DATETIME DATE                  not null,
   VERSIONNO            NUMBER(10)            not null,
   BIDTYPE              VARCHAR2(10),
   RESIDUAL_DCF         NUMBER(18,8),
   RESIDUAL_DCF_REASON_FLAG NUMBER(5),
   DCF_ABS_NEGATIVE_PERF_TOTAL NUMBER(18,8)
)
   tablespace PSSTAB02;

comment on table FPP_FORECAST_RESIDUAL_DCF is
'This report delivers the forecast residual default contribution factors (DCF) effective for a billing period (aligned to the settlement week)';

comment on column FPP_FORECAST_RESIDUAL_DCF.CONSTRAINTID is
'Constraint ID (binding constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_FORECAST_RESIDUAL_DCF.EFFECTIVE_START_DATETIME is
'Effective period start date and time (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time) of the effective period for this residual default contribution factor related to the constraint. This is the billing period over which these default contribution factors will be effective / applied. In most cases this will align to the settlement week, however there are some cases (like a new constraint) where the effective start date will be prorated to align with the change.';

comment on column FPP_FORECAST_RESIDUAL_DCF.EFFECTIVE_END_DATETIME is
'Effective period end date and time (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time) of the effective period for this residual default contribution factor related to the constraint. This is the billing period over which these default contribution factors will be effective / applied. Effective end date will align with the end of a settlement week.';

comment on column FPP_FORECAST_RESIDUAL_DCF.VERSIONNO is
'The version number. In most cases this version will be the historical performance calculation run number from the FPP database (which is different from the FPP run number), however there are some cases (like a new constraint) where the version number listed here will be the FPP run number from the FPP database (this will be where the effective start date time will be prorated to align with the detection of this change).';

comment on column FPP_FORECAST_RESIDUAL_DCF.BIDTYPE is
'Bid type (the bid type saved in relation to constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_FORECAST_RESIDUAL_DCF.RESIDUAL_DCF is
'Calculated residual default contribution factor from the historical performance period. For further details please see the FPP procedure document.';

comment on column FPP_FORECAST_RESIDUAL_DCF.RESIDUAL_DCF_REASON_FLAG is
'The reason flag showing the decision matrix for the residual default contribution factor (DCF)';

comment on column FPP_FORECAST_RESIDUAL_DCF.DCF_ABS_NEGATIVE_PERF_TOTAL is
'The sum of absolute negative performance in MWHz for the combination of constraint (raise or lower). This is used as the denominator in normalising default contribution factors (DCF). For further details please see the FPP procedure document. >0 = Performance was calculated for the units NULL = Performance for the units was unavailable';

alter table FPP_FORECAST_RESIDUAL_DCF
   add constraint FPP_FORECAST_RESIDUAL_DCF_PK primary key (CONSTRAINTID, EFFECTIVE_START_DATETIME, EFFECTIVE_END_DATETIME, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: FPP_HIST_PERFORMANCE                                  */
/*==============================================================*/
create table FPP_HIST_PERFORMANCE (
   FPP_UNITID           VARCHAR2(20)          not null,
   EFFECTIVE_START_DATETIME DATE                  not null,
   EFFECTIVE_END_DATETIME DATE                  not null,
   VERSIONNO            NUMBER(10)            not null,
   HIST_PERIOD_START_DATETIME DATE,
   HIST_PERIOD_END_DATETIME DATE,
   REG_HIST_RAISE_PERFORMANCE NUMBER(18,5),
   REG_HIST_LOWER_PERFORMANCE NUMBER(18,5),
   FPP_HIST_RAISE_PERFORMANCE NUMBER(18,5),
   FPP_HIST_LOWER_PERFORMANCE NUMBER(18,5)
)
   tablespace PSSTAB02;

comment on table FPP_HIST_PERFORMANCE is
'This report delivers the historical performance calculated based on a historical period and effective for a billing period (aligned to the settlement week)';

comment on column FPP_HIST_PERFORMANCE.FPP_UNITID is
'FPP Unit ID (registered DUID/ TNI)';

comment on column FPP_HIST_PERFORMANCE.EFFECTIVE_START_DATETIME is
'Effective period start date and time (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time) of the effective period for this historical performance values related to the FPP unit ID. This is the billing period over which these historical performance values will be effective / applied over. This will align to the settlement week.';

comment on column FPP_HIST_PERFORMANCE.EFFECTIVE_END_DATETIME is
'Effective period end date and time (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time) of the effective period for this historical performance values related to the FPP unit ID. This is the billing period over which these historical performance values will be effective / applied over. This will align to the settlement week.';

comment on column FPP_HIST_PERFORMANCE.VERSIONNO is
'Version (FPP historical performance calculation run number from the FPP database) Note that due to the these historical calculations, the version numbers listed here are different to the normal FPP run number version for trading interval calculations.';

comment on column FPP_HIST_PERFORMANCE.HIST_PERIOD_START_DATETIME is
'Historical period start date and time (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time) of the historical period for this historical performance calculation related to the FPP unit ID. This is the historical period of trading intervals that feed into the historical performance calculation. This will align to the settlement week.';

comment on column FPP_HIST_PERFORMANCE.HIST_PERIOD_END_DATETIME is
'Historical period end date and time (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time) of the historical period for this historical performance calculation related to the FPP unit ID. This is the historical period of trading intervals that feed into the historical performance calculation. This will align to the settlement week.';

comment on column FPP_HIST_PERFORMANCE.REG_HIST_RAISE_PERFORMANCE is
'Calculated regulation historical raise performance from the historical performance period (substitute raise performance when live performance is unavailable and default raise performance used for default contribution factor calculation) - please see the NER and FPP procedure documents for further information';

comment on column FPP_HIST_PERFORMANCE.REG_HIST_LOWER_PERFORMANCE is
'Calculated regulation historical lower performance from the historical performance period (substitute lower performance when live performance is unavailable and default lower performance used for default contribution factor calculation) - please see the NER and FPP procedure documents for further information';

comment on column FPP_HIST_PERFORMANCE.FPP_HIST_RAISE_PERFORMANCE is
'Calculated FPP historical raise performance from the historical performance period (substitute raise performance calculated used for negative contribution factor calculation when live performance is unavailable) - please see the NER and FPP procedure documents for further information';

comment on column FPP_HIST_PERFORMANCE.FPP_HIST_LOWER_PERFORMANCE is
'Calculated FPP historical lower performance from the historical performance period (substitute lower performance calculated used for negative contribution factor calculation when live performance is unavailable) - please see the NER and FPP procedure documents for further information';

alter table FPP_HIST_PERFORMANCE
   add constraint FPP_HIST_PERFORMANCE_PK primary key (FPP_UNITID, EFFECTIVE_START_DATETIME, EFFECTIVE_END_DATETIME, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: FPP_HIST_REGION_PERFORMANCE                           */
/*==============================================================*/
create table FPP_HIST_REGION_PERFORMANCE (
   REGIONID             VARCHAR2(20)          not null,
   EFFECTIVE_START_DATETIME DATE                  not null,
   EFFECTIVE_END_DATETIME DATE                  not null,
   VERSIONNO            NUMBER(10)            not null,
   HIST_PERIOD_START_DATETIME DATE,
   HIST_PERIOD_END_DATETIME DATE,
   REG_HIST_RAISE_PERFORMANCE NUMBER(18,5),
   REG_HIST_LOWER_PERFORMANCE NUMBER(18,5),
   FPP_HIST_RAISE_PERFORMANCE NUMBER(18,5),
   FPP_HIST_LOWER_PERFORMANCE NUMBER(18,5)
)
   tablespace PSSTAB02;

comment on table FPP_HIST_REGION_PERFORMANCE is
'This report delivers the historical region performance calculated based on a historical period and effective for a billing period (aligned to the settlement week). This calculated historical region performance calculation is an input into the constraint level residual default contribution factor (DCF) which is published in FPP_FORECAST_RESIDUAL_DCF';

comment on column FPP_HIST_REGION_PERFORMANCE.REGIONID is
'Region ID of the historical region performance';

comment on column FPP_HIST_REGION_PERFORMANCE.EFFECTIVE_START_DATETIME is
'Effective period start date and time (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time) of the effective period for this historical performance values related to the region ID. This is the billing period over which these historical performance values will be effective / applied over. This will align to the settlement week.';

comment on column FPP_HIST_REGION_PERFORMANCE.EFFECTIVE_END_DATETIME is
'Effective period end date and time (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time) of the effective period for this historical performance values related to the region ID. This is the billing period over which these historical performance values will be effective / applied over. This will align to the settlement week.';

comment on column FPP_HIST_REGION_PERFORMANCE.VERSIONNO is
'Version (FPP historical performance calculation run number from the FPP database) Note that due to the these historical calculations, the version numbers listed here are different to the normal FPP run number version for trading interval calculations.';

comment on column FPP_HIST_REGION_PERFORMANCE.HIST_PERIOD_START_DATETIME is
'Historical period start date and time (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time) of the historical period for this historical performance calculation related to the region ID. This is the historical period of trading intervals that feed into the historical performance calculation. This will align to the settlement week.';

comment on column FPP_HIST_REGION_PERFORMANCE.HIST_PERIOD_END_DATETIME is
'Historical period end date and time (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time) of the historical period for this historical performance calculation related to the region ID. This is the historical period of trading intervals that feed into the historical performance calculation. This will align to the settlement week.';

comment on column FPP_HIST_REGION_PERFORMANCE.REG_HIST_RAISE_PERFORMANCE is
'Calculated regulation historical raise performance from the historical performance period (substitute raise performance when live performance is unavailable and default raise performance used for default contribution factor calculation) - please see the NER and FPP procedure documents for further information';

comment on column FPP_HIST_REGION_PERFORMANCE.REG_HIST_LOWER_PERFORMANCE is
'Calculated regulation historical lower performance from the historical performance period (substitute lower performance when live performance is unavailable and default lower performance used for default contribution factor calculation) - please see the NER and FPP procedure documents for further information';

comment on column FPP_HIST_REGION_PERFORMANCE.FPP_HIST_RAISE_PERFORMANCE is
'Calculated FPP historical raise performance from the historical performance period (substitute raise performance calculated used for negative contribution factor calculation when live performance is unavailable) - please see the NER and FPP procedure documents for further information';

comment on column FPP_HIST_REGION_PERFORMANCE.FPP_HIST_LOWER_PERFORMANCE is
'Calculated FPP historical lower performance from the historical performance period (substitute lower performance calculated used for negative contribution factor calculation when live performance is unavailable) - please see the NER and FPP procedure documents for further information';

alter table FPP_HIST_REGION_PERFORMANCE
   add constraint FPP_HIST_REGION_PERFORMANCE_PK primary key (REGIONID, EFFECTIVE_START_DATETIME, EFFECTIVE_END_DATETIME, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: FPP_P5_FWD_EST_COST                                   */
/*==============================================================*/
create table FPP_P5_FWD_EST_COST (
   RUN_DATETIME         DATE                  not null,
   RUNNO                NUMBER(5)             not null,
   INTERVAL_DATETIME    DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   FPP_UNITID           VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(5)             not null,
   BIDTYPE              VARCHAR2(10),
   RELEVANT_REGIONS     VARCHAR2(200),
   EST_UNUSED_FCAS      NUMBER(18,8),
   PARTICIPANTID        VARCHAR2(20)
)
   tablespace PSSTAB03;

comment on table FPP_P5_FWD_EST_COST is
'This report delivers the forward estimated unit cost based on P5min runs. These high-level estimates (i.e. assuming that all is unused FCAS) will be provided for each constraint for each 5 minute pre-dispatch interval.';

comment on column FPP_P5_FWD_EST_COST.RUN_DATETIME is
'The run date and time of the 5 minute predispatch case that was the trigger for the constraint FCAS processor execution (as the FCAS requirement data is the basis of these forward estimates)';

comment on column FPP_P5_FWD_EST_COST.RUNNO is
'The 5 minute predispatch case run number that was the trigger for the constraint FCAS processor execution (as the FCAS requirement data is the basis for these forward estimates)';

comment on column FPP_P5_FWD_EST_COST.INTERVAL_DATETIME is
'Date and time of the trading interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_P5_FWD_EST_COST.CONSTRAINTID is
'Constraint ID (binding constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_P5_FWD_EST_COST.FPP_UNITID is
'FPP Unit ID (registered DUID / TNI)';

comment on column FPP_P5_FWD_EST_COST.VERSIONNO is
'The version number of the effective default contribution factor for the unit / constraint combination taken from the FPP_FORECAST_DEFAULT_CF table';

comment on column FPP_P5_FWD_EST_COST.BIDTYPE is
'Bid type (the bid type saved in relation to constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_P5_FWD_EST_COST.RELEVANT_REGIONS is
'Relevant regions (a comma separated list of the relevant regions for the constraint from FCAS data)';

comment on column FPP_P5_FWD_EST_COST.EST_UNUSED_FCAS is
'Estimated unused recovery FCAS in AUD (the forward financial estimate of the recovery of unused FCAS, assuming that all is unused FCAS). This value will be either 0 (nil), or a negative value (debit) only. For details on the calculation, please see FPP procedure and supporting documentation. Please note that there may be cases where there is not yet a default contribution factor for the constraint (for example if there is a new constraint binding in p5min that has not yet bound in dispatch). In these scenarios there will be no EST_UNUSED_FCAS row sent for the constraint, as it is currently incalculable until a corresponding default contribution factor is calculated from a dispatch run with this constraint binding.';

comment on column FPP_P5_FWD_EST_COST.PARTICIPANTID is
'Participant ID';

alter table FPP_P5_FWD_EST_COST
   add constraint FPP_P5_FWD_EST_COST_PK primary key (RUN_DATETIME, RUNNO, INTERVAL_DATETIME, CONSTRAINTID, FPP_UNITID, VERSIONNO)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: FPP_P5_FWD_EST_RESIDUALRATE                           */
/*==============================================================*/
create table FPP_P5_FWD_EST_RESIDUALRATE (
   RUN_DATETIME         DATE                  not null,
   RUNNO                NUMBER(5)             not null,
   INTERVAL_DATETIME    DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(5)             not null,
   BIDTYPE              VARCHAR2(10),
   RELEVANT_REGIONS     VARCHAR2(200),
   EST_UNUSED_FCAS      NUMBER(18,8)
)
   tablespace PSSTAB03;

comment on table FPP_P5_FWD_EST_RESIDUALRATE is
'This report delivers the forward estimated residual cost rate based on P5min runs. These high-level estimates (i.e. assuming that all is unused FCAS) will be provided for each constraint for each 5 minute pre-dispatch interval.';

comment on column FPP_P5_FWD_EST_RESIDUALRATE.RUN_DATETIME is
'The run date and time of the 5 minute predispatch case that was the trigger for the constraint FCAS processor execution (as the FCAS requirement data is the basis of these forward estimates)';

comment on column FPP_P5_FWD_EST_RESIDUALRATE.RUNNO is
'The 5 minute predispatch case run number that was the trigger for the constraint FCAS processor execution (as the FCAS requirement data is the basis for these forward estimates)';

comment on column FPP_P5_FWD_EST_RESIDUALRATE.INTERVAL_DATETIME is
'Date and time of the trading interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_P5_FWD_EST_RESIDUALRATE.CONSTRAINTID is
'Constraint ID (binding constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_P5_FWD_EST_RESIDUALRATE.VERSIONNO is
'The version number of the effective default contribution factor for the unit / constraint combination taken from the FPP_FORECAST_DEFAULT_CF table';

comment on column FPP_P5_FWD_EST_RESIDUALRATE.BIDTYPE is
'Bid type (the bid type saved in relation to constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_P5_FWD_EST_RESIDUALRATE.RELEVANT_REGIONS is
'Relevant regions (a comma separated list of the relevant regions for the constraint from FCAS data)';

comment on column FPP_P5_FWD_EST_RESIDUALRATE.EST_UNUSED_FCAS is
'Estimated unused recovery FCAS in AUD/MWh (the forward financial estimate of the recovery of unused FCAS, assuming that all is unused FCAS). This value will be either 0 (nil), or a negative value (debit) only. For details on the calculation, please see FPP procedure and supporting documentation. Please note that there may be cases where there is not yet a residual default contribution factor for the constraint (for example if there is a new constraint binding in p5min that has not yet bound in dispatch). In these scenarios there will be no EST_UNUSED_FCAS row sent for the constraint, as this is currently incalculable until a corresponding residual default contribution factor is calculated from a dispatch run with this constraint binding.';

alter table FPP_P5_FWD_EST_RESIDUALRATE
   add constraint FPP_P5_FWD_EST_RESIDUALRATE_PK primary key (RUN_DATETIME, RUNNO, INTERVAL_DATETIME, CONSTRAINTID, VERSIONNO)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: FPP_PD_FWD_EST_COST                                   */
/*==============================================================*/
create table FPP_PD_FWD_EST_COST (
   PREDISPATCHSEQNO     VARCHAR2(20)          not null,
   RUN_DATETIME         DATE                  not null,
   RUNNO                NUMBER(5)             not null,
   INTERVAL_DATETIME    DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   FPP_UNITID           VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(5)             not null,
   BIDTYPE              VARCHAR2(10),
   RELEVANT_REGIONS     VARCHAR2(200),
   EST_UNUSED_FCAS      NUMBER(18,8),
   PARTICIPANTID        VARCHAR2(20)
)
   tablespace PSSTAB02;

comment on table FPP_PD_FWD_EST_COST is
'This report delivers the forward estimated unit cost based on PREDISPATCH runs. These high-level estimates (i.e. assuming that all is unused FCAS) will be provided for each constraint for each 30 minute pre-dispatch interval.';

comment on column FPP_PD_FWD_EST_COST.PREDISPATCHSEQNO is
'Predispatch sequence number for the 30 minute predispatch case that triggers the constraint FCAS processor run';

comment on column FPP_PD_FWD_EST_COST.RUN_DATETIME is
'The run date and time of the 30 minute predispatch case that was the trigger for the constraint FCAS processor execution (as the FCAS requirement data is the basis of these forward estimates)';

comment on column FPP_PD_FWD_EST_COST.RUNNO is
'The 30 minute predispatch case run number that was the trigger for the constraint FCAS processor execution (as the FCAS requirement data is the basis for these forward estimates)';

comment on column FPP_PD_FWD_EST_COST.INTERVAL_DATETIME is
'Date and time of the 30 minute predispatch interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_PD_FWD_EST_COST.CONSTRAINTID is
'Constraint ID (binding constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_PD_FWD_EST_COST.FPP_UNITID is
'FPP Unit ID (registered DUID / TNI)';

comment on column FPP_PD_FWD_EST_COST.VERSIONNO is
'The version number of the effective default contribution factor for the unit / constraint combination taken from the FPP_FORECAST_DEFAULT_CF table';

comment on column FPP_PD_FWD_EST_COST.BIDTYPE is
'Bid type (the bid type saved in relation to constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_PD_FWD_EST_COST.RELEVANT_REGIONS is
'Relevant regions (a comma separated list of the relevant regions for the constraint from FCAS data)';

comment on column FPP_PD_FWD_EST_COST.EST_UNUSED_FCAS is
'Estimated unused recovery FCAS in AUD (the forward financial estimate of the recovery of unused FCAS, assuming that all is unused FCAS). This value will be either 0 (nil), or a negative value (debit) only. For details on the calculation, please see FPP procedure and supporting documentation. Please note that there may be cases where there is not yet a default contribution factor for the constraint (for example if there is a new constraint binding in predispatch that has not yet bound in dispatch). In these scenarios there will be no EST_UNUSED_FCAS row sent for the constraint, as it is currently incalculable until a corresponding default contribution factor is calculated from a dispatch run with this constraint binding.';

comment on column FPP_PD_FWD_EST_COST.PARTICIPANTID is
'Participant ID';

alter table FPP_PD_FWD_EST_COST
   add constraint FPP_PD_FWD_EST_COST_PK primary key (PREDISPATCHSEQNO, RUN_DATETIME, RUNNO, INTERVAL_DATETIME, CONSTRAINTID, FPP_UNITID, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: FPP_PD_FWD_EST_RESIDUALRATE                           */
/*==============================================================*/
create table FPP_PD_FWD_EST_RESIDUALRATE (
   PREDISPATCHSEQNO     VARCHAR2(20)          not null,
   RUN_DATETIME         DATE                  not null,
   RUNNO                NUMBER(5)             not null,
   INTERVAL_DATETIME    DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(5)             not null,
   BIDTYPE              VARCHAR2(10),
   RELEVANT_REGIONS     VARCHAR2(200),
   EST_UNUSED_FCAS      NUMBER(18,8)
)
   tablespace PSSTAB02;

comment on table FPP_PD_FWD_EST_RESIDUALRATE is
'This report delivers the forward estimated residual cost rate based on PREDISPATCH runs. These high-level estimates (i.e. assuming that all is unused FCAS) will be provided for each constraint for each 30 minute pre- dispatch interval.';

comment on column FPP_PD_FWD_EST_RESIDUALRATE.PREDISPATCHSEQNO is
'Predispatch sequence number for the 30 minute predispatch case that triggers the constraint FCAS processor run';

comment on column FPP_PD_FWD_EST_RESIDUALRATE.RUN_DATETIME is
'The run date and time of the 30 minute predispatch case that was the trigger for the constraint FCAS processor execution (as the FCAS requirement data is the basis of these forward estimates)';

comment on column FPP_PD_FWD_EST_RESIDUALRATE.RUNNO is
'The 30 minute predispatch case run number that was the trigger for the constraint FCAS processor execution (as the FCAS requirement data is the basis for these forward estimates)';

comment on column FPP_PD_FWD_EST_RESIDUALRATE.INTERVAL_DATETIME is
'Date and time of the 30 minute predispatch interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_PD_FWD_EST_RESIDUALRATE.CONSTRAINTID is
'Constraint ID (binding constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_PD_FWD_EST_RESIDUALRATE.VERSIONNO is
'The version number of the effective default contribution factor for the unit / constraint combination taken from the FPP_FORECAST_DEFAULT_CF table';

comment on column FPP_PD_FWD_EST_RESIDUALRATE.BIDTYPE is
'Bid type (the bid type saved in relation to constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_PD_FWD_EST_RESIDUALRATE.RELEVANT_REGIONS is
'Relevant regions (a comma separated list of the relevant regions for the constraint from FCAS data)';

comment on column FPP_PD_FWD_EST_RESIDUALRATE.EST_UNUSED_FCAS is
'Estimated unused recovery FCAS in AUD/MWh (the forward financial estimate of the recovery of unused FCAS, assuming that all is unused FCAS). This value will be either 0 (nil), or a negative value (debit) only. For details on the calculation, please see FPP procedure and supporting documentation. Please note that there may be cases where there is not yet a residual default contribution factor for the constraint (for example if there is a new constraint binding in predispatch that has not yet bound in dispatch). In these scenarios there will be no EST_UNUSED_FCAS row sent for the constraint, as this is currently incalculable until a corresponding residual default contribution factor is calculated from a dispatch run with this constraint binding.';

alter table FPP_PD_FWD_EST_RESIDUALRATE
   add constraint FPP_PD_FWD_EST_RESIDUALRATE_PK primary key (PREDISPATCHSEQNO, RUN_DATETIME, RUNNO, INTERVAL_DATETIME, CONSTRAINTID, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: FPP_PERFORMANCE                                       */
/*==============================================================*/
create table FPP_PERFORMANCE (
   INTERVAL_DATETIME    DATE                  not null,
   FPP_UNITID           VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(5)             not null,
   RAISE_PERFORMANCE    NUMBER(18,5),
   RAISE_REASON_FLAG    NUMBER(5),
   LOWER_PERFORMANCE    NUMBER(18,5),
   LOWER_REASON_FLAG    NUMBER(5),
   PARTICIPANTID        VARCHAR2(20)
)
   tablespace PSSTAB02;

comment on table FPP_PERFORMANCE is
'This report delivers the calculated performance value for each 5 minute trading interval for each FPP unit';

comment on column FPP_PERFORMANCE.INTERVAL_DATETIME is
'Date and time of the trading interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_PERFORMANCE.FPP_UNITID is
'FPP Unit ID (registered DUID/ TNI)';

comment on column FPP_PERFORMANCE.VERSIONNO is
'Version (FPP run number from the FPP database)';

comment on column FPP_PERFORMANCE.RAISE_PERFORMANCE is
'Raise performance value in MWHz units (calculated by FPP for that trading interval taken from FPP database)';

comment on column FPP_PERFORMANCE.RAISE_REASON_FLAG is
'The reason flag showing the decision matrix for the raise performance value Supported values are: 0 = Performance is calculated based on good input data 1 = Performance is Null as unit is a Non Primary DUID in the group 2 = Performance against the Primary DUID representing the group 4 = Performance is Null as Input data is bad or unavailable 8 = Performance is Null as FM is unreliable 12 = Performance is Null as Input data is bad or unavailable and FM is unreliable 6 = Performance against the Primary DUID representing the group is Null as Input data is bad or unavailable 10 = Performance against the Primary DUID representing the group is Null as FM is unreliable 14 = Performance against the Primary DUID representing the group is Null as Input data is bad or unavailable  and FM is unreliable';

comment on column FPP_PERFORMANCE.LOWER_PERFORMANCE is
'Lower performance value in MWHz units (calculated by FPP for that trading interval taken from FPP database)';

comment on column FPP_PERFORMANCE.LOWER_REASON_FLAG is
'The reason flag showing the decision matrix for the lower performance value Supported values are: 0 = Performance is calculated based on good input data 1 = Performance is Null as unit is a Non Primary DUID in the group 2 = Performance against the Primary DUID representing the group 4 = Performance is Null as Input data is bad or unavailable 8 = Performance is Null as FM is unreliable 12 = Performance is Null as Input data is bad or unavailable and FM is unreliable 6 = Performance against the Primary DUID representing the group is Null as Input data is bad or unavailable 10 = Performance against the Primary DUID representing the group is Null as FM is unreliable 14 = Performance against the Primary DUID representing the group is Null as Input data is bad or unavailable  and FM is unreliable';

comment on column FPP_PERFORMANCE.PARTICIPANTID is
'Participant ID';

alter table FPP_PERFORMANCE
   add constraint FPP_PERFORMANCE_PK primary key (INTERVAL_DATETIME, FPP_UNITID, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: FPP_RCR                                               */
/*==============================================================*/
create table FPP_RCR (
   INTERVAL_DATETIME    DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(5)             not null,
   BIDTYPE              VARCHAR2(10),
   RCR                  NUMBER(18,5),
   RCR_REASON_FLAG      NUMBER(5)
)
   tablespace PSSTAB02;

comment on table FPP_RCR is
'This report delivers the calculated RCR for each constraint for each 5 minute trading interval';

comment on column FPP_RCR.INTERVAL_DATETIME is
'Date and time of the trading interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_RCR.CONSTRAINTID is
'Constraint ID (binding constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_RCR.VERSIONNO is
'Version (FPP run number from the FPP database)';

comment on column FPP_RCR.BIDTYPE is
'Bid type (the bid type saved in relation to constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_RCR.RCR is
'RCR (the calculated requirement for corrective response from FPP database)';

comment on column FPP_RCR.RCR_REASON_FLAG is
'The reason flag showing the decision matrix for the requirement for corrective response (RCR) calculation Supported values are: 0 = RCR is calculated based on good input data 1 = RCR is 0 as FM is unreliable 2 = RCR is 0 as the percentage of units with unavailable input or bad data is greater than the threshold percentage';

alter table FPP_RCR
   add constraint FPP_RCR_PK primary key (INTERVAL_DATETIME, CONSTRAINTID, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: FPP_REGION_FREQ_MEASURE                               */
/*==============================================================*/
create table FPP_REGION_FREQ_MEASURE (
   INTERVAL_DATETIME    DATE                  not null,
   MEASUREMENT_DATETIME DATE                  not null,
   REGIONID             VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(5)             not null,
   FREQ_DEVIATION_HZ    NUMBER(18,8),
   HZ_QUALITY_FLAG      NUMBER(5),
   FREQ_MEASURE_HZ      NUMBER(18,8),
   FM_ALIGNMENT_FLAG    NUMBER(5)
)
   tablespace PSSTAB02;

comment on table FPP_REGION_FREQ_MEASURE is
'This report delivers the curated 4 second frequency deviation and frequency measure data for each region';

comment on column FPP_REGION_FREQ_MEASURE.INTERVAL_DATETIME is
'Date and time of the trading interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_REGION_FREQ_MEASURE.MEASUREMENT_DATETIME is
'Date and time of the SCADA data (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_REGION_FREQ_MEASURE.REGIONID is
'Region ID of the frequency deviation / frequency measure';

comment on column FPP_REGION_FREQ_MEASURE.VERSIONNO is
'Version (FPP run number from the FPP database)';

comment on column FPP_REGION_FREQ_MEASURE.FREQ_DEVIATION_HZ is
'Frequency Deviation (4 second frequency deviation in Hz for that region)';

comment on column FPP_REGION_FREQ_MEASURE.HZ_QUALITY_FLAG is
'Frequency Quality (4 second frequency deviation quality for that region) Supported values are: 0 = Bad Quality 1 = Good Quality 2 = Suspect Quality -1 = Value not used as input in the FPP calculation (generally used to indicate when SCADA data was late arriving after the FPP calculation commenced)';

comment on column FPP_REGION_FREQ_MEASURE.FREQ_MEASURE_HZ is
'Calculated 4 second Frequency Measure for that region from FPP database';

comment on column FPP_REGION_FREQ_MEASURE.FM_ALIGNMENT_FLAG is
'Alignment Flag (4 second frequency deviation is aligned with 4 second frequency measure for that region) Supported values are: 0 = Misaligned 1 = Aligned';

alter table FPP_REGION_FREQ_MEASURE
   add constraint FPP_REGION_FREQ_MEASURE_PK primary key (INTERVAL_DATETIME, MEASUREMENT_DATETIME, REGIONID, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: FPP_RESIDUAL_CF                                       */
/*==============================================================*/
create table FPP_RESIDUAL_CF (
   INTERVAL_DATETIME    DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(5)             not null,
   BIDTYPE              VARCHAR2(10),
   RESIDUAL_CF          NUMBER(18,8),
   NEGATIVE_RESIDUAL_CF NUMBER(18,8),
   RESIDUAL_DCF         NUMBER(18,8),
   RESIDUAL_CF_REASON_FLAG NUMBER(5),
   CF_ABS_POSITIVE_PERF_TOTAL NUMBER(18,8),
   CF_ABS_NEGATIVE_PERF_TOTAL NUMBER(18,8),
   NCF_ABS_NEGATIVE_PERF_TOTAL NUMBER(18,8)
)
   tablespace PSSTAB02;

comment on table FPP_RESIDUAL_CF is
'This report delivers the calculated residual contribution factor value for each 5 minute trading interval for each constraint';

comment on column FPP_RESIDUAL_CF.INTERVAL_DATETIME is
'Date and time of the trading interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_RESIDUAL_CF.CONSTRAINTID is
'Constraint ID (binding constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_RESIDUAL_CF.VERSIONNO is
'Version (FPP run number from the FPP database)';

comment on column FPP_RESIDUAL_CF.BIDTYPE is
'Bid type (the bid type saved in relation to constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_RESIDUAL_CF.RESIDUAL_CF is
'Residual Contribution Factor (the calculated residual contribution factor for the constraint ID for that trading interval) - for further details please see the FPP procedure document';

comment on column FPP_RESIDUAL_CF.NEGATIVE_RESIDUAL_CF is
'Negative Residual Contribution Factor (the calculated negative residual contribution factor for the constraint ID for that trading interval) - for further details please see the FPP procedure document';

comment on column FPP_RESIDUAL_CF.RESIDUAL_DCF is
'The Residual Default Contribution Factor (the calculated residual default contribution factor based on historical performance for the constraint ID for that trading interval) that is effective for this trading interval, which joins back to FPP_FORECAST_RESIDUAL_DCF - for further details please see the FPP procedure document';

comment on column FPP_RESIDUAL_CF.RESIDUAL_CF_REASON_FLAG is
'The reason flag showing the decision matrix for the residual contribution factor (CF) Supported values are: 0 = CF is calculated based on good input data 8 = CF is 0 because FM is unreliable. 16 = CF is 0 because more than 50 percent input is bad or not available.';

comment on column FPP_RESIDUAL_CF.CF_ABS_POSITIVE_PERF_TOTAL is
'The sum of absolute positive performance in MWHz for the combination of constraint / bid type (raise or lower). This is used as the denominator in normalising contribution factors (CF) where a units performance is positive. For further details please see the FPP procedure document. >0 = Performance was calculated for the units NULL = Performance for the units was unavailable';

comment on column FPP_RESIDUAL_CF.CF_ABS_NEGATIVE_PERF_TOTAL is
'The sum of absolute negative performance in MWHz for the combination of constraint / bid type (raise or lower). This is used as the denominator in normalising contribution factors (CF) where a units performance is negative. For further details please see the FPP procedure document. >0 = Performance was calculated for the units NULL = Performance for the units was unavailable';

comment on column FPP_RESIDUAL_CF.NCF_ABS_NEGATIVE_PERF_TOTAL is
'The sum of absolute negative performance in MWHz for the combination of constraint / bid type (raise or lower). This is used as the denominator in normalising negative contribution factors (NCF). For further details please see the FPP procedure document. >0 = Performance was calculated for the units NULL = Performance for the units was unavailable 0 = When NCF is zero (i.e. CF is positive), then this total will be represented as zero';

alter table FPP_RESIDUAL_CF
   add constraint FPP_RESIDUAL_CF_PK primary key (INTERVAL_DATETIME, CONSTRAINTID, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: FPP_RESIDUAL_PERFORMANCE                              */
/*==============================================================*/
create table FPP_RESIDUAL_PERFORMANCE (
   INTERVAL_DATETIME    DATE                  not null,
   REGIONID             VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(5)             not null,
   RAISE_PERFORMANCE    NUMBER(18,5),
   RAISE_REASON_FLAG    NUMBER(5),
   LOWER_PERFORMANCE    NUMBER(18,5),
   LOWER_REASON_FLAG    NUMBER(5)
)
   tablespace PSSTAB01;

comment on table FPP_RESIDUAL_PERFORMANCE is
'This report delivers the calculated residual performance value for each 5 minute trading interval';

comment on column FPP_RESIDUAL_PERFORMANCE.INTERVAL_DATETIME is
'Date and time of the trading interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_RESIDUAL_PERFORMANCE.REGIONID is
'Region ID of the residual performance';

comment on column FPP_RESIDUAL_PERFORMANCE.VERSIONNO is
'Version (FPP run number from the FPP database)';

comment on column FPP_RESIDUAL_PERFORMANCE.RAISE_PERFORMANCE is
'Raise performance value in MWHz units (calculated by FPP for that trading interval taken from FPP database)';

comment on column FPP_RESIDUAL_PERFORMANCE.RAISE_REASON_FLAG is
'The reason flag showing the decision matrix for the raise performance value Supported values are: 0 = Performance is calculated based on good input data 4 = Performance is Null as Input data is bad or unavailable 8 = Performance is Null as FM is unreliable 12 = Performance is Null as Input data is bad or unavailable and FM is unreliable';

comment on column FPP_RESIDUAL_PERFORMANCE.LOWER_PERFORMANCE is
'Lower performance value in MWHz units (calculated by FPP for that trading interval taken from FPP database)';

comment on column FPP_RESIDUAL_PERFORMANCE.LOWER_REASON_FLAG is
'The reason flag showing the decision matrix for the lower performance value Supported values are: 0 = Performance is calculated based on good input data 4 = Performance is Null as Input data is bad or unavailable 8 = Performance is Null as FM is unreliable 12 = Performance is Null as Input data is bad or unavailable and FM is unreliable';

alter table FPP_RESIDUAL_PERFORMANCE
   add constraint FPP_RESIDUAL_PERFORMANCE_PK primary key (INTERVAL_DATETIME, REGIONID, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: FPP_RUN                                               */
/*==============================================================*/
create table FPP_RUN (
   INTERVAL_DATETIME    DATE                  not null,
   VERSIONNO            NUMBER(5)             not null,
   FPPRUN_DATETIME      DATE,
   RUN_STATUS           VARCHAR2(20),
   AUTHORISED_DATETIME  DATE
)
   tablespace PSSTAB01;

comment on table FPP_RUN is
'This report delivers details of the 5-minute FPP calculation engine success failure outcome saved in FPP database';

comment on column FPP_RUN.INTERVAL_DATETIME is
'Date and time of the trading interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_RUN.VERSIONNO is
'Version (FPP run number from the FPP database)';

comment on column FPP_RUN.FPPRUN_DATETIME is
'Completion time of the FPP calculation run (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_RUN.RUN_STATUS is
'The status of FPP_RUN at the time the data model extract report was run. Typical value is Completed';

comment on column FPP_RUN.AUTHORISED_DATETIME is
'Date and time of the authorisation of this FPP calculation run (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time) - Note may be delayed in cases of ex-post calculation runs';

alter table FPP_RUN
   add constraint FPP_RUN_PK primary key (INTERVAL_DATETIME, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: FPP_UNIT_MW                                           */
/*==============================================================*/
create table FPP_UNIT_MW (
   INTERVAL_DATETIME    DATE                  not null,
   MEASUREMENT_DATETIME DATE                  not null,
   FPP_UNITID           VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(5)             not null,
   MEASURED_MW          NUMBER(18,8),
   MW_QUALITY_FLAG      NUMBER(5),
   SCHEDULED_MW         NUMBER(18,5),
   DEVIATION_MW         NUMBER(18,5),
   PARTICIPANTID        VARCHAR2(20)
)
   tablespace PSSTAB03;

comment on table FPP_UNIT_MW is
'This report delivers the curated 4 second measurement MW data for each FPP unit';

comment on column FPP_UNIT_MW.INTERVAL_DATETIME is
'Date and time of the trading interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_UNIT_MW.MEASUREMENT_DATETIME is
'Date and time of the SCADA data (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_UNIT_MW.FPP_UNITID is
'FPP Unit ID (registered DUID/TNI)';

comment on column FPP_UNIT_MW.VERSIONNO is
'Version (FPP run number from the FPP database)';

comment on column FPP_UNIT_MW.MEASURED_MW is
'Measured MW (4 second SCADA measurement in MW)';

comment on column FPP_UNIT_MW.MW_QUALITY_FLAG is
'MW Quality (4 second SCADA measurement Quality) Supported values are: 0 = Bad Quality 1 = Good Quality 2 = Suspect Quality -1 = Value not used as input in the FPP calculation (generally used to indicate when SCADA data was late arriving after the FPP calculation commenced)';

comment on column FPP_UNIT_MW.SCHEDULED_MW is
'Scheduled MW (reference trajectory value from FPP calculation process)';

comment on column FPP_UNIT_MW.DEVIATION_MW is
'Unit Deviation (output of the FPP calculation process)';

comment on column FPP_UNIT_MW.PARTICIPANTID is
'Participant ID';

alter table FPP_UNIT_MW
   add constraint FPP_UNIT_MW_PK primary key (INTERVAL_DATETIME, MEASUREMENT_DATETIME, FPP_UNITID, VERSIONNO)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: FPP_USAGE                                             */
/*==============================================================*/
create table FPP_USAGE (
   INTERVAL_DATETIME    DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   VERSIONNO            NUMBER(5)             not null,
   BIDTYPE              VARCHAR2(10),
   REGULATION_MW        NUMBER(18,8),
   USED_MW              NUMBER(18,8),
   USAGE_VALUE          NUMBER(18,8),
   USAGE_REASON_FLAG    NUMBER(5)
)
   tablespace PSSTAB02;

comment on table FPP_USAGE is
'This report delivers the calculated usage for each constraint for each 5 minute trading interval';

comment on column FPP_USAGE.INTERVAL_DATETIME is
'Date and time of the trading interval (DD/MM/YYYY HH24:MI:SS) fixed to the UTC+10 time zone (NEM time)';

comment on column FPP_USAGE.CONSTRAINTID is
'Constraint ID (binding constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_USAGE.VERSIONNO is
'Version (FPP run number from the FPP database)';

comment on column FPP_USAGE.BIDTYPE is
'Bid type (the bid type saved in relation to constraint ID from FCAS data used in FPP calculations)';

comment on column FPP_USAGE.REGULATION_MW is
'Enabled regulation MW used in the FPP calculation (from FPP database)';

comment on column FPP_USAGE.USED_MW is
'Maximum used regulation MW value (quantity of regulation FCAS that was calculated to be used in MW)';

comment on column FPP_USAGE.USAGE_VALUE is
'Usage (calculation of the proportion of regulation FCAS that was calculated to be used)';

comment on column FPP_USAGE.USAGE_REASON_FLAG is
'The reason flag showing the decision matrix for the usage calculation Supported values are: 0 = Usage is calculated based on good input data 1 = Usage is 0 as FM is unreliable 2 = Usage is 0 as the percentage of units with unavailable input or bad data is greater than the threshold percentage';

alter table FPP_USAGE
   add constraint FPP_USAGE_PK primary key (INTERVAL_DATETIME, CONSTRAINTID, VERSIONNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: GDINSTRUCT                                            */
/*==============================================================*/
create table GDINSTRUCT (
   DUID                 VARCHAR2(10),
   STATIONID            VARCHAR2(10),
   REGIONID             VARCHAR2(10),
   ID                   NUMBER(22,0)          not null,
   INSTRUCTIONTYPEID    VARCHAR2(10),
   INSTRUCTIONSUBTYPEID VARCHAR2(10),
   INSTRUCTIONCLASSID   VARCHAR2(10),
   REASON               VARCHAR2(64),
   INSTLEVEL            NUMBER(6,0),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   PARTICIPANTID        VARCHAR2(10),
   ISSUEDTIME           DATE,
   TARGETTIME           DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table GDINSTRUCT is
'GDINSTRUCT shows all manually issued dispatch instructions for a dispatchable unit. Ancillary Service instructions are to enable and to disable (i.e. 2 separate instructions) a service. Non-conforming units are also instructed via this facility. However, this facility is not the same as the market notice.';

comment on column GDINSTRUCT.DUID is
'Dispatchable unit identifier';

comment on column GDINSTRUCT.STATIONID is
'Station Identifier';

comment on column GDINSTRUCT.REGIONID is
'Region Identifier';

comment on column GDINSTRUCT.ID is
'Instruction ID (sequential number)';

comment on column GDINSTRUCT.INSTRUCTIONTYPEID is
'Instruction type';

comment on column GDINSTRUCT.INSTRUCTIONSUBTYPEID is
'Instruction sub type';

comment on column GDINSTRUCT.INSTRUCTIONCLASSID is
'Instruction class';

comment on column GDINSTRUCT.REASON is
'Reason';

comment on column GDINSTRUCT.INSTLEVEL is
'Instruction target level';

comment on column GDINSTRUCT.AUTHORISEDDATE is
'Authorised date';

comment on column GDINSTRUCT.AUTHORISEDBY is
'User authorised by';

comment on column GDINSTRUCT.PARTICIPANTID is
'Unique participant identifier';

comment on column GDINSTRUCT.ISSUEDTIME is
'Date / time issued';

comment on column GDINSTRUCT.TARGETTIME is
'Date / time instruction to apply';

comment on column GDINSTRUCT.LASTCHANGED is
'Last date and time record changed';

alter table GDINSTRUCT
   add constraint GDINSTRUCT_PK primary key (ID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: GDINSTRUCT_LCX                                        */
/*==============================================================*/
create index GDINSTRUCT_LCX on GDINSTRUCT (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: GDINSTRUCT_NDX2                                       */
/*==============================================================*/
create index GDINSTRUCT_NDX2 on GDINSTRUCT (
   DUID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: GDINSTRUCT_NDX3                                       */
/*==============================================================*/
create index GDINSTRUCT_NDX3 on GDINSTRUCT (
   TARGETTIME ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: GENCONDATA                                            */
/*==============================================================*/
create table GENCONDATA (
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   GENCONID             VARCHAR2(20)          not null,
   CONSTRAINTTYPE       VARCHAR2(2),
   CONSTRAINTVALUE      NUMBER(16,6),
   DESCRIPTION          VARCHAR2(256),
   STATUS               VARCHAR2(8),
   GENERICCONSTRAINTWEIGHT NUMBER(16,6),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   DYNAMICRHS           NUMBER(15,5),
   LASTCHANGED          DATE,
   DISPATCH             VARCHAR2(1),
   PREDISPATCH          VARCHAR2(1),
   STPASA               VARCHAR2(1),
   MTPASA               VARCHAR2(1),
   IMPACT               VARCHAR2(64),
   SOURCE               VARCHAR2(128),
   LIMITTYPE            VARCHAR2(64),
   REASON               VARCHAR2(256),
   MODIFICATIONS        VARCHAR2(256),
   ADDITIONALNOTES      VARCHAR2(256),
   P5MIN_SCOPE_OVERRIDE VARCHAR2(2),
   LRC                  VARCHAR2(1),
   LOR                  VARCHAR2(1),
   FORCE_SCADA          NUMBER(1,0)
)
   tablespace PSSTAB01;

comment on table GENCONDATA is
'GENCONDATA sets out the generic constraints contained within a generic constraint set invoked in PASA, predispatch and dispatch.
Fields enable selective application of invoked constraints in the Dispatch, Predispatch, ST PASA or MT PASA processes.';

comment on column GENCONDATA.EFFECTIVEDATE is
'Effective date of this constraint';

comment on column GENCONDATA.VERSIONNO is
'Version with respect to the effective date';

comment on column GENCONDATA.GENCONID is
'Unique ID for the constraint';

comment on column GENCONDATA.CONSTRAINTTYPE is
'The logical operator (=, >=, <=)';

comment on column GENCONDATA.CONSTRAINTVALUE is
'the RHS value used if there is no dynamic RHS defined in GenericConstraintRHS';

comment on column GENCONDATA.DESCRIPTION is
'Detail of the plant that is not in service';

comment on column GENCONDATA.STATUS is
'Not used';

comment on column GENCONDATA.GENERICCONSTRAINTWEIGHT is
'The constraint violation penalty factor';

comment on column GENCONDATA.AUTHORISEDDATE is
'Date record authorised';

comment on column GENCONDATA.AUTHORISEDBY is
'User authorising record';

comment on column GENCONDATA.DYNAMICRHS is
'Not used';

comment on column GENCONDATA.LASTCHANGED is
'Last date and time record changed';

comment on column GENCONDATA.DISPATCH is
'Flag: constraint RHS used for Dispatch? 1-used, 0-not used';

comment on column GENCONDATA.PREDISPATCH is
'Flag to indicate if the constraint RHS is to be used for PreDispatch, 1-used, 0-not used';

comment on column GENCONDATA.STPASA is
'Flag to indicate if the constraint RHS is to be used for ST PASA, 1-used, 0-not used';

comment on column GENCONDATA.MTPASA is
'Flag to indicate if the constraint RHS is to be used for MT PASA, 1-used, 0-not used';

comment on column GENCONDATA.IMPACT is
'The device(s) that is affected by the constraint e.g. Interconnector, Generator(s) or Cutset';

comment on column GENCONDATA.SOURCE is
'The source of the constraint formulation';

comment on column GENCONDATA.LIMITTYPE is
'The limit type of the constraint e.g. Transient Stability, Voltage Stability';

comment on column GENCONDATA.REASON is
'The contingency or reason for the constraint';

comment on column GENCONDATA.MODIFICATIONS is
'Details of the changes made to this version of the constraint';

comment on column GENCONDATA.ADDITIONALNOTES is
'Extra notes on the constraint';

comment on column GENCONDATA.P5MIN_SCOPE_OVERRIDE is
'Extra notes on the constraint: NULL = Dispatch RHS applied in 5MPD, PD = PreDispatch RHS applied in 5MPD';

comment on column GENCONDATA.LRC is
'Flag to indicate if PASA LRC run uses the constraint; 1-used, 0-not used';

comment on column GENCONDATA.LOR is
'Flag to indicate if PASA LOR run uses the constraint; 1-used, 0-not used';

comment on column GENCONDATA.FORCE_SCADA is
'Flags Constraints for which NEMDE must use "InitialMW" values instead of "WhatOfInitialMW" for Intervention Pricing runs';

alter table GENCONDATA
   add constraint GENCONDATA_PK primary key (EFFECTIVEDATE, VERSIONNO, GENCONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: GENCONDATA_LCX                                        */
/*==============================================================*/
create index GENCONDATA_LCX on GENCONDATA (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: GENCONSET                                             */
/*==============================================================*/
create table GENCONSET (
   GENCONSETID          VARCHAR2(20)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   GENCONID             VARCHAR2(20)          not null,
   GENCONEFFDATE        DATE,
   GENCONVERSIONNO      NUMBER(3,0),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table GENCONSET is
'GENCONSET sets out generic constraint sets that are invoked and revoked, and may contain many generic constraints (GENCONDATA).';

comment on column GENCONSET.GENCONSETID is
'Unique ID for the Constraint Set';

comment on column GENCONSET.EFFECTIVEDATE is
'Date this record becomes effective';

comment on column GENCONSET.VERSIONNO is
'Version no of the record for the given effective date';

comment on column GENCONSET.GENCONID is
'Generic Contraint ID';

comment on column GENCONSET.GENCONEFFDATE is
'Since market start in 1998 these fields have not been used and any data that has been populated in the fields should be ignored';

comment on column GENCONSET.GENCONVERSIONNO is
'Since market start in 1998 these fields have not been used and any data that has been populated in the fields should be ignored';

comment on column GENCONSET.LASTCHANGED is
'Last date and time record changed';

alter table GENCONSET
   add constraint GENCONSET_PK primary key (GENCONSETID, EFFECTIVEDATE, VERSIONNO, GENCONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: GENCONSET_LCX                                         */
/*==============================================================*/
create index GENCONSET_LCX on GENCONSET (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: GENCONSETINVOKE                                       */
/*==============================================================*/
create table GENCONSETINVOKE (
   INVOCATION_ID        NUMBER(9)             not null,
   STARTDATE            DATE                  not null,
   STARTPERIOD          NUMBER(3,0)           not null,
   GENCONSETID          VARCHAR2(20)          not null,
   ENDDATE              DATE,
   ENDPERIOD            NUMBER(3,0),
   STARTAUTHORISEDBY    VARCHAR2(15),
   ENDAUTHORISEDBY      VARCHAR2(15),
   INTERVENTION         VARCHAR2(1),
   ASCONSTRAINTTYPE     VARCHAR2(10),
   LASTCHANGED          DATE,
   STARTINTERVALDATETIME DATE,
   ENDINTERVALDATETIME  DATE,
   SYSTEMNORMAL         VARCHAR2(1)
)
   tablespace PSSTAB02;

comment on table GENCONSETINVOKE is
'GENCONSETINVOKE provides details of invoked and revoked generic constraints. GENCONSETINVOKE is the key table for determining what constraints are active in dispatch, predispatch and PASA.
GENCONSETINVOKE also indicates whether constraints are for interconnector limits, ancillary services, etc.';

comment on column GENCONSETINVOKE.INVOCATION_ID is
'Abstract unique identifier for the record. Allows Invocations to be modified without affecting PK values';

comment on column GENCONSETINVOKE.STARTDATE is
'Market date of start';

comment on column GENCONSETINVOKE.STARTPERIOD is
'The first dispatch interval of the invocation being the dispatch interval number starting from1 at 04:05.';

comment on column GENCONSETINVOKE.GENCONSETID is
'Unique generic constraint set identifier';

comment on column GENCONSETINVOKE.ENDDATE is
'Market date end';

comment on column GENCONSETINVOKE.ENDPERIOD is
'Dispatch interval number end';

comment on column GENCONSETINVOKE.STARTAUTHORISEDBY is
'User authorising invoke, indicating a constraint set invocation is applicable (i.e. non-null). A null value indicates inactive invocation.';

comment on column GENCONSETINVOKE.ENDAUTHORISEDBY is
'user authorising revoke.';

comment on column GENCONSETINVOKE.INTERVENTION is
'0 is not intervention, 1 is intervention and causes dispatch to solve twice.';

comment on column GENCONSETINVOKE.ASCONSTRAINTTYPE is
'Constraint type (e.g. ancillary services). This also flags where a constraint is an interconnector or intra-region network limit.';

comment on column GENCONSETINVOKE.LASTCHANGED is
'Last date and time record changed';

comment on column GENCONSETINVOKE.STARTINTERVALDATETIME is
'The settlement date and time corresponding to the first interval to which the constraint set is to be applied.';

comment on column GENCONSETINVOKE.ENDINTERVALDATETIME is
'The settlement date and time corresponding to the last interval to which the constraint set is to be applied.';

comment on column GENCONSETINVOKE.SYSTEMNORMAL is
'Flag to indicate if the constraint set is a system normal (1) or an outage set (0)';

alter table GENCONSETINVOKE
   add constraint GENCONSETINV_PK primary key (INVOCATION_ID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: GENCONSETINVOKE_LCX                                   */
/*==============================================================*/
create index GENCONSETINVOKE_LCX on GENCONSETINVOKE (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: GENCONSETTRK                                          */
/*==============================================================*/
create table GENCONSETTRK (
   GENCONSETID          VARCHAR2(20)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   DESCRIPTION          VARCHAR2(256),
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE,
   COVERAGE             VARCHAR2(64),
   MODIFICATIONS        VARCHAR2(256),
   SYSTEMNORMAL         VARCHAR2(1),
   OUTAGE               VARCHAR2(256)
)
   tablespace PSSTAB02;

comment on table GENCONSETTRK is
'GENCONSETTRK assists in determining the correct version of a generic constraint set that has been invoked in GENCONSETINVOKE.';

comment on column GENCONSETTRK.GENCONSETID is
'Unique ID for the Constraint Set';

comment on column GENCONSETTRK.EFFECTIVEDATE is
'Date this record becomes effective';

comment on column GENCONSETTRK.VERSIONNO is
'Version no of the record for the given effective date';

comment on column GENCONSETTRK.DESCRIPTION is
'Description of the constraint';

comment on column GENCONSETTRK.AUTHORISEDBY is
'The person who authorised the constraint set';

comment on column GENCONSETTRK.AUTHORISEDDATE is
'The date and time of authorising the constraint set';

comment on column GENCONSETTRK.LASTCHANGED is
'Last date and time record changed';

comment on column GENCONSETTRK.COVERAGE is
'The region the constraint set is located in or a special grouping (e.g. CHIMERA)';

comment on column GENCONSETTRK.MODIFICATIONS is
'Details of the changes made to this version of the constraint set';

comment on column GENCONSETTRK.SYSTEMNORMAL is
'Not used as of 2005 End of Year Release [was Flag to indicate if the constraint set is a system normal (1) or and an outage set (0)]';

comment on column GENCONSETTRK.OUTAGE is
'Detail of the plant that is not in service';

alter table GENCONSETTRK
   add constraint GENCONSETTRK_PK primary key (GENCONSETID, EFFECTIVEDATE, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: GENCONSETTRK_LCX                                      */
/*==============================================================*/
create index GENCONSETTRK_LCX on GENCONSETTRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: GENERICCONSTRAINTRHS                                  */
/*==============================================================*/
create table GENERICCONSTRAINTRHS (
   GENCONID             VARCHAR2(20)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(22,0)          not null,
   SCOPE                VARCHAR2(2)           not null,
   TERMID               NUMBER(4,0)           not null,
   GROUPID              NUMBER(3,0),
   SPD_ID               VARCHAR2(21),
   SPD_TYPE             VARCHAR2(1),
   FACTOR               NUMBER(16,6),
   OPERATION            VARCHAR2(10),
   DEFAULTVALUE         NUMBER(16,6),
   PARAMETERTERM1       VARCHAR2(12),
   PARAMETERTERM2       VARCHAR2(12),
   PARAMETERTERM3       VARCHAR2(12),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table GENERICCONSTRAINTRHS is
'GENERICCONSTRAINTRHS sets out details of generic constraint Right Hand Side (RHS) formulations for dispatch (DS), predispatch (PD) and Short Term PASA (ST). GENERICCONSTRAINTRHS also includes general expressions (EQ) used in the dispatch, predispatch and PASA time frames.
GENERICCONSTRAINTRHS replaces data previously available via the "Constraint Library Excel spreadsheet.';

comment on column GENERICCONSTRAINTRHS.GENCONID is
'Generic Constraint Identifier';

comment on column GENERICCONSTRAINTRHS.EFFECTIVEDATE is
'Effective date of this record';

comment on column GENERICCONSTRAINTRHS.VERSIONNO is
'Version no of this record for the effective date';

comment on column GENERICCONSTRAINTRHS.SCOPE is
'Scope of RHS term (DS, PD, ST or EQ)';

comment on column GENERICCONSTRAINTRHS.TERMID is
'The unique identifier for the a constraint RHS term';

comment on column GENERICCONSTRAINTRHS.GROUPID is
'ID of super-term, if this is a sub-term';

comment on column GENERICCONSTRAINTRHS.SPD_ID is
'ID defining data source';

comment on column GENERICCONSTRAINTRHS.SPD_TYPE is
'ID describing type of data source';

comment on column GENERICCONSTRAINTRHS.FACTOR is
'Multiplier applied to operator result';

comment on column GENERICCONSTRAINTRHS.OPERATION is
'Unitary operator to apply to data value';

comment on column GENERICCONSTRAINTRHS.DEFAULTVALUE is
'Default value if primary source given by SPD_ID and SPD_TYPE not available.';

comment on column GENERICCONSTRAINTRHS.PARAMETERTERM1 is
'The unique identifier for the first term (logic expression) to use in a Branch term';

comment on column GENERICCONSTRAINTRHS.PARAMETERTERM2 is
'The unique identifier for the second term (logic<=0 result) to use in a Branch term';

comment on column GENERICCONSTRAINTRHS.PARAMETERTERM3 is
'The unique identifier for the third term (logic>0 result) to use in a Branch term';

comment on column GENERICCONSTRAINTRHS.LASTCHANGED is
'Last date and time record changed';

alter table GENERICCONSTRAINTRHS
   add constraint GENERICCONSTRAINTRHS_PK primary key (GENCONID, EFFECTIVEDATE, VERSIONNO, SCOPE, TERMID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: GENERICCONSTRAINTRHS_LCHD_IDX                         */
/*==============================================================*/
create index GENERICCONSTRAINTRHS_LCHD_IDX on GENERICCONSTRAINTRHS (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: GENERICEQUATIONDESC                                   */
/*==============================================================*/
create table GENERICEQUATIONDESC (
   EQUATIONID           VARCHAR2(20)          not null,
   DESCRIPTION          VARCHAR2(256),
   LASTCHANGED          DATE,
   IMPACT               VARCHAR2(64),
   SOURCE               VARCHAR2(128),
   LIMITTYPE            VARCHAR2(64),
   REASON               VARCHAR2(256),
   MODIFICATIONS        VARCHAR2(256),
   ADDITIONALNOTES      VARCHAR2(256)
)
   tablespace PSSTAB01;

comment on table GENERICEQUATIONDESC is
'GENERICEQUATIONDESC defines a generic equation identifier with a description. The formulation of the generic equation is detailed in GENERICEQUATIONRHS.';

comment on column GENERICEQUATIONDESC.EQUATIONID is
'Generic Equation Identifier';

comment on column GENERICEQUATIONDESC.DESCRIPTION is
'Generic Equation Description';

comment on column GENERICEQUATIONDESC.LASTCHANGED is
'Last date and time record changed';

comment on column GENERICEQUATIONDESC.IMPACT is
'The device(s) affected by the constraint (e.g. Interconnector, Generator(s) or Cutset)';

comment on column GENERICEQUATIONDESC.SOURCE is
'The source of the constraint formulation';

comment on column GENERICEQUATIONDESC.LIMITTYPE is
'The limit type of the constraint e.g. Transient Stability, Voltage Stability';

comment on column GENERICEQUATIONDESC.REASON is
'The contingency or reason for the constraint';

comment on column GENERICEQUATIONDESC.MODIFICATIONS is
'Details of the changes made to this version of the generic equation RHS';

comment on column GENERICEQUATIONDESC.ADDITIONALNOTES is
'Extra notes on the constraint';

alter table GENERICEQUATIONDESC
   add constraint GENERICEQUATIONDESC_PK primary key (EQUATIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: GENERICEQUATIONDS_LCHD_IDX                            */
/*==============================================================*/
create index GENERICEQUATIONDS_LCHD_IDX on GENERICEQUATIONDESC (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: GENERICEQUATIONRHS                                    */
/*==============================================================*/
create table GENERICEQUATIONRHS (
   EQUATIONID           VARCHAR2(20)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   TERMID               NUMBER(3,0)           not null,
   GROUPID              NUMBER(3,0),
   SPD_ID               VARCHAR2(21),
   SPD_TYPE             VARCHAR2(1),
   FACTOR               NUMBER(16,6),
   OPERATION            VARCHAR2(10),
   DEFAULTVALUE         NUMBER(16,6),
   PARAMETERTERM1       VARCHAR2(12),
   PARAMETERTERM2       VARCHAR2(12),
   PARAMETERTERM3       VARCHAR2(12),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table GENERICEQUATIONRHS is
'GENERICEQUATIONRHS stores the formulation of commonly used Generic Constraint Right Hand Side Equations referenced from Generic Constraint Right Hand Side definitions stored in GENERICCONSTRAINTRHS. The Generic Equation definitions are versioned and the latest effective version is applied to the dispatch process.';

comment on column GENERICEQUATIONRHS.EQUATIONID is
'Generic Equation Identifier';

comment on column GENERICEQUATIONRHS.EFFECTIVEDATE is
'Effective date of this record';

comment on column GENERICEQUATIONRHS.VERSIONNO is
'Version no of this record for the effective date';

comment on column GENERICEQUATIONRHS.TERMID is
'The unique identifier for the a equation RHS term';

comment on column GENERICEQUATIONRHS.GROUPID is
'ID of super-term, if this is a sub-term';

comment on column GENERICEQUATIONRHS.SPD_ID is
'ID defining data source';

comment on column GENERICEQUATIONRHS.SPD_TYPE is
'ID describing type of data source';

comment on column GENERICEQUATIONRHS.FACTOR is
'Multiplier applied to operator result';

comment on column GENERICEQUATIONRHS.OPERATION is
'Unitary operator to apply to data value';

comment on column GENERICEQUATIONRHS.DEFAULTVALUE is
'Default value if primary source given by SPD_ID and SPD_TYPE not available.';

comment on column GENERICEQUATIONRHS.PARAMETERTERM1 is
'The unique identifier for the first term (logic expression) to use in a Branch term';

comment on column GENERICEQUATIONRHS.PARAMETERTERM2 is
'The unique identifier for the second term (logic<=0 result) to use in a Branch term';

comment on column GENERICEQUATIONRHS.PARAMETERTERM3 is
'The unique identifier for the third term (logic>0 result) to use in a Branch term';

comment on column GENERICEQUATIONRHS.LASTCHANGED is
'Last date and time record changed';

alter table GENERICEQUATIONRHS
   add constraint GENERICEQUATIONRHS_PK primary key (EQUATIONID, EFFECTIVEDATE, VERSIONNO, TERMID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: GENERICEQUATION_LCHD_IDX                              */
/*==============================================================*/
create index GENERICEQUATION_LCHD_IDX on GENERICEQUATIONRHS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: GENMETER                                              */
/*==============================================================*/
create table GENMETER (
   METERID              VARCHAR2(12)          not null,
   GENSETID             VARCHAR2(20),
   CONNECTIONPOINTID    VARCHAR2(10),
   STATIONID            VARCHAR2(10),
   METERTYPE            VARCHAR2(20),
   METERCLASS           VARCHAR2(10),
   VOLTAGELEVEL         NUMBER(6,0),
   APPLYDATE            DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   AUTHORISEDBY         VARCHAR2(10),
   AUTHORISEDDATE       DATE,
   COMDATE              DATE,
   DECOMDATE            DATE,
   ENDDATE              DATE,
   STARTDATE            DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table GENMETER is
'GENMETER shows details of generator meter sets.';

comment on column GENMETER.METERID is
'Meter Id';

comment on column GENMETER.GENSETID is
'Generator Set ID';

comment on column GENMETER.CONNECTIONPOINTID is
'Not used';

comment on column GENMETER.STATIONID is
'Station Identifier';

comment on column GENMETER.METERTYPE is
'LOAD';

comment on column GENMETER.METERCLASS is
'WATT or AUXILARY';

comment on column GENMETER.VOLTAGELEVEL is
'Voltage';

comment on column GENMETER.APPLYDATE is
'Application date';

comment on column GENMETER.VERSIONNO is
'Version no of the record for the given effective date';

comment on column GENMETER.AUTHORISEDBY is
'AEMO user authorising';

comment on column GENMETER.AUTHORISEDDATE is
'Date authorised';

comment on column GENMETER.COMDATE is
'Not used';

comment on column GENMETER.DECOMDATE is
'Not used';

comment on column GENMETER.ENDDATE is
'Not used';

comment on column GENMETER.STARTDATE is
'Not used';

comment on column GENMETER.LASTCHANGED is
'Last date and time record changed';

alter table GENMETER
   add constraint GENMETERS_PK primary key (METERID, APPLYDATE, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: GENMETER_LCX                                          */
/*==============================================================*/
create index GENMETER_LCX on GENMETER (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: GENMETER_NDX2                                         */
/*==============================================================*/
create index GENMETER_NDX2 on GENMETER (
   STATIONID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: GENUNITMTRINPERIOD                                    */
/*==============================================================*/
create table GENUNITMTRINPERIOD (
   PARTICIPANTID        VARCHAR2(10)          not null,
   SETTLEMENTDATE       DATE                  not null,
   VERSIONNO            NUMBER(6,0)           not null,
   CONNECTIONPOINTID    VARCHAR2(10)          not null,
   PERIODID             NUMBER(3,0)           not null,
   GENUNITID            VARCHAR2(10),
   STATIONID            VARCHAR2(10),
   IMPORTENERGYVALUE    NUMBER(16,6),
   EXPORTENERGYVALUE    NUMBER(16,6),
   IMPORTREACTIVEVALUE  NUMBER(16,6),
   EXPORTREACTIVEVALUE  NUMBER(16,6),
   LASTCHANGED          DATE,
   MDA                  VARCHAR2(10)          not null,
   LOCAL_RETAILER       VARCHAR2(10)         default 'POOLNSW'  not null
)
   tablespace PSSTAB03;

comment on table GENUNITMTRINPERIOD is
'GENUNITMTRINPERIOD shows meter reading by period for each generator meter. GENUNITMTRINPERIOD covers generated power flowing into the system. It is used to calculate settlement values.';

comment on column GENUNITMTRINPERIOD.PARTICIPANTID is
'Unique participant identifier';

comment on column GENUNITMTRINPERIOD.SETTLEMENTDATE is
'Trading date of meter data submitted';

comment on column GENUNITMTRINPERIOD.VERSIONNO is
'Version no of the record for the given effective date';

comment on column GENUNITMTRINPERIOD.CONNECTIONPOINTID is
'Connection Point NMI';

comment on column GENUNITMTRINPERIOD.PERIODID is
'Period number where 1 period ending 00:30 EST';

comment on column GENUNITMTRINPERIOD.GENUNITID is
'Physical unit ID';

comment on column GENUNITMTRINPERIOD.STATIONID is
'Station Identifier';

comment on column GENUNITMTRINPERIOD.IMPORTENERGYVALUE is
'Energy sent to the pool (MWh)';

comment on column GENUNITMTRINPERIOD.EXPORTENERGYVALUE is
'Energy received from the pool (MWh)';

comment on column GENUNITMTRINPERIOD.IMPORTREACTIVEVALUE is
'Reactive power sent to the network';

comment on column GENUNITMTRINPERIOD.EXPORTREACTIVEVALUE is
'Reactive power received from the network';

comment on column GENUNITMTRINPERIOD.LASTCHANGED is
'Last date and time record changed';

comment on column GENUNITMTRINPERIOD.MDA is
'Relevant Metering Data Agent';

comment on column GENUNITMTRINPERIOD.LOCAL_RETAILER is
'Local Retailer for this NMI';

alter table GENUNITMTRINPERIOD
   add constraint GENUNITMTRINPERD_PK primary key (SETTLEMENTDATE, MDA, VERSIONNO, CONNECTIONPOINTID, PARTICIPANTID, LOCAL_RETAILER, PERIODID)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Index: GENUNITMTRINPERIOD_LCX                                */
/*==============================================================*/
create index GENUNITMTRINPERIOD_LCX on GENUNITMTRINPERIOD (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Index: GENUNITMTRINPERIOD_NDX2                               */
/*==============================================================*/
create index GENUNITMTRINPERIOD_NDX2 on GENUNITMTRINPERIOD (
   STATIONID ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: GENUNITS                                              */
/*==============================================================*/
create table GENUNITS (
   GENSETID             VARCHAR2(20)          not null,
   STATIONID            VARCHAR2(10),
   SETLOSSFACTOR        NUMBER(16,6),
   CDINDICATOR          VARCHAR2(10),
   AGCFLAG              VARCHAR2(2),
   SPINNINGFLAG         VARCHAR2(2),
   VOLTLEVEL            NUMBER(6,0),
   REGISTEREDCAPACITY   NUMBER(6,0),
   DISPATCHTYPE         VARCHAR2(20),
   STARTTYPE            VARCHAR2(20),
   MKTGENERATORIND      VARCHAR2(10),
   NORMALSTATUS         VARCHAR2(10),
   MAXCAPACITY          NUMBER(6,0),
   GENSETTYPE           VARCHAR2(15),
   GENSETNAME           VARCHAR2(40),
   LASTCHANGED          DATE,
   CO2E_EMISSIONS_FACTOR NUMBER(18,8),
   CO2E_ENERGY_SOURCE   VARCHAR2(100),
   CO2E_DATA_SOURCE     VARCHAR2(20),
   MINCAPACITY          NUMBER(6,0),
   REGISTEREDMINCAPACITY NUMBER(6,0),
   MAXSTORAGECAPACITY   NUMBER(15,5)
)
   tablespace PSSTAB01;

comment on table GENUNITS is
'GENUNITS shows Genset details for each physical unit with the relevant station.';

comment on column GENUNITS.GENSETID is
'Physical Unit identifier';

comment on column GENUNITS.STATIONID is
'Station Identifier';

comment on column GENUNITS.SETLOSSFACTOR is
'Not used';

comment on column GENUNITS.CDINDICATOR is
'Centrally dispatched Indicator';

comment on column GENUNITS.AGCFLAG is
'AGC Available flag';

comment on column GENUNITS.SPINNINGFLAG is
'Not used';

comment on column GENUNITS.VOLTLEVEL is
'Voltage level';

comment on column GENUNITS.REGISTEREDCAPACITY is
'Registered capacity';

comment on column GENUNITS.DISPATCHTYPE is
'Identifies LOAD, GENERATOR or BIDIRECTIONAL. This will likely expand to more generic models as new technology types are integrated into the NEM.';

comment on column GENUNITS.STARTTYPE is
'Fast / Slow / Not Dispatched';

comment on column GENUNITS.MKTGENERATORIND is
'Market Generator Indicator Flag';

comment on column GENUNITS.NORMALSTATUS is
'On / Off for load';

comment on column GENUNITS.MAXCAPACITY is
'Maximum capacity';

comment on column GENUNITS.GENSETTYPE is
'Genset type';

comment on column GENUNITS.GENSETNAME is
'Genset name';

comment on column GENUNITS.LASTCHANGED is
'Last date and time record changed';

comment on column GENUNITS.CO2E_EMISSIONS_FACTOR is
'The emissions factor for the generating unit, as calculated by Settlements staff members';

comment on column GENUNITS.CO2E_ENERGY_SOURCE is
'The energy source for the generating unit, as used in the calculation of the CO2-e emissions factor.  Distinct from the Energy Source for a generating unit published as part of the Registration Master List';

comment on column GENUNITS.CO2E_DATA_SOURCE is
'An indicator as to the source of the emission factor used in the calculation of the index. The applicable values for this field would be NTNDP which indicates the emission factor is quoted from the National Transmission Network Development Plan or Estimated to indicate the emission factor has been calculated using an internal AEMO procedure based upon the Department of Climate Change and Energy Efficiency NGA factors';

comment on column GENUNITS.MINCAPACITY is
'Minimum capacity only for load side of BDU, otherwise 0 (MW)';

comment on column GENUNITS.REGISTEREDMINCAPACITY is
'Registered minimum capacity only for load side of BDU, otherwise 0 (MW)';

comment on column GENUNITS.MAXSTORAGECAPACITY is
'The rated storage capacity (MWh), information only';

alter table GENUNITS
   add constraint GENUNIT_PK primary key (GENSETID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: GENUNITS_LCX                                          */
/*==============================================================*/
create index GENUNITS_LCX on GENUNITS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: GENUNITS_UNIT                                         */
/*==============================================================*/
create table GENUNITS_UNIT (
   GENSETID             VARCHAR2(20)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(6,0)           not null,
   UNIT_GROUPING_LABEL  VARCHAR2(20)          not null,
   UNIT_COUNT           NUMBER(10,0),
   UNIT_SIZE            NUMBER(8,3),
   UNIT_MAX_SIZE        NUMBER(8,3),
   AGGREGATION_FLAG     NUMBER(1,0),
   LASTCHANGED          DATE,
   UNITMINSIZE          NUMBER(8,3),
   MAXSTORAGECAPACITY   NUMBER(15,5),
   REGISTEREDCAPACITY   NUMBER(8,3),
   REGISTEREDMINCAPACITY NUMBER(8,3)
)
   tablespace PSSTAB01;

comment on table GENUNITS_UNIT is
'Physical units within a Gen Unit Set';

comment on column GENUNITS_UNIT.GENSETID is
'System wide unique Generating Set ID';

comment on column GENUNITS_UNIT.EFFECTIVEDATE is
'Effective Date of this detail record';

comment on column GENUNITS_UNIT.VERSIONNO is
'Version with respect to the effective date';

comment on column GENUNITS_UNIT.UNIT_GROUPING_LABEL is
'Label of Physical Units within the station';

comment on column GENUNITS_UNIT.UNIT_COUNT is
'Number of units in this Gen Unit grouping';

comment on column GENUNITS_UNIT.UNIT_SIZE is
'Nameplate Capacity for each unit in this grouping';

comment on column GENUNITS_UNIT.UNIT_MAX_SIZE is
'Maximum Capacity for each unit in this grouping';

comment on column GENUNITS_UNIT.AGGREGATION_FLAG is
'Deprecated as this flag is moved to DUDETAIL table with IESS release.';

comment on column GENUNITS_UNIT.LASTCHANGED is
'Date/Time when record was changed';

comment on column GENUNITS_UNIT.UNITMINSIZE is
'Only applicable for the LOAD side of BDU (MW)';

comment on column GENUNITS_UNIT.MAXSTORAGECAPACITY is
'The rated storage capacity (MWh), information only';

comment on column GENUNITS_UNIT.REGISTEREDCAPACITY is
'Registered capacity for normal operations';

comment on column GENUNITS_UNIT.REGISTEREDMINCAPACITY is
'Only applicable for the LOAD side of BDU (MW)';

alter table GENUNITS_UNIT
   add constraint GENUNITS_UNIT_PK primary key (GENSETID, EFFECTIVEDATE, VERSIONNO, UNIT_GROUPING_LABEL)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: GST_BAS_CLASS                                         */
/*==============================================================*/
create table GST_BAS_CLASS (
   BAS_CLASS            VARCHAR2(30)          not null,
   DESCRIPTION          VARCHAR2(100),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table GST_BAS_CLASS is
'GST_BAS_CLASS contains a static list of BAS (Business Activity Statement) classifications supported by the MMS.';

comment on column GST_BAS_CLASS.BAS_CLASS is
'The BAS classification';

comment on column GST_BAS_CLASS.DESCRIPTION is
'Description of the BAS classification';

comment on column GST_BAS_CLASS.LASTCHANGED is
'Last date and time the record changed';

alter table GST_BAS_CLASS
   add constraint GST_BAS_CLASS_PK primary key (BAS_CLASS)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: GST_BAS_CLASS_LCX                                     */
/*==============================================================*/
create index GST_BAS_CLASS_LCX on GST_BAS_CLASS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: GST_RATE                                              */
/*==============================================================*/
create table GST_RATE (
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   BAS_CLASS            VARCHAR2(30)          not null,
   GST_RATE             NUMBER(8,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table GST_RATE is
'GST_RATE maintains the GST rates on a BAS (Business Activity Statement) class basis.';

comment on column GST_RATE.EFFECTIVEDATE is
'The effective date of the data set';

comment on column GST_RATE.VERSIONNO is
'The version number of the data set';

comment on column GST_RATE.BAS_CLASS is
'The BAS classification';

comment on column GST_RATE.GST_RATE is
'The GST rate that applies to this BAS classification';

comment on column GST_RATE.LASTCHANGED is
'Last date and time the record changed';

alter table GST_RATE
   add constraint GST_RATE_PK primary key (EFFECTIVEDATE, VERSIONNO, BAS_CLASS)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: GST_RATE_LCX                                          */
/*==============================================================*/
create index GST_RATE_LCX on GST_RATE (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: GST_TRANSACTION_CLASS                                 */
/*==============================================================*/
create table GST_TRANSACTION_CLASS (
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   TRANSACTION_TYPE     VARCHAR2(30)          not null,
   BAS_CLASS            VARCHAR2(30)          not null,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table GST_TRANSACTION_CLASS is
'GST_TRANSACTION_CLASS maps NEM settlement transaction types with BAS (Business Activity Statement) classifications.';

comment on column GST_TRANSACTION_CLASS.EFFECTIVEDATE is
'The effective date of the data set';

comment on column GST_TRANSACTION_CLASS.VERSIONNO is
'The version number of the data set';

comment on column GST_TRANSACTION_CLASS.TRANSACTION_TYPE is
'NEM settlement transaction type';

comment on column GST_TRANSACTION_CLASS.BAS_CLASS is
'The BAS classification that the transaction type corresponds to';

comment on column GST_TRANSACTION_CLASS.LASTCHANGED is
'Last date and time the record changed';

alter table GST_TRANSACTION_CLASS
   add constraint GST_TRANS_CLASS_PK primary key (EFFECTIVEDATE, VERSIONNO, TRANSACTION_TYPE, BAS_CLASS)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: GST_TRAN_CLASS_LCX                                    */
/*==============================================================*/
create index GST_TRAN_CLASS_LCX on GST_TRANSACTION_CLASS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: GST_TRANSACTION_TYPE                                  */
/*==============================================================*/
create table GST_TRANSACTION_TYPE (
   TRANSACTION_TYPE     VARCHAR2(30)          not null,
   DESCRIPTION          VARCHAR2(100),
   GL_FINANCIALCODE     VARCHAR2(10),
   GL_TCODE             VARCHAR2(15),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table GST_TRANSACTION_TYPE is
'GST_TRANSACTION_TYPE shows a static list of transaction types supported by the MMS.';

comment on column GST_TRANSACTION_TYPE.TRANSACTION_TYPE is
'The transaction type';

comment on column GST_TRANSACTION_TYPE.DESCRIPTION is
'Description of the transaction type';

comment on column GST_TRANSACTION_TYPE.LASTCHANGED is
'Last date and time the record changed';

alter table GST_TRANSACTION_TYPE
   add constraint GST_TRANSACTION_TYPE_PK primary key (TRANSACTION_TYPE)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: GST_TRANSACTION_TYPE_LCX                              */
/*==============================================================*/
create index GST_TRANSACTION_TYPE_LCX on GST_TRANSACTION_TYPE (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: INSTRUCTIONSUBTYPE                                    */
/*==============================================================*/
create table INSTRUCTIONSUBTYPE (
   INSTRUCTIONTYPEID    VARCHAR2(10)          not null,
   INSTRUCTIONSUBTYPEID VARCHAR2(10)          not null,
   DESCRIPTION          VARCHAR2(64),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table INSTRUCTIONSUBTYPE is
'Each Dispatch instruction (GD instruct) has a type and subtype. INSTRUCTIONSUBTYPE, together with INSTRUCTIONTYPE, sets out valid instruction types.';

comment on column INSTRUCTIONSUBTYPE.INSTRUCTIONTYPEID is
'Instruction type';

comment on column INSTRUCTIONSUBTYPE.INSTRUCTIONSUBTYPEID is
'Subtype for each dispatch instruction type, for example governor off.';

comment on column INSTRUCTIONSUBTYPE.DESCRIPTION is
'Description of instruction subtype';

comment on column INSTRUCTIONSUBTYPE.LASTCHANGED is
'Last date and time record changed';

alter table INSTRUCTIONSUBTYPE
   add constraint INSTRUCTIONSUBTYPE_PK primary key (INSTRUCTIONTYPEID, INSTRUCTIONSUBTYPEID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: INSTRUCTIONSUBTYPE_LCX                                */
/*==============================================================*/
create index INSTRUCTIONSUBTYPE_LCX on INSTRUCTIONSUBTYPE (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: INSTRUCTIONTYPE                                       */
/*==============================================================*/
create table INSTRUCTIONTYPE (
   INSTRUCTIONTYPEID    VARCHAR2(10)          not null,
   DESCRIPTION          VARCHAR2(64),
   REGIONID             VARCHAR2(10),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table INSTRUCTIONTYPE is
'Dispatch instruction (GD instruct) has types and subtypes. INSTRUCTIONTYPE, together with INSTRUCTIONSUBTYPE, sets out valid instruction types.';

comment on column INSTRUCTIONTYPE.INSTRUCTIONTYPEID is
'Dispatch instruction type for example FCAS service.';

comment on column INSTRUCTIONTYPE.DESCRIPTION is
'Description of instruction type';

comment on column INSTRUCTIONTYPE.REGIONID is
'Region id if regional instruction only.';

comment on column INSTRUCTIONTYPE.LASTCHANGED is
'Last date and time record changed';

alter table INSTRUCTIONTYPE
   add constraint INSTRUCTIONTYPE_PK primary key (INSTRUCTIONTYPEID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: INSTRUCTIONTYPE_LCX                                   */
/*==============================================================*/
create index INSTRUCTIONTYPE_LCX on INSTRUCTIONTYPE (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: INTCONTRACT                                           */
/*==============================================================*/
create table INTCONTRACT (
   CONTRACTID           VARCHAR2(10)          not null,
   PARTICIPANTID        VARCHAR2(10),
   DUID                 VARCHAR2(10),
   STARTDATE            DATE,
   ENDDATE              DATE,
   STARTPERIOD          NUMBER(3,0),
   ENDPERIOD            NUMBER(3,0),
   DEREGISTRATIONDATE   DATE,
   DEREGISTRATIONPERIOD NUMBER(3,0),
   LASTCHANGED          DATE,
   REGIONID             VARCHAR2(10)
)
   tablespace PSSTAB01;

comment on table INTCONTRACT is
'INTCONTRACT shows intervention contract details. These are specific to each intervention.';

comment on column INTCONTRACT.CONTRACTID is
'Intervention Contract Identifier';

comment on column INTCONTRACT.PARTICIPANTID is
'Unique participant identifier';

comment on column INTCONTRACT.DUID is
'Dispatchable Unit ID';

comment on column INTCONTRACT.STARTDATE is
'Starting Date of Contract';

comment on column INTCONTRACT.ENDDATE is
'Terminate Date of contract';

comment on column INTCONTRACT.STARTPERIOD is
'Starting period of contract';

comment on column INTCONTRACT.ENDPERIOD is
'Terminate period of contract in trading interval';

comment on column INTCONTRACT.DEREGISTRATIONDATE is
'Not Used';

comment on column INTCONTRACT.DEREGISTRATIONPERIOD is
'Not Used';

comment on column INTCONTRACT.LASTCHANGED is
'Last changed date/time';

comment on column INTCONTRACT.REGIONID is
'Region Identifier';

alter table INTCONTRACT
   add constraint INTCONTRACT_PK primary key (CONTRACTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: INTCONTRACT_LCX                                       */
/*==============================================================*/
create index INTCONTRACT_LCX on INTCONTRACT (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: INTCONTRACTAMOUNT                                     */
/*==============================================================*/
create table INTCONTRACTAMOUNT (
   CONTRACTID           VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   PERIODID             NUMBER(3,0)           not null,
   AMOUNT               NUMBER(16,6),
   RCF                  CHAR(1),
   LASTCHANGED          DATE                  not null
)
   tablespace PSSTAB01;

comment on table INTCONTRACTAMOUNT is
'INTCONTRACTAMOUNT shows intervention contract amounts.';

comment on column INTCONTRACTAMOUNT.CONTRACTID is
'Intervention Contract Identifier';

comment on column INTCONTRACTAMOUNT.VERSIONNO is
'Intervention Contract Version';

comment on column INTCONTRACTAMOUNT.PERIODID is
'Period Identifier based on calendar settlement date - YYYYMMDDPP.';

comment on column INTCONTRACTAMOUNT.AMOUNT is
'Intervention Amount for Trading Interval';

comment on column INTCONTRACTAMOUNT.RCF is
'Regional Recovery Flag';

comment on column INTCONTRACTAMOUNT.LASTCHANGED is
'Last date and time record changed';

alter table INTCONTRACTAMOUNT
   add constraint INTCONTRACTAMOUNT_PK primary key (CONTRACTID, VERSIONNO, PERIODID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: INTCONTRACTAMOUNT_LCX                                 */
/*==============================================================*/
create index INTCONTRACTAMOUNT_LCX on INTCONTRACTAMOUNT (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: INTCONTRACTAMOUNTTRK                                  */
/*==============================================================*/
create table INTCONTRACTAMOUNTTRK (
   CONTRACTID           VARCHAR2(10)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table INTCONTRACTAMOUNTTRK is
'INTCONTRACTAMOUNTTRK shows the latest valid version of each intervention contract.';

comment on column INTCONTRACTAMOUNTTRK.CONTRACTID is
'Intervention Contract Identifier';

comment on column INTCONTRACTAMOUNTTRK.VERSIONNO is
'Intervention Contract Version';

comment on column INTCONTRACTAMOUNTTRK.AUTHORISEDBY is
'User name';

comment on column INTCONTRACTAMOUNTTRK.AUTHORISEDDATE is
'Date contract was authorised';

comment on column INTCONTRACTAMOUNTTRK.LASTCHANGED is
'Last date and time record changed';

alter table INTCONTRACTAMOUNTTRK
   add constraint INTCONTRACTAMOUNTTRK_PK primary key (CONTRACTID, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: INTCONTRACTAMOUNTTRK_LCX                              */
/*==============================================================*/
create index INTCONTRACTAMOUNTTRK_LCX on INTCONTRACTAMOUNTTRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: INTERCONNECTOR                                        */
/*==============================================================*/
create table INTERCONNECTOR (
   INTERCONNECTORID     VARCHAR2(10)          not null,
   REGIONFROM           VARCHAR2(10),
   RSOID                VARCHAR2(10),
   REGIONTO             VARCHAR2(10),
   DESCRIPTION          VARCHAR2(64),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table INTERCONNECTOR is
'INTERCONNECTOR sets out valid identifiers for each interconnector.';

comment on column INTERCONNECTOR.INTERCONNECTORID is
'Unique Id of this interconnector';

comment on column INTERCONNECTOR.REGIONFROM is
'Starting region of the interconnect';

comment on column INTERCONNECTOR.RSOID is
'Not used';

comment on column INTERCONNECTOR.REGIONTO is
'Ending region of the interconnect';

comment on column INTERCONNECTOR.DESCRIPTION is
'Description of interconnector';

comment on column INTERCONNECTOR.LASTCHANGED is
'Last date and time record changed';

alter table INTERCONNECTOR
   add constraint INTERCONNECTOR_PK primary key (INTERCONNECTORID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: INTERCONNECTOR_LCX                                    */
/*==============================================================*/
create index INTERCONNECTOR_LCX on INTERCONNECTOR (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: INTERCONNECTORALLOC                                   */
/*==============================================================*/
create table INTERCONNECTORALLOC (
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(5,0)           not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   REGIONID             VARCHAR2(10)          not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   ALLOCATION           NUMBER(12,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table INTERCONNECTORALLOC is
'INTERCONNECTORALLOC shows allocations of interconnector residues to Network Service Providers.';

comment on column INTERCONNECTORALLOC.EFFECTIVEDATE is
'Effective Date of Allocation Details';

comment on column INTERCONNECTORALLOC.VERSIONNO is
'Version No in respect to effective date';

comment on column INTERCONNECTORALLOC.INTERCONNECTORID is
'Interconnector identifier';

comment on column INTERCONNECTORALLOC.REGIONID is
'Region Identifier';

comment on column INTERCONNECTORALLOC.PARTICIPANTID is
'Unique participant identifier';

comment on column INTERCONNECTORALLOC.ALLOCATION is
'Allocation % / 100';

comment on column INTERCONNECTORALLOC.LASTCHANGED is
'Last date and time record changed';

alter table INTERCONNECTORALLOC
   add constraint INTERCONNECTORALLOC_PK primary key (EFFECTIVEDATE, VERSIONNO, INTERCONNECTORID, REGIONID, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: INTERCONNECTORALLOC_LCX                               */
/*==============================================================*/
create index INTERCONNECTORALLOC_LCX on INTERCONNECTORALLOC (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: INTERCONNECTORCONSTRAINT                              */
/*==============================================================*/
create table INTERCONNECTORCONSTRAINT (
   RESERVEOVERALLLOADFACTOR NUMBER(5,2),
   FROMREGIONLOSSSHARE  NUMBER(5,2),
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   MAXMWIN              NUMBER(15,5),
   MAXMWOUT             NUMBER(15,5),
   LOSSCONSTANT         NUMBER(15,6),
   LOSSFLOWCOEFFICIENT  NUMBER(27,17),
   EMSMEASURAND         VARCHAR2(40),
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   DYNAMICRHS           VARCHAR2(1),
   IMPORTLIMIT          NUMBER(6,0),
   EXPORTLIMIT          NUMBER(6,0),
   OUTAGEDERATIONFACTOR NUMBER(15,5),
   NONPHYSICALLOSSFACTOR NUMBER(15,5),
   OVERLOADFACTOR60SEC  NUMBER(15,5),
   OVERLOADFACTOR6SEC   NUMBER(15,5),
   LASTCHANGED          DATE,
   FCASSUPPORTUNAVAILABLE NUMBER(1,0),
   ICTYPE               VARCHAR2(10)
)
   tablespace PSSTAB01;

comment on table INTERCONNECTORCONSTRAINT is
'INTERCONNECTORCONSTRAINT sets out Interconnector limit data used as defaults in dispatch, predispatch and STPASA and used by SPD in calculating flows. INTERCONNECTORCONSTRAINT includes an additional field to restrict an interconnector from support transfer of FCAS.';

comment on column INTERCONNECTORCONSTRAINT.RESERVEOVERALLLOADFACTOR is
'SPD Factor';

comment on column INTERCONNECTORCONSTRAINT.FROMREGIONLOSSSHARE is
'Loss share attributable to from region';

comment on column INTERCONNECTORCONSTRAINT.EFFECTIVEDATE is
'Date that this limit is effective from';

comment on column INTERCONNECTORCONSTRAINT.VERSIONNO is
'Version for this date';

comment on column INTERCONNECTORCONSTRAINT.INTERCONNECTORID is
'Unique Id of this interconnector';

comment on column INTERCONNECTORCONSTRAINT.MAXMWIN is
'Limit of energy flowing into the RegionFrom';

comment on column INTERCONNECTORCONSTRAINT.MAXMWOUT is
'Limit of energy flowing out of the Region';

comment on column INTERCONNECTORCONSTRAINT.LOSSCONSTANT is
'Constant Loss factor';

comment on column INTERCONNECTORCONSTRAINT.LOSSFLOWCOEFFICIENT is
'Linear coefficient of loss factor calculation';

comment on column INTERCONNECTORCONSTRAINT.EMSMEASURAND is
'Identifies the EMS entity that represents the interconnector flow';

comment on column INTERCONNECTORCONSTRAINT.AUTHORISEDBY is
'User authorising record';

comment on column INTERCONNECTORCONSTRAINT.AUTHORISEDDATE is
'Date record authorised';

comment on column INTERCONNECTORCONSTRAINT.DYNAMICRHS is
'Not used';

comment on column INTERCONNECTORCONSTRAINT.IMPORTLIMIT is
'Interconnector import limit';

comment on column INTERCONNECTORCONSTRAINT.EXPORTLIMIT is
'Interconnector export limit';

comment on column INTERCONNECTORCONSTRAINT.OUTAGEDERATIONFACTOR is
'SPD Factor';

comment on column INTERCONNECTORCONSTRAINT.NONPHYSICALLOSSFACTOR is
'Factor for non-physical losses rerun';

comment on column INTERCONNECTORCONSTRAINT.OVERLOADFACTOR60SEC is
'Interconnector overload for 60 sec';

comment on column INTERCONNECTORCONSTRAINT.OVERLOADFACTOR6SEC is
'Interconnector overload for 6 sec';

comment on column INTERCONNECTORCONSTRAINT.LASTCHANGED is
'Last date and time record changed';

comment on column INTERCONNECTORCONSTRAINT.FCASSUPPORTUNAVAILABLE is
'Flag to indicate that the interconnector cannot support FCAS Transfers';

comment on column INTERCONNECTORCONSTRAINT.ICTYPE is
'Interconnector type - Currently either "REGULATED" or "MNSP"';

alter table INTERCONNECTORCONSTRAINT
   add constraint INTCCONSTRAINT_PK primary key (EFFECTIVEDATE, VERSIONNO, INTERCONNECTORID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: INTERCONNECTORCONSTRAINT_LCX                          */
/*==============================================================*/
create index INTERCONNECTORCONSTRAINT_LCX on INTERCONNECTORCONSTRAINT (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: INTERCONNMWFLOW                                       */
/*==============================================================*/
create table INTERCONNMWFLOW (
   SETTLEMENTDATE       DATE                  not null,
   VERSIONNO            NUMBER(6,0)           not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   PERIODID             NUMBER(3,0)           not null,
   IMPORTENERGYVALUE    NUMBER(15,6),
   EXPORTENERGYVALUE    NUMBER(15,6),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table INTERCONNMWFLOW is
'INTERCONNMWFLOW shows Metered Interconnector flow data. INTERCONNMWFLOW shows the meter data provided by Meter Data Providers to MSATS.
Despite the name, this view shows metered energy (MWh) and not power flow (MW).';

comment on column INTERCONNMWFLOW.SETTLEMENTDATE is
'Settlement Date (based on Trading day, not dispatch day - i.e. period 1 ends 00:30)';

comment on column INTERCONNMWFLOW.VERSIONNO is
'Meter Data Version number';

comment on column INTERCONNMWFLOW.INTERCONNECTORID is
'Interconnector Identifier';

comment on column INTERCONNMWFLOW.PERIODID is
'Settlement Period identifier (half hour period)';

comment on column INTERCONNMWFLOW.IMPORTENERGYVALUE is
'Imported Energy value (MWh)';

comment on column INTERCONNMWFLOW.EXPORTENERGYVALUE is
'Exported Energy value (MWh)';

comment on column INTERCONNMWFLOW.LASTCHANGED is
'Record creation timestamp';

alter table INTERCONNMWFLOW
   add constraint INTERCONNMWFLOW_PK primary key (SETTLEMENTDATE, VERSIONNO, INTERCONNECTORID, PERIODID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: INTERCONNMWFLOW_LCIDX                                 */
/*==============================================================*/
create index INTERCONNMWFLOW_LCIDX on INTERCONNMWFLOW (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: INTERMITTENT_CLUSTER_AVAIL                            */
/*==============================================================*/
create table INTERMITTENT_CLUSTER_AVAIL (
   TRADINGDATE          DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   OFFERDATETIME        DATE                  not null,
   CLUSTERID            VARCHAR2(20)          not null,
   PERIODID             NUMBER(3,0)           not null,
   ELEMENTS_UNAVAILABLE NUMBER(5,0),
   ELEMENTS_AVAILABLE   NUMBER(5,0)
)
   tablespace PSSTAB02;

comment on table INTERMITTENT_CLUSTER_AVAIL is
'A submission of expected plant availability for an intermittent generating unit cluster, by Trading Day and Trading Interval.';

comment on column INTERMITTENT_CLUSTER_AVAIL.TRADINGDATE is
'The trading day to which the availability submission applies';

comment on column INTERMITTENT_CLUSTER_AVAIL.DUID is
'Unique Identifier of Dispatchable Unit';

comment on column INTERMITTENT_CLUSTER_AVAIL.OFFERDATETIME is
'Date and Time when this cluster availability submission was loaded';

comment on column INTERMITTENT_CLUSTER_AVAIL.CLUSTERID is
'Unique Cluster Identifier for this cluster within the DUID';

comment on column INTERMITTENT_CLUSTER_AVAIL.PERIODID is
'Trading interval number (148) within this TRADINGDATE for which ELEMENTS_UNAVAILABLE applies';

comment on column INTERMITTENT_CLUSTER_AVAIL.ELEMENTS_UNAVAILABLE is
'Number of elements within this CLUSTERID (turbines for wind, or inverters for solar) that are not available for this TRADINGDATE and PERIODID (scheduled maintenance in AWEFS/ASEFS). Value between 0 and the registered Number of Cluster Elements.Value = 0 means no elements unavailable';

comment on column INTERMITTENT_CLUSTER_AVAIL.ELEMENTS_AVAILABLE is
'Number of elements within this CLUSTERID (turbines for wind, or inverters for solar) that are available for this TRADINGDATE and PERIODID (scheduled maintenance in AWEFS/ASEFS). Value between 0 and the registered Number of Cluster Elements. Value = 0 means no elements available';

alter table INTERMITTENT_CLUSTER_AVAIL
   add constraint INTERMITTENT_CLUSTER_AVAIL_PK primary key (TRADINGDATE, DUID, OFFERDATETIME, CLUSTERID, PERIODID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: INTERMITTENT_CLUSTER_AVAIL_DAY                        */
/*==============================================================*/
create table INTERMITTENT_CLUSTER_AVAIL_DAY (
   TRADINGDATE          DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   OFFERDATETIME        DATE                  not null,
   CLUSTERID            VARCHAR2(20)          not null
)
   tablespace PSSTAB02;

comment on table INTERMITTENT_CLUSTER_AVAIL_DAY is
'Summary record for an availability submission for an intermittent generating unit cluster for a Trading Day.';

comment on column INTERMITTENT_CLUSTER_AVAIL_DAY.TRADINGDATE is
'Trading Day for which this cluster availability submission applies';

comment on column INTERMITTENT_CLUSTER_AVAIL_DAY.DUID is
'Unique Identifier of Dispatchable Unit';

comment on column INTERMITTENT_CLUSTER_AVAIL_DAY.OFFERDATETIME is
'Date and Time when this cluster availability submission was loaded';

comment on column INTERMITTENT_CLUSTER_AVAIL_DAY.CLUSTERID is
'Unique Cluster Identifier for this cluster within the DUID';

alter table INTERMITTENT_CLUSTER_AVAIL_DAY
   add constraint INTERMITTENT_CLUST_AVL_DAY_PK primary key (TRADINGDATE, DUID, OFFERDATETIME, CLUSTERID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: INTERMITTENT_DS_PRED                                  */
/*==============================================================*/
create table INTERMITTENT_DS_PRED (
   RUN_DATETIME         DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   OFFERDATETIME        DATE                  not null,
   INTERVAL_DATETIME    DATE                  not null,
   ORIGIN               VARCHAR2(20)          not null,
   FORECAST_PRIORITY    NUMBER(10,0)          not null,
   FORECAST_MEAN        NUMBER(18,8),
   FORECAST_POE10       NUMBER(18,8),
   FORECAST_POE50       NUMBER(18,8),
   FORECAST_POE90       NUMBER(18,8)
)
   tablespace PSSTAB02;

comment on table INTERMITTENT_DS_PRED is
'Unconstrained Intermittent Generation Forecasts (UIGF) for Dispatch';

comment on column INTERMITTENT_DS_PRED.RUN_DATETIME is
'Date and Time when the forecast applies (dispatch interval ending)';

comment on column INTERMITTENT_DS_PRED.DUID is
'DUID (or Area for non-scheduled) where this forecast applies';

comment on column INTERMITTENT_DS_PRED.OFFERDATETIME is
'Date and Time when this forecast submission was loaded';

comment on column INTERMITTENT_DS_PRED.INTERVAL_DATETIME is
'Date and Time when the forecast applies (dispatch interval ending)';

comment on column INTERMITTENT_DS_PRED.ORIGIN is
'Origin of this forecast (PARTICIPANTID, AWEFS/ASEFS, or another vendor)';

comment on column INTERMITTENT_DS_PRED.FORECAST_PRIORITY is
'Unsuppressed forecasts with higher priority values are used in Dispatch in preference to unsuppressed forecasts with lower priority values';

comment on column INTERMITTENT_DS_PRED.FORECAST_MEAN is
'Forecast MW value for this interval_DateTime';

comment on column INTERMITTENT_DS_PRED.FORECAST_POE10 is
'Forecast 10% POE MW value for this interval_DateTime';

comment on column INTERMITTENT_DS_PRED.FORECAST_POE50 is
'Forecast 50% POE MW value for this interval_DateTime. Used in Dispatch.';

comment on column INTERMITTENT_DS_PRED.FORECAST_POE90 is
'Forecast 90% POE MW value for this interval_DateTime';

alter table INTERMITTENT_DS_PRED
   add constraint INTERMITTENT_DS_PRED_PK primary key (RUN_DATETIME, DUID, OFFERDATETIME, INTERVAL_DATETIME, ORIGIN, FORECAST_PRIORITY)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: INTERMITTENT_DS_RUN                                   */
/*==============================================================*/
create table INTERMITTENT_DS_RUN (
   RUN_DATETIME         DATE                  not null,
   DUID                 Varchar2(20)          not null,
   OFFERDATETIME        DATE                  not null,
   ORIGIN               Varchar2(20)          not null,
   FORECAST_PRIORITY    NUMBER(10,0)          not null,
   AUTHORISEDBY         Varchar2(20),
   COMMENTS             Varchar2(200),
   LASTCHANGED          DATE,
   MODEL                Varchar2(30),
   PARTICIPANT_TIMESTAMP DATE,
   SUPPRESSED_AEMO      NUMBER(1,0),
   SUPPRESSED_PARTICIPANT NUMBER(1,0),
   TRANSACTION_ID       Varchar2(100)
)
   tablespace PSSTAB02;

comment on table INTERMITTENT_DS_RUN is
'Unconstrained Intermittent Generation Forecasts (UIGF) for Dispatch.';

comment on column INTERMITTENT_DS_RUN.RUN_DATETIME is
'Date and Time where the forecast applies (dispatch interval ending)';

comment on column INTERMITTENT_DS_RUN.DUID is
'DUID (or Area for non-scheduled) where this forecast applies';

comment on column INTERMITTENT_DS_RUN.OFFERDATETIME is
'Date and Time when this forecast submission was loaded.';

comment on column INTERMITTENT_DS_RUN.ORIGIN is
'Origin of this forecast (PARTICIPANTID, AWEFS/ASEFS, or another vendor)';

comment on column INTERMITTENT_DS_RUN.FORECAST_PRIORITY is
'Unsuppressed forecasts with higher priority values are used in Dispatch in preference to unsuppressed forecasts with lower priority values.';

comment on column INTERMITTENT_DS_RUN.AUTHORISEDBY is
'Authorising officer of this forecast (applicable for participant forecasts only). This column is not made available to the public.';

comment on column INTERMITTENT_DS_RUN.COMMENTS is
'Comments relating to the forecast. This column is not made available to the public.';

comment on column INTERMITTENT_DS_RUN.LASTCHANGED is
'Last date and time the record changed.';

comment on column INTERMITTENT_DS_RUN.MODEL is
'Metadata relating to the forecast. This column is not made available to the public.';

comment on column INTERMITTENT_DS_RUN.PARTICIPANT_TIMESTAMP is
'Participant can document when the forecast was created';

comment on column INTERMITTENT_DS_RUN.SUPPRESSED_AEMO is
'Was this forecast suppressed by AEMO? Suppressed = 1,Not suppressed =0';

comment on column INTERMITTENT_DS_RUN.SUPPRESSED_PARTICIPANT is
'Was this forecast suppressed by the participant? Suppressed submissions may not be used,  Suppressed = 1, Not suppressed =0';

comment on column INTERMITTENT_DS_RUN.TRANSACTION_ID is
'Uniquely identifies this interaction';

alter table INTERMITTENT_DS_RUN
   add constraint INTERMITTENT_DS_RUN_PK primary key (RUN_DATETIME, DUID, OFFERDATETIME, ORIGIN, FORECAST_PRIORITY)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: INTERMITTENT_FORECAST_TRK                             */
/*==============================================================*/
create table INTERMITTENT_FORECAST_TRK (
   SETTLEMENTDATE       DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   ORIGIN               VARCHAR2(20),
   FORECAST_PRIORITY    NUMBER(10,0),
   OFFERDATETIME        DATE
)
   tablespace PSSTAB02;

comment on table INTERMITTENT_FORECAST_TRK is
'Uniquely tracks which Intermittent Generation forecast was used for the DUID in which Dispatch run';

comment on column INTERMITTENT_FORECAST_TRK.SETTLEMENTDATE is
'DateTime of the Dispatch run (dispatch interval ending)';

comment on column INTERMITTENT_FORECAST_TRK.DUID is
'Tracks to INTERMITTENT_DS_RUN.DUID';

comment on column INTERMITTENT_FORECAST_TRK.ORIGIN is
'Tracks to INTERMITTENT_DS_RUN.ORIGIN, except when the forecast used is either SCADA or FCST or Last Target';

comment on column INTERMITTENT_FORECAST_TRK.FORECAST_PRIORITY is
'Tracks to INTERMITTENT_DS_RUN.FORECAST_PRIORITY, except for -1 which denotes SCADA or FCST, and 0 which denotes Last Target';

comment on column INTERMITTENT_FORECAST_TRK.OFFERDATETIME is
'Tracks to INTERMITTENT_DS_RUN.OFFERDATETIME';

alter table INTERMITTENT_FORECAST_TRK
   add constraint INTERMITTENT_FORECAST_TRK_PK primary key (SETTLEMENTDATE, DUID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: INTERMITTENT_GEN_FCST                                 */
/*==============================================================*/
create table INTERMITTENT_GEN_FCST (
   RUN_DATETIME         DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   START_INTERVAL_DATETIME DATE                  not null,
   END_INTERVAL_DATETIME DATE                  not null,
   VERSIONNO            NUMBER(10,0),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table INTERMITTENT_GEN_FCST is
'Identifying record for a given forecast of an intermittent generation. This table is the version table for the INTERMITTENT_GEN_FCST_DATA table which stores the individual forecast values. AEMO plans to remove this table in a Data Model version release after 5.6.';

comment on column INTERMITTENT_GEN_FCST.RUN_DATETIME is
'Date Time of forecast (AEST).';

comment on column INTERMITTENT_GEN_FCST.DUID is
'Identifier of the intermittent generator.';

comment on column INTERMITTENT_GEN_FCST.START_INTERVAL_DATETIME is
'Date Time (AEST) of the first half-hour interval being forecast.';

comment on column INTERMITTENT_GEN_FCST.END_INTERVAL_DATETIME is
'Date Time (AEST) of the final half-hour interval being forecast.';

comment on column INTERMITTENT_GEN_FCST.VERSIONNO is
'Versioning information for resolution back to AEMO''s wind generation forecasting system.';

comment on column INTERMITTENT_GEN_FCST.LASTCHANGED is
'Date Time record was created';

alter table INTERMITTENT_GEN_FCST
   add constraint PK_INTERMITTENT_GEN_FCST primary key (RUN_DATETIME, DUID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: INTERMITTENT_GEN_FCST_DATA                            */
/*==============================================================*/
create table INTERMITTENT_GEN_FCST_DATA (
   RUN_DATETIME         DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   INTERVAL_DATETIME    DATE                  not null,
   POWERMEAN            NUMBER(9,3),
   POWERPOE50           NUMBER(9,3),
   POWERPOELOW          NUMBER(9,3),
   POWERPOEHIGH         NUMBER(9,3),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table INTERMITTENT_GEN_FCST_DATA is
'Stores the forecast generation (MW) for each interval within a given forecast of an intermittent generator. AEMO plans to remove this table in a Data Model version release after 5.6.';

comment on column INTERMITTENT_GEN_FCST_DATA.RUN_DATETIME is
'Date Time of forecast (AEST).';

comment on column INTERMITTENT_GEN_FCST_DATA.DUID is
'Identifier of the intermittent generator';

comment on column INTERMITTENT_GEN_FCST_DATA.INTERVAL_DATETIME is
'Date Time (AEST) of the halfhour interval being forecast';

comment on column INTERMITTENT_GEN_FCST_DATA.POWERMEAN is
'The average forecast value in MW at the interval end';

comment on column INTERMITTENT_GEN_FCST_DATA.POWERPOE50 is
'50% probability of exceedance forecast value in MW at the interval end';

comment on column INTERMITTENT_GEN_FCST_DATA.POWERPOELOW is
'90% probability of exceedance forecast value in MW at the interval end';

comment on column INTERMITTENT_GEN_FCST_DATA.POWERPOEHIGH is
'10% probability of exceedance forecast value in MW at the interval end';

comment on column INTERMITTENT_GEN_FCST_DATA.LASTCHANGED is
'Date Time record was created';

alter table INTERMITTENT_GEN_FCST_DATA
   add constraint PK_INTERMITTENT_GEN_FCST_DATA primary key (RUN_DATETIME, DUID, INTERVAL_DATETIME)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: INTERMITTENT_GEN_FCST_P5_PRED                         */
/*==============================================================*/
create table INTERMITTENT_GEN_FCST_P5_PRED (
   FORECAST_RUN_DATETIME DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   OFFERDATETIME        DATE                  not null,
   PROVIDERID           VARCHAR2(20)          not null,
   FORECAST_PRIORITY    NUMBER(10,0)          not null,
   INTERVAL_DATETIME    DATE                  not null,
   FORECAST_TYPE        VARCHAR2(20)          not null,
   FORECAST_VALUE       NUMBER(18,8)
)
   tablespace PSSTAB03;

comment on table INTERMITTENT_GEN_FCST_P5_PRED is
'Contains forecast predictions for intermittent wind and solar units, with a 5-minute resolution over the hour-ahead P5MIN timeframe. This is the child table of the parent table INTERMITTENT_GEN_FCST_P5_RUN, which contains the corresponding forecast runs.';

comment on column INTERMITTENT_GEN_FCST_P5_PRED.FORECAST_RUN_DATETIME is
'Datetime (interval ending) when this forecast run is valid. It aligns with run_datetime in downstream processes, unless a forecast run is missed, in this case the previous run is used.';

comment on column INTERMITTENT_GEN_FCST_P5_PRED.DUID is
'Dispatchable unit identifier for which this forecast applies.';

comment on column INTERMITTENT_GEN_FCST_P5_PRED.OFFERDATETIME is
'Datetime when this forecast submission was loaded.';

comment on column INTERMITTENT_GEN_FCST_P5_PRED.PROVIDERID is
'Forecast provider identifier';

comment on column INTERMITTENT_GEN_FCST_P5_PRED.FORECAST_PRIORITY is
'Priority of forecast run, higher number is used in preference to lower number for the same provider.';

comment on column INTERMITTENT_GEN_FCST_P5_PRED.INTERVAL_DATETIME is
'Datetime (interval-ending) for the period that this forecast applies to, within the current forecast_run_datetime.';

comment on column INTERMITTENT_GEN_FCST_P5_PRED.FORECAST_TYPE is
'Type of forecast, for example, POE_10, POE_50, POE_90, MEAN and so on.';

comment on column INTERMITTENT_GEN_FCST_P5_PRED.FORECAST_VALUE is
'Forecast value in MW.';

alter table INTERMITTENT_GEN_FCST_P5_PRED
   add constraint INTERMIT_GEN_FCST_P5_PRED_PK primary key (FORECAST_RUN_DATETIME, DUID, OFFERDATETIME, PROVIDERID, FORECAST_PRIORITY, INTERVAL_DATETIME, FORECAST_TYPE)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: INTERMITTENT_GEN_FCST_P5_RUN                          */
/*==============================================================*/
create table INTERMITTENT_GEN_FCST_P5_RUN (
   FORECAST_RUN_DATETIME DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   OFFERDATETIME        DATE                  not null,
   PROVIDERID           VARCHAR2(20)          not null,
   FORECAST_PRIORITY    NUMBER(10,0)          not null,
   PROVIDER_TIMESTAMP   DATE,
   REMARKS              VARCHAR2(300),
   MODEL_USED           VARCHAR2(30),
   SUPPRESSED_PROVIDER  NUMBER(1,0),
   TRANSACTION_ID       VARCHAR2(100),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table INTERMITTENT_GEN_FCST_P5_RUN is
'Contains forecast runs for intermittent wind and solar units, with a 5-minute resolution over the hour-ahead P5MIN timeframe. This is the parent table to the child table INTERMITTENT_GEN_FCST_P5_PRED, which contains the corresponding forecast predictions over the full horizon.';

comment on column INTERMITTENT_GEN_FCST_P5_RUN.FORECAST_RUN_DATETIME is
'Datetime (interval ending) when this forecast run is valid. It aligns with run_datetime in downstream processes, unless a forecast run is missed, in this case the previous run is used.';

comment on column INTERMITTENT_GEN_FCST_P5_RUN.DUID is
'Dispatchable unit identifier for which this forecast applies.';

comment on column INTERMITTENT_GEN_FCST_P5_RUN.OFFERDATETIME is
'Datetime when this forecast submission was loaded.';

comment on column INTERMITTENT_GEN_FCST_P5_RUN.PROVIDERID is
'Forecast provider identifier';

comment on column INTERMITTENT_GEN_FCST_P5_RUN.FORECAST_PRIORITY is
'Priority of forecast run, higher number is used in preference to lower number for the same provider.';

comment on column INTERMITTENT_GEN_FCST_P5_RUN.PROVIDER_TIMESTAMP is
'Datetime when the provider created the forecast.';

comment on column INTERMITTENT_GEN_FCST_P5_RUN.REMARKS is
'Comments relating to the forecast run. This column is not made available to the public.';

comment on column INTERMITTENT_GEN_FCST_P5_RUN.MODEL_USED is
'Metadata describing the model used to produce the forecast run. This column is not made available to the public.';

comment on column INTERMITTENT_GEN_FCST_P5_RUN.SUPPRESSED_PROVIDER is
'Flag indicating if the forecast run was suppressed by the provider when submitted. Suppressed forecasts are not used by downstream systems. Suppressed = 1, Unsuppressed = 0.';

comment on column INTERMITTENT_GEN_FCST_P5_RUN.TRANSACTION_ID is
'Transaction identifier for receiving the forecast run.';

comment on column INTERMITTENT_GEN_FCST_P5_RUN.LASTCHANGED is
'Datetime when the forecast run was written into AEMO database.';

alter table INTERMITTENT_GEN_FCST_P5_RUN
   add constraint INTERMIT_GEN_FCST_P5_RUN_PK primary key (FORECAST_RUN_DATETIME, DUID, OFFERDATETIME, PROVIDERID, FORECAST_PRIORITY)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: INTERMITTENT_GEN_FCST_PRED                            */
/*==============================================================*/
create table INTERMITTENT_GEN_FCST_PRED (
   FORECAST_RUN_DATETIME DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   OFFERDATETIME        DATE                  not null,
   PROVIDERID           VARCHAR2(20)          not null,
   FORECAST_PRIORITY    NUMBER(10,0)          not null,
   INTERVAL_DATETIME    DATE                  not null,
   FORECAST_TYPE        VARCHAR2(20)          not null,
   FORECAST_VALUE       NUMBER(18,8)
)
   tablespace PSSTAB03;

comment on table INTERMITTENT_GEN_FCST_PRED is
'Contains forecast predictions for intermittent wind and solar units, with a 30-minute resolution over the week-ahead PD/STPASA timeframe. This is the child table of the parent table INTERMITTENT_GEN_FCST_RUN, which contains the corresponding forecast runs.';

comment on column INTERMITTENT_GEN_FCST_PRED.FORECAST_RUN_DATETIME is
'Datetime (interval ending) when this forecast run is valid. It aligns with run_datetime in downstream processes, unless a forecast run is missed, in this case the previous run is used.';

comment on column INTERMITTENT_GEN_FCST_PRED.DUID is
'Dispatchable unit identifier for which this forecast applies.';

comment on column INTERMITTENT_GEN_FCST_PRED.OFFERDATETIME is
'Datetime when this forecast submission was loaded.';

comment on column INTERMITTENT_GEN_FCST_PRED.PROVIDERID is
'Forecast provider identifier';

comment on column INTERMITTENT_GEN_FCST_PRED.FORECAST_PRIORITY is
'Priority of forecast run, higher number is used in preference to lower number for the same provider.';

comment on column INTERMITTENT_GEN_FCST_PRED.INTERVAL_DATETIME is
'Datetime (interval-ending) for the period that this forecast applies to, within the current forecast_run_datetime.';

comment on column INTERMITTENT_GEN_FCST_PRED.FORECAST_TYPE is
'Type of forecast, for example, POE_10, POE_50, POE_90, MEAN and so on.';

comment on column INTERMITTENT_GEN_FCST_PRED.FORECAST_VALUE is
'Forecast value in MW.';

alter table INTERMITTENT_GEN_FCST_PRED
   add constraint INTERMIT_GEN_FCST_PRED_PK primary key (FORECAST_RUN_DATETIME, DUID, OFFERDATETIME, PROVIDERID, FORECAST_PRIORITY, INTERVAL_DATETIME, FORECAST_TYPE)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: INTERMITTENT_GEN_FCST_RUN                             */
/*==============================================================*/
create table INTERMITTENT_GEN_FCST_RUN (
   FORECAST_RUN_DATETIME DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   OFFERDATETIME        DATE                  not null,
   PROVIDERID           VARCHAR2(20)          not null,
   FORECAST_PRIORITY    NUMBER(10,0)          not null,
   PROVIDER_TIMESTAMP   DATE,
   REMARKS              VARCHAR2(300),
   MODEL_USED           VARCHAR2(30),
   SUPPRESSED_PROVIDER  NUMBER(1,0),
   TRANSACTION_ID       VARCHAR2(100),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table INTERMITTENT_GEN_FCST_RUN is
'Contains forecast runs for intermittent wind and solar units, with a 30-minute resolution over the week-ahead PD/STPASA timeframe. This is the parent table to the child table INTERMITTENT_GEN_FCST_PRED, which contains the corresponding forecast predictions over the full horizon.';

comment on column INTERMITTENT_GEN_FCST_RUN.FORECAST_RUN_DATETIME is
'Datetime (interval ending) when this forecast run is valid. It aligns with run_datetime in downstream processes, unless a forecast run is missed, in this case the previous run is used.';

comment on column INTERMITTENT_GEN_FCST_RUN.DUID is
'Dispatchable unit identifier for which this forecast applies.';

comment on column INTERMITTENT_GEN_FCST_RUN.OFFERDATETIME is
'Datetime when this forecast submission was loaded.';

comment on column INTERMITTENT_GEN_FCST_RUN.PROVIDERID is
'Forecast provider identifier';

comment on column INTERMITTENT_GEN_FCST_RUN.FORECAST_PRIORITY is
'Priority of forecast run, higher number is used in preference to lower number for the same provider.';

comment on column INTERMITTENT_GEN_FCST_RUN.PROVIDER_TIMESTAMP is
'Datetime when the provider created the forecast.';

comment on column INTERMITTENT_GEN_FCST_RUN.REMARKS is
'Comments relating to the forecast run. This column is not made available to the public.';

comment on column INTERMITTENT_GEN_FCST_RUN.MODEL_USED is
'Metadata describing the model used to produce the forecast run. This column is not made available to the public.';

comment on column INTERMITTENT_GEN_FCST_RUN.SUPPRESSED_PROVIDER is
'Flag indicating if the forecast run was suppressed by the provider when submitted. Suppressed forecasts are not used by downstream systems. Suppressed = 1, Unsuppressed = 0.';

comment on column INTERMITTENT_GEN_FCST_RUN.TRANSACTION_ID is
'Transaction identifier for receiving the forecast run';

comment on column INTERMITTENT_GEN_FCST_RUN.LASTCHANGED is
'Datetime when the forecast run was written into AEMO database.';

alter table INTERMITTENT_GEN_FCST_RUN
   add constraint INTERMIT_GEN_FCST_RUN_PK primary key (FORECAST_RUN_DATETIME, DUID, OFFERDATETIME, PROVIDERID, FORECAST_PRIORITY)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: INTERMITTENT_GEN_LIMIT                                */
/*==============================================================*/
create table INTERMITTENT_GEN_LIMIT (
   TRADINGDATE          DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   OFFERDATETIME        DATE                  not null,
   PERIODID             NUMBER(3,0)           not null,
   UPPERMWLIMIT         NUMBER(6)
)
   tablespace PSSTAB02;

comment on table INTERMITTENT_GEN_LIMIT is
'A submission of Upper MW Limit for an intermittent generating unit, by Trading Day and Trading Interval';

comment on column INTERMITTENT_GEN_LIMIT.TRADINGDATE is
'Trading Day for which this unit availability submission applies';

comment on column INTERMITTENT_GEN_LIMIT.DUID is
'Unique Identifier of Dispatchable Unit';

comment on column INTERMITTENT_GEN_LIMIT.OFFERDATETIME is
'Date and Time when this unit availability submission was loaded';

comment on column INTERMITTENT_GEN_LIMIT.PERIODID is
'Trading interval number (1...48) within this TRADINGDATE for which UPPERMWLIMIT applies';

comment on column INTERMITTENT_GEN_LIMIT.UPPERMWLIMIT is
'Maximum imposed MW limit (down regulation in AWEFS/ASEFS). Value between 0 and the registered DUID Maximum Capacity. Value = -1 means no limit applies.';

alter table INTERMITTENT_GEN_LIMIT
   add constraint INTERMITTENT_GEN_LIMIT_PK primary key (TRADINGDATE, DUID, OFFERDATETIME, PERIODID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: INTERMITTENT_GEN_LIMIT_DAY                            */
/*==============================================================*/
create table INTERMITTENT_GEN_LIMIT_DAY (
   TRADINGDATE          DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   OFFERDATETIME        DATE                  not null,
   PARTICIPANTID        VARCHAR2(20),
   LASTCHANGED          DATE,
   AUTHORISEDBYUSER     VARCHAR2(20),
   AUTHORISEDBYPARTICIPANTID VARCHAR2(20)
)
   tablespace PSSTAB02;

comment on table INTERMITTENT_GEN_LIMIT_DAY is
'Summary record for an Upper MW Limit submission for an intermittent generating unit for a Trading Day';

comment on column INTERMITTENT_GEN_LIMIT_DAY.TRADINGDATE is
'Trading Day for which this unit availability submission applies';

comment on column INTERMITTENT_GEN_LIMIT_DAY.DUID is
'Unique Identifier of Dispatchable Unit';

comment on column INTERMITTENT_GEN_LIMIT_DAY.OFFERDATETIME is
'Date and Time when this unit availability submission was loaded';

comment on column INTERMITTENT_GEN_LIMIT_DAY.PARTICIPANTID is
'Unique participant identifier';

comment on column INTERMITTENT_GEN_LIMIT_DAY.LASTCHANGED is
'Last date and time record changed';

comment on column INTERMITTENT_GEN_LIMIT_DAY.AUTHORISEDBYUSER is
'User entering the unit availability submission';

comment on column INTERMITTENT_GEN_LIMIT_DAY.AUTHORISEDBYPARTICIPANTID is
'Participant entering the unit availability submission';

alter table INTERMITTENT_GEN_LIMIT_DAY
   add constraint INTERMITTENT_GEN_LIMIT_DAY_PK primary key (TRADINGDATE, DUID, OFFERDATETIME)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: INTERMITTENT_GEN_SCADA                                */
/*==============================================================*/
create table INTERMITTENT_GEN_SCADA (
   RUN_DATETIME         DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   SCADA_TYPE           VARCHAR2(20)          not null,
   SCADA_VALUE          NUMBER(15,5),
   SCADA_QUALITY        VARCHAR2(20),
   LASTCHANGED          DATE
)
   tablespace PSSTAB03;

comment on table INTERMITTENT_GEN_SCADA is
'INTERMITTENT_GEN_SCADA provides the SCADA Availability for every intermittent generating unit, including Elements Available (wind turbines/solar inverters) and Local Limit';

comment on column INTERMITTENT_GEN_SCADA.RUN_DATETIME is
'Date Time of the dispatch interval (interval ending)';

comment on column INTERMITTENT_GEN_SCADA.DUID is
'Dispatchable Unit Identifier';

comment on column INTERMITTENT_GEN_SCADA.SCADA_TYPE is
'SCADA snapshot for intermittent generating unit at start of interval for a specified SCADA signal type. ELAV = Total Elements Available (# turbines for wind farms, # inverters for solar farms); LOCL = Local Limit (MW).';

comment on column INTERMITTENT_GEN_SCADA.SCADA_VALUE is
'SCADA value snapshot for intermittent generating unit at start of interval for a specified SCADA signal type.';

comment on column INTERMITTENT_GEN_SCADA.SCADA_QUALITY is
'SCADA quality snapshot for intermittent generating unit at start of interval for a specified SCADA signal type.';

comment on column INTERMITTENT_GEN_SCADA.LASTCHANGED is
'Last date and time record changed';

alter table INTERMITTENT_GEN_SCADA
   add constraint INTERMITTENT_GEN_SCADA_PK primary key (RUN_DATETIME, DUID, SCADA_TYPE)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: INTERMITTENT_P5_PRED                                  */
/*==============================================================*/
create table INTERMITTENT_P5_PRED (
   RUN_DATETIME         DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   OFFERDATETIME        DATE                  not null,
   INTERVAL_DATETIME    DATE                  not null,
   ORIGIN               VARCHAR2(20)          not null,
   FORECAST_PRIORITY    NUMBER(10,0)          not null,
   FORECAST_MEAN        NUMBER(18,8),
   FORECAST_POE10       NUMBER(18,8),
   FORECAST_POE50       NUMBER(18,8),
   FORECAST_POE90       NUMBER(18,8)
)
   tablespace PSSTAB02;

comment on table INTERMITTENT_P5_PRED is
'Unconstrained Intermittent Generation Forecasts (UIGF) for 5-Minute Pre-dispatch. AEMO plans to remove this table in a Data Model version release after 5.6.';

comment on column INTERMITTENT_P5_PRED.RUN_DATETIME is
'Date and Time of the first interval of 5-Minute Predispatch where the forecast applies (dispatch interval ending)';

comment on column INTERMITTENT_P5_PRED.DUID is
'DUID (or Area for non-scheduled) where this forecast applies';

comment on column INTERMITTENT_P5_PRED.OFFERDATETIME is
'Date and Time when this forecast submission was loaded';

comment on column INTERMITTENT_P5_PRED.INTERVAL_DATETIME is
'Interval within the current RUN_DATETIME where this forecast applies (dispatch interval ending)';

comment on column INTERMITTENT_P5_PRED.ORIGIN is
'Origin of this forecast (PARTICIPANTID, AWEFS/ASEFS, or another vendor)';

comment on column INTERMITTENT_P5_PRED.FORECAST_PRIORITY is
'Unsuppressed forecasts with higher priority values are used in 5-Minute Predispatch in preference to unsuppressed forecasts with lower priority values';

comment on column INTERMITTENT_P5_PRED.FORECAST_MEAN is
'Forecast MW value for this interval_DateTime';

comment on column INTERMITTENT_P5_PRED.FORECAST_POE10 is
'Forecast 10% POE MW value for this interval_DateTime';

comment on column INTERMITTENT_P5_PRED.FORECAST_POE50 is
'Forecast 50% POE MW value for this interval_DateTime.';

comment on column INTERMITTENT_P5_PRED.FORECAST_POE90 is
'Forecast 90% POE MW value for this interval_DateTime';

alter table INTERMITTENT_P5_PRED
   add constraint INTERMITTENT_P5_PRED_PK primary key (RUN_DATETIME, DUID, OFFERDATETIME, INTERVAL_DATETIME, ORIGIN, FORECAST_PRIORITY)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: INTERMITTENT_P5_RUN                                   */
/*==============================================================*/
create table INTERMITTENT_P5_RUN (
   RUN_DATETIME         DATE                  not null,
   DUID                 Varchar2(20)          not null,
   OFFERDATETIME        DATE                  not null,
   ORIGIN               Varchar2(20)          not null,
   FORECAST_PRIORITY    NUMBER(10,0)          not null,
   AUTHORISEDBY         Varchar2(20),
   COMMENTS             Varchar2(200),
   LASTCHANGED          DATE,
   MODEL                Varchar2(30),
   PARTICIPANT_TIMESTAMP DATE,
   SUPPRESSED_AEMO      NUMBER(1,0),
   SUPPRESSED_PARTICIPANT NUMBER(1,0),
   TRANSACTION_ID       Varchar2(100)
)
   tablespace PSSTAB02;

comment on table INTERMITTENT_P5_RUN is
'Unconstrained Intermittent Generation Forecasts (UIGF) for 5-Minute Pre-dispatch. AEMO plans to remove this table in a Data Model version release after 5.6.';

comment on column INTERMITTENT_P5_RUN.RUN_DATETIME is
'Date and Time of the first interval of 5-minute pre-dispatch where the forecast applies.';

comment on column INTERMITTENT_P5_RUN.DUID is
'DUID (or Area for non-scheduled) where this forecast applies';

comment on column INTERMITTENT_P5_RUN.OFFERDATETIME is
'Date and Time when this forecast submission was loaded';

comment on column INTERMITTENT_P5_RUN.ORIGIN is
'Origin of this forecast (PARTICIPANTID, AWEFS/ASEFS, or another vendor)';

comment on column INTERMITTENT_P5_RUN.FORECAST_PRIORITY is
'Unsuppressed forecasts with higher priority values are used in 5-Minute Predispatch in preference to unsuppressed forecasts with lower priority values';

comment on column INTERMITTENT_P5_RUN.AUTHORISEDBY is
'Authorising officer of this forecast';

comment on column INTERMITTENT_P5_RUN.COMMENTS is
'Comments relating to the forecast';

comment on column INTERMITTENT_P5_RUN.LASTCHANGED is
'Last date and time the record changed.';

comment on column INTERMITTENT_P5_RUN.MODEL is
'Metadata relating to the forecast.';

comment on column INTERMITTENT_P5_RUN.PARTICIPANT_TIMESTAMP is
'Participant can document when the forecast was created';

comment on column INTERMITTENT_P5_RUN.SUPPRESSED_AEMO is
'Was this forecast suppressed by AEMO? Suppressed = 1,Not suppressed =0';

comment on column INTERMITTENT_P5_RUN.SUPPRESSED_PARTICIPANT is
'Was this forecast suppressed by the participant? Suppressed submissions may not be used, Suppressed = 1, Not suppressed =0';

comment on column INTERMITTENT_P5_RUN.TRANSACTION_ID is
'Uniquely identifies this interaction';

alter table INTERMITTENT_P5_RUN
   add constraint INTERMITTENT_P5_RUN_PK primary key (RUN_DATETIME, DUID, OFFERDATETIME, ORIGIN, FORECAST_PRIORITY)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: INTRAREGIONALLOC                                      */
/*==============================================================*/
create table INTRAREGIONALLOC (
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(5,0)           not null,
   REGIONID             VARCHAR2(10)          not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   ALLOCATION           NUMBER(12,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table INTRAREGIONALLOC is
'INTRAREGIONALLOC shows allocations of intra-regional residues to participants.';

comment on column INTRAREGIONALLOC.EFFECTIVEDATE is
'Effective Date of Allocation Details';

comment on column INTRAREGIONALLOC.VERSIONNO is
'Version No in respect to effective date';

comment on column INTRAREGIONALLOC.REGIONID is
'Region Identifier';

comment on column INTRAREGIONALLOC.PARTICIPANTID is
'Unique participant identifier';

comment on column INTRAREGIONALLOC.ALLOCATION is
'Allocation Percent / 100';

comment on column INTRAREGIONALLOC.LASTCHANGED is
'Last changed date/time';

alter table INTRAREGIONALLOC
   add constraint INTRAREGIONALLOC_PK primary key (EFFECTIVEDATE, VERSIONNO, REGIONID, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: INTRAREGIONALLOC_LCX                                  */
/*==============================================================*/
create index INTRAREGIONALLOC_LCX on INTRAREGIONALLOC (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: IRFMAMOUNT                                            */
/*==============================================================*/
create table IRFMAMOUNT (
   IRFMID               VARCHAR2(10)          not null,
   EFFECTIVEDATE        DATE,
   VERSIONNO            NUMBER(3,0)           not null,
   PERIODID             NUMBER(4,0)           not null,
   AMOUNT               NUMBER(15,5),
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table IRFMAMOUNT is
'IRFMAMOUNT sets out settlement amounts associated with Industrial Relations Forced Majeure events.';

comment on column IRFMAMOUNT.IRFMID is
'Unique Industrial Relations Force Majeure event';

comment on column IRFMAMOUNT.EFFECTIVEDATE is
'Date of event';

comment on column IRFMAMOUNT.VERSIONNO is
'Version number of record of event';

comment on column IRFMAMOUNT.PERIODID is
'Settlement period';

comment on column IRFMAMOUNT.AMOUNT is
'Total settlement amount in $';

comment on column IRFMAMOUNT.AUTHORISEDBY is
'Person authorising amount';

comment on column IRFMAMOUNT.AUTHORISEDDATE is
'Authorised date';

comment on column IRFMAMOUNT.LASTCHANGED is
'last changed';

alter table IRFMAMOUNT
   add constraint IRFMAMOUNT_PK primary key (IRFMID, VERSIONNO, PERIODID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: IRFMAMOUNT_LCX                                        */
/*==============================================================*/
create index IRFMAMOUNT_LCX on IRFMAMOUNT (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: IRFMEVENTS                                            */
/*==============================================================*/
create table IRFMEVENTS (
   IRFMID               VARCHAR2(10)          not null,
   STARTDATE            DATE,
   STARTPERIOD          NUMBER(3,0),
   ENDDATE              DATE,
   ENDPERIOD            NUMBER(3,0),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table IRFMEVENTS is
'IRFMEVENTS sets out specific Industrial Relations Forced Majeure events.';

alter table IRFMEVENTS
   add constraint IRFMEVENTS_PK primary key (IRFMID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: IRFMEVENTS_LCX                                        */
/*==============================================================*/
create index IRFMEVENTS_LCX on IRFMEVENTS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: LOSSFACTORMODEL                                       */
/*==============================================================*/
create table LOSSFACTORMODEL (
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   REGIONID             VARCHAR2(10)          not null,
   DEMANDCOEFFICIENT    NUMBER(27,17),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table LOSSFACTORMODEL is
'LOSSFACTORMODEL sets out the demand coefficients for each interconnector, used by LP Solver modelling of interconnector flows.';

comment on column LOSSFACTORMODEL.EFFECTIVEDATE is
'Calendar date data set is effective';

comment on column LOSSFACTORMODEL.VERSIONNO is
'Version number within effective date of the status proposed';

comment on column LOSSFACTORMODEL.INTERCONNECTORID is
'The unique identifier for the interconnector.';

comment on column LOSSFACTORMODEL.REGIONID is
'The unique region identifier for a connection point of the interconnector';

comment on column LOSSFACTORMODEL.DEMANDCOEFFICIENT is
'The coefficient applied to the region demand in the calculation of the interconnector loss factor';

comment on column LOSSFACTORMODEL.LASTCHANGED is
'Last date and time record changed';

alter table LOSSFACTORMODEL
   add constraint LFMOD_PK primary key (EFFECTIVEDATE, VERSIONNO, INTERCONNECTORID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: LOSSFACTORMODEL_LCX                                   */
/*==============================================================*/
create index LOSSFACTORMODEL_LCX on LOSSFACTORMODEL (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: LOSSMODEL                                             */
/*==============================================================*/
create table LOSSMODEL (
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   PERIODID             VARCHAR2(20),
   LOSSSEGMENT          NUMBER(6,0)           not null,
   MWBREAKPOINT         NUMBER(6,0),
   LOSSFACTOR           NUMBER(16,6),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table LOSSMODEL is
'LOSSMODEL sets out segment breakpoints in loss model for each interconnector, used by LP Solver modelling of interconnector flows.';

comment on column LOSSMODEL.EFFECTIVEDATE is
'Calendar date data set is effective';

comment on column LOSSMODEL.VERSIONNO is
'Version number within effective date';

comment on column LOSSMODEL.INTERCONNECTORID is
'Interconnector identifier';

comment on column LOSSMODEL.PERIODID is
'Not used';

comment on column LOSSMODEL.LOSSSEGMENT is
'Segment Identifier (1 to 80 at present)';

comment on column LOSSMODEL.MWBREAKPOINT is
'MW Value for segment';

comment on column LOSSMODEL.LOSSFACTOR is
'Not used';

comment on column LOSSMODEL.LASTCHANGED is
'Last date and time record changed';

alter table LOSSMODEL
   add constraint LOSSMODEL_PK primary key (EFFECTIVEDATE, VERSIONNO, INTERCONNECTORID, LOSSSEGMENT)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: LOSSMODEL_LCX                                         */
/*==============================================================*/
create index LOSSMODEL_LCX on LOSSMODEL (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MARKETFEE                                             */
/*==============================================================*/
create table MARKETFEE (
   MARKETFEEID          VARCHAR2(10)          not null,
   MARKETFEEPERIOD      VARCHAR2(20),
   MARKETFEETYPE        VARCHAR2(12),
   DESCRIPTION          VARCHAR2(64),
   LASTCHANGED          DATE,
   GL_TCODE             VARCHAR2(15),
   GL_FINANCIALCODE     VARCHAR2(10),
   FEE_CLASS            VARCHAR2(40),
   METER_TYPE           VARCHAR2(20),
   METER_SUBTYPE        VARCHAR2(20)
)
   tablespace PSSTAB01;

comment on table MARKETFEE is
'MARKETFEE sets out fee type and period for each market fee.';

comment on column MARKETFEE.MARKETFEEID is
'Identifier for Market Fee';

comment on column MARKETFEE.MARKETFEEPERIOD is
'Period type - PERIOD, DAILY, WEEKLY';

comment on column MARKETFEE.MARKETFEETYPE is
'Type - MW or $';

comment on column MARKETFEE.DESCRIPTION is
'Description of market fee';

comment on column MARKETFEE.LASTCHANGED is
'Last date and time record changed';

comment on column MARKETFEE.METER_TYPE is
'The Energy Type for the Market Fees Calculation. E.g of Meter Types are CUSTOMER, GENERATOR, NREG, BDU etc. If Meter Type is mentioned as ALL then all the Meter Types for that Participant Category will be used in the Fee calculation';

comment on column MARKETFEE.METER_SUBTYPE is
'The Meter Sub Type values are ACE, ASOE or ALL. ACE represent ACE_MWH value , ASOE represent ASOE_MWH value and ALL represent sum of ACE_MWh and ASOE_MWh';

alter table MARKETFEE
   add constraint MARKETFEE_PK primary key (MARKETFEEID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MARKETFEE_LCX                                         */
/*==============================================================*/
create index MARKETFEE_LCX on MARKETFEE (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MARKETFEEDATA                                         */
/*==============================================================*/
create table MARKETFEEDATA (
   MARKETFEEID          VARCHAR2(10)          not null,
   MARKETFEEVERSIONNO   NUMBER(3,0)           not null,
   EFFECTIVEDATE        DATE                  not null,
   MARKETFEEVALUE       NUMBER(22,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MARKETFEEDATA is
'MARKETFEEDATA sets out actual fee rates, as adjusted from time to time.';

comment on column MARKETFEEDATA.MARKETFEEID is
'Identifier for Market Fee';

comment on column MARKETFEEDATA.MARKETFEEVERSIONNO is
'Version of fees for this id';

comment on column MARKETFEEDATA.EFFECTIVEDATE is
'Date on which this data becomes effective';

comment on column MARKETFEEDATA.MARKETFEEVALUE is
'Market fee rate/MWh, a dollar amount';

comment on column MARKETFEEDATA.LASTCHANGED is
'Last date and time record changed';

alter table MARKETFEEDATA
   add constraint MARKETFEEDATA_PK primary key (MARKETFEEID, MARKETFEEVERSIONNO, EFFECTIVEDATE)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MARKETFEEDATA_LCX                                     */
/*==============================================================*/
create index MARKETFEEDATA_LCX on MARKETFEEDATA (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MARKETFEETRK                                          */
/*==============================================================*/
create table MARKETFEETRK (
   MARKETFEEVERSIONNO   NUMBER(3,0)           not null,
   EFFECTIVEDATE        DATE                  not null,
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MARKETFEETRK is
'MARKETFEETRK sets out versions of each market fee used and its effective date.';

comment on column MARKETFEETRK.MARKETFEEVERSIONNO is
'Version of fees for this ID';

comment on column MARKETFEETRK.EFFECTIVEDATE is
'Effective Date of Market notice';

comment on column MARKETFEETRK.AUTHORISEDBY is
'User authorising record';

comment on column MARKETFEETRK.AUTHORISEDDATE is
'Date record authorised';

comment on column MARKETFEETRK.LASTCHANGED is
'Last date and time record changed';

alter table MARKETFEETRK
   add constraint MARKETFEETRK_PK primary key (MARKETFEEVERSIONNO, EFFECTIVEDATE)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MARKETFEETRK_LCX                                      */
/*==============================================================*/
create index MARKETFEETRK_LCX on MARKETFEETRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MARKETNOTICEDATA                                      */
/*==============================================================*/
create table MARKETNOTICEDATA (
   NOTICEID             NUMBER(10,0)          not null,
   EFFECTIVEDATE        DATE,
   TYPEID               VARCHAR2(25),
   NOTICETYPE           VARCHAR2(25),
   LASTCHANGED          DATE,
   REASON               VARCHAR2(2000),
   EXTERNALREFERENCE    VARCHAR2(255)
)
   tablespace PSSTAB01;

comment on table MARKETNOTICEDATA is
'MARKETNOTICEDATA shows market notices data provided to all participants (market) and specific participants (participant).';

comment on column MARKETNOTICEDATA.NOTICEID is
'Notice Identifier';

comment on column MARKETNOTICEDATA.EFFECTIVEDATE is
'Effective Date of Market notice';

comment on column MARKETNOTICEDATA.TYPEID is
'Market Notice Type Identifier (Market - all participants. Participant - selected participants)';

comment on column MARKETNOTICEDATA.NOTICETYPE is
'Market Notice Type';

comment on column MARKETNOTICEDATA.LASTCHANGED is
'Last date and time record changed';

comment on column MARKETNOTICEDATA.REASON is
'Detail of market notices.';

comment on column MARKETNOTICEDATA.EXTERNALREFERENCE is
'External Reference for extra data pertaining to market notice';

alter table MARKETNOTICEDATA
   add constraint MARKETNOTICEDATA_PK primary key (NOTICEID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MARKETNOTICEDATA_LCX                                  */
/*==============================================================*/
create index MARKETNOTICEDATA_LCX on MARKETNOTICEDATA (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MARKETNOTICETYPE                                      */
/*==============================================================*/
create table MARKETNOTICETYPE (
   TYPEID               VARCHAR2(25)          not null,
   DESCRIPTION          VARCHAR2(64),
   RAISEDBY             VARCHAR2(10),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MARKETNOTICETYPE is
'MARKETNOTICETYPE sets out the different types of market notices (e.g. market systems).';

comment on column MARKETNOTICETYPE.TYPEID is
'Identifier for market notice type';

comment on column MARKETNOTICETYPE.DESCRIPTION is
'Type description';

comment on column MARKETNOTICETYPE.RAISEDBY is
'Not used';

comment on column MARKETNOTICETYPE.LASTCHANGED is
'Last date and time record changed';

alter table MARKETNOTICETYPE
   add constraint MARKETNOTICETYPE_PK primary key (TYPEID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MARKETNOTICETYPE_LCX                                  */
/*==============================================================*/
create index MARKETNOTICETYPE_LCX on MARKETNOTICETYPE (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MARKETSUSPENSION                                      */
/*==============================================================*/
create table MARKETSUSPENSION (
   SUSPENSIONID         VARCHAR2(10)          not null,
   STARTDATE            DATE,
   STARTPERIOD          NUMBER(3,0),
   ENDDATE              DATE,
   ENDPERIOD            NUMBER(3,0),
   REASON               VARCHAR2(64),
   STARTAUTHORISEDBY    VARCHAR2(15),
   ENDAUTHORISEDBY      VARCHAR2(15),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MARKETSUSPENSION is
'MARKETSUSPENSION is obsolete from 2017 End of Year DM4.27 Release. 
MARKETSUSPENSION sets out a start and end periods of any market suspension and the reason.';

comment on column MARKETSUSPENSION.SUSPENSIONID is
'Unique identifier for suspension';

comment on column MARKETSUSPENSION.STARTDATE is
'Start date of suspension';

comment on column MARKETSUSPENSION.STARTPERIOD is
'Start trading interval of suspension';

comment on column MARKETSUSPENSION.ENDDATE is
'End Date of suspension';

comment on column MARKETSUSPENSION.ENDPERIOD is
'End trading interval of suspension';

comment on column MARKETSUSPENSION.REASON is
'Reason for suspension';

comment on column MARKETSUSPENSION.STARTAUTHORISEDBY is
'User authorising start';

comment on column MARKETSUSPENSION.ENDAUTHORISEDBY is
'User authorising end';

comment on column MARKETSUSPENSION.LASTCHANGED is
'Last date and time record changed';

alter table MARKETSUSPENSION
   add constraint MARKETSUSPENSION_PK primary key (SUSPENSIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MARKETSUSPENSION_LCX                                  */
/*==============================================================*/
create index MARKETSUSPENSION_LCX on MARKETSUSPENSION (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MARKETSUSREGION                                       */
/*==============================================================*/
create table MARKETSUSREGION (
   SUSPENSIONID         VARCHAR2(10)          not null,
   REGIONID             VARCHAR2(10)          not null,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MARKETSUSREGION is
'MARKETSUSREGION  is obsolete from 2017 End of Year DM4.27 Release. 
MARKETSUSREGION sets out a regions affected by a market suspension.';

comment on column MARKETSUSREGION.SUSPENSIONID is
'Unique identifier of suspension';

comment on column MARKETSUSREGION.REGIONID is
'Differentiates this region from all other regions';

comment on column MARKETSUSREGION.LASTCHANGED is
'Last date and time record changed';

alter table MARKETSUSREGION
   add constraint MARKETSUSREGION_PK primary key (SUSPENSIONID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MARKETSUSREGION_LCX                                   */
/*==============================================================*/
create index MARKETSUSREGION_LCX on MARKETSUSREGION (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MARKET_FEE_CAT_EXCL                                   */
/*==============================================================*/
create table MARKET_FEE_CAT_EXCL (
   MARKETFEEID          VARCHAR2(20)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSION_DATETIME     DATE                  not null,
   PARTICIPANT_CATEGORYID VARCHAR2(20)          not null
)
   tablespace PSSTAB01;

comment on table MARKET_FEE_CAT_EXCL is
'Market fee exclusions for participant categories.';

comment on column MARKET_FEE_CAT_EXCL.MARKETFEEID is
'The excluded market fee';

comment on column MARKET_FEE_CAT_EXCL.EFFECTIVEDATE is
'The date the exclusion is effective from';

comment on column MARKET_FEE_CAT_EXCL.VERSION_DATETIME is
'The version information for this record';

comment on column MARKET_FEE_CAT_EXCL.PARTICIPANT_CATEGORYID is
'Participant category to be excluded from this market fee';

alter table MARKET_FEE_CAT_EXCL
   add constraint PK_MARKET_FEE_CAT_EXCL primary key (MARKETFEEID, EFFECTIVEDATE, VERSION_DATETIME, PARTICIPANT_CATEGORYID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: MARKET_FEE_CAT_EXCL_TRK                               */
/*==============================================================*/
create table MARKET_FEE_CAT_EXCL_TRK (
   MARKETFEEID          VARCHAR2(20)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSION_DATETIME     DATE                  not null,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MARKET_FEE_CAT_EXCL_TRK is
'Tracking table for market fee exclusions for participant categories.';

comment on column MARKET_FEE_CAT_EXCL_TRK.MARKETFEEID is
'The excluded market fee';

comment on column MARKET_FEE_CAT_EXCL_TRK.EFFECTIVEDATE is
'The date the exclusion is effective from';

comment on column MARKET_FEE_CAT_EXCL_TRK.VERSION_DATETIME is
'The version information for this record';

comment on column MARKET_FEE_CAT_EXCL_TRK.LASTCHANGED is
'Last date and time the record changed';

alter table MARKET_FEE_CAT_EXCL_TRK
   add constraint PK_MARKET_FEE_CAT_EXCL_TRK primary key (MARKETFEEID, EFFECTIVEDATE, VERSION_DATETIME)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: MARKET_FEE_EXCLUSION                                  */
/*==============================================================*/
create table MARKET_FEE_EXCLUSION (
   PARTICIPANTID        VARCHAR2(10)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   MARKETFEEID          VARCHAR2(10)          not null,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MARKET_FEE_EXCLUSION is
'MARKET_FEE_EXCLUSION shows the list of market fees from which a participant is excluded from funding after a particular settlement date.';

comment on column MARKET_FEE_EXCLUSION.PARTICIPANTID is
'Unique participant identifier';

comment on column MARKET_FEE_EXCLUSION.EFFECTIVEDATE is
'Date on which this data becomes effective';

comment on column MARKET_FEE_EXCLUSION.VERSIONNO is
'Version of fees for this ID';

comment on column MARKET_FEE_EXCLUSION.MARKETFEEID is
'Identifier for Market Fee';

comment on column MARKET_FEE_EXCLUSION.LASTCHANGED is
'Last date and time record changed';

alter table MARKET_FEE_EXCLUSION
   add constraint MARKET_FEE_EXCLUSION_PK primary key (PARTICIPANTID, EFFECTIVEDATE, VERSIONNO, MARKETFEEID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MARKET_FEE_EXCLUSION_LCX                              */
/*==============================================================*/
create index MARKET_FEE_EXCLUSION_LCX on MARKET_FEE_EXCLUSION (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MARKET_FEE_EXCLUSIONTRK                               */
/*==============================================================*/
create table MARKET_FEE_EXCLUSIONTRK (
   PARTICIPANTID        VARCHAR2(10)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MARKET_FEE_EXCLUSIONTRK is
'MARKET_FEE_EXCLUSIONTRK shows authorisation details of participant market fee exclusion data sets.';

comment on column MARKET_FEE_EXCLUSIONTRK.PARTICIPANTID is
'Unique participant identifier';

comment on column MARKET_FEE_EXCLUSIONTRK.EFFECTIVEDATE is
'Date on which this data becomes effective';

comment on column MARKET_FEE_EXCLUSIONTRK.VERSIONNO is
'Version of fees for this ID';

comment on column MARKET_FEE_EXCLUSIONTRK.AUTHORISEDBY is
'User authorising record';

comment on column MARKET_FEE_EXCLUSIONTRK.AUTHORISEDDATE is
'Date record authorised';

comment on column MARKET_FEE_EXCLUSIONTRK.LASTCHANGED is
'Last date and time record changed';

alter table MARKET_FEE_EXCLUSIONTRK
   add constraint MARKET_FEE_EXCLUSIONTRK_PK primary key (PARTICIPANTID, EFFECTIVEDATE, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MARKET_FEE_EXCLUSIONTRK_LCX                           */
/*==============================================================*/
create index MARKET_FEE_EXCLUSIONTRK_LCX on MARKET_FEE_EXCLUSIONTRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MARKET_PRICE_THRESHOLDS                               */
/*==============================================================*/
create table MARKET_PRICE_THRESHOLDS (
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(4,0)           not null,
   VOLL                 NUMBER(15,5),
   MARKETPRICEFLOOR     NUMBER(15,5),
   ADMINISTERED_PRICE_THRESHOLD NUMBER(15,5),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MARKET_PRICE_THRESHOLDS is
'MARKET_PRICE_THRESHOLDS sets out the market cap , floor and administered price thresholds applying to the electricity market';

comment on column MARKET_PRICE_THRESHOLDS.EFFECTIVEDATE is
'Calendar date that this record becomes effective';

comment on column MARKET_PRICE_THRESHOLDS.VERSIONNO is
'version no for the effective date';

comment on column MARKET_PRICE_THRESHOLDS.VOLL is
'value of lost load if total supply falls short of demand after load management then involuntary load';

comment on column MARKET_PRICE_THRESHOLDS.MARKETPRICEFLOOR is
'The floor price that the spot market price will not fall below.';

comment on column MARKET_PRICE_THRESHOLDS.ADMINISTERED_PRICE_THRESHOLD is
'Threshold value beyond which Aggregate Prices per Region over 336 Trade Intervals (Energy), or 2016 Dispatch Intervals (FCAS), will result in an Administered Price declaration';

comment on column MARKET_PRICE_THRESHOLDS.AUTHORISEDDATE is
'date data authorised';

comment on column MARKET_PRICE_THRESHOLDS.AUTHORISEDBY is
'user authorising';

comment on column MARKET_PRICE_THRESHOLDS.LASTCHANGED is
'Last date and time record changed';

alter table MARKET_PRICE_THRESHOLDS
   add constraint MARKET_PRICE_THRESHOLDS_PK primary key (EFFECTIVEDATE, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MARKET_PRICE_THRESHOLDS_LCX                           */
/*==============================================================*/
create index MARKET_PRICE_THRESHOLDS_LCX on MARKET_PRICE_THRESHOLDS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MARKET_SUSPEND_REGIME_SUM                             */
/*==============================================================*/
create table MARKET_SUSPEND_REGIME_SUM (
   SUSPENSION_ID        VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   START_INTERVAL       DATE                  not null,
   END_INTERVAL         DATE,
   PRICING_REGIME       VARCHAR2(20),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MARKET_SUSPEND_REGIME_SUM is
'Tracks the evolution of pricing regimes applied to the suspended region and from which Dispatch Interval';

comment on column MARKET_SUSPEND_REGIME_SUM.SUSPENSION_ID is
'Unique identifier for this suspension event';

comment on column MARKET_SUSPEND_REGIME_SUM.REGIONID is
'Region(s) covered by this evolution of the event';

comment on column MARKET_SUSPEND_REGIME_SUM.START_INTERVAL is
'First Dispatch interval from which this regime applies';

comment on column MARKET_SUSPEND_REGIME_SUM.END_INTERVAL is
'Last Dispatch interval for which this regime applies';

comment on column MARKET_SUSPEND_REGIME_SUM.PRICING_REGIME is
'Pricing Regime applied';

comment on column MARKET_SUSPEND_REGIME_SUM.LASTCHANGED is
'Last date and time record changed';

alter table MARKET_SUSPEND_REGIME_SUM
   add constraint MARKET_SUSPEND_REGIME_SUM_PK primary key (SUSPENSION_ID, REGIONID, START_INTERVAL)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: MARKET_SUSPEND_REGION_SUM                             */
/*==============================================================*/
create table MARKET_SUSPEND_REGION_SUM (
   SUSPENSION_ID        VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   INITIAL_INTERVAL     DATE,
   END_REGION_INTERVAL  DATE,
   END_SUSPENSION_INTERVAL DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MARKET_SUSPEND_REGION_SUM is
'Summary of Market Suspension timings';

comment on column MARKET_SUSPEND_REGION_SUM.SUSPENSION_ID is
'Unique identifier for this suspension event';

comment on column MARKET_SUSPEND_REGION_SUM.REGIONID is
'Region(s) covered by the Suspension event';

comment on column MARKET_SUSPEND_REGION_SUM.INITIAL_INTERVAL is
'Initial interval of the Suspension event';

comment on column MARKET_SUSPEND_REGION_SUM.END_REGION_INTERVAL is
'Last Dispatch interval for the Suspension event for this Region';

comment on column MARKET_SUSPEND_REGION_SUM.END_SUSPENSION_INTERVAL is
'Last Dispatch interval for the Suspension event';

comment on column MARKET_SUSPEND_REGION_SUM.LASTCHANGED is
'Last DateTime the Suspension was administered';

alter table MARKET_SUSPEND_REGION_SUM
   add constraint MARKET_SUSPEND_REGION_SUM_PK primary key (SUSPENSION_ID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: MARKET_SUSPEND_SCHEDULE                               */
/*==============================================================*/
create table MARKET_SUSPEND_SCHEDULE (
   EFFECTIVEDATE        DATE                  not null,
   DAY_TYPE             VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   PERIODID             NUMBER(3,0)           not null,
   ENERGY_RRP           NUMBER(15,5),
   R6_RRP               NUMBER(15,5),
   R60_RRP              NUMBER(15,5),
   R5_RRP               NUMBER(15,5),
   RREG_RRP             NUMBER(15,5),
   L6_RRP               NUMBER(15,5),
   L60_RRP              NUMBER(15,5),
   L5_RRP               NUMBER(15,5),
   LREG_RRP             NUMBER(15,5),
   LASTCHANGED          DATE,
   L1_RRP               NUMBER(15,5),
   R1_RRP               NUMBER(15,5)
)
   tablespace PSSTAB01;

comment on table MARKET_SUSPEND_SCHEDULE is
'Trading prices that will apply in the event of a market suspension event updated weekly.';

comment on column MARKET_SUSPEND_SCHEDULE.EFFECTIVEDATE is
'Calendar date from when this record set is effective';

comment on column MARKET_SUSPEND_SCHEDULE.DAY_TYPE is
'Distinguishes which record set to apply - at time of writing this was Business or Non-business day but may change in the future depending on outcome of consultation';

comment on column MARKET_SUSPEND_SCHEDULE.REGIONID is
'Region affected.';

comment on column MARKET_SUSPEND_SCHEDULE.PERIODID is
'48 intervals for a day, midnight base (equates to 00:30 - 00:00)';

comment on column MARKET_SUSPEND_SCHEDULE.ENERGY_RRP is
'Energy Price applied for this period for this Day Type';

comment on column MARKET_SUSPEND_SCHEDULE.R6_RRP is
'Raise 6Sec contingency Price applied for this period for this Day Type';

comment on column MARKET_SUSPEND_SCHEDULE.R60_RRP is
'Raise 60Sec contingency Price applied for this period for this Day Type';

comment on column MARKET_SUSPEND_SCHEDULE.R5_RRP is
'Raise 5Min contingency Price applied for this period for this Day Type';

comment on column MARKET_SUSPEND_SCHEDULE.RREG_RRP is
'Raise Regulation contingency Price applied for this period for this Day Type';

comment on column MARKET_SUSPEND_SCHEDULE.L6_RRP is
'Lower 6Sec contingency Price applied for this period for this Day Type';

comment on column MARKET_SUSPEND_SCHEDULE.L60_RRP is
'Lower 60Sec contingency Price applied for this period for this Day Type';

comment on column MARKET_SUSPEND_SCHEDULE.L5_RRP is
'Lower 5Min contingency Price applied for this period for this Day Type';

comment on column MARKET_SUSPEND_SCHEDULE.LREG_RRP is
'Lower Regulation Price applied for this period for this Day Type';

comment on column MARKET_SUSPEND_SCHEDULE.LASTCHANGED is
'Last date and time record changed';

comment on column MARKET_SUSPEND_SCHEDULE.L1_RRP is
'Lower 1Sec contingency Price applied for this period for this Day Type';

comment on column MARKET_SUSPEND_SCHEDULE.R1_RRP is
'Raise 1Sec contingency Price applied for this period for this Day Type';

alter table MARKET_SUSPEND_SCHEDULE
   add constraint MARKET_SUSPEND_SCHEDULE_PK primary key (EFFECTIVEDATE, DAY_TYPE, REGIONID, PERIODID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: MARKET_SUSPEND_SCHEDULE_TRK                           */
/*==============================================================*/
create table MARKET_SUSPEND_SCHEDULE_TRK (
   EFFECTIVEDATE        DATE                  not null,
   SOURCE_START_DATE    DATE,
   SOURCE_END_DATE      DATE,
   COMMENTS             VARCHAR2(1000),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MARKET_SUSPEND_SCHEDULE_TRK is
'Parent table for pricing regimes used in suspensions';

comment on column MARKET_SUSPEND_SCHEDULE_TRK.EFFECTIVEDATE is
'Calendar date from when this record set is effective';

comment on column MARKET_SUSPEND_SCHEDULE_TRK.SOURCE_START_DATE is
'Start Date of the date range for the source data';

comment on column MARKET_SUSPEND_SCHEDULE_TRK.SOURCE_END_DATE is
'End Date of the date range for the source data';

comment on column MARKET_SUSPEND_SCHEDULE_TRK.COMMENTS is
'Reason why this regime was applied';

comment on column MARKET_SUSPEND_SCHEDULE_TRK.AUTHORISEDDATE is
'DateTime this record set was loaded';

comment on column MARKET_SUSPEND_SCHEDULE_TRK.LASTCHANGED is
'Last date and time record changed';

alter table MARKET_SUSPEND_SCHEDULE_TRK
   add constraint MARKET_SUSPEND_SCHEDULE_TRK_PK primary key (EFFECTIVEDATE)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: MAS_CP_CHANGE                                         */
/*==============================================================*/
create table MAS_CP_CHANGE (
   NMI                  VARCHAR2(10)          not null,
   STATUS_FLAG          VARCHAR2(1),
   CP_OLD_SECURITY_CODE VARCHAR2(4),
   CP_NEW_SECURITY_CODE VARCHAR2(4),
   OLD_LOCAL_NETWORK_PROVIDER VARCHAR2(10),
   OLD_LOCAL_RETAILER   VARCHAR2(10),
   OLD_FINANCIAL_PARTICIPANT VARCHAR2(10),
   OLD_METERING_DATA_AGENT VARCHAR2(10),
   OLD_RETAILER_OF_LAST_RESORT VARCHAR2(10),
   OLD_RESPONSIBLE_PERSON VARCHAR2(10),
   NEW_LOCAL_NETWORK_PROVIDER VARCHAR2(10),
   NEW_LOCAL_RETAILER   VARCHAR2(10),
   NEW_FINANCIAL_PARTICIPANT VARCHAR2(10),
   NEW_METERING_DATA_AGENT VARCHAR2(10),
   NEW_RETAILER_OF_LAST_RESORT VARCHAR2(10),
   NEW_RESPONSIBLE_PERSON VARCHAR2(10),
   OLD_LNSP_OK          VARCHAR2(1),
   OLD_LR_OK            VARCHAR2(1),
   OLD_FRMP_OK          VARCHAR2(1),
   OLD_MDA_OK           VARCHAR2(1),
   OLD_ROLR_OK          VARCHAR2(1),
   OLD_RP_OK            VARCHAR2(1),
   NEW_LNSP_OK          VARCHAR2(1),
   NEW_LR_OK            VARCHAR2(1),
   NEW_FRMP_OK          VARCHAR2(1),
   NEW_MDA_OK           VARCHAR2(1),
   NEW_ROLR_OK          VARCHAR2(1),
   NEW_RP_OK            VARCHAR2(1),
   PRUDENTIAL_OK        VARCHAR2(1),
   INITIAL_CHANGE_DATE  DATE,
   CURRENT_CHANGE_DATE  DATE,
   CP_NAME              VARCHAR2(30),
   CP_DETAIL_1          VARCHAR2(30),
   CP_DETAIL_2          VARCHAR2(30),
   CITY_SUBURB          VARCHAR2(30),
   STATE                VARCHAR2(3),
   POST_CODE            VARCHAR2(4),
   TX_NODE              VARCHAR2(4),
   AGGREGATE_DATA       VARCHAR2(1),
   AVERAGE_DAILY_LOAD_KWH NUMBER(8,0),
   DISTRIBUTION_LOSS    NUMBER(5,4),
   OLD_LSNP_TEXT        VARCHAR2(30),
   OLD_LR_TEXT          VARCHAR2(30),
   OLD_FRMP_TEXT        VARCHAR2(30),
   OLD_MDA_TEXT         VARCHAR2(30),
   OLD_ROLR_TEXT        VARCHAR2(30),
   OLD_RP_TEXT          VARCHAR2(30),
   NEW_LSNP_TEXT        VARCHAR2(30),
   NEW_LR_TEXT          VARCHAR2(30),
   NEW_FRMP_TEXT        VARCHAR2(30),
   NEW_MDA_TEXT         VARCHAR2(30),
   NEW_ROLR_TEXT        VARCHAR2(30),
   NEW_RP_TEXT          VARCHAR2(30),
   LASTCHANGED          DATE,
   NMI_CLASS            VARCHAR2(9),
   METERING_TYPE        VARCHAR2(9),
   JURISDICTION         VARCHAR2(3),
   CREATE_DATE          DATE,
   EXPIRY_DATE          DATE,
   METER_READ_DATE      DATE
)
   tablespace PSSTAB01;

comment on table MAS_CP_CHANGE is
'MAS_CP_CHANGE records pending changes to the current MAS configuration.';

comment on column MAS_CP_CHANGE.NMI is
'National Metering Identifier';

comment on column MAS_CP_CHANGE.STATUS_FLAG is
'Active/Inactive flag';

comment on column MAS_CP_CHANGE.CP_OLD_SECURITY_CODE is
'Old Security Code';

comment on column MAS_CP_CHANGE.CP_NEW_SECURITY_CODE is
'New Security Code';

comment on column MAS_CP_CHANGE.OLD_LOCAL_NETWORK_PROVIDER is
'Old Local Network Provider';

comment on column MAS_CP_CHANGE.OLD_LOCAL_RETAILER is
'Old Local Retailer';

comment on column MAS_CP_CHANGE.OLD_FINANCIAL_PARTICIPANT is
'Old FRMP';

comment on column MAS_CP_CHANGE.OLD_METERING_DATA_AGENT is
'Old Metering Data Agent';

comment on column MAS_CP_CHANGE.OLD_RETAILER_OF_LAST_RESORT is
'Old Retailer of Last Resort';

comment on column MAS_CP_CHANGE.OLD_RESPONSIBLE_PERSON is
'Old Responsible Person';

comment on column MAS_CP_CHANGE.NEW_LOCAL_NETWORK_PROVIDER is
'New Local Network Provider';

comment on column MAS_CP_CHANGE.NEW_LOCAL_RETAILER is
'New Local Retailer';

comment on column MAS_CP_CHANGE.NEW_FINANCIAL_PARTICIPANT is
'New FRMP';

comment on column MAS_CP_CHANGE.NEW_METERING_DATA_AGENT is
'New Metering Data Agent';

comment on column MAS_CP_CHANGE.NEW_RETAILER_OF_LAST_RESORT is
'New Retailer of Last Resort';

comment on column MAS_CP_CHANGE.NEW_RESPONSIBLE_PERSON is
'New Responsible Person';

comment on column MAS_CP_CHANGE.OLD_LNSP_OK is
'Old LNSP approval flag';

comment on column MAS_CP_CHANGE.OLD_LR_OK is
'Old LR approval flag';

comment on column MAS_CP_CHANGE.OLD_FRMP_OK is
'Old FRMP approval flag';

comment on column MAS_CP_CHANGE.OLD_MDA_OK is
'Old MDA approval flag';

comment on column MAS_CP_CHANGE.OLD_ROLR_OK is
'Old ROLR approval flag';

comment on column MAS_CP_CHANGE.OLD_RP_OK is
'Old RP approval flag';

comment on column MAS_CP_CHANGE.NEW_LNSP_OK is
'New LNSP approval flag';

comment on column MAS_CP_CHANGE.NEW_LR_OK is
'New LR approval flag';

comment on column MAS_CP_CHANGE.NEW_FRMP_OK is
'New FRMP approval flag';

comment on column MAS_CP_CHANGE.NEW_MDA_OK is
'New MDA approval flag';

comment on column MAS_CP_CHANGE.NEW_ROLR_OK is
'New ROLR approval flag';

comment on column MAS_CP_CHANGE.NEW_RP_OK is
'New RP approval flag';

comment on column MAS_CP_CHANGE.PRUDENTIAL_OK is
'Prudential check flag';

comment on column MAS_CP_CHANGE.INITIAL_CHANGE_DATE is
'Initial change date';

comment on column MAS_CP_CHANGE.CURRENT_CHANGE_DATE is
'Current change date';

comment on column MAS_CP_CHANGE.CP_NAME is
'Connection point name';

comment on column MAS_CP_CHANGE.CP_DETAIL_1 is
'Connection point detail 1';

comment on column MAS_CP_CHANGE.CP_DETAIL_2 is
'Connection point detail 2';

comment on column MAS_CP_CHANGE.CITY_SUBURB is
'Connection point City/Suburb';

comment on column MAS_CP_CHANGE.STATE is
'State of Australia';

comment on column MAS_CP_CHANGE.POST_CODE is
'Connection point postcode';

comment on column MAS_CP_CHANGE.TX_NODE is
'Connection point TNI';

comment on column MAS_CP_CHANGE.AGGREGATE_DATA is
'Aggregate data Flag (YIN)';

comment on column MAS_CP_CHANGE.AVERAGE_DAILY_LOAD_KWH is
'Average Daily load in KWh';

comment on column MAS_CP_CHANGE.DISTRIBUTION_LOSS is
'Distribution loss factors';

comment on column MAS_CP_CHANGE.OLD_LSNP_TEXT is
'Old LNSP text field (LNSP misspelt in name)';

comment on column MAS_CP_CHANGE.OLD_LR_TEXT is
'Old LR text field';

comment on column MAS_CP_CHANGE.OLD_FRMP_TEXT is
'Old FRMP text field';

comment on column MAS_CP_CHANGE.OLD_MDA_TEXT is
'Old MDA text field';

comment on column MAS_CP_CHANGE.OLD_ROLR_TEXT is
'Old ROLR text field';

comment on column MAS_CP_CHANGE.OLD_RP_TEXT is
'Old RP text field';

comment on column MAS_CP_CHANGE.NEW_LSNP_TEXT is
'New LNSP text field (LNSP misspelt in name)';

comment on column MAS_CP_CHANGE.NEW_LR_TEXT is
'New LR text field';

comment on column MAS_CP_CHANGE.NEW_FRMP_TEXT is
'New FRMP text field';

comment on column MAS_CP_CHANGE.NEW_MDA_TEXT is
'New MDA text field';

comment on column MAS_CP_CHANGE.NEW_ROLR_TEXT is
'New ROLR text field';

comment on column MAS_CP_CHANGE.NEW_RP_TEXT is
'New RP text field';

comment on column MAS_CP_CHANGE.LASTCHANGED is
'Last changed date.';

comment on column MAS_CP_CHANGE.NMI_CLASS is
'Class of National Metering Identifier to allow for different business rules to apply';

comment on column MAS_CP_CHANGE.METERING_TYPE is
'Type of metering installation (e.g. BASIC, MRIM, COMMS)';

comment on column MAS_CP_CHANGE.JURISDICTION is
'Area; for application of rules';

comment on column MAS_CP_CHANGE.CREATE_DATE is
'Set by the system with today''s date when the change record is created.';

comment on column MAS_CP_CHANGE.EXPIRY_DATE is
'Set by the system (and cannot be changed).';

comment on column MAS_CP_CHANGE.METER_READ_DATE is
'Date of meter reading';

alter table MAS_CP_CHANGE
   add constraint PK_MAS_CP_CHANGE primary key (NMI)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MAS_CP_CHANGE_LCX                                     */
/*==============================================================*/
create index MAS_CP_CHANGE_LCX on MAS_CP_CHANGE (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MAS_CP_MASTER                                         */
/*==============================================================*/
create table MAS_CP_MASTER (
   NMI                  VARCHAR2(10)          not null,
   CP_SECURITY_CODE     VARCHAR2(4),
   IN_USE               VARCHAR2(1),
   VALID_FROM_DATE      DATE                  not null,
   VALID_TO_DATE        DATE                  not null,
   LOCAL_NETWORK_PROVIDER VARCHAR2(10),
   LOCAL_RETAILER       VARCHAR2(10),
   FINANCIAL_PARTICIPANT VARCHAR2(10),
   METERING_DATA_AGENT  VARCHAR2(10),
   RETAILER_OF_LAST_RESORT VARCHAR2(10),
   RESPONSIBLE_PERSON   VARCHAR2(10),
   CP_NAME              VARCHAR2(30),
   CP_DETAIL_1          VARCHAR2(30),
   CP_DETAIL_2          VARCHAR2(30),
   CITY_SUBURB          VARCHAR2(30),
   STATE                VARCHAR2(3),
   POST_CODE            VARCHAR2(4),
   TX_NODE              VARCHAR2(4),
   AGGREGATE_DATA       VARCHAR2(1),
   AVERAGE_DAILY_LOAD_KWH NUMBER(8,0),
   DISTRIBUTION_LOSS    NUMBER(5,4),
   LSNP_TEXT            VARCHAR2(30),
   LR_TEXT              VARCHAR2(30),
   FRMP_TEXT            VARCHAR2(30),
   MDA_TEXT             VARCHAR2(30),
   ROLR_TEXT            VARCHAR2(30),
   RP_TEXT              VARCHAR2(30),
   LASTCHANGED          DATE,
   NMI_CLASS            VARCHAR2(9),
   METERING_TYPE        VARCHAR2(9),
   JURISDICTION         VARCHAR2(3)
)
   tablespace PSSTAB02;

comment on table MAS_CP_MASTER is
'MAS_CP_MASTER shows the current MAS configuration.';

comment on column MAS_CP_MASTER.NMI is
'National Metering Identifier';

comment on column MAS_CP_MASTER.CP_SECURITY_CODE is
'Security Code';

comment on column MAS_CP_MASTER.IN_USE is
'Active/Inactive Status flag (NEW/N/Y/X)';

comment on column MAS_CP_MASTER.VALID_FROM_DATE is
'Record valid from date';

comment on column MAS_CP_MASTER.VALID_TO_DATE is
'Record valid to date';

comment on column MAS_CP_MASTER.LOCAL_NETWORK_PROVIDER is
'LNSP';

comment on column MAS_CP_MASTER.LOCAL_RETAILER is
'Local Retailer';

comment on column MAS_CP_MASTER.FINANCIAL_PARTICIPANT is
'FRMP';

comment on column MAS_CP_MASTER.METERING_DATA_AGENT is
'MDA';

comment on column MAS_CP_MASTER.RETAILER_OF_LAST_RESORT is
'ROLR';

comment on column MAS_CP_MASTER.RESPONSIBLE_PERSON is
'Responsible Person';

comment on column MAS_CP_MASTER.CP_NAME is
'Connection point name';

comment on column MAS_CP_MASTER.CP_DETAIL_1 is
'Connection point detail 1';

comment on column MAS_CP_MASTER.CP_DETAIL_2 is
'Connection point detail 2';

comment on column MAS_CP_MASTER.CITY_SUBURB is
'Connection point city/suburb';

comment on column MAS_CP_MASTER.STATE is
'State of Australia';

comment on column MAS_CP_MASTER.POST_CODE is
'Connection point postcode';

comment on column MAS_CP_MASTER.TX_NODE is
'Connection point TNI';

comment on column MAS_CP_MASTER.AGGREGATE_DATA is
'Aggregate data flag (YIN)';

comment on column MAS_CP_MASTER.AVERAGE_DAILY_LOAD_KWH is
'Average daily load in KWh';

comment on column MAS_CP_MASTER.DISTRIBUTION_LOSS is
'Distribution loss factor';

comment on column MAS_CP_MASTER.LSNP_TEXT is
'LNSP text field (name has misspelt LNSP)';

comment on column MAS_CP_MASTER.LR_TEXT is
'LR text field';

comment on column MAS_CP_MASTER.FRMP_TEXT is
'FRMP text field';

comment on column MAS_CP_MASTER.MDA_TEXT is
'MDA text field';

comment on column MAS_CP_MASTER.ROLR_TEXT is
'ROLR text field';

comment on column MAS_CP_MASTER.RP_TEXT is
'RP text field';

comment on column MAS_CP_MASTER.LASTCHANGED is
'Last changed date';

alter table MAS_CP_MASTER
   add constraint PK_MAS_CP_MASTER primary key (NMI, VALID_FROM_DATE)
      using index tablespace PSSIND02;

alter table MAS_CP_MASTER
   add constraint UC_MAS_CP_MASTER unique (NMI, VALID_TO_DATE)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: MAS_CP_MASTER_LCX                                     */
/*==============================================================*/
create index MAS_CP_MASTER_LCX on MAS_CP_MASTER (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: MCC_CASESOLUTION                                      */
/*==============================================================*/
create table MCC_CASESOLUTION (
   RUN_DATETIME         DATE                  not null
)
   tablespace PSSTAB01;

comment on table MCC_CASESOLUTION is
'Top level table for each MCC dispatch rerun process. Note there will be one record for each dispatch interval';

comment on column MCC_CASESOLUTION.RUN_DATETIME is
'5-minute Dispatch Run identifier';

alter table MCC_CASESOLUTION
   add constraint MCC_CASESOLUTION_PK primary key (RUN_DATETIME)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: MCC_CONSTRAINTSOLUTION                                */
/*==============================================================*/
create table MCC_CONSTRAINTSOLUTION (
   RUN_DATETIME         DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   RHS                  NUMBER(15,5),
   MARGINALVALUE        NUMBER(15,5)
)
   tablespace PSSTAB01;

comment on table MCC_CONSTRAINTSOLUTION is
'Constraint solution data from the MCC dispatch rerun process. Note only constraints with a non-zero marginal value are published.';

comment on column MCC_CONSTRAINTSOLUTION.RUN_DATETIME is
'5-minute Dispatch Run identifier';

comment on column MCC_CONSTRAINTSOLUTION.CONSTRAINTID is
'Generic Constraint identifier (synonymous with GenConID)';

comment on column MCC_CONSTRAINTSOLUTION.RHS is
'Generic Constraint RHS Value for this MCC run';

comment on column MCC_CONSTRAINTSOLUTION.MARGINALVALUE is
'Generic Constraint Marginal Value for this MCC run';

alter table MCC_CONSTRAINTSOLUTION
   add constraint MCC_CONSTRAINTSOLUTION_PK primary key (RUN_DATETIME, CONSTRAINTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: METERDATA                                             */
/*==============================================================*/
create table METERDATA (
   PARTICIPANTID        VARCHAR2(10)          not null,
   PERIODID             NUMBER(3,0)           not null,
   SETTLEMENTDATE       DATE                  not null,
   METERRUNNO           NUMBER(6,0)           not null,
   CONNECTIONPOINTID    VARCHAR2(10)          not null,
   IMPORTENERGYVALUE    NUMBER(9,6),
   EXPORTENERGYVALUE    NUMBER(9,6),
   IMPORTREACTIVEVALUE  NUMBER(9,6),
   EXPORTREACTIVEVALUE  NUMBER(9,6),
   HOSTDISTRIBUTOR      VARCHAR2(10)          not null,
   LASTCHANGED          DATE,
   MDA                  VARCHAR2(10)          not null
)
   tablespace PSSTAB02;

comment on table METERDATA is
'METERDATA sets out a meter data for each customer connection point. METERDATA covers market load. Use the field METERRUNNO to match the meter data version for each settlement run.';

comment on column METERDATA.PARTICIPANTID is
'Unique participant identifier';

comment on column METERDATA.PERIODID is
'Settlement period identifier (half hour period)';

comment on column METERDATA.SETTLEMENTDATE is
'Settlement date';

comment on column METERDATA.METERRUNNO is
'Data version no';

comment on column METERDATA.CONNECTIONPOINTID is
'Transmission Node Identifier (TNI).  Identifies a Transmission NetworkConnection Point.';

comment on column METERDATA.IMPORTENERGYVALUE is
'Imported energy value (MWh)';

comment on column METERDATA.EXPORTENERGYVALUE is
'Exported energy value (MWh)';

comment on column METERDATA.IMPORTREACTIVEVALUE is
'Not used';

comment on column METERDATA.EXPORTREACTIVEVALUE is
'Not used';

comment on column METERDATA.HOSTDISTRIBUTOR is
'Local Retailer participant identifier';

comment on column METERDATA.LASTCHANGED is
'Last date and time record changed';

comment on column METERDATA.MDA is
'Defaults to MSATS';

alter table METERDATA
   add constraint METERDATA_PK primary key (SETTLEMENTDATE, MDA, METERRUNNO, CONNECTIONPOINTID, PARTICIPANTID, HOSTDISTRIBUTOR, PERIODID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: METERDATA_LCX                                         */
/*==============================================================*/
create index METERDATA_LCX on METERDATA (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: METERDATATRK                                          */
/*==============================================================*/
create table METERDATATRK (
   SETTLEMENTDATE       DATE                  not null,
   METERRUNNO           NUMBER(6,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   FILENAME             VARCHAR2(40),
   ACKFILENAME          VARCHAR2(40),
   CONNECTIONPOINTID    VARCHAR2(10)          not null,
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   METERINGDATAAGENT    VARCHAR2(10)          not null,
   HOSTDISTRIBUTOR      VARCHAR2(10)          not null,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table METERDATATRK is
'METERDATATRK records meter data files submitted for each connection point on a daily basis. The same data is provided in METERDATA period by period (i.e. 48 records), whereas METERDATATRK shows one record per day for each file submitted for a connection point.';

comment on column METERDATATRK.SETTLEMENTDATE is
'Settlement calendar date';

comment on column METERDATATRK.METERRUNNO is
'Meter data version number';

comment on column METERDATATRK.PARTICIPANTID is
'Participant identifier';

comment on column METERDATATRK.FILENAME is
'Meter file name (MSATS file name)';

comment on column METERDATATRK.ACKFILENAME is
'Not used';

comment on column METERDATATRK.CONNECTIONPOINTID is
'Transmission Node Identifier (TNI)';

comment on column METERDATATRK.AUTHORISEDDATE is
'Date processed';

comment on column METERDATATRK.AUTHORISEDBY is
'Not used';

comment on column METERDATATRK.METERINGDATAAGENT is
'Defaults to MSATS';

comment on column METERDATATRK.HOSTDISTRIBUTOR is
'Local retailer participant identifier';

comment on column METERDATATRK.LASTCHANGED is
'Last date and time record changed';

alter table METERDATATRK
   add constraint METERDATATRK_PK primary key (SETTLEMENTDATE, METERINGDATAAGENT, METERRUNNO, CONNECTIONPOINTID, PARTICIPANTID, HOSTDISTRIBUTOR)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: METERDATATRK_LCX                                      */
/*==============================================================*/
create index METERDATATRK_LCX on METERDATATRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: METERDATA_AGGREGATE_READS                             */
/*==============================================================*/
create table METERDATA_AGGREGATE_READS (
   CASE_ID              NUMBER(15,0)          not null,
   SETTLEMENTDATE       DATE                  not null,
   CONNECTIONPOINTID    VARCHAR2(20)          not null,
   METER_TYPE           VARCHAR2(20)          not null,
   FRMP                 VARCHAR2(20)          not null,
   LR                   VARCHAR2(20)          not null,
   PERIODID             NUMBER(3,0)           not null,
   IMPORTVALUE          NUMBER(18,8)          not null,
   EXPORTVALUE          NUMBER(18,8)          not null,
   LASTCHANGED          DATE
)
   tablespace PSSTAB03;

comment on table METERDATA_AGGREGATE_READS is
'Publishes aggregated metering data associated with a wholesale connection point for a given CASE_ID';

comment on column METERDATA_AGGREGATE_READS.CASE_ID is
'Case Identifier';

comment on column METERDATA_AGGREGATE_READS.SETTLEMENTDATE is
'Settlement date within the case';

comment on column METERDATA_AGGREGATE_READS.CONNECTIONPOINTID is
'Connection Point ID';

comment on column METERDATA_AGGREGATE_READS.METER_TYPE is
'The meter type for the read, one of: CUSTOMER; GENERATOR; EMBEDDED_GENERATOR';

comment on column METERDATA_AGGREGATE_READS.FRMP is
'The financially responsible market participantid';

comment on column METERDATA_AGGREGATE_READS.LR is
'The local retailer at the connection point id';

comment on column METERDATA_AGGREGATE_READS.PERIODID is
'Trading Interval.';

comment on column METERDATA_AGGREGATE_READS.IMPORTVALUE is
'The import(pool-centric) value for the meter read (MWh)';

comment on column METERDATA_AGGREGATE_READS.EXPORTVALUE is
'The export(pool-centric) value for the meter read (MWh)';

comment on column METERDATA_AGGREGATE_READS.LASTCHANGED is
'Last changed date for the record';

alter table METERDATA_AGGREGATE_READS
   add constraint METERDATA_AGGREGATE_READS_PK primary key (CASE_ID, SETTLEMENTDATE, CONNECTIONPOINTID, METER_TYPE, FRMP, LR, PERIODID)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: METERDATA_GEN_DUID                                    */
/*==============================================================*/
create table METERDATA_GEN_DUID (
   INTERVAL_DATETIME    date                  not null,
   DUID                 varchar2(10)          not null,
   MWH_READING          number(18,8),
   LASTCHANGED          date
)
   tablespace PSSTAB02;

comment on table METERDATA_GEN_DUID is
'Recorded actual generation of non-scheduled units where SCADA data is available.';

comment on column METERDATA_GEN_DUID.INTERVAL_DATETIME is
'Timestamp of the recorded interval';

comment on column METERDATA_GEN_DUID.DUID is
'Unit ID';

comment on column METERDATA_GEN_DUID.MWH_READING is
'MW reading';

comment on column METERDATA_GEN_DUID.LASTCHANGED is
'Timestamp of last record change';

alter table METERDATA_GEN_DUID
   add constraint METERDATA_GEN_DUID_PK primary key (INTERVAL_DATETIME, DUID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: METERDATA_GEN_DUID_LCX                                */
/*==============================================================*/
create index METERDATA_GEN_DUID_LCX on METERDATA_GEN_DUID (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: METERDATA_INDIVIDUAL_READS                            */
/*==============================================================*/
create table METERDATA_INDIVIDUAL_READS (
   CASE_ID              NUMBER(15,0)          not null,
   SETTLEMENTDATE       DATE                  not null,
   METER_ID             VARCHAR2(20)          not null,
   METER_ID_SUFFIX      VARCHAR2(20)          not null,
   FRMP                 VARCHAR2(20)          not null,
   LR                   VARCHAR2(20)          not null,
   PERIODID             NUMBER(3,0)           not null,
   CONNECTIONPOINTID    VARCHAR2(20)          not null,
   METER_TYPE           VARCHAR2(20)          not null,
   IMPORTVALUE          NUMBER(18,8)          not null,
   EXPORTVALUE          NUMBER(18,8)          not null,
   LASTCHANGED          DATE
)
   tablespace PSSTAB03;

comment on table METERDATA_INDIVIDUAL_READS is
'Publishes metering data associated with individual metering points for a given CASE_ID';

comment on column METERDATA_INDIVIDUAL_READS.CASE_ID is
'Case Identifier';

comment on column METERDATA_INDIVIDUAL_READS.SETTLEMENTDATE is
'Settlement date within the case';

comment on column METERDATA_INDIVIDUAL_READS.METER_ID is
'The National Metering Identifier (NMI)';

comment on column METERDATA_INDIVIDUAL_READS.METER_ID_SUFFIX is
'The National Metering Identifier (NMI) data stream';

comment on column METERDATA_INDIVIDUAL_READS.FRMP is
'The financially responsible market participantid';

comment on column METERDATA_INDIVIDUAL_READS.LR is
'The local retailer at the connection point id';

comment on column METERDATA_INDIVIDUAL_READS.PERIODID is
'Trading Interval.';

comment on column METERDATA_INDIVIDUAL_READS.CONNECTIONPOINTID is
'Connection Point ID';

comment on column METERDATA_INDIVIDUAL_READS.METER_TYPE is
'The meter type for the read, one of: CUSTOMER; GENERATOR; EMBEDDED_GENERATOR';

comment on column METERDATA_INDIVIDUAL_READS.IMPORTVALUE is
'The import(pool-centric) value for the meter read (MWh)';

comment on column METERDATA_INDIVIDUAL_READS.EXPORTVALUE is
'The export(pool-centric) value for the meter read (MWh)';

comment on column METERDATA_INDIVIDUAL_READS.LASTCHANGED is
'Last changed date for the record';

alter table METERDATA_INDIVIDUAL_READS
   add constraint METERDATA_INDIVIDUAL_READS_PK primary key (CASE_ID, SETTLEMENTDATE, METER_ID, METER_ID_SUFFIX, PERIODID)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: METERDATA_INTERCONNECTOR                              */
/*==============================================================*/
create table METERDATA_INTERCONNECTOR (
   CASE_ID              NUMBER(15,0)          not null,
   SETTLEMENTDATE       DATE                  not null,
   INTERCONNECTORID     VARCHAR2(20)          not null,
   PERIODID             NUMBER(3,0)           not null,
   IMPORTVALUE          NUMBER(18,8),
   EXPORTVALUE          NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table METERDATA_INTERCONNECTOR is
'Publishes metering data associated with wholesale interconnectors for a given CASE_ID';

comment on column METERDATA_INTERCONNECTOR.CASE_ID is
'Case Identifier';

comment on column METERDATA_INTERCONNECTOR.SETTLEMENTDATE is
'Settlement date within the case';

comment on column METERDATA_INTERCONNECTOR.INTERCONNECTORID is
'Interconnector Identifier';

comment on column METERDATA_INTERCONNECTOR.PERIODID is
'Trading Interval.';

comment on column METERDATA_INTERCONNECTOR.IMPORTVALUE is
'The import direction value for the meter read (MWh)';

comment on column METERDATA_INTERCONNECTOR.EXPORTVALUE is
'The export direction value for the meter read (MWh)';

comment on column METERDATA_INTERCONNECTOR.LASTCHANGED is
'Last changed date for the record';

alter table METERDATA_INTERCONNECTOR
   add constraint METERDATA_INTERCONNECTOR_PK primary key (CASE_ID, SETTLEMENTDATE, INTERCONNECTORID, PERIODID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: METERDATA_SAPS                                        */
/*==============================================================*/
create table METERDATA_SAPS (
   CASE_ID              NUMBER(15,0)          not null,
   SETTLEMENTDATE       DATE                  not null,
   CONNECTIONPOINT_ID   VARCHAR2(20)          not null,
   METER_TYPE           VARCHAR2(20)          not null,
   FRMP                 VARCHAR2(20)          not null,
   LR                   VARCHAR2(20)          not null,
   PERIODID             NUMBER(4,0)           not null,
   IMPORTVALUE          NUMBER(18,8),
   EXPORTVALUE          NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB03;

comment on table METERDATA_SAPS is
'The SAPS Meter data for MSRP and Retailer used in the Settlement Calculation';

comment on column METERDATA_SAPS.CASE_ID is
'The Metering Case ID used for Settlements';

comment on column METERDATA_SAPS.SETTLEMENTDATE is
'The Settlement Date for that Week';

comment on column METERDATA_SAPS.CONNECTIONPOINT_ID is
'The SAPS Connection Point Id';

comment on column METERDATA_SAPS.METER_TYPE is
'The Meter Type Identifier , CUSTOMER or MSRP';

comment on column METERDATA_SAPS.FRMP is
'The Financial Responsible Market Participant';

comment on column METERDATA_SAPS.LR is
'The Local Retailer';

comment on column METERDATA_SAPS.PERIODID is
'The Period ID Identifier';

comment on column METERDATA_SAPS.IMPORTVALUE is
'The Sent Out Energy in MWh';

comment on column METERDATA_SAPS.EXPORTVALUE is
'The Consumed Energy in MWh';

comment on column METERDATA_SAPS.LASTCHANGED is
'The Date time of the record last updated or inserted.';

alter table METERDATA_SAPS
   add constraint METERDATA_SAPS_PK primary key (CASE_ID, SETTLEMENTDATE, CONNECTIONPOINT_ID, METER_TYPE, FRMP, LR, PERIODID)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: METERDATA_TRK                                         */
/*==============================================================*/
create table METERDATA_TRK (
   CASE_ID              NUMBER(15,0)          not null,
   AGGREGATE_READS_LOAD_DATETIME DATE,
   INDIVIDUAL_READS_LOAD_DATETIME DATE,
   STARTDATE            DATE,
   ENDDATE              DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table METERDATA_TRK is
'Tracking table for the publication of wholesale settlement data associated with BILLING run';

comment on column METERDATA_TRK.CASE_ID is
'Case Identifier';

comment on column METERDATA_TRK.AGGREGATE_READS_LOAD_DATETIME is
'Timestamp of the aggregated reads being loaded for this case';

comment on column METERDATA_TRK.INDIVIDUAL_READS_LOAD_DATETIME is
'Timestamp of the non aggregated reads being loaded for this case';

comment on column METERDATA_TRK.STARTDATE is
'The start date of data associated with the CASE_ID';

comment on column METERDATA_TRK.ENDDATE is
'The end date of data associated with the Case_ID';

comment on column METERDATA_TRK.LASTCHANGED is
'Last changed date for the record';

alter table METERDATA_TRK
   add constraint METERDATA_TRK_PK primary key (CASE_ID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: METERDATA_WDR_READS                                   */
/*==============================================================*/
create table METERDATA_WDR_READS (
   MARKET_ID            VARCHAR2(20)          not null,
   CASE_ID              NUMBER(15,0)          not null,
   SETTLEMENTDATE       DATE                  not null,
   METER_ID             VARCHAR2(20)          not null,
   TNI                  VARCHAR2(20),
   FRMP                 VARCHAR2(20),
   DRSP                 VARCHAR2(20),
   PERIODID             NUMBER(3,0)           not null,
   METEREDQUANTITYIMPORT NUMBER(18,8),
   METEREDQUANTITYEXPORT NUMBER(18,8),
   BASELINEQUANTITY     NUMBER(18,8),
   QUALITYFLAG          VARCHAR2(20),
   ISNONCOMPLIANT       NUMBER(1,0),
   BASELINECALCULATIONID VARCHAR2(100)
)
   tablespace PSSTAB02;

comment on table METERDATA_WDR_READS is
'Metering Data WDR Readings';

comment on column METERDATA_WDR_READS.MARKET_ID is
'Unique identifier for the market to which this metering record applies.  Always equal to NEM in the current system.';

comment on column METERDATA_WDR_READS.CASE_ID is
'Unique identifier for the metering case.';

comment on column METERDATA_WDR_READS.SETTLEMENTDATE is
'The settlement date for the metering record';

comment on column METERDATA_WDR_READS.METER_ID is
'Unique identifier for the meter to which the metering record applies';

comment on column METERDATA_WDR_READS.TNI is
'Unique identifier for the transmission node to which this meter belongs on the settlement date';

comment on column METERDATA_WDR_READS.FRMP is
'Unique identifier for the participant acting as the FRMP for this NMI on the settlement date';

comment on column METERDATA_WDR_READS.DRSP is
'Unique identifier for the participant acting as the DRSP for this NMI on the settlement date';

comment on column METERDATA_WDR_READS.PERIODID is
'Trading interval identifier, with Period 1 being the first TI for the calendar day, i.e interval ending 00:05.';

comment on column METERDATA_WDR_READS.METEREDQUANTITYIMPORT is
'Metered quantity Import in MWh for the NMI in the trading interval.  A negative value indicates net consumption, while a positive value indicates net generation';

comment on column METERDATA_WDR_READS.METEREDQUANTITYEXPORT is
'Metered quantity Export in MWh for the NMI in the trading interval.  A negative value indicates net consumption, while a positive value indicates net generation';

comment on column METERDATA_WDR_READS.BASELINEQUANTITY is
'Baseline quantity in MWh for the NMI in the trading interval.  A negative value indicates net consumption, while a positive value indicates the net generation';

comment on column METERDATA_WDR_READS.QUALITYFLAG is
'Quality flag for the meter read.  Where multiple datastreams exist against the NMI with different quality flags for each read, the lowest quality flag will be published against the NMI for the interval.';

comment on column METERDATA_WDR_READS.ISNONCOMPLIANT is
'A value of TRUE (indicated by 1) for this column indicates that financial settlement of WDR transactions for this NMI should not proceed for the settlement date and trading interval. Possible values are 1 and 0.';

comment on column METERDATA_WDR_READS.BASELINECALCULATIONID is
'A reference to the baseline run that produced the baseline quantity for this NMI and interval';

alter table METERDATA_WDR_READS
   add constraint METERDATA_WDR_READS_PK primary key (MARKET_ID, CASE_ID, SETTLEMENTDATE, METER_ID, PERIODID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: MMS_DATA_MODEL_AUDIT                                  */
/*==============================================================*/
create table MMS_DATA_MODEL_AUDIT (
   INSTALLATION_DATE    DATE                  not null,
   MMSDM_VERSION        VARCHAR2(20)          not null,
   INSTALL_TYPE         VARCHAR2(10)          not null,
   SCRIPT_VERSION       VARCHAR2(20),
   NEM_CHANGE_NOTICE    VARCHAR2(20),
   PROJECT_TITLE        VARCHAR2(200),
   USERNAME             VARCHAR2(40),
   STATUS               VARCHAR2(10)
)
   tablespace PSSTAB01;

comment on table MMS_DATA_MODEL_AUDIT is
'MMS_DATA_MODEL_AUDIT shows the audit trail of scripts applied to this installation of MMS Data Model. Participants should ensure that if a database is cloned the content of this table is copied to the target database.';

comment on column MMS_DATA_MODEL_AUDIT.INSTALLATION_DATE is
'The date in which the changes to the MMS Data Model were installed';

comment on column MMS_DATA_MODEL_AUDIT.MMSDM_VERSION is
'The  version of MMS Data Model after the script has been applied';

comment on column MMS_DATA_MODEL_AUDIT.INSTALL_TYPE is
'The type of the patch applied. Valid entries are: FULL, UPGRADE, DML';

comment on column MMS_DATA_MODEL_AUDIT.SCRIPT_VERSION is
'The  version of the patch set to the MMS Data Model';

comment on column MMS_DATA_MODEL_AUDIT.NEM_CHANGE_NOTICE is
'The NEM Change notice for which this MMS Data Model applies';

comment on column MMS_DATA_MODEL_AUDIT.PROJECT_TITLE is
'The name of the business project for which these changes to the MMS Data Model apply';

comment on column MMS_DATA_MODEL_AUDIT.USERNAME is
'The USER applying this script';

comment on column MMS_DATA_MODEL_AUDIT.STATUS is
'The status of the upgrade. Valid entries are STARTED, FAILED, SUCCESS';

alter table MMS_DATA_MODEL_AUDIT
   add constraint MMS_DATA_MODEL_AUDIT_PK primary key (INSTALLATION_DATE, MMSDM_VERSION, INSTALL_TYPE)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: MNSP_BIDOFFERPERIOD                                   */
/*==============================================================*/
create table MNSP_BIDOFFERPERIOD (
   LINKID               VARCHAR2(20)          not null,
   TRADINGDATE          DATE                  not null,
   OFFERDATETIME        TIMESTAMP(3)          not null,
   PERIODID             NUMBER(3,0)           not null,
   MAXAVAIL             NUMBER(8,3),
   FIXEDLOAD            NUMBER(8,3),
   RAMPUPRATE           NUMBER(6),
   BANDAVAIL1           NUMBER(8,3),
   BANDAVAIL2           NUMBER(8,3),
   BANDAVAIL3           NUMBER(8,3),
   BANDAVAIL4           NUMBER(8,3),
   BANDAVAIL5           NUMBER(8,3),
   BANDAVAIL6           NUMBER(8,3),
   BANDAVAIL7           NUMBER(8,3),
   BANDAVAIL8           NUMBER(8,3),
   BANDAVAIL9           NUMBER(8,3),
   BANDAVAIL10          NUMBER(8,3),
   PASAAVAILABILITY     NUMBER(8,3),
   RECALL_PERIOD        NUMBER(8,3)
)
   tablespace PSSTAB03;

comment on table MNSP_BIDOFFERPERIOD is
'MNSP_BIDOFFERPERIOD shows availability for 5-minute periods for a specific Bid and LinkID for the given Trading Date and period. MNSP_BIDOFFERPERIOD is a child to MNSP_DAYOFFER and links to BIDOFFERFILETRK for 5MS Bids.';

comment on column MNSP_BIDOFFERPERIOD.LINKID is
'Identifier for each of the two MNSP Interconnector Links. Each link pertains to the direction from and to.';

comment on column MNSP_BIDOFFERPERIOD.TRADINGDATE is
'The trading date this bid is for';

comment on column MNSP_BIDOFFERPERIOD.OFFERDATETIME is
'Time this bid was processed and loaded';

comment on column MNSP_BIDOFFERPERIOD.PERIODID is
'Period ID, 1 to 288';

comment on column MNSP_BIDOFFERPERIOD.MAXAVAIL is
'Maximum planned availability MW';

comment on column MNSP_BIDOFFERPERIOD.FIXEDLOAD is
'Fixed unit output, in MW. A value of NULL means no fixed load so the unit is dispatched according to bid and the market.';

comment on column MNSP_BIDOFFERPERIOD.RAMPUPRATE is
'Ramp rate (MW / min) in the positive direction of flow for this MNSP link for this half-hour period';

comment on column MNSP_BIDOFFERPERIOD.BANDAVAIL1 is
'Availability at price band 1';

comment on column MNSP_BIDOFFERPERIOD.BANDAVAIL2 is
'Availability at price band 2';

comment on column MNSP_BIDOFFERPERIOD.BANDAVAIL3 is
'Availability at price band 3';

comment on column MNSP_BIDOFFERPERIOD.BANDAVAIL4 is
'Availability at price band 4';

comment on column MNSP_BIDOFFERPERIOD.BANDAVAIL5 is
'Availability at price band 5';

comment on column MNSP_BIDOFFERPERIOD.BANDAVAIL6 is
'Availability at price band 6';

comment on column MNSP_BIDOFFERPERIOD.BANDAVAIL7 is
'Availability at price band 7';

comment on column MNSP_BIDOFFERPERIOD.BANDAVAIL8 is
'Availability at price band 8';

comment on column MNSP_BIDOFFERPERIOD.BANDAVAIL9 is
'Availability at price band 9';

comment on column MNSP_BIDOFFERPERIOD.BANDAVAIL10 is
'Availability at price band 10';

comment on column MNSP_BIDOFFERPERIOD.PASAAVAILABILITY is
'Allows for future use for Energy bids, being the physical plant capability including any capability potentially available within 24 hours';

comment on column MNSP_BIDOFFERPERIOD.RECALL_PERIOD is
'The advance notice(in hours) that a Scheduled Resource requires to achieve the PASA Availability MW for this trading interval';

alter table MNSP_BIDOFFERPERIOD
   add constraint MNSP_BIDOFFERPERIOD_PK primary key (TRADINGDATE, LINKID, OFFERDATETIME, PERIODID)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: MNSP_DAYOFFER                                         */
/*==============================================================*/
create table MNSP_DAYOFFER (
   SETTLEMENTDATE       DATE                  not null,
   OFFERDATE            TIMESTAMP(3)          not null,
   VERSIONNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   LINKID               VARCHAR2(10)          not null,
   ENTRYTYPE            VARCHAR2(20),
   REBIDEXPLANATION     VARCHAR2(500),
   PRICEBAND1           NUMBER(9,2),
   PRICEBAND2           NUMBER(9,2),
   PRICEBAND3           NUMBER(9,2),
   PRICEBAND4           NUMBER(9,2),
   PRICEBAND5           NUMBER(9,2),
   PRICEBAND6           NUMBER(9,2),
   PRICEBAND7           NUMBER(9,2),
   PRICEBAND8           NUMBER(9,2),
   PRICEBAND9           NUMBER(9,2),
   PRICEBAND10          NUMBER(9,2),
   LASTCHANGED          DATE,
   MR_FACTOR            NUMBER(16,6),
   REBID_EVENT_TIME     VARCHAR2(20),
   REBID_AWARE_TIME     VARCHAR2(20),
   REBID_DECISION_TIME  VARCHAR2(20),
   REBID_CATEGORY       VARCHAR2(1),
   REFERENCE_ID         VARCHAR2(100)
)
   tablespace PSSTAB02;

comment on table MNSP_DAYOFFER is
'MNSP_DAYOFFER updates as bids are processed. All bids are available as part of next day market data. MNSP_DAYOFFER is the parent table to MNSP_BIDOFFERPERIOD, and joins to BIDOFFERFILETRK for 5MS Bids.';

comment on column MNSP_DAYOFFER.SETTLEMENTDATE is
'Market Date from which bid is active';

comment on column MNSP_DAYOFFER.OFFERDATE is
'Time this bid was processed and loaded';

comment on column MNSP_DAYOFFER.VERSIONNO is
'Version of data for other key data - a higher version for same key data will take precedence';

comment on column MNSP_DAYOFFER.PARTICIPANTID is
'Participant Identifier';

comment on column MNSP_DAYOFFER.LINKID is
'Identifier for each of the two MNSP Interconnector Links. Each link pertains to the direction from and to.';

comment on column MNSP_DAYOFFER.ENTRYTYPE is
'Bid type. Either Rebid or Daily';

comment on column MNSP_DAYOFFER.REBIDEXPLANATION is
'Explanation for all rebids and inflexibilities';

comment on column MNSP_DAYOFFER.PRICEBAND1 is
'Price for Availability Band 1';

comment on column MNSP_DAYOFFER.PRICEBAND2 is
'Price for Availability Band 2';

comment on column MNSP_DAYOFFER.PRICEBAND3 is
'Price for Availability Band 3';

comment on column MNSP_DAYOFFER.PRICEBAND4 is
'Price for Availability Band 4';

comment on column MNSP_DAYOFFER.PRICEBAND5 is
'Price for Availability Band 5';

comment on column MNSP_DAYOFFER.PRICEBAND6 is
'Price for Availability Band 6';

comment on column MNSP_DAYOFFER.PRICEBAND7 is
'Price for Availability Band 7';

comment on column MNSP_DAYOFFER.PRICEBAND8 is
'Price for Availability Band 8';

comment on column MNSP_DAYOFFER.PRICEBAND9 is
'Price for Availability Band 9';

comment on column MNSP_DAYOFFER.PRICEBAND10 is
'Price for Availability Band 10';

comment on column MNSP_DAYOFFER.LASTCHANGED is
'Last date and time record changed';

comment on column MNSP_DAYOFFER.MR_FACTOR is
'Mandatory Restriction Offer Factor';

comment on column MNSP_DAYOFFER.REBID_EVENT_TIME is
'The time of the event(s) or other occurrence(s) cited/adduced as the reason for the rebid. Required for rebids, not required for fixed load or low ramp rates. Expected in the format: HH:MM:SS e.g. 20:11:00';

comment on column MNSP_DAYOFFER.REBID_AWARE_TIME is
'Intended to support the Rebidding and Technical Parameters Guideline. The time at which the participant became aware of the event(s) / occurrence(s) that prompted the rebid. Not validated by AEMO';

comment on column MNSP_DAYOFFER.REBID_DECISION_TIME is
'Intended to support the Rebidding and Technical Parameters Guideline. The time at which the participant made the decision to rebid. Not validated by AEMO';

comment on column MNSP_DAYOFFER.REBID_CATEGORY is
'Intended to support the Rebidding and Technical Parameters Guideline. A provided rebid category. Not validated by AEMO';

comment on column MNSP_DAYOFFER.REFERENCE_ID is
'A participants unique Reference Id';

alter table MNSP_DAYOFFER
   add constraint MNSP_DAYOFFER_PK primary key (SETTLEMENTDATE, OFFERDATE, VERSIONNO, PARTICIPANTID, LINKID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MNSP_DAYOFFER_LCX                                     */
/*==============================================================*/
create index MNSP_DAYOFFER_LCX on MNSP_DAYOFFER (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MNSP_FILETRK                                          */
/*==============================================================*/
create table MNSP_FILETRK (
   SETTLEMENTDATE       DATE                  not null,
   OFFERDATE            DATE                  not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   FILENAME             VARCHAR2(40)          not null,
   STATUS               VARCHAR2(10),
   ACKFILENAME          VARCHAR2(40),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MNSP_FILETRK is
'MNSP_FILETRK shows all MNSPOFFERS transmitted to the MMS system.';

comment on column MNSP_FILETRK.SETTLEMENTDATE is
'Market Date from which bid is active';

comment on column MNSP_FILETRK.OFFERDATE is
'The actual date and time the bid file was submitted by the participant';

comment on column MNSP_FILETRK.PARTICIPANTID is
'Participant Identifier';

comment on column MNSP_FILETRK.FILENAME is
'File name for default bids, bids, rebids, re-offers or meter files, as appropriate to table';

comment on column MNSP_FILETRK.STATUS is
'Load status [SUCCESSFUL/CORRUPT]';

comment on column MNSP_FILETRK.ACKFILENAME is
'Acknowledge file name for bids, rebids';

comment on column MNSP_FILETRK.LASTCHANGED is
'Last date and time record changed';

alter table MNSP_FILETRK
   add constraint MNSP_FILETRK_PK primary key (SETTLEMENTDATE, OFFERDATE, PARTICIPANTID, FILENAME)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MNSP_FILETRK_LCX                                      */
/*==============================================================*/
create index MNSP_FILETRK_LCX on MNSP_FILETRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MNSP_INTERCONNECTOR                                   */
/*==============================================================*/
create table MNSP_INTERCONNECTOR (
   LINKID               VARCHAR2(10)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   INTERCONNECTORID     VARCHAR2(10),
   FROMREGION           VARCHAR2(10),
   TOREGION             VARCHAR2(10),
   MAXCAPACITY          NUMBER(5,0),
   TLF                  NUMBER(12,7),
   LHSFACTOR            NUMBER(12,7),
   METERFLOWCONSTANT    NUMBER(12,7),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   LASTCHANGED          DATE,
   FROM_REGION_TLF      NUMBER(12,7),
   TO_REGION_TLF        NUMBER(12,7)
)
   tablespace PSSTAB01;

comment on table MNSP_INTERCONNECTOR is
'MNSP_INTERCONNECTOR sets out attributes of each interconnector.';

comment on column MNSP_INTERCONNECTOR.LINKID is
'Identifier for each of the two MNSP Interconnector Links. Each link pertains to the direction from and to.';

comment on column MNSP_INTERCONNECTOR.EFFECTIVEDATE is
'Date when Interconnector becomes effective';

comment on column MNSP_INTERCONNECTOR.VERSIONNO is
'Version of data for other key data - a higher version for same key data will take precedence';

comment on column MNSP_INTERCONNECTOR.INTERCONNECTORID is
'Interconnector Identifier';

comment on column MNSP_INTERCONNECTOR.FROMREGION is
'Nominated source region for Interconnector';

comment on column MNSP_INTERCONNECTOR.TOREGION is
'Nominated destination region for Interconnector';

comment on column MNSP_INTERCONNECTOR.MAXCAPACITY is
'Maximum capacity';

comment on column MNSP_INTERCONNECTOR.TLF is
'Transmission Loss Factor (redundant from May 2012)';

comment on column MNSP_INTERCONNECTOR.LHSFACTOR is
'Factor applied to the LHS of constraint equations; set by AEMO';

comment on column MNSP_INTERCONNECTOR.METERFLOWCONSTANT is
'Obsolete; no longer applied. Ignore.';

comment on column MNSP_INTERCONNECTOR.AUTHORISEDDATE is
'Date of authorisation. Nominal date but required to enable Interconnector.';

comment on column MNSP_INTERCONNECTOR.AUTHORISEDBY is
'Authorising officer';

comment on column MNSP_INTERCONNECTOR.LASTCHANGED is
'Last date and time record changed';

comment on column MNSP_INTERCONNECTOR.FROM_REGION_TLF is
'Transmission Loss Factor for Link "From Region" end';

comment on column MNSP_INTERCONNECTOR.TO_REGION_TLF is
'Transmission Loss Factor for Link at "To Region" end';

alter table MNSP_INTERCONNECTOR
   add constraint MNSP_INTERCONNECTOR_PK primary key (LINKID, EFFECTIVEDATE, VERSIONNO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MNSP_INTERCONNECTOR_LCX                               */
/*==============================================================*/
create index MNSP_INTERCONNECTOR_LCX on MNSP_INTERCONNECTOR (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MNSP_OFFERTRK                                         */
/*==============================================================*/
create table MNSP_OFFERTRK (
   SETTLEMENTDATE       DATE                  not null,
   OFFERDATE            DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   FILENAME             VARCHAR2(40)          not null,
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MNSP_OFFERTRK is
'MNSP_OFFERTRK records all valid MNSPOFFERS loaded into the MMS system. The authorised date reflects the date and time of the load. MNSP_OFFERTRK is key for tracking MNSP bid submission.';

alter table MNSP_OFFERTRK
   add constraint MNSP_OFFERTRK_PK primary key (SETTLEMENTDATE, OFFERDATE, VERSIONNO, PARTICIPANTID, FILENAME)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MNSP_OFFERTRK_LCX                                     */
/*==============================================================*/
create index MNSP_OFFERTRK_LCX on MNSP_OFFERTRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MNSP_PARTICIPANT                                      */
/*==============================================================*/
create table MNSP_PARTICIPANT (
   INTERCONNECTORID     VARCHAR2(10)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MNSP_PARTICIPANT is
'MNSP_PARTICIPANT registers MNSP ownership.';

comment on column MNSP_PARTICIPANT.INTERCONNECTORID is
'Interconnector Identifier';

comment on column MNSP_PARTICIPANT.EFFECTIVEDATE is
'Calendar date when Interconnector ownership becomes effective';

comment on column MNSP_PARTICIPANT.VERSIONNO is
'Version of data for other key data - a higher version for same key data takes precedence';

comment on column MNSP_PARTICIPANT.PARTICIPANTID is
'Participant Identifier';

comment on column MNSP_PARTICIPANT.LASTCHANGED is
'Last date and time record changed';

alter table MNSP_PARTICIPANT
   add constraint MNSP_PARTICIPANT_PK primary key (INTERCONNECTORID, EFFECTIVEDATE, VERSIONNO, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MNSP_PARTICIPANT_LCX                                  */
/*==============================================================*/
create index MNSP_PARTICIPANT_LCX on MNSP_PARTICIPANT (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MNSP_PEROFFER                                         */
/*==============================================================*/
create table MNSP_PEROFFER (
   SETTLEMENTDATE       DATE                  not null,
   OFFERDATE            DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   LINKID               VARCHAR2(10)          not null,
   PERIODID             NUMBER(22,0)          not null,
   MAXAVAIL             NUMBER(6,0),
   BANDAVAIL1           NUMBER(6,0),
   BANDAVAIL2           NUMBER(6,0),
   BANDAVAIL3           NUMBER(6,0),
   BANDAVAIL4           NUMBER(6,0),
   BANDAVAIL5           NUMBER(6,0),
   BANDAVAIL6           NUMBER(6,0),
   BANDAVAIL7           NUMBER(6,0),
   BANDAVAIL8           NUMBER(6,0),
   BANDAVAIL9           NUMBER(6,0),
   BANDAVAIL10          NUMBER(6,0),
   LASTCHANGED          DATE,
   FIXEDLOAD            NUMBER(12,6),
   RAMPUPRATE           NUMBER(6,0),
   PASAAVAILABILITY     NUMBER(12,0),
   MR_CAPACITY          NUMBER(6,0)
)
   tablespace PSSTAB02;

comment on table MNSP_PEROFFER is
'MNSP_PEROFFER shows period by period availability and other period data pertaining to a specific bid and LinkID for the given Settlement Date.
MNSP_PEROFFER is a child to MNSP_DAYOFFER and links to MNSP_OFFERTRK.';

comment on column MNSP_PEROFFER.SETTLEMENTDATE is
'Market Date from which bid is active';

comment on column MNSP_PEROFFER.OFFERDATE is
'Offer date for bid';

comment on column MNSP_PEROFFER.VERSIONNO is
'Version of data for other key data - a higher version for same key data will take precedence';

comment on column MNSP_PEROFFER.PARTICIPANTID is
'Participant Identifier';

comment on column MNSP_PEROFFER.LINKID is
'Identifier for each of the two MNSP Interconnector Links. Each link pertains to the direction from and to.';

comment on column MNSP_PEROFFER.PERIODID is
'Trading Interval number';

comment on column MNSP_PEROFFER.MAXAVAIL is
'Maximum planned availability MW';

comment on column MNSP_PEROFFER.BANDAVAIL1 is
'Band Availability for current Period';

comment on column MNSP_PEROFFER.BANDAVAIL2 is
'Band Availability for current Period';

comment on column MNSP_PEROFFER.BANDAVAIL3 is
'Band Availability for current Period';

comment on column MNSP_PEROFFER.BANDAVAIL4 is
'Band Availability for current Period';

comment on column MNSP_PEROFFER.BANDAVAIL5 is
'Band Availability for current Period';

comment on column MNSP_PEROFFER.BANDAVAIL6 is
'Band Availability for current Period';

comment on column MNSP_PEROFFER.BANDAVAIL7 is
'Band Availability for current Period';

comment on column MNSP_PEROFFER.BANDAVAIL8 is
'Band Availability for current Period';

comment on column MNSP_PEROFFER.BANDAVAIL9 is
'Band Availability for current Period';

comment on column MNSP_PEROFFER.BANDAVAIL10 is
'Band Availability for current Period';

comment on column MNSP_PEROFFER.LASTCHANGED is
'Last date and time record changed';

comment on column MNSP_PEROFFER.FIXEDLOAD is
'Inflexibility flag and availability. Fixed unit output MW. A value of zero means no fixed load so the unit is dispatched according to bid and market (rather than zero fixed load)';

comment on column MNSP_PEROFFER.RAMPUPRATE is
'Ramp rate (MW / min) in the positive direction of flow for this MNSP link for this half-hour period';

comment on column MNSP_PEROFFER.PASAAVAILABILITY is
'Allows for future use for energy bids, being the physical plant capability including any capability potentially available within 24 hours';

comment on column MNSP_PEROFFER.MR_CAPACITY is
'Mandatory Restriction Offer amount';

alter table MNSP_PEROFFER
   add constraint MNSP_PEROFFER_PK primary key (SETTLEMENTDATE, OFFERDATE, VERSIONNO, PARTICIPANTID, LINKID, PERIODID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: MNSP_PEROFFER_LCX                                     */
/*==============================================================*/
create index MNSP_PEROFFER_LCX on MNSP_PEROFFER (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: MR_DAYOFFER_STACK                                     */
/*==============================================================*/
create table MR_DAYOFFER_STACK (
   MR_DATE              DATE                  not null,
   REGIONID             VARCHAR2(10)          not null,
   VERSION_DATETIME     DATE                  not null,
   STACK_POSITION       NUMBER(3,0)           not null,
   DUID                 VARCHAR2(10),
   AUTHORISED           NUMBER(1,0),
   OFFER_SETTLEMENTDATE DATE,
   OFFER_OFFERDATE      DATE,
   OFFER_VERSIONNO      NUMBER(3,0),
   OFFER_TYPE           VARCHAR2(20),
   LAOF                 NUMBER(16,6),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MR_DAYOFFER_STACK is
'MR_DAYOFFER_STACK defines the Stack order for each version of the Acceptance Schedule, including all units submitting MR offers for that event. MR_DAYOFFER_STACK is the child to MR_EVENT_SCHEDULE, and parent to MR_PEROFFER_STACK.';

comment on column MR_DAYOFFER_STACK.MR_DATE is
'Mandatory Restriction imposition date';

comment on column MR_DAYOFFER_STACK.REGIONID is
'Unique RegionID';

comment on column MR_DAYOFFER_STACK.VERSION_DATETIME is
'Allows many Stack versions';

comment on column MR_DAYOFFER_STACK.STACK_POSITION is
'Loss Adjusted Offer Factor Stack order starting at 1';

comment on column MR_DAYOFFER_STACK.DUID is
'Dispatchable Unit ID or LinkID';

comment on column MR_DAYOFFER_STACK.AUTHORISED is
'Confirms the unit is allowed to Contribute MR Capacity';

comment on column MR_DAYOFFER_STACK.OFFER_SETTLEMENTDATE is
'Foreign key reference to XXXX_DayOffer.SettlementDate';

comment on column MR_DAYOFFER_STACK.OFFER_OFFERDATE is
'Foreign key reference to XXXX_DayOffer.OfferDate';

comment on column MR_DAYOFFER_STACK.OFFER_VERSIONNO is
'Foreign key reference to XXXX_DayOffer.VersionNo';

comment on column MR_DAYOFFER_STACK.OFFER_TYPE is
'Source tables - ENERGY or MNSP';

comment on column MR_DAYOFFER_STACK.LAOF is
'Loss Adjusted Offer Factor = TLF times MR_Factor';

comment on column MR_DAYOFFER_STACK.LASTCHANGED is
'Date and time the record was last inserted/modified';

alter table MR_DAYOFFER_STACK
   add constraint MR_DAYOFFER_STACK_PK primary key (MR_DATE, REGIONID, VERSION_DATETIME, STACK_POSITION)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MR_DAYOFFER_STACK_LCX                                 */
/*==============================================================*/
create index MR_DAYOFFER_STACK_LCX on MR_DAYOFFER_STACK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MR_EVENT                                              */
/*==============================================================*/
create table MR_EVENT (
   MR_DATE              DATE                  not null,
   REGIONID             VARCHAR2(10)          not null,
   DESCRIPTION          VARCHAR2(200),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(20),
   OFFER_CUT_OFF_TIME   DATE,
   SETTLEMENT_COMPLETE  NUMBER(1,0),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MR_EVENT is
'MR_EVENT defines an MR Event for a given region on a specific trading date.';

comment on column MR_EVENT.MR_DATE is
'Mandatory Restriction imposition date';

comment on column MR_EVENT.REGIONID is
'Unique RegionID';

comment on column MR_EVENT.DESCRIPTION is
'Description of MR';

comment on column MR_EVENT.AUTHORISEDDATE is
'Required for MR_Event to take effect';

comment on column MR_EVENT.AUTHORISEDBY is
'Ignored - Tracking purpose only';

comment on column MR_EVENT.OFFER_CUT_OFF_TIME is
'Cut off after when new Offers and Scaling Factor changes are disallowed';

comment on column MR_EVENT.SETTLEMENT_COMPLETE is
'Flag:1 = MR settlement figures locked. Do not recalculate, 	0 = MR settlements to be recalculated';

comment on column MR_EVENT.LASTCHANGED is
'Date/Time record inserted/modified';

alter table MR_EVENT
   add constraint MR_EVENT_PK primary key (MR_DATE, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MR_EVENT_LCX                                          */
/*==============================================================*/
create index MR_EVENT_LCX on MR_EVENT (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MR_EVENT_SCHEDULE                                     */
/*==============================================================*/
create table MR_EVENT_SCHEDULE (
   MR_DATE              DATE                  not null,
   REGIONID             VARCHAR2(10)          not null,
   VERSION_DATETIME     DATE                  not null,
   DEMAND_EFFECTIVEDATE DATE,
   DEMAND_OFFERDATE     DATE,
   DEMAND_VERSIONNO     NUMBER(3,0),
   AUTHORISEDBY         VARCHAR2(20),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MR_EVENT_SCHEDULE is
'MR_EVENT_SCHEDULE defines the Stack version of the Acceptance Schedule and is the parent table to MR_DayOffer_Stack and MR_PerOffer_Stack.';

comment on column MR_EVENT_SCHEDULE.MR_DATE is
'Mandatory Restriction imposition date';

comment on column MR_EVENT_SCHEDULE.REGIONID is
'Unique RegionID';

comment on column MR_EVENT_SCHEDULE.VERSION_DATETIME is
'Effective Date/Time of Schedule; Allows many Stack versions';

comment on column MR_EVENT_SCHEDULE.DEMAND_EFFECTIVEDATE is
'Foreign key reference to ResDemandTrk.EffectiveDate';

comment on column MR_EVENT_SCHEDULE.DEMAND_OFFERDATE is
'Foreign key reference to ResDemandTrk.OfferDate';

comment on column MR_EVENT_SCHEDULE.DEMAND_VERSIONNO is
'Foreign key reference to ResDemandTrk.VersionNo';

comment on column MR_EVENT_SCHEDULE.AUTHORISEDBY is
'Authorised person confirming Offer Stack (AKA Acceptance)';

comment on column MR_EVENT_SCHEDULE.AUTHORISEDDATE is
'Date and time the Offer Stack confirmed';

comment on column MR_EVENT_SCHEDULE.LASTCHANGED is
'Date and time the record was inserted/modified';

alter table MR_EVENT_SCHEDULE
   add constraint MR_EVENT_SCHEDULE_PK primary key (MR_DATE, REGIONID, VERSION_DATETIME)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MR_EVENT_SCHEDULE_LCX                                 */
/*==============================================================*/
create index MR_EVENT_SCHEDULE_LCX on MR_EVENT_SCHEDULE (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MR_PEROFFER_STACK                                     */
/*==============================================================*/
create table MR_PEROFFER_STACK (
   MR_DATE              DATE                  not null,
   REGIONID             VARCHAR2(10)          not null,
   VERSION_DATETIME     DATE                  not null,
   STACK_POSITION       NUMBER(3,0)           not null,
   PERIODID             NUMBER(3,0)           not null,
   DUID                 VARCHAR2(10),
   ACCEPTED_CAPACITY    NUMBER(6,0),
   DEDUCTED_CAPACITY    NUMBER(6,0),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MR_PEROFFER_STACK is
'MR_PEROFFER_STACK defines the accepted capacity on a period basis for the Acceptance Schedule, is a child table to MR_DayOffer_Stack and only includes records or units with accepted_capacity > 0 for the specific period.';

comment on column MR_PEROFFER_STACK.MR_DATE is
'Mandatory Restriction imposition date';

comment on column MR_PEROFFER_STACK.REGIONID is
'Unique RegionID';

comment on column MR_PEROFFER_STACK.VERSION_DATETIME is
'Allows many Period Stack versions for the one Scaling Factor stack';

comment on column MR_PEROFFER_STACK.STACK_POSITION is
'LAOF Stack order';

comment on column MR_PEROFFER_STACK.PERIODID is
'Trade Period for the MR Offer';

comment on column MR_PEROFFER_STACK.DUID is
'Dispatchable Unit ID or LinkID. Only required here for CSV reports';

comment on column MR_PEROFFER_STACK.ACCEPTED_CAPACITY is
'MR Capacity to be Dispatched';

comment on column MR_PEROFFER_STACK.DEDUCTED_CAPACITY is
'Requested capacity reduction amount';

comment on column MR_PEROFFER_STACK.LASTCHANGED is
'Date and time the record was last  inserted/modified';

alter table MR_PEROFFER_STACK
   add constraint MR_PEROFFER_STACK_PK primary key (MR_DATE, REGIONID, VERSION_DATETIME, STACK_POSITION, PERIODID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MR_PEROFFER_STACK_LCX                                 */
/*==============================================================*/
create index MR_PEROFFER_STACK_LCX on MR_PEROFFER_STACK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MTPASACONSTRAINTSOLUTION_D                            */
/*==============================================================*/
create table MTPASACONSTRAINTSOLUTION_D (
   DATETIME             DATE                  not null,
   CONSTRAINT_ID        VARCHAR2(20)          not null,
   DEGREE_OF_VIOLATION  NUMBER(16,6),
   LASTCHANGED          DATE,
   RUN_DATETIME         DATE
)
   tablespace PSSTAB01;

comment on table MTPASACONSTRAINTSOLUTION_D is
'MTPASACONSTRAINTSOLUTION_D sets out MT PASA constraint solution results, where constraints are binding.';

comment on column MTPASACONSTRAINTSOLUTION_D.DATETIME is
'Date constraint is binding';

comment on column MTPASACONSTRAINTSOLUTION_D.CONSTRAINT_ID is
'Constraint Identifier';

comment on column MTPASACONSTRAINTSOLUTION_D.DEGREE_OF_VIOLATION is
'The degree in MW by which the constraint would be violated if the solution could not solve. This could be due to incorrect penalties etc. This figure should always be 0.';

comment on column MTPASACONSTRAINTSOLUTION_D.LASTCHANGED is
'Last changed data and time.';

comment on column MTPASACONSTRAINTSOLUTION_D.RUN_DATETIME is
'The run date and time';

alter table MTPASACONSTRAINTSOLUTION_D
   add constraint MTPASACONSTRAINTSOLUTION_D_PK primary key (DATETIME, CONSTRAINT_ID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MTPASACONSOLUTION_D_LCX                               */
/*==============================================================*/
create index MTPASACONSOLUTION_D_LCX on MTPASACONSTRAINTSOLUTION_D (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MTPASAINTERCONNECTORSOLUTION_D                        */
/*==============================================================*/
create table MTPASAINTERCONNECTORSOLUTION_D (
   DATETIME             DATE                  not null,
   INTERCONNECTOR_ID    VARCHAR2(12)          not null,
   POSITIVE_INTERCONNECTOR_FLOW NUMBER(16,6),
   POSITIVE_TRANSFER_LIMITS NUMBER(16,6),
   POSITIVE_BINDING     VARCHAR2(10),
   NEGATIVE_INTERCONNECTOR_FLOW NUMBER(16,6),
   NEGATIVE_TRANSFER_LIMITS NUMBER(16,6),
   NEGATIVE_BINDING     VARCHAR2(10),
   LASTCHANGED          DATE,
   RUN_DATETIME         DATE
)
   tablespace PSSTAB01;

comment on table MTPASAINTERCONNECTORSOLUTION_D is
'MTPASAINTERCONNECTORSOLUTION_D shows interconnector results for MT PASA, shown region by region.';

comment on column MTPASAINTERCONNECTORSOLUTION_D.DATETIME is
'Date of results. One record for each day for next two years.';

comment on column MTPASAINTERCONNECTORSOLUTION_D.INTERCONNECTOR_ID is
'Interconnector Identifier';

comment on column MTPASAINTERCONNECTORSOLUTION_D.POSITIVE_INTERCONNECTOR_FLOW is
'The MW flow out';

comment on column MTPASAINTERCONNECTORSOLUTION_D.POSITIVE_TRANSFER_LIMITS is
'The MW transfer limits out';

comment on column MTPASAINTERCONNECTORSOLUTION_D.POSITIVE_BINDING is
'Indication of a binding limit in the out direction';

comment on column MTPASAINTERCONNECTORSOLUTION_D.NEGATIVE_INTERCONNECTOR_FLOW is
'The MW flow in';

comment on column MTPASAINTERCONNECTORSOLUTION_D.NEGATIVE_TRANSFER_LIMITS is
'the MW transfer limits in';

comment on column MTPASAINTERCONNECTORSOLUTION_D.NEGATIVE_BINDING is
'Indication of a binding limit in the in direction';

comment on column MTPASAINTERCONNECTORSOLUTION_D.LASTCHANGED is
'Last change date and time';

comment on column MTPASAINTERCONNECTORSOLUTION_D.RUN_DATETIME is
'The run date and time';

alter table MTPASAINTERCONNECTORSOLUTION_D
   add constraint MTPASAINTERCONSOLUTION_D_PK primary key (DATETIME, INTERCONNECTOR_ID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MTPASAINTERCONSOLUTION_D_LCX                          */
/*==============================================================*/
create index MTPASAINTERCONSOLUTION_D_LCX on MTPASAINTERCONNECTORSOLUTION_D (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MTPASAREGIONSOLUTION_D                                */
/*==============================================================*/
create table MTPASAREGIONSOLUTION_D (
   DATETIME             DATE                  not null,
   REGION_ID            VARCHAR2(12)          not null,
   RUN_DATETIME         DATE,
   RESERVE_CONDITION    VARCHAR2(50),
   RESERVE_SURPLUS      NUMBER(16,6),
   CAPACITY_REQUIREMENT NUMBER(16,6),
   MINIMUM_RESERVE_REQUIREMENT NUMBER(16,6),
   REGION_DEMAND_10POE  NUMBER(16,6),
   DEMAND_MINUS_SCHEDULED_LOAD NUMBER(16,6),
   CONSTRAINED_CAPACITY NUMBER(16,6),
   UNCONSTRAINED_CAPACITY NUMBER(16,6),
   NET_INTERCHANGE      NUMBER(16,6),
   ENERGY_REQUIREMENT_10POE NUMBER(16,6),
   REPORTED_BLOCK_ID    NUMBER(16,6),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MTPASAREGIONSOLUTION_D is
'MTPASAREGIONSOLUTION_D shows region results for MT PASA, showing predicted demand and any capacity limits.';

comment on column MTPASAREGIONSOLUTION_D.DATETIME is
'Date of results. One record for each day for next two years.';

comment on column MTPASAREGIONSOLUTION_D.REGION_ID is
'Region Identifier';

comment on column MTPASAREGIONSOLUTION_D.RUN_DATETIME is
'The run date and time';

comment on column MTPASAREGIONSOLUTION_D.RESERVE_CONDITION is
'The regional reserve condition';

comment on column MTPASAREGIONSOLUTION_D.RESERVE_SURPLUS is
'Regional reserve surplus value';

comment on column MTPASAREGIONSOLUTION_D.CAPACITY_REQUIREMENT is
'Capacity in MW required to meet demand';

comment on column MTPASAREGIONSOLUTION_D.MINIMUM_RESERVE_REQUIREMENT is
'Minimum required regional reserve value';

comment on column MTPASAREGIONSOLUTION_D.REGION_DEMAND_10POE is
'Regional 10% Probability of Exceedance demand forecast value';

comment on column MTPASAREGIONSOLUTION_D.DEMAND_MINUS_SCHEDULED_LOAD is
'Regional demand minus the scheduled load value';

comment on column MTPASAREGIONSOLUTION_D.CONSTRAINED_CAPACITY is
'The total regional capacity due to energy and network constraints';

comment on column MTPASAREGIONSOLUTION_D.UNCONSTRAINED_CAPACITY is
'The total regional capacity, subject to network constraints.';

comment on column MTPASAREGIONSOLUTION_D.NET_INTERCHANGE is
'Regional net MW import via interconnectors';

comment on column MTPASAREGIONSOLUTION_D.ENERGY_REQUIREMENT_10POE is
'Regional energy required to meet demand';

comment on column MTPASAREGIONSOLUTION_D.REPORTED_BLOCK_ID is
'The load duration curve block that is recorded in the report.';

comment on column MTPASAREGIONSOLUTION_D.LASTCHANGED is
'Last change date and time.';

alter table MTPASAREGIONSOLUTION_D
   add constraint MTPASAREGIONSOLUTION_D_PK primary key (DATETIME, REGION_ID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MTPASAREGIONSOLUTION_D_LCX                            */
/*==============================================================*/
create index MTPASAREGIONSOLUTION_D_LCX on MTPASAREGIONSOLUTION_D (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MTPASA_CASERESULT                                     */
/*==============================================================*/
create table MTPASA_CASERESULT (
   RUN_DATETIME         DATE                  not null,
   RUN_NO               NUMBER(4)             not null,
   PLEXOS_VERSION       VARCHAR2(20),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MTPASA_CASERESULT is
'MTPASA solution header table';

comment on column MTPASA_CASERESULT.RUN_DATETIME is
'Date processing of the run begins.';

comment on column MTPASA_CASERESULT.RUN_NO is
'Unique run id.';

comment on column MTPASA_CASERESULT.PLEXOS_VERSION is
'Version of PLEXOS used';

comment on column MTPASA_CASERESULT.LASTCHANGED is
'Last date and time record changed';

alter table MTPASA_CASERESULT
   add constraint MTPASA_CASERESULT_PK primary key (RUN_DATETIME, RUN_NO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: MTPASA_CASESOLUTION                                   */
/*==============================================================*/
create table MTPASA_CASESOLUTION (
   RUN_DATETIME         DATE                  not null,
   RUN_NO               NUMBER(3,0)           not null,
   PASAVERSION          VARCHAR2(10),
   RESERVECONDITION     NUMBER(1,0),
   LORCONDITION         NUMBER(1,0),
   CAPACITYOBJFUNCTION  NUMBER(12,3),
   CAPACITYOPTION       NUMBER(12,3),
   MAXSURPLUSRESERVEOPTION NUMBER(12,3),
   MAXSPARECAPACITYOPTION NUMBER(12,3),
   INTERCONNECTORFLOWPENALTY NUMBER(12,3),
   LASTCHANGED          DATE,
   RUNTYPE              VARCHAR2(50),
   RELIABILITYLRCDEMANDOPTION NUMBER(12,3),
   OUTAGELRCDEMANDOPTION NUMBER(12,3),
   LORDEMANDOPTION      NUMBER(12,3),
   RELIABILITYLRCCAPACITYOPTION VARCHAR2(10),
   OUTAGELRCCAPACITYOPTION VARCHAR2(10),
   LORCAPACITYOPTION    VARCHAR2(10),
   LORUIGFOPTION        NUMBER(3,0),
   RELIABILITYLRCUIGFOPTION NUMBER(3,0),
   OUTAGELRCUIGFOPTION  NUMBER(3,0)
)
   tablespace PSSTAB01;

comment on table MTPASA_CASESOLUTION is
'MTPASA_CASESOLUTION is obsolete from 2017 End of Year DM4.27 Release. 
MTPASA_CASESOLUTION holds one record for each entire solution.
Change Notice 379 announced the replacement of the MT PASA data model so all MTPASAxxx tables become obsolete, replaced by MTPASA_xxx tables.';

comment on column MTPASA_CASESOLUTION.RUN_DATETIME is
'Date processing of the run begins.  Generated from the solution file caseid';

comment on column MTPASA_CASESOLUTION.RUN_NO is
'Unique run id.  Generated from the solution file caseid';

comment on column MTPASA_CASESOLUTION.PASAVERSION is
'Version of the PASA solver used to solve this case';

comment on column MTPASA_CASESOLUTION.RESERVECONDITION is
'Low Reserve Condition (LRC) flag for the case (1 - LRC in the case, 0 - No LRCs in the case) for capacity run';

comment on column MTPASA_CASESOLUTION.LORCONDITION is
'Lack of Reserve Condition (LOR) flag for the case indicates the most severe condition in the case  (3 = LOR3, 2 = LOR2, 1 = LOR1, 0 = No LOR)';

comment on column MTPASA_CASESOLUTION.CAPACITYOBJFUNCTION is
'Objective Function from the Capacity Adequacy run';

comment on column MTPASA_CASESOLUTION.CAPACITYOPTION is
'Not populated as of 2005 End of Year Release; was the demand forecast used for capacity adequacy assessment: 0 = no assessment, 1 = 10%, 2 = 50%, 3 = 90%';

comment on column MTPASA_CASESOLUTION.MAXSURPLUSRESERVEOPTION is
'Not populated as of 2005 End of Year Release; was the demand forecast used for assessment of Maximum surplus Reserve: 0 = no assessment, 1 = 10%, 2 = 50%, 3 = 90%';

comment on column MTPASA_CASESOLUTION.MAXSPARECAPACITYOPTION is
'Not populated as of 2005 End of Year Release; was the demand forecast used for assessment of Maximum Spare Capacity: 0 = no assessment, 1 = 10%, 2 = 50%, 3 = 90%';

comment on column MTPASA_CASESOLUTION.INTERCONNECTORFLOWPENALTY is
'The penalty for non-zero interconnector flow';

comment on column MTPASA_CASESOLUTION.LASTCHANGED is
'Date and time the record was created or modified';

comment on column MTPASA_CASESOLUTION.RUNTYPE is
'Discontinued in Dec 2005; was description of the constraints included in this run, being either System Normal and Planned Outage Constraints or System Normal Constraints Only';

comment on column MTPASA_CASESOLUTION.RELIABILITYLRCDEMANDOPTION is
'Specifies the Probability of Exceedence (POE) demand forecast for Reliability LRC assessment (0 if no assessment, 10 for 10%, 50 for 50%, 90 for 90%)';

comment on column MTPASA_CASESOLUTION.OUTAGELRCDEMANDOPTION is
'Specifies the Probability of Exceedence (POE) demand forecast for outage LRC assessment (0 if no assessment, 10 for 10%, 50 for 50%, 90 for 90%)';

comment on column MTPASA_CASESOLUTION.LORDEMANDOPTION is
'Specifies the Probability of Exceedence (POE) demand forecast for LOR assessment (0 if no assessment, 10 for 10%, 50 for 50%, 90 for 90%)';

comment on column MTPASA_CASESOLUTION.RELIABILITYLRCCAPACITYOPTION is
'Generation Availability to be used in Reliability LRC run (either PASA or MARKET)';

comment on column MTPASA_CASESOLUTION.OUTAGELRCCAPACITYOPTION is
'Generation Availability to be used in Outage LRC run (either PASA or MARKET)';

comment on column MTPASA_CASESOLUTION.LORCAPACITYOPTION is
'Generation Availability to be used in LOR run (either PASA or MARKET)';

comment on column MTPASA_CASESOLUTION.LORUIGFOPTION is
'UIGF POE forecast availability used for this option';

comment on column MTPASA_CASESOLUTION.RELIABILITYLRCUIGFOPTION is
'UIGF POE forecast availability used for this option';

comment on column MTPASA_CASESOLUTION.OUTAGELRCUIGFOPTION is
'UIGF POE forecast availability used for this option';

alter table MTPASA_CASESOLUTION
   add constraint MTPASA_CASESOLUTION_PK primary key (RUN_DATETIME, RUN_NO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MTPASA_CASESOLUTION_LCX                               */
/*==============================================================*/
create index MTPASA_CASESOLUTION_LCX on MTPASA_CASESOLUTION (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MTPASA_CASE_SET                                       */
/*==============================================================*/
create table MTPASA_CASE_SET (
   RUN_DATETIME         DATE                  not null,
   RUN_NO               NUMBER(3,0)           not null,
   CASESETID            NUMBER(3,0),
   RUNTYPEID            NUMBER(1,0),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MTPASA_CASE_SET is
'MTPASA_CASE_SET is obsolete from 2005 End of Year Release. The RUNTYPE added to the primary key of the detail tables for MTPASA allows for the different types of runs for each case.
MTPASA_CASE_SET allows a MT PASA scenario to be linked across runs.';

comment on column MTPASA_CASE_SET.RUN_DATETIME is
'Date processing of the run begins. Generated from the solution file CASEID.';

comment on column MTPASA_CASE_SET.RUN_NO is
'Unique run id. Generated from the solution file CASEID.';

comment on column MTPASA_CASE_SET.CASESETID is
'Unique id to link a set of cases run from the same inputs';

comment on column MTPASA_CASE_SET.RUNTYPEID is
'Unique id for type of run, being either';

comment on column MTPASA_CASE_SET.LASTCHANGED is
'Date the solution was loaded';

alter table MTPASA_CASE_SET
   add constraint MTPASA_CASE_SET_PK primary key (RUN_DATETIME, RUN_NO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: MTPASA_CASE_SET_LCX                                   */
/*==============================================================*/
create index MTPASA_CASE_SET_LCX on MTPASA_CASE_SET (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: MTPASA_CONSTRAINTRESULT                               */
/*==============================================================*/
create table MTPASA_CONSTRAINTRESULT (
   RUN_DATETIME         DATE                  not null,
   RUN_NO               NUMBER(4)             not null,
   RUNTYPE              VARCHAR2(20)          not null,
   DEMAND_POE_TYPE      VARCHAR2(20)          not null,
   DAY                  DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   EFFECTIVEDATE        DATE,
   VERSIONNO            NUMBER(3,0),
   PERIODID             NUMBER(3,0),
   PROBABILITYOFBINDING NUMBER(8,5),
   PROBABILITYOFVIOLATION NUMBER(8,5),
   CONSTRAINTVIOLATION90 NUMBER(12,2),
   CONSTRAINTVIOLATION50 NUMBER(12,2),
   CONSTRAINTVIOLATION10 NUMBER(12,2),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table MTPASA_CONSTRAINTRESULT is
'Constraint results for Binding or Violating Constraints';

comment on column MTPASA_CONSTRAINTRESULT.RUN_DATETIME is
'Date processing of the run begins.';

comment on column MTPASA_CONSTRAINTRESULT.RUN_NO is
'Unique run id.';

comment on column MTPASA_CONSTRAINTRESULT.RUNTYPE is
'Type of run. Always RELIABILITY';

comment on column MTPASA_CONSTRAINTRESULT.DEMAND_POE_TYPE is
'Demand POE type used. Value is POE10';

comment on column MTPASA_CONSTRAINTRESULT.DAY is
'Day this result is for';

comment on column MTPASA_CONSTRAINTRESULT.CONSTRAINTID is
'The unique identifier for the constraint. Only binding or violating constraints are reported';

comment on column MTPASA_CONSTRAINTRESULT.EFFECTIVEDATE is
'The effective date of the constraint used';

comment on column MTPASA_CONSTRAINTRESULT.VERSIONNO is
'The version of the constraint used';

comment on column MTPASA_CONSTRAINTRESULT.PERIODID is
'Half hourly period reported, selected as period of maximum NEM scheduled demand (calculated as maximum of scheduled demands, averaged across iterations and reference years)';

comment on column MTPASA_CONSTRAINTRESULT.PROBABILITYOFBINDING is
'Proportion of a constraint binding, across iterations and reference years';

comment on column MTPASA_CONSTRAINTRESULT.PROBABILITYOFVIOLATION is
'Proportion of a constraint violating, across iterations and reference years';

comment on column MTPASA_CONSTRAINTRESULT.CONSTRAINTVIOLATION90 is
'The 90th percentile violation degree for this constraint, across iterations and reference years (MW)';

comment on column MTPASA_CONSTRAINTRESULT.CONSTRAINTVIOLATION50 is
'The 50th percentile violation degree for this constraint, across iterations and reference years (MW)';

comment on column MTPASA_CONSTRAINTRESULT.CONSTRAINTVIOLATION10 is
'The 10th percentile violation degree for this constraint, across iterations and reference years (MW)';

comment on column MTPASA_CONSTRAINTRESULT.LASTCHANGED is
'Last date and time record changed';

alter table MTPASA_CONSTRAINTRESULT
   add constraint MTPASA_CONSTRAINTRESULT_PK primary key (RUN_DATETIME, RUN_NO, RUNTYPE, DEMAND_POE_TYPE, DAY, CONSTRAINTID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: MTPASA_CONSTRAINTSOLUTION                             */
/*==============================================================*/
create table MTPASA_CONSTRAINTSOLUTION (
   RUN_DATETIME         DATE                  not null,
   RUN_NO               NUMBER(3,0)           not null,
   ENERGYBLOCK          DATE                  not null,
   DAY                  DATE                  not null,
   LDCBLOCK             NUMBER(3,0)           not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   CAPACITYRHS          NUMBER(12,2),
   CAPACITYMARGINALVALUE NUMBER(12,2),
   CAPACITYVIOLATIONDEGREE NUMBER(12,2),
   LASTCHANGED          DATE,
   RUNTYPE              VARCHAR2(20)         default 'OUTAGE_LRC'  not null
)
   tablespace PSSTAB02;

comment on table MTPASA_CONSTRAINTSOLUTION is
'MTPASA_CONSTRAINTSOLUTION is obsolete from 2017 End of Year DM4.27 Release. 
The MTPASA_CONSTRAINTSOLUTION table holds the binding and violated constraint results from the capacity evaluation, including the RHS value.
Change Notice 379 announced the replacement of the MT PASA data model so all MTPASAxxx tables become obsolete, replaced by MTPASA_xxx tables.';

comment on column MTPASA_CONSTRAINTSOLUTION.RUN_DATETIME is
'Date processing of the run begins';

comment on column MTPASA_CONSTRAINTSOLUTION.RUN_NO is
'Unique run id.  Generated from the solution file caseid';

comment on column MTPASA_CONSTRAINTSOLUTION.ENERGYBLOCK is
'Sunday at start of the week for this solutions energy block.  Generated from the solution file energy block';

comment on column MTPASA_CONSTRAINTSOLUTION.DAY is
'Day this solution is for.  Generated from the solution file periodid';

comment on column MTPASA_CONSTRAINTSOLUTION.LDCBLOCK is
'LDC block this solution is for.  Generated from the solution file periodid';

comment on column MTPASA_CONSTRAINTSOLUTION.CONSTRAINTID is
'The unique identifier for the constraint';

comment on column MTPASA_CONSTRAINTSOLUTION.CAPACITYRHS is
'The RHS value in the capacity evaluation';

comment on column MTPASA_CONSTRAINTSOLUTION.CAPACITYMARGINALVALUE is
'Capacity adequacy assessment marginal value; 0 if not binding';

comment on column MTPASA_CONSTRAINTSOLUTION.CAPACITYVIOLATIONDEGREE is
'Capacity adequacy assessment violation degree; 0 if not violating';

comment on column MTPASA_CONSTRAINTSOLUTION.LASTCHANGED is
'Date the solution was loaded';

comment on column MTPASA_CONSTRAINTSOLUTION.RUNTYPE is
'Type of run.  Values are RELIABILITY_LRC and OUTAGE_LRC';

alter table MTPASA_CONSTRAINTSOLUTION
   add constraint MTPASA_CONSTRAINTSOLUTION_PK primary key (RUN_DATETIME, RUN_NO, RUNTYPE, ENERGYBLOCK, DAY, LDCBLOCK, CONSTRAINTID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: MTPASA_CONSTRAINTSOLUTION_NDX2                        */
/*==============================================================*/
create index MTPASA_CONSTRAINTSOLUTION_NDX2 on MTPASA_CONSTRAINTSOLUTION (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: MTPASA_CONSTRAINTSUMMARY                              */
/*==============================================================*/
create table MTPASA_CONSTRAINTSUMMARY (
   RUN_DATETIME         DATE                  not null,
   RUN_NO               NUMBER(4)             not null,
   RUNTYPE              VARCHAR2(20)          not null,
   DEMAND_POE_TYPE      VARCHAR2(20)          not null,
   DAY                  DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   EFFECTIVEDATE        DATE,
   VERSIONNO            NUMBER(3,0),
   AGGREGATION_PERIOD   VARCHAR2(20)          not null,
   CONSTRAINTHOURSBINDING NUMBER(12,2),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table MTPASA_CONSTRAINTSUMMARY is
'Constraint Summary results over aggregation periods';

comment on column MTPASA_CONSTRAINTSUMMARY.RUN_DATETIME is
'Date processing of the run begins.';

comment on column MTPASA_CONSTRAINTSUMMARY.RUN_NO is
'Unique run id.';

comment on column MTPASA_CONSTRAINTSUMMARY.RUNTYPE is
'Type of run. Always RELIABILITY';

comment on column MTPASA_CONSTRAINTSUMMARY.DEMAND_POE_TYPE is
'Demand POE type used. Value is POE10';

comment on column MTPASA_CONSTRAINTSUMMARY.DAY is
'Day this result is for';

comment on column MTPASA_CONSTRAINTSUMMARY.CONSTRAINTID is
'The unique identifier for the constraint. Only binding or violating constraints are reported';

comment on column MTPASA_CONSTRAINTSUMMARY.EFFECTIVEDATE is
'The effective date of the constraint used';

comment on column MTPASA_CONSTRAINTSUMMARY.VERSIONNO is
'The version of the constraintID';

comment on column MTPASA_CONSTRAINTSUMMARY.AGGREGATION_PERIOD is
'Period data is aggregated over. Values are PEAK, SHOULDER, OFFPEAK. PEAK = 14:00-19:59, SHOULDER = 07:00-13:59 and 20:00-21:59, OFFPEAK = 22:00-06:59';

comment on column MTPASA_CONSTRAINTSUMMARY.CONSTRAINTHOURSBINDING is
'Constraint hours binding or violating for period, averaged across iterations and reference years';

comment on column MTPASA_CONSTRAINTSUMMARY.LASTCHANGED is
'Last date and time record changed';

alter table MTPASA_CONSTRAINTSUMMARY
   add constraint MTPASA_CONSTRAINTSUMMARY_PK primary key (RUN_DATETIME, RUN_NO, RUNTYPE, DEMAND_POE_TYPE, DAY, CONSTRAINTID, AGGREGATION_PERIOD)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: MTPASA_DUIDAVAILABILITY                               */
/*==============================================================*/
create table MTPASA_DUIDAVAILABILITY (
   PUBLISH_DATETIME     DATE                  not null,
   DAY                  DATE                  not null,
   REGIONID             VARCHAR2(20)          not null,
   DUID                 VARCHAR2(20)          not null,
   PASAAVAILABILITY     NUMBER(12,0),
   LATEST_OFFER_DATETIME DATE,
   LASTCHANGED          DATE,
   CARRYOVERSTATUS      NUMBER(1,0),
   PASAUNITSTATE        VARCHAR2(20),
   PASARECALLTIME       NUMBER(4)
)
   tablespace PSSTAB03;

comment on table MTPASA_DUIDAVAILABILITY is
'Offered PASA Availability of the scheduled generator DUID for each day over the Medium Term PASA period. The data in this table is input data to the MT PASA process it is not part of the MTPASA solution. The availability does not reflect any energy limitations in the MT PASA offers';

comment on column MTPASA_DUIDAVAILABILITY.PUBLISH_DATETIME is
'Date Time the report was published.';

comment on column MTPASA_DUIDAVAILABILITY.DAY is
'Date on which the PASA availability of DUID applies.';

comment on column MTPASA_DUIDAVAILABILITY.REGIONID is
'NEM Region.';

comment on column MTPASA_DUIDAVAILABILITY.DUID is
'NEM DUID.';

comment on column MTPASA_DUIDAVAILABILITY.PASAAVAILABILITY is
'Offered PASA Availability of Scheduled generator DUID for the day.';

comment on column MTPASA_DUIDAVAILABILITY.LATEST_OFFER_DATETIME is
'Date Time of the latest offer used for DUID for this date.';

comment on column MTPASA_DUIDAVAILABILITY.LASTCHANGED is
'Last date and time record changed';

comment on column MTPASA_DUIDAVAILABILITY.CARRYOVERSTATUS is
'Status of a reported capacity value (e.g. 1 for Yes, 0 for No)';

comment on column MTPASA_DUIDAVAILABILITY.PASAUNITSTATE is
'The unit state value';

comment on column MTPASA_DUIDAVAILABILITY.PASARECALLTIME is
'The recall time value';

alter table MTPASA_DUIDAVAILABILITY
   add constraint MTPASA_DUIDAVAILABILITY_PK primary key (PUBLISH_DATETIME, DAY, REGIONID, DUID)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: MTPASA_INTERCONNECTORRESULT                           */
/*==============================================================*/
create table MTPASA_INTERCONNECTORRESULT (
   RUN_DATETIME         DATE                  not null,
   RUN_NO               NUMBER(4)             not null,
   RUNTYPE              VARCHAR2(20)          not null,
   DEMAND_POE_TYPE      VARCHAR2(20)          not null,
   DAY                  DATE                  not null,
   INTERCONNECTORID     VARCHAR2(20)          not null,
   PERIODID             NUMBER(3,0),
   FLOW90               NUMBER(12,2),
   FLOW50               NUMBER(12,2),
   FLOW10               NUMBER(12,2),
   PROBABILITYOFBINDINGEXPORT NUMBER(8,5),
   PROBABILITYOFBINDINGIMPORT NUMBER(8,5),
   CALCULATEDEXPORTLIMIT NUMBER(12,2),
   CALCULATEDIMPORTLIMIT NUMBER(12,2),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table MTPASA_INTERCONNECTORRESULT is
'Interconnector results for interval of max demand per day';

comment on column MTPASA_INTERCONNECTORRESULT.RUN_DATETIME is
'Date processing of the run begins.';

comment on column MTPASA_INTERCONNECTORRESULT.RUN_NO is
'Unique run id.';

comment on column MTPASA_INTERCONNECTORRESULT.RUNTYPE is
'Type of run. Always RELIABILITY';

comment on column MTPASA_INTERCONNECTORRESULT.DEMAND_POE_TYPE is
'Demand POE type used. Value is POE10';

comment on column MTPASA_INTERCONNECTORRESULT.DAY is
'Day this result is for';

comment on column MTPASA_INTERCONNECTORRESULT.INTERCONNECTORID is
'The unique identifier for the interconnector';

comment on column MTPASA_INTERCONNECTORRESULT.PERIODID is
'Half hourly period reported, selected as period of maximum NEM scheduled demand (calculated as maximum of scheduled demands, averaged across iterations and reference years)';

comment on column MTPASA_INTERCONNECTORRESULT.FLOW90 is
'The 90th percentile for flows, across iterations and reference years. Positive values indicate exporting, negative values indicate importing (MW)';

comment on column MTPASA_INTERCONNECTORRESULT.FLOW50 is
'The 50th percentile for flows, across iterations and reference years. Positive values indicate exporting, negative values indicate importing (MW)';

comment on column MTPASA_INTERCONNECTORRESULT.FLOW10 is
'The 10th percentile for flows, across iterations and reference years. Positive values indicate exporting, negative values indicate importing (MW)';

comment on column MTPASA_INTERCONNECTORRESULT.PROBABILITYOFBINDINGEXPORT is
'Proportion of iterations and reference years with interconnector constrained when exporting';

comment on column MTPASA_INTERCONNECTORRESULT.PROBABILITYOFBINDINGIMPORT is
'Proportion of iterations and reference years with interconnector constrained when importing';

comment on column MTPASA_INTERCONNECTORRESULT.CALCULATEDEXPORTLIMIT is
'Calculated Interconnector limit of exporting energy on the basis of invoked constraints and static interconnector export limit, averaged across iterations and reference years';

comment on column MTPASA_INTERCONNECTORRESULT.CALCULATEDIMPORTLIMIT is
'Calculated Interconnector limit of importing energy on the basis of invoked constraints and static interconnector import limit, averaged across iterations and reference years';

comment on column MTPASA_INTERCONNECTORRESULT.LASTCHANGED is
'Last date and time record changed';

alter table MTPASA_INTERCONNECTORRESULT
   add constraint MTPASA_INTERCONNECTORRESULT_PK primary key (RUN_DATETIME, RUN_NO, RUNTYPE, DEMAND_POE_TYPE, DAY, INTERCONNECTORID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: MTPASA_INTERCONNECTORSOLUTION                         */
/*==============================================================*/
create table MTPASA_INTERCONNECTORSOLUTION (
   RUN_DATETIME         DATE                  not null,
   RUN_NO               NUMBER(3,0)           not null,
   ENERGYBLOCK          DATE                  not null,
   DAY                  DATE                  not null,
   LDCBLOCK             NUMBER(3,0)           not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   CAPACITYMWFLOW       NUMBER(12,2),
   CAPACITYMARGINALVALUE NUMBER(12,2),
   CAPACITYVIOLATIONDEGREE NUMBER(12,2),
   CALCULATEDEXPORTLIMIT NUMBER(12,2),
   CALCULATEDIMPORTLIMIT NUMBER(12,2),
   LASTCHANGED          DATE,
   RUNTYPE              VARCHAR2(20)         default 'OUTAGE_LRC'  not null,
   EXPORTLIMITCONSTRAINTID VARCHAR2(20),
   IMPORTLIMITCONSTRAINTID VARCHAR2(20)
)
   tablespace PSSTAB02;

comment on table MTPASA_INTERCONNECTORSOLUTION is
'MTPASA_INTERCONNECTORSOLUTION is obsolete from 2017 End of Year DM4.27 Release. 
The MTPASA_INTERCONNECTORSOLUTION table shows the results of the capacity evaluation for Interconnectors, including the calculated limits for the ldcblock within the day.
Change Notice 379 announced the replacement of the MT PASA data model so all MTPASAxxx tables become obsolete, replaced by MTPASA_xxx tables (see Change Notices 400, 400a and 400b).';

comment on column MTPASA_INTERCONNECTORSOLUTION.RUN_DATETIME is
'Date processing of the run begins';

comment on column MTPASA_INTERCONNECTORSOLUTION.RUN_NO is
'Unique run id. Generated from the solution file caseid';

comment on column MTPASA_INTERCONNECTORSOLUTION.ENERGYBLOCK is
'Sunday at start of the week for this solutions energy block. Generated from the solution file energy block';

comment on column MTPASA_INTERCONNECTORSOLUTION.DAY is
'Day this solution is for. Generated from the solution file periodid';

comment on column MTPASA_INTERCONNECTORSOLUTION.LDCBLOCK is
'LDC block this solution is for. Generated from the solution file periodid';

comment on column MTPASA_INTERCONNECTORSOLUTION.INTERCONNECTORID is
'The unique identifier for the interconnector';

comment on column MTPASA_INTERCONNECTORSOLUTION.CAPACITYMWFLOW is
'Interconnector loading level (MW) that can be reached in case of capacity scarcity in neighbouring regions subject to network and energy constraints';

comment on column MTPASA_INTERCONNECTORSOLUTION.CAPACITYMARGINALVALUE is
'Capacity adequacy assessment marginal value; 0 if not binding';

comment on column MTPASA_INTERCONNECTORSOLUTION.CAPACITYVIOLATIONDEGREE is
'Capacity adequacy assessment violation degree, 0 if not violating; where CapacityMWFLow <= export + violation Degree (Deficit)CapacityMWFLow >= import + CapacityViolationDegree (Deficit)';

comment on column MTPASA_INTERCONNECTORSOLUTION.CALCULATEDEXPORTLIMIT is
'Calculated Interconnector limit of exporting energy on the basis of invoked constraints and static interconnector export limit';

comment on column MTPASA_INTERCONNECTORSOLUTION.CALCULATEDIMPORTLIMIT is
'Calculated Interconnector limit of importing energy on the basis of invoked constraints and static interconnector import limit. Note unlike the input interconnector import limit this is a directional quantity and should be defined with respect to the interconnector flow';

comment on column MTPASA_INTERCONNECTORSOLUTION.LASTCHANGED is
'Date the solution was loaded';

comment on column MTPASA_INTERCONNECTORSOLUTION.RUNTYPE is
'Type of run.  Values are RELIABILITY_LRC and OUTAGE_LRC';

comment on column MTPASA_INTERCONNECTORSOLUTION.EXPORTLIMITCONSTRAINTID is
'ID of the constraint that sets the Interconnector Export Limit';

comment on column MTPASA_INTERCONNECTORSOLUTION.IMPORTLIMITCONSTRAINTID is
'ID of the constraint that sets the Interconnector Import Limit';

alter table MTPASA_INTERCONNECTORSOLUTION
   add constraint MTPASA_INTERCONNECTORSOLN_PK primary key (RUN_DATETIME, RUN_NO, RUNTYPE, ENERGYBLOCK, DAY, LDCBLOCK, INTERCONNECTORID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: MTPASA_INTERCONNECTORSOLN_NDX2                        */
/*==============================================================*/
create index MTPASA_INTERCONNECTORSOLN_NDX2 on MTPASA_INTERCONNECTORSOLUTION (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: MTPASA_INTERMITTENT_AVAIL                             */
/*==============================================================*/
create table MTPASA_INTERMITTENT_AVAIL (
   TRADINGDATE          DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   OFFERDATETIME        DATE                  not null,
   CLUSTERID            VARCHAR2(20)          not null,
   LASTCHANGED          DATE,
   ELEMENTS_UNAVAILABLE NUMBER(5,0),
   ELEMENTS_AVAILABLE   NUMBER(5,0)
)
   tablespace PSSTAB02;

comment on table MTPASA_INTERMITTENT_AVAIL is
'A submission of expected plant availability for intermittent generators for use in MTPASA intermittent generation forecasts';

comment on column MTPASA_INTERMITTENT_AVAIL.TRADINGDATE is
'Trading Day for which this cluster availability submission applies';

comment on column MTPASA_INTERMITTENT_AVAIL.DUID is
'Unique Identifier of Dispatchable Unit';

comment on column MTPASA_INTERMITTENT_AVAIL.OFFERDATETIME is
'Date and Time when this cluster availability submission was loaded';

comment on column MTPASA_INTERMITTENT_AVAIL.CLUSTERID is
'Unique Cluster Identifier for this cluster within the DUID';

comment on column MTPASA_INTERMITTENT_AVAIL.LASTCHANGED is
'Last date and time record changed';

comment on column MTPASA_INTERMITTENT_AVAIL.ELEMENTS_UNAVAILABLE is
'Number of elements within this CLUSTERID (turbines for wind, or inverters for solar) that are not available for this TRADINGDATE. Value between 0 and the registered Number of Cluster Elements.Value = 0 means no elements unavailable';

comment on column MTPASA_INTERMITTENT_AVAIL.ELEMENTS_AVAILABLE is
'Number of elements within this CLUSTERID (turbines for wind, or inverters for solar) that are available for this TRADINGDATE. Value between 0 and the registered Number of Cluster Elements. Value = 0 means no elements available';

alter table MTPASA_INTERMITTENT_AVAIL
   add constraint MTPASA_INTERMITTENT_AVAIL_PK primary key (TRADINGDATE, DUID, OFFERDATETIME, CLUSTERID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: MTPASA_INTERMITTENT_LIMIT                             */
/*==============================================================*/
create table MTPASA_INTERMITTENT_LIMIT (
   TRADINGDATE          DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   OFFERDATETIME        DATE                  not null,
   LASTCHANGED          DATE,
   UPPERMWLIMIT         NUMBER(6),
   AUTHORISEDBYUSER     VARCHAR2(20),
   AUTHORISEDBYPARTICIPANTID VARCHAR2(20)
)
   tablespace PSSTAB02;

comment on table MTPASA_INTERMITTENT_LIMIT is
'A submission of expected maximum availability for intermittent generators for use in MTPASA intermittent generation
forecasts';

comment on column MTPASA_INTERMITTENT_LIMIT.TRADINGDATE is
'Trading Day for which this unit availability submission applies';

comment on column MTPASA_INTERMITTENT_LIMIT.DUID is
'Unique Identifier of Dispatchable Unit';

comment on column MTPASA_INTERMITTENT_LIMIT.OFFERDATETIME is
'Date time file processed';

comment on column MTPASA_INTERMITTENT_LIMIT.LASTCHANGED is
'Last date and time record changed';

comment on column MTPASA_INTERMITTENT_LIMIT.UPPERMWLIMIT is
'Maximum imposed MW limit. Value between 0 and the registered DUID Maximum Capacity.Value = -1 means no limit applies.';

comment on column MTPASA_INTERMITTENT_LIMIT.AUTHORISEDBYUSER is
'User entering the unit availability submission';

comment on column MTPASA_INTERMITTENT_LIMIT.AUTHORISEDBYPARTICIPANTID is
'Participant entering the unit availability submission';

alter table MTPASA_INTERMITTENT_LIMIT
   add constraint MTPASA_INTERMITTENT_LIMIT_PK primary key (TRADINGDATE, DUID, OFFERDATETIME)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: MTPASA_LOLPRESULT                                     */
/*==============================================================*/
create table MTPASA_LOLPRESULT (
   RUN_DATETIME         DATE                  not null,
   RUN_NO               NUMBER(4)             not null,
   RUNTYPE              VARCHAR2(20)          not null,
   DAY                  DATE                  not null,
   REGIONID             VARCHAR2(20)          not null,
   WORST_INTERVAL_PERIODID NUMBER(3,0),
   WORST_INTERVAL_DEMAND NUMBER(12,2),
   WORST_INTERVAL_INTGEN NUMBER(12,2),
   WORST_INTERVAL_DSP   NUMBER(12,2),
   LOSSOFLOADPROBABILITY NUMBER(8,5),
   LOSSOFLOADMAGNITUDE  VARCHAR2(20),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table MTPASA_LOLPRESULT is
'Results for Loss of Load Probability (LOLP) run per day';

comment on column MTPASA_LOLPRESULT.RUN_DATETIME is
'Date processing of the run begins.';

comment on column MTPASA_LOLPRESULT.RUN_NO is
'Unique run id.';

comment on column MTPASA_LOLPRESULT.RUNTYPE is
'Type of run. Always LOLP';

comment on column MTPASA_LOLPRESULT.DAY is
'Day this result is for';

comment on column MTPASA_LOLPRESULT.REGIONID is
'The unique region identifier';

comment on column MTPASA_LOLPRESULT.WORST_INTERVAL_PERIODID is
'The half hourly interval period with the highest LOLP, or highest region demand if LOLP = 0 for all intervals (1..48)';

comment on column MTPASA_LOLPRESULT.WORST_INTERVAL_DEMAND is
'The Abstract Operational Demand for the worst interval in this region (MW)';

comment on column MTPASA_LOLPRESULT.WORST_INTERVAL_INTGEN is
'The half hourly aggregate intermittent generation for the worst interval in this region (MW)';

comment on column MTPASA_LOLPRESULT.WORST_INTERVAL_DSP is
'The half hourly aggregate demand side participation for the worst interval period in this region (MW)';

comment on column MTPASA_LOLPRESULT.LOSSOFLOADPROBABILITY is
'Loss of Load Probability for the worst interval in this region';

comment on column MTPASA_LOLPRESULT.LOSSOFLOADMAGNITUDE is
'Loss of Load Magnitude for the worst interval in this region. Values are LOW, MEDIUM, HIGH';

comment on column MTPASA_LOLPRESULT.LASTCHANGED is
'Last date and time record changed';

alter table MTPASA_LOLPRESULT
   add constraint MTPASA_LOLPRESULT_PK primary key (RUN_DATETIME, RUN_NO, RUNTYPE, DAY, REGIONID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: MTPASA_OFFERDATA                                      */
/*==============================================================*/
create table MTPASA_OFFERDATA (
   PARTICIPANTID        VARCHAR2(20)          not null,
   OFFERDATETIME        DATE                  not null,
   UNITID               VARCHAR2(20)          not null,
   EFFECTIVEDATE        DATE                  not null,
   ENERGY               NUMBER(9),
   CAPACITY1            NUMBER(9),
   CAPACITY2            NUMBER(9),
   CAPACITY3            NUMBER(9),
   CAPACITY4            NUMBER(9),
   CAPACITY5            NUMBER(9),
   CAPACITY6            NUMBER(9),
   CAPACITY7            NUMBER(9),
   LASTCHANGED          DATE,
   UNITSTATE1           VARCHAR2(20),
   UNITSTATE2           VARCHAR2(20),
   UNITSTATE3           VARCHAR2(20),
   UNITSTATE4           VARCHAR2(20),
   UNITSTATE5           VARCHAR2(20),
   UNITSTATE6           VARCHAR2(20),
   UNITSTATE7           VARCHAR2(20),
   RECALLTIME1          NUMBER(4),
   RECALLTIME2          NUMBER(4),
   RECALLTIME3          NUMBER(4),
   RECALLTIME4          NUMBER(4),
   RECALLTIME5          NUMBER(4),
   RECALLTIME6          NUMBER(4),
   RECALLTIME7          NUMBER(4)
)
   tablespace PSSTAB03;

comment on table MTPASA_OFFERDATA is
'Participant submitted Offers for MTPASA process';

comment on column MTPASA_OFFERDATA.PARTICIPANTID is
'Unique participant identifier';

comment on column MTPASA_OFFERDATA.OFFERDATETIME is
'Date time file processed';

comment on column MTPASA_OFFERDATA.UNITID is
'either duid or mnsp linkid';

comment on column MTPASA_OFFERDATA.EFFECTIVEDATE is
'trade date when the offer becomes effective';

comment on column MTPASA_OFFERDATA.ENERGY is
'weekly energy constraint value';

comment on column MTPASA_OFFERDATA.CAPACITY1 is
'capacity value day 1 (sunday)';

comment on column MTPASA_OFFERDATA.CAPACITY2 is
'capacity value day 2 (monday)';

comment on column MTPASA_OFFERDATA.CAPACITY3 is
'capacity value day 3 (tuesday)';

comment on column MTPASA_OFFERDATA.CAPACITY4 is
'capacity value day 4 (wednesday)';

comment on column MTPASA_OFFERDATA.CAPACITY5 is
'capacity value day 5 (thursday)';

comment on column MTPASA_OFFERDATA.CAPACITY6 is
'capacity value day 6 (friday)';

comment on column MTPASA_OFFERDATA.CAPACITY7 is
'capacity value day 7 (saturday)';

comment on column MTPASA_OFFERDATA.LASTCHANGED is
'timestamp when record last changed';

comment on column MTPASA_OFFERDATA.UNITSTATE1 is
'The unit state value for day 1 Sunday';

comment on column MTPASA_OFFERDATA.UNITSTATE2 is
'The unit state value for day 2 Monday';

comment on column MTPASA_OFFERDATA.UNITSTATE3 is
'The unit state value for day 3 Tuesday';

comment on column MTPASA_OFFERDATA.UNITSTATE4 is
'The unit state value for 4 Wednesday';

comment on column MTPASA_OFFERDATA.UNITSTATE5 is
'The unit state value for day 5 Thursday';

comment on column MTPASA_OFFERDATA.UNITSTATE6 is
'The unit state value for day 6 Friday';

comment on column MTPASA_OFFERDATA.UNITSTATE7 is
'The unit state value for day 7 Saturday';

comment on column MTPASA_OFFERDATA.RECALLTIME1 is
'The recall time associated with the unit state for day 1 Sunday';

comment on column MTPASA_OFFERDATA.RECALLTIME2 is
'The recall time associated with the unit state for day 2 Monday';

comment on column MTPASA_OFFERDATA.RECALLTIME3 is
'The recall time associated with the unit state for day 3 Tuesday';

comment on column MTPASA_OFFERDATA.RECALLTIME4 is
'The recall time associated with the unit state for day 4 Wednesday';

comment on column MTPASA_OFFERDATA.RECALLTIME5 is
'The recall time associated with the unit state for day 5 Thursday';

comment on column MTPASA_OFFERDATA.RECALLTIME6 is
'The recall time associated with the unit state for day 6 Friday';

comment on column MTPASA_OFFERDATA.RECALLTIME7 is
'The recall time associated with the unit state for day 7 Saturday';

alter table MTPASA_OFFERDATA
   add constraint MTPASA_OFFERDATA_PK primary key (PARTICIPANTID, OFFERDATETIME, UNITID, EFFECTIVEDATE)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Index: MTPASA_OFFERDATA_LCX                                  */
/*==============================================================*/
create index MTPASA_OFFERDATA_LCX on MTPASA_OFFERDATA (
   LASTCHANGED ASC
)
tablespace PSSIND03;

/*==============================================================*/
/* Table: MTPASA_OFFERFILETRK                                   */
/*==============================================================*/
create table MTPASA_OFFERFILETRK (
   PARTICIPANTID        VARCHAR2(20)          not null,
   OFFERDATETIME        DATE                  not null,
   FILENAME             VARCHAR2(200)
)
   tablespace PSSTAB01;

comment on table MTPASA_OFFERFILETRK is
'Participant submitted Offers for MTPASA process';

comment on column MTPASA_OFFERFILETRK.PARTICIPANTID is
'Unique participant identifier';

comment on column MTPASA_OFFERFILETRK.OFFERDATETIME is
'Date time file processed';

comment on column MTPASA_OFFERFILETRK.FILENAME is
'Submitted file name';

alter table MTPASA_OFFERFILETRK
   add constraint MTPASA_OFFERFILETRK_PK primary key (PARTICIPANTID, OFFERDATETIME)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: MTPASA_REGIONAVAILABILITY                             */
/*==============================================================*/
create table MTPASA_REGIONAVAILABILITY (
   PUBLISH_DATETIME     DATE                  not null,
   DAY                  DATE                  not null,
   REGIONID             VARCHAR2(20)          not null,
   PASAAVAILABILITY_SCHEDULED NUMBER(12,0),
   LATEST_OFFER_DATETIME DATE,
   ENERGYUNCONSTRAINEDCAPACITY NUMBER(12,0),
   ENERGYCONSTRAINEDCAPACITY NUMBER(12,0),
   NONSCHEDULEDGENERATION NUMBER(12,2),
   DEMAND10             NUMBER(12,2),
   DEMAND50             NUMBER(12,2),
   ENERGYREQDEMAND10    NUMBER(12,2),
   ENERGYREQDEMAND50    NUMBER(12,2),
   LASTCHANGED          DATE,
   DEMAND10MIN          NUMBER(12,2),
   DEMAND10MAX          NUMBER(12,2),
   DEMAND50MIN          NUMBER(12,2),
   DEMAND50MAX          NUMBER(12,2),
   CARRYOVERCAPACITY    NUMBER(12,0)
)
   tablespace PSSTAB03;

comment on table MTPASA_REGIONAVAILABILITY is
'Stores the Region-aggregate offered PASA Availability of scheduled generators for each day over the Medium Term PASA period. The data in this table is an aggregate of input data to the MT PASA process it is not part of the MTPASA solution. The aggregate availability does not reflect any energy limitations in the MT PASA offers.';

comment on column MTPASA_REGIONAVAILABILITY.PUBLISH_DATETIME is
'Date Time the report was published.';

comment on column MTPASA_REGIONAVAILABILITY.DAY is
'Date on which the aggregation applies.';

comment on column MTPASA_REGIONAVAILABILITY.REGIONID is
'NEM Region.';

comment on column MTPASA_REGIONAVAILABILITY.PASAAVAILABILITY_SCHEDULED is
'Aggregate of the offered PASA Availability for all Scheduled generators in this region.';

comment on column MTPASA_REGIONAVAILABILITY.LATEST_OFFER_DATETIME is
'Date Time of the latest offer used in the aggregation for this region and date.';

comment on column MTPASA_REGIONAVAILABILITY.ENERGYUNCONSTRAINEDCAPACITY is
'Region energy unconstrained MW capacity';

comment on column MTPASA_REGIONAVAILABILITY.ENERGYCONSTRAINEDCAPACITY is
'Region energy constrained MW capacity';

comment on column MTPASA_REGIONAVAILABILITY.NONSCHEDULEDGENERATION is
'Allowance made for non-scheduled generation in the demand forecast (MW)';

comment on column MTPASA_REGIONAVAILABILITY.DEMAND10 is
'10% probability demand (ex non-scheduled demand)';

comment on column MTPASA_REGIONAVAILABILITY.DEMAND50 is
'50% probability demand (ex non-scheduled demand)';

comment on column MTPASA_REGIONAVAILABILITY.ENERGYREQDEMAND10 is
'Total weekly operational as generated consumption (POE 10)';

comment on column MTPASA_REGIONAVAILABILITY.ENERGYREQDEMAND50 is
'Total weekly operational as generated consumption (POE 50)';

comment on column MTPASA_REGIONAVAILABILITY.LASTCHANGED is
'Last date and time record changed';

comment on column MTPASA_REGIONAVAILABILITY.DEMAND10MIN is
'Minimum of the Operational Load as Generated (OPGEN) peaks that occur in all ref years for the P10 traces (MW).';

comment on column MTPASA_REGIONAVAILABILITY.DEMAND10MAX is
'Maximum of the Operational Load as Generated (OPGEN) peaks that occur in all ref years for the P10 traces (MW).';

comment on column MTPASA_REGIONAVAILABILITY.DEMAND50MIN is
'Minimum of the Operational Load as Generated (OPGEN) peaks that occur in all ref years for the P50 traces (MW).';

comment on column MTPASA_REGIONAVAILABILITY.DEMAND50MAX is
'Maximum of the Operational Load as Generated (OPGEN) peaks that occur in all ref years for the P50 traces (MW).';

comment on column MTPASA_REGIONAVAILABILITY.CARRYOVERCAPACITY is
'Split of the CARRYOVER component of aggregate capacity vs the currently reported capacity.';

alter table MTPASA_REGIONAVAILABILITY
   add constraint MTPASA_REGIONAVAILABILITY_PK primary key (PUBLISH_DATETIME, DAY, REGIONID)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: MTPASA_REGIONAVAIL_TRK                                */
/*==============================================================*/
create table MTPASA_REGIONAVAIL_TRK (
   PUBLISH_DATETIME     DATE                  not null,
   STARTDATE            DATE,
   ENDDATE              DATE,
   LATEST_OFFER_DATETIME DATE
)
   tablespace PSSTAB01;

comment on table MTPASA_REGIONAVAIL_TRK is
'The tracking table to assist in versioning of the region-aggregate offered PASA Availability data published to the MTPASA_REGIONAVAILABILITY table.';

comment on column MTPASA_REGIONAVAIL_TRK.PUBLISH_DATETIME is
'Date Time the report was published.';

comment on column MTPASA_REGIONAVAIL_TRK.STARTDATE is
'First date of the report inclusive.';

comment on column MTPASA_REGIONAVAIL_TRK.ENDDATE is
'Last date of the report inclusive.';

comment on column MTPASA_REGIONAVAIL_TRK.LATEST_OFFER_DATETIME is
'Date Time of the latest offer used in the report.';

alter table MTPASA_REGIONAVAIL_TRK
   add constraint MTPASA_REGIONAVAIL_TRK_PK primary key (PUBLISH_DATETIME)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: MTPASA_REGIONITERATION                                */
/*==============================================================*/
create table MTPASA_REGIONITERATION (
   RUN_DATETIME         DATE                  not null,
   RUN_NO               NUMBER(4)             not null,
   RUNTYPE              VARCHAR2(20)          not null,
   DEMAND_POE_TYPE      VARCHAR2(20)          not null,
   AGGREGATION_PERIOD   VARCHAR2(20)          not null,
   PERIOD_ENDING        DATE                  not null,
   REGIONID             VARCHAR2(20)          not null,
   USE_ITERATION_ID     NUMBER(5)             not null,
   USE_ITERATION_EVENT_NUMBER NUMBER(12,2),
   USE_ITERATION_EVENT_AVERAGE NUMBER(12,2),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table MTPASA_REGIONITERATION is
'Region results for Unserved Energy (USE)';

comment on column MTPASA_REGIONITERATION.RUN_DATETIME is
'Date processing of the run begins.';

comment on column MTPASA_REGIONITERATION.RUN_NO is
'Unique run id.';

comment on column MTPASA_REGIONITERATION.RUNTYPE is
'Type of run. Always RELIABILITY';

comment on column MTPASA_REGIONITERATION.DEMAND_POE_TYPE is
'Demand POE type used. Value is POE10 or POE50';

comment on column MTPASA_REGIONITERATION.AGGREGATION_PERIOD is
'Period data is aggregated over. Values are YEAR';

comment on column MTPASA_REGIONITERATION.PERIOD_ENDING is
'Datetime of day at end of period (i.e. last day of year reported)';

comment on column MTPASA_REGIONITERATION.REGIONID is
'The unique region identifier';

comment on column MTPASA_REGIONITERATION.USE_ITERATION_ID is
'Iteration ID, only produced for iterations showing unserved energy>0';

comment on column MTPASA_REGIONITERATION.USE_ITERATION_EVENT_NUMBER is
'Number of half hours showing unserved energy over year, for iteration';

comment on column MTPASA_REGIONITERATION.USE_ITERATION_EVENT_AVERAGE is
'Average unserved energy event size for iteration over year (MW)';

comment on column MTPASA_REGIONITERATION.LASTCHANGED is
'Last date and time record changed';

alter table MTPASA_REGIONITERATION
   add constraint MTPASA_REGIONITERATION_PK primary key (RUN_DATETIME, RUN_NO, RUNTYPE, DEMAND_POE_TYPE, AGGREGATION_PERIOD, PERIOD_ENDING, REGIONID, USE_ITERATION_ID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: MTPASA_REGIONRESULT                                   */
/*==============================================================*/
create table MTPASA_REGIONRESULT (
   RUN_DATETIME         DATE                  not null,
   RUN_NO               NUMBER(4)             not null,
   RUNTYPE              VARCHAR2(20)          not null,
   DEMAND_POE_TYPE      VARCHAR2(20)          not null,
   DAY                  DATE                  not null,
   REGIONID             VARCHAR2(20)          not null,
   PERIODID             NUMBER(3,0),
   DEMAND               NUMBER(12,2),
   AGGREGATEINSTALLEDCAPACITY NUMBER(12,2),
   NUMBEROFITERATIONS   NUMBER(12,2),
   USE_NUMBEROFITERATIONS NUMBER(12,2),
   USE_MAX              NUMBER(12,2),
   USE_UPPERQUARTILE    NUMBER(12,2),
   USE_MEDIAN           NUMBER(12,2),
   USE_LOWERQUARTILE    NUMBER(12,2),
   USE_MIN              NUMBER(12,2),
   USE_AVERAGE          NUMBER(12,2),
   USE_EVENT_AVERAGE    NUMBER(12,2),
   TOTALSCHEDULEDGEN90  NUMBER(12,2),
   TOTALSCHEDULEDGEN50  NUMBER(12,2),
   TOTALSCHEDULEDGEN10  NUMBER(12,2),
   TOTALINTERMITTENTGEN90 NUMBER(12,2),
   TOTALINTERMITTENTGEN50 NUMBER(12,2),
   TOTALINTERMITTENTGEN10 NUMBER(12,2),
   DEMANDSIDEPARTICIPATION90 NUMBER(12,2),
   DEMANDSIDEPARTICIPATION50 NUMBER(12,2),
   DEMANDSIDEPARTICIPATION10 NUMBER(12,2),
   LASTCHANGED          DATE,
   TOTALSEMISCHEDULEGEN90 NUMBER(12,2),
   TOTALSEMISCHEDULEGEN50 NUMBER(12,2),
   TOTALSEMISCHEDULEGEN10 NUMBER(12,2),
   TOTALAVAILABLEGENMIN NUMBER(12,2),
   TOTALAVAILABLEGEN10  NUMBER(12,2),
   TOTALAVAILABLEGEN50  NUMBER(12,2),
   TOTALAVAILABLEGEN90  NUMBER(12,2),
   TOTALAVAILABLEGENMAX NUMBER(12,2)
)
   tablespace PSSTAB02;

comment on table MTPASA_REGIONRESULT is
'Region results for interval of max demand per day.';

comment on column MTPASA_REGIONRESULT.RUN_DATETIME is
'Date processing of the run begins.';

comment on column MTPASA_REGIONRESULT.RUN_NO is
'Unique run id.';

comment on column MTPASA_REGIONRESULT.RUNTYPE is
'Type of run. Always RELIABILITY';

comment on column MTPASA_REGIONRESULT.DEMAND_POE_TYPE is
'Demand POE type used. Value is POE10';

comment on column MTPASA_REGIONRESULT.DAY is
'Day this result is for';

comment on column MTPASA_REGIONRESULT.REGIONID is
'The unique region identifier';

comment on column MTPASA_REGIONRESULT.PERIODID is
'Half hourly period reported, selected as period of maximum NEM scheduled demand (calculated as maximum of scheduled demands, averaged across iterations and reference years)';

comment on column MTPASA_REGIONRESULT.DEMAND is
'Demand value from selected half hourly interval (MW)';

comment on column MTPASA_REGIONRESULT.AGGREGATEINSTALLEDCAPACITY is
'The total installed capacity of all generation (MW)';

comment on column MTPASA_REGIONRESULT.NUMBEROFITERATIONS is
'Total number of iterations and reference years performed';

comment on column MTPASA_REGIONRESULT.USE_NUMBEROFITERATIONS is
'Number of iterations and reference years with unserved energy>0';

comment on column MTPASA_REGIONRESULT.USE_MAX is
'Maximum unserved energy, across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.USE_UPPERQUARTILE is
'Upper quartile unserved energy, across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.USE_MEDIAN is
'Median unserved energy, across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.USE_LOWERQUARTILE is
'Lower quartile unserved energy, across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.USE_MIN is
'Minimum unserved energy, across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.USE_AVERAGE is
'Average unserved energy, across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.USE_EVENT_AVERAGE is
'Average unserved energy event size, across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.TOTALSCHEDULEDGEN90 is
'The 90th percentile for scheduled generation across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.TOTALSCHEDULEDGEN50 is
'The 50th percentile for scheduled generation across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.TOTALSCHEDULEDGEN10 is
'The 10th percentile for scheduled generation across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.TOTALINTERMITTENTGEN90 is
'The 90th percentile for intermittent generation, across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.TOTALINTERMITTENTGEN50 is
'The 50th percentile for intermittent generation, across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.TOTALINTERMITTENTGEN10 is
'The 10th percentile for intermittent generation, across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.DEMANDSIDEPARTICIPATION90 is
'The 90th percentile for demand side participation, across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.DEMANDSIDEPARTICIPATION50 is
'The 50th percentile for demand side participation, across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.DEMANDSIDEPARTICIPATION10 is
'The 10th percentile for demand side participation, across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.LASTCHANGED is
'Last date and time record changed';

comment on column MTPASA_REGIONRESULT.TOTALSEMISCHEDULEGEN90 is
'The 90% percentile for semi-scheduled generation across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.TOTALSEMISCHEDULEGEN50 is
'The 50% percentile for semi-scheduled generation across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.TOTALSEMISCHEDULEGEN10 is
'The 10% percentile for semi-scheduled generation across iterations and reference years (MW)';

comment on column MTPASA_REGIONRESULT.TOTALAVAILABLEGENMIN is
'Minimum available capacity, across iterations and reference years (MW).';

comment on column MTPASA_REGIONRESULT.TOTALAVAILABLEGEN10 is
'The 10% percentile for available capacity, across iterations and reference years (MW).';

comment on column MTPASA_REGIONRESULT.TOTALAVAILABLEGEN50 is
'The 50% percentile for available capacity, across iterations and reference years (MW).';

comment on column MTPASA_REGIONRESULT.TOTALAVAILABLEGEN90 is
'The 90% percentile for available capacity, across iterations and reference years (MW).';

comment on column MTPASA_REGIONRESULT.TOTALAVAILABLEGENMAX is
'Maximum available capacity, across iterations and reference years (MW).';

alter table MTPASA_REGIONRESULT
   add constraint MTPASA_REGIONRESULT_PK primary key (RUN_DATETIME, RUN_NO, RUNTYPE, DEMAND_POE_TYPE, DAY, REGIONID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: MTPASA_REGIONSOLUTION                                 */
/*==============================================================*/
create table MTPASA_REGIONSOLUTION (
   RUN_DATETIME         DATE                  not null,
   RUN_NO               NUMBER(3,0)           not null,
   ENERGYBLOCK          DATE                  not null,
   DAY                  DATE                  not null,
   LDCBLOCK             NUMBER(3,0)           not null,
   REGIONID             VARCHAR2(10)          not null,
   DEMAND10             NUMBER(12,2),
   RESERVEREQ           NUMBER(12,2),
   CAPACITYREQ          NUMBER(12,2),
   ENERGYREQDEMAND10    NUMBER(12,2),
   UNCONSTRAINEDCAPACITY NUMBER(12,0),
   CONSTRAINEDCAPACITY  NUMBER(12,0),
   NETINTERCHANGEUNDERSCARCITY NUMBER(12,2),
   SURPLUSCAPACITY      NUMBER(12,2),
   SURPLUSRESERVE       NUMBER(12,2),
   RESERVECONDITION     NUMBER(1,0),
   MAXSURPLUSRESERVE    NUMBER(12,2),
   MAXSPARECAPACITY     NUMBER(12,2),
   LORCONDITION         NUMBER(1,0),
   AGGREGATECAPACITYAVAILABLE NUMBER(12,2),
   AGGREGATESCHEDULEDLOAD NUMBER(12,2),
   LASTCHANGED          DATE,
   AGGREGATEPASAAVAILABILITY NUMBER(12,0),
   RUNTYPE              VARCHAR2(20)         default 'OUTAGE_LRC'  not null,
   CALCULATEDLOR1LEVEL  NUMBER(16,6),
   CALCULATEDLOR2LEVEL  NUMBER(16,6),
   MSRNETINTERCHANGEUNDERSCARCITY NUMBER(12,2),
   LORNETINTERCHANGEUNDERSCARCITY NUMBER(12,2),
   TOTALINTERMITTENTGENERATION NUMBER(15,5),
   DEMAND50             NUMBER(12,2),
   DEMAND_AND_NONSCHEDGEN NUMBER(15,5),
   UIGF                 NUMBER(12,2),
   SEMISCHEDULEDCAPACITY NUMBER(12,2),
   LOR_SEMISCHEDULEDCAPACITY NUMBER(12,2),
   DEFICITRESERVE       NUMBER(16,6),
   MAXUSEFULRESPONSE    NUMBER(12,2),
   MURNETINTERCHANGEUNDERSCARCITY NUMBER(12,2),
   LORTOTALINTERMITTENTGENERATION NUMBER(15,5),
   ENERGYREQDEMAND50    number(12,2)
)
   tablespace PSSTAB02;

comment on table MTPASA_REGIONSOLUTION is
'MTPASA_CASESOLUTION is obsolete from 2017 End of Year DM4.27 Release. 
The MTPASA_REGIONSOLUTION table shows the results of the regional capacity, maximum surplus reserve and maximum spare capacity evaluations for each day and ldcblock of the study.';

comment on column MTPASA_REGIONSOLUTION.RUN_DATETIME is
'Date processing of the run begins';

comment on column MTPASA_REGIONSOLUTION.RUN_NO is
'Unique run id. Generated from the solution file caseid';

comment on column MTPASA_REGIONSOLUTION.ENERGYBLOCK is
'Sunday at start of the week for this solutions energy block. Generated from the solution file energy block';

comment on column MTPASA_REGIONSOLUTION.DAY is
'Day this solution is for. Generated from the solution file periodid';

comment on column MTPASA_REGIONSOLUTION.LDCBLOCK is
'LDC block this solution is for. Generated from the solution file periodid';

comment on column MTPASA_REGIONSOLUTION.REGIONID is
'The unique region identifier';

comment on column MTPASA_REGIONSOLUTION.DEMAND10 is
'Input value for 10% probability demand';

comment on column MTPASA_REGIONSOLUTION.RESERVEREQ is
'Not used from 21/05/2010. Prior to 21/05/2010: Input reserve requirement';

comment on column MTPASA_REGIONSOLUTION.CAPACITYREQ is
'Not used from 21/05/2010. Prior to 21/05/2010: CA Demand + Reserve Requirement';

comment on column MTPASA_REGIONSOLUTION.ENERGYREQDEMAND10 is
'Sum of: (Region Period Demand - given Demand10)/PeriodLength(sum by Energy Block, entered in first period of energy block, GWh)';

comment on column MTPASA_REGIONSOLUTION.UNCONSTRAINEDCAPACITY is
'Region energy unconstrained MW capacity subject to network security constraints';

comment on column MTPASA_REGIONSOLUTION.CONSTRAINEDCAPACITY is
'Region energy constrained MW capacity subject to energy and network security constraints';

comment on column MTPASA_REGIONSOLUTION.NETINTERCHANGEUNDERSCARCITY is
'Calculated in capacity adequacy evaluation: Export if > 0, Import if < 0.';

comment on column MTPASA_REGIONSOLUTION.SURPLUSCAPACITY is
'Regional surplus capacity MW, +/- values indicate surplus/deficit capacity';

comment on column MTPASA_REGIONSOLUTION.SURPLUSRESERVE is
'Not used from 21/05/2010. Prior to 21/05/2010: Regional reserve surplus. +/-0 values indicate surplus/deficit reserve';

comment on column MTPASA_REGIONSOLUTION.RESERVECONDITION is
'The regional reserve condition: 0 = Adequate, 1 = LRC';

comment on column MTPASA_REGIONSOLUTION.MAXSURPLUSRESERVE is
'The Maximum generation (MW) that could be withdrawn from this region without incurring a Low Reserve Condition.';

comment on column MTPASA_REGIONSOLUTION.MAXSPARECAPACITY is
'The Maximum Spare Capacity evaluated for this region in this period. Calculated for each region in turn';

comment on column MTPASA_REGIONSOLUTION.LORCONDITION is
'The LOR Condition determined from the Maximum Spare Capacity value: 0 = no condition, 1 = LOR1 condition, 2 = LOR2 condition, 3 = LOR3 condition';

comment on column MTPASA_REGIONSOLUTION.AGGREGATECAPACITYAVAILABLE is
'Sum of  MAXAVAIL quantities offered by all Scheduled Generators in a given Region for a given PERIODID.';

comment on column MTPASA_REGIONSOLUTION.AGGREGATESCHEDULEDLOAD is
'Sum of  MAXAVAIL quantities bid by of all Scheduled Loads in a given Region for a given PERIODID.';

comment on column MTPASA_REGIONSOLUTION.LASTCHANGED is
'Date the solution was loaded';

comment on column MTPASA_REGIONSOLUTION.AGGREGATEPASAAVAILABILITY is
'Sum of  PASAAVAILABILITY quantities offered by all Scheduled Generators in a given Region for a given PERIODID.';

comment on column MTPASA_REGIONSOLUTION.RUNTYPE is
'Type of run.  Values are RELIABILITY_LRC and OUTAGE_LRC';

comment on column MTPASA_REGIONSOLUTION.CALCULATEDLOR1LEVEL is
'Region Reserve Level for LOR1 used. Can be static value or calculated value if an interconnector is a credible contingency';

comment on column MTPASA_REGIONSOLUTION.CALCULATEDLOR2LEVEL is
'Region Reserve Level for LOR2 used. Can be static value or calculated value if an interconnector is a credible contingency';

comment on column MTPASA_REGIONSOLUTION.MSRNETINTERCHANGEUNDERSCARCITY is
'Net interconnector flow from the region for this interval from the MSR assessment';

comment on column MTPASA_REGIONSOLUTION.LORNETINTERCHANGEUNDERSCARCITY is
'Net interconnector flow from the region for this interval from the LOR assessment';

comment on column MTPASA_REGIONSOLUTION.TOTALINTERMITTENTGENERATION is
'Allowance made for non-scheduled generation in the demand forecast (MW).';

comment on column MTPASA_REGIONSOLUTION.DEMAND50 is
'Input value for 50% probability demand';

comment on column MTPASA_REGIONSOLUTION.DEMAND_AND_NONSCHEDGEN is
'Sum of Cleared Scheduled generation, imported generation (at the region boundary) and allowances made for non-scheduled generation (MW).';

comment on column MTPASA_REGIONSOLUTION.UIGF is
'Regional aggregated Unconstrained Intermittent Generation Forecast of Semi-scheduled generation (MW).';

comment on column MTPASA_REGIONSOLUTION.SEMISCHEDULEDCAPACITY is
'Aggregate Regional UIGF availability';

comment on column MTPASA_REGIONSOLUTION.LOR_SEMISCHEDULEDCAPACITY is
'Aggregate Regional UIGF availability for LOR';

comment on column MTPASA_REGIONSOLUTION.DEFICITRESERVE is
'Regional reserve deficit (MW)';

comment on column MTPASA_REGIONSOLUTION.MAXUSEFULRESPONSE is
'The Maximum market response (MW) needed for the region to eliminate a Low Reserve Condition (LRC)';

comment on column MTPASA_REGIONSOLUTION.MURNETINTERCHANGEUNDERSCARCITY is
'Net interconnector flow from the region for this interval from the MRR assessment';

comment on column MTPASA_REGIONSOLUTION.LORTOTALINTERMITTENTGENERATION is
'Allowance made for non-scheduled generation in the LOR assessment';

comment on column MTPASA_REGIONSOLUTION.ENERGYREQDEMAND50 is
'Sum of: (Region Period Demand - given Demand50)/PeriodLength (sum by Energy Block, entered in first period of energy block, GWh)';

alter table MTPASA_REGIONSOLUTION
   add constraint MTPASA_REGIONSOLUTION_PK primary key (RUN_DATETIME, RUN_NO, RUNTYPE, ENERGYBLOCK, DAY, LDCBLOCK, REGIONID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: MTPASA_REGIONSOLUTION_NDX2                            */
/*==============================================================*/
create index MTPASA_REGIONSOLUTION_NDX2 on MTPASA_REGIONSOLUTION (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: MTPASA_REGIONSUMMARY                                  */
/*==============================================================*/
create table MTPASA_REGIONSUMMARY (
   RUN_DATETIME         DATE                  not null,
   RUN_NO               NUMBER(4)             not null,
   RUNTYPE              VARCHAR2(20)          not null,
   DEMAND_POE_TYPE      VARCHAR2(20)          not null,
   AGGREGATION_PERIOD   VARCHAR2(20)          not null,
   PERIOD_ENDING        DATE                  not null,
   REGIONID             VARCHAR2(20)          not null,
   NATIVEDEMAND         NUMBER(12,2),
   USE_PERCENTILE10     NUMBER(12,2),
   USE_PERCENTILE20     NUMBER(12,2),
   USE_PERCENTILE30     NUMBER(12,2),
   USE_PERCENTILE40     NUMBER(12,2),
   USE_PERCENTILE50     NUMBER(12,2),
   USE_PERCENTILE60     NUMBER(12,2),
   USE_PERCENTILE70     NUMBER(12,2),
   USE_PERCENTILE80     NUMBER(12,2),
   USE_PERCENTILE90     NUMBER(12,2),
   USE_PERCENTILE100    NUMBER(12,2),
   USE_AVERAGE          NUMBER(12,2),
   NUMBEROFITERATIONS   NUMBER(12,2),
   USE_NUMBEROFITERATIONS NUMBER(12,2),
   USE_EVENT_MAX        NUMBER(12,2),
   USE_EVENT_UPPERQUARTILE NUMBER(12,2),
   USE_EVENT_MEDIAN     NUMBER(12,2),
   USE_EVENT_LOWERQUARTILE NUMBER(12,2),
   USE_EVENT_MIN        NUMBER(12,2),
   WEIGHT               NUMBER(16,6),
   USE_WEIGHTED_AVG     NUMBER(16,6),
   LRC                  NUMBER(12,2),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table MTPASA_REGIONSUMMARY is
'Region Results summary over aggregation periods.';

comment on column MTPASA_REGIONSUMMARY.RUN_DATETIME is
'Date processing of the run begins.';

comment on column MTPASA_REGIONSUMMARY.RUN_NO is
'Unique run id.';

comment on column MTPASA_REGIONSUMMARY.RUNTYPE is
'Type of run. Always RELIABILITY';

comment on column MTPASA_REGIONSUMMARY.DEMAND_POE_TYPE is
'Demand POE type used. Value are POE10, POE50';

comment on column MTPASA_REGIONSUMMARY.AGGREGATION_PERIOD is
'Period data is aggregated over. Values are YEAR, MONTH';

comment on column MTPASA_REGIONSUMMARY.PERIOD_ENDING is
'Datetime of day at end of period (i.e. last day of month or year reported)';

comment on column MTPASA_REGIONSUMMARY.REGIONID is
'The unique region identifier';

comment on column MTPASA_REGIONSUMMARY.NATIVEDEMAND is
'Native demand calculated from Operational As Generated trace supplied by Energy Forecasting';

comment on column MTPASA_REGIONSUMMARY.USE_PERCENTILE10 is
'Unserved energy period amount at the 10th percentile of iterations and reference years (MWh)';

comment on column MTPASA_REGIONSUMMARY.USE_PERCENTILE20 is
'Unserved energy period amount at the 20th percentile of iterations and reference years (MWh)';

comment on column MTPASA_REGIONSUMMARY.USE_PERCENTILE30 is
'Unserved energy period amount at the 30th percentile of iterations and reference years (MWh)';

comment on column MTPASA_REGIONSUMMARY.USE_PERCENTILE40 is
'Unserved energy period amount at the 40th percentile of iterations and reference years (MWh)';

comment on column MTPASA_REGIONSUMMARY.USE_PERCENTILE50 is
'Unserved energy period amount at the 50th percentile of iterations and reference years (MWh)';

comment on column MTPASA_REGIONSUMMARY.USE_PERCENTILE60 is
'Unserved energy period amount at the 60th percentile of iterations and reference years (MWh)';

comment on column MTPASA_REGIONSUMMARY.USE_PERCENTILE70 is
'Unserved energy period amount at the 70th percentile of iterations and reference years (MWh)';

comment on column MTPASA_REGIONSUMMARY.USE_PERCENTILE80 is
'Unserved energy period amount at the 80th percentile of iterations and reference years (MWh)';

comment on column MTPASA_REGIONSUMMARY.USE_PERCENTILE90 is
'Unserved energy period amount at the 90th percentile of iterations and reference years (MWh)';

comment on column MTPASA_REGIONSUMMARY.USE_PERCENTILE100 is
'Unserved energy period amount at the 100th percentile of iterations and reference years (MWh)';

comment on column MTPASA_REGIONSUMMARY.USE_AVERAGE is
'Average period unserved energy across iterations and reference years (MWh)';

comment on column MTPASA_REGIONSUMMARY.NUMBEROFITERATIONS is
'Total number of iterations and reference years performed';

comment on column MTPASA_REGIONSUMMARY.USE_NUMBEROFITERATIONS is
'Number of iterations and reference years showing unserved energy';

comment on column MTPASA_REGIONSUMMARY.USE_EVENT_MAX is
'Maximum unserved energy event size across all half hourly intervals and iterations and reference years that have unserved energy>0 (MW)';

comment on column MTPASA_REGIONSUMMARY.USE_EVENT_UPPERQUARTILE is
'Upper quartile unserved energy event size across all half hourly intervals and iterations and reference years that have unserved energy>0 (MW)';

comment on column MTPASA_REGIONSUMMARY.USE_EVENT_MEDIAN is
'Median unserved energy event size across all half hourly intervals and iterations and reference years that have unserved energy>0 (MW)';

comment on column MTPASA_REGIONSUMMARY.USE_EVENT_LOWERQUARTILE is
'Lower quartile unserved energy event size across all half hourly intervals and iterations and reference years that have unserved energy>0 (MW)';

comment on column MTPASA_REGIONSUMMARY.USE_EVENT_MIN is
'Minimum unserved energy event size across all half hourly intervals and iterations and reference years that have unserved energy>0 (MW)';

comment on column MTPASA_REGIONSUMMARY.WEIGHT is
'Fixed Values of 0.696 for 50 POE and 0.304 for 10 POE.';

comment on column MTPASA_REGIONSUMMARY.USE_WEIGHTED_AVG is
'Weighted average USE per region = (USE_AVERAGE_POE10/NATIVE_DEMAND_POE_10*WEIGHT_POE_10 + USE_AVERAGE_POE50/NATIVE_DEMAND_POE_50*WEIGHT_POE_50)*100';

comment on column MTPASA_REGIONSUMMARY.LRC is
'LRC Condition reported (Value=1) if USE_WEIGHTED_AVG >= 0.002% otherwise (Value=0)';

comment on column MTPASA_REGIONSUMMARY.LASTCHANGED is
'Last date and time record changed';

alter table MTPASA_REGIONSUMMARY
   add constraint MTPASA_REGIONSUMMARY_PK primary key (RUN_DATETIME, RUN_NO, RUNTYPE, DEMAND_POE_TYPE, AGGREGATION_PERIOD, PERIOD_ENDING, REGIONID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: MTPASA_RESERVELIMIT                                   */
/*==============================================================*/
create table MTPASA_RESERVELIMIT (
   EFFECTIVEDATE        DATE                  not null,
   VERSION_DATETIME     DATE                  not null,
   RESERVELIMITID       VARCHAR2(20)          not null,
   DESCRIPTION          VARCHAR2(200),
   RHS                  NUMBER(16,6),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MTPASA_RESERVELIMIT is
'MT PASA input table defining a MT PASA Reserve Requirement within a single set. An MT PASA Reserve Requirement can span more than one region.';

comment on column MTPASA_RESERVELIMIT.EFFECTIVEDATE is
'Trade date when the set of reserve requirements become effective';

comment on column MTPASA_RESERVELIMIT.VERSION_DATETIME is
'Timestamp when the set of reserve requirements become effective';

comment on column MTPASA_RESERVELIMIT.RESERVELIMITID is
'MT PASA Reserve Requirement identifier';

comment on column MTPASA_RESERVELIMIT.DESCRIPTION is
'Description of this Reserve Requirement';

comment on column MTPASA_RESERVELIMIT.RHS is
'Right hand side value for this Reserve requirement';

comment on column MTPASA_RESERVELIMIT.LASTCHANGED is
'Timestamp the record was last modified.';

alter table MTPASA_RESERVELIMIT
   add constraint PK_MTPASA_RESERVELIMIT primary key (EFFECTIVEDATE, VERSION_DATETIME, RESERVELIMITID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: MTPASA_RESERVELIMITSOLUTION                           */
/*==============================================================*/
create table MTPASA_RESERVELIMITSOLUTION (
   RUN_DATETIME         DATE                  not null,
   RUN_NO               NUMBER(3,0)           not null,
   RUNTYPE              VARCHAR2(20)          not null,
   ENERGYBLOCK          DATE                  not null,
   DAY                  DATE                  not null,
   LDCBLOCK             NUMBER(3,0)           not null,
   RESERVELIMITID       VARCHAR2(20)          not null,
   MARGINALVALUE        NUMBER(16,6),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table MTPASA_RESERVELIMITSOLUTION is
'MTPASA_RESERVELIMITSOLUTION is obsolete from 2017 End of Year DM4.27 Release. 
MT PASA Solution table reporting whether a MT PASA Reserve requirement is binding for each day and LDC block of the run.';

comment on column MTPASA_RESERVELIMITSOLUTION.RUN_DATETIME is
'Date processing of the run begins';

comment on column MTPASA_RESERVELIMITSOLUTION.RUN_NO is
'Unique run ID. Generated from the solution file Case ID.';

comment on column MTPASA_RESERVELIMITSOLUTION.RUNTYPE is
'Type of run. Values are RELIABILITY_LRC and OUTAGE_LRC';

comment on column MTPASA_RESERVELIMITSOLUTION.ENERGYBLOCK is
'Sunday at start of the week for this solutions energy block. Generated from the solution file energy block.';

comment on column MTPASA_RESERVELIMITSOLUTION.DAY is
'Day this solution is for. Generated from the solution file period id.';

comment on column MTPASA_RESERVELIMITSOLUTION.LDCBLOCK is
'Load Duration Curve block this solution is for. Generated from the solution file period id.';

comment on column MTPASA_RESERVELIMITSOLUTION.RESERVELIMITID is
'The unique identifier of the MT PASA LRC Reserve Requirement.';

comment on column MTPASA_RESERVELIMITSOLUTION.MARGINALVALUE is
'Marginal Value of the Reserve Requirement Constraint. A non-zero value indicates that the reserve requirement is binding.';

comment on column MTPASA_RESERVELIMITSOLUTION.LASTCHANGED is
'Timestamp the record was last modified.';

alter table MTPASA_RESERVELIMITSOLUTION
   add constraint PK_MTPASA_RESERVELIMITSOLUTION primary key (RUN_DATETIME, RUN_NO, RUNTYPE, ENERGYBLOCK, DAY, LDCBLOCK, RESERVELIMITID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: MTPASA_RESERVELIMIT_REGION                            */
/*==============================================================*/
create table MTPASA_RESERVELIMIT_REGION (
   EFFECTIVEDATE        DATE                  not null,
   VERSION_DATETIME     DATE                  not null,
   RESERVELIMITID       VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   COEF                 NUMBER(16,6),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MTPASA_RESERVELIMIT_REGION is
'MT PASA input table to define the regions that are part of a single MT PASA Reserve Requirement';

comment on column MTPASA_RESERVELIMIT_REGION.EFFECTIVEDATE is
'Trade date when the set of reserve requirements become effective';

comment on column MTPASA_RESERVELIMIT_REGION.VERSION_DATETIME is
'Timestamp when the set of reserve requirements become effective';

comment on column MTPASA_RESERVELIMIT_REGION.RESERVELIMITID is
'MT PASA Reserve requirement identifier';

comment on column MTPASA_RESERVELIMIT_REGION.REGIONID is
'Region ID - identifier of a NEM region included in this requirement';

comment on column MTPASA_RESERVELIMIT_REGION.COEF is
'Coefficient for the region in this reserve requirement';

comment on column MTPASA_RESERVELIMIT_REGION.LASTCHANGED is
'Timestamp the record was last modified';

alter table MTPASA_RESERVELIMIT_REGION
   add constraint PK_MTPASA_RESERVELIMIT_REGION primary key (EFFECTIVEDATE, VERSION_DATETIME, RESERVELIMITID, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: MTPASA_RESERVELIMIT_SET                               */
/*==============================================================*/
create table MTPASA_RESERVELIMIT_SET (
   EFFECTIVEDATE        DATE                  not null,
   VERSION_DATETIME     DATE                  not null,
   RESERVELIMIT_SET_ID  VARCHAR2(20),
   DESCRIPTION          VARCHAR2(200),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(20),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table MTPASA_RESERVELIMIT_SET is
'MT PASA input table defining a set of MT PASA Reserve Requirements. Note only one set can be active on a given date.';

comment on column MTPASA_RESERVELIMIT_SET.EFFECTIVEDATE is
'Trade date when the set of reserve requirements become effective';

comment on column MTPASA_RESERVELIMIT_SET.VERSION_DATETIME is
'Timestamp when the set of reserve requirements become effective';

comment on column MTPASA_RESERVELIMIT_SET.RESERVELIMIT_SET_ID is
'MT PASA LRC Reserve Requirement Set Identifier';

comment on column MTPASA_RESERVELIMIT_SET.DESCRIPTION is
'Description of this set of Reserve Requirements';

comment on column MTPASA_RESERVELIMIT_SET.AUTHORISEDDATE is
'Date the requirement set was authorised';

comment on column MTPASA_RESERVELIMIT_SET.AUTHORISEDBY is
'User authorising this requirement set';

comment on column MTPASA_RESERVELIMIT_SET.LASTCHANGED is
'Timestamp the record was last modified';

alter table MTPASA_RESERVELIMIT_SET
   add constraint PK_MTPASA_RESERVELIMIT_SET primary key (EFFECTIVEDATE, VERSION_DATETIME)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: NEGATIVE_RESIDUE                                      */
/*==============================================================*/
create table NEGATIVE_RESIDUE (
   SETTLEMENTDATE       date                  not null,
   NRM_DATETIME         date                  not null,
   DIRECTIONAL_INTERCONNECTORID varchar2(30)          not null,
   NRM_ACTIVATED_FLAG   number(1,0),
   CUMUL_NEGRESIDUE_AMOUNT number(15,5),
   CUMUL_NEGRESIDUE_PREV_TI number(15,5),
   NEGRESIDUE_CURRENT_TI number(15,5),
   NEGRESIDUE_PD_NEXT_TI number(15,5),
   PRICE_REVISION       varchar2(30),
   PREDISPATCHSEQNO     varchar2(20),
   EVENT_ACTIVATED_DI   date,
   EVENT_DEACTIVATED_DI date,
   DI_NOTBINDING_COUNT  number(2,0),
   DI_VIOLATED_COUNT    number(2,0),
   NRMCONSTRAINT_BLOCKED_FLAG number(1,0)
)
   tablespace PSSTAB03;

comment on table NEGATIVE_RESIDUE is
'Shows the inputs provided to the Negative Residue Constraints in the Dispatch horizon';

comment on column NEGATIVE_RESIDUE.SETTLEMENTDATE is
'Dispatch Interval';

comment on column NEGATIVE_RESIDUE.NRM_DATETIME is
'The time that residue information is processed';

comment on column NEGATIVE_RESIDUE.DIRECTIONAL_INTERCONNECTORID is
'Negative residue related direction interconnector id';

comment on column NEGATIVE_RESIDUE.NRM_ACTIVATED_FLAG is
'Is 1 if negative residue process is on, else is 0';

comment on column NEGATIVE_RESIDUE.CUMUL_NEGRESIDUE_AMOUNT is
'Negative residue triggering amount';

comment on column NEGATIVE_RESIDUE.CUMUL_NEGRESIDUE_PREV_TI is
'Previous trading interval cumulative negative residue amount';

comment on column NEGATIVE_RESIDUE.NEGRESIDUE_CURRENT_TI is
'Current trading interval negative residue amount';

comment on column NEGATIVE_RESIDUE.NEGRESIDUE_PD_NEXT_TI is
'The cumulative negative residue for the next trading interval (PD)';

comment on column NEGATIVE_RESIDUE.PRICE_REVISION is
'SubjectToReview, Indeterminate, Accepted or Rejected';

comment on column NEGATIVE_RESIDUE.PREDISPATCHSEQNO is
'Predispatch sequence number';

comment on column NEGATIVE_RESIDUE.EVENT_ACTIVATED_DI is
'The starting DI when NRM event is active';

comment on column NEGATIVE_RESIDUE.EVENT_DEACTIVATED_DI is
'The finishing DI when NRM event stops being active.';

comment on column NEGATIVE_RESIDUE.DI_NOTBINDING_COUNT is
'Count of the number of DIs not binding by this constraint';

comment on column NEGATIVE_RESIDUE.DI_VIOLATED_COUNT is
'Count of the number of DIs violated by this constraint';

comment on column NEGATIVE_RESIDUE.NRMCONSTRAINT_BLOCKED_FLAG is
'1 if constraint is blocked, else 0';

alter table NEGATIVE_RESIDUE
   add constraint NEGATIVE_RESIDUE_PK primary key (SETTLEMENTDATE, NRM_DATETIME, DIRECTIONAL_INTERCONNECTORID)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: NETWORK_EQUIPMENTDETAIL                               */
/*==============================================================*/
create table NETWORK_EQUIPMENTDETAIL (
   SUBSTATIONID         VARCHAR(30)           not null,
   EQUIPMENTTYPE        VARCHAR(10)           not null,
   EQUIPMENTID          VARCHAR(30)           not null,
   VALIDFROM            TIMESTAMP(3)          not null,
   VALIDTO              TIMESTAMP(3),
   VOLTAGE              VARCHAR(20),
   DESCRIPTION          VARCHAR(100),
   LASTCHANGED          TIMESTAMP(3),
   ELEMENTID            NUMBER(15,0)          not null
)
   tablespace PSSTAB02;

comment on table NETWORK_EQUIPMENTDETAIL is
'NETWORK_EQUIPMENTDETAIL Provides details on equipment that may have outages or ratings. A single piece of equipment may have multiple records if its details change.
A line will typically have at least two valid records at a time, once for each end of the line.';

comment on column NETWORK_EQUIPMENTDETAIL.SUBSTATIONID is
'ID uniquely identifying the substation this equipment is located at';

comment on column NETWORK_EQUIPMENTDETAIL.EQUIPMENTTYPE is
'The type of equipment. Valid values are:
LINE = Line
TRANS = Transformer
CB = Circuit breaker
ISOL = Isolator
CAP = Capacitor
REAC = Reactor
UNIT = Unit';

comment on column NETWORK_EQUIPMENTDETAIL.EQUIPMENTID is
'A unique identifier for this type of equipment at this substation';

comment on column NETWORK_EQUIPMENTDETAIL.VALIDFROM is
'The date that this record is applies from (inclusive)';

comment on column NETWORK_EQUIPMENTDETAIL.VALIDTO is
'The date that this record applies until (exclusive)';

comment on column NETWORK_EQUIPMENTDETAIL.VOLTAGE is
'The voltage in KV for this equipment.
Transformers may have multiple voltages defined.
E.g. 132_110_33';

comment on column NETWORK_EQUIPMENTDETAIL.DESCRIPTION is
'A short description for this equipment.';

comment on column NETWORK_EQUIPMENTDETAIL.LASTCHANGED is
'The time that this record was last changed.';

comment on column NETWORK_EQUIPMENTDETAIL.ELEMENTID is
'Equipment element id';

alter table NETWORK_EQUIPMENTDETAIL
   add constraint PK_NETWORK_EQUIPMENTDETAIL primary key (ELEMENTID, SUBSTATIONID, EQUIPMENTTYPE, EQUIPMENTID, VALIDFROM)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: NETWORK_EQUIPMENTDETAIL_LCX                           */
/*==============================================================*/
create index NETWORK_EQUIPMENTDETAIL_LCX on NETWORK_EQUIPMENTDETAIL (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: NETWORK_OUTAGECONSTRAINTSET                           */
/*==============================================================*/
create table NETWORK_OUTAGECONSTRAINTSET (
   OUTAGEID             NUMBER(15,0)          not null,
   GENCONSETID          VARCHAR(50)           not null,
   STARTINTERVAL        DATE,
   ENDINTERVAL          DATE
)
   tablespace PSSTAB02;

comment on table NETWORK_OUTAGECONSTRAINTSET is
'NETWORK_OUTAGECONSTRAINTSET lists the Constraint Set or Sets that are expected to be invoked for the outage once it is confirmed to proceed.';

comment on column NETWORK_OUTAGECONSTRAINTSET.OUTAGEID is
'ID uniquely identifying the outage';

comment on column NETWORK_OUTAGECONSTRAINTSET.GENCONSETID is
'ID for the constraint set';

comment on column NETWORK_OUTAGECONSTRAINTSET.STARTINTERVAL is
'The dispatch interval that this constraint applies from';

comment on column NETWORK_OUTAGECONSTRAINTSET.ENDINTERVAL is
'The dispatch interval that this constraint applies until.';

alter table NETWORK_OUTAGECONSTRAINTSET
   add constraint PK_NETWORK_OUTAGECONSTRAINTSET primary key (OUTAGEID, GENCONSETID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: NETWORK_OUTAGEDETAIL                                  */
/*==============================================================*/
create table NETWORK_OUTAGEDETAIL (
   OUTAGEID             NUMBER(15,0)          not null,
   SUBSTATIONID         VARCHAR(30)           not null,
   EQUIPMENTTYPE        VARCHAR(10)           not null,
   EQUIPMENTID          VARCHAR(30)           not null,
   STARTTIME            DATE                  not null,
   ENDTIME              DATE,
   SUBMITTEDDATE        DATE,
   OUTAGESTATUSCODE     VARCHAR(10),
   RESUBMITREASON       VARCHAR(50),
   RESUBMITOUTAGEID     NUMBER(15,0),
   RECALLTIMEDAY        NUMBER(10,0),
   RECALLTIMENIGHT      NUMBER(10,0),
   LASTCHANGED          TIMESTAMP(3),
   REASON               VARCHAR2(100),
   ISSECONDARY          NUMBER(1,0),
   ACTUAL_STARTTIME     DATE,
   ACTUAL_ENDTIME       DATE,
   COMPANYREFCODE       VARCHAR2(20),
   ELEMENTID            NUMBER(15,0)          not null
)
   tablespace PSSTAB02;

comment on table NETWORK_OUTAGEDETAIL is
'Lists asset owners planned outages for transmission equipment. This also includes details for transmission equipment that will not have an outage, but associated secondary equipment has an outage and a related constraint set may be invoked. This scenario is indicated by the ISSECONDARY field in the table';

comment on column NETWORK_OUTAGEDETAIL.OUTAGEID is
'ID uniquely identifying the outage';

comment on column NETWORK_OUTAGEDETAIL.SUBSTATIONID is
'The substation this equipment is located at';

comment on column NETWORK_OUTAGEDETAIL.EQUIPMENTTYPE is
'The type of equipment. Valid values are:
LINE = Line
TRANS = Transformer
CB = Circuit breaker
ISOL = Isolator
CAP = Capacitor
REAC = Reactor
UNIT = Unit';

comment on column NETWORK_OUTAGEDETAIL.EQUIPMENTID is
'A unique identifier for this equipment at this substation, and based on its type';

comment on column NETWORK_OUTAGEDETAIL.STARTTIME is
'The planned starting date and time of the outage';

comment on column NETWORK_OUTAGEDETAIL.ENDTIME is
'The planned ending date and time of the outage';

comment on column NETWORK_OUTAGEDETAIL.SUBMITTEDDATE is
'The date and time this outage was first submitted';

comment on column NETWORK_OUTAGEDETAIL.OUTAGESTATUSCODE is
'A code representing the status of the outage.
The OUTAGESTATUSCODE table will store a detailed description of each code.';

comment on column NETWORK_OUTAGEDETAIL.RESUBMITREASON is
'Changes to an outage key details may require the outage to be resubmitted.
A new outage id will then be allocated and the outage will be reassessed.
This field will detail the reason for the change.';

comment on column NETWORK_OUTAGEDETAIL.RESUBMITOUTAGEID is
'The new outage id created from a resubmit.';

comment on column NETWORK_OUTAGEDETAIL.RECALLTIMEDAY is
'The recall time in minutes during the day';

comment on column NETWORK_OUTAGEDETAIL.RECALLTIMENIGHT is
'The recall time in minutes during the night';

comment on column NETWORK_OUTAGEDETAIL.LASTCHANGED is
'The time that this record was last changed';

comment on column NETWORK_OUTAGEDETAIL.REASON is
'The reason provided by the asset owner for this outage';

comment on column NETWORK_OUTAGEDETAIL.ISSECONDARY is
'1 = The outage is for a secondary piece of equipment that has an associated constraint set. The transmission equipment is still in service. 0 = The outage is for the transmission equipment';

comment on column NETWORK_OUTAGEDETAIL.ACTUAL_STARTTIME is
'The actual starting date/time of the outage';

comment on column NETWORK_OUTAGEDETAIL.ACTUAL_ENDTIME is
'The actual ending date/time of the outage';

comment on column NETWORK_OUTAGEDETAIL.COMPANYREFCODE is
'The asset owners reference code for this outage';

comment on column NETWORK_OUTAGEDETAIL.ELEMENTID is
'Equipment element id';

alter table NETWORK_OUTAGEDETAIL
   add constraint PK_NETWORK_OUTAGEDETAIL primary key (OUTAGEID, ELEMENTID, SUBSTATIONID, EQUIPMENTTYPE, EQUIPMENTID, STARTTIME)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: NETWORK_OUTAGEDETAIL_LCX                              */
/*==============================================================*/
create index NETWORK_OUTAGEDETAIL_LCX on NETWORK_OUTAGEDETAIL (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: NETWORK_OUTAGESTATUSCODE                              */
/*==============================================================*/
create table NETWORK_OUTAGESTATUSCODE (
   OUTAGESTATUSCODE     VARCHAR(10)           not null,
   DESCRIPTION          VARCHAR(100),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table NETWORK_OUTAGESTATUSCODE is
'NETWORK_OUTAGESTATUSCODE describes the different outage status codes';

comment on column NETWORK_OUTAGESTATUSCODE.OUTAGESTATUSCODE is
'A code representing the status of an outage';

comment on column NETWORK_OUTAGESTATUSCODE.DESCRIPTION is
'A description of the status code';

comment on column NETWORK_OUTAGESTATUSCODE.LASTCHANGED is
'The time that this record was last changed';

alter table NETWORK_OUTAGESTATUSCODE
   add constraint PK_NETWORK_OUTAGESTATUSCODE primary key (OUTAGESTATUSCODE)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: NETWORK_RATING                                        */
/*==============================================================*/
create table NETWORK_RATING (
   SPD_ID               VARCHAR(21)           not null,
   VALIDFROM            DATE                  not null,
   VALIDTO              DATE,
   REGIONID             VARCHAR(10),
   SUBSTATIONID         VARCHAR(30),
   EQUIPMENTTYPE        VARCHAR(10),
   EQUIPMENTID          VARCHAR(30),
   RATINGLEVEL          VARCHAR(10),
   ISDYNAMIC            NUMBER(1,0),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table NETWORK_RATING is
'NETWORK_RATING defines a list of the equipment ratings that may be used as inputs to market constraints.
If the rating is flagged as dynamic then in real-time the rating will be dynamically determined and the static value will be used as a fallback value should the dynamic value fail.
Note:
In some rare cases equipment has ratings provided from more than one TNSP. This is identified by a different SPD Id. The value used in the NEM is normally the more restrictive of the two values.';

comment on column NETWORK_RATING.SPD_ID is
'ID defining this data source for use in constraints';

comment on column NETWORK_RATING.VALIDFROM is
'The date that this record is applies from (inclusive)';

comment on column NETWORK_RATING.VALIDTO is
'The date that this record applies until (exclusive)';

comment on column NETWORK_RATING.REGIONID is
'The region that this rating is for';

comment on column NETWORK_RATING.SUBSTATIONID is
'The substation the equipment is located at';

comment on column NETWORK_RATING.EQUIPMENTTYPE is
'The type of equipment. Valid values are:
LINE = Line
TRANS = Transformer
CB = Circuit breaker
ISOL = Isolator
CAP = Capacitor
REAC = Reactor
UNIT = Unit';

comment on column NETWORK_RATING.EQUIPMENTID is
'A unique identifier for this equipment at this substation, and based on its type';

comment on column NETWORK_RATING.RATINGLEVEL is
'The rating level of the value used, one of:
NORM = Continuous rating value. Applied under pre-contingent conditions.
EMER = Continuous rating value. Applied under pre-contingent conditions
LDSH = Load Shedding';

comment on column NETWORK_RATING.ISDYNAMIC is
'One of:
1 = Normally uses dynamic ratings
0 = No dynamic ratings, static ratings are used';

comment on column NETWORK_RATING.LASTCHANGED is
'The time that this record was last changed';

alter table NETWORK_RATING
   add constraint PK_NETWORK_RATING primary key (SPD_ID, VALIDFROM)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: NETWORK_RATING_LCX                                    */
/*==============================================================*/
create index NETWORK_RATING_LCX on NETWORK_RATING (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: NETWORK_REALTIMERATING                                */
/*==============================================================*/
create table NETWORK_REALTIMERATING (
   SETTLEMENTDATE       DATE                  not null,
   SPD_ID               VARCHAR(21)           not null,
   RATINGVALUE          NUMBER(16,6)          not null
)
   tablespace PSSTAB02;

comment on table NETWORK_REALTIMERATING is
'The NETWORK_REALTIMERATING table shows the equipment rating values in MVA used as inputs to constraints in the dispatch solution. This includes values for both static and dynamic ratings. The NETWORK_RATING table can be used to determine the physical equipment the rating is for based on the SPD_ID value.';

comment on column NETWORK_REALTIMERATING.SETTLEMENTDATE is
'The dispatch interval the rating applies to';

comment on column NETWORK_REALTIMERATING.SPD_ID is
'ID defining this data source for use in constraints';

comment on column NETWORK_REALTIMERATING.RATINGVALUE is
'The defined equipment rating value in MVA';

alter table NETWORK_REALTIMERATING
   add constraint PK_NETWORK_REALTIMERATING primary key (SETTLEMENTDATE, SPD_ID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: NETWORK_STATICRATING                                  */
/*==============================================================*/
create table NETWORK_STATICRATING (
   SUBSTATIONID         VARCHAR(30)           not null,
   EQUIPMENTTYPE        VARCHAR(10)           not null,
   EQUIPMENTID          VARCHAR(30)           not null,
   RATINGLEVEL          VARCHAR(10)           not null,
   APPLICATIONID        VARCHAR(20)           not null,
   VALIDFROM            DATE                  not null,
   VALIDTO              DATE,
   RATINGVALUE          NUMBER(16,6),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table NETWORK_STATICRATING is
'NETWORK_STATICRATING lists the static rating values that will apply for a Rating Application ID.
This data does not provide information for when the rating actually applies in the NEM. This is dependent on the Rating Application definition.
For information on the Rating Applications please refer to the information published on the AEMO website under the topic "Transmission Equipment Ratings". The Rating Applications are referred to as Alternate Value Application Ratings.
Ratings that normally use dynamic values will also have static rating values defined. These are used as a fallback if the dynamic rating fails.';

comment on column NETWORK_STATICRATING.SUBSTATIONID is
'The substation the equipment is located at';

comment on column NETWORK_STATICRATING.EQUIPMENTTYPE is
'The type of equipment. Valid values are:
LINE = Line
TRANS = Transformer
CB = Circuit breaker
ISOL = Isolator
CAP = Capacitor
REAC = Reactor
UNIT = Unit';

comment on column NETWORK_STATICRATING.EQUIPMENTID is
'A unique identifier for this type of equipment at this substation';

comment on column NETWORK_STATICRATING.RATINGLEVEL is
'The rating level of the value used, one of:
NORM = Continuous rating value. Applied under pre-contingent conditions.
EMER = Continuous rating value. Applied under pre-contingent conditions
LDSH = Load Shedding';

comment on column NETWORK_STATICRATING.APPLICATIONID is
'The applicationid which defines the application timeframes of this rating.';

comment on column NETWORK_STATICRATING.VALIDFROM is
'The date that this record is applies from (inclusive)';

comment on column NETWORK_STATICRATING.VALIDTO is
'The date that this record applies until (exclusive)';

comment on column NETWORK_STATICRATING.RATINGVALUE is
'The rating value in MVA that applies. This may be positive or negative depending on which side of the nominal MW flow direction the rating value applies.
Flow into a transmission device is positive, flow out of the device is negative.';

comment on column NETWORK_STATICRATING.LASTCHANGED is
'The time that this record was last changed.';

alter table NETWORK_STATICRATING
   add constraint PK_NETWORK_STATICRATING primary key (SUBSTATIONID, EQUIPMENTTYPE, EQUIPMENTID, RATINGLEVEL, APPLICATIONID, VALIDFROM)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: NETWORK_STATICRATING_LCX                              */
/*==============================================================*/
create index NETWORK_STATICRATING_LCX on NETWORK_STATICRATING (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: NETWORK_SUBSTATIONDETAIL                              */
/*==============================================================*/
create table NETWORK_SUBSTATIONDETAIL (
   SUBSTATIONID         VARCHAR(30)           not null,
   VALIDFROM            TIMESTAMP(3)          not null,
   VALIDTO              TIMESTAMP(3),
   DESCRIPTION          VARCHAR(100),
   REGIONID             VARCHAR(10),
   OWNERID              VARCHAR(30),
   LASTCHANGED          TIMESTAMP(3)
)
   tablespace PSSTAB01;

comment on table NETWORK_SUBSTATIONDETAIL is
'NETWORK_SUBSTATIONDETAIL sets out the attributes of sub-stations across time';

comment on column NETWORK_SUBSTATIONDETAIL.SUBSTATIONID is
'ID uniquely identifying this substation';

comment on column NETWORK_SUBSTATIONDETAIL.VALIDFROM is
'The record is valid from this date (inclusive)';

comment on column NETWORK_SUBSTATIONDETAIL.VALIDTO is
'The record is valid up until this date (exclusive)';

comment on column NETWORK_SUBSTATIONDETAIL.DESCRIPTION is
'Description of the substation';

comment on column NETWORK_SUBSTATIONDETAIL.REGIONID is
'The NEM region the substation is in';

comment on column NETWORK_SUBSTATIONDETAIL.OWNERID is
'The TNSP who is responsible for this substation';

comment on column NETWORK_SUBSTATIONDETAIL.LASTCHANGED is
'The time that this record was last changed.';

alter table NETWORK_SUBSTATIONDETAIL
   add constraint PK_NETWORK_SUBSTATIONDETAIL primary key (SUBSTATIONID, VALIDFROM)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: NETWORK_SUBSTATIONDETAIL_LCX                          */
/*==============================================================*/
create index NETWORK_SUBSTATIONDETAIL_LCX on NETWORK_SUBSTATIONDETAIL (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: OARTRACK                                              */
/*==============================================================*/
create table OARTRACK (
   SETTLEMENTDATE       DATE                  not null,
   OFFERDATE            DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   FILENAME             VARCHAR2(40),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(10),
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table OARTRACK is
'OARTRACK shows an audit trail of bids for a particular settlement day. Corrupt bids do not update OARTRACK, but are just in OFFERFILETRK.';

comment on column OARTRACK.SETTLEMENTDATE is
'Settlement date';

comment on column OARTRACK.OFFERDATE is
'Date file offered';

comment on column OARTRACK.VERSIONNO is
'Version no for this offer date';

comment on column OARTRACK.PARTICIPANTID is
'Unique participant identifier';

comment on column OARTRACK.FILENAME is
'Load file name';

comment on column OARTRACK.AUTHORISEDDATE is
'Date record authorised';

comment on column OARTRACK.AUTHORISEDBY is
'User authorising record';

comment on column OARTRACK.LASTCHANGED is
'Last date and time record changed';

alter table OARTRACK
   add constraint OARTRACK_PK primary key (SETTLEMENTDATE, OFFERDATE, VERSIONNO, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: OARTRACK_NDX2                                         */
/*==============================================================*/
create index OARTRACK_NDX2 on OARTRACK (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: OARTRACK_LCX                                          */
/*==============================================================*/
create index OARTRACK_LCX on OARTRACK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: OFFERAGCDATA                                          */
/*==============================================================*/
create table OFFERAGCDATA (
   CONTRACTID           VARCHAR2(10)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   AVAILABILITY         NUMBER(4,0),
   UPPERLIMIT           NUMBER(4,0),
   LOWERLIMIT           NUMBER(4,0),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   FILENAME             VARCHAR2(40),
   LASTCHANGED          DATE,
   PERIODID             NUMBER(3,0)           not null,
   AGCUP                NUMBER(3,0),
   AGCDOWN              NUMBER(3,0)
)
   tablespace PSSTAB01;

comment on table OFFERAGCDATA is
'OFFERAGCDATA shows availability reoffers of Automatic Generation Control.';

comment on column OFFERAGCDATA.CONTRACTID is
'Contract Identifier';

comment on column OFFERAGCDATA.EFFECTIVEDATE is
'Market date of offer';

comment on column OFFERAGCDATA.VERSIONNO is
'Version no of record';

comment on column OFFERAGCDATA.AVAILABILITY is
'Availability flag (0 or 1)';

comment on column OFFERAGCDATA.UPPERLIMIT is
'Upper control limit. This is used by SPD.';

comment on column OFFERAGCDATA.LOWERLIMIT is
'Lower control limit MW. This is used by SPD.';

comment on column OFFERAGCDATA.AUTHORISEDDATE is
'Authorised date';

comment on column OFFERAGCDATA.AUTHORISEDBY is
'Authorised by';

comment on column OFFERAGCDATA.FILENAME is
'Name of reoffer file';

comment on column OFFERAGCDATA.LASTCHANGED is
'Last date and time record changed';

comment on column OFFERAGCDATA.PERIODID is
'Market day trading interval number';

comment on column OFFERAGCDATA.AGCUP is
'AGC Ramp Rate Up. This is used by SPD.';

comment on column OFFERAGCDATA.AGCDOWN is
'AGC Ramp Rate Down. This is used by SPD.';

alter table OFFERAGCDATA
   add constraint OFFERAGCDATA_PK primary key (CONTRACTID, EFFECTIVEDATE, VERSIONNO, PERIODID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: OFFERAGCDATA_NDX2                                     */
/*==============================================================*/
create index OFFERAGCDATA_NDX2 on OFFERAGCDATA (
   CONTRACTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: OFFERAGCDATA_LCX                                      */
/*==============================================================*/
create index OFFERAGCDATA_LCX on OFFERAGCDATA (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: OFFERASTRK                                            */
/*==============================================================*/
create table OFFERASTRK (
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   FILENAME             VARCHAR2(40),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table OFFERASTRK is
'OFFERASTRK tracks successfully acknowledged ancillary service reoffers.';

comment on column OFFERASTRK.EFFECTIVEDATE is
'Market day starting at 4:00 am';

comment on column OFFERASTRK.VERSIONNO is
'Version of the offer for that date';

comment on column OFFERASTRK.PARTICIPANTID is
'Participant ID';

comment on column OFFERASTRK.FILENAME is
'Submitted file name.';

comment on column OFFERASTRK.LASTCHANGED is
'Last changed date and time.';

alter table OFFERASTRK
   add constraint OFFERASTRK_PK primary key (EFFECTIVEDATE, VERSIONNO, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: OFFERASTRK_LCX                                        */
/*==============================================================*/
create index OFFERASTRK_LCX on OFFERASTRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: OFFERFILETRK                                          */
/*==============================================================*/
create table OFFERFILETRK (
   OFFERDATE            DATE                  not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   STATUS               VARCHAR2(10),
   ACKFILENAME          VARCHAR2(40),
   ENDDATE              DATE,
   FILENAME             VARCHAR2(40)          not null,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table OFFERFILETRK is
'OFFERFILETRK shows an audit trail of all bid files submitted containing energy bids, including corrupt bids/rebids.';

comment on column OFFERFILETRK.OFFERDATE is
'Date file offered';

comment on column OFFERFILETRK.PARTICIPANTID is
'Unique participant identifier';

comment on column OFFERFILETRK.STATUS is
'Load status [SUCCESSFUL/CORRUPT]';

comment on column OFFERFILETRK.ACKFILENAME is
'Acknowledge file name';

comment on column OFFERFILETRK.ENDDATE is
'Not used';

comment on column OFFERFILETRK.FILENAME is
'Load file name';

comment on column OFFERFILETRK.LASTCHANGED is
'Last date and time record changed';

alter table OFFERFILETRK
   add constraint OFFERFILETRK_PK primary key (OFFERDATE, FILENAME, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: OFFERFILETRK_NDX2                                     */
/*==============================================================*/
create index OFFERFILETRK_NDX2 on OFFERFILETRK (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: OFFERFILETRK_LCX                                      */
/*==============================================================*/
create index OFFERFILETRK_LCX on OFFERFILETRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: OFFERGOVDATA                                          */
/*==============================================================*/
create table OFFERGOVDATA (
   CONTRACTID           VARCHAR2(10)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   PERIODID             NUMBER(3,0)           not null,
   SEC6AVAILUP          NUMBER(6,0),
   SEC6AVAILDOWN        NUMBER(6,0),
   SEC60AVAILUP         NUMBER(6,0),
   SEC60AVAILDOWN       NUMBER(6,0),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   FILENAME             VARCHAR2(40),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table OFFERGOVDATA is
'OFFERGOVDATA sets out reoffers of governor (6 and 60 second FCAS) availability.';

comment on column OFFERGOVDATA.CONTRACTID is
'Contract Version No.';

comment on column OFFERGOVDATA.EFFECTIVEDATE is
'Effective Date of Re-Offer';

comment on column OFFERGOVDATA.VERSIONNO is
'Version No. of Re-Offer';

comment on column OFFERGOVDATA.PERIODID is
'Market day trading interval number';

comment on column OFFERGOVDATA.SEC6AVAILUP is
'Availability for 6 Second Raise (0 or 1. ''0''= unavailable, ''1'' = available)';

comment on column OFFERGOVDATA.SEC6AVAILDOWN is
'Availability for 6 Second Lower (0 or 1)';

comment on column OFFERGOVDATA.SEC60AVAILUP is
'Availability for 60 Second Raise (0 or 1)';

comment on column OFFERGOVDATA.SEC60AVAILDOWN is
'Availability for 60 Second Lower (0 or 1)';

comment on column OFFERGOVDATA.AUTHORISEDDATE is
'Date Contract was Authorised';

comment on column OFFERGOVDATA.AUTHORISEDBY is
'User Name';

comment on column OFFERGOVDATA.FILENAME is
'File name of Re-Offer file';

comment on column OFFERGOVDATA.LASTCHANGED is
'Last date and time record changed';

alter table OFFERGOVDATA
   add constraint OFFERGOVDATA_PK primary key (CONTRACTID, EFFECTIVEDATE, VERSIONNO, PERIODID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: OFFERGOVDATA_NDX2                                     */
/*==============================================================*/
create index OFFERGOVDATA_NDX2 on OFFERGOVDATA (
   CONTRACTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: OFFERGOVDATA_LCX                                      */
/*==============================================================*/
create index OFFERGOVDATA_LCX on OFFERGOVDATA (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: OFFERLSHEDDATA                                        */
/*==============================================================*/
create table OFFERLSHEDDATA (
   CONTRACTID           VARCHAR2(10)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   AVAILABLELOAD        NUMBER(4,0),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   FILENAME             VARCHAR2(40),
   LASTCHANGED          DATE,
   PERIODID             NUMBER(3,0)           not null
)
   tablespace PSSTAB01;

comment on table OFFERLSHEDDATA is
'OFFERLSHEDDATA shows reoffers of load shed including available load shed quantity. This Table may also be used for NSCAS and Type 1 transitional services procured by AEMO under the ISF framework during 2025 and prior to the implementation of all system changes. During this time descriptions in these tables may not be correct.';

comment on column OFFERLSHEDDATA.CONTRACTID is
'Contract identifier';

comment on column OFFERLSHEDDATA.EFFECTIVEDATE is
'Market date of reoffer';

comment on column OFFERLSHEDDATA.VERSIONNO is
'Version No of reoffer';

comment on column OFFERLSHEDDATA.AVAILABLELOAD is
'Available load';

comment on column OFFERLSHEDDATA.AUTHORISEDDATE is
'Authorised date';

comment on column OFFERLSHEDDATA.AUTHORISEDBY is
'Authorised by';

comment on column OFFERLSHEDDATA.FILENAME is
'Name of reoffer file';

comment on column OFFERLSHEDDATA.LASTCHANGED is
'Last date and time record changed';

comment on column OFFERLSHEDDATA.PERIODID is
'Market day trading interval number';

alter table OFFERLSHEDDATA
   add constraint OFFERLSHEDDATA_PK primary key (CONTRACTID, EFFECTIVEDATE, VERSIONNO, PERIODID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: OFFERLSHEDDATA_LCX                                    */
/*==============================================================*/
create index OFFERLSHEDDATA_LCX on OFFERLSHEDDATA (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: OFFERRESTARTDATA                                      */
/*==============================================================*/
create table OFFERRESTARTDATA (
   CONTRACTID           VARCHAR2(10)          not null,
   OFFERDATE            DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   AVAILABILITY         VARCHAR2(3),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   FILENAME             VARCHAR2(40),
   LASTCHANGED          DATE,
   PERIODID             NUMBER(3,0)           not null
)
   tablespace PSSTAB01;

comment on table OFFERRESTARTDATA is
'OFFERRESTARTDATA sets out reoffers of system restart availability.';

comment on column OFFERRESTARTDATA.CONTRACTID is
'Contract identifier';

comment on column OFFERRESTARTDATA.OFFERDATE is
'Effective date of contract';

comment on column OFFERRESTARTDATA.VERSIONNO is
'Version No of contract';

comment on column OFFERRESTARTDATA.AVAILABILITY is
'Available load';

comment on column OFFERRESTARTDATA.AUTHORISEDDATE is
'Authorised date';

comment on column OFFERRESTARTDATA.AUTHORISEDBY is
'Authorised by';

comment on column OFFERRESTARTDATA.FILENAME is
'Name of reoffer file';

comment on column OFFERRESTARTDATA.LASTCHANGED is
'Last date and time record changed';

comment on column OFFERRESTARTDATA.PERIODID is
'Market day trading interval number';

alter table OFFERRESTARTDATA
   add constraint OFFERRESTARTDATA_PK primary key (CONTRACTID, OFFERDATE, VERSIONNO, PERIODID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: OFFERRESTARTDATA_LCX                                  */
/*==============================================================*/
create index OFFERRESTARTDATA_LCX on OFFERRESTARTDATA (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: OFFERRPOWERDATA                                       */
/*==============================================================*/
create table OFFERRPOWERDATA (
   CONTRACTID           VARCHAR2(10)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   PERIODID             NUMBER(3,0)           not null,
   AVAILABILITY         NUMBER(3,0),
   MTA                  NUMBER(6,0),
   MTG                  NUMBER(6,0),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   FILENAME             VARCHAR2(40),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table OFFERRPOWERDATA is
'OFFERRPOWERDATA shows reoffers of reactive power capability and settlement measurements. This Table may also be used for NSCAS and Type 1 transitional services procured by AEMO under the ISF framework during 2025 and prior to the implementation of all system changes. During this time descriptions in these tables may not be correct.';

comment on column OFFERRPOWERDATA.CONTRACTID is
'Contract Version No.';

comment on column OFFERRPOWERDATA.EFFECTIVEDATE is
'Contract Version No.';

comment on column OFFERRPOWERDATA.VERSIONNO is
'Version No. of Re-Offer';

comment on column OFFERRPOWERDATA.PERIODID is
'Market trading interval';

comment on column OFFERRPOWERDATA.AVAILABILITY is
'Availability of service';

comment on column OFFERRPOWERDATA.MTA is
'Reactive Power Absorption Capability (MVar)';

comment on column OFFERRPOWERDATA.MTG is
'Reactive Power Generation Capability (MVar)';

comment on column OFFERRPOWERDATA.AUTHORISEDDATE is
'Date Contract was Authorised';

comment on column OFFERRPOWERDATA.AUTHORISEDBY is
'User Name';

comment on column OFFERRPOWERDATA.FILENAME is
'File name of Re-Offer file';

comment on column OFFERRPOWERDATA.LASTCHANGED is
'Last date and time record changed';

alter table OFFERRPOWERDATA
   add constraint OFFERRPOWERDATA_PK primary key (CONTRACTID, EFFECTIVEDATE, VERSIONNO, PERIODID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: OFFERRPOWERDATA_NDX2                                  */
/*==============================================================*/
create index OFFERRPOWERDATA_NDX2 on OFFERRPOWERDATA (
   CONTRACTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: OFFERRPOWERDATA_LCX                                   */
/*==============================================================*/
create index OFFERRPOWERDATA_LCX on OFFERRPOWERDATA (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: OFFERULOADINGDATA                                     */
/*==============================================================*/
create table OFFERULOADINGDATA (
   CONTRACTID           VARCHAR2(10)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   AVAILABLELOAD        NUMBER(4,0),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   FILENAME             VARCHAR2(40),
   LASTCHANGED          DATE,
   PERIODID             NUMBER(3,0)           not null
)
   tablespace PSSTAB01;

comment on table OFFERULOADINGDATA is
'OFFERULOADINGDATA shows reoffers of rapid unit loading capability.';

comment on column OFFERULOADINGDATA.CONTRACTID is
'Contract identifier';

comment on column OFFERULOADINGDATA.EFFECTIVEDATE is
'Effective date of contract';

comment on column OFFERULOADINGDATA.VERSIONNO is
'Version No of contract';

comment on column OFFERULOADINGDATA.AVAILABLELOAD is
'Available load';

comment on column OFFERULOADINGDATA.AUTHORISEDDATE is
'Authorised date';

comment on column OFFERULOADINGDATA.AUTHORISEDBY is
'Authorised by';

comment on column OFFERULOADINGDATA.FILENAME is
'Name of reoffer file';

comment on column OFFERULOADINGDATA.LASTCHANGED is
'Last date and time record changed';

comment on column OFFERULOADINGDATA.PERIODID is
'Market day trading interval number';

alter table OFFERULOADINGDATA
   add constraint OFFERULOADINGDATA_PK primary key (CONTRACTID, EFFECTIVEDATE, VERSIONNO, PERIODID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: OFFERULOADINGDATA_NDX2                                */
/*==============================================================*/
create index OFFERULOADINGDATA_NDX2 on OFFERULOADINGDATA (
   CONTRACTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: OFFERULOADINGDATA_LCX                                 */
/*==============================================================*/
create index OFFERULOADINGDATA_LCX on OFFERULOADINGDATA (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: OFFERUNLOADINGDATA                                    */
/*==============================================================*/
create table OFFERUNLOADINGDATA (
   CONTRACTID           VARCHAR2(10)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   AVAILABLELOAD        NUMBER(4,0),
   AUTHORISEDDATE       DATE,
   AUTHORISEDBY         VARCHAR2(15),
   FILENAME             VARCHAR2(40),
   LASTCHANGED          DATE,
   PERIODID             NUMBER(3,0)           not null
)
   tablespace PSSTAB01;

comment on table OFFERUNLOADINGDATA is
'OFFERUNLOADINGDATA shows reoffers of rapid unit unloading capability.';

comment on column OFFERUNLOADINGDATA.CONTRACTID is
'Contract identifier';

comment on column OFFERUNLOADINGDATA.EFFECTIVEDATE is
'Market date of reoffer';

comment on column OFFERUNLOADINGDATA.VERSIONNO is
'Version No of reoffer';

comment on column OFFERUNLOADINGDATA.AVAILABLELOAD is
'Available load';

comment on column OFFERUNLOADINGDATA.AUTHORISEDDATE is
'Authorised date';

comment on column OFFERUNLOADINGDATA.AUTHORISEDBY is
'Authorised by';

comment on column OFFERUNLOADINGDATA.FILENAME is
'Name of reoffer file';

comment on column OFFERUNLOADINGDATA.LASTCHANGED is
'Last date and time record changed';

comment on column OFFERUNLOADINGDATA.PERIODID is
'Market day trading interval number';

alter table OFFERUNLOADINGDATA
   add constraint OFFERUNLOADINGDATA_PK primary key (CONTRACTID, EFFECTIVEDATE, VERSIONNO, PERIODID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: OFFERUNLOADINGDATA_NDX2                               */
/*==============================================================*/
create index OFFERUNLOADINGDATA_NDX2 on OFFERUNLOADINGDATA (
   CONTRACTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: OFFERUNLOADINGDATA_LCX                                */
/*==============================================================*/
create index OFFERUNLOADINGDATA_LCX on OFFERUNLOADINGDATA (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: OVERRIDERRP                                           */
/*==============================================================*/
create table OVERRIDERRP (
   REGIONID             VARCHAR2(10)          not null,
   STARTDATE            DATE                  not null,
   STARTPERIOD          NUMBER(3,0)           not null,
   ENDDATE              DATE,
   ENDPERIOD            NUMBER(3,0),
   RRP                  NUMBER(15,0),
   DESCRIPTION          VARCHAR2(128),
   AUTHORISESTART       VARCHAR2(15),
   AUTHORISEEND         VARCHAR2(15),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table OVERRIDERRP is
'OVERRIDERRP shows details of override price periods.';

comment on column OVERRIDERRP.REGIONID is
'Region Identifier';

comment on column OVERRIDERRP.STARTDATE is
'Starting date of override';

comment on column OVERRIDERRP.STARTPERIOD is
'Starting period of override';

comment on column OVERRIDERRP.ENDDATE is
'Termination date of override';

comment on column OVERRIDERRP.ENDPERIOD is
'Terminate period of override';

comment on column OVERRIDERRP.RRP is
'Dispatch Price';

comment on column OVERRIDERRP.DESCRIPTION is
'Description of reason for override';

comment on column OVERRIDERRP.AUTHORISESTART is
'Authorise Start of Override';

comment on column OVERRIDERRP.AUTHORISEEND is
'Authorise End of Override';

comment on column OVERRIDERRP.LASTCHANGED is
'Last date and time record changed';

alter table OVERRIDERRP
   add constraint OVERRIDERRP_PK primary key (STARTDATE, STARTPERIOD, REGIONID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: OVERRIDERRP_LCX                                       */
/*==============================================================*/
create index OVERRIDERRP_LCX on OVERRIDERRP (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: P5MIN_BLOCKEDCONSTRAINT                               */
/*==============================================================*/
create table P5MIN_BLOCKEDCONSTRAINT (
   RUN_DATETIME         DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null
)
   tablespace PSSTAB01;

comment on table P5MIN_BLOCKEDCONSTRAINT is
'P5MIN Blocked Constraints lists any constraints that were blocked in a P5MIN run. If no constraints are blocked, there will be no rows for that 5 minute predispatch run.';

comment on column P5MIN_BLOCKEDCONSTRAINT.RUN_DATETIME is
'5-minute Predispatch Run';

comment on column P5MIN_BLOCKEDCONSTRAINT.CONSTRAINTID is
'Generic Constraint identifier (synonymous with GenConID)';

alter table P5MIN_BLOCKEDCONSTRAINT
   add constraint P5MIN_BLOCKEDCONSTRAINT_PK primary key (RUN_DATETIME, CONSTRAINTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: P5MIN_CASESOLUTION                                    */
/*==============================================================*/
create table P5MIN_CASESOLUTION (
   RUN_DATETIME         DATE                  not null,
   STARTINTERVAL_DATETIME VARCHAR2(20),
   TOTALOBJECTIVE       NUMBER(27,10),
   NONPHYSICALLOSSES    NUMBER(1,0),
   TOTALAREAGENVIOLATION NUMBER(15,5),
   TOTALINTERCONNECTORVIOLATION NUMBER(15,5),
   TOTALGENERICVIOLATION NUMBER(15,5),
   TOTALRAMPRATEVIOLATION NUMBER(15,5),
   TOTALUNITMWCAPACITYVIOLATION NUMBER(15,5),
   TOTAL5MINVIOLATION   NUMBER(15,5),
   TOTALREGVIOLATION    NUMBER(15,5),
   TOTAL6SECVIOLATION   NUMBER(15,5),
   TOTAL60SECVIOLATION  NUMBER(15,5),
   TOTALENERGYCONSTRVIOLATION NUMBER(15,5),
   TOTALENERGYOFFERVIOLATION NUMBER(15,5),
   TOTALASPROFILEVIOLATION NUMBER(15,5),
   TOTALFASTSTARTVIOLATION NUMBER(15,5),
   LASTCHANGED          DATE,
   INTERVENTION         Number(2,0)
)
   tablespace PSSTAB02;

comment on table P5MIN_CASESOLUTION is
'The five-minute predispatch (P5Min) is a MMS system providing projected dispatch for 12 Dispatch cycles (one hour). The 5-minute Predispatch cycle runs every 5-minutes to produce a dispatch and pricing schedule to a 5-minute resolution covering the next hour, a total of twelve periods.
P5MIN_CASESOLUTION shows one record containing results pertaining to the entire solution.';

comment on column P5MIN_CASESOLUTION.RUN_DATETIME is
'Unique Timestamp Identifier for this study';

comment on column P5MIN_CASESOLUTION.STARTINTERVAL_DATETIME is
'Date and Time of first interval in study';

comment on column P5MIN_CASESOLUTION.TOTALOBJECTIVE is
'The Objective function from the LP';

comment on column P5MIN_CASESOLUTION.NONPHYSICALLOSSES is
'Flag to indicate non-physical losses occurred in this study';

comment on column P5MIN_CASESOLUTION.TOTALAREAGENVIOLATION is
'Sum of Regional Energy balance violations';

comment on column P5MIN_CASESOLUTION.TOTALINTERCONNECTORVIOLATION is
'Sum of Interconnector violations of standing data limits';

comment on column P5MIN_CASESOLUTION.TOTALGENERICVIOLATION is
'Sum of Generic Constraint violations';

comment on column P5MIN_CASESOLUTION.TOTALRAMPRATEVIOLATION is
'Sum of Unit Ramp Rate violations';

comment on column P5MIN_CASESOLUTION.TOTALUNITMWCAPACITYVIOLATION is
'Sum of unit capacity violations';

comment on column P5MIN_CASESOLUTION.TOTAL5MINVIOLATION is
'Sum of regional 5 min FCAS violations';

comment on column P5MIN_CASESOLUTION.TOTALREGVIOLATION is
'Sum of regional regulation FCAS violations';

comment on column P5MIN_CASESOLUTION.TOTAL6SECVIOLATION is
'Sum of regional 6 sec FCAS violations';

comment on column P5MIN_CASESOLUTION.TOTAL60SECVIOLATION is
'Sum of regional 60 sec FCAS violations';

comment on column P5MIN_CASESOLUTION.TOTALENERGYCONSTRVIOLATION is
'Sum of unit energy constrained violations';

comment on column P5MIN_CASESOLUTION.TOTALENERGYOFFERVIOLATION is
'Sum of unit offer violations';

comment on column P5MIN_CASESOLUTION.TOTALASPROFILEVIOLATION is
'Sum of unit FCAS profile offer violations';

comment on column P5MIN_CASESOLUTION.TOTALFASTSTARTVIOLATION is
'Sum of unit Fast start profile violations';

comment on column P5MIN_CASESOLUTION.LASTCHANGED is
'Last changed date and time of this record';

comment on column P5MIN_CASESOLUTION.INTERVENTION is
'Flag to indicate if this Predispatch case includes an intervention pricing run: 0 = case does not include an intervention pricing run, 1 = case does include an intervention pricing run. This field has a default value of 0 and is not nullable';

alter table P5MIN_CASESOLUTION
   add constraint P5MIN_CASESOLUTION_PK primary key (RUN_DATETIME)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: P5MIN_CASESOLUTION_LCX                                */
/*==============================================================*/
create index P5MIN_CASESOLUTION_LCX on P5MIN_CASESOLUTION (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: P5MIN_CONSTRAINTSOLUTION                              */
/*==============================================================*/
create table P5MIN_CONSTRAINTSOLUTION (
   RUN_DATETIME         DATE                  not null,
   INTERVAL_DATETIME    DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   RHS                  NUMBER(15,5),
   MARGINALVALUE        NUMBER(15,5),
   VIOLATIONDEGREE      NUMBER(15,5),
   LASTCHANGED          DATE,
   DUID                 VARCHAR2(20),
   GENCONID_EFFECTIVEDATE DATE,
   GENCONID_VERSIONNO   NUMBER(22,0),
   LHS                  number(15,5),
   INTERVENTION         Number(2,0)
)
   tablespace PSSTAB03;

comment on table P5MIN_CONSTRAINTSOLUTION is
'The Five-Minute Pre-Dispatch (P5Min) is a MMS system providing projected dispatch for 12 Dispatch cycles (one hour). The Five-Minute Pre-dispatch cycle runs every 5-minutes to produce a dispatch and pricing schedule to a 5-minute resolution covering the next hour, a total of twelve periods.
P5MIN_CONSTRAINTSOLUTION shows binding and violated constraint results from the capacity evaluation, including the RHS value.';

comment on column P5MIN_CONSTRAINTSOLUTION.RUN_DATETIME is
'Unique Timestamp Identifier for this study';

comment on column P5MIN_CONSTRAINTSOLUTION.INTERVAL_DATETIME is
'The unique identifier for the interval within this study';

comment on column P5MIN_CONSTRAINTSOLUTION.CONSTRAINTID is
'Constraint identifier (synonymous with GenConID)';

comment on column P5MIN_CONSTRAINTSOLUTION.RHS is
'Right Hand Side value in the capacity evaluation';

comment on column P5MIN_CONSTRAINTSOLUTION.MARGINALVALUE is
'Marginal cost of constraint (>0 if binding)';

comment on column P5MIN_CONSTRAINTSOLUTION.VIOLATIONDEGREE is
'Amount of Violation (>0 if  violating)';

comment on column P5MIN_CONSTRAINTSOLUTION.LASTCHANGED is
'Last date and time record changed';

comment on column P5MIN_CONSTRAINTSOLUTION.DUID is
'DUID to which the Constraint is confidential. Null denotes non-confidential';

comment on column P5MIN_CONSTRAINTSOLUTION.GENCONID_EFFECTIVEDATE is
'Effective date of the Generic Constraint (ConstraintID). This field is used to track the version of this generic constraint applied in this dispatch interval';

comment on column P5MIN_CONSTRAINTSOLUTION.GENCONID_VERSIONNO is
'Version number of the Generic Constraint (ConstraintID). This field is used to track the version of this generic constraint applied in this dispatch interval';

comment on column P5MIN_CONSTRAINTSOLUTION.LHS is
'Aggregation of the constraints LHS term solution values';

comment on column P5MIN_CONSTRAINTSOLUTION.INTERVENTION is
'Flag to indicate if this result set was sourced from the pricing run (INTERVENTION=0) or the physical run(INTERVENTION=1). In the event there is not intervention in the market, both pricing and physical runs correspond to INTERVENTION=0)';

alter table P5MIN_CONSTRAINTSOLUTION
   add constraint P5MIN_CONSTRAINTSOLUTION_PK primary key (RUN_DATETIME, CONSTRAINTID, INTERVAL_DATETIME)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Index: P5MIN_CONSTRAINTSOLUTION_LCX                          */
/*==============================================================*/
create index P5MIN_CONSTRAINTSOLUTION_LCX on P5MIN_CONSTRAINTSOLUTION (
   LASTCHANGED ASC
)
tablespace PSSIND03;

/*==============================================================*/
/* Table: P5MIN_FCAS_REQUIREMENT                                */
/*==============================================================*/
create table P5MIN_FCAS_REQUIREMENT (
   RUN_DATETIME         DATE                  not null,
   INTERVAL_DATETIME    DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   BIDTYPE              VARCHAR2(10)          not null,
   INTERVENTION         NUMBER(2,0),
   CONSTRAINT_EFFECTIVEDATE DATE,
   CONSTRAINT_VERSIONNO NUMBER(3,0),
   MARGINALVALUE        NUMBER(18,8),
   BASE_COST            NUMBER(18,8),
   ADJUSTED_COST        NUMBER(18,8),
   ESTIMATED_CMPF       NUMBER(18,8),
   ESTIMATED_CRMPF      NUMBER(18,8),
   RECOVERY_FACTOR_CMPF NUMBER(18,8),
   RECOVERY_FACTOR_CRMPF NUMBER(18,8),
   LASTCHANGED          DATE
)
   tablespace PSSTAB03;

comment on table P5MIN_FCAS_REQUIREMENT is
'5-minute Predispatch constraint tracking for Regional FCAS recovery';

comment on column P5MIN_FCAS_REQUIREMENT.RUN_DATETIME is
'First interval of the 5-minute Predispatch case';

comment on column P5MIN_FCAS_REQUIREMENT.INTERVAL_DATETIME is
'Datetime of the 5-minute Predispatch interval';

comment on column P5MIN_FCAS_REQUIREMENT.CONSTRAINTID is
'ConstraintID Join to table GenConData';

comment on column P5MIN_FCAS_REQUIREMENT.REGIONID is
'Region Identifier';

comment on column P5MIN_FCAS_REQUIREMENT.BIDTYPE is
'DUID offered type';

comment on column P5MIN_FCAS_REQUIREMENT.INTERVENTION is
'Intervention flag';

comment on column P5MIN_FCAS_REQUIREMENT.CONSTRAINT_EFFECTIVEDATE is
'Constraint EffectiveDate Join to table GenConData';

comment on column P5MIN_FCAS_REQUIREMENT.CONSTRAINT_VERSIONNO is
'Constraint Version number Join to table GenConData';

comment on column P5MIN_FCAS_REQUIREMENT.MARGINALVALUE is
'Marginal $ value for energy';

comment on column P5MIN_FCAS_REQUIREMENT.BASE_COST is
'The base cost of the constraint for this service, before the regulation/contingency split';

comment on column P5MIN_FCAS_REQUIREMENT.ADJUSTED_COST is
'The adjusted cost of the constraint for this service, after the regulation/contingency split';

comment on column P5MIN_FCAS_REQUIREMENT.ESTIMATED_CMPF is
'An estimated value for the constraint CMPF, based on 5- minute Predispatch data';

comment on column P5MIN_FCAS_REQUIREMENT.ESTIMATED_CRMPF is
'An estimated value for the constraint CRMPF, based on 5-minute Predispatch data';

comment on column P5MIN_FCAS_REQUIREMENT.RECOVERY_FACTOR_CMPF is
'Estimated recovery factor for CMPF based recovery';

comment on column P5MIN_FCAS_REQUIREMENT.RECOVERY_FACTOR_CRMPF is
'Estimated recovery for CRMPF based recovery';

comment on column P5MIN_FCAS_REQUIREMENT.LASTCHANGED is
'Last changed date for the record';

alter table P5MIN_FCAS_REQUIREMENT
   add constraint P5MIN_FCAS_REQUIREMENT_PK primary key (RUN_DATETIME, INTERVAL_DATETIME, CONSTRAINTID, REGIONID, BIDTYPE)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: P5MIN_FCAS_REQ_CONSTRAINT                             */
/*==============================================================*/
create table P5MIN_FCAS_REQ_CONSTRAINT (
   RUN_DATETIME         DATE                  not null,
   RUNNO                NUMBER(5)             not null,
   INTERVAL_DATETIME    DATE                  not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(20)          not null,
   BIDTYPE              VARCHAR2(10)          not null,
   LHS                  NUMBER(15,5),
   RHS                  NUMBER(15,5),
   MARGINALVALUE        NUMBER(15,5),
   RRP                  NUMBER(15,5),
   REGIONAL_ENABLEMENT  NUMBER(15,5),
   CONSTRAINT_ENABLEMENT NUMBER(15,5),
   REGION_BASE_COST     NUMBER(18,8),
   BASE_COST            NUMBER(18,8),
   ADJUSTED_COST        NUMBER(18,8),
   P_REGULATION         NUMBER(18,8)
)
   tablespace PSSTAB03;

comment on table P5MIN_FCAS_REQ_CONSTRAINT is
'The constraint level FCAS cost / price details for constraint FCAS processor runs. This enhanced output table format is established for the constraint FCAS processor release required for the Frequency Performance Payments (FPP) release. This enhanced output is hierarchical in nature, with the parent *_FCAS_REQ_RUN table holding the details about the triggering case run and time, and the child *_FCAS_REQ_CONSTRAINT table holding the constraint level details of FCAS costs / prices.';

comment on column P5MIN_FCAS_REQ_CONSTRAINT.RUN_DATETIME is
'The run date and time of the 5 minute predispatch case that triggers the constraint FCAS processor run';

comment on column P5MIN_FCAS_REQ_CONSTRAINT.RUNNO is
'The 5 minute predispatch case run number that has triggers the constraint FCAS processor run';

comment on column P5MIN_FCAS_REQ_CONSTRAINT.INTERVAL_DATETIME is
'The 5 minute interval date and time of the 5 minute predispatch interval that was processed by the constraint FCAS processor';

comment on column P5MIN_FCAS_REQ_CONSTRAINT.CONSTRAINTID is
'ConstraintID join to table GenConData';

comment on column P5MIN_FCAS_REQ_CONSTRAINT.REGIONID is
'Region identifier';

comment on column P5MIN_FCAS_REQ_CONSTRAINT.BIDTYPE is
'DUID offered type';

comment on column P5MIN_FCAS_REQ_CONSTRAINT.LHS is
'Constraints summed LHS - aggregate LHS Solution values from the physical run from the P5MIN_CONSTRAINTSOLUTION table';

comment on column P5MIN_FCAS_REQ_CONSTRAINT.RHS is
'Constraints RHS value used in the solution - may be either dynamic (calculated) or static from the physical run from the P5MIN_CONSTRAINTSOLUTION table';

comment on column P5MIN_FCAS_REQ_CONSTRAINT.MARGINALVALUE is
'Shadow price of constraint from the P5MIN_CONSTRAINTSOLUTION table from the physical run.';

comment on column P5MIN_FCAS_REQ_CONSTRAINT.RRP is
'Bid type prices for the region coming from the pricing run of the P5MIN_REGIONSOLUTION table';

comment on column P5MIN_FCAS_REQ_CONSTRAINT.REGIONAL_ENABLEMENT is
'The dispatched MW for the bid type inside the region from the physical run of the P5MIN_REGIONSOLUTION table';

comment on column P5MIN_FCAS_REQ_CONSTRAINT.CONSTRAINT_ENABLEMENT is
'MW enabled for this bid type within the constraint';

comment on column P5MIN_FCAS_REQ_CONSTRAINT.REGION_BASE_COST is
'The regional payment allocated to the constraint for the interval pro- rated based on marginal value ratios over the binding constraints for that service in that region (this is an intermediate calculation to get to the base cost)';

comment on column P5MIN_FCAS_REQ_CONSTRAINT.BASE_COST is
'The base cost of the constraint, before the regulation/contingency split';

comment on column P5MIN_FCAS_REQ_CONSTRAINT.ADJUSTED_COST is
'The adjusted cost of the constraint for this service, after the regulation/contingency split';

comment on column P5MIN_FCAS_REQ_CONSTRAINT.P_REGULATION is
'The adjusted marginal value of the constraint for FPP recovery (blank for constraints without REG terms)';

alter table P5MIN_FCAS_REQ_CONSTRAINT
   add constraint P5MIN_FCAS_REQ_CONSTRAINT_PK primary key (RUN_DATETIME, RUNNO, INTERVAL_DATETIME, CONSTRAINTID, REGIONID, BIDTYPE)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: P5MIN_FCAS_REQ_RUN                                    */
/*==============================================================*/
create table P5MIN_FCAS_REQ_RUN (
   RUN_DATETIME         DATE                  not null,
   RUNNO                NUMBER(5)             not null,
   LASTCHANGED          DATE
)
   tablespace PSSTAB02;

comment on table P5MIN_FCAS_REQ_RUN is
'The constraint FCAS processor run details. This enhanced output table format is established for the constraint FCAS processor release required for the Frequency Performance Payments (FPP) release. This enhanced output is hierarchical in nature, with the parent *_FCAS_REQ_RUN table holding the details about the triggering case run and time, and the child *_FCAS_REQ_CONSTRAINT table holding the constraint level details of FCAS costs / prices.';

comment on column P5MIN_FCAS_REQ_RUN.RUN_DATETIME is
'The run date and time of the 5 minute predispatch case that triggers the constraint FCAS processor run';

comment on column P5MIN_FCAS_REQ_RUN.RUNNO is
'The 5 minute predispatch case run number that has triggers the constraint FCAS processor run';

comment on column P5MIN_FCAS_REQ_RUN.LASTCHANGED is
'The last time the constraint FCAS processor was executed for this case run time.';

alter table P5MIN_FCAS_REQ_RUN
   add constraint P5MIN_FCAS_REQ_RUN_PK primary key (RUN_DATETIME, RUNNO)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: P5MIN_INTERCONNECTORSOLN                              */
/*==============================================================*/
create table P5MIN_INTERCONNECTORSOLN (
   RUN_DATETIME         DATE                  not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   INTERVAL_DATETIME    DATE                  not null,
   METEREDMWFLOW        NUMBER(15,5),
   MWFLOW               NUMBER(15,5),
   MWLOSSES             NUMBER(15,5),
   MARGINALVALUE        NUMBER(15,5),
   VIOLATIONDEGREE      NUMBER(15,5),
   MNSP                 NUMBER(1,0),
   EXPORTLIMIT          NUMBER(15,5),
   IMPORTLIMIT          NUMBER(15,5),
   MARGINALLOSS         NUMBER(15,5),
   EXPORTGENCONID       VARCHAR2(20),
   IMPORTGENCONID       VARCHAR2(20),
   FCASEXPORTLIMIT      NUMBER(15,5),
   FCASIMPORTLIMIT      NUMBER(15,5),
   LASTCHANGED          DATE,
   LOCAL_PRICE_ADJUSTMENT_EXPORT NUMBER(10,2),
   LOCALLY_CONSTRAINED_EXPORT NUMBER(1,0),
   LOCAL_PRICE_ADJUSTMENT_IMPORT NUMBER(10,2),
   LOCALLY_CONSTRAINED_IMPORT NUMBER(1,0),
   INTERVENTION         Number(2,0)
)
   tablespace PSSTAB03;

comment on table P5MIN_INTERCONNECTORSOLN is
'The five-minute predispatch (P5Min) is a MMS system providing projected dispatch for 12 Dispatch cycles (one hour). The 5-minute Predispatch cycle runs every 5-minutes to produce a dispatch and pricing schedule to a 5-minute resolution covering the next hour, a total of twelve periods.
P5MIN_INTERCONNECTORSOLN sets out the results of the capacity evaluation for Interconnectors, including the calculated limits for the interval.';

comment on column P5MIN_INTERCONNECTORSOLN.RUN_DATETIME is
'Unique Timestamp Identifier for this study';

comment on column P5MIN_INTERCONNECTORSOLN.INTERCONNECTORID is
'Interconnector identifier';

comment on column P5MIN_INTERCONNECTORSOLN.INTERVAL_DATETIME is
'The unique identifier for the interval within this study';

comment on column P5MIN_INTERCONNECTORSOLN.METEREDMWFLOW is
'SCADA MW Flow measured at Run start. For periods subsequent to the first period of a P5MIN run, this value represents the cleared target for the previous period of that P5MIN run.';

comment on column P5MIN_INTERCONNECTORSOLN.MWFLOW is
'Cleared Interconnector loading level (MW)';

comment on column P5MIN_INTERCONNECTORSOLN.MWLOSSES is
'Interconnector Losses at cleared flow';

comment on column P5MIN_INTERCONNECTORSOLN.MARGINALVALUE is
'Marginal cost of Interconnector standing data limits (if binding)';

comment on column P5MIN_INTERCONNECTORSOLN.VIOLATIONDEGREE is
'Violation of Interconnector standing data limits';

comment on column P5MIN_INTERCONNECTORSOLN.MNSP is
'Flag indicating MNSP registration';

comment on column P5MIN_INTERCONNECTORSOLN.EXPORTLIMIT is
'Calculated Interconnector limit of exporting energy on the basis of invoked constraints and static interconnector export limit';

comment on column P5MIN_INTERCONNECTORSOLN.IMPORTLIMIT is
'Calculated Interconnector limit of importing energy on the basis of invoked constraints and static interconnector import limit. Note unlike the input interconnector import limit this is a directional quantity and should be defined with respect to the interconnector flow.';

comment on column P5MIN_INTERCONNECTORSOLN.MARGINALLOSS is
'Marginal loss factor at the cleared flow';

comment on column P5MIN_INTERCONNECTORSOLN.EXPORTGENCONID is
'Generic Constraint setting the export limit';

comment on column P5MIN_INTERCONNECTORSOLN.IMPORTGENCONID is
'Generic Constraint setting the import limit';

comment on column P5MIN_INTERCONNECTORSOLN.FCASEXPORTLIMIT is
'Calculated export limit applying to energy + Frequency Controlled Ancillary Services.';

comment on column P5MIN_INTERCONNECTORSOLN.FCASIMPORTLIMIT is
'Calculated import limit applying to energy + Frequency Controlled Ancillary Services.';

comment on column P5MIN_INTERCONNECTORSOLN.LASTCHANGED is
'Last changed date of this record';

comment on column P5MIN_INTERCONNECTORSOLN.LOCAL_PRICE_ADJUSTMENT_EXPORT is
'Aggregate Constraint contribution cost of this Interconnector: Sum(MarginalValue x Factor) for all relevant Constraints, for Export (Factor >= 0)';

comment on column P5MIN_INTERCONNECTORSOLN.LOCALLY_CONSTRAINED_EXPORT is
'Key for Local_Price_Adjustment_Export: 2 = at least one Outage Constraint; 1 = at least 1 System Normal Constraint (and no Outage Constraint); 0 = No System Normal or Outage Constraints';

comment on column P5MIN_INTERCONNECTORSOLN.LOCAL_PRICE_ADJUSTMENT_IMPORT is
'Aggregate Constraint contribution cost of this Interconnector: Sum(MarginalValue x Factor) for all relevant Constraints, for Import (Factor >= 0)';

comment on column P5MIN_INTERCONNECTORSOLN.LOCALLY_CONSTRAINED_IMPORT is
'Key for Local_Price_Adjustment_Import: 2 = at least one Outage Constraint; 1 = at least 1 System Normal Constraint (and no Outage Constraint); 0 = No System Normal or Outage Constraints';

comment on column P5MIN_INTERCONNECTORSOLN.INTERVENTION is
'Flag to indicate if this result set was sourced from the pricing run (INTERVENTION=0) or the physical run (INTERVENTION=1). In the event there is not intervention in the market, both pricing and physical runs correspond to INTERVENTION=0)';

alter table P5MIN_INTERCONNECTORSOLN
   add constraint P5MIN_INTERCONNECTORSOLN_PK primary key (RUN_DATETIME, INTERCONNECTORID, INTERVAL_DATETIME)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Index: P5MIN_INTERCONNECTORSOLN_LCX                          */
/*==============================================================*/
create index P5MIN_INTERCONNECTORSOLN_LCX on P5MIN_INTERCONNECTORSOLN (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: P5MIN_INTERMITTENT_FCST_TRK                           */
/*==============================================================*/
create table P5MIN_INTERMITTENT_FCST_TRK (
   RUN_DATETIME         DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   FORECAST_RUN_DATETIME DATE,
   PROVIDERID           VARCHAR2(20),
   FORECAST_PRIORITY    NUMBER(10,0),
   OFFERDATETIME        DATE
)
   tablespace PSSTAB02;

comment on table P5MIN_INTERMITTENT_FCST_TRK is
'Uniquely tracks which Intermittent Generation forecast run (from INTERMITTENT_GEN_FCST_P5_RUN) was used for the DUID in which 5-Minute Pre-dispatch run.';

comment on column P5MIN_INTERMITTENT_FCST_TRK.RUN_DATETIME is
'Datetime (interval ending) of the 5-Minute Pre-dispatch run.';

comment on column P5MIN_INTERMITTENT_FCST_TRK.DUID is
'Dispatchable unit identifier, tracks to INTERMITTENT_GEN_FCST_P5_RUN.DUID.';

comment on column P5MIN_INTERMITTENT_FCST_TRK.FORECAST_RUN_DATETIME is
'Datetime (interval ending) when this forecast run is valid. It aligns with run_datetime, unless a forecast run is missed in which case the previous run is used. Tracks to INTERMITTENT_GEN_FCST_P5_RUN.FORECAST_RUN_DATETIME.';

comment on column P5MIN_INTERMITTENT_FCST_TRK.PROVIDERID is
'Provider of the forecast run used for the 5MPD run, tracks to INTERMITTENT_GEN_FCST_P5_RUN.PROVIDERID.';

comment on column P5MIN_INTERMITTENT_FCST_TRK.FORECAST_PRIORITY is
'Priority of the forecast run used for the 5MPD run, tracks to INTERMITTENT_GEN_FCST_P5_RUN.FORECAST_PRIORITY.';

comment on column P5MIN_INTERMITTENT_FCST_TRK.OFFERDATETIME is
'Submission datetime of the forecast run used for the 5MPD run, tracks to INTERMITTENT_GEN_FCST_P5_RUN.OFFERDATETIME.';

alter table P5MIN_INTERMITTENT_FCST_TRK
   add constraint P5MIN_INTERMITTENT_FCST_TRK_PK primary key (RUN_DATETIME, DUID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: P5MIN_INTERSENSITIVITIES                              */
/*==============================================================*/
create table P5MIN_INTERSENSITIVITIES (
   RUN_DATETIME         DATE                  not null,
   INTERCONNECTORID     VARCHAR2(20)          not null,
   INTERVAL_DATETIME    DATE                  not null,
   INTERVENTION         NUMBER(1,0)           not null,
   INTERVENTION_ACTIVE  NUMBER(1,0),
   MWFLOW1              NUMBER(15,5),
   MWFLOW2              NUMBER(15,5),
   MWFLOW3              NUMBER(15,5),
   MWFLOW4              NUMBER(15,5),
   MWFLOW5              NUMBER(15,5),
   MWFLOW6              NUMBER(15,5),
   MWFLOW7              NUMBER(15,5),
   MWFLOW8              NUMBER(15,5),
   MWFLOW9              NUMBER(15,5),
   MWFLOW10             NUMBER(15,5),
   MWFLOW11             NUMBER(15,5),
   MWFLOW12             NUMBER(15,5),
   MWFLOW13             NUMBER(15,5),
   MWFLOW14             NUMBER(15,5),
   MWFLOW15             NUMBER(15,5),
   MWFLOW16             NUMBER(15,5),
   MWFLOW17             NUMBER(15,5),
   MWFLOW18             NUMBER(15,5),
   MWFLOW19             NUMBER(15,5),
   MWFLOW20             NUMBER(15,5),
   MWFLOW21             NUMBER(15,5),
   MWFLOW22             NUMBER(15,5),
   MWFLOW23             NUMBER(15,5),
   MWFLOW24             NUMBER(15,5),
   MWFLOW25             NUMBER(15,5),
   MWFLOW26             NUMBER(15,5),
   MWFLOW27             NUMBER(15,5),
   MWFLOW28             NUMBER(15,5),
   MWFLOW29             NUMBER(15,5),
   MWFLOW30             NUMBER(15,5),
   MWFLOW31             NUMBER(15,5),
   MWFLOW32             NUMBER(15,5),
   MWFLOW33             NUMBER(15,5),
   MWFLOW34             NUMBER(15,5),
   MWFLOW35             NUMBER(15,5),
   MWFLOW36             NUMBER(15,5),
   MWFLOW37             NUMBER(15,5),
   MWFLOW38             NUMBER(15,5),
   MWFLOW39             NUMBER(15,5),
   MWFLOW40             NUMBER(15,5),
   MWFLOW41             NUMBER(15,5),
   MWFLOW42             NUMBER(15,5),
   MWFLOW43             NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB03;

comment on table P5MIN_INTERSENSITIVITIES is
'Price Sensitivies for 5MinPD solution. New solution every 5 minutes. Current Scenarios defined in P5MIN_SCENARIODEMANDTRK/P5MIN_SCENARIODEMAND';

comment on column P5MIN_INTERSENSITIVITIES.RUN_DATETIME is
'Definitive Run from which this solution derives';

comment on column P5MIN_INTERSENSITIVITIES.INTERCONNECTORID is
'Interconnector identifier';

comment on column P5MIN_INTERSENSITIVITIES.INTERVAL_DATETIME is
'The unique identifier for the interval within this study';

comment on column P5MIN_INTERSENSITIVITIES.INTERVENTION is
'Flag to indicate if this result set was sourced from the pricing run (INTERVENTION=0) or the physical run (INTERVENTION=1). In the event there is not intervention in the market, both pricing and physical runs correspond to INTERVENTION=0)';

comment on column P5MIN_INTERSENSITIVITIES.INTERVENTION_ACTIVE is
'Flag to indicate if the sensitivity run contains an active intervention constraint: 0 = No, 1 = Yes';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW1 is
'MW Flow value. Flow1 = MW flow for given Interconnector for Scenario 1';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW2 is
'MW Flow value. Flow2 = MW flow for given Interconnector for Scenario 2';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW3 is
'MW Flow value. Flow3 = MW flow for given Interconnector for Scenario 3';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW4 is
'MW Flow value. Flow4 = MW flow for given Interconnector for Scenario 4';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW5 is
'MW Flow value. Flow5 = MW flow for given Interconnector for Scenario 5';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW6 is
'MW Flow value. Flow6 = MW flow for given Interconnector for Scenario 6';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW7 is
'MW Flow value. Flow7 = MW flow for given Interconnector for Scenario 7';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW8 is
'MW Flow value. Flow8 = MW flow for given Interconnector for Scenario 8';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW9 is
'MW Flow value. Flow9 = MW flow for given Interconnector for Scenario 9';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW10 is
'MW Flow value. Flow10 = MW flow for given Interconnector for Scenario 10';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW11 is
'MW Flow value. Flow11 = MW flow for given Interconnector for Scenario 11';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW12 is
'MW Flow value. Flow12 = MW flow for given Interconnector for Scenario 12';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW13 is
'MW Flow value. Flow13 = MW flow for given Interconnector for Scenario 13';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW14 is
'MW Flow value. Flow14 = MW flow for given Interconnector for Scenario 14';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW15 is
'MW Flow value. Flow15 = MW flow for given Interconnector for Scenario 15';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW16 is
'MW Flow value. Flow16 = MW flow for given Interconnector for Scenario 16';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW17 is
'MW Flow value. Flow17 = MW flow for given Interconnector for Scenario 17';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW18 is
'MW Flow value. Flow18 = MW flow for given Interconnector for Scenario 18';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW19 is
'MW Flow value. Flow19 = MW flow for given Interconnector for Scenario 19';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW20 is
'MW Flow value. Flow20 = MW flow for given Interconnector for Scenario 20';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW21 is
'MW Flow value. Flow21 = MW flow for given Interconnector for Scenario 21';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW22 is
'MW Flow value. Flow22 = MW flow for given Interconnector for Scenario 22';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW23 is
'MW Flow value. Flow23 = MW flow for given Interconnector for Scenario 23';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW24 is
'MW Flow value. Flow24 = MW flow for given Interconnector for Scenario 24';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW25 is
'MW Flow value. Flow25 = MW flow for given Interconnector for Scenario 25';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW26 is
'MW Flow value. Flow26 = MW flow for given Interconnector for Scenario 26';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW27 is
'MW Flow value. Flow27 = MW flow for given Interconnector for Scenario 27';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW28 is
'MW Flow value. Flow28 = MW flow for given Interconnector for Scenario 28';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW29 is
'MW Flow value. Flow29 = MW flow for given Interconnector for Scenario 29';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW30 is
'MW Flow value. Flow30 = MW flow for given Interconnector for Scenario 30';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW31 is
'MW Flow value. Flow31 = MW flow for given Interconnector for Scenario 31';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW32 is
'MW Flow value. Flow32 = MW flow for given Interconnector for Scenario 32';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW33 is
'MW Flow value. Flow33 = MW flow for given Interconnector for Scenario 33';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW34 is
'MW Flow value. Flow34 = MW flow for given Interconnector for Scenario 34';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW35 is
'MW Flow value. Flow35 = MW flow for given Interconnector for Scenario 35';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW36 is
'MW Flow value. Flow36 = MW flow for given Interconnector for Scenario 36';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW37 is
'MW Flow value. Flow37 = MW flow for given Interconnector for Scenario 37';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW38 is
'MW Flow value. Flow38 = MW flow for given Interconnector for Scenario 38';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW39 is
'MW Flow value. Flow39 = MW flow for given Interconnector for Scenario 39';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW40 is
'MW Flow value. Flow40 = MW flow for given Interconnector for Scenario 40';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW41 is
'MW Flow value. Flow41 = MW flow for given Interconnector for Scenario 41';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW42 is
'MW Flow value. Flow42 = MW flow for given Interconnector for Scenario 42';

comment on column P5MIN_INTERSENSITIVITIES.MWFLOW43 is
'MW Flow value. Flow43 = MW flow for given Interconnector for Scenario 43';

comment on column P5MIN_INTERSENSITIVITIES.LASTCHANGED is
'Timestamp when this record was last modified';

alter table P5MIN_INTERSENSITIVITIES
   add constraint P5MIN_INTERSENSITIVITIES_PK primary key (RUN_DATETIME, INTERCONNECTORID, INTERVAL_DATETIME)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: P5MIN_LOCAL_PRICE                                     */
/*==============================================================*/
create table P5MIN_LOCAL_PRICE (
   RUN_DATETIME         DATE                  not null,
   INTERVAL_DATETIME    DATE                  not null,
   DUID                 VARCHAR2(20)          not null,
   LOCAL_PRICE_ADJUSTMENT NUMBER(10, 2),
   LOCALLY_CONSTRAINED  NUMBER(1,0)
)
   tablespace PSSTAB03;

comment on table P5MIN_LOCAL_PRICE is
'Sets out local pricing offsets associated with each DUID connection point for each dispatch period';

comment on column P5MIN_LOCAL_PRICE.RUN_DATETIME is
'Unique Timestamp Identifier for this study';

comment on column P5MIN_LOCAL_PRICE.INTERVAL_DATETIME is
'The unique identifier for the interval within this study';

comment on column P5MIN_LOCAL_PRICE.DUID is
'Dispatchable unit identifier';

comment on column P5MIN_LOCAL_PRICE.LOCAL_PRICE_ADJUSTMENT is
'Aggregate Constraint contribution cost of this unit: Sum(MarginalValue x Factor) for all relevant Constraints';

comment on column P5MIN_LOCAL_PRICE.LOCALLY_CONSTRAINED is
'Key for Local_Price_Adjustment: 2 = at least one Outage Constraint; 1 = at least 1 System Normal Constraint (and no Outage Constraint); 0 = No System Normal or Outage Constraints';

alter table P5MIN_LOCAL_PRICE
   add constraint P5MIN_LOCAL_PRICE_PK primary key (RUN_DATETIME, INTERVAL_DATETIME, DUID)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: P5MIN_PRICESENSITIVITIES                              */
/*==============================================================*/
create table P5MIN_PRICESENSITIVITIES (
   RUN_DATETIME         DATE                  not null,
   REGIONID             VARCHAR2(20)          not null,
   INTERVAL_DATETIME    DATE                  not null,
   INTERVENTION         NUMBER(1,0)           not null,
   INTERVENTION_ACTIVE  NUMBER(1,0),
   RRP1                 NUMBER(15,5),
   RRP2                 NUMBER(15,5),
   RRP3                 NUMBER(15,5),
   RRP4                 NUMBER(15,5),
   RRP5                 NUMBER(15,5),
   RRP6                 NUMBER(15,5),
   RRP7                 NUMBER(15,5),
   RRP8                 NUMBER(15,5),
   RRP9                 NUMBER(15,5),
   RRP10                NUMBER(15,5),
   RRP11                NUMBER(15,5),
   RRP12                NUMBER(15,5),
   RRP13                NUMBER(15,5),
   RRP14                NUMBER(15,5),
   RRP15                NUMBER(15,5),
   RRP16                NUMBER(15,5),
   RRP17                NUMBER(15,5),
   RRP18                NUMBER(15,5),
   RRP19                NUMBER(15,5),
   RRP20                NUMBER(15,5),
   RRP21                NUMBER(15,5),
   RRP22                NUMBER(15,5),
   RRP23                NUMBER(15,5),
   RRP24                NUMBER(15,5),
   RRP25                NUMBER(15,5),
   RRP26                NUMBER(15,5),
   RRP27                NUMBER(15,5),
   RRP28                NUMBER(15,5),
   RRP29                NUMBER(15,5),
   RRP30                NUMBER(15,5),
   RRP31                NUMBER(15,5),
   RRP32                NUMBER(15,5),
   RRP33                NUMBER(15,5),
   RRP34                NUMBER(15,5),
   RRP35                NUMBER(15,5),
   RRP36                NUMBER(15,5),
   RRP37                NUMBER(15,5),
   RRP38                NUMBER(15,5),
   RRP39                NUMBER(15,5),
   RRP40                NUMBER(15,5),
   RRP41                NUMBER(15,5),
   RRP42                NUMBER(15,5),
   RRP43                NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB03;

comment on table P5MIN_PRICESENSITIVITIES is
'Price Sensitivies for 5MinPD solution. New solution every 5 minutes. Current Scenarios defined in P5MIN_SCENARIODEMANDTRK/P5MIN_SCENARIODEMAND';

comment on column P5MIN_PRICESENSITIVITIES.RUN_DATETIME is
'Definitive Run from which this solution derives';

comment on column P5MIN_PRICESENSITIVITIES.REGIONID is
'Region';

comment on column P5MIN_PRICESENSITIVITIES.INTERVAL_DATETIME is
'The unique identifier for the interval within this study';

comment on column P5MIN_PRICESENSITIVITIES.INTERVENTION is
'Whether an Intervention constraint was defined in this run';

comment on column P5MIN_PRICESENSITIVITIES.INTERVENTION_ACTIVE is
'Flag to indicate if the sensitivity run contains an active intervention constraint: 0 = No, 1 = Yes';

comment on column P5MIN_PRICESENSITIVITIES.RRP1 is
'Regional Reference price for scenario 1';

comment on column P5MIN_PRICESENSITIVITIES.RRP2 is
'Regional Reference price for scenario 2';

comment on column P5MIN_PRICESENSITIVITIES.RRP3 is
'Regional Reference price for scenario 3';

comment on column P5MIN_PRICESENSITIVITIES.RRP4 is
'Regional Reference price for scenario 4';

comment on column P5MIN_PRICESENSITIVITIES.RRP5 is
'Regional Reference price for scenario 5';

comment on column P5MIN_PRICESENSITIVITIES.RRP6 is
'Regional Reference price for scenario 6';

comment on column P5MIN_PRICESENSITIVITIES.RRP7 is
'Regional Reference price for scenario 7';

comment on column P5MIN_PRICESENSITIVITIES.RRP8 is
'Regional Reference price for scenario 8';

comment on column P5MIN_PRICESENSITIVITIES.RRP9 is
'Regional Reference price for scenario 9';

comment on column P5MIN_PRICESENSITIVITIES.RRP10 is
'Regional Reference price for scenario 10';

comment on column P5MIN_PRICESENSITIVITIES.RRP11 is
'Regional Reference price for scenario 11';

comment on column P5MIN_PRICESENSITIVITIES.RRP12 is
'Regional Reference price for scenario 12';

comment on column P5MIN_PRICESENSITIVITIES.RRP13 is
'Regional Reference price for scenario 13';

comment on column P5MIN_PRICESENSITIVITIES.RRP14 is
'Regional Reference price for scenario 14';

comment on column P5MIN_PRICESENSITIVITIES.RRP15 is
'Regional Reference price for scenario 15';

comment on column P5MIN_PRICESENSITIVITIES.RRP16 is
'Regional Reference price for scenario 16';

comment on column P5MIN_PRICESENSITIVITIES.RRP17 is
'Regional Reference price for scenario 17';

comment on column P5MIN_PRICESENSITIVITIES.RRP18 is
'Regional Reference price for scenario 18';

comment on column P5MIN_PRICESENSITIVITIES.RRP19 is
'Regional Reference price for scenario 19';

comment on column P5MIN_PRICESENSITIVITIES.RRP20 is
'Regional Reference price for scenario 20';

comment on column P5MIN_PRICESENSITIVITIES.RRP21 is
'Regional Reference price for scenario 21';

comment on column P5MIN_PRICESENSITIVITIES.RRP22 is
'Regional Reference price for scenario 22';

comment on column P5MIN_PRICESENSITIVITIES.RRP23 is
'Regional Reference price for scenario 23';

comment on column P5MIN_PRICESENSITIVITIES.RRP24 is
'Regional Reference price for scenario 24';

comment on column P5MIN_PRICESENSITIVITIES.RRP25 is
'Regional Reference price for scenario 25';

comment on column P5MIN_PRICESENSITIVITIES.RRP26 is
'Regional Reference price for scenario 26';

comment on column P5MIN_PRICESENSITIVITIES.RRP27 is
'Regional Reference price for scenario 27';

comment on column P5MIN_PRICESENSITIVITIES.RRP28 is
'Regional Reference price for scenario 28';

comment on column P5MIN_PRICESENSITIVITIES.RRP29 is
'Regional Reference price for scenario 29';

comment on column P5MIN_PRICESENSITIVITIES.RRP30 is
'Regional Reference price for scenario 30';

comment on column P5MIN_PRICESENSITIVITIES.RRP31 is
'Regional Reference price for scenario 31';

comment on column P5MIN_PRICESENSITIVITIES.RRP32 is
'Regional Reference price for scenario 32';

comment on column P5MIN_PRICESENSITIVITIES.RRP33 is
'Regional Reference price for scenario 33';

comment on column P5MIN_PRICESENSITIVITIES.RRP34 is
'Regional Reference price for scenario 34';

comment on column P5MIN_PRICESENSITIVITIES.RRP35 is
'Regional Reference price for scenario 35';

comment on column P5MIN_PRICESENSITIVITIES.RRP36 is
'Regional Reference price for scenario 36';

comment on column P5MIN_PRICESENSITIVITIES.RRP37 is
'Regional Reference price for scenario 37';

comment on column P5MIN_PRICESENSITIVITIES.RRP38 is
'Regional Reference price for scenario 38';

comment on column P5MIN_PRICESENSITIVITIES.RRP39 is
'Regional Reference price for scenario 39';

comment on column P5MIN_PRICESENSITIVITIES.RRP40 is
'Regional Reference price for scenario 40';

comment on column P5MIN_PRICESENSITIVITIES.RRP41 is
'Regional Reference price for scenario 41';

comment on column P5MIN_PRICESENSITIVITIES.RRP42 is
'Regional Reference price for scenario 42';

comment on column P5MIN_PRICESENSITIVITIES.RRP43 is
'Regional Reference price for scenario 43';

comment on column P5MIN_PRICESENSITIVITIES.LASTCHANGED is
'Timestamp when this record was last modified';

alter table P5MIN_PRICESENSITIVITIES
   add constraint P5MIN_PRICESENSITIVITIES_PK primary key (RUN_DATETIME, REGIONID, INTERVAL_DATETIME)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Table: P5MIN_REGIONSOLUTION                                  */
/*==============================================================*/
create table P5MIN_REGIONSOLUTION (
   RUN_DATETIME         DATE                  not null,
   INTERVAL_DATETIME    DATE                  not null,
   REGIONID             VARCHAR2(10)          not null,
   RRP                  NUMBER(15,5),
   ROP                  NUMBER(15,5),
   EXCESSGENERATION     NUMBER(15,5),
   RAISE6SECRRP         NUMBER(15,5),
   RAISE6SECROP         NUMBER(15,5),
   RAISE60SECRRP        NUMBER(15,5),
   RAISE60SECROP        NUMBER(15,5),
   RAISE5MINRRP         NUMBER(15,5),
   RAISE5MINROP         NUMBER(15,5),
   RAISEREGRRP          NUMBER(15,5),
   RAISEREGROP          NUMBER(15,5),
   LOWER6SECRRP         NUMBER(15,5),
   LOWER6SECROP         NUMBER(15,5),
   LOWER60SECRRP        NUMBER(15,5),
   LOWER60SECROP        NUMBER(15,5),
   LOWER5MINRRP         NUMBER(15,5),
   LOWER5MINROP         NUMBER(15,5),
   LOWERREGRRP          NUMBER(15,5),
   LOWERREGROP          NUMBER(15,5),
   TOTALDEMAND          NUMBER(15,5),
   AVAILABLEGENERATION  NUMBER(15,5),
   AVAILABLELOAD        NUMBER(15,5),
   DEMANDFORECAST       NUMBER(15,5),
   DISPATCHABLEGENERATION NUMBER(15,5),
   DISPATCHABLELOAD     NUMBER(15,5),
   NETINTERCHANGE       NUMBER(15,5),
   LOWER5MINDISPATCH    NUMBER(15,5),
   LOWER5MINIMPORT      NUMBER(15,5),
   LOWER5MINLOCALDISPATCH NUMBER(15,5),
   LOWER5MINLOCALREQ    NUMBER(15,5),
   LOWER5MINREQ         NUMBER(15,5),
   LOWER60SECDISPATCH   NUMBER(15,5),
   LOWER60SECIMPORT     NUMBER(15,5),
   LOWER60SECLOCALDISPATCH NUMBER(15,5),
   LOWER60SECLOCALREQ   NUMBER(15,5),
   LOWER60SECREQ        NUMBER(15,5),
   LOWER6SECDISPATCH    NUMBER(15,5),
   LOWER6SECIMPORT      NUMBER(15,5),
   LOWER6SECLOCALDISPATCH NUMBER(15,5),
   LOWER6SECLOCALREQ    NUMBER(15,5),
   LOWER6SECREQ         NUMBER(15,5),
   RAISE5MINDISPATCH    NUMBER(15,5),
   RAISE5MINIMPORT      NUMBER(15,5),
   RAISE5MINLOCALDISPATCH NUMBER(15,5),
   RAISE5MINLOCALREQ    NUMBER(15,5),
   RAISE5MINREQ         NUMBER(15,5),
   RAISE60SECDISPATCH   NUMBER(15,5),
   RAISE60SECIMPORT     NUMBER(15,5),
   RAISE60SECLOCALDISPATCH NUMBER(15,5),
   RAISE60SECLOCALREQ   NUMBER(15,5),
   RAISE60SECREQ        NUMBER(15,5),
   RAISE6SECDISPATCH    NUMBER(15,5),
   RAISE6SECIMPORT      NUMBER(15,5),
   RAISE6SECLOCALDISPATCH NUMBER(15,5),
   RAISE6SECLOCALREQ    NUMBER(15,5),
   RAISE6SECREQ         NUMBER(15,5),
   AGGREGATEDISPATCHERROR NUMBER(15,5),
   INITIALSUPPLY        NUMBER(15,5),
   CLEAREDSUPPLY        NUMBER(15,5),
   LOWERREGIMPORT       NUMBER(15,5),
   LOWERREGDISPATCH     NUMBER(15,5),
   LOWERREGLOCALDISPATCH NUMBER(15,5),
   LOWERREGLOCALREQ     NUMBER(15,5),
   LOWERREGREQ          NUMBER(15,5),
   RAISEREGIMPORT       NUMBER(15,5),
   RAISEREGDISPATCH     NUMBER(15,5),
   RAISEREGLOCALDISPATCH NUMBER(15,5),
   RAISEREGLOCALREQ     NUMBER(15,5),
   RAISEREGREQ          NUMBER(15,5),
   RAISE5MINLOCALVIOLATION NUMBER(15,5),
   RAISEREGLOCALVIOLATION NUMBER(15,5),
   RAISE60SECLOCALVIOLATION NUMBER(15,5),
   RAISE6SECLOCALVIOLATION NUMBER(15,5),
   LOWER5MINLOCALVIOLATION NUMBER(15,5),
   LOWERREGLOCALVIOLATION NUMBER(15,5),
   LOWER60SECLOCALVIOLATION NUMBER(15,5),
   LOWER6SECLOCALVIOLATION NUMBER(15,5),
   RAISE5MINVIOLATION   NUMBER(15,5),
   RAISEREGVIOLATION    NUMBER(15,5),
   RAISE60SECVIOLATION  NUMBER(15,5),
   RAISE6SECVIOLATION   NUMBER(15,5),
   LOWER5MINVIOLATION   NUMBER(15,5),
   LOWERREGVIOLATION    NUMBER(15,5),
   LOWER60SECVIOLATION  NUMBER(15,5),
   LOWER6SECVIOLATION   NUMBER(15,5),
   LASTCHANGED          DATE,
   TOTALINTERMITTENTGENERATION NUMBER(15,5),
   DEMAND_AND_NONSCHEDGEN NUMBER(15,5),
   UIGF                 NUMBER(15,5),
   SEMISCHEDULE_CLEAREDMW NUMBER(15,5),
   SEMISCHEDULE_COMPLIANCEMW NUMBER(15,5),
   INTERVENTION         Number(2,0),
   SS_SOLAR_UIGF        Number(15,5),
   SS_WIND_UIGF         Number (15,5),
   SS_SOLAR_CLEAREDMW   Number(15,5),
   SS_WIND_CLEAREDMW    Number(15,5),
   SS_SOLAR_COMPLIANCEMW Number(15,5),
   SS_WIND_COMPLIANCEMW Number(15,5),
   WDR_INITIALMW        NUMBER(15,5),
   WDR_AVAILABLE        NUMBER(15,5),
   WDR_DISPATCHED       NUMBER(15,5),
   SS_SOLAR_AVAILABILITY NUMBER(15,5),
   SS_WIND_AVAILABILITY NUMBER(15,5),
   RAISE1SECRRP         NUMBER(15,5),
   RAISE1SECROP         NUMBER(15,5),
   LOWER1SECRRP         NUMBER(15,5),
   LOWER1SECROP         NUMBER(15,5),
   RAISE1SECLOCALDISPATCH NUMBER(15,5),
   LOWER1SECLOCALDISPATCH NUMBER(15,5),
   BDU_ENERGY_STORAGE   NUMBER(15,5),
   BDU_MIN_AVAIL        NUMBER(15,5),
   BDU_MAX_AVAIL        NUMBER(15,5),
   BDU_CLEAREDMW_GEN    NUMBER(15,5),
   BDU_CLEAREDMW_LOAD   NUMBER(15,5),
   BDU_INITIAL_ENERGY_STORAGE NUMBER(15,5),
   DECGEN_INITIAL_ENERGY_STORAGE NUMBER(15,5)
)
   tablespace PSSTAB03;

comment on table P5MIN_REGIONSOLUTION is
'The five-minute predispatch (P5Min) is a MMS system providing projected dispatch for 12 Dispatch cycles (one hour). The 5-minute Predispatch cycle runs every 5-minutes to produce a dispatch and pricing schedule to a 5-minute resolution covering the next hour, a total of twelve periods.
P5MIN_REGIONSOLUTION shows the results of the regional capacity, maximum surplus reserve and maximum spare capacity evaluations for each period of the study.';

comment on column P5MIN_REGIONSOLUTION.RUN_DATETIME is
'Unique Timestamp Identifier for this study';

comment on column P5MIN_REGIONSOLUTION.INTERVAL_DATETIME is
'The unique identifier for the interval within this study';

comment on column P5MIN_REGIONSOLUTION.REGIONID is
'Region Identifier';

comment on column P5MIN_REGIONSOLUTION.RRP is
'Region Reference Price (Energy)';

comment on column P5MIN_REGIONSOLUTION.ROP is
'Region Override Price (Energy)';

comment on column P5MIN_REGIONSOLUTION.EXCESSGENERATION is
'Total Energy Imbalance (MW)';

comment on column P5MIN_REGIONSOLUTION.RAISE6SECRRP is
'Region Reference Price (Raise6Sec)';

comment on column P5MIN_REGIONSOLUTION.RAISE6SECROP is
'Original regional price (Raise6Sec)';

comment on column P5MIN_REGIONSOLUTION.RAISE60SECRRP is
'Region Reference Price (Raise60Sec)';

comment on column P5MIN_REGIONSOLUTION.RAISE60SECROP is
'Original regional price (Raise60Sec)';

comment on column P5MIN_REGIONSOLUTION.RAISE5MINRRP is
'Region Reference Price (Raise5Min)';

comment on column P5MIN_REGIONSOLUTION.RAISE5MINROP is
'Original regional price (Raise5Min)';

comment on column P5MIN_REGIONSOLUTION.RAISEREGRRP is
'Region Reference Price (RaiseReg)';

comment on column P5MIN_REGIONSOLUTION.RAISEREGROP is
'Original regional price (RaiseReg)';

comment on column P5MIN_REGIONSOLUTION.LOWER6SECRRP is
'Region Reference Price (Lower6Sec)';

comment on column P5MIN_REGIONSOLUTION.LOWER6SECROP is
'Original regional price (Lower6Sec)';

comment on column P5MIN_REGIONSOLUTION.LOWER60SECRRP is
'Region Reference Price (Lower60Sec)';

comment on column P5MIN_REGIONSOLUTION.LOWER60SECROP is
'Original regional price (Lower60Sec)';

comment on column P5MIN_REGIONSOLUTION.LOWER5MINRRP is
'Region Reference Price (Lower5Min)';

comment on column P5MIN_REGIONSOLUTION.LOWER5MINROP is
'Original regional price (Lower5Min)';

comment on column P5MIN_REGIONSOLUTION.LOWERREGRRP is
'Region Reference Price (LowerReg)';

comment on column P5MIN_REGIONSOLUTION.LOWERREGROP is
'Original regional price (LowerReg)';

comment on column P5MIN_REGIONSOLUTION.TOTALDEMAND is
'Regional Demand - NB NOT net of Interconnector flows or Loads';

comment on column P5MIN_REGIONSOLUTION.AVAILABLEGENERATION is
'Regional Available generation';

comment on column P5MIN_REGIONSOLUTION.AVAILABLELOAD is
'Regional Available Load';

comment on column P5MIN_REGIONSOLUTION.DEMANDFORECAST is
'Predicted change in regional demand for this interval';

comment on column P5MIN_REGIONSOLUTION.DISPATCHABLEGENERATION is
'Regional Generation Dispatched';

comment on column P5MIN_REGIONSOLUTION.DISPATCHABLELOAD is
'Regional Load Dispatched';

comment on column P5MIN_REGIONSOLUTION.NETINTERCHANGE is
'Net interconnector Flows';

comment on column P5MIN_REGIONSOLUTION.LOWER5MINDISPATCH is
'Not used since Dec 2003. Lower 5 min MW dispatch';

comment on column P5MIN_REGIONSOLUTION.LOWER5MINIMPORT is
'Not used since Dec 2003. Lower 5 min MW imported';

comment on column P5MIN_REGIONSOLUTION.LOWER5MINLOCALDISPATCH is
'Lower 5 min local dispatch';

comment on column P5MIN_REGIONSOLUTION.LOWER5MINLOCALREQ is
'Not used since Dec 2003. Lower 5 min local requirement';

comment on column P5MIN_REGIONSOLUTION.LOWER5MINREQ is
'Not used since Dec 2003. Lower 5 min total requirement';

comment on column P5MIN_REGIONSOLUTION.LOWER60SECDISPATCH is
'Not used since Dec 2003. Lower 60 sec MW dispatch';

comment on column P5MIN_REGIONSOLUTION.LOWER60SECIMPORT is
'Not used since Dec 2003. Lower 60 sec MW imported';

comment on column P5MIN_REGIONSOLUTION.LOWER60SECLOCALDISPATCH is
'Lower 60 sec local dispatch';

comment on column P5MIN_REGIONSOLUTION.LOWER60SECLOCALREQ is
'Not used since Dec 2003. Lower 60 sec local requirement';

comment on column P5MIN_REGIONSOLUTION.LOWER60SECREQ is
'Not used since Dec 2003. Lower 60 sec total requirement';

comment on column P5MIN_REGIONSOLUTION.LOWER6SECDISPATCH is
'Not used since Dec 2003. Lower 6 sec MW dispatch';

comment on column P5MIN_REGIONSOLUTION.LOWER6SECIMPORT is
'Not used since Dec 2003. Lower 6 sec MW imported';

comment on column P5MIN_REGIONSOLUTION.LOWER6SECLOCALDISPATCH is
'Lower 6 sec local dispatch';

comment on column P5MIN_REGIONSOLUTION.LOWER6SECLOCALREQ is
'Not used since Dec 2003. Lower 6 sec local requirement';

comment on column P5MIN_REGIONSOLUTION.LOWER6SECREQ is
'Not used since Dec 2003. Lower 6 sec total requirement';

comment on column P5MIN_REGIONSOLUTION.RAISE5MINDISPATCH is
'Not used since Dec 2003. Total Raise 5 min MW dispatch';

comment on column P5MIN_REGIONSOLUTION.RAISE5MINIMPORT is
'Not used since Dec 2003. Raise 5 min MW imported';

comment on column P5MIN_REGIONSOLUTION.RAISE5MINLOCALDISPATCH is
'Raise 5 min local dispatch';

comment on column P5MIN_REGIONSOLUTION.RAISE5MINLOCALREQ is
'Not used since Dec 2003. Raise 5 min local requirement';

comment on column P5MIN_REGIONSOLUTION.RAISE5MINREQ is
'Not used since Dec 2003. Raise 5 min total requirement';

comment on column P5MIN_REGIONSOLUTION.RAISE60SECDISPATCH is
'Not used since Dec 2003. Raise 60 sec MW dispatch';

comment on column P5MIN_REGIONSOLUTION.RAISE60SECIMPORT is
'Not used since Dec 2003. Raise 60 sec MW imported';

comment on column P5MIN_REGIONSOLUTION.RAISE60SECLOCALDISPATCH is
'Raise 50 sec local dispatch';

comment on column P5MIN_REGIONSOLUTION.RAISE60SECLOCALREQ is
'Not used since Dec 2003. Raise 60 sec local requirement';

comment on column P5MIN_REGIONSOLUTION.RAISE60SECREQ is
'Not used since Dec 2003. Raise 60 sec total requirement';

comment on column P5MIN_REGIONSOLUTION.RAISE6SECDISPATCH is
'Not used since Dec 2003. Raise 6 sec MW dispatch';

comment on column P5MIN_REGIONSOLUTION.RAISE6SECIMPORT is
'Not used since Dec 2003. Raise 6 sec MW imported';

comment on column P5MIN_REGIONSOLUTION.RAISE6SECLOCALDISPATCH is
'Raise 6 sec local dispatch';

comment on column P5MIN_REGIONSOLUTION.RAISE6SECLOCALREQ is
'Not used since Dec 2003. Raise 6 sec local requirement';

comment on column P5MIN_REGIONSOLUTION.RAISE6SECREQ is
'Not used since Dec 2003. Raise 6 sec total requirement';

comment on column P5MIN_REGIONSOLUTION.AGGREGATEDISPATCHERROR is
'Aggregate dispatch error applied';

comment on column P5MIN_REGIONSOLUTION.INITIALSUPPLY is
'Sum of initial generation and import for region';

comment on column P5MIN_REGIONSOLUTION.CLEAREDSUPPLY is
'Sum of cleared generation and import for region';

comment on column P5MIN_REGIONSOLUTION.LOWERREGIMPORT is
'Not used since Dec 2003. Lower Regulation MW imported';

comment on column P5MIN_REGIONSOLUTION.LOWERREGDISPATCH is
'Not used since Dec 2003. Total Lower Regulation dispatch';

comment on column P5MIN_REGIONSOLUTION.LOWERREGLOCALDISPATCH is
'Lower Regulation local dispatch';

comment on column P5MIN_REGIONSOLUTION.LOWERREGLOCALREQ is
'Not used since Dec 2003. Lower Regulation local requirement';

comment on column P5MIN_REGIONSOLUTION.LOWERREGREQ is
'Not used since Dec 2003. Lower Regulation total requirement';

comment on column P5MIN_REGIONSOLUTION.RAISEREGIMPORT is
'Not used since Dec 2003. Raise Regulation MW imported';

comment on column P5MIN_REGIONSOLUTION.RAISEREGDISPATCH is
'Not used since Dec 2003. Total Raise Regulation dispatch';

comment on column P5MIN_REGIONSOLUTION.RAISEREGLOCALDISPATCH is
'Raise Regulation local dispatch';

comment on column P5MIN_REGIONSOLUTION.RAISEREGLOCALREQ is
'Not used since Dec 2003. Raise Regulation local requirement';

comment on column P5MIN_REGIONSOLUTION.RAISEREGREQ is
'Not used since Dec 2003. Raise Regulation total requirement';

comment on column P5MIN_REGIONSOLUTION.RAISE5MINLOCALVIOLATION is
'Not used since Dec 2003. Violation (MW) of Raise 5 min local requirement';

comment on column P5MIN_REGIONSOLUTION.RAISEREGLOCALVIOLATION is
'Not used since Dec 2003. Violation (MW) of Raise Reg local requirement';

comment on column P5MIN_REGIONSOLUTION.RAISE60SECLOCALVIOLATION is
'Not used since Dec 2003. Violation (MW) of Raise 60 sec local requirement';

comment on column P5MIN_REGIONSOLUTION.RAISE6SECLOCALVIOLATION is
'Not used since Dec 2003. Violation (MW) of Raise 6 sec local requirement';

comment on column P5MIN_REGIONSOLUTION.LOWER5MINLOCALVIOLATION is
'Not used since Dec 2003. Violation (MW) of Lower 5 min local requirement';

comment on column P5MIN_REGIONSOLUTION.LOWERREGLOCALVIOLATION is
'Not used since Dec 2003. Violation (MW) of Lower Reg local requirement';

comment on column P5MIN_REGIONSOLUTION.LOWER60SECLOCALVIOLATION is
'Not used since Dec 2003. Violation (MW) of Lower 60 sec local requirement';

comment on column P5MIN_REGIONSOLUTION.LOWER6SECLOCALVIOLATION is
'Not used since Dec 2003. Violation (MW) of Lower 6 sec local requirement';

comment on column P5MIN_REGIONSOLUTION.RAISE5MINVIOLATION is
'Not used since Dec 2003. Violation (MW) of Raise 5 min requirement';

comment on column P5MIN_REGIONSOLUTION.RAISEREGVIOLATION is
'Not used since Dec 2003. Violation (MW) of Raise Reg requirement';

comment on column P5MIN_REGIONSOLUTION.RAISE60SECVIOLATION is
'Not used since Dec 2003. Violation (MW) of Raise 60 seconds requirement';

comment on column P5MIN_REGIONSOLUTION.RAISE6SECVIOLATION is
'Not used since Dec 2003. Violation (MW) of Raise 6 seconds requirement';

comment on column P5MIN_REGIONSOLUTION.LOWER5MINVIOLATION is
'Not used since Dec 2003. Violation (MW) of Lower 5 min requirement';

comment on column P5MIN_REGIONSOLUTION.LOWERREGVIOLATION is
'Not used since Dec 2003. Violation (MW) of Lower Reg requirement';

comment on column P5MIN_REGIONSOLUTION.LOWER60SECVIOLATION is
'Not used since Dec 2003. Violation (MW) of Lower 60 seconds requirement';

comment on column P5MIN_REGIONSOLUTION.LOWER6SECVIOLATION is
'Not used since Dec 2003. Violation (MW) of Lower 6 seconds requirement';

comment on column P5MIN_REGIONSOLUTION.LASTCHANGED is
'Last date and time record changed';

comment on column P5MIN_REGIONSOLUTION.TOTALINTERMITTENTGENERATION is
'Allowance made for non-scheduled generation in the demand forecast (MW).';

comment on column P5MIN_REGIONSOLUTION.DEMAND_AND_NONSCHEDGEN is
'Sum of Cleared Scheduled generation, imported generation (at the region boundary) and allowances made for non-scheduled generation (MW).';

comment on column P5MIN_REGIONSOLUTION.UIGF is
'Regional aggregated Unconstrained Intermittent Generation Forecast of Semi-scheduled generation (MW).';

comment on column P5MIN_REGIONSOLUTION.SEMISCHEDULE_CLEAREDMW is
'Regional aggregated Semi-Schedule generator Cleared MW';

comment on column P5MIN_REGIONSOLUTION.SEMISCHEDULE_COMPLIANCEMW is
'Regional aggregated Semi-Schedule generator Cleared MW where Semi-Dispatch cap is enforced';

comment on column P5MIN_REGIONSOLUTION.INTERVENTION is
'Flag to indicate if this result set was sourced from the pricing run (INTERVENTION=0) or the physical run (INTERVENTION=1). In the event there is not intervention in the market, both pricing and physical runs correspond to INTERVENTION=0';

comment on column P5MIN_REGIONSOLUTION.SS_SOLAR_UIGF is
'Regional aggregated Unconstrained Intermittent Generation Forecast of Semi-scheduled generation (MW) where the primary fuel source is solar';

comment on column P5MIN_REGIONSOLUTION.SS_WIND_UIGF is
'Regional aggregated Unconstrained Intermittent Generation Forecast of Semi-scheduled generation (MW) where the primary fuel source is wind';

comment on column P5MIN_REGIONSOLUTION.SS_SOLAR_CLEAREDMW is
'Regional aggregated Semi-Schedule generator Cleared MW where the primary fuel source is solar';

comment on column P5MIN_REGIONSOLUTION.SS_WIND_CLEAREDMW is
'Regional aggregated Semi-Schedule generator Cleared MW where the primary fuel source is wind';

comment on column P5MIN_REGIONSOLUTION.SS_SOLAR_COMPLIANCEMW is
'Regional aggregated Semi-Schedule generator Cleared MW where Semi-Dispatch cap is enforced and the primary fuel source is solar';

comment on column P5MIN_REGIONSOLUTION.SS_WIND_COMPLIANCEMW is
'Regional aggregated Semi-Schedule generator Cleared MW where Semi-Dispatch cap is enforced and the primary fuel source is wind';

comment on column P5MIN_REGIONSOLUTION.WDR_INITIALMW is
'Regional aggregated MW value at start of interval for Wholesale Demand Response (WDR) units';

comment on column P5MIN_REGIONSOLUTION.WDR_AVAILABLE is
'Regional aggregated available MW for Wholesale Demand Response (WDR) units';

comment on column P5MIN_REGIONSOLUTION.WDR_DISPATCHED is
'Regional aggregated dispatched MW for Wholesale Demand Response (WDR) units';

comment on column P5MIN_REGIONSOLUTION.SS_SOLAR_AVAILABILITY is
'For Semi-Scheduled units. Aggregate Energy Availability from Solar units in that region';

comment on column P5MIN_REGIONSOLUTION.SS_WIND_AVAILABILITY is
'For Semi-Scheduled units. Aggregate Energy Availability from Wind units in that region';

comment on column P5MIN_REGIONSOLUTION.RAISE1SECRRP is
'Regional Raise 1Sec Price - R1Price attribute after capping/flooring';

comment on column P5MIN_REGIONSOLUTION.RAISE1SECROP is
'Raise1Sec Regional Original Price - uncapped/unfloored and unscaled';

comment on column P5MIN_REGIONSOLUTION.LOWER1SECRRP is
'Regional Lower 1Sec Price - RegionSolution element L1Price attribute';

comment on column P5MIN_REGIONSOLUTION.LOWER1SECROP is
'Lower1Sec Regional Original Price - uncapped/unfloored and unscaled';

comment on column P5MIN_REGIONSOLUTION.RAISE1SECLOCALDISPATCH is
'Total Raise1Sec Dispatched in Region - RegionSolution element R1Dispatch attribute';

comment on column P5MIN_REGIONSOLUTION.LOWER1SECLOCALDISPATCH is
'Total Lower1Sec Dispatched in Region - RegionSolution element L1Dispatch attribute';

comment on column P5MIN_REGIONSOLUTION.BDU_ENERGY_STORAGE is
'Regional aggregated energy storage where the DUID type is BDU (MWh)';

comment on column P5MIN_REGIONSOLUTION.BDU_MIN_AVAIL is
'Total available load side BDU summated for region (MW)';

comment on column P5MIN_REGIONSOLUTION.BDU_MAX_AVAIL is
'Total available generation side BDU summated for region (MW)';

comment on column P5MIN_REGIONSOLUTION.BDU_CLEAREDMW_GEN is
'Regional aggregated cleared MW where the DUID type is BDU. Net of export (Generation)';

comment on column P5MIN_REGIONSOLUTION.BDU_CLEAREDMW_LOAD is
'Regional aggregated cleared MW where the DUID type is BDU. Net of import (Load)';

comment on column P5MIN_REGIONSOLUTION.BDU_INITIAL_ENERGY_STORAGE is
'Energy Storage for BDU at the start of the interval(MWh) - Region Aggregated';

comment on column P5MIN_REGIONSOLUTION.DECGEN_INITIAL_ENERGY_STORAGE is
'Energy storage for Daily Energy Constrained Scheduled Generating Units at the start of the interval(MWh) - Region Aggregated';

alter table P5MIN_REGIONSOLUTION
   add constraint P5MIN_REGIONSOLUTION_PK primary key (RUN_DATETIME, REGIONID, INTERVAL_DATETIME)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Index: P5MIN_REGIONSOLUTION_LCX                              */
/*==============================================================*/
create index P5MIN_REGIONSOLUTION_LCX on P5MIN_REGIONSOLUTION (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: P5MIN_ROOFTOP_PV_FCST_TRK                             */
/*==============================================================*/
create table P5MIN_ROOFTOP_PV_FCST_TRK (
   RUN_DATETIME         DATE                  not null,
   AREAID               VARCHAR2(10)          not null,
   FORECAST_RUN_DATETIME DATE,
   PROVIDERID           VARCHAR2(20),
   FORECAST_PRIORITY    NUMBER(10,0),
   OFFERDATETIME        DATE
)
   tablespace PSSTAB02;

comment on table P5MIN_ROOFTOP_PV_FCST_TRK is
'Uniquely tracks which Rooftop PV forecast run (from ROOFTOP_PV_FCST_P5_RUN) was used for the Area in which 5-Minute Pre-dispatch run.';

comment on column P5MIN_ROOFTOP_PV_FCST_TRK.RUN_DATETIME is
'Datetime (interval ending) of the 5-Minute Pre-dispatch run.';

comment on column P5MIN_ROOFTOP_PV_FCST_TRK.AREAID is
'Area identifier aligning with the load forecasting areas, tracks to ROOFTOP_PV_FCST_P5_RUN.AREAID.';

comment on column P5MIN_ROOFTOP_PV_FCST_TRK.FORECAST_RUN_DATETIME is
'Datetime (interval ending) when this forecast run is valid. It aligns with run_datetime, unless a forecast run is missed, in this case the previous run is used. Tracks to ROOFTOP_PV_FCST_P5_RUN.FORECAST_RUN_DATETIME.';

comment on column P5MIN_ROOFTOP_PV_FCST_TRK.PROVIDERID is
'Provider identifier of the forecast run used for the DS run, tracks to ROOFTOP_PV_FCST_P5_RUN.PROVIDERID.';

comment on column P5MIN_ROOFTOP_PV_FCST_TRK.FORECAST_PRIORITY is
'Priority of the forecast run used for the DS run, tracks to ROOFTOP_PV_FCST_P5_RUN.FORECAST_PRIORITY.';

comment on column P5MIN_ROOFTOP_PV_FCST_TRK.OFFERDATETIME is
'Submission datetime of the forecast run used for the DS run, tracks to ROOFTOP_PV_FCST_P5_RUN.OFFERDATETIME.';

alter table P5MIN_ROOFTOP_PV_FCST_TRK
   add constraint P5MIN_ROOFTOP_PV_FCST_TRK_PK primary key (RUN_DATETIME, AREAID)
      using index tablespace PSSIND02;

/*==============================================================*/
/* Table: P5MIN_SCENARIODEMAND                                  */
/*==============================================================*/
create table P5MIN_SCENARIODEMAND (
   EFFECTIVEDATE        DATE                  not null,
   VERSION_DATETIME     DATE                  not null,
   SCENARIO             NUMBER(2,0)           not null,
   REGIONID             VARCHAR2(20)          not null,
   DELTAMW              NUMBER(4,0)
)
   tablespace PSSTAB01;

comment on table P5MIN_SCENARIODEMAND is
'The P5Min scenario MW offsets';

comment on column P5MIN_SCENARIODEMAND.EFFECTIVEDATE is
'The effective date of this set of scenarios';

comment on column P5MIN_SCENARIODEMAND.VERSION_DATETIME is
'The version of this set of scenarios';

comment on column P5MIN_SCENARIODEMAND.SCENARIO is
'The scenario identifier';

comment on column P5MIN_SCENARIODEMAND.REGIONID is
'The region to which to apply the deltaMW for this SCENARIO';

comment on column P5MIN_SCENARIODEMAND.DELTAMW is
'The MW offset to apply to region total demand for this SCENARIO';

alter table P5MIN_SCENARIODEMAND
   add constraint P5MIN_SCENARIODEMAND_PK primary key (EFFECTIVEDATE, VERSION_DATETIME, REGIONID, SCENARIO)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: P5MIN_SCENARIODEMANDTRK                               */
/*==============================================================*/
create table P5MIN_SCENARIODEMANDTRK (
   EFFECTIVEDATE        DATE                  not null,
   VERSION_DATETIME     DATE                  not null,
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table P5MIN_SCENARIODEMANDTRK is
'Tracks the 5Min scenario offset updates across time';

comment on column P5MIN_SCENARIODEMANDTRK.EFFECTIVEDATE is
'The effective date of this set of scenarios';

comment on column P5MIN_SCENARIODEMANDTRK.VERSION_DATETIME is
'The version of this set of scenarios';

comment on column P5MIN_SCENARIODEMANDTRK.AUTHORISEDDATE is
'The datetime that the scenario update was authorised';

comment on column P5MIN_SCENARIODEMANDTRK.LASTCHANGED is
'The datetime that the record was last changed';

alter table P5MIN_SCENARIODEMANDTRK
   add constraint P5MIN_SCENARIODEMANDTRK_PK primary key (EFFECTIVEDATE, VERSION_DATETIME)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Table: P5MIN_UNITSOLUTION                                    */
/*==============================================================*/
create table P5MIN_UNITSOLUTION (
   RUN_DATETIME         DATE                  not null,
   INTERVAL_DATETIME    DATE                  not null,
   DUID                 VARCHAR2(10)          not null,
   CONNECTIONPOINTID    VARCHAR2(12),
   TRADETYPE            NUMBER(2,0),
   AGCSTATUS            NUMBER(2,0),
   INITIALMW            NUMBER(15,5),
   TOTALCLEARED         NUMBER(15,5),
   RAMPDOWNRATE         NUMBER(15,5),
   RAMPUPRATE           NUMBER(15,5),
   LOWER5MIN            NUMBER(15,5),
   LOWER60SEC           NUMBER(15,5),
   LOWER6SEC            NUMBER(15,5),
   RAISE5MIN            NUMBER(15,5),
   RAISE60SEC           NUMBER(15,5),
   RAISE6SEC            NUMBER(15,5),
   LOWERREG             NUMBER(15,5),
   RAISEREG             NUMBER(15,5),
   AVAILABILITY         NUMBER(15,5),
   RAISE6SECFLAGS       NUMBER(3,0),
   RAISE60SECFLAGS      NUMBER(3,0),
   RAISE5MINFLAGS       NUMBER(3,0),
   RAISEREGFLAGS        NUMBER(3,0),
   LOWER6SECFLAGS       NUMBER(3,0),
   LOWER60SECFLAGS      NUMBER(3,0),
   LOWER5MINFLAGS       NUMBER(3,0),
   LOWERREGFLAGS        NUMBER(3,0),
   LASTCHANGED          DATE,
   SEMIDISPATCHCAP      NUMBER(3,0),
   INTERVENTION         Number(2,0),
   DISPATCHMODETIME     NUMBER(4,0),
   CONFORMANCE_MODE     NUMBER(6,0),
   UIGF                 NUMBER(15,5),
   RAISE1SEC            NUMBER(15,5),
   RAISE1SECFLAGS       NUMBER(3,0),
   LOWER1SEC            NUMBER(15,5),
   LOWER1SECFLAGS       NUMBER(3,0),
   INITIAL_ENERGY_STORAGE NUMBER(15,5),
   ENERGY_STORAGE       NUMBER(15,5),
   ENERGY_STORAGE_MIN   NUMBER(15,5),
   ENERGY_STORAGE_MAX   NUMBER(15,5),
   MIN_AVAILABILITY     NUMBER(15,5),
   ELEMENT_CAP          NUMBER(5,0)
)
   tablespace PSSTAB03;

comment on table P5MIN_UNITSOLUTION is
'The five-minute predispatch (P5Min) is a MMS system providing projected dispatch for 12 Dispatch cycles (one hour). The 5-minute Predispatch cycle runs every 5-minutes to produce a dispatch and pricing schedule to a 5-minute resolution covering the next hour, a total of twelve periods.
P5MIN_UNITSOLUTION shows the Unit results from the capacity evaluations for each period of the study.';

comment on column P5MIN_UNITSOLUTION.RUN_DATETIME is
'Unique Timestamp Identifier for this study';

comment on column P5MIN_UNITSOLUTION.INTERVAL_DATETIME is
'The unique identifier for the interval within this study';

comment on column P5MIN_UNITSOLUTION.DUID is
'Dispatchable unit identifier';

comment on column P5MIN_UNITSOLUTION.CONNECTIONPOINTID is
'Connection point identifier for DUID';

comment on column P5MIN_UNITSOLUTION.TRADETYPE is
'Generator or Load';

comment on column P5MIN_UNITSOLUTION.AGCSTATUS is
'AGC Status from EMS: 1 = on, 0 = off';

comment on column P5MIN_UNITSOLUTION.INITIALMW is
'Initial MW at start of period. For periods subsequent to the first period of a P5MIN run, this value represents the cleared target for the previous period of that P5MIN run. Negative values when Bi-directional Unit start from importing power, otherwise positive.';

comment on column P5MIN_UNITSOLUTION.TOTALCLEARED is
'Target MW for end of period. Negative values when Bi-directional Unit is importing power, otherwise positive.';

comment on column P5MIN_UNITSOLUTION.RAMPDOWNRATE is
'Ramp down rate (lessor of bid or telemetered rate).';

comment on column P5MIN_UNITSOLUTION.RAMPUPRATE is
'Ramp up rate (lessor of bid or telemetered rate).';

comment on column P5MIN_UNITSOLUTION.LOWER5MIN is
'Lower 5 min reserve target';

comment on column P5MIN_UNITSOLUTION.LOWER60SEC is
'Lower 60 sec reserve target';

comment on column P5MIN_UNITSOLUTION.LOWER6SEC is
'Lower 6 sec reserve target';

comment on column P5MIN_UNITSOLUTION.RAISE5MIN is
'Raise 5 min reserve target';

comment on column P5MIN_UNITSOLUTION.RAISE60SEC is
'Raise 60 sec reserve target';

comment on column P5MIN_UNITSOLUTION.RAISE6SEC is
'Raise 6 sec reserve target';

comment on column P5MIN_UNITSOLUTION.LOWERREG is
'Lower Regulation reserve target';

comment on column P5MIN_UNITSOLUTION.RAISEREG is
'Raise Regulation reserve target';

comment on column P5MIN_UNITSOLUTION.AVAILABILITY is
'For Scheduled units, this is the MAXAVAIL bid availability For Semi-scheduled units, this is the lower of MAXAVAIL bid availability and UIGF';

comment on column P5MIN_UNITSOLUTION.RAISE6SECFLAGS is
'Raise 6sec status flag';

comment on column P5MIN_UNITSOLUTION.RAISE60SECFLAGS is
'Raise 60sec status flag';

comment on column P5MIN_UNITSOLUTION.RAISE5MINFLAGS is
'Raise 5min status flag';

comment on column P5MIN_UNITSOLUTION.RAISEREGFLAGS is
'Raise Reg status flag';

comment on column P5MIN_UNITSOLUTION.LOWER6SECFLAGS is
'Lower 6sec status flag';

comment on column P5MIN_UNITSOLUTION.LOWER60SECFLAGS is
'Lower 60sec status flag';

comment on column P5MIN_UNITSOLUTION.LOWER5MINFLAGS is
'Lower 5min status flag';

comment on column P5MIN_UNITSOLUTION.LOWERREGFLAGS is
'Lower Reg status flag';

comment on column P5MIN_UNITSOLUTION.LASTCHANGED is
'Last date and time record changed';

comment on column P5MIN_UNITSOLUTION.SEMIDISPATCHCAP is
'Boolean representation flagging if the Target is Capped';

comment on column P5MIN_UNITSOLUTION.INTERVENTION is
'Flag to indicate if this result set was sourced from the pricing run (INTERVENTION=0) or the physical run(INTERVENTION=1). In the event there is not intervention in the market, both pricing and physical runs correspond to INTERVENTION=0';

comment on column P5MIN_UNITSOLUTION.DISPATCHMODETIME is
'Minutes for which the unit has been in the current DISPATCHMODE. From NEMDE TRADERSOLUTION element FSTARGETMODETIME attribute.';

comment on column P5MIN_UNITSOLUTION.CONFORMANCE_MODE is
'Mode specific to units within an aggregate. 0 - no monitoring, 1 - aggregate monitoring, 2 - individual monitoring due to constraint';

comment on column P5MIN_UNITSOLUTION.UIGF is
'For Semi-Scheduled units. Unconstrained Intermittent Generation Forecast value provided to NEMDE';

comment on column P5MIN_UNITSOLUTION.RAISE1SEC is
'Dispatched Raise1Sec - TraderSolution element R1Target attribute';

comment on column P5MIN_UNITSOLUTION.RAISE1SECFLAGS is
'TraderSolution element R1Flags attribute';

comment on column P5MIN_UNITSOLUTION.LOWER1SEC is
'Dispatched Lower1Sec - TraderSolution element L1Target attribute';

comment on column P5MIN_UNITSOLUTION.LOWER1SECFLAGS is
'TraderSolution element L1Flags attribute';

comment on column P5MIN_UNITSOLUTION.INITIAL_ENERGY_STORAGE is
'The energy storage at the start of this dispatch interval(MWh)';

comment on column P5MIN_UNITSOLUTION.ENERGY_STORAGE is
'The projected energy storage based on cleared energy and regulation FCAS dispatch(MWh)';

comment on column P5MIN_UNITSOLUTION.ENERGY_STORAGE_MIN is
'BDU only - Minimum Energy Storage constraint limit (MWh)';

comment on column P5MIN_UNITSOLUTION.ENERGY_STORAGE_MAX is
'BDU only - Maximum Energy Storage constraint limit (MWh)';

comment on column P5MIN_UNITSOLUTION.MIN_AVAILABILITY is
'BDU only. Load side availability (BidOfferPeriod.MAXAVAIL where DIRECTION = LOAD).';

comment on column P5MIN_UNITSOLUTION.ELEMENT_CAP is
'Cap on the number of turbines or inverters at a DUID.';

alter table P5MIN_UNITSOLUTION
   add constraint P5MIN_UNITSOLUTION_PK primary key (RUN_DATETIME, DUID, INTERVAL_DATETIME)
      using index tablespace PSSIND03;

/*==============================================================*/
/* Index: P5MIN_UNITSOLUTION_LCX                                */
/*==============================================================*/
create index P5MIN_UNITSOLUTION_LCX on P5MIN_UNITSOLUTION (
   LASTCHANGED ASC
)
tablespace PSSIND02;

/*==============================================================*/
/* Table: PARTICIPANT                                           */
/*==============================================================*/
create table PARTICIPANT (
   PARTICIPANTID        VARCHAR2(10)          not null,
   PARTICIPANTCLASSID   VARCHAR2(20),
   NAME                 VARCHAR2(80),
   DESCRIPTION          VARCHAR2(64),
   ACN                  VARCHAR2(9),
   PRIMARYBUSINESS      VARCHAR2(40),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table PARTICIPANT is
'PARTICIPANT sets out Participant ID, name and class for all participants.';

comment on column PARTICIPANT.PARTICIPANTID is
'Unique participant identifier';

comment on column PARTICIPANT.PARTICIPANTCLASSID is
'Class of participant';

comment on column PARTICIPANT.NAME is
'Full name of participant';

comment on column PARTICIPANT.DESCRIPTION is
'Not used';

comment on column PARTICIPANT.ACN is
'Australian Company Number; Nine Numbers XXX-XXX-XXX';

comment on column PARTICIPANT.PRIMARYBUSINESS is
'Identifies primary business activity of participant';

comment on column PARTICIPANT.LASTCHANGED is
'Last date and time record changed';

alter table PARTICIPANT
   add constraint PARTICIPANT_PK primary key (PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: PARTICIPANT_LCX                                       */
/*==============================================================*/
create index PARTICIPANT_LCX on PARTICIPANT (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: PARTICIPANTACCOUNT                                    */
/*==============================================================*/
create table PARTICIPANTACCOUNT (
   ACCOUNTNAME          VARCHAR2(80),
   PARTICIPANTID        VARCHAR2(10)          not null,
   ACCOUNTNUMBER        VARCHAR2(16),
   BANKNAME             VARCHAR2(16),
   BANKNUMBER           NUMBER(10,0),
   BRANCHNAME           VARCHAR2(16),
   BRANCHNUMBER         NUMBER(10,0),
   BSBNUMBER            VARCHAR2(20),
   NEMMCOCREDITACCOUNTNUMBER NUMBER(10,0),
   NEMMCODEBITACCOUNTNUMBER NUMBER(10,0),
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   EFFECTIVEDATE        DATE,
   LASTCHANGED          DATE,
   ABN                  VARCHAR2(20)
)
   tablespace PSSTAB01;

comment on table PARTICIPANTACCOUNT is
'PARTICIPANTACCOUNT shows financial details on participants.';

comment on column PARTICIPANTACCOUNT.ACCOUNTNAME is
'Name of the account';

comment on column PARTICIPANTACCOUNT.PARTICIPANTID is
'Unique participant identifier';

comment on column PARTICIPANTACCOUNT.ACCOUNTNUMBER is
'Account number';

comment on column PARTICIPANTACCOUNT.BANKNAME is
'Bank name';

comment on column PARTICIPANTACCOUNT.BANKNUMBER is
'Bank number';

comment on column PARTICIPANTACCOUNT.BRANCHNAME is
'Branch name';

comment on column PARTICIPANTACCOUNT.BRANCHNUMBER is
'Branch number';

comment on column PARTICIPANTACCOUNT.BSBNUMBER is
'BSB number';

comment on column PARTICIPANTACCOUNT.NEMMCOCREDITACCOUNTNUMBER is
'AEMO credit account number';

comment on column PARTICIPANTACCOUNT.NEMMCODEBITACCOUNTNUMBER is
'AEMO debit account number';

comment on column PARTICIPANTACCOUNT.AUTHORISEDBY is
'User authorising record';

comment on column PARTICIPANTACCOUNT.AUTHORISEDDATE is
'Authorised date';

comment on column PARTICIPANTACCOUNT.EFFECTIVEDATE is
'Date record authorised';

comment on column PARTICIPANTACCOUNT.LASTCHANGED is
'Last date and time record changed';

comment on column PARTICIPANTACCOUNT.ABN is
'Australian Business Number';

alter table PARTICIPANTACCOUNT
   add constraint PARTICIPANTACCOUNT_PK primary key (PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: PARTICIPANTACCOUNT_LCX                                */
/*==============================================================*/
create index PARTICIPANTACCOUNT_LCX on PARTICIPANTACCOUNT (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: PARTICIPANTCATEGORY                                   */
/*==============================================================*/
create table PARTICIPANTCATEGORY (
   PARTICIPANTCATEGORYID VARCHAR2(10)          not null,
   DESCRIPTION          VARCHAR2(64),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table PARTICIPANTCATEGORY is
'PARTICIPANTCATEGORY sets out valid participant categories.';

comment on column PARTICIPANTCATEGORY.PARTICIPANTCATEGORYID is
'Participant category identifier';

comment on column PARTICIPANTCATEGORY.DESCRIPTION is
'Category description';

comment on column PARTICIPANTCATEGORY.LASTCHANGED is
'Last date and time record changed';

alter table PARTICIPANTCATEGORY
   add constraint PARTICIPANTCATEGORY_PK primary key (PARTICIPANTCATEGORYID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: PARTICIPANTCATEGORY_LCX                               */
/*==============================================================*/
create index PARTICIPANTCATEGORY_LCX on PARTICIPANTCATEGORY (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: PARTICIPANTCATEGORYALLOC                              */
/*==============================================================*/
create table PARTICIPANTCATEGORYALLOC (
   PARTICIPANTCATEGORYID VARCHAR2(10)          not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table PARTICIPANTCATEGORYALLOC is
'PARTICIPANTCATEGORYALLOC sets out the assignment of participants to particular categories.';

comment on column PARTICIPANTCATEGORYALLOC.PARTICIPANTCATEGORYID is
'Category unique identifier';

comment on column PARTICIPANTCATEGORYALLOC.PARTICIPANTID is
'Unique participant identifier';

comment on column PARTICIPANTCATEGORYALLOC.LASTCHANGED is
'Last date and time record changed';

alter table PARTICIPANTCATEGORYALLOC
   add constraint PARTICIPANTCATALLOC_PK primary key (PARTICIPANTCATEGORYID, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: PARTICIPANTCATEGORYALLOC_LCX                          */
/*==============================================================*/
create index PARTICIPANTCATEGORYALLOC_LCX on PARTICIPANTCATEGORYALLOC (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: PARTICIPANTCLASS                                      */
/*==============================================================*/
create table PARTICIPANTCLASS (
   PARTICIPANTCLASSID   VARCHAR2(20)          not null,
   DESCRIPTION          VARCHAR2(64),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table PARTICIPANTCLASS is
'PARTICIPANTCLASS sets out valid participant classifications.';

comment on column PARTICIPANTCLASS.PARTICIPANTCLASSID is
'Class of participant';

comment on column PARTICIPANTCLASS.DESCRIPTION is
'Description of participant class';

comment on column PARTICIPANTCLASS.LASTCHANGED is
'Last date and time record changed';

alter table PARTICIPANTCLASS
   add constraint PARTCLASS_PK primary key (PARTICIPANTCLASSID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: PARTICIPANTCLASS_LCX                                  */
/*==============================================================*/
create index PARTICIPANTCLASS_LCX on PARTICIPANTCLASS (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: PARTICIPANTCREDITDETAIL                               */
/*==============================================================*/
create table PARTICIPANTCREDITDETAIL (
   EFFECTIVEDATE        DATE                  not null,
   PARTICIPANTID        VARCHAR2(10)          not null,
   CREDITLIMIT          NUMBER(10,0),
   AUTHORISEDBY         VARCHAR2(15),
   AUTHORISEDDATE       DATE,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on column PARTICIPANTCREDITDETAIL.LASTCHANGED is
'Last date and time record changed';

alter table PARTICIPANTCREDITDETAIL
   add constraint PARTCREDDET_PK primary key (EFFECTIVEDATE, PARTICIPANTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: PARTICIPANTCREDITDETAIL_LCX                           */
/*==============================================================*/
create index PARTICIPANTCREDITDETAIL_LCX on PARTICIPANTCREDITDETAIL (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: PARTICIPANTCREDITDET_NDX2                             */
/*==============================================================*/
create index PARTICIPANTCREDITDET_NDX2 on PARTICIPANTCREDITDETAIL (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: PARTICIPANTNOTICETRK                                  */
/*==============================================================*/
create table PARTICIPANTNOTICETRK (
   PARTICIPANTID        VARCHAR2(10)          not null,
   NOTICEID             NUMBER(10,0)          not null,
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table PARTICIPANTNOTICETRK is
'PARTICIPANTNOTICETRK provides the cross-reference between participant market notices and participants.';

comment on column PARTICIPANTNOTICETRK.PARTICIPANTID is
'Unique participant identifier';

comment on column PARTICIPANTNOTICETRK.NOTICEID is
'Market notice identifier';

comment on column PARTICIPANTNOTICETRK.LASTCHANGED is
'Last date and time record changed';

alter table PARTICIPANTNOTICETRK
   add constraint PARTICIPANTNOTICETRK_PK primary key (PARTICIPANTID, NOTICEID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: PARTICIPANTNOTICETRK_NDX2                             */
/*==============================================================*/
create index PARTICIPANTNOTICETRK_NDX2 on PARTICIPANTNOTICETRK (
   PARTICIPANTID ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Index: PARTICIPANTNOTICETRK_LCX                              */
/*==============================================================*/
create index PARTICIPANTNOTICETRK_LCX on PARTICIPANTNOTICETRK (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: PARTICIPANT_BANDFEE_ALLOC                             */
/*==============================================================*/
create table PARTICIPANT_BANDFEE_ALLOC (
   PARTICIPANTID        VARCHAR2(10)          not null,
   MARKETFEEID          VARCHAR2(10)          not null,
   EFFECTIVEDATE        DATE                  not null,
   VERSIONNO            NUMBER(3,0)           not null,
   PARTICIPANTCATEGORYID VARCHAR2(10)          not null,
   MARKETFEEVALUE       NUMBER(15,5),
   LASTCHANGED          DATE
)
   tablespace PSSTAB01;

comment on table PARTICIPANT_BANDFEE_ALLOC is
'PARTICIPANT_BANDFEE_ALLOC shows the market fee for each Participant/Participant Category over time.';

comment on column PARTICIPANT_BANDFEE_ALLOC.PARTICIPANTID is
'Unique participant identifier';

comment on column PARTICIPANT_BANDFEE_ALLOC.MARKETFEEID is
'Identifier for Market Fee';

comment on column PARTICIPANT_BANDFEE_ALLOC.EFFECTIVEDATE is
'Date on which this data becomes effective.';

comment on column PARTICIPANT_BANDFEE_ALLOC.VERSIONNO is
'Period identifier';

comment on column PARTICIPANT_BANDFEE_ALLOC.PARTICIPANTCATEGORYID is
'The participant category that the market fee recovery amount pertains to.';

comment on column PARTICIPANT_BANDFEE_ALLOC.MARKETFEEVALUE is
'The value of this market fee';

comment on column PARTICIPANT_BANDFEE_ALLOC.LASTCHANGED is
'Last date and time record changed';

alter table PARTICIPANT_BANDFEE_ALLOC
   add constraint PARTICIPANT_BANDFEE_ALLOC_PK primary key (PARTICIPANTID, MARKETFEEID, EFFECTIVEDATE, VERSIONNO, PARTICIPANTCATEGORYID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: PARTICIPANT_BANDFEE_ALOC_LCX                          */
/*==============================================================*/
create index PARTICIPANT_BANDFEE_ALOC_LCX on PARTICIPANT_BANDFEE_ALLOC (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: PASACASESOLUTION                                      */
/*==============================================================*/
create table PASACASESOLUTION (
   CASEID               VARCHAR2(20)          not null,
   SOLUTIONCOMPLETE     NUMBER(16,6),
   PASAVERSION          NUMBER(27,10),
   EXCESSGENERATION     NUMBER(16,6),
   DEFICITCAPACITY      NUMBER(16,6),
   LASTCHANGED          DATE,
   DATETIME             DATE
)
   tablespace PSSTAB01;

comment on table PASACASESOLUTION is
'PASACASESOLUTION sets out ST PASA case listing providing details of each STPASA case run.';

comment on column PASACASESOLUTION.CASEID is
'PASA Case Identifier';

comment on column PASACASESOLUTION.SOLUTIONCOMPLETE is
'Not used';

comment on column PASACASESOLUTION.PASAVERSION is
'Software version identifier';

comment on column PASACASESOLUTION.EXCESSGENERATION is
'Excess generation detected flag';

comment on column PASACASESOLUTION.DEFICITCAPACITY is
'Deficit capacity detected flag';

comment on column PASACASESOLUTION.LASTCHANGED is
'Last date and time record changed';

comment on column PASACASESOLUTION.DATETIME is
'Scheduled date and time of the run';

alter table PASACASESOLUTION
   add constraint PASACASESOLUTION_PK primary key (CASEID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: PASACASESOLUTION_LCX                                  */
/*==============================================================*/
create index PASACASESOLUTION_LCX on PASACASESOLUTION (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: PASACONSTRAINTSOLUTION                                */
/*==============================================================*/
create table PASACONSTRAINTSOLUTION (
   CASEID               VARCHAR2(20)          not null,
   CONSTRAINTID         VARCHAR2(20)          not null,
   PERIODID             VARCHAR2(20)          not null,
   CAPACITYMARGINALVALUE NUMBER(16,6),
   CAPACITYVIOLATIONDEGREE NUMBER(16,6),
   EXCESSGENMARGINALVALUE NUMBER(16,6),
   EXCESSGENVIOLATIONDEGREE NUMBER(16,6),
   LASTCHANGED          DATE,
   DATETIME             DATE
)
   tablespace PSSTAB01;

comment on table PASACONSTRAINTSOLUTION is
'PASACONSTRAINTSOLUTION records the latest binding STPASA constraint details for each period. For each solution, the latest recalculation for each period overwrites the previous entry.';

comment on column PASACONSTRAINTSOLUTION.CASEID is
'PASA Case Identifier';

comment on column PASACONSTRAINTSOLUTION.CONSTRAINTID is
'Generic Constraint Id';

comment on column PASACONSTRAINTSOLUTION.PERIODID is
'PASA Interval (48 half hours)';

comment on column PASACONSTRAINTSOLUTION.CAPACITYMARGINALVALUE is
'Binding value of capacity and adequacy (if insufficient to measure)';

comment on column PASACONSTRAINTSOLUTION.CAPACITYVIOLATIONDEGREE is
'Deficit MW of surplus capacity';

comment on column PASACONSTRAINTSOLUTION.EXCESSGENMARGINALVALUE is
'Binding value of dispatch generator above aggregate self dispatch';

comment on column PASACONSTRAINTSOLUTION.EXCESSGENVIOLATIONDEGREE is
'Deficit of generator above aggregate self dispatch level';

comment on column PASACONSTRAINTSOLUTION.LASTCHANGED is
'Last date and time record changed';

comment on column PASACONSTRAINTSOLUTION.DATETIME is
'Date and time of the end of the period';

alter table PASACONSTRAINTSOLUTION
   add constraint PASACONSTRAINTSOLUTION_PK primary key (PERIODID, CONSTRAINTID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: PASACONSTRAINTSOLUTION_LCX                            */
/*==============================================================*/
create index PASACONSTRAINTSOLUTION_LCX on PASACONSTRAINTSOLUTION (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: PASAINTERCONNECTORSOLUTION                            */
/*==============================================================*/
create table PASAINTERCONNECTORSOLUTION (
   CASEID               VARCHAR2(20)          not null,
   INTERCONNECTORID     VARCHAR2(10)          not null,
   PERIODID             VARCHAR2(20)          not null,
   CAPACITYMWFLOW       NUMBER(16,6),
   CAPACITYMARGINALVALUE NUMBER(16,6),
   CAPACITYVIOLATIONDEGREE NUMBER(16,6),
   EXCESSGENMWFLOW      NUMBER(16,6),
   EXCESSGENMARGINALVALUE NUMBER(16,6),
   EXCESSGENVIOLATIONDEGREE NUMBER(16,6),
   LASTCHANGED          DATE,
   IMPORTLIMIT          NUMBER(15,5),
   EXPORTLIMIT          NUMBER(15,5),
   DATETIME             DATE
)
   tablespace PSSTAB01;

comment on table PASAINTERCONNECTORSOLUTION is
'PASAINTERCONNECTORSOLUTION records ST PASA interconnector solutions for the latest period.';

comment on column PASAINTERCONNECTORSOLUTION.CASEID is
'PASA Case Identifier';

comment on column PASAINTERCONNECTORSOLUTION.INTERCONNECTORID is
'Interconnector Id';

comment on column PASAINTERCONNECTORSOLUTION.PERIODID is
'PASA Interval (48 half hours)';

comment on column PASAINTERCONNECTORSOLUTION.CAPACITYMWFLOW is
'Capacity MW flow';

comment on column PASAINTERCONNECTORSOLUTION.CAPACITYMARGINALVALUE is
'Marginal value in capacity';

comment on column PASAINTERCONNECTORSOLUTION.CAPACITYVIOLATIONDEGREE is
'Violation value in capacity';

comment on column PASAINTERCONNECTORSOLUTION.EXCESSGENMWFLOW is
'Excess generation MW flow';

comment on column PASAINTERCONNECTORSOLUTION.EXCESSGENMARGINALVALUE is
'Marginal value in excess generation';

comment on column PASAINTERCONNECTORSOLUTION.EXCESSGENVIOLATIONDEGREE is
'Violation value in excess generation';

comment on column PASAINTERCONNECTORSOLUTION.LASTCHANGED is
'Last date and time record changed';

comment on column PASAINTERCONNECTORSOLUTION.IMPORTLIMIT is
'Calculated import limit';

comment on column PASAINTERCONNECTORSOLUTION.EXPORTLIMIT is
'Calculated export limit';

comment on column PASAINTERCONNECTORSOLUTION.DATETIME is
'Date and time of the end of the period';

alter table PASAINTERCONNECTORSOLUTION
   add constraint PASAINTERCONNECTORSOLUTION_PK primary key (PERIODID, INTERCONNECTORID)
      using index tablespace PSSIND01;

/*==============================================================*/
/* Index: PASAINTERCONNECTORSOLUTIO_LCX                         */
/*==============================================================*/
create index PASAINTERCONNECTORSOLUTIO_LCX on PASAINTERCONNECTORSOLUTION (
   LASTCHANGED ASC
)
tablespace PSSIND01;

/*==============================================================*/
/* Table: PASAREGIONSOLUTION                                    */
/*==============================================================*/
create table PASAREGIONSOLUTION (
   CASEID               VARCHAR2(20)          not null,
   REGIONID             VARCHAR2(10)          not null,
   PERIODID             VARCHAR2(20)          not null,
   DEMAND10             NUMBER(16,6),
   DEMAND50             NUMBER(16,6),
   DEMAND90             NUMBER(16,6),
   UNCONSTRAINEDCAPACITY NUMBER(16,6),
   CONSTRAINEDCAPACITY  NUMBER(16,6),
   CAPACITYSURPLUS      NUMBER(16,6),
   RESERVEREQ           NUMBER(16,6),
   RESERVECONDITION     NUMBER(16,6),
   RESERVESURPLUS       NUMBER(16,6),
   LOADREJECTIONRESERVEREQ NUMBER(16,6),
   LOADREJECTIONRESERVESURPLUS NUMBER(16,6),
   NETINTERCHANGEUNDEREXCESS NUMBER(16,6),
   NETINTERCHANGEUNDERSCARCITY NUMBER(16,6),
   LASTCHANGED          DATE,
   EXCESSGENERATION     NUMBER(22,0),
   ENERGYREQUIRED       NUMBER(15,5),
   CAPACITYREQUIRED     NUMBER(15,5),
   DATETIME             DATE
)
   tablespace PSSTAB01;

comment on table PASAREGIONSOLUTION is
'PASAREGIONSOLUTION shows the Regional solution for ST PASA showing reserves for each half-hour period. This table (PASAREGIONSOLUTION_D) shows the latest calculated result for each period.';

comment on column PASAREGIONSOLUTION.CASEID is
'PASA Case Identifier';

comment on column PASAREGIONSOLUTION.REGIONID is
'Region Identifier';

comment on column PASAREGIONSOLUTION.PERIODID is
'PASA Interval (48 half hours)';

comment on column PASAREGIONSOLUTION.DEMAND10 is
'10% exceedence forecast';

comment on column PASAREGIONSOLUTION.DEMAND50 is
'50% exceedence forecast';

comment on column PASAREGIONSOLUTION.DEMAND90 is
'90% exceedence forecast';

comment on column PASAREGIONSOLUTION.UNCONSTRAINEDCAPACITY is
'Unconstrained capacity';

comment on column PASAREGIONSOLUTION.CONSTRAINEDCAPACITY is
'Constrained capacity';

comment on column PASAREGIONSOLUTION.CAPACITYSURPLUS is
'Surplus capacity';

comment on column PASAREGIONSOLUTION.RESERVEREQ is
'Reserve requirement';

comment on column PASAREGIONSOLUTION.RESERVECONDITION is
'Reserve condition';

comment on column PASAREGIONSOLUTION.RESERVESURPLUS is
'Reserve surplus';

comment on column PASAREGIONSOLUTION.LOADREJECTIONRESERVEREQ is
'Load rejection reserve requirement';

comment on