/*==============================================================*/
/* Data Model:     AEMO Gas Data Model v2.2 mySQL               */
/* Created on:     2022/08/28 16:37:56                          */
/*==============================================================*/


/*==============================================================*/
/* Table: GAS_BASIN                                             */
/*==============================================================*/
create table GAS_BASIN
(
   BASINID              numeric(20,0) not null  comment 'A unique AEMO defined Facility Identifier',
   BASINNAME            varchar(100)  comment 'The name of the basin. If short name exists then short name included in report',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_BASIN comment 'This report displays a list of all basins';

alter table GAS_BASIN
   add primary key (BASINID);

/*==============================================================*/
/* Table: GAS_CAPACITY_TYPE                                     */
/*==============================================================*/
create table GAS_CAPACITY_TYPE
(
   CAPACITYTYPE         varchar(20) not null  comment 'Capacity Type - The type of capacity being detailed. Valid entries are: STORAGE, MDQ',
   DESCRIPTION          varchar(800)  comment 'Description of Capacity Type',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_CAPACITY_TYPE comment 'Capacity Type - The type of capacity detailed. Valid entries';

alter table GAS_CAPACITY_TYPE
   add primary key (CAPACITYTYPE);

/*==============================================================*/
/* Table: GAS_CONNECTIONPOINT_CAPACITY                          */
/*==============================================================*/
create table GAS_CONNECTIONPOINT_CAPACITY
(
   CONNECTIONPOINTID    numeric(10,0) not null  comment 'Unique connection point identifier.',
   EFFECTIVEDATE        datetime not null  comment 'Effective date of the record',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record',
   CAPACITYQUANTITY     numeric(18,3)  comment 'Quantity of available capacity in TJs and support just 3 decimal places during the defined period',
   DESCRIPTION          varchar(255)  comment 'Description for reasons or comments directly related to the quantity or change in Outlook quantity in relation to the Facility.'
);

alter table GAS_CONNECTIONPOINT_CAPACITY comment 'Provide details of the gate station capacity on a particular';

alter table GAS_CONNECTIONPOINT_CAPACITY
   add primary key (CONNECTIONPOINTID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_CONTACT                                           */
/*==============================================================*/
create table GAS_CONTACT
(
   PERSONID             numeric(10,0) not null  comment 'Unique person identifier',
   TITLE                varchar(10)  comment 'Title of person',
   FIRSTNAME            varchar(40)  comment 'First Name of person',
   LASTNAME             varchar(40)  comment 'Last Name of person',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_CONTACT comment 'The registered user list of responsible contact persons for ';

alter table GAS_CONTACT
   add primary key (PERSONID);

/*==============================================================*/
/* Table: GAS_CONTACT_DETAIL                                    */
/*==============================================================*/
create table GAS_CONTACT_DETAIL
(
   PERSONID             numeric(10,0) not null  comment 'Unique person identifier',
   EFFECTIVEDATE        datetime not null  comment 'Effective date for this operating state',
   VERSIONDATETIME      datetime not null  comment 'Version date time for this operating state',
   POSITION             varchar(40)  comment 'Position of person',
   PHONE                varchar(30)  comment 'Landline phone of person',
   FAX                  varchar(30)  comment 'Facsimile of person',
   MOBILE               varchar(30)  comment 'Mobile phone of person',
   EMAIL                varchar(255)  comment 'Email address of person',
   AUTHORISEDDATE       datetime  comment 'Date and time this operating state was authorised',
   AUTHORISEDBY         varchar(30)  comment 'User which authorised the change'
);

alter table GAS_CONTACT_DETAIL comment 'The registered user list details of responsible contact pers';

alter table GAS_CONTACT_DETAIL
   add primary key (PERSONID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_DAILY_FLOW_ACTUAL                                 */
/*==============================================================*/
create table GAS_DAILY_FLOW_ACTUAL
(
   GASDATE              datetime not null  comment 'Date of gas day',
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility Identifier',
   CONNECTIONPOINTID    numeric(10,0) not null  comment 'Unique connection point identifier.',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record',
   FLOWDIRECTION        varchar(20) not null  comment 'Direction of flow for the submission record. Must be one of RECEIPT or DELIVERY',
   ACTUALQUANTITY       numeric(18,3)  comment 'The actual flow quantity reported in TJ to the nearest gigajoule.',
   QUALITY              varchar(5)  comment 'Indicates whether meter data for the submission date is available',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_DAILY_FLOW_ACTUAL comment 'Provides on each gas day D, the BB facility operators daily';

alter table GAS_DAILY_FLOW_ACTUAL
   add primary key (GASDATE, FACILITYID, CONNECTIONPOINTID, VERSIONDATETIME, FLOWDIRECTION);

/*==============================================================*/
/* Table: GAS_DAILY_STORAGE_ACTUAL                              */
/*==============================================================*/
create table GAS_DAILY_STORAGE_ACTUAL
(
   GASDATE              datetime not null  comment 'Date of gas day',
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility Identifier',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record',
   ACTUALQUANTITY       numeric(18,3)  comment 'The actual flow quantity reported in TJ to the nearest gigajoule.',
   LASTCHANGED          datetime  comment 'Date and time record was last modified',
   CUSHIONGASQUANTITY   numeric(18,3)  comment 'The quantity of natural gas that must be retained in the Storage or LNGImport facility in order to maintain the required pressure and deliverability rates'
);

alter table GAS_DAILY_STORAGE_ACTUAL comment 'Provide on each gas day D, the actual quantity of natural ga';

alter table GAS_DAILY_STORAGE_ACTUAL
   add primary key (GASDATE, FACILITYID, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_DATA_MODEL_AUDIT                                  */
/*==============================================================*/
create table GAS_DATA_MODEL_AUDIT
(
   INSTALLATION_DATE    datetime not null  comment 'The date in which the changes to the Gas Data Model were installed',
   GASDM_VERSION        varchar(20) not null  comment 'The version of Gas Data Model after the script has been applied',
   INSTALL_TYPE         varchar(10) not null  comment 'The type of the patch applied. Valid entries are: FULL, UPGRADE, DML',
   SCRIPT_VERSION       varchar(20)  comment 'The  version of the patch set to the Gas Data Model',
   GAS_CHANGE_NOTICE    varchar(20)  comment 'The Gas Change notice for which this Gas Data Model applies',
   PROJECT_TITLE        varchar(200)  comment 'The name of the business project for which these changes to the Gas Data Model apply',
   USERNAME             varchar(40)  comment 'The USER applying this script',
   STATUS               varchar(10)  comment 'The status of the upgrade. Valid entries are STARTED, FAILED, SUCCESS'
);

alter table GAS_DATA_MODEL_AUDIT comment 'GAS_DATA_MODEL_AUDIT shows the audit trail of scripts applie';

alter table GAS_DATA_MODEL_AUDIT
   add primary key (INSTALLATION_DATE, GASDM_VERSION, INSTALL_TYPE);

/*==============================================================*/
/* Table: GAS_DEVFACILITY                                       */
/*==============================================================*/
create table GAS_DEVFACILITY
(
   DEVFACILITYID        numeric(20,0) not null  comment 'A unique AEMO defined Development Facility Identifier',
   DESCRIPTION          varchar(100)  comment 'Description of development facility',
   FACILITYTYPE         varchar(40)  comment 'The facility development type',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_DEVFACILITY comment 'Details of all Facility Developments';

alter table GAS_DEVFACILITY
   add primary key (DEVFACILITYID);

/*==============================================================*/
/* Table: GAS_DEVFACILITY_DETAIL                                */
/*==============================================================*/
create table GAS_DEVFACILITY_DETAIL
(
   DEVFACILITYID        numeric(20,0) not null  comment 'A unique AEMO defined Development Facility Identifier',
   EFFECTIVEDATE        datetime not null  comment 'The effective date of the submission',
   VERSIONDATETIME      datetime not null  comment 'Time a successful submission is accepted by AEMO systems',
   PROPOSEDNAME         varchar(100)  comment 'The name of the Facility development',
   MINNAMEPLATE         numeric(18,3)  comment 'The lower estimate of nameplate rating capacity',
   MAXNAMEPLATE         numeric(18,3)  comment 'The upper estimate of nameplate rating capacity',
   LOCATION             varchar(200)  comment 'The location of the development facility',
   PLANNEDCOMMISSIONFROM varchar(7)  comment 'The planned start date of commissioning',
   PLANNEDCOMMISSIONTO  varchar(7)  comment 'The planned end date of commissioning',
   DEVELOPMENTSTAGE     varchar(200)  comment 'The current stage of the development facility being, PROPOSED, COMMITTED, CANCELLED, ENDED',
   COMMENTS             varchar(400)  comment 'Any additional comments included in the submission',
   AUTHORISEDDATE       datetime  comment 'Date and time this operating state was authorised',
   AUTHORISEDBY         varchar(30)  comment 'Reporting entity for the facility',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_DEVFACILITY_DETAIL comment 'Details of all Facility Developments over time';

alter table GAS_DEVFACILITY_DETAIL
   add primary key (DEVFACILITYID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_DEVFACILITY_FACILITY                              */
/*==============================================================*/
create table GAS_DEVFACILITY_FACILITY
(
   DEVFACILITYID        numeric(20,0) not null  comment 'A unique AEMO defined Development Facility Identifier',
   EFFECTIVEDATE        datetime not null  comment 'The effective date of the submission',
   VERSIONDATETIME      datetime not null  comment 'Time a successful submission is accepted by AEMO systems',
   FACILITYID           numeric(20,0) not null  comment 'Unique facility identifier'
);

alter table GAS_DEVFACILITY_FACILITY comment 'Any facility that is related to dev facility';

alter table GAS_DEVFACILITY_FACILITY
   add primary key (DEVFACILITYID, EFFECTIVEDATE, VERSIONDATETIME, FACILITYID);

/*==============================================================*/
/* Table: GAS_FACILITY                                          */
/*==============================================================*/
create table GAS_FACILITY
(
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility Identifier',
   FACILITYNAME         varchar(100)  comment 'Name of the Facility',
   FACILITYTYPE         varchar(40)  comment 'Describes type of Facility.Valid entries are: PIPE, PROD or STOR',
   LASTCHANGED          datetime  comment 'Date and time record was last modified',
   FACILITYSHORTNAME    varchar(30)  comment 'An abbreviation of the facility name'
);

alter table GAS_FACILITY comment 'Provides details of all facilities';

alter table GAS_FACILITY
   add primary key (FACILITYID);

/*==============================================================*/
/* Table: GAS_FACILITY_DETAIL                                   */
/*==============================================================*/
create table GAS_FACILITY_DETAIL
(
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility Identifier',
   EFFECTIVEDATE        datetime not null  comment 'Effective date of the record',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record',
   GASDAYSTARTHOUR      numeric(2,0)  comment 'Column no longer populated',
   AUTHORISEDDATE       datetime  comment 'Date and time this operating state was authorised',
   AUTHORISEDBY         varchar(30)  comment 'User that authorised the change',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_FACILITY_DETAIL comment 'The details of all gas plant facilities over time';

alter table GAS_FACILITY_DETAIL
   add primary key (FACILITYID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_FACILITY_DETAIL_CAPACITY                          */
/*==============================================================*/
create table GAS_FACILITY_DETAIL_CAPACITY
(
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility identifier',
   EFFECTIVEDATE        datetime not null  comment 'Date capacity comes into effect',
   CAPACITYTYPE         varchar(20) not null  comment 'Capacity Type - Must be either STORAGE or MDQ.',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record.',
   LOWRANGE             numeric(3,0)  comment 'Percentage value used for (low) validation of capacity outlook',
   HIRANGE              numeric(3,0)  comment 'Percentage value used for (high) validation of capacity outlook',
   UNITS                varchar(5)  comment 'Unit of measurement. Only TJ is used',
   COMMENTS             varchar(255)  comment 'Free text description of record',
   AUTHORISEDDATE       datetime  comment 'Date and time this operating state was authorised',
   AUTHORISEDBY         varchar(30)  comment 'User which authorised the change'
);

alter table GAS_FACILITY_DETAIL_CAPACITY comment 'The standing details on capacity of all gas plant facilities';

alter table GAS_FACILITY_DETAIL_CAPACITY
   add primary key (FACILITYID, EFFECTIVEDATE, CAPACITYTYPE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_FACILITY_NODE                                     */
/*==============================================================*/
create table GAS_FACILITY_NODE
(
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility Identifier',
   EFFECTIVEDATE        datetime not null  comment 'Effective date of the record',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record',
   NODEID               numeric(10,0) not null  comment 'Unique Node Identifier'
);

alter table GAS_FACILITY_NODE comment 'Identifies the set of nodes linked to facilities. A facility';

alter table GAS_FACILITY_NODE
   add primary key (FACILITYID, EFFECTIVEDATE, VERSIONDATETIME, NODEID);

/*==============================================================*/
/* Table: GAS_FACILITY_NODE_CONNECTION                          */
/*==============================================================*/
create table GAS_FACILITY_NODE_CONNECTION
(
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility Identifier',
   EFFECTIVEDATE        datetime not null  comment 'Effective date of the record',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record',
   NODEID               numeric(10,0) not null  comment 'Unique Node Identifier',
   CONNECTIONPOINTID    numeric(10,0) not null  comment 'Unique connection point identifier.',
   FLOWDIRECTION        varchar(20) not null  comment 'Direction of flow for the submission record. Must be one of RECEIPT or DELIVERY',
   ISEXEMPT             numeric(1,0)  comment 'Submissions are not required if this is set to 1.',
   HASAGGREGATIONPRIORITY numeric(1,0)  comment 'If 1, this will be used for aggregating.',
   CONNECTIONPOINTNAME  varchar(200)  comment 'Name of the Connection point',
   MIRN                 varchar(20)  comment 'Meter Installation Registration Number  (MIRN). Unique identifier for the Victorian DTS delivery or supply point, used to retrieve the Daily Actual Flow and Nomination and Forecast from DWGM.',
   HASFORECAST          numeric(1,0)  comment 'Used to determine the MIRN required for Nomination and Forecast from DWGM.',
   CONNECTIONPOINTSHORTNAME varchar(30)  comment 'An abbreviation of the connection point name',
   EXEMPTIONDESCRIPTION varchar(200)  comment 'Description of the exemption'
);

alter table GAS_FACILITY_NODE_CONNECTION comment 'Identifies the set of connections associated with a node. Th';

alter table GAS_FACILITY_NODE_CONNECTION
   add primary key (FACILITYID, EFFECTIVEDATE, VERSIONDATETIME, NODEID, CONNECTIONPOINTID, FLOWDIRECTION);

/*==============================================================*/
/* Table: GAS_FACILITY_NODE_TRK                                 */
/*==============================================================*/
create table GAS_FACILITY_NODE_TRK
(
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility Identifier',
   EFFECTIVEDATE        datetime not null  comment 'Effective date of the record',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record',
   AUTHORISEDDATE       datetime  comment 'Any records without a date are ignored by AEMO systems.',
   AUTHORISEDBY         varchar(30)  comment 'User that authorised the change.',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_FACILITY_NODE_TRK comment 'A facility tracking record. A parent record relating to sets';

alter table GAS_FACILITY_NODE_TRK
   add primary key (FACILITYID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_FACILITY_OPSTATE                                  */
/*==============================================================*/
create table GAS_FACILITY_OPSTATE
(
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility identifier',
   EFFECTIVEDATE        datetime not null  comment 'Effective date for this operating state',
   VERSIONDATETIME      datetime not null  comment 'Version date time for this operating state',
   OPERATINGSTATE       varchar(20)  comment 'Operating status - active, inactive',
   AUTHORISEDDATE       datetime  comment 'Date and time this operating state was authorised',
   AUTHORISEDBY         varchar(30)  comment 'User which authorised the change',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_FACILITY_OPSTATE comment 'The operating status of all facilities over time';

alter table GAS_FACILITY_OPSTATE
   add primary key (FACILITYID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_FACILITY_OWNER                                    */
/*==============================================================*/
create table GAS_FACILITY_OWNER
(
   COMPANYID            numeric(10,0) not null  comment 'Unique company identifier',
   EFFECTIVEDATE        datetime not null  comment 'Effective date for this operating state',
   VERSIONDATETIME      datetime not null  comment 'Version date time for this operating state',
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility identifier'
);

alter table GAS_FACILITY_OWNER comment 'The detail of facility ownership over time';

alter table GAS_FACILITY_OWNER
   add primary key (COMPANYID, EFFECTIVEDATE, VERSIONDATETIME, FACILITYID);

/*==============================================================*/
/* Table: GAS_FACILITY_OWNER_TRK                                */
/*==============================================================*/
create table GAS_FACILITY_OWNER_TRK
(
   COMPANYID            numeric(10,0) not null  comment 'Unique company identifier',
   EFFECTIVEDATE        datetime not null  comment 'Effective date for this operating state',
   VERSIONDATETIME      datetime not null  comment 'Version date time for this operating state',
   AUTHORISEDDATE       datetime  comment 'Date and time this operating state was authorised',
   AUTHORISEDBY         varchar(30)  comment 'User which authorised the change',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_FACILITY_OWNER_TRK comment 'The tracking of Facility ownership over time';

alter table GAS_FACILITY_OWNER_TRK
   add primary key (COMPANYID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_FACILITY_SUMMARY                                  */
/*==============================================================*/
create table GAS_FACILITY_SUMMARY
(
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility Identifier',
   STARTDATE            datetime not null  comment 'Effective date of the record',
   ENDDATE              datetime  comment 'End date of the record',
   NODEID               numeric(10,0)  comment 'Unique Node Identifier',
   CONNECTIONPOINTID    numeric(10,0) not null  comment 'Unique connection point identifier.',
   FLOWDIRECTION        varchar(20) not null  comment 'Direction of flow for the submission record. Must be one of RECEIPT or DELIVERY',
   CONNECTIONPOINTNAME  varchar(200)  comment 'Name of the Connection point',
   STATEID              numeric(10,0)  comment 'Unique state identifier',
   STATE                varchar(5)  comment 'Name of the state',
   FACILITYNAME         varchar(100)  comment 'Name of the Facility',
   FACILITYTYPE         varchar(40)  comment 'Describes type of Facility.Valid entries are: PIPE, PROD or STOR',
   FACILITYTYPEDESCRIPTION varchar(800)  comment 'Description of the facility type',
   GASDAYSTARTHOUR      numeric(2,0)  comment 'Column No longer populated',
   COMPANYID            numeric(10,0)  comment 'Unique company identifier',
   COMPANYNAME          varchar(50)  comment 'Name of company',
   LOCATIONID           numeric(10,0)  comment 'Unique Node Location Identifier',
   LOCATIONNAME         varchar(100)  comment 'Name of the location.',
   LOCATIONTYPE         varchar(40)  comment 'Type of location',
   LASTCHANGED          datetime  comment 'Last changed date for the record'
);

alter table GAS_FACILITY_SUMMARY comment 'A summary of gas FACILITY, capacity and ownership through ti';

alter table GAS_FACILITY_SUMMARY
   add primary key (FACILITYID, STARTDATE, CONNECTIONPOINTID, FLOWDIRECTION);

/*==============================================================*/
/* Table: GAS_FACILITY_TYPE                                     */
/*==============================================================*/
create table GAS_FACILITY_TYPE
(
   FACILITYTYPE         varchar(40) not null  comment 'Identifier for a facility type',
   DESCRIPTION          varchar(800)  comment 'Description of the facility type',
   ACTIONTYPE           varchar(40)  comment 'Used in aggregation',
   ISREPORTABLE         numeric(1,0)  comment '1 if reportable, otherwise 0',
   LASTCHANGED          datetime  comment 'Last time the record was changed'
);

alter table GAS_FACILITY_TYPE comment 'A lookup table identifying the type of facilities stored in ';

alter table GAS_FACILITY_TYPE
   add primary key (FACILITYTYPE);

/*==============================================================*/
/* Table: GAS_FIELD                                             */
/*==============================================================*/
create table GAS_FIELD
(
   FIELDID              numeric(20,0) not null  comment 'A unique AEMO defined Field Identifier',
   LASTCHANGED          datetime  comment 'Date and time record was last modified',
   STATEID              numeric(10,0)  comment 'The state ID the field interest is in',
   STATE                varchar(5)  comment 'The state the field interest is in'
);

alter table GAS_FIELD comment 'Field location';

alter table GAS_FIELD
   add primary key (FIELDID);

/*==============================================================*/
/* Table: GAS_FIELDINTEREST                                     */
/*==============================================================*/
create table GAS_FIELDINTEREST
(
   FIELDINTERESTID      numeric(20,0) not null  comment 'A unique AEMO defined Field Interest Identifier',
   FIELDID              numeric(20,0)  comment 'A unique AEMO defined Field Identifier',
   ANNUALREPORTINGDATE  varchar(30)  comment 'The annual reporting date of the field interest'
);

alter table GAS_FIELDINTEREST comment 'Field Interest annual reporting date';

alter table GAS_FIELDINTEREST
   add primary key (FIELDINTERESTID);

/*==============================================================*/
/* Table: GAS_FIELDINTEREST_DETAIL                              */
/*==============================================================*/
create table GAS_FIELDINTEREST_DETAIL
(
   FIELDINTERESTID      numeric(20,0) not null  comment 'A unique AEMO defined Field Interest Identifier',
   EFFECTIVEDATE        datetime not null  comment 'The date on which the record takes effect',
   VERSIONDATETIME      datetime not null  comment 'Time a successful submission is accepted by AEMO systems',
   PETROLEUMTENEMENTS   varchar(300)  comment 'The petroleum tenements which are the subject of the BB field interest',
   PROCESSINGFACILITIES varchar(300)  comment 'The processing facility used to process gas from the field',
   RESOURCECLASSIFICATION varchar(100)  comment 'Classification of the resources in the field as conventional or unconventional',
   RESOURCESUBCLASSIFICATION varchar(100)  comment 'Any further sub-classification of the resources',
   NATUREOFGAS          varchar(100)  comment 'The nature of the gas in the field using classifications in the BB Procedures',
   TENEMENTSHARE        numeric(10,3)  comment 'The field interest share of the petroleum tenements',
   AUTHORISEDDATE       datetime  comment 'Date and time this operating state was authorised',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_FIELDINTEREST_DETAIL comment 'This report displays information about Field Interests';

alter table GAS_FIELDINTEREST_DETAIL
   add primary key (FIELDINTERESTID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_FIELDINTEREST_PARTICIPANT                         */
/*==============================================================*/
create table GAS_FIELDINTEREST_PARTICIPANT
(
   FIELDINTERESTID      numeric(20,0) not null  comment 'A unique AEMO defined Field Interest Identifier',
   EFFECTIVEDATE        datetime not null  comment 'The date on which the record takes effect',
   VERSIONDATETIME      datetime not null  comment 'Time a successful submission is accepted by AEMO systems',
   COMPANYID            numeric(10,0) not null  comment 'The company ID of the responsible field owner',
   AUTHORISEDDATE       datetime  comment 'Date and time this operating state was authorised',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_FIELDINTEREST_PARTICIPANT comment 'The responsible participant of the field interest';

alter table GAS_FIELDINTEREST_PARTICIPANT
   add primary key (FIELDINTERESTID, EFFECTIVEDATE, VERSIONDATETIME, COMPANYID);

/*==============================================================*/
/* Table: GAS_FIELDINTEREST_SHARE                               */
/*==============================================================*/
create table GAS_FIELDINTEREST_SHARE
(
   FIELDINTERESTID      numeric(20,0) not null  comment 'A unique AEMO defined Field Interest Identifier',
   EFFECTIVEDATE        datetime not null  comment 'The date on which the record takes effect',
   VERSIONDATETIME      datetime not null  comment 'Time a successful submission is accepted by AEMO systems',
   REPORTEDCOMPANY      varchar(100) not null  comment 'Member of the field owner group',
   PERCENTAGESHARE      numeric(18,8)  comment 'The percentage share of the group member'
);

alter table GAS_FIELDINTEREST_SHARE comment 'Field owner group members';

alter table GAS_FIELDINTEREST_SHARE
   add primary key (FIELDINTERESTID, EFFECTIVEDATE, VERSIONDATETIME, REPORTEDCOMPANY);

/*==============================================================*/
/* Table: GAS_FIELDINTEREST_SHARETRK                            */
/*==============================================================*/
create table GAS_FIELDINTEREST_SHARETRK
(
   FIELDINTERESTID      numeric(20,0) not null  comment 'A unique AEMO defined Field Interest Identifier',
   EFFECTIVEDATE        datetime not null  comment 'The date on which the record takes effect',
   VERSIONDATETIME      datetime not null  comment 'Time a successful submission is accepted by AEMO systems',
   AUTHORISEDDATE       datetime  comment 'Date and time this operating state was authorised',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_FIELDINTEREST_SHARETRK comment 'Filed Share tracking details over time';

alter table GAS_FIELDINTEREST_SHARETRK
   add primary key (FIELDINTERESTID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_FIELD_DETAIL                                      */
/*==============================================================*/
create table GAS_FIELD_DETAIL
(
   FIELDID              numeric(20,0) not null  comment 'A unique AEMO defined Field Identifier',
   EFFECTIVEDATE        datetime not null  comment 'The date on which the record takes effect',
   VERSIONDATETIME      datetime not null  comment 'Time a successful submission is accepted by AEMO systems',
   FIELDNAME            varchar(100)  comment 'The name of the Field in which the Field Interest is located',
   FIELDSHORTNAME       varchar(30)  comment 'The short name of the field',
   DESCRIPTION          varchar(400)  comment 'Additional information relating to the field',
   LOCATIONDESCRIPTION  varchar(200)  comment 'Additional information relating to the location of the field',
   AUTHORISEDDATE       datetime  comment 'Date and time this operating state was authorised',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_FIELD_DETAIL comment 'Field details';

alter table GAS_FIELD_DETAIL
   add primary key (FIELDID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_GSH_TRADES                                        */
/*==============================================================*/
create table GAS_GSH_TRADES
(
   TRADE_DATE           datetime not null  comment 'Date and time the trade is executed',
   TRANSACTION_TYPE     varchar(20) not null  comment 'The type of the trade. Valid values are AUTO_MATCHED, OFF_MARKET_TRADE',
   PRODUCT              varchar(80) not null  comment 'The product delivery period for the Gas Supply Hub market',
   LOCATION             varchar(100) not null  comment 'Product location for the Gas Supply Hub market',
   NO_OF_TRADES         numeric(10,0)  comment 'Count of number of trades',
   TRADE_PRICE          numeric(18,8) not null  comment 'Price value of the trade in $/GJ',
   DAILY_QTY_GJ         numeric(18,8)  comment 'Volume of gas transacted on the Trading Platform in GJ/day',
   START_DATE           datetime  comment 'The start gas day for the order delivery period',
   END_DATE             datetime  comment 'The end gas day for the order delivery period',
   MANUAL_TRADE         varchar(50)  comment 'Whether the trade is a manual trade',
   LASTCHANGED          datetime  comment 'The date the record was last updated'
);

alter table GAS_GSH_TRADES comment 'Provides participants (buyer and seller) a confirmation when';

alter table GAS_GSH_TRADES
   add primary key (TRADE_DATE, TRANSACTION_TYPE, PRODUCT, LOCATION, TRADE_PRICE);

/*==============================================================*/
/* Table: GAS_LINEPACK_ADEQUACY                                 */
/*==============================================================*/
create table GAS_LINEPACK_ADEQUACY
(
   GASDATE              datetime not null  comment 'Date of gas day applicable under either the pipeline contract or market rules',
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility identifier',
   VERSIONDATETIME      datetime not null  comment 'Version date time for this operating state',
   FLAG                 varchar(5)  comment 'Expected adequacy for the gas day. Valid entries are GREEN, AMBER or RED',
   DESCRIPTION          varchar(800)  comment 'Free text description of record'
);

alter table GAS_LINEPACK_ADEQUACY comment 'The linepack adequacy (flag) for each of the (non-exempt) BB';

alter table GAS_LINEPACK_ADEQUACY
   add primary key (GASDATE, FACILITYID, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_LNG_SHIPMENT                                      */
/*==============================================================*/
create table GAS_LNG_SHIPMENT
(
   TRANSACTIONID        varchar(40) not null  comment 'Unique shipment identifier',
   FACILITYID           numeric(20,0) not null  comment 'Unique facility identifier',
   VERSIONDATETIME      datetime not null  comment 'Time a successful submission is accepted by AEMO systems',
   SHIPMENTDATE         datetime  comment 'For LNG export facility, the departure date. For LNG import facility, the date unloading commences at the LNG import facility',
   VOLUMEPJ             numeric(18,8)  comment 'Volume of the shipment in PJ',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_LNG_SHIPMENT comment 'This table displays a list of all LNG shipments';

alter table GAS_LNG_SHIPMENT
   add primary key (TRANSACTIONID, FACILITYID, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_LNG_TRANSACTION                                   */
/*==============================================================*/
create table GAS_LNG_TRANSACTION
(
   TRANSACTIONSTARTDATE datetime not null  comment 'Transaction start date',
   TRANSACTIONENDDATE   datetime  comment 'Transaction end date',
   VOLWEIGHTPRICE       numeric(18,8)  comment 'The volume weighted price for the reporting period',
   VOLUMEPJ             numeric(18,8)  comment 'The total volume of the transactions',
   SUPPLYPERIODSTART    datetime  comment 'The earliest start date of all transactions captured in the reporting period',
   SUPPLYPERIODEND      datetime  comment 'The latest end date of all transactions captured in the reporting period'
);

alter table GAS_LNG_TRANSACTION comment 'This table displays LNG transaction aggregated data';

alter table GAS_LNG_TRANSACTION
   add primary key (TRANSACTIONSTARTDATE);

/*==============================================================*/
/* Table: GAS_LOCATION                                          */
/*==============================================================*/
create table GAS_LOCATION
(
   LOCATIONID           numeric(10,0) not null  comment 'Unique Node Location Identifier',
   LOCATIONNAME         varchar(100)  comment 'Name of the location.',
   LOCATIONTYPE         varchar(40)  comment 'Type of location',
   STATEID              numeric(10,0)  comment 'Unique state identifier',
   DESCRIPTION          varchar(800)  comment 'Location description',
   LASTCHANGED          datetime  comment 'Last time the record was changed.'
);

alter table GAS_LOCATION comment 'Lists all production and demand locations within the Bulleti';

alter table GAS_LOCATION
   add primary key (LOCATIONID);

/*==============================================================*/
/* Table: GAS_LOCATION_NODE                                     */
/*==============================================================*/
create table GAS_LOCATION_NODE
(
   LOCATIONID           numeric(10,0) not null  comment 'Unique Location Identifier',
   EFFECTIVEDATE        datetime not null  comment 'Effective date of the record',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record',
   NODEID               numeric(10,0) not null  comment 'Unique Node Identifier'
);

alter table GAS_LOCATION_NODE comment 'Provides a list of nodes within a location.';

alter table GAS_LOCATION_NODE
   add primary key (LOCATIONID, EFFECTIVEDATE, VERSIONDATETIME, NODEID);

/*==============================================================*/
/* Table: GAS_LOCATION_NODE_TRK                                 */
/*==============================================================*/
create table GAS_LOCATION_NODE_TRK
(
   LOCATIONID           numeric(10,0) not null  comment 'Unique Location identifier.',
   EFFECTIVEDATE        datetime not null  comment 'Effective date of the record',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record',
   AUTHORISEDDATE       datetime  comment 'Date and time this operating state was authorised',
   AUTHORISEDBY         varchar(30)  comment 'User which authorised the change',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_LOCATION_NODE_TRK comment 'Provides a record of node tracking.';

alter table GAS_LOCATION_NODE_TRK
   add primary key (LOCATIONID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_LOCATION_OPSTATE                                  */
/*==============================================================*/
create table GAS_LOCATION_OPSTATE
(
   LOCATIONID           numeric(10,0) not null  comment 'Unique Location identifier.',
   EFFECTIVEDATE        datetime not null  comment 'Effective date of the record',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record',
   OPERATINGSTATE       varchar(20)  comment 'Operating status - active, inactive.',
   AUTHORISEDDATE       datetime  comment 'Date and time this operating state was authorised.',
   AUTHORISEDBY         varchar(30)  comment 'User that authorised the change',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_LOCATION_OPSTATE comment 'The operating status of all locations over time.';

alter table GAS_LOCATION_OPSTATE
   add primary key (LOCATIONID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_MARKET                                            */
/*==============================================================*/
create table GAS_MARKET
(
   MARKETID             varchar(20) not null  comment 'Unique market identifier',
   DESCRIPTION          varchar(255)  comment 'Market description',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_MARKET comment 'The list of defined Gas Markets against which the Participan';

alter table GAS_MARKET
   add primary key (MARKETID);

/*==============================================================*/
/* Table: GAS_MEDIUM_TERM_CAP_OUTLOOK                           */
/*==============================================================*/
create table GAS_MEDIUM_TERM_CAP_OUTLOOK
(
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility Identifier',
   FROMGASDATE          datetime not null  comment 'Start date (inclusive) of the record',
   TOGASDATE            datetime  comment 'End date (exclusive) of the record',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record',
   CAPACITYTYPE         varchar(20) not null  comment 'Capacity Type - Must be either STORAGE or MDQ',
   FLOWDIRECTION        varchar(20) not null  comment 'Direction of flow for the submission record. Must be one of RECEIPT or DELIVERY',
   RECEIPTLOCATION      numeric(10,0) not null  comment 'Connection Point ID that best represents the receipt location',
   DELIVERYLOCATION     numeric(10,0) not null  comment 'Connection Point ID that best represents the Delivery Location',
   OUTLOOKQUANTITY      numeric(18,3)  comment 'Quantity of available capacity during the defined period',
   CAPACITYDESCRIPTION  varchar(1000)  comment 'Describes the meaning of the capacity Number Provided',
   DESCRIPTION          varchar(1000)  comment 'Description for reasons or comments directly related to the quantity or change in Outlook quantity in relation to the Facility',
   ACTIVEFLAG           numeric(1,0)  comment 'Active flag for the record.  A value of 1 indicates the record is currently active',
   DISABLEDDATETIME     datetime  comment 'Audit timestamp of the record',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_MEDIUM_TERM_CAP_OUTLOOK comment 'Provide the details of activities that may affect the daily ';

alter table GAS_MEDIUM_TERM_CAP_OUTLOOK
   add primary key (FACILITYID, FROMGASDATE, VERSIONDATETIME, CAPACITYTYPE, FLOWDIRECTION, RECEIPTLOCATION, DELIVERYLOCATION);

/*==============================================================*/
/* Table: GAS_NAMEPLATE_RATING                                  */
/*==============================================================*/
create table GAS_NAMEPLATE_RATING
(
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility Identifier',
   EFFECTIVEDATE        datetime not null  comment 'Effective date of the record',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record.',
   CAPACITYTYPE         varchar(20) not null  comment 'Capacity Type - Must be either STORAGE or MDQ',
   FLOWDIRECTION        varchar(20) not null  comment 'Direction of flow for the submission record. Must be either RECEIPT or DELIVERY.',
   RECEIPTLOCATION      numeric(10,0) not null  comment 'Connection Point ID that best represents the Receipt location',
   DELIVERYLOCATION     numeric(10,0) not null  comment 'Connection Point ID that best represents the Delivery Location',
   CAPACITYQUANTITY     numeric(18,3)  comment 'Quantity of available capacity during the defined period.',
   CAPACITYDESCRIPTION  varchar(1000)  comment 'Describes the meaning of the capacity Number Provided',
   DESCRIPTION          varchar(1000)  comment 'Description for reasons or comments directly related to the quantity or change in Outlook quantity in relation to the Facility.'
);

alter table GAS_NAMEPLATE_RATING comment 'Provide the nameplate rating of each facility or information';

alter table GAS_NAMEPLATE_RATING
   add primary key (FACILITYID, EFFECTIVEDATE, VERSIONDATETIME, CAPACITYTYPE, FLOWDIRECTION, RECEIPTLOCATION, DELIVERYLOCATION);

/*==============================================================*/
/* Table: GAS_NODE                                              */
/*==============================================================*/
create table GAS_NODE
(
   NODEID               numeric(10,0) not null  comment 'Unique Node Identifier',
   STATEID              numeric(10,0)  comment 'Unique StateIdentifier',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_NODE comment 'A container for up to four connections';

alter table GAS_NODE
   add primary key (NODEID);

/*==============================================================*/
/* Table: GAS_NODE_OPSTATE                                      */
/*==============================================================*/
create table GAS_NODE_OPSTATE
(
   NODEID               numeric(10,0) not null  comment 'Unique Node Identifier',
   EFFECTIVEDATE        datetime not null  comment 'Effective date of the record',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record',
   OPERATINGSTATE       varchar(20)  comment 'Operating status - active, inactive.',
   AUTHORISEDDATE       datetime  comment 'Date and time this operating state was authorised.',
   AUTHORISEDBY         varchar(30)  comment 'User that authorised the change.',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_NODE_OPSTATE comment 'The operating status of all facility nodes over time';

alter table GAS_NODE_OPSTATE
   add primary key (NODEID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_NOMINATIONS_FORECAST                              */
/*==============================================================*/
create table GAS_NOMINATIONS_FORECAST
(
   GASDATE              datetime not null  comment 'Date of gas day',
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility Identifier',
   CONNECTIONPOINTID    numeric(10,0) not null  comment 'Unique connection point identifier.',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record',
   FLOWDIRECTION        varchar(20) not null  comment 'Direction of flow for the submission record. Must be one of RECEIPT or DELIVERY',
   NOMINATIONQUANTITY   numeric(18,3)  comment 'Receipt or delivery nomination quantity in TJ'
);

alter table GAS_NOMINATIONS_FORECAST comment 'For BB pipelines forming part of a Declared Transmission Sys';

alter table GAS_NOMINATIONS_FORECAST
   add primary key (GASDATE, FACILITYID, CONNECTIONPOINTID, VERSIONDATETIME, FLOWDIRECTION);

/*==============================================================*/
/* Table: GAS_NOMINATIONS_FORECAST_AGG                          */
/*==============================================================*/
create table GAS_NOMINATIONS_FORECAST_AGG
(
   GASDATE              datetime not null  comment 'Date of gas day',
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility Identifier',
   LOCATIONID           numeric(10,0) not null  comment 'Unique Node Location Identifier',
   VERSIONDATETIME      datetime not null  comment 'Version date time of the record',
   DEMAND               numeric(18,3)  comment 'Usage type expressed in TJ. Three decimal places is not shown if the value has trailing zeros after the decimal place.',
   SUPPLY               numeric(18,3)  comment 'Usage type expressed in TJ. Three decimal places is not shown if the value has trailing zeros after the decimal place',
   TRANSFERIN           numeric(18,3)  comment 'Usage type expressed in TJ. Only applicable to BB pipelines. Three decimal places is not shown if the value has trailing zeros after the decimal place.',
   TRANSFEROUT          numeric(18,3)  comment 'Usage type expressed in TJ. Only applicable to BB pipelines. Three decimal places is not shown if the value has trailing zeros after the decimal place.',
   COMPLETENESS         numeric(3,0)  comment 'Completeness of aggregated figure in percentage',
   LASTCHANGED          datetime  comment 'Last changed date for the record'
);

alter table GAS_NOMINATIONS_FORECAST_AGG comment 'Provides the aggregate of forecast and nominations by locati';

alter table GAS_NOMINATIONS_FORECAST_AGG
   add primary key (GASDATE, FACILITYID, LOCATIONID, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_PARTICIPANT                                       */
/*==============================================================*/
create table GAS_PARTICIPANT
(
   COMPANYID            numeric(10,0) not null  comment 'Unique company identifier',
   DESCRIPTION          varchar(800)  comment 'Name of company',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_PARTICIPANT comment 'The list of all registered organisations i.e  the BB Partici';

alter table GAS_PARTICIPANT
   add primary key (COMPANYID);

/*==============================================================*/
/* Table: GAS_PARTICIPANT_CONTACT                               */
/*==============================================================*/
create table GAS_PARTICIPANT_CONTACT
(
   COMPANYID            numeric(10,0) not null  comment 'Unique company identifier',
   EFFECTIVEDATE        datetime not null  comment 'Effective date for this operating state',
   VERSIONDATETIME      datetime not null  comment 'Version date time for this operating state',
   PERSONID             numeric(10,0) not null  comment 'Unique person identifier'
);

alter table GAS_PARTICIPANT_CONTACT comment 'The detail of contacts against a gas participant over time';

alter table GAS_PARTICIPANT_CONTACT
   add primary key (COMPANYID, EFFECTIVEDATE, VERSIONDATETIME, PERSONID);

/*==============================================================*/
/* Table: GAS_PARTICIPANT_CONTACT_TRK                           */
/*==============================================================*/
create table GAS_PARTICIPANT_CONTACT_TRK
(
   COMPANYID            numeric(10,0) not null  comment 'Unique company identifier',
   EFFECTIVEDATE        datetime not null  comment 'Effective date for this operating state',
   VERSIONDATETIME      datetime not null  comment 'Version date time for this operating state',
   AUTHORISEDDATE       datetime  comment 'Date and time this operating state was authorised',
   AUTHORISEDBY         varchar(30)  comment 'User which authorised the change'
);

alter table GAS_PARTICIPANT_CONTACT_TRK comment 'The tracking of sets of contacts against a gas participant o';

alter table GAS_PARTICIPANT_CONTACT_TRK
   add primary key (COMPANYID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_PARTICIPANT_DETAIL                                */
/*==============================================================*/
create table GAS_PARTICIPANT_DETAIL
(
   COMPANYID            numeric(10,0) not null  comment 'Unique company identifier',
   EFFECTIVEDATE        datetime not null  comment 'Effective date for this operating state',
   VERSIONDATETIME      datetime not null  comment 'Version date time for this operating state',
   COMPANYNAME          varchar(50)  comment 'Name of company',
   ABN                  varchar(30)  comment 'Australian Business Number of company',
   ADDRESSTYPE          varchar(40)  comment 'Type of address eg. Head Office, Postal Address etc.',
   ADDRESS              varchar(120)  comment 'Address of company based on AddressType',
   LOCALE               varchar(40)  comment 'City location of company',
   JURISDICTION         varchar(5)  comment 'State location of company',
   POSTCODE             varchar(4)  comment 'Postcode of company',
   COMPANYPHONE         varchar(30)  comment 'Main (reception) phone of company',
   COMPANYFAX           varchar(30)  comment 'Main (reception) facsimile of company',
   AUTHORISEDDATE       datetime  comment 'Date and time this operating state was authorised',
   AUTHORISEDBY         varchar(30)  comment 'User which authorised the change',
   ORGANISATIONTYPENAME varchar(40)  comment 'The type of organisation'
);

alter table GAS_PARTICIPANT_DETAIL comment 'The details of all registered organisations i.e  the BB Part';

alter table GAS_PARTICIPANT_DETAIL
   add primary key (COMPANYID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_PARTICIPANT_OPSTATE                               */
/*==============================================================*/
create table GAS_PARTICIPANT_OPSTATE
(
   MARKETID             varchar(20) not null  comment 'Unique Market identifier',
   COMPANYID            numeric(10,0) not null  comment 'Unique company identifier',
   EFFECTIVEDATE        datetime not null  comment 'Effective date for this operating state',
   VERSIONDATETIME      datetime not null  comment 'Version date time for this operating state',
   OPERATINGSTATE       varchar(20)  comment 'Operating status - active, inactive',
   AUTHORISEDDATE       datetime  comment 'Date and time this operating state was authorised',
   AUTHORISEDBY         varchar(30)  comment 'User which authorised the change'
);

alter table GAS_PARTICIPANT_OPSTATE comment 'The operating status of all registered organisations i.e  th';

alter table GAS_PARTICIPANT_OPSTATE
   add primary key (MARKETID, COMPANYID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_RESERVES_AND_RESOURCES                            */
/*==============================================================*/
create table GAS_RESERVES_AND_RESOURCES
(
   FIELDINTERESTID      numeric(20,0) not null  comment 'A unique AEMO defined Field Interest Identifier',
   EFFECTIVEDATE        datetime not null  comment 'The date on which the record takes effect',
   VERSIONDATETIME      datetime not null  comment 'Time a successful submission is accepted by AEMO systems',
   DEVELOPEDRESERVE1P   numeric(18,3)  comment 'An estimate of the BB field interest 1P developed reserves',
   DEVELOPEDRESERVE2P   numeric(18,3)  comment 'An estimate of the BB field interest 2P developed reserves',
   DEVELOPEDRESERVE3P   numeric(18,3)  comment 'An estimate of the BB field interest 3P developed reserves',
   UNDEVELOPEDRESERVE1P numeric(18,3)  comment 'An estimate of the BB field interest 1P undeveloped reserves',
   UNDEVELOPEDRESERVE2P numeric(18,3)  comment 'An estimate of the BB field interest 2P undeveloped reserves',
   UNDEVELOPEDRESERVE3P numeric(18,3)  comment 'An estimate of the BB field interest 3P undeveloped reserves',
   RESOURCES2C          numeric(18,3)  comment 'An estimate of the BB field interest 2C resources',
   PRODUCTIONCHANGERESERVE2P numeric(18,3)  comment 'An estimate of the total movement in the BB field interest 2P reserves since the end of prior reporting year due to the production of gas',
   PROVEDAREAEXTENSIONRESERVE2P numeric(18,3)  comment 'An estimate of the total movement in the BB field interest 2P reserves since the end of prior reporting year due to the extension of a fields proved area',
   PERCENTAGECHANGERESERVE2P numeric(18,3)  comment 'An estimate of the total movement in the BB field interest 2P reserves since the end of prior reporting year due to a percentage change in the BB field interest',
   UPWARDREVISION3PRESERVETO2P numeric(18,3)  comment 'An estimate of the total movement in the BB field interest 2P reserves since the end of prior reporting year due to an upward revision of 2P reserves arising from the reclassification of 3P reserves or resources to 2P reserves',
   DOWNWARDREVISION2PRESERVETO3P numeric(18,3)  comment 'An estimate of the total movement in the BB field interest 2P reserves since the end of prior reporting year due to a downward revision of 2P reserves arising from the reclassification of 2P reserves to 3P reserves or resources',
   OTHERREVISIONSRESERVE2P numeric(18,3)  comment 'An estimate of the total movement in the BB field interest 2P reserves since the end of prior reporting year due to other revisions',
   MATURITYSUBCLASS2P   varchar(100)  comment 'The project maturity sub-class for the 2P reserves',
   MATURITYSUBCLASS2C   varchar(100)  comment 'The project maturity sub-class for the 2C resources',
   MINDATE2P            datetime  comment 'The earliest estimated date for the production of the 2P reserves',
   MAXDATE2P            datetime  comment 'The latest estimated date for the production of the 2P reserves',
   MINDATE2C            datetime  comment 'The earliest estimated date for the production of the 2C resources',
   MAXDATE2C            datetime  comment 'The latest estimated date for the production of the 2C resources',
   EXPECTEDBARRIERS2C   varchar(400)  comment 'A list of any barriers to the commercial recovery of the 2C resources',
   RESOURCESESTIMATEMETHOD varchar(200)  comment 'The resources assessment method used to prepare the reserves and resources estimates',
   CONVERSIONFACTORQTYTCFTOPJ numeric(18,3)  comment 'The conversion factor used to convert quantities measured in trillions of cubic feet to PJ',
   ECONOMICASSUMPTION   varchar(400)  comment 'The key economic assumptions in the forecast case used to prepare the reserves and resources estimates and the source of the assumptions',
   UPDATEREASON         varchar(400)  comment 'The reason for the update',
   PREPAREDBY           varchar(100)  comment 'The name of the person who prepared the estimates',
   INDEPENDENTESTIMATE  varchar(3)  comment 'Whether the qualified gas industry professional who prepared, or supervised the preparation of, the reserves and resources estimates is independent of the BB reporting entity',
   AUTHORISEDDATE       datetime  comment 'Date and time this operating state was authorised',
   LASTCHANGED          datetime  comment 'Date and time record was last modified'
);

alter table GAS_RESERVES_AND_RESOURCES comment 'This table displays information about Field Reserves and Res';

alter table GAS_RESERVES_AND_RESOURCES
   add primary key (FIELDINTERESTID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_SEC_CAP_ORDERS                                    */
/*==============================================================*/
create table GAS_SEC_CAP_ORDERS
(
   ORDERID              varchar(40) not null  comment 'Unique identifier for the order',
   SUBMITTINGCOMPANYID  numeric(10,0)  comment 'Name of the company submitting the data to AEMO',
   VERSIONDATETIME      datetime  comment 'Version timestamp of the record',
   FACILITYID           numeric(10,0)  comment 'Unique facility Identifier',
   BUYSELL              varchar(5)  comment 'Indicates whether the shipper is either looking to buy or sell spare capacity',
   AVAILABLEQUANTITY    numeric(18,3)  comment 'Quantity of actual flow for the gas date',
   PRICE                numeric(18,2)  comment 'Spare capacity price expressed in $ per GJ',
   FROMGASDATE          datetime  comment 'Date spare capacity comes into effect',
   TOGASDATE            datetime  comment 'Date spare capacity is effective to',
   RECEIPTLOCATION      numeric(10,0)  comment 'Connection Point ID that best represents the receipt location',
   DELIVERYLOCATION     numeric(10,0)  comment 'Connection Point ID that best represents the Delivery Location',
   CONTACTDETAILS       varchar(1000)  comment 'Name and number of the shippers contact person',
   LASTCHANGED          datetime  comment 'Last changed date for the record'
);

alter table GAS_SEC_CAP_ORDERS comment 'Information related to the bids and offers of secondary pipe';

alter table GAS_SEC_CAP_ORDERS
   add primary key (ORDERID);

/*==============================================================*/
/* Index: GAS_SEC_CAP_ORDERS_IDX1                               */
/*==============================================================*/
create index GAS_SEC_CAP_ORDERS_IDX1 on GAS_SEC_CAP_ORDERS
(
   SUBMITTINGCOMPANYID,
   VERSIONDATETIME
);

/*==============================================================*/
/* Index: GAS_SEC_CAP_ORDERS_IDX2                               */
/*==============================================================*/
create index GAS_SEC_CAP_ORDERS_IDX2 on GAS_SEC_CAP_ORDERS
(
   FROMGASDATE,
   TOGASDATE
);

/*==============================================================*/
/* Table: GAS_SEC_CAP_ORDERS_TRK                                */
/*==============================================================*/
create table GAS_SEC_CAP_ORDERS_TRK
(
   SUBMITTINGCOMPANYID  numeric(10,0) not null  comment 'Name of the company submitting the data to AEMO',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record',
   LASTCHANGED          datetime  comment 'Last changed date for the record'
);

alter table GAS_SEC_CAP_ORDERS_TRK comment 'Bids and offer tracking information of secondary pipeline ca';

alter table GAS_SEC_CAP_ORDERS_TRK
   add primary key (SUBMITTINGCOMPANYID, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_SEC_CAP_TRADES                                    */
/*==============================================================*/
create table GAS_SEC_CAP_TRADES
(
   TRADEID              varchar(40) not null  comment 'Unique identifier for the trade',
   SUBMITTINGCOMPANYID  numeric(10,0)  comment 'Name of the company submitting the data to AEMO',
   VERSIONDATETIME      datetime  comment 'Version timestamp of the record',
   FACILITYID           numeric(10,0)  comment 'Unique facility Identifier',
   GASDATE              datetime  comment 'Gas date',
   RECEIPTLOCATION      numeric(10,0)  comment 'Connection Point ID that best represents the Receipt location',
   DELIVERYLOCATION     numeric(10,0)  comment 'Connection Point ID that best represents the Delivery Location',
   NAMEPLATECAPACITY    numeric(18,3)  comment 'Official pipeline capacity expressed in GJ',
   DAILYNOMINATIONS     numeric(18,3)  comment 'Daily aggregate quantity of gas (expressed in GJ) nominated for delivery from the pipeline',
   DAILYUTILISATIONPERCENT numeric(18,2)  comment 'Percentage of the pipeline capacity that is utilised per day',
   AVAILABLECAPACITY    numeric(18,3)  comment 'Operational capacity minus nominations each day. Expressed in GJ',
   CAPACITYONOFFER      numeric(18,3)  comment 'Sum of total capacity offered for sale expressed in GJ',
   DAILYCAPACITYTRADED  numeric(18,3)  comment 'Sum of total daily sold capacity expressed in GJ',
   DAILYCAPACITY        numeric(18,3)  comment 'Operational capacity',
   CONTRACTEDCAPACITY   numeric(18,3)  comment 'Firm-forward, contracted pipeline capacity expressed in GJ.',
   AVERAGEANNUALCAPACITYTRADED numeric(18,3)  comment 'Sum of total annual sold capacity divided by number of days, year to date. Expressed in GJ.',
   LASTCHANGED          datetime  comment 'Last changed date for the record'
);

alter table GAS_SEC_CAP_TRADES comment 'Information related to summary trading data, which is derive';

alter table GAS_SEC_CAP_TRADES
   add primary key (TRADEID);

/*==============================================================*/
/* Index: GAS_SEC_CAP_TRADES_IDX1                               */
/*==============================================================*/
create index GAS_SEC_CAP_TRADES_IDX1 on GAS_SEC_CAP_TRADES
(
   SUBMITTINGCOMPANYID,
   VERSIONDATETIME
);

/*==============================================================*/
/* Index: GAS_SEC_CAP_TRADES_IDX2                               */
/*==============================================================*/
create index GAS_SEC_CAP_TRADES_IDX2 on GAS_SEC_CAP_TRADES
(
   GASDATE
);

/*==============================================================*/
/* Table: GAS_SEC_CAP_TRADES_TRK                                */
/*==============================================================*/
create table GAS_SEC_CAP_TRADES_TRK
(
   SUBMITTINGCOMPANYID  numeric(10,0) not null  comment 'Name of the company submitting the data to AEMO',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record',
   LASTCHANGED          datetime  comment 'Last changed date for the record'
);

alter table GAS_SEC_CAP_TRADES_TRK comment 'Information related to summary trading data, which is derive';

alter table GAS_SEC_CAP_TRADES_TRK
   add primary key (SUBMITTINGCOMPANYID, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_SHIPPER_LIST                                      */
/*==============================================================*/
create table GAS_SHIPPER_LIST
(
   FACILITYID           numeric(20,0) not null  comment 'A unique AEMO defined Facility Identifier',
   EFFECTIVEDATE        datetime not null  comment 'Gas date that corresponding record takes effect',
   VERSIONDATETIME      datetime not null  comment 'Time a successful submission is accepted by AEMO systems',
   SHIPPERNAME          varchar(200) not null  comment 'The name of the shipper who holds the capacity'
);

alter table GAS_SHIPPER_LIST comment 'A list of shippers who have contracted primary Storage, Comp';

alter table GAS_SHIPPER_LIST
   add primary key (FACILITYID, EFFECTIVEDATE, VERSIONDATETIME, SHIPPERNAME);

/*==============================================================*/
/* Table: GAS_SHORT_TERM_CAP_OUTLOOK                            */
/*==============================================================*/
create table GAS_SHORT_TERM_CAP_OUTLOOK
(
   GASDATE              datetime not null  comment 'Date of gas day',
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility Identifier',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record',
   CAPACITYTYPE         varchar(20) not null  comment 'Capacity Type - Must be either STORAGE or MDQ',
   FLOWDIRECTION        varchar(20) not null  comment 'Direction of flow for the submission record. Must be one of RECEIPT or DELIVERY',
   RECEIPTLOCATION      numeric(10,0) not null  comment 'Connection Point ID that best represents the receipt location',
   DELIVERYLOCATION     numeric(10,0) not null  comment 'Connection Point ID that best represents the Delivery Location',
   OUTLOOKQUANTITY      numeric(18,3)  comment 'Quantity of available capacity during the defined period',
   CAPACITYDESCRIPTION  varchar(1000)  comment 'Describes the meaning of the capacity Number Provided',
   DESCRIPTION          varchar(1000)  comment 'Description for reasons or comments directly related to the quantity or change in Outlook quantity in relation to the Facility'
);

alter table GAS_SHORT_TERM_CAP_OUTLOOK comment 'Provides on each gas day D, the BB facility operators good ';

alter table GAS_SHORT_TERM_CAP_OUTLOOK
   add primary key (GASDATE, FACILITYID, VERSIONDATETIME, CAPACITYTYPE, FLOWDIRECTION, RECEIPTLOCATION, DELIVERYLOCATION);

/*==============================================================*/
/* Table: GAS_SHORT_TERM_SWAP_TRANS                             */
/*==============================================================*/
create table GAS_SHORT_TERM_SWAP_TRANS
(
   PERIODSTARTDATE      datetime not null  comment 'The time period start date',
   PERIODENDDATE        datetime  comment 'The time period end date',
   STATE                varchar(5) not null  comment 'The state where the transaction occurred',
   QUANTITY             numeric(18,8)  comment 'Total volume of the transactions where trade date is in the reporting period for the given state',
   VOLUMEWEIGHTEDPRICE  numeric(18,8)  comment 'Volume weighted price of transactions where trade date is in the reporting period for the given State',
   TRANSACTIONTYPE      varchar(255)  comment 'Whether the swap is a location swap, time swap or both location and time swap',
   SUPPLYPERIODSTART    datetime  comment 'The earliest start date of all transactions in the reporting period for the given state',
   SUPPLYPERIODEND      datetime  comment 'The latest end date of all transactions in the reporting period for the given state'
);

alter table GAS_SHORT_TERM_SWAP_TRANS comment 'These reports display information regarding short-term gas s';

alter table GAS_SHORT_TERM_SWAP_TRANS
   add primary key (PERIODSTARTDATE, STATE);

/*==============================================================*/
/* Table: GAS_SHORT_TERM_TRANS                                  */
/*==============================================================*/
create table GAS_SHORT_TERM_TRANS
(
   PERIODSTARTDATE      datetime not null  comment 'The time period start date',
   PERIODENDDATE        datetime  comment 'The time period end date',
   STATE                varchar(5) not null  comment 'The state where the transaction occurred',
   QUANTITY             numeric(18,8)  comment 'Total volume of the transactions where trade date is in the reporting period for the given state',
   VOLUMEWEIGHTEDPRICE  numeric(18,8)  comment 'Volume weighted price of transactions where trade date is in the reporting period for the given State',
   TRANSACTIONTYPE      varchar(255)  comment 'Transaction Type is Supply for these short-term transactions reports',
   SUPPLYPERIODSTART    datetime  comment 'The earliest start date of all transactions in the reporting period for the given state',
   SUPPLYPERIODEND      datetime  comment 'The latest end date of all transactions in the reporting period for the given state'
);

alter table GAS_SHORT_TERM_TRANS comment 'These reports display information regarding short-term gas t';

alter table GAS_SHORT_TERM_TRANS
   add primary key (PERIODSTARTDATE, STATE);

/*==============================================================*/
/* Table: GAS_STATE                                             */
/*==============================================================*/
create table GAS_STATE
(
   STATEID              numeric(10,0) not null  comment 'Unique state identifier',
   STATE                varchar(5)  comment 'State details',
   STATENAME            varchar(50)  comment 'Name of the state',
   LASTCHANGED          datetime  comment 'Last changed date for the record'
);

alter table GAS_STATE comment 'List of states.';

alter table GAS_STATE
   add primary key (STATEID);

/*==============================================================*/
/* Table: GAS_STORAGE_CAPACITY_TRANS                            */
/*==============================================================*/
create table GAS_STORAGE_CAPACITY_TRANS
(
   TRADEID              numeric(20,0) not null  comment 'A unique AEMO defined Transaction Identifier',
   VERSIONDATETIME      datetime not null  comment 'Time a successful submission is accepted by AEMO systems',
   TRADEDATE            datetime  comment 'The date the transaction was entered into',
   FROMGASDATE          datetime  comment 'The start date of the transaction',
   TOGASDATE            datetime  comment 'The end date of the transaction',
   FACILITYID           numeric(20,0)  comment 'The gas storage facility ID for the facility by means of which the service is provided',
   PRIORITY             varchar(255)  comment 'The priority given to the service to which the transaction relates',
   MAXIMUMSTORAGEQUANTITY numeric(18,3)  comment 'The storage capacity the subject of the transaction (in GJ)',
   INJECTIONCAPACITY    numeric(18,3)  comment 'The injection capacity (in GJ/day)',
   WITHDRAWALCAPACITY   numeric(18,3)  comment 'The withdrawal capacity (in GJ/day)',
   PRICE                numeric(18,3)  comment 'The transaction price (in $/GJ/day or where relevant, in $/GJ)',
   PRICESTRUCTURE       varchar(255)  comment 'The price structure applicable to the transaction',
   PRICEESCALATIONMECHANISM varchar(255)  comment 'Any price escalation mechanism applicable to the transaction',
   CANCELLED            numeric(1,0)  comment 'Whether the record has been cancelled',
   LASTCHANGED          datetime  comment 'The date the record was last updated'
);

alter table GAS_STORAGE_CAPACITY_TRANS comment 'Gas storage capacity transactions';

alter table GAS_STORAGE_CAPACITY_TRANS
   add primary key (TRADEID, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_UNCONTRACTED_CAP_OUTLOOK                          */
/*==============================================================*/
create table GAS_UNCONTRACTED_CAP_OUTLOOK
(
   OUTLOOKYEAR          numeric(4,0) not null  comment 'The outlook year',
   OUTLOOKMONTH         numeric(4,0) not null  comment 'The outlook month.',
   FACILITYID           numeric(10,0) not null  comment 'Unique Facility Identifier',
   VERSIONDATETIME      datetime not null  comment 'Version timestamp of the record',
   CAPACITYTYPE         varchar(20) not null  comment 'Capacity Type - Must be either STORAGE or MDQ',
   FLOWDIRECTION        varchar(20) not null  comment 'Direction of flow for the submission record. Must be one of RECEIPT or DELIVERY',
   RECEIPTLOCATION      numeric(10,0) not null  comment 'Connection Point ID that best represents the receipt location',
   DELIVERYLOCATION     numeric(10,0) not null  comment 'Connection Point ID that best represents the Delivery Location',
   OUTLOOKQUANTITY      numeric(18,3)  comment 'Quantity of available capacity during the defined period',
   CAPACITYDESCRIPTION  varchar(1000)  comment 'Describes the meaning of the capacity Number Provided',
   DESCRIPTION          varchar(1000)  comment 'Description for reasons or comments directly related to the quantity or change in Outlook quantity in relation to the Facility'
);

alter table GAS_UNCONTRACTED_CAP_OUTLOOK comment 'Provide the uncontracted capacity of the facility over the n';

alter table GAS_UNCONTRACTED_CAP_OUTLOOK
   add primary key (OUTLOOKYEAR, OUTLOOKMONTH, FACILITYID, VERSIONDATETIME, CAPACITYTYPE, FLOWDIRECTION, RECEIPTLOCATION, DELIVERYLOCATION);

/*==============================================================*/
/* Table: GSH_AUCTION_BID                                       */
/*==============================================================*/
create table GSH_AUCTION_BID
(
   GAS_DATE             datetime not null  comment 'Single Gas Date of Submission',
   PARTICIPANT_CODE     varchar(20) not null  comment 'Unique Participant Code',
   SUBMISSION_TIME      datetime not null  comment 'Date and Time of the submission',
   BID_ID               numeric(8,0) not null  comment 'Unique Bid ID within Bid Profile',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_AUCTION_BID comment 'Information for a single bid contained within a bid profile';

alter table GSH_AUCTION_BID
   add primary key (GAS_DATE, PARTICIPANT_CODE, SUBMISSION_TIME, BID_ID);

/*==============================================================*/
/* Table: GSH_AUCTION_BID_PRODUCT                               */
/*==============================================================*/
create table GSH_AUCTION_BID_PRODUCT
(
   GAS_DATE             datetime not null  comment 'Single Gas Date of Submission',
   PARTICIPANT_CODE     varchar(20) not null  comment 'Unique Participant Code',
   SUBMISSION_TIME      datetime not null  comment 'Date and Time of the submission',
   BID_ID               numeric(8,0) not null  comment 'Unique Bid ID within Bid Profile',
   AUCTION_PRODUCT_TYPE varchar(20) not null  comment 'The product type, for example Forward Haul, Backhaul, Compression',
   RECEIPT_POINT_ID     numeric(8,0) not null  comment 'The unique identifier of the Receipt Point as defined in the Transportation Service Point Register. This is the point where gas is injected into the pipeline',
   RECEIPT_POINT_NAME   varchar(255)  comment 'The descriptive name of the Receipt Point as defined in the Transportation Service Point Register.',
   RP_DWGM_REFERENCE    varchar(255)  comment 'The DWGM accreditation reference associated with the Receipt Point.',
   DELIVERY_POINT_ID    numeric(8,0) not null  comment 'The unique identifier of the Delivery Point as defined in the Transportation Service Point Register. This is the point where gas is withdrawn from the pipeline. Mandatory for service points registered under Part 24 of the rules.',
   DELIVERY_POINT_NAME  varchar(255)  comment 'The descriptive name of the Delivery Point as defined in the Transportation Service Point Register.',
   DP_DWGM_REFERENCE    varchar(255)  comment 'The DWGM accreditation reference associated with the Delivery Point.',
   SERVICE_REFERENCE    varchar(255)  comment 'The service reference issued by the facility operator.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_AUCTION_BID_PRODUCT comment 'Information for a single auction product contained within a ';

alter table GSH_AUCTION_BID_PRODUCT
   add primary key (GAS_DATE, PARTICIPANT_CODE, SUBMISSION_TIME, BID_ID, AUCTION_PRODUCT_TYPE, DELIVERY_POINT_ID, RECEIPT_POINT_ID);

/*==============================================================*/
/* Table: GSH_AUCTION_BID_PROFILE                               */
/*==============================================================*/
create table GSH_AUCTION_BID_PROFILE
(
   GAS_DATE             datetime not null  comment 'Single Gas Date of Submission',
   PARTICIPANT_CODE     varchar(20) not null  comment 'Unique Participant Code',
   SUBMISSION_TIME      datetime not null  comment 'Date and Time of the submission',
   LAST_UPDATED_BY      varchar(20)  comment 'The user name who submitted or updated the bid.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_AUCTION_BID_PROFILE comment 'Top-level table for a profile of auction bids on a given gas';

alter table GSH_AUCTION_BID_PROFILE
   add primary key (GAS_DATE, PARTICIPANT_CODE, SUBMISSION_TIME);

/*==============================================================*/
/* Table: GSH_AUCTION_BID_STEPS                                 */
/*==============================================================*/
create table GSH_AUCTION_BID_STEPS
(
   GAS_DATE             datetime not null  comment 'Single Gas Date of Submission',
   PARTICIPANT_CODE     varchar(20) not null  comment 'Unique Participant Code',
   SUBMISSION_TIME      datetime not null  comment 'Date and Time of the submission',
   BID_ID               numeric(8,0) not null  comment 'Unique Bid ID within Bid Profile',
   STEP_NUMBER          numeric(6,0) not null  comment 'Unique Step ID within the Bid',
   STEP_QUANTITY        numeric(18,8)  comment 'Quantity for this Bid Step',
   STEP_PRICE           numeric(18,8)  comment 'Price for this Bid Step',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_AUCTION_BID_STEPS comment 'Information for a single step contained within a bid.';

alter table GSH_AUCTION_BID_STEPS
   add primary key (GAS_DATE, PARTICIPANT_CODE, SUBMISSION_TIME, BID_ID, STEP_NUMBER);

/*==============================================================*/
/* Table: GSH_AUCTION_BID_SUBMISSION                            */
/*==============================================================*/
create table GSH_AUCTION_BID_SUBMISSION
(
   SUBMISSION_ID        numeric(8,0) not null  comment 'Unique ID for a bid entry',
   PARTICIPANT_CODE     varchar(20)  comment 'Unique Participant Code',
   SUBMISSION_TIME      datetime  comment 'Date and Time of the submission',
   FROM_GAS_DATE        datetime  comment 'The first gas day of the auction bid. Disregard the time component as this is not applicable.',
   TO_GAS_DATE          datetime  comment 'The last gas day of the auction bid. Disregard the time component as this is not applicable.',
   LAST_UPDATED         datetime  comment 'The date and time the bid was submitted or updated.',
   LAST_UPDATED_BY      varchar(100)  comment 'The user name who submitted or updated the bid.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_AUCTION_BID_SUBMISSION comment 'Provides a confirmation to trading participants that their a';

alter table GSH_AUCTION_BID_SUBMISSION
   add primary key (SUBMISSION_ID);

/*==============================================================*/
/* Table: GSH_AUCTION_BID_TRK                                   */
/*==============================================================*/
create table GSH_AUCTION_BID_TRK
(
   AUCTION_DATE         datetime not null  comment 'The date of the auction.',
   VERSION_DATETIME     datetime not null  comment 'The date and time that the auction was run.',
   GAS_DATE             datetime not null  comment 'The gas date for which capacity is assigned to the shipper',
   PARTICIPANT_ID       varchar(20) not null  comment 'Unique GSH Participant ID',
   PARTICIPANT_CODE     varchar(20)  comment 'The unique identifier of the participant',
   BID_ID               numeric(8,0) not null  comment 'Unique Bid ID within Bid Profile',
   SUBMISSION_TIME      datetime  comment 'Date and Time of the submission',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_AUCTION_BID_TRK comment 'List of bids considered for an auction';

alter table GSH_AUCTION_BID_TRK
   add primary key (AUCTION_DATE, GAS_DATE, VERSION_DATETIME, PARTICIPANT_ID, BID_ID);

/*==============================================================*/
/* Table: GSH_AUCTION_CURTAIL_NOTICE                            */
/*==============================================================*/
create table GSH_AUCTION_CURTAIL_NOTICE
(
   NOTICE_ID            numeric(8,0) not null  comment 'The unique identifier of the notice.',
   FACILITY_ID          numeric(8,0) not null  comment 'The unique facility identifier as defined in the Transportation Facility Register.',
   CURTAILED            varchar(3)  comment 'A flag to indicate curtailment',
   MATERIAL             varchar(3)  comment 'Flag to indicate material curtailment',
   DESCRIPTION          varchar(255)  comment 'A description of the curtailment.',
   EFFECTIVE_DATE_TIME  datetime not null  comment 'The date and time the curtailment is effective from.',
   FACILITY_NAME        varchar(100)  comment 'The descriptive name of the facility.',
   AUCTION_SERVICE      varchar(20) not null  comment 'The auction service impacted by curtailment',
   GAS_DATE             datetime not null  comment 'The gas date for which the auction service is curtailed.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_AUCTION_CURTAIL_NOTICE comment 'Contains details of curtailment of nominations against secon';

alter table GSH_AUCTION_CURTAIL_NOTICE
   add primary key (NOTICE_ID, FACILITY_ID, AUCTION_SERVICE, EFFECTIVE_DATE_TIME, GAS_DATE);

/*==============================================================*/
/* Table: GSH_AUCTION_PRICE_VOLUME                              */
/*==============================================================*/
create table GSH_AUCTION_PRICE_VOLUME
(
   AUCTION_DATE         datetime not null  comment 'The date of the auction run. Disregard the time component as this is not applicable.',
   GAS_DATE             datetime not null  comment 'The gas date that the auction quantities and prices are valid for. Disregard the time component as this is not applicable.',
   FACILITY_ID          numeric(8,0) not null  comment 'The unique identifier of the facility.',
   FACILITY_NAME        varchar(100)  comment 'The descriptive name of the facility.',
   LIMIT_ID             numeric(8,0) not null  comment 'The unique identifier of the auction limit.',
   LIMIT_NAME           varchar(100)  comment 'The descriptive name of the auction limit.',
   LIMIT_TYPE           varchar(100)  comment 'The type of auction limit',
   CLEARED_QUANTITY     numeric(12,4)  comment 'The capacity quantity cleared at auction (GJ/day).',
   CLEARED_PRICE        numeric(12,4)  comment 'The cleared auction price ($/GJ).',
   HIGH_PRICE_SENSITIVITY numeric(12,4)  comment 'The high price sensitivity factor applied to the price.',
   LOW_PRICE_SENSITIVITY numeric(12,4)  comment 'The low price sensitivity factor applied to the price.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_AUCTION_PRICE_VOLUME comment 'The cleared quantity, cleared price and price sensitivity fo';

alter table GSH_AUCTION_PRICE_VOLUME
   add primary key (AUCTION_DATE, GAS_DATE, FACILITY_ID, LIMIT_ID);

/*==============================================================*/
/* Table: GSH_AUCTION_QUANTITIES                                */
/*==============================================================*/
create table GSH_AUCTION_QUANTITIES
(
   AUCTION_DATE         datetime not null  comment 'The date of the auction. Disregard the time component as this is not applicable.',
   GAS_DATE             datetime not null  comment 'The gas date that the auction quantities and transitional rights are valid for. Disregard the time component as this is not applicable.',
   FACILITY_ID          numeric(8,0) not null  comment 'The unique identifier of the facility.',
   FACILITY_NAME        varchar(100)  comment 'The descriptive name of the facility.',
   LIMIT_ID             numeric(8,0) not null  comment 'The unique identifier of the auction limit.',
   LIMIT_NAME           varchar(100)  comment 'The descriptive name of the auction limit.',
   LIMIT_TYPE           varchar(100) not null  comment 'The type of auction limit',
   LIMIT_QUANTITY       numeric(10,0)  comment 'The available auction quantity of the limit in GJ.',
   TRANSITIONAL_RIGHTS_QUANTITY numeric(10,0)  comment 'The quantity of nominations against grandfathered rights in GJ, for the previous gas day.',
   OPERATIONAL_OVERRIDE varchar(3)  comment 'Flag to indicate whether the facility operator has applied an operational override to the limit quantity, for example Yes or No.',
   VERSION_DATETIME     datetime  comment 'The date and time that the auction was run.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_AUCTION_QUANTITIES comment 'Auction limit quantities for each product component availabl';

alter table GSH_AUCTION_QUANTITIES
   add primary key (AUCTION_DATE, GAS_DATE, FACILITY_ID, LIMIT_ID, LIMIT_TYPE);

/*==============================================================*/
/* Table: GSH_AUCTION_RESULTS                                   */
/*==============================================================*/
create table GSH_AUCTION_RESULTS
(
   AUCTION_TRANSACTION_ID numeric(12,0) not null  comment 'The unique identifier of the auction transaction.',
   AUCTION_DATE         datetime  comment 'The date of the auction. Disregard the time component as this is not applicable.',
   FACILITY_ID          numeric(8,0)  comment 'The unique facility identifier as defined in the Transportation Facility Register.',
   FACILITY_NAME        varchar(100)  comment 'The descriptive name of the facility as defined in the Transportation Facility Register.',
   PARTICIPANT_ID       varchar(20)  comment 'The unique AEMO defined GSH Participant identifier.',
   PARTICIPANT_CODE     varchar(20)  comment 'The unique identifier of the participant',
   PARTICIPANT_NAME     varchar(80)  comment 'The descriptive name of the participant.',
   SERVICE_REFERENCE    varchar(255)  comment 'The service reference issued by the facility operator.',
   RECEIPT_POINT_ID     numeric(8,0)  comment 'The unique identifier of the Receipt Point as defined in the Transportation Service Point Register. This is the point where gas is injected into the pipeline',
   RECEIPT_POINT_NAME   varchar(255)  comment 'The descriptive name of the Receipt Point as defined in the Transportation Service Point Register.',
   RP_DWGM_REFERENCE    varchar(255)  comment 'The DWGM accreditation reference associated with the Receipt Point.',
   DELIVERY_POINT_ID    numeric(8,0)  comment 'The unique identifier of the Delivery Point as defined in the Transportation Service Point Register. This is the point where gas is withdrawn from the pipeline. Mandatory for service points registered under Part 24 of the rules.',
   DELIVERY_POINT_NAME  varchar(255)  comment 'The descriptive name of the Delivery Point as defined in the Transportation Service Point Register.',
   DP_DWGM_REFERENCE    varchar(255)  comment 'The DWGM accreditation reference associated with the Delivery Point.',
   AUCTION_QUANTITY_WON numeric(8,0)  comment 'The capacity quantity won at auction (GJ/day) by the participant.',
   GAS_DATE             datetime  comment 'The gas date for which capacity is assigned to the shipper, in ISO 8601 datetime format (yyyy-mm-ddThh:mm:ss).',
   PRODUCT_PRICE        numeric(18,8)  comment 'The price of the auction product won at auction ($/GJ) by the participant.',
   VERSION_DATETIME     datetime  comment 'The date and time that the auction was run.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_AUCTION_RESULTS comment 'Specifies the details of the capacity won at auction';

alter table GSH_AUCTION_RESULTS
   add primary key (AUCTION_TRANSACTION_ID);

/*==============================================================*/
/* Table: GSH_AUCTION_RUN                                       */
/*==============================================================*/
create table GSH_AUCTION_RUN
(
   AUCTION_DATE         datetime not null  comment 'The date of the auction.',
   GAS_DATE             datetime not null  comment 'The gas date for which capacity is assigned to the shipper.',
   AUCTION_RUN_DATETIME datetime not null  comment 'The date and time that the auction was run.',
   PARTICIPANT_ID       varchar(20) not null  comment 'Unique GSH Participant ID',
   PARTICIPANT_CODE     varchar(20)  comment 'The unique identifier of the participant',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_AUCTION_RUN comment 'Specifies the Auction Run has completed, issued to any parti';

alter table GSH_AUCTION_RUN
   add primary key (AUCTION_DATE, GAS_DATE, AUCTION_RUN_DATETIME, PARTICIPANT_ID);

/*==============================================================*/
/* Table: GSH_BENCHMARK_PRICE                                   */
/*==============================================================*/
create table GSH_BENCHMARK_PRICE
(
   GAS_DATE             datetime not null  comment 'The gas date that the Wallumbilla benchmark price applies to',
   PRODUCT_LOCATION     varchar(80) not null  comment 'The product location that the benchmark price applies to. Can be one of the following: WALLUMBILLA, RBP, SWQP, QGP.',
   PRODUCT_TYPE         varchar(80) not null  comment 'The type of the product that the Wallumbilla benchmark price is calculated for',
   BENCHMARK_PRICE      numeric(18,8)  comment 'The calculated Wallumbilla benchmark price for the specified product type for the specified gas date',
   IS_FIRM              varchar(1)  comment 'Flag to determine if the Wallumbilla benchmark price is firm or not. Can be either: Y or N',
   LASTCHANGED          datetime  comment 'The date and time the report was generated'
);

alter table GSH_BENCHMARK_PRICE comment 'The report contains the benchmark prices for the day ahead (';

alter table GSH_BENCHMARK_PRICE
   add primary key (GAS_DATE, PRODUCT_LOCATION, PRODUCT_TYPE);

/*==============================================================*/
/* Table: GSH_CAPACITY_TRANSACTION                              */
/*==============================================================*/
create table GSH_CAPACITY_TRANSACTION
(
   TRANSACTION_ID       varchar(50) not null  comment 'Unique capacity transaction id',
   VERSION_DATETIME     datetime not null  comment 'Updated version within transaction id',
   TRADE_DATE           datetime  comment 'The date the trade was made. Disregard the time component as this is not applicable.',
   FROM_GAS_DATE        datetime  comment 'The effective start date of the trade. Disregard the time component as this is not applicable.',
   TO_GAS_DATE          datetime  comment 'The effective end date of the trade. Disregard the time component as this is not applicable.',
   TRADE_TYPE           varchar(30)  comment 'The type of trade',
   FACILITY_ID          numeric(10,0)  comment 'The unique identifier of the Part 24 facility. Mandatory for facilities registered under Part 24 of the rules.',
   FACILITY_NAME        varchar(100)  comment 'The descriptive name of the facility.',
   TRANSPORTATION_SERVICE_TYPE varchar(12)  comment 'The transportation service type',
   FLOW_DIRECTION       varchar(20)  comment 'The direction of the flow for non-Part 24 facilities, relative to the general physical direction of the facility. Valid values are:NORTHNORTH_EASTNORTH_WESTEASTSOUTH_EASTSOUTHSOUTH_WESTWEST',
   RECEIPT_ZONE_ID      numeric(10,0)  comment 'The unique identifier of the Part 24 Receipt Zone. This is the zone in which the receipt point that was reported to AEMO is specified. Mandatory for service points registered under Part 24 of the rules.',
   RECEIPT_ZONE_NAME    varchar(255)  comment 'The descriptive name of the Receipt Zone.',
   DELIVERY_ZONE_ID     numeric(10,0)  comment 'The unique identifier of the Part 24 Delivery Zone. This is the zone in which the delivery point that was reported to AEMO is specified. Mandatory for service points registered under Part 24 of the rules.',
   DELIVERY_ZONE_NAME   varchar(255)  comment 'The descriptive name of the Delivery Zone.',
   PARK_LOAN_POINT_ID   numeric(10,0)  comment 'The unique identifier of the Part 24 Park and Loan service point. Mandatory for service points registered under Part 24 of the rules.',
   PARK_LOAN_POINT_NAME varchar(255)  comment 'The descriptive name of the service point.',
   STANDARD_OTSA        varchar(255)  comment 'Flag to indicate whether a standard OTSA was used',
   PRIORITY             varchar(255)  comment 'The priority of the traded capacity. Mandatory for GBB bilateral trades. Null for GSH trades.',
   MAX_DAILY_QUANTITY   numeric(10,0)  comment 'The traded maximum daily quantity (MDQ) (GJ/day).',
   MAX_HOURLY_QUANTITY  numeric(10,0)  comment 'The traded maximum hourly quantity (GJ/hour). Null for GSH trades.',
   PRICE                numeric(18,8)  comment 'The price of the traded capacity ($/GJ/day).',
   PRICE_STRUCTURE      varchar(255)  comment 'The price structure applied over the term of the trade. Mandatory for GBB bilateral trades. Null for GSH trades.',
   PRICE_ESCALATION_MECHANISM varchar(255)  comment 'The price escalation mechanism applied over the term of the trade. Null for GSH trades.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_CAPACITY_TRANSACTION comment 'Consolidated list of all secondary capacity trades which inc';

alter table GSH_CAPACITY_TRANSACTION
   add primary key (TRANSACTION_ID, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_CAP_XFR_AUCTION_NOTICE                            */
/*==============================================================*/
create table GSH_CAP_XFR_AUCTION_NOTICE
(
   NOTICE_ID            numeric(8,0) not null  comment 'The unique identifier of the notice.',
   NOTICE_DATE          datetime  comment 'The date of the notice',
   GAS_DATE             datetime  comment 'The gas date of the notice. Disregard the time component as this is not applicable.',
   FACILITY_ID          numeric(8,0)  comment 'The unique identifier of the facility.',
   FACILITY_NAME        varchar(100)  comment 'The descriptive name of the facility.',
   MARKET               varchar(20)  comment 'The market that has triggered the notice, for example CTP or DAA.',
   NOTICE_TYPE          varchar(50)  comment 'The type of notice, for example Cancel or Delay for CTP, Exclude or Delay for DAA.',
   DESCRIPTION          varchar(1000)  comment 'The description of the notice.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_CAP_XFR_AUCTION_NOTICE comment 'Notifies participants when the market delay or cancellation ';

alter table GSH_CAP_XFR_AUCTION_NOTICE
   add primary key (NOTICE_ID);

/*==============================================================*/
/* Table: GSH_CAP_XFR_NOTICE                                    */
/*==============================================================*/
create table GSH_CAP_XFR_NOTICE
(
   CAPACITY_TRANSFER_ID varchar(20) not null  comment 'The unique AEMO defined Capacity Transfer transaction identifier.',
   FACILITY_ID          numeric(8,0)  comment 'The unique facility identifier as defined in the Transportation Facility Register.',
   FACILITY_NAME        varchar(100)  comment 'The descriptive name of the facility as defined in the Transportation Facility Register.',
   PARTICIPANT_ID       varchar(20)  comment 'The unique AEMO defined GSH Participant identifier.',
   PARTICIPANT_NAME     varchar(80)  comment 'The descriptive name of the participant.',
   SERVICE_REFERENCE    varchar(255)  comment 'The service reference issued by the facility operator.',
   STTM_CONTRACT_REFERENCE varchar(255)  comment 'The STTM registered facility service identifier (CRN) associated with the service reference.',
   RECEIPT_POINT_ID     numeric(8,0)  comment 'The unique identifier of the Receipt Point as defined in the Transportation Service Point Register. This is the point where gas is injected into the pipeline',
   RECEIPT_POINT_NAME   varchar(255)  comment 'The descriptive name of the Receipt Point as defined in the Transportation Service Point Register.',
   DELIVERY_POINT_ID    numeric(8,0)  comment 'The unique identifier of the Delivery Point as defined in the Transportation Service Point Register. This is the point where gas is withdrawn from the pipeline.',
   DELIVERY_POINT_NAME  varchar(255)  comment 'The descriptive name of the Delivery Point as defined in the Transportation Service Point Register.',
   RECEIPT_ZONE_ID      numeric(8,0)  comment 'The unique identifier of the Receipt Zone as defined in the Transportation Service Point Register.',
   RECEIPT_ZONE_NAME    varchar(255)  comment 'The descriptive name of the Receipt Zone as defined in the Transportation Service Point Register.',
   DELIVERY_ZONE_ID     numeric(8,0)  comment 'The unique identifier of the Delivery Zone as defined in the Transportation Service Point Register.',
   DELIVERY_ZONE_NAME   varchar(255)  comment 'The descriptive name of the Delivery Zone as defined in the Transportation Service Point Register.',
   GAS_DATE             datetime  comment 'The gas date that the capacity change applies to. Disregard the time component as this is not applicable.',
   INCREMENTAL_QUANTITY_CHANGE numeric(10,0)  comment 'The incremental capacity quantity adjustment (GJ/day) to be applied to the service reference for this transfer date and gas date. This is a signed value (+/-).',
   OVERALL_QUANTITY_CHANGE numeric(10,0)  comment 'The overall capacity quantity adjustment (GJ/day) applied to the relevant contracts for this transfer date and gas date. This is a signed value (+/-).',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_CAP_XFR_NOTICE comment 'Capacity transfer notice includes a list of capacity quantit';

alter table GSH_CAP_XFR_NOTICE
   add primary key (CAPACITY_TRANSFER_ID);

/*==============================================================*/
/* Table: GSH_CONTRACT_DETAILS                                  */
/*==============================================================*/
create table GSH_CONTRACT_DETAILS
(
   PARTICIPANT_ID       varchar(20) not null  comment 'The unique AEMO defined GSH Participant identifier.',
   PARTICIPANT_NAME     varchar(80)  comment 'The descriptive name of the participant.',
   PARTICIPANT_CODE     varchar(20)  comment 'The participant code used in the ETS.',
   FACILITY_ID          numeric(8,0) not null  comment 'The unique facility identifier as defined in the Transportation Facility Register.',
   FACILITY_NAME        varchar(100)  comment 'The descriptive name of the facility as defined in the Transportation Facility Register.',
   SERVICE_REFERENCE    varchar(255) not null  comment 'The service reference issued by the facility operator.',
   SERVICE_REFERENCE_DESCRIPTION varchar(255)  comment 'Brief description of the service reference.',
   STTM_CONTRACT_REFERENCE varchar(255)  comment 'The STTM registered facility service identifier (CRN) associated with the service reference if applicable.',
   CONTRACT_TYPE        varchar(20)  comment 'Indicates if a contract can be used for capacity trades, auctions or both. For example, Trade, Auction or All.',
   START_DATE           datetime  comment 'Effective start date of the contract. Disregard the time component as this is not applicable.',
   END_DATE             datetime  comment 'Effective end date of the contract. Disregard the time component as this is not applicable.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_CONTRACT_DETAILS comment 'Provides participants with a list of all the facility operat';

alter table GSH_CONTRACT_DETAILS
   add primary key (PARTICIPANT_ID, FACILITY_ID, SERVICE_REFERENCE);

/*==============================================================*/
/* Table: GSH_CONTRACT_REFERENCES                               */
/*==============================================================*/
create table GSH_CONTRACT_REFERENCES
(
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code used in the ETS.',
   PRODUCT_NAME         varchar(80) not null  comment 'The capacity product name.',
   BUY_SELL_POSITION    varchar(4) not null  comment 'Specifies the position that the contract reference must be used for, for example Buy, Sell or Both.',
   FROM_GAS_DATE        datetime not null  comment 'The first gas date that the contract reference takes effect from. Disregard the time component as this is not applicable.',
   TO_GAS_DATE          datetime not null  comment 'The last gas date that the contract reference takes effect until. Disregard the time component as this is not applicable.',
   STEP_NUMBER          numeric(6,0) not null  comment 'The step number for the corresponding capacity quantity.',
   SERVICE_REFERENCE    varchar(255)  comment 'The service reference issued by the facility operator.',
   SERVICE_REFERENCE_STATUS varchar(20)  comment 'The status of the service reference, for example Valid, Missing or Expired.',
   STTM_CONTRACT_REFERENCE varchar(255)  comment 'The STTM registered facility service identifier (CRN) associated with the service reference.',
   STTM_CONTRACT_REFERENCE_STATUS varchar(20)  comment 'The status of the STTM contract reference, for example Valid, Missing or Expired.',
   DWGM_REFERENCE       varchar(255)  comment 'The DWGM accreditation reference associated with the transfer.',
   DWGM_REFERENCE_STATUS varchar(20)  comment 'The status of the DWGM accreditation reference, for example Valid, Missing or Expired.',
   STEP_QUANTITY        numeric(10,0)  comment 'The step quantity to applied to the contract reference (GJ/day).',
   VERSION_DATETIME     datetime not null  comment 'The date and time the preference was updated.',
   LAST_UPDATED         datetime  comment 'The date and time the preference was updated.',
   LAST_UPDATED_BY      varchar(20)  comment 'The user name who updated the preference.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_CONTRACT_REFERENCES comment 'This report is available to participants as an audit trail o';

alter table GSH_CONTRACT_REFERENCES
   add primary key (PARTICIPANT_CODE, PRODUCT_NAME, BUY_SELL_POSITION, FROM_GAS_DATE, TO_GAS_DATE, STEP_NUMBER, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_DELIVERED_QUANTITY                                */
/*==============================================================*/
create table GSH_DELIVERED_QUANTITY
(
   GAS_DATE             datetime not null  comment 'The delivery gas day of the trade',
   TRANSACTION_ID       varchar(20) not null  comment 'Unique identifier for the transaction',
   VERSION_DATETIME     datetime not null  comment 'The date and time of when the entry was last modified in the source table.',
   BUYER_PARTICIPANT_CODE varchar(20)  comment 'The participant code used in the Exchange Trading System (ETS)',
   BUYER_PARTICIPANT_NAME varchar(80)  comment 'The name for the participant on buy side of the trade',
   SELLER_PARTICIPANT_CODE varchar(20)  comment 'The participant code used in the Exchange Trading System (ETS) for the participant on the seller side of the trade',
   SELLER_PARTICIPANT_NAME varchar(80)  comment 'The name for the participant on seller side of the trade',
   PRODUCT_LOCATION     varchar(80)  comment 'The product location for the Gas Supply Hub market',
   TRANSACTION_QUANTITY numeric(18,8)  comment 'Initial gas volume of the transaction in GJ/day',
   SETTLEMENTS_OFF_MARKET varchar(1)  comment 'Indicates whether the buyer and seller wish to settle the delivery variance outside of the GSH market. Valid values are: Y, N',
   DELIVERED_QUANTITY   numeric(18,8)  comment 'The delivered quantity in GJ/day',
   REASON_FOR_VARIATION varchar(80)  comment 'The reason for the variance in the delivered quantity',
   STATUS               varchar(20)  comment 'The status of the delivery confirmation. Valid values are: SUBMITTED, CONFIRMED',
   LAST_UPDATED_BY_PARTICIPANT varchar(20)  comment 'The participant code used in the Exchange Trading System (ETS) for the participant who last updated the delivered quantity record',
   LAST_UPDATED         datetime  comment 'The date and time the delivered quantity was updated i.e. saved into database',
   LASTCHANGED          datetime  comment 'The date and time the report was generated',
   ETS_TRADE_ID         varchar(20)  comment 'The exchange transaction ID associated with the delivery obligation for transactions in non-netted products. This field can be cross referenced with the TRADE_ID field in the Trade Execution report to identify the transaction price. 
             This field is empty for trades involving netted products'
);

alter table GSH_DELIVERED_QUANTITY comment 'The purpose of this report is to provide the Participants (b';

alter table GSH_DELIVERED_QUANTITY
   add primary key (GAS_DATE, TRANSACTION_ID, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_DELIVERY_NETT_POINTS                              */
/*==============================================================*/
create table GSH_DELIVERY_NETT_POINTS
(
   MARKET_ID            varchar(20) not null  comment 'Unique Market ID',
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code',
   PRODUCT_LOCATION     varchar(80) not null  comment 'Product location for the GSH market',
   VERSION_DATETIME     datetime not null  comment 'Date and time preference entered',
   POSITION_TYPE        varchar(20) not null  comment 'Type of position used. Valid values are: NET, BUY, SELL',
   PRIORITY             numeric(3,0) not null  comment 'Order of priority for overriding delivery points',
   DELIVERY_POINT       varchar(40)  comment 'Name of delivery point',
   VOLUME               numeric(18,8)  comment 'Amount (Gj) to assign to delivery point',
   LAST_UPDATED         datetime  comment 'The date and time the preference was updated',
   LASTCHANGED          datetime  comment 'The date and time the report was generated'
);

alter table GSH_DELIVERY_NETT_POINTS comment 'List of delivery point preferences if delivery points are ov';

alter table GSH_DELIVERY_NETT_POINTS
   add primary key (MARKET_ID, PARTICIPANT_CODE, PRODUCT_LOCATION, VERSION_DATETIME, POSITION_TYPE, PRIORITY);

/*==============================================================*/
/* Table: GSH_DELIVERY_NETT_PREFS                               */
/*==============================================================*/
create table GSH_DELIVERY_NETT_PREFS
(
   MARKET_ID            varchar(20) not null  comment 'Unique Market ID',
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code',
   PRODUCT_LOCATION     varchar(80) not null  comment 'Product location for the GSH market',
   VERSION_DATETIME     datetime not null  comment 'Date and time preference entered',
   START_DATE           datetime  comment 'Date preference takes effect from',
   END_DATE             datetime  comment 'Date preference takes effect until',
   CHOSEN_POSITION      varchar(20)  comment 'Type of position used. Valid values are: NET, BUY/SELL',
   USE_TRANSACTIONS     varchar(1)  comment 'Whether original delivery points specified in trades will be used. Valid values are Y | N',
   USE_GROUPS           varchar(1)  comment 'Indicates if positions are matched using groups. Valid values are Y | N',
   LAST_UPDATED         datetime  comment 'The date and time the preference was updated',
   LAST_UPDATED_BY      varchar(20)  comment 'The user name who updated the preference',
   LASTCHANGED          datetime  comment 'The date and time the report was generated'
);

alter table GSH_DELIVERY_NETT_PREFS comment 'Provide a history of Delivery Netting Preferences entered th';

alter table GSH_DELIVERY_NETT_PREFS
   add primary key (MARKET_ID, PARTICIPANT_CODE, PRODUCT_LOCATION, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_DELIVERY_OBLIGATIONS                              */
/*==============================================================*/
create table GSH_DELIVERY_OBLIGATIONS
(
   NETTING_TRANSACTION_ID varchar(20) not null  comment 'The unique identifier of the delivery transaction.',
   BUYER_PARTICIPANT_CODE varchar(20)  comment 'The unique identifier for the participant on buy side of the trade that has an obligation to receipt gas
             OR
             The unique code for the participant with net zero position.',
   BUYER_PARTICIPANT_NAME varchar(80)  comment 'The name for the participant on buy side of the trade that has an obligation to receipt gas
             OR
             The name of the participant with net zero position.',
   SELLER_PARTICIPANT_CODE varchar(20)  comment 'The unique identifier for the participant on seller side of the trade that has an obligation to deliver gas
             OR
             The unique code for the participant with net zero position.',
   SELLER_PARTICIPANT_NAME varchar(80)  comment 'The name for the participant on seller side of the trade that has an obligation to deliver gas
             OR
             The name of the participant with net zero position.',
   FROM_GAS_DATE        datetime  comment 'The first gas date in the netting period.',
   TO_GAS_DATE          datetime  comment 'The last gas date of the netting period.',
   PRODUCT_LOCATION     varchar(80)  comment 'The product location, product grouping code for the Gas Supply Hub market',
   DELIVERY_QUANTITY    numeric(18,8)  comment 'The quantity of the delivery transaction GJ/day
             OR
             Zero if the participant has a zero net position.',
   DELIVERY_POINT       varchar(40)  comment 'Delivery point/s and associated quantities determined as part of SUC7407a  Perform Participant Location Netting.
             OR
             NULL if the participant has a zero net position.',
   DELIVERY_TYPE_ALERT  varchar(20)  comment 'This field is empty if the transaction is an output of the delivery-netting module.
             If the transaction is an original executed trade (due to a system failure), this field shows Individual Contract Delivery.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated',
   ETS_TRADE_ID         varchar(20)  comment 'The exchange transaction ID associated with the delivery obligation for transactions in non-netted products. This field can be cross referenced with the TRADE_ID field in the Trade Execution report to identify the transaction price. This field is empty for trades involving netted products.',
   RECEIPT_POINT        varchar(40)  comment 'Receipt point specified by the buyer or seller in an order submission for capacity products'
);

alter table GSH_DELIVERY_OBLIGATIONS comment 'The purpose of this report is to provide the Participants wi';

alter table GSH_DELIVERY_OBLIGATIONS
   add primary key (NETTING_TRANSACTION_ID);

/*==============================================================*/
/* Table: GSH_FACILITIES                                        */
/*==============================================================*/
create table GSH_FACILITIES
(
   FACILITY_ID          numeric(8,0) not null  comment 'The unique identifier of the facility.',
   EFFECTIVEDATE        datetime not null  comment 'Effective date of the record',
   VERSION_DATETIME     datetime not null  comment 'Version timestamp of the record',
   FACILITY_NAME        varchar(100)  comment 'The name of the facility.',
   FACILITY_DESCRIPTION varchar(400)  comment 'The descriptive name of the facility.',
   FACILITY_TYPE        varchar(20)  comment 'The type of facility, for example pipe or compressor.',
   OPERATOR_ID          varchar(20)  comment 'The unique identifier of the Facility Operator that operates the facility.',
   OPERATOR_NAME        varchar(100)  comment 'The descriptive name of the participant.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_FACILITIES comment 'Facilities currently registered, for capacity trading and da';

alter table GSH_FACILITIES
   add primary key (FACILITY_ID, EFFECTIVEDATE, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_HISTORICAL_SUMMARY                                */
/*==============================================================*/
create table GSH_HISTORICAL_SUMMARY
(
   GAS_DATE             datetime not null  comment 'The gas date when the delivery of trades occurred',
   PRODUCT_LOCATION     varchar(80) not null  comment 'Product location for the Gas Supply Hub market',
   VOLUME_WEIGHTED_AVERAGE_PRICE numeric(18,8)  comment 'The volume weighted average price for trades delivered in the specified gas day at the specified location in $/GJ',
   TOTAL_QUANTITY       numeric(18,8)  comment 'Total traded quantities for the specified gas day at the specified location in GJ/day',
   ROLLING_WEIGHTED_AVERAGE_PRICE numeric(18,8)  comment 'The rolling 30 days volume weighted average price for trades delivered in the last 30 days at the specified location in $/GJ',
   LASTCHANGED          datetime  comment 'The date and time the report was generated'
);

alter table GSH_HISTORICAL_SUMMARY comment 'The report provides a summary of all trades delivered in the';

alter table GSH_HISTORICAL_SUMMARY
   add primary key (GAS_DATE, PRODUCT_LOCATION);

/*==============================================================*/
/* Table: GSH_NOTIONAL_POINTS                                   */
/*==============================================================*/
create table GSH_NOTIONAL_POINTS
(
   FACILITY_ID          numeric(8,0) not null  comment 'Unique Facility ID for this market',
   SERVICE_POINT_ID     numeric(8,0) not null  comment 'The unique identifier of the service point.',
   EFFECTIVEDATE        datetime not null  comment 'The effective date of the notional point.',
   VERSION_DATETIME     datetime not null  comment 'The version timestamp of the record',
   SERVICE_POINT_NAME   varchar(100)  comment 'The descriptive name of the service point.',
   PHYSICAL_POINT_NAME  varchar(100) not null  comment 'The descriptive name of the physical point associated with the notional point.',
   GBB_CONNECTION_POINT_ID numeric(8,0)  comment 'The corresponding GBB connection point identifier.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_NOTIONAL_POINTS comment 'Notional Point currently registered under Part 24 of the Rul';

alter table GSH_NOTIONAL_POINTS
   add primary key (FACILITY_ID, EFFECTIVEDATE, VERSION_DATETIME, SERVICE_POINT_ID, PHYSICAL_POINT_NAME);

/*==============================================================*/
/* Table: GSH_ORDERS                                            */
/*==============================================================*/
create table GSH_ORDERS
(
   MARKET_ID            varchar(20) not null  comment 'Unique Market ID',
   ORDER_ID             varchar(20) not null  comment 'Unique order id generated by the Trading Platform on submission of the order',
   PARTICIPANT_CODE     varchar(20)  comment 'The participant code used in the Exchange Trading System (ETS)',
   PARTICIPANT_NAME     varchar(80)  comment 'The name of the participant that the order belongs to',
   PRODUCT_LOCATION     varchar(80)  comment 'Product location for the Gas Supply Hub market',
   PRODUCT_TYPE         varchar(80)  comment 'The product delivery period for the Gas Supply Hub market',
   FROM_GAS_DATE        datetime  comment 'The start gas day for the order delivery period',
   TO_GAS_DATE          datetime  comment 'The end gas day for the order delivery period',
   ORDER_DATE           datetime  comment 'The date and time that the order is submitted on',
   ORDER_STATUS         varchar(20)  comment 'The status of the order. Valid values are: NEW, UPDATE, CANCELLED',
   ORDER_SIDE           varchar(5)  comment 'This field is to indicate whether the order is a Bid or Offer',
   ORDER_QUANTITY       numeric(18,8)  comment 'Gas volume of the order in GJ/day',
   ORDER_PRICE          numeric(18,8)  comment 'Price value of the order in $/GJ',
   EXPIRY_TYPE          varchar(80)  comment 'Order expiry type, for example (Good Till Cancelled, Good For Day, Good Till Date)',
   EXPIRY_DATE          datetime  comment 'Date and time the order will expire. Only applies to "Good_Till_Date" expiry type',
   DELIVERY_POINT       varchar(40)  comment 'Delivery Point specified by the seller in an order submission for non-netted commodity products, or the buyer or seller in an order submission for capacity products',
   LAST_UPDATED         datetime  comment 'The date and time the bid/offer was updated i.e. saved into database',
   LAST_UPDATED_BY      varchar(20)  comment 'The user name used in the bid/offer submission',
   LASTCHANGED          datetime  comment 'The date and time the report was generated',
   ALL_OR_NONE          numeric(1,0)  comment 'Populated if the trading participant selects the "All or None" option when submitting an order',
   DURATION             varchar(20)  comment 'Time from order submission until expiry. Data has day format (ISO 8601)',
   HIDDEN_VOLUME        numeric(18,8)  comment 'Populated if the trading participant specifies a "Hidden Volume" when submitting an order',
   RECEIPT_POINT        varchar(40)  comment 'Receipt Point specified by the buyer or seller in an order submission for capacity products',
   VALID_SERVICE_REFERENCE varchar(3)  comment 'Flag to indicate whether a valid service reference exists for the capacity product. Will be NULL for orders involving commodity products',
   VALID_STTM_CONTRACT_REFERENCE varchar(3)  comment 'Flag to indicate whether a valid STTM contract reference exists for the capacity product. Will be NULL for orders involving commodity products and non-STTM integrated capacity products',
   VALID_DWGM_REFERENCE varchar(3)  comment 'Flag to indicate whether a valid DWGM accreditation reference exists for the capacity product where a DWGM interface point has been specified. Will be NULL for orders involving commodity products and capacity products that do not have a DWGM interface point specified'
);

alter table GSH_ORDERS comment 'The purpose of this report is to provide the Participant a c';

alter table GSH_ORDERS
   add primary key (MARKET_ID, ORDER_ID);

/*==============================================================*/
/* Table: GSH_PARK_SERVICES                                     */
/*==============================================================*/
create table GSH_PARK_SERVICES
(
   FACILITY_ID          numeric(8,0) not null  comment 'Unique Facility ID for this market',
   EFFECTIVEDATE        datetime not null  comment 'The effective date of the park service.',
   VERSION_DATETIME     datetime not null  comment 'The version timestamp of the record',
   PARK_PRODUCT_NAME    varchar(100) not null  comment 'The descriptive name of the park product.',
   PARK_SERVICE_POINT_ID numeric(8,0)  comment 'The unique identifier of the service point at which transportation capacity is required to use this park service.',
   PARK_SERVICE_POINT_NAME varchar(100)  comment 'The descriptive name of the park service point.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_PARK_SERVICES comment 'Park Services currently registered under Part 24 of the Rule';

alter table GSH_PARK_SERVICES
   add primary key (FACILITY_ID, EFFECTIVEDATE, VERSION_DATETIME, PARK_PRODUCT_NAME);

/*==============================================================*/
/* Table: GSH_PARTICIPANTS                                      */
/*==============================================================*/
create table GSH_PARTICIPANTS
(
   PARTICIPANT_ID       varchar(20) not null  comment 'The unique identifier of the participant',
   PARTICIPANT_CODE     varchar(20)  comment 'The participant code',
   ORGANISATION_NAME    varchar(80)  comment 'Name of the organisation who has the ABN',
   ORGANISATION_CODE    varchar(20)  comment 'Unique code of the organisation',
   TRADING_NAME         varchar(80)  comment 'Trading name of the organisation',
   HEAD_OFFICE_ADDRESS_LINE_1 varchar(80)  comment 'The head office address of the organisation',
   HEAD_OFFICE_ADDRESS_LINE_2 varchar(80)  comment 'The head office address of the organisation',
   HEAD_OFFICE_ADDRESS_LINE_3 varchar(80)  comment 'The head office address of the organisation',
   HEAD_OFFICE_ADDRESS_LINE_4 varchar(80)  comment 'The head office address of the organisation',
   PHONE_NUMBER         varchar(80)  comment 'The phone number of the organisation',
   FAX_NUMBER           varchar(80)  comment 'The fax number for the organisation',
   ABN                  varchar(20)  comment 'ABN of the organisation',
   ARBN                 varchar(20)  comment 'ARBN of the organisation',
   ACN                  varchar(20)  comment 'ACN of the organisation',
   CLASSIFICATIONS      varchar(255)  comment 'The classification of the company in the GSH market, for example trading, view-only, etc.',
   LASTCHANGED          datetime  comment 'The date and time the report is generated',
   MARKET_ID            varchar(20) not null default 'Gas Supply Hub'  comment 'The market in which the company is a participant, for example GSH, DAA, ALL.',
   AGENT_ID             varchar(20)  comment 'The unique participant identifier of the agent appointed to trade on the participants behalf.',
   AGENT_NAME           varchar(80)  comment 'Name of the agent organisation'
);

alter table GSH_PARTICIPANTS comment 'The report contains the registration details of all particip';

alter table GSH_PARTICIPANTS
   add primary key (MARKET_ID, PARTICIPANT_ID);

/*==============================================================*/
/* Table: GSH_PARTICIPANT_CONTACTS                              */
/*==============================================================*/
create table GSH_PARTICIPANT_CONTACTS
(
   CONTACT_ID           varchar(20) not null  comment 'The unique identifier for the contact',
   PARTICIPANT_CODE     varchar(20)  comment 'The participant code',
   PARTICIPANT_NAME     varchar(80)  comment 'The trading participant name',
   ROLE                 varchar(80)  comment 'Contact type, for example: Trading Contact, Gas Delivery Contact, System contact, Regulatory contact, Head office contact, Primary settlements contact, Secondary settlements contact, Settlements manager contact, CFO, CEO, First Escalation settlements, Second escalation settlements, Third escalation settlements',
   FIRST_NAME           varchar(80)  comment 'First name of the contact',
   LAST_NAME            varchar(80)  comment 'Last name of the contact',
   SALUTATION           varchar(20)  comment 'Salutation of the contact',
   JOB_TITLE            varchar(80)  comment 'Job title for the contact',
   ADDRESS_LINE_1       varchar(80)  comment 'The address of the contact',
   ADDRESS_LINE_2       varchar(80)  comment 'The address of the contact',
   ADDRESS_LINE_3       varchar(80)  comment 'The address of the contact',
   ADDRESS_LINE_4       varchar(80)  comment 'The address of the contact',
   BUSINESS_PHONE_NUMBER varchar(80)  comment 'The business phone number for the contact',
   MOBILE_NUMBER        varchar(80)  comment 'Mobile number of the contact',
   FAX_NUMBER           varchar(80)  comment 'Fax number for the contact',
   EMAIL_ADDRESS        varchar(255)  comment 'Email address of the contact',
   LASTCHANGED          datetime  comment 'The date and time the report is issued'
);

alter table GSH_PARTICIPANT_CONTACTS comment 'The purpose of this report is to provide the participant wit';

alter table GSH_PARTICIPANT_CONTACTS
   add primary key (CONTACT_ID);

/*==============================================================*/
/* Table: GSH_PARTICIPANT_OPSTATE                               */
/*==============================================================*/
create table GSH_PARTICIPANT_OPSTATE
(
   MARKET_ID            varchar(20) not null  comment 'Unique Market ID',
   PARTICIPANT_ID       varchar(20) not null  comment 'Unique Participant ID for this market',
   EFFECTIVEDATE        datetime not null  comment 'The Effective date of the Participant Operating State',
   VERSION_DATETIME     datetime not null  comment 'Versioning with EffectiveDate',
   STATUS               varchar(30)  comment 'Current State of the Product (ACTIVE/INACTIVE)',
   AUTHORISEDDATE       datetime  comment 'The date of the latest authorisation record',
   LASTCHANGED          datetime  comment 'Last changed date for the record'
);

alter table GSH_PARTICIPANT_OPSTATE comment 'Operating Status of Participant';

alter table GSH_PARTICIPANT_OPSTATE
   add primary key (MARKET_ID, PARTICIPANT_ID, EFFECTIVEDATE, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_PIPELINE_SEGMENTS                                 */
/*==============================================================*/
create table GSH_PIPELINE_SEGMENTS
(
   FACILITY_ID          numeric(8,0) not null  comment 'The unique identifier of the parent facility.',
   EFFECTIVEDATE        datetime not null  comment 'The effective date of the segment.',
   VERSION_DATETIME     datetime not null  comment 'The version timestamp of the record',
   SEGMENT_ID           numeric(8,0) not null  comment 'The unique identifier of the pipeline segment or compression service.',
   FACILITY_NAME        varchar(100)  comment 'The descriptive name of the parent facility.',
   SEGMENT_NAME         varchar(100)  comment 'The descriptive name of the pipeline segment or compression service.',
   SEGMENT_TYPE         varchar(20)  comment 'The segment type, for example Forward Haul, Backhaul or Compression Service',
   FROM_ID              numeric(8,0) not null  comment 'The unique identifier of the zone for forward haul pipeline segments and compression services or the service point for backhaul pipeline segments.',
   FROM_NAME            varchar(100)  comment 'The descriptive name of the zone or service point.',
   TO_ID                numeric(8,0) not null  comment 'The unique identifier of the zone for forward haul pipeline segments and compression services or the service point for backhaul pipeline segments.',
   TO_NAME              varchar(100)  comment 'The descriptive name of the zone or service point.',
   CONNECTION_TYPE      varchar(20) not null  comment 'Type of connection (eg. SERVICE_POINT, ZONE)',
   FOR_AUCTION_PUBLICATION varchar(3)  comment 'Flag to indicate if the product component can be published in the public auction reports, for example yes or no.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_PIPELINE_SEGMENTS comment 'Pipeline Segments currently registered under Part 24 of the ';

alter table GSH_PIPELINE_SEGMENTS
   add primary key (FACILITY_ID, EFFECTIVEDATE, VERSION_DATETIME, SEGMENT_ID, CONNECTION_TYPE, FROM_ID, TO_ID);

/*==============================================================*/
/* Table: GSH_PRD_AUCT_FWD_EXPOSURE                             */
/*==============================================================*/
create table GSH_PRD_AUCT_FWD_EXPOSURE
(
   MARKET_ID            varchar(20)  comment 'Unique Market ID',
   ORGANISATION_CODE    varchar(20) not null  comment 'The unique code for the organisation.',
   ORGANISATION_NAME    varchar(80)  comment 'The name of the organisation.',
   PRUDENTIAL_RUN_ID    numeric(4,0) not null  comment 'The run number of the prudential run.',
   PRUDENTIAL_DATETIME  datetime not null  comment 'The date and time of the prudential run.',
   GAS_DATE             datetime not null  comment 'The gas day. Disregard the time component as this is not applicable.',
   BIDS_FORWARD_EXP_AMT_NET_GST numeric(18,8)  comment 'The sum of all day ahead auction bids forward exposure amounts allocated to the organisation on this gas dayexclusive of the bids forward exposure GST amount.',
   BIDS_FORWARD_EXP_GST_AMT numeric(18,8)  comment 'The total amount of GST for the day ahead auction bids forward exposure.',
   BID_QUANTITY         numeric(18,8)  comment 'The total quantity for auction bids.'
);

alter table GSH_PRD_AUCT_FWD_EXPOSURE comment 'The purpose of the Prudential Exposure report is to provide ';

alter table GSH_PRD_AUCT_FWD_EXPOSURE
   add primary key (ORGANISATION_CODE, PRUDENTIAL_RUN_ID, PRUDENTIAL_DATETIME, GAS_DATE);

/*==============================================================*/
/* Table: GSH_PRD_BANK_GUARANTEES                               */
/*==============================================================*/
create table GSH_PRD_BANK_GUARANTEES
(
   ORGANISATION_CODE    varchar(20) not null  comment 'The unique code for the organisation',
   PRUDENTIAL_RUN_ID    numeric(4,0) not null  comment 'The run number of the prudential run.',
   PRUDENTIAL_DATETIME  datetime not null  comment 'The date and time of the prudential run',
   BANK_GUARANTEE_NO    varchar(20) not null  comment 'The reference number for the bank guarantee included in the prudential run',
   ORGANISATION_NAME    varchar(80)  comment 'The name of the organisation',
   EFFECTIVE_FROM_DATE  datetime  comment 'The start date of the bank guarantee',
   EXPIRY_DATE          datetime  comment 'Expiry date for the bank guarantee',
   PRUDENTIAL_EXPIRY_DATE datetime  comment 'The prudential expiry date of the guarantee',
   AMOUNT               numeric(18,8)  comment 'The dollar amount of the guarantee applicable to the prudential run',
   BANK_GUARANTEE_ALERT varchar(255)  comment 'The field will show the warning Renew Bank Guarantee Warning if the prudential expiry date of the bank guarantee is within the next 30 days (PRUDENTIAL_EXPIRY_DATE is less than or equal to PRUDENTIAL_DATETIME + 30 days), otherwise it will have no value'
);

alter table GSH_PRD_BANK_GUARANTEES comment 'The purpose of the Prudential Exposure report is to provide ';

alter table GSH_PRD_BANK_GUARANTEES
   add primary key (ORGANISATION_CODE, PRUDENTIAL_RUN_ID, PRUDENTIAL_DATETIME, BANK_GUARANTEE_NO);

/*==============================================================*/
/* Table: GSH_PRD_EST_MARKET_EXPOSURE                           */
/*==============================================================*/
create table GSH_PRD_EST_MARKET_EXPOSURE
(
   ORGANISATION_CODE    varchar(20) not null  comment 'The unique code for the organisation',
   PRUDENTIAL_RUN_ID    numeric(4,0) not null  comment 'The run number of the prudential run',
   PRUDENTIAL_DATETIME  datetime not null  comment 'The date and time of the prudential run',
   ORGANISATION_NAME    varchar(80)  comment 'The name of the organisation',
   TOTAL_BANK_GUARANTEES numeric(18,8)  comment 'Total amount of all valid bank guarantees included in the prudential run',
   TRADING_LIMIT        numeric(18,8)  comment 'The calculated trading limit for the organisation
             (organisations trading limit is calculated by multiplying the total valid bank guarantee amounts with trading limit parameter)',
   SECURITY_DEPOSITS    numeric(18,8)  comment 'Total amount of security deposits included in the prudential run',
   EARLY_PAYMENTS       numeric(18,8)  comment 'The sum of all applicable early payment amounts for the organisation',
   OUTSTANDING_AMOUNT   numeric(18,8)  comment 'The total outstanding amounts for the organisation
             (The organisations oustandings amount equals to Settlement exposure amount, plus reallocation amounts for gas days less than prudential run date where the reallocation amounts were not included in settlement run, minus Security deposits minus Early payments)',
   REALLOC_ADJ_AMT_TO_OUTSTANDING numeric(18,8)  comment 'The total reallocations amounts used as adjustment to outstandings in the prudential run, this include all reallocations amounts not included in settlement runs and are associated with gas days covered by the prudential run where the reallocation gas day is less than the prudential run date',
   SETTLEMENT_EXPOSURE_AMOUNT numeric(18,8)  comment 'The sum of all commodity and capacity settlement amounts owing up to the current date for the organisation, including estimates of non-invoiced amounts',
   TRADES_FORWARD_EXPOSURE_AMOUNT numeric(18,8)  comment 'The sum of all commodity and capacity trades forward exposure amounts allocated to the organisationinclusive of the trades forward exposure GST amount.',
   FORWARD_REALLOCATIONS_AMOUNT numeric(18,8)  comment 'The aggregated value of all debit and credit authorised reallocations which apply to the prudential forward exposure period for reallocation',
   FORWARD_EXPOSURE_AMOUNT numeric(18,8)  comment 'The sum of all forward transaction exposure amounts and all authorised reallocations amounts which apply to gas days in the prudential forward exposure period for reallocation for the organisation  inclusive of the trades forward exposure GST amount
             (The forward exposure is the organisations total exposure amount attributable to all forward dates submitted/executed orders which the delivery dates are in the future, and all authorised reallocations amounts which apply to gas days in the prudential forward exposure period for reallocation)',
   PRUDENTIAL_EXPOSURE  numeric(18,8)  comment 'The organisations prudential exposure is calculated by aggregating all associated participants outstanding amounts, settlement exposure, the organisations forward exposure and the forward exposure GST amount.',
   PERCENTAGE_EXPOSURE  numeric(18,8)  comment 'The percentage of prudential exposure as compared to the trading limit
             (organisations percentage exposure is calculated by dividing the organisation prudential exposure amount by the trading limit)',
   TRADING_MARGIN       numeric(18,8)  comment 'The trading margin of the organisation
             (organisations trading margin is calculated by subtracting the organisation prudential exposure amount from the trading limit)',
   BANK_GUARANTEE_RATIO numeric(18,8)  comment 'The ratio between the organisations total bank guarantees and the sum of the organisations total bank guarantees and the organisations total security deposits in this prudential run',
   EXPOSURE_NOTIFICATION varchar(80)  comment 'The field contains exposure notifications for organisation, it can have one of the following notifications or both depending on the conditions met as the following:
             Trading Limit Breach: if the organisation has gone over the trading limit (trading margin value < 0)
             Bank Guarantee Ratio Breach: If the bank guarantee ratio is deemed insufficient (refer to FRDG7603 for more details) 
             If none of the above conditions are met then the field will contain no value',
   LASTCHANGED          datetime  comment 'The date and time the report was generated',
   DAA_SETTLEMENT_EXPOSURE_AMOUNT numeric(18,8)  comment 'The sum of all day ahead auction settlement amounts owing up to the current date for the organisation, including estimates of non-invoiced amounts',
   DAA_FORWARD_EXPOSURE_AMOUNT numeric(18,8)  comment 'The sum of all day ahead auction bids forward exposure amounts allocated to the organisationinclusive of the day ahead auction bids forward exposure GST amount'
);

alter table GSH_PRD_EST_MARKET_EXPOSURE comment 'The purpose of the Prudential Exposure report is to provide ';

alter table GSH_PRD_EST_MARKET_EXPOSURE
   add primary key (ORGANISATION_CODE, PRUDENTIAL_RUN_ID, PRUDENTIAL_DATETIME);

/*==============================================================*/
/* Table: GSH_PRD_REALLOCATIONS                                 */
/*==============================================================*/
create table GSH_PRD_REALLOCATIONS
(
   ORGANISATION_CODE    varchar(20) not null  comment 'The unique code for the organisation',
   PRUDENTIAL_RUN_ID    numeric(4,0) not null  comment 'The run number of the prudential run.',
   PRUDENTIAL_DATETIME  datetime not null  comment 'The date and time of the prudential run',
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code for the participant who lodged the security deposit',
   REALLOCATION_ID      varchar(20) not null  comment 'The unique identifier for the reallocation',
   ORGANISATION_NAME    varchar(80)  comment 'The name of the organisation',
   REALLOCATION_ROLE    varchar(80)  comment 'The role of the participant in the reallocation',
   COUNTERPARTY_PARTICIPANT_CODE varchar(20)  comment 'The participant code for the counterparty of the reallocation',
   START_DATE           datetime  comment 'The date on which the reallocation becomes into effect',
   END_DATE             datetime  comment 'The date on which the reallocation ends',
   AGREEMENT_TYPE       varchar(80)  comment 'The agreement type of the reallocation',
   REALLOC_ADJ_AMT_TO_OUTSTANDING numeric(18,8)  comment 'The total dollar amount of the reallocation, which is used in this prudential run as an adjustment to outstandings',
   FORWARD_REALLOCATION_AMOUNT numeric(18,8)  comment 'The total dollar amount of the reallocation, which is used in this prudential run in calculating the forward exposure amount'
);

alter table GSH_PRD_REALLOCATIONS comment 'The purpose of the Prudential Exposure report is to provide ';

alter table GSH_PRD_REALLOCATIONS
   add primary key (ORGANISATION_CODE, PRUDENTIAL_RUN_ID, PRUDENTIAL_DATETIME, PARTICIPANT_CODE, REALLOCATION_ID);

/*==============================================================*/
/* Table: GSH_PRD_SECURITY_DEPOSITS                             */
/*==============================================================*/
create table GSH_PRD_SECURITY_DEPOSITS
(
   ORGANISATION_CODE    varchar(20) not null  comment 'The unique code for the organisation',
   PRUDENTIAL_RUN_ID    numeric(4,0) not null  comment 'The run number of the prudential run.',
   PRUDENTIAL_DATETIME  datetime not null  comment 'The date and time of the prudential run',
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code for the participant who lodged the security deposit',
   SECURITY_DEPOSIT_ID  varchar(20) not null  comment 'The unique identifier for a security deposit which is included in the prudential run (Note: only security deposits that are maturing on a day later than the prudential date time are included in this report)',
   ORGANISATION_NAME    varchar(80)  comment 'The name of the organisation',
   EFFECTIVE_FROM_DATE  datetime  comment 'The start date of the security deposit',
   MATURITY_DATE        datetime  comment 'The maturity date of the security deposit',
   TOTAL_SECURITY_DEPOSIT_AMOUNT numeric(18,8)  comment 'All cash being held and not yet paid back via settlement',
   NON_APPLIED_SEC_DEPOSIT_AMT numeric(18,8)  comment 'All cash that is used to directly reduce outstandings (not in a settlement run)',
   INTEREST_ACCT_ID     varchar(20)  comment 'The Interest Account Id for calculating the Interest Payment. This is NULL if the INTEREST_CALC_TYPE = FIXED',
   INTEREST_CALC_TYPE   varchar(20)  comment 'Fixed or daily'
);

alter table GSH_PRD_SECURITY_DEPOSITS comment 'The purpose of the Prudential Exposure report is to provide ';

alter table GSH_PRD_SECURITY_DEPOSITS
   add primary key (ORGANISATION_CODE, PRUDENTIAL_RUN_ID, PRUDENTIAL_DATETIME, PARTICIPANT_CODE, SECURITY_DEPOSIT_ID);

/*==============================================================*/
/* Table: GSH_PRD_SETTLEMENT_AMOUNTS                            */
/*==============================================================*/
create table GSH_PRD_SETTLEMENT_AMOUNTS
(
   ORGANISATION_CODE    varchar(20) not null  comment 'The unique code for the organisation',
   PRUDENTIAL_RUN_ID    numeric(4,0) not null  comment 'The run number of the prudential run.',
   PRUDENTIAL_DATETIME  datetime not null  comment 'The date and time of the prudential run',
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code for the participant which the outstanding amount belongs to',
   BILLING_PERIOD       varchar(20) not null  comment 'The billing period',
   ORGANISATION_NAME    varchar(80)  comment 'The name of the organisation',
   SETTLEMENT_TYPE      varchar(20)  comment 'The run type of the settlement run (i.e. Final, Revision, etc)',
   SETTLEMENT_VERSION_ID numeric(18,0)  comment 'The version number of the settlement run used in this billing period',
   SETTLEMENT_EXPOSURE_AMOUNT numeric(18,8)  comment 'The outstanding amount for the participant as per the settlement run',
   PUBLICATION_DATE     datetime  comment 'The publication date of the settlement',
   PAYMENT_DUE_DATE     datetime  comment 'The due date of the payment for the outstanding amount'
);

alter table GSH_PRD_SETTLEMENT_AMOUNTS comment 'The purpose of the Prudential Exposure report is to provide ';

alter table GSH_PRD_SETTLEMENT_AMOUNTS
   add primary key (ORGANISATION_CODE, PRUDENTIAL_RUN_ID, PRUDENTIAL_DATETIME, PARTICIPANT_CODE, BILLING_PERIOD);

/*==============================================================*/
/* Table: GSH_PRD_TRADE_FWD_EXPOSURE                            */
/*==============================================================*/
create table GSH_PRD_TRADE_FWD_EXPOSURE
(
   ORGANISATION_CODE    varchar(20) not null  comment 'The unique code for the organisation',
   PRUDENTIAL_RUN_ID    numeric(4,0) not null  comment 'The run number of the prudential run.',
   PRUDENTIAL_DATETIME  datetime not null  comment 'The date and time of the prudential run',
   GAS_DATE             datetime not null  comment 'The gas day',
   PRODUCT_LOCATION     varchar(80) not null  comment 'The product location for the Gas Supply Hub market',
   ORGANISATION_NAME    varchar(80)  comment 'The name of the organisation',
   TRADES_FORWARD_EXP_AMT_NET_GST numeric(18,8)  comment 'The sum of all trades forward exposure amounts allocated to the organisation on this gas day  exclusive of the trades forward exposure GST amount',
   TRADES_FORWARD_EXP_GST_AMT numeric(18,8)  comment 'The total amount of GST for the trades forward exposure',
   BUY_ORDER_QUANTITY   numeric(18,8)  comment 'The total quantity for positively-valued orders',
   BUY_TRADE_QUANTITY   numeric(18,8)  comment 'The total quantity for positively-valued trades',
   SELL_ORDER_QUANTITY  numeric(18,8)  comment 'The total quantity for negatively-valued orders',
   SELL_TRADE_QUANTITY  numeric(18,8)  comment 'The total quantity for negatively-valued trades',
   BUY_LOAD_WEIGHTED_AVG_PRICE numeric(18,8)  comment 'The load weighted average price for all positively-valued orders and trades',
   SELL_LOAD_WEIGHTED_AVG_PRICE numeric(18,8)  comment 'The load weighted average price for all negatively-valued trades'
);

alter table GSH_PRD_TRADE_FWD_EXPOSURE comment 'The purpose of the Prudential Exposure report is to provide ';

alter table GSH_PRD_TRADE_FWD_EXPOSURE
   add primary key (ORGANISATION_CODE, PRUDENTIAL_RUN_ID, PRUDENTIAL_DATETIME, GAS_DATE, PRODUCT_LOCATION);

/*==============================================================*/
/* Table: GSH_RCPT_DELIV_POINT_PREFS                            */
/*==============================================================*/
create table GSH_RCPT_DELIV_POINT_PREFS
(
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code used in the ETS.',
   PRODUCT_NAME         varchar(80) not null  comment 'The capacity product name.',
   BUY_SELL_POSITION    varchar(4) not null  comment 'Specifies the position that the contract reference must be used for, for example Buy, Sell or Both.',
   STEP_NUMBER          numeric(6,0) not null  comment 'The step number for the corresponding capacity quantity.',
   VERSION_DATETIME     datetime not null  comment 'The date and time the preference was updated.',
   FROM_GAS_DATE        datetime  comment 'The first gas date that the contract reference takes effect from. Disregard the time component as this is not applicable.',
   TO_GAS_DATE          datetime  comment 'The last gas date that the contract reference takes effect until. Disregard the time component as this is not applicable.',
   OVERRIDE_FLAG        varchar(3)  comment 'A flag to indicate whether the receipt and delivery points specified in the original order should override this preference, for example Yes or No.',
   PRODUCT_RECEIPT_POINT varchar(40)  comment 'The default Receipt Point for Spread Product',
   RECEIPT_POINT_ID     numeric(8,0)  comment 'The unique identifier of the Receipt Point as defined in the Transportation Service Point Register. This is the point where gas is injected into the pipeline',
   RECEIPT_POINT_NAME   varchar(255)  comment 'The descriptive name of the Receipt Point as defined in the Transportation Service Point Register.',
   PRODUCT_DELIVERY_POINT varchar(40)  comment 'The default Delivery Point for Spread Product',
   DELIVERY_POINT_ID    numeric(8,0)  comment 'The unique identifier of the Delivery Point as defined in the Transportation Service Point Register. This is the point where gas is withdrawn from the pipeline. Mandatory for service points registered under Part 24 of the rules.',
   DELIVERY_POINT_NAME  varchar(255)  comment 'The descriptive name of the Delivery Point as defined in the Transportation Service Point Register.',
   STEP_QUANTITY        numeric(10,0)  comment 'The step quantity to applied to the contract reference (GJ/day).',
   LAST_UPDATED         datetime  comment 'The date and time the preference was updated.',
   LAST_UPDATED_BY      varchar(20)  comment 'The user name who updated the preference.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_RCPT_DELIV_POINT_PREFS comment 'This report is available to participants as an audit trail o';

alter table GSH_RCPT_DELIV_POINT_PREFS
   add primary key (PARTICIPANT_CODE, PRODUCT_NAME, BUY_SELL_POSITION, STEP_NUMBER, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_REALLOC_DAILY_AMOUNT                              */
/*==============================================================*/
create table GSH_REALLOC_DAILY_AMOUNT
(
   REALLOCATION_ID      varchar(20) not null  comment 'The unique identifier for the reallocation',
   GAS_DATE             datetime not null  comment 'The gas date in the reallocation period that the daily amount applies to',
   AMOUNT               numeric(18,8)  comment 'The daily reallocation amount which can be dollar or quantity GJ amount'
);

alter table GSH_REALLOC_DAILY_AMOUNT comment 'The purpose of the Reallocation Confirmation report is to pr';

alter table GSH_REALLOC_DAILY_AMOUNT
   add primary key (REALLOCATION_ID, GAS_DATE);

/*==============================================================*/
/* Table: GSH_REALLOC_REALLOCATION                              */
/*==============================================================*/
create table GSH_REALLOC_REALLOCATION
(
   REALLOCATION_ID      varchar(20) not null  comment 'The unique identifier for the reallocation',
   DEBIT_PARTICIPANT_CODE varchar(20)  comment 'The participant code for the participant on the debit side of the reallocation',
   DEBIT_PARTICIPANT_NAME varchar(80)  comment 'The name for the participant on the debit side of the reallocation',
   DEBIT_PARTICIPANT_REFERENCE varchar(80)  comment 'The reference for the reallocation for the participant on the debit side of the reallocation',
   CREDIT_PARTICIPANT_CODE varchar(20)  comment 'The participant code for the participant on the credit side of the reallocation',
   CREDIT_PARTICIPANT_NAME varchar(80)  comment 'The name for the participant on the credit side of the reallocation',
   CREDIT_PARTICIPANT_REFERENCE varchar(80)  comment 'The reference for the reallocation for the participant on the credit side of the reallocation',
   START_DATE           datetime  comment 'The date on which the reallocation comes into effect',
   END_DATE             datetime  comment 'The date on which the reallocation ends',
   AGREEMENT_TYPE       varchar(80)  comment 'The type of the reallocation agreement',
   PRODUCT_LOCATION     varchar(80)  comment 'The product location used to determine the reference price (only applicable to Fixed Quantity Amount reallocation agreement type)',
   REALLOCATION_STATUS  varchar(80)  comment 'The status of the agreement which can be submitted, authorised, cancelled, or expired',
   REASON               varchar(255)  comment 'The reason explaining why a reallocation has been cancelled or expired (only applies to Cancellation and Expiration of report)
             (Note: for details on what  the REASON field value should be refer to SUC7704-Cancel Reallocation in the Reallocations System Requirements Document)',
   LAST_UPDATED         datetime  comment 'The date and time the reallocation was last updated i.e. saved into database',
   LASTCHANGED          datetime  comment 'The date and time the report was generated'
);

alter table GSH_REALLOC_REALLOCATION comment 'The purpose of the Reallocation Confirmation report is to pr';

alter table GSH_REALLOC_REALLOCATION
   add primary key (REALLOCATION_ID);

/*==============================================================*/
/* Table: GSH_REVISED_AUCT_QUANTITIES                           */
/*==============================================================*/
create table GSH_REVISED_AUCT_QUANTITIES
(
   AUCTION_DATE         datetime not null  comment 'The date of the auction run. Disregard the time component as this is not applicable.',
   GAS_DATE             datetime not null  comment 'The effective gas date of the capacity won at auction Disregard the time component as this is not applicable.',
   FACILITY_ID          numeric(8,0) not null  comment 'The unique identifier of the facility.',
   FACILITY_NAME        varchar(100)  comment 'The descriptive name of the facility.',
   SERVICE_TYPE         varchar(20) not null  comment 'The type of service, for example Forward Haul, Backhaul or Compression.',
   AGG_CLEARED_QUANTITY numeric(10,0)  comment 'The aggregated initial cleared capacity quantity as determined by the auction (GJ/day).',
   AGG_REVISED_QUANTITY numeric(10,0)  comment 'The aggregated revised auction capacity quantity (GJ/day).',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_REVISED_AUCT_QUANTITIES comment 'The report lists the following items for each facility and s';

alter table GSH_REVISED_AUCT_QUANTITIES
   add primary key (AUCTION_DATE, GAS_DATE, FACILITY_ID, SERVICE_TYPE);

/*==============================================================*/
/* Table: GSH_SECDEPOSIT_INTEREST_RATE                          */
/*==============================================================*/
create table GSH_SECDEPOSIT_INTEREST_RATE
(
   INTEREST_ACCT_ID     varchar(20) not null  comment 'The Interest Account Id for calculating the Interest Payment. This is NULL if the INTEREST_CALC_TYPE = FIXED',
   EFFECTIVEDATE        datetime not null  comment 'The Effective Date of the Interest Rate Change',
   VERSION_DATETIME     datetime not null  comment 'The Version Date time of the Effective Date for the Interest Rate Change',
   INTEREST_RATE        numeric(18,8)  comment 'The Interest Rate for the Interest Account ID as on the Effective Date'
);

alter table GSH_SECDEPOSIT_INTEREST_RATE comment 'The Security Deposit Interest Rate On Daily Basis. This is t';

alter table GSH_SECDEPOSIT_INTEREST_RATE
   add primary key (INTEREST_ACCT_ID, EFFECTIVEDATE, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_SERVICE_POINTS                                    */
/*==============================================================*/
create table GSH_SERVICE_POINTS
(
   FACILITY_ID          numeric(8,0) not null  comment 'The unique identifier of the parent facility.',
   EFFECTIVEDATE        datetime not null  comment 'The effective date of the service point.',
   VERSION_DATETIME     datetime not null  comment 'The version timestamp of the record',
   SERVICE_POINT_ID     numeric(8,0) not null  comment 'The unique identifier of the service point.',
   FACILITY_NAME        varchar(100)  comment 'The descriptive name of the parent facility.',
   SERVICE_POINT_NAME   varchar(100)  comment 'The descriptive name of the service point.',
   SERVICE_POINT_TYPE   varchar(100) not null  comment 'The type of service point, for example forward haul receipt point, forward haul delivery point, backhaul receipt point, backhaul delivery point, compression receipt point or compression delivery point.',
   SERVICE_POINT_DESCRIPTION varchar(255)  comment 'The description of the service point.',
   AUCTION_PRODUCT_TYPE varchar(20) not null  comment 'Auction Product Type for Service Point',
   NOTIONAL_POINT       varchar(3)  comment 'Flag to indicate if the service point is a notional point, for example Yes or No.',
   ZONE_ID              numeric(8,0)  comment 'The unique identifier of the zone in which the service point is located.',
   ZONE_NAME            varchar(100)  comment 'The descriptive name of the zone.',
   GBB_CONNECTION_POINT_ID numeric(8,0)  comment 'The corresponding GBB connection point identifier.',
   GBB_CONNECTION_POINT_NAME varchar(100)  comment 'The descriptive name of the corresponding GBB connection point.',
   FOR_AUCTION_PUBLICATION varchar(3)  comment 'Flag to indicate if the product component can be published in the public auction reports, for example yes or no.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_SERVICE_POINTS comment 'Service Points currently registered under Part 24 of the Rul';

alter table GSH_SERVICE_POINTS
   add primary key (FACILITY_ID, EFFECTIVEDATE, VERSION_DATETIME, SERVICE_POINT_ID, SERVICE_POINT_TYPE, AUCTION_PRODUCT_TYPE);

/*==============================================================*/
/* Table: GSH_SHIPPER_CAP_XFR_NOTICE                            */
/*==============================================================*/
create table GSH_SHIPPER_CAP_XFR_NOTICE
(
   CAPACITY_TRANSFER_ID varchar(20) not null  comment 'The unique AEMO defined Capacity Transfer transaction identifier.',
   GAS_DATE             datetime  comment 'The gas date for the capacity change. Disregard the time component as this is not applicable.',
   PARTICIPANT_CODE     varchar(20)  comment 'The participant code used in the ETS.',
   PARTICIPANT_NAME     varchar(80)  comment 'The descriptive name of the participant.',
   FACILITY_ID          numeric(8,0)  comment 'The unique facility identifier as defined in the Transportation Facility Register.',
   FACILITY_NAME        varchar(100)  comment 'The descriptive name of the facility as defined in the Transportation Facility Register.',
   SERVICE_REFERENCE    varchar(255)  comment 'The service reference issued by the facility operator.',
   STTM_CONTRACT_REFERENCE varchar(255)  comment 'The STTM registered facility service identifier (CRN) associated with the service reference.',
   DWGM_REFERENCE       varchar(255)  comment 'The DWGM accreditation reference associated with the transfer.',
   RECEIPT_POINT_ID     numeric(8,0)  comment 'The unique identifier of the Receipt Point as defined in the Transportation Service Point Register. This is the point where gas is injected into the pipeline',
   RECEIPT_POINT_NAME   varchar(255)  comment 'The descriptive name of the Receipt Point as defined in the Transportation Service Point Register.',
   DELIVERY_POINT_ID    numeric(8,0)  comment 'The unique identifier of the Delivery Point as defined in the Transportation Service Point Register. This is the point where gas is withdrawn from the pipeline.',
   DELIVERY_POINT_NAME  varchar(255)  comment 'The descriptive name of the Delivery Point as defined in the Transportation Service Point Register.',
   RECEIPT_ZONE_ID      numeric(8,0)  comment 'The unique identifier of the Receipt Zone as defined in the Transportation Service Point Register.',
   RECEIPT_ZONE_NAME    varchar(255)  comment 'The descriptive name of the Receipt Zone as defined in the Transportation Service Point Register.',
   DELIVERY_ZONE_ID     numeric(8,0)  comment 'The unique identifier of the Delivery Zone as defined in the Transportation Service Point Register.',
   DELIVERY_ZONE_NAME   varchar(255)  comment 'The descriptive name of the Delivery Zone as defined in the Transportation Service Point Register.',
   STATUS               varchar(20)  comment 'The status of the capacity transfer, for example Transferred, Partial, Pending, Rejected or Cancelled.',
   XFR_SYSTEM           varchar(20)  comment 'The system that has processed the capacity transfer, for example Facility operator, AEMO, AEMO STTM or AEMO DWGM.',
   REASON               varchar(40)  comment 'The reason for a rejected or cancelled capacity transfer, for example Buyer delivery contract, Seller delivery fault, system issue or matching record not found.',
   TRANSACTION_COMPLETED_FLAG varchar(3)  comment 'Flag to indicate whether the capacity transfer process is completed, for example yes or no.',
   OVERALL_QUANTITY_CHANGE numeric(10,0)  comment 'The overall capacity quantity adjustment (GJ/day) applied to the relevant contracts for this transfer date and gas date. This is a signed value (+/-).',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_SHIPPER_CAP_XFR_NOTICE comment 'Provides the status of a trading participants capacity tran';

alter table GSH_SHIPPER_CAP_XFR_NOTICE
   add primary key (CAPACITY_TRANSFER_ID);

/*==============================================================*/
/* Table: GSH_SUPP_ADHOC                                        */
/*==============================================================*/
create table GSH_SUPP_ADHOC
(
   SETTLEMENT_RUN_ID    numeric(18,0) not null  comment 'The unique identifier for the settlement run',
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code',
   GAS_DATE             datetime not null  comment 'The gas date for the ad-hoc payment or charge',
   TYPE                 varchar(20) not null  comment 'Type of the ad-hoc amount whether it is a charge or a payment',
   BILLING_PERIOD       varchar(20)  comment 'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run',
   AMOUNT               numeric(18,8)  comment 'Total amount of ad-hoc charge or payment for the gas date excluding GST',
   GST_AMOUNT           numeric(18,8)  comment 'Total amount of GST on the ad-hoc payment or charge for the gas date',
   ADHOC_DESCRIPTION    varchar(255)  comment 'The description of the adhoc payment'
);

alter table GSH_SUPP_ADHOC comment 'The Settlements Supporting Data Report is generated when Fin';

alter table GSH_SUPP_ADHOC
   add primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, GAS_DATE, TYPE);

/*==============================================================*/
/* Table: GSH_SUPP_AUCTION_QUANTITIES                           */
/*==============================================================*/
create table GSH_SUPP_AUCTION_QUANTITIES
(
   MARKET_ID            varchar(20)  comment 'Unique Market ID',
   SETTLEMENT_RUN_ID    numeric(18,0) not null  comment 'The unique identifier for the settlement run.',
   BILLING_PERIOD       varchar(20)  comment 'The billing period for the settlement run.',
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code.',
   GAS_DATE             datetime  comment 'The gas date when the quantity is delivered. Disregard the time component as this is not applicable.',
   AUCTION_TRANSACTION_ID numeric(12,0) not null  comment 'The unique identifier of the auction transaction included in the settlement run.',
   AUCTION_QUANTITY     numeric(10,0)  comment 'The initial capacity quantity won at auction.',
   REVISED_AUCTION_QUANTITY numeric(10,0)  comment 'The actual auction quantity in GJ. This is based on the revised quantity reported by facility operators on D+1.',
   PRODUCT_PRICE        numeric(18,8)  comment 'The price of the auction product won at auction ($/GJ) by the participant.'
);

alter table GSH_SUPP_AUCTION_QUANTITIES comment 'Details of Auction Quantities won used in Settlements';

alter table GSH_SUPP_AUCTION_QUANTITIES
   add primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, AUCTION_TRANSACTION_ID);

/*==============================================================*/
/* Table: GSH_SUPP_CAPACITY_AUCTION                             */
/*==============================================================*/
create table GSH_SUPP_CAPACITY_AUCTION
(
   MARKET_ID            varchar(20)  comment 'Unique Market ID',
   SETTLEMENT_RUN_ID    numeric(18,0) not null  comment 'The unique identifier for the settlement run.',
   BILLING_PERIOD       varchar(20)  comment 'The billing period for the settlement run.',
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code.',
   GAS_DATE             datetime not null  comment 'The gas date for the capacity auction payment or charge. Disregard the time component as this is not applicable.',
   TYPE                 varchar(20) not null  comment 'Type of the capacity auction amount whether it is a charge or a payment.',
   AMOUNT               numeric(18,8)  comment 'Total amount of trading charge or payment for the gas date excluding GST.',
   GST_AMOUNT           numeric(18,8)  comment 'Total amount of GST on the capacity auction payment or charge for the gas date.'
);

alter table GSH_SUPP_CAPACITY_AUCTION comment 'Includes Auction items won that are included in the settleme';

alter table GSH_SUPP_CAPACITY_AUCTION
   add primary key (SETTLEMENT_RUN_ID, GAS_DATE, PARTICIPANT_CODE, TYPE);

/*==============================================================*/
/* Table: GSH_SUPP_CAPACITY_TRANSFERS                           */
/*==============================================================*/
create table GSH_SUPP_CAPACITY_TRANSFERS
(
   MARKET_ID            varchar(20)  comment 'Unique Market ID',
   SETTLEMENT_RUN_ID    numeric(18,0) not null  comment 'The unique identifier for the settlement run.',
   BILLING_PERIOD       varchar(20)  comment 'The billing period for the settlement run.',
   PARTICIPANT_CODE     varchar(20)  comment 'The participant code.',
   GAS_DATE             datetime  comment 'The gas date for the capacity variance payment or charge. Disregard the time component as this is not applicable.',
   CAPACITY_TRANSFER_ID varchar(20) not null  comment 'The unique identifier of the capacity transfer record included in the settlement run.',
   TRADE_SIDE           varchar(20)  comment 'This field specified whether the participant was the buyer or the seller in the trade.',
   CAPACITY_QUANTITY    numeric(18,8)  comment 'The initial capacity quantity transfer in GJ',
   CONFIRMED_CAPACITY_QUANTITY numeric(18,8)  comment 'The confirmed capacity quantity in GJ.',
   INVALID_QUANTITY     numeric(18,8)  comment 'Invalid quantity to indicate the amount of a short sell or amount that a buyer could not accept due to incorrect contract.'
);

alter table GSH_SUPP_CAPACITY_TRANSFERS comment 'Provides a list of all individual capacity transfer records ';

alter table GSH_SUPP_CAPACITY_TRANSFERS
   add primary key (SETTLEMENT_RUN_ID, CAPACITY_TRANSFER_ID);

/*==============================================================*/
/* Table: GSH_SUPP_CAPACITY_VARIANCE                            */
/*==============================================================*/
create table GSH_SUPP_CAPACITY_VARIANCE
(
   MARKET_ID            varchar(20)  comment 'Unique Market ID',
   SETTLEMENT_RUN_ID    numeric(18,0) not null  comment 'The unique identifier for the settlement run.',
   BILLING_PERIOD       varchar(20)  comment 'The billing period for the settlement run.',
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code.',
   GAS_DATE             datetime not null  comment 'The gas date for the capacity variance payment or charge. Disregard the time component as this is not applicable.',
   TYPE                 varchar(20) not null  comment 'Type of the capacity variance amount whether it is a charge or a payment.',
   AMOUNT               numeric(18,8)  comment 'Total amount of trading charge or payment for the gas date excluding GST.',
   GST_AMOUNT           numeric(18,8)  comment 'Total amount of GST on the capacity variance payment or charge for the gas date.'
);

alter table GSH_SUPP_CAPACITY_VARIANCE comment 'The Delivery Variance subreport provides a breakdown of tota';

alter table GSH_SUPP_CAPACITY_VARIANCE
   add primary key (SETTLEMENT_RUN_ID, GAS_DATE, PARTICIPANT_CODE, TYPE);

/*==============================================================*/
/* Table: GSH_SUPP_DELIVERED_QUANTITY                           */
/*==============================================================*/
create table GSH_SUPP_DELIVERED_QUANTITY
(
   SETTLEMENT_RUN_ID    numeric(18,0) not null  comment 'The unique identifier for the settlement run',
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code',
   GAS_DATE             datetime not null  comment 'The gas date that the quantity is delivered',
   TRADE_ID             varchar(20) not null  comment 'The unique identifier for the executed trade included in the settlement run',
   BILLING_PERIOD       varchar(20)  comment 'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run',
   TRADE_SIDE           varchar(20)  comment 'This field specified whether the participant was the buyer or the seller in the trade',
   TRADE_QUANTITY       numeric(18,8)  comment 'The initial quantity of the executed trade',
   DELIVERED_QUANTITY   numeric(18,8)  comment 'The delivered quantity in GJ'
);

alter table GSH_SUPP_DELIVERED_QUANTITY comment 'The Settlements Supporting Data Report is generated when Fin';

alter table GSH_SUPP_DELIVERED_QUANTITY
   add primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, GAS_DATE, TRADE_ID);

/*==============================================================*/
/* Table: GSH_SUPP_DELIVERY_VARIANCE                            */
/*==============================================================*/
create table GSH_SUPP_DELIVERY_VARIANCE
(
   SETTLEMENT_RUN_ID    numeric(18,0) not null  comment 'The unique identifier for the settlement run',
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code',
   GAS_DATE             datetime not null  comment 'The gas date for the delivery variance payment or charge',
   TYPE                 varchar(20) not null  comment 'Type of the delivery variance amount whether it is a charge or a payment',
   BILLING_PERIOD       varchar(20)  comment 'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run',
   AMOUNT               numeric(18,8)  comment 'Total amount of delivery variance charge or payment for the gas date excluding GST',
   GST_AMOUNT           numeric(18,8)  comment 'Total amount of GST on the delivery variance payment or charge for the gas date'
);

alter table GSH_SUPP_DELIVERY_VARIANCE comment 'The Settlements Supporting Data Report is generated when Fin';

alter table GSH_SUPP_DELIVERY_VARIANCE
   add primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, GAS_DATE, TYPE);

/*==============================================================*/
/* Table: GSH_SUPP_EXECUTED_TRADES                              */
/*==============================================================*/
create table GSH_SUPP_EXECUTED_TRADES
(
   SETTLEMENT_RUN_ID    numeric(18,0) not null  comment 'The unique identifier for the settlement run',
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code',
   TRADE_ID             varchar(20) not null  comment 'The unique identifier for the executed trade included in the settlement run',
   GAS_DATE             datetime not null  comment 'The gas date for the trade was executed on',
   BILLING_PERIOD       varchar(20)  comment 'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run',
   TRADE_SIDE           varchar(20)  comment 'This field specified whether the participant was the buyer or the seller in the trade',
   TRADE_QUANTITY       numeric(18,8)  comment 'The quantity of the executed trade',
   TRADE_PRICE          numeric(18,8)  comment 'The price of the executed trade'
);

alter table GSH_SUPP_EXECUTED_TRADES comment 'The Settlements Supporting Data Report is generated when Fin';

alter table GSH_SUPP_EXECUTED_TRADES
   add primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, TRADE_ID, GAS_DATE);

/*==============================================================*/
/* Table: GSH_SUPP_FEE                                          */
/*==============================================================*/
create table GSH_SUPP_FEE
(
   SETTLEMENT_RUN_ID    numeric(18,0) not null  comment 'The unique identifier for the settlement run',
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code',
   GAS_DATE             datetime not null  comment 'The gas date for the fee',
   FEE_TYPE             varchar(20) not null  comment 'The type of fee whether it is a market fee or a transaction fee',
   BILLING_PERIOD       varchar(20)  comment 'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run',
   AMOUNT               numeric(18,8)  comment 'The fee amount excluding GST',
   GST_AMOUNT           numeric(18,8)  comment 'Total amount of GST on the fee for the gas date'
);

alter table GSH_SUPP_FEE comment 'The Settlements Supporting Data Report is generated when Fin';

alter table GSH_SUPP_FEE
   add primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, GAS_DATE, FEE_TYPE);

/*==============================================================*/
/* Table: GSH_SUPP_PHYSICAL_GAS                                 */
/*==============================================================*/
create table GSH_SUPP_PHYSICAL_GAS
(
   SETTLEMENT_RUN_ID    numeric(18,0) not null  comment 'The unique identifier for the settlement run',
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code',
   GAS_DATE             datetime not null  comment 'The gas date for the physical payment or charge',
   TYPE                 varchar(20) not null  comment 'Type of the physical gas amount whether it is a charge or a payment',
   BILLING_PERIOD       varchar(20)  comment 'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run',
   AMOUNT               numeric(18,8)  comment 'Total amount of trading charge or payment for the gas date excluding GST',
   GST_AMOUNT           numeric(18,8)  comment 'Total amount of GST on the physical gas payment or charge for the gas date'
);

alter table GSH_SUPP_PHYSICAL_GAS comment 'The Settlements Supporting Data Report is generated when Fin';

alter table GSH_SUPP_PHYSICAL_GAS
   add primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, GAS_DATE, TYPE);

/*==============================================================*/
/* Table: GSH_SUPP_REALLOCATIONS                                */
/*==============================================================*/
create table GSH_SUPP_REALLOCATIONS
(
   SETTLEMENT_RUN_ID    numeric(18,0) not null  comment 'The unique identifier for the settlement run',
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code',
   GAS_DATE             datetime not null  comment 'The gas date for the reallocation',
   REALLOCATION_ID      varchar(20) not null  comment 'The unique identifier for the reallocation',
   BILLING_PERIOD       varchar(20)  comment 'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run',
   REALLOCATION_SIDE    varchar(20)  comment 'This field specifies whether the participant was the credit or the debit party of the reallocation agreement',
   AGREEMENT_TYPE       varchar(80)  comment 'The type of the reallocation agreement',
   REFERENCE_PRICE      numeric(18,8)  comment 'The reference price used for Fixed Quantity Amount agreement type',
   AMOUNT               numeric(18,8)  comment 'The daily dollar or quantity amount of the reallocation for the specified gas date'
);

alter table GSH_SUPP_REALLOCATIONS comment 'The Settlements Supporting Data Report is generated when Fin';

alter table GSH_SUPP_REALLOCATIONS
   add primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, GAS_DATE, REALLOCATION_ID);

/*==============================================================*/
/* Table: GSH_SUPP_SERVICES                                     */
/*==============================================================*/
create table GSH_SUPP_SERVICES
(
   SETTLEMENT_RUN_ID    numeric(18,0) not null  comment 'The unique identifier for the settlement run.',
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code.',
   GAS_DATE             datetime not null  comment 'The gas date for the delivery variance payment or charge. Disregard the time component as this is not applicable.',
   TYPE                 varchar(20) not null  comment 'Type of the delivery variance amount whether it is a charge or a payment.',
   BILLING_PERIOD       varchar(20)  comment 'The billing period for the settlement run.',
   AMOUNT               numeric(18,8)  comment 'Total amount of delivery variance charge or payment for the gas date excluding GST.',
   GST_AMOUNT           numeric(18,8)  comment 'Total amount of GST on the delivery variance payment or charge for the gas date.'
);

alter table GSH_SUPP_SERVICES comment 'This report adds details from the settlements run for the ne';

alter table GSH_SUPP_SERVICES
   add primary key (GAS_DATE, SETTLEMENT_RUN_ID, PARTICIPANT_CODE, TYPE);

/*==============================================================*/
/* Table: GSH_SUPP_SETTLEMENT_RUN                               */
/*==============================================================*/
create table GSH_SUPP_SETTLEMENT_RUN
(
   SETTLEMENT_RUN_ID    numeric(18,0) not null  comment 'The unique identifier for the settlement run',
   BILLING_PERIOD       varchar(20)  comment 'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run',
   SETTLEMENT_RUN_TYPE  varchar(20)  comment 'The run type for the settlement, for example final, revision, initial',
   START_GAS_DATE       datetime  comment 'The first gas date of the period for which settlement is run',
   END_GAS_DATE         datetime  comment 'The last gas date of the period for which settlement is run',
   LASTCHANGED          datetime  comment 'The date and time the report was generated'
);

alter table GSH_SUPP_SETTLEMENT_RUN comment 'The Settlements Supporting Data Report is generated when Fin';

alter table GSH_SUPP_SETTLEMENT_RUN
   add primary key (SETTLEMENT_RUN_ID);

/*==============================================================*/
/* Table: GSH_SUPP_SETTLEMENT_SUMMARY                           */
/*==============================================================*/
create table GSH_SUPP_SETTLEMENT_SUMMARY
(
   SETTLEMENT_RUN_ID    numeric(18,0) not null  comment 'The unique identifier for the settlement run',
   PARTICIPANT_CODE     varchar(20) not null  comment 'The participant code',
   REVISION_SETTLEMENT_RUN_ID numeric(18,0) not null  comment 'Identifier of the revision included if the settlement run is of type Final.',
   BILLING_PERIOD       varchar(20)  comment 'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run',
   PARTICIPANT_NAME     varchar(80)  comment 'The name of the participant',
   NET_SETTLEMENT_AMOUNT numeric(18,8)  comment 'Sum of all payments and charges exclusive of GST',
   NET_SETTLEMENT_AMOUNT_GST numeric(18,8)  comment 'Sum of all GST amounts on payments and charges',
   TOTAL_REALLOCATION_DEBIT_AMT numeric(18,8)  comment 'Sum of all reallocation debit amounts included in the settlement',
   TOTAL_REALLOCATION_CREDIT_AMT numeric(18,8)  comment 'Sum of all reallocation credit amounts included in the settlement',
   TOTAL_SECURITY_DEPOSITS numeric(18,8)  comment 'The total of all security deposits included in the settlement run. This value will always be a negative value for consistency with invoices (multiplied with -1)',
   SECURITY_DEPOSIT_INTEREST numeric(18,8)  comment 'Total interest amounts calculated on security deposits. This value will always be a negative value for consistency with invoices (multiplied with -1)',
   SETTLEMENT_ADJUSTMENT_AMOUNT numeric(18,8)  comment 'Adjustment amount of the settlement if applicable',
   INTEREST_ON_REVISION numeric(18,8)  comment 'Amount of interest calculated for revision statement if the run type is Revision',
   TOTAL_PAYABLE_AMOUNT numeric(18,8)  comment 'The net payable amount for this settlement run'
);

alter table GSH_SUPP_SETTLEMENT_SUMMARY comment 'The Settlements Supporting Data Report is generated when Fin';

alter table GSH_SUPP_SETTLEMENT_SUMMARY
   add primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, REVISION_SETTLEMENT_RUN_ID);

/*==============================================================*/
/* Table: GSH_TRADES                                            */
/*==============================================================*/
create table GSH_TRADES
(
   MARKET_ID            varchar(20) not null  comment 'Unique Market ID',
   TRADE_ID             varchar(20) not null  comment 'Unique trade id generated by the Trading Platform at the time a transaction is executed',
   BUYER_PARTICIPANT_CODE varchar(20)  comment 'The participant code used in the Exchange Trading System (ETS) for the participant on buy side of the trade',
   BUYER_PARTICIPANT_NAME varchar(80)  comment 'The name for the participant on buy side of the trade',
   SELLER_PARTICIPANT_CODE varchar(20)  comment 'The participant code used in the Exchange Trading System (ETS) for the participant on the seller side of the trade',
   SELLER_PARTICIPANT_NAME varchar(80)  comment 'The name for the participant on seller side of the trade',
   FROM_GAS_DATE        datetime  comment 'The start gas day for the order delivery period',
   TO_GAS_DATE          datetime  comment 'The end gas day for the order delivery period',
   TRADE_DATETIME       datetime  comment 'Date and time the trade is executed',
   TRADE_TYPE           varchar(20)  comment 'The type of the trade. Valid values are: AUTO_MATCHED, OFF_MARKET_TRADE',
   PRODUCT_LOCATION     varchar(80)  comment 'Product location for the Gas Supply Hub market',
   PRODUCT_TYPE         varchar(80)  comment 'The product delivery period for the Gas Supply Hub market',
   TRADE_QUANTITY       numeric(18,8)  comment 'Volume of gas transacted on the Trading Platform in GJ/day',
   TRADE_PRICE          numeric(18,8)  comment 'Price value of the trade in $/GJ',
   DELIVERY_POINT       varchar(40)  comment 'Delivery Point specified by the seller in an order submission for non-netted commodity products, or the buyer or seller in an order submission for capacity products',
   ORDER_ID             varchar(20)  comment 'The id of the order used in the trade from the buyer or the seller side depending on the participant getting the report. This field will be null if the TRADE_TYPE is off-market or rarely if the network goes down',
   LAST_UPDATED         datetime  comment 'The date and time the trade was updated i.e. saved into database',
   LASTCHANGED          datetime  comment 'The date and time the report was generated',
   BUYER_USER_NAME      varchar(100)  comment 'Name of the buyer''''s account that made submission to the exchange',
   SELLER_USER_NAME     varchar(100)  comment 'Name of the seller''''s account that made submission to the exchange',
   RELATIONSHIP_ID      varchar(20)  comment 'Populated for transactions executed as part of a spread product. Trades triggered by the matching of orders in a spread product will have the same RELATIONSHIP_ID',
   RECEIPT_POINT        varchar(40)  comment 'Receipt Point specified by the buyer or seller in an order submission for capacity products',
   VALID_SERVICE_REFERENCE varchar(3)  comment 'Flag to indicate whether a valid service reference exists for the traded capacity product. Will be NULL for trades involving commodity products',
   VALID_STTM_CONTRACT_REFERENCE varchar(3)  comment 'Flag to indicate whether a valid STTM contract reference exists for the traded capacity product. Will be NULL for trades involving commodity products and non-STTM integrated capacity products',
   VALID_DWGM_REFERENCE varchar(3)  comment 'Flag to indicate whether a valid DWGM accreditation reference exists for the traded capacity product where a DWGM interface point has been specified. Will be NULL for trades involving commodity products and capacity products that do not have a DWGM interface point specified',
   TRADE_STATUS         varchar(20)  comment 'Status of the trade, for example Accepted, Rejected or Cancelled. Capacity trades are rejected if a valid service reference or STTM contract reference does not exist for the product',
   GAS_FUTURE_ID        varchar(20)  comment 'The EFP Trade ID for matching trades'
);

alter table GSH_TRADES comment 'The purpose of this report is to provide the Participants (b';

alter table GSH_TRADES
   add primary key (MARKET_ID, TRADE_ID);

/*==============================================================*/
/* Table: GSH_TRADES_STAGING                                    */
/*==============================================================*/
create table GSH_TRADES_STAGING
(
   RECORD_ID            varchar(20) not null  comment 'The unique identifier of the record',
   MARKET_ID            varchar(20)  comment 'The unique Market ID',
   PARTICIPANT_ID       varchar(20)  comment 'The unique identifier of the participant',
   GAS_FUTURE_ID        varchar(20)  comment 'The EFP Trade ID for matching trades',
   PRODUCT_GROUP_CODE   varchar(20)  comment 'AEMO uses a default value only: GAS-WAL. Data submitted by participants is not stored',
   PRODUCT_TYPE_CODE    varchar(20)  comment 'AEMO uses a default value only: NG Months. Data submitted by participants is not stored',
   DELIVERY_POINT       varchar(40)  comment 'AEMO uses a default value only: WAL HP Trade Point. Data submitted by participants is not stored',
   START_DATE           datetime  comment 'The gas start day for the EFP trade period',
   END_DATE             datetime  comment 'The gas end day for the EFP trade period',
   TRADE_TYPE           varchar(20)  comment 'Buy or Sell',
   VOLUME               numeric(18,8)  comment 'Trade quantities measured in GJ for each Gas Day in the Delivery Period',
   PRICE                numeric(18,8)  comment 'Trade price',
   STATUS               varchar(20)  comment 'Submission Status: PENDING, FAIL, MATCH, OVERRIDE',
   VALIDATION_MSG       varchar(255)  comment 'Submission validation message',
   SUBMITTED_TIME       datetime  comment 'Data submission time',
   LASTCHANGED          datetime  comment 'The date and time the report is generated'
);

alter table GSH_TRADES_STAGING comment 'Provides trading participants a confirmation whenever an EFP';

alter table GSH_TRADES_STAGING
   add primary key (RECORD_ID);

/*==============================================================*/
/* Table: GSH_TRADING_CONTACTS                                  */
/*==============================================================*/
create table GSH_TRADING_CONTACTS
(
   CONTACT_ID           varchar(20) not null  comment 'The unique identifier for the contact',
   PARTICIPANT_CODE     varchar(20)  comment 'The participant code',
   PARTICIPANT_NAME     varchar(80)  comment 'The trading participant name',
   ROLE                 varchar(80)  comment 'Contact type, for this report it will only be two types: Trading Contact, Gas Delivery Contact',
   FIRST_NAME           varchar(80)  comment 'First name of the contact',
   LAST_NAME            varchar(80)  comment 'Last name of the contact',
   SALUTATION           varchar(20)  comment 'Salutation of the contact',
   JOB_TITLE            varchar(80)  comment 'Job title for the contact',
   ADDRESS_LINE_1       varchar(80)  comment 'The address of the contact',
   ADDRESS_LINE_2       varchar(80)  comment 'The address of the contact',
   ADDRESS_LINE_3       varchar(80)  comment 'The address of the contact',
   ADDRESS_LINE_4       varchar(80)  comment 'The address of the contact',
   BUSINESS_PHONE_NUMBER varchar(80)  comment 'The business phone number for the contact',
   MOBILE_NUMBER        varchar(80)  comment 'Mobile number of the contact',
   FAX_NUMBER           varchar(80)  comment 'Fax number for the contact',
   EMAIL_ADDRESS        varchar(255)  comment 'Email address of the contact',
   LASTCHANGED          datetime  comment 'The date and time the report is issued'
);

alter table GSH_TRADING_CONTACTS comment 'The purpose of this report is to provide all GSH participant';

alter table GSH_TRADING_CONTACTS
   add primary key (CONTACT_ID);

/*==============================================================*/
/* Table: GSH_TRANSACTION_SUMMARY                               */
/*==============================================================*/
create table GSH_TRANSACTION_SUMMARY
(
   CURRENTDATE          datetime not null  comment 'The current gas date',
   PRODUCT_LOCATION     varchar(80) not null  comment 'Product location for the Gas Supply Hub market',
   PRODUCT_TYPE         varchar(80) not null  comment 'The product delivery period for the Gas Supply Hub market',
   FROM_GAS_DATE        datetime not null  comment 'The start gas day for the order delivery period',
   TO_GAS_DATE          datetime not null  comment 'The end gas day for the order delivery period',
   HIGH_PRICE           numeric(18,8)  comment 'Highest price between all trades executed today per product in $/GJ',
   LOW_PRICE            numeric(18,8)  comment 'Lowest price between all trades executed today per product in $/GJ',
   OPEN_PRICE           numeric(18,8)  comment 'The first price of all trades executed today per product in $/GJ',
   CLOSE_PRICE          numeric(18,8)  comment 'The last price of all trades executed today per product in $/GJ',
   VOLUME_WEIGHTED_AVERAGE_PRICE numeric(18,8)  comment 'The volume weighted average price for all trades executed today per product in $/GJ',
   TOTAL_QUANTITY       numeric(18,8)  comment 'Total traded quantities for all trades executed today per product in GJ/day',
   TOTAL_NUMBER_OF_TRADES numeric(18,0)  comment 'Total number of all executed trades today per product',
   LASTCHANGED          datetime  comment 'The date and time the report was generated'
);

alter table GSH_TRANSACTION_SUMMARY comment 'The report provides a summary of all trades executed in the ';

alter table GSH_TRANSACTION_SUMMARY
   add primary key (CURRENTDATE, PRODUCT_LOCATION, PRODUCT_TYPE, FROM_GAS_DATE, TO_GAS_DATE);

/*==============================================================*/
/* Table: GSH_ZONES                                             */
/*==============================================================*/
create table GSH_ZONES
(
   ZONE_ID              numeric(8,0) not null  comment 'The unique identifier of the zone.',
   EFFECTIVEDATE        datetime not null  comment 'Effective date of the record',
   VERSION_DATETIME     datetime not null  comment 'Version timestamp of the record',
   ZONE_NAME            varchar(100)  comment 'The descriptive name of the zone.',
   ZONE_TYPE            varchar(20)  comment 'The type of zone, for example pipeline receipt, pipeline delivery, compression receipt or compression delivery.',
   ZONE_DESCRIPTION     varchar(400)  comment 'The description of the zone.',
   FOR_AUCTION_PUBLICATION varchar(3)  comment 'Flag to indicate if the product component can be published in the public auction reports, for example yes or no.',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_ZONES comment 'Zones currently registered under Part 24 of the Rules for th';

alter table GSH_ZONES
   add primary key (ZONE_ID, EFFECTIVEDATE, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_ZONE_CURTAILMENT                                  */
/*==============================================================*/
create table GSH_ZONE_CURTAILMENT
(
   FACILITY_ID          numeric(8,0) not null  comment 'The unique identifier of the parent facility.',
   FACILITY_NAME        varchar(100)  comment 'The descriptive name of the parent facility.',
   SERVICE_POINT_ID     numeric(8,0) not null  comment 'The unique identifier of the service point.',
   SERVICE_POINT_NAME   varchar(100)  comment 'The descriptive name of the service point.',
   SERVICE_POINT_TYPE   varchar(100)  comment 'The type of service point, for example forward haul receipt point, forward haul delivery point, backhaul receipt point, backhaul delivery point, compression receipt point or compression delivery point.',
   SERVICE_POINT_DESCRIPTION varchar(255)  comment 'The description of the service point.',
   GAS_DATE             datetime not null  comment 'The effective gas date of the rejected nomination quantity.',
   CURTAILED_QUANTITY   numeric(10,0)  comment 'The curtailed quantity of nominations by the facility operator for transportation capacity that has been transferred pursuant to an operational transfer (GJ)',
   LASTCHANGED          datetime  comment 'The date and time the report was generated.'
);

alter table GSH_ZONE_CURTAILMENT comment 'Contains details of curtailment of nominations against secon';

alter table GSH_ZONE_CURTAILMENT
   add primary key (FACILITY_ID, SERVICE_POINT_ID, GAS_DATE);

