/*==============================================================*/
/* Data Model:     AEMO Gas Data Model v2.2 postgreSQL          */
/* Created on:     2022/08/28 16:38:20                          */
/*==============================================================*/


/*==============================================================*/
/* Table: GAS_BASIN                                             */
/*==============================================================*/
create table GAS_BASIN (
   BASINID              NUMERIC(20,0)        not null,
   BASINNAME            VARCHAR(100)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_BASIN is
'This report displays a list of all basins';

comment on column GAS_BASIN.BASINID is
'A unique AEMO defined Facility Identifier';

comment on column GAS_BASIN.BASINNAME is
'The name of the basin. If short name exists then short name included in report';

comment on column GAS_BASIN.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_BASIN
   add constraint GAS_BASIN_PK primary key (BASINID);

/*==============================================================*/
/* Table: GAS_CAPACITY_TYPE                                     */
/*==============================================================*/
create table GAS_CAPACITY_TYPE (
   CAPACITYTYPE         VARCHAR(20)          not null,
   DESCRIPTION          VARCHAR(800)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_CAPACITY_TYPE is
'Capacity Type - The type of capacity detailed. Valid entries are: STORAGE, MDQ';

comment on column GAS_CAPACITY_TYPE.CAPACITYTYPE is
'Capacity Type - The type of capacity being detailed. Valid entries are: STORAGE, MDQ';

comment on column GAS_CAPACITY_TYPE.DESCRIPTION is
'Description of Capacity Type';

comment on column GAS_CAPACITY_TYPE.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_CAPACITY_TYPE
   add constraint GAS_CAPACITY_TYPE_PK primary key (CAPACITYTYPE);

/*==============================================================*/
/* Table: GAS_CONNECTIONPOINT_CAPACITY                          */
/*==============================================================*/
create table GAS_CONNECTIONPOINT_CAPACITY (
   CONNECTIONPOINTID    NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   CAPACITYQUANTITY     NUMERIC(18,3)        null,
   DESCRIPTION          VARCHAR(255)         null
);

comment on table GAS_CONNECTIONPOINT_CAPACITY is
'Provide details of the gate station capacity on a particular day.';

comment on column GAS_CONNECTIONPOINT_CAPACITY.CONNECTIONPOINTID is
'Unique connection point identifier.';

comment on column GAS_CONNECTIONPOINT_CAPACITY.EFFECTIVEDATE is
'Effective date of the record';

comment on column GAS_CONNECTIONPOINT_CAPACITY.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_CONNECTIONPOINT_CAPACITY.CAPACITYQUANTITY is
'Quantity of available capacity in TJs and support just 3 decimal places during the defined period';

comment on column GAS_CONNECTIONPOINT_CAPACITY.DESCRIPTION is
'Description for reasons or comments directly related to the quantity or change in Outlook quantity in relation to the Facility.';

alter table GAS_CONNECTIONPOINT_CAPACITY
   add constraint GAS_CONNECTPOINT_CAPACITY_PK primary key (CONNECTIONPOINTID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_CONTACT                                           */
/*==============================================================*/
create table GAS_CONTACT (
   PERSONID             NUMERIC(10,0)        not null,
   TITLE                VARCHAR(10)          null,
   FIRSTNAME            VARCHAR(40)          null,
   LASTNAME             VARCHAR(40)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_CONTACT is
'The registered user list of responsible contact persons for each BB Participant organisation as identified at registration or otherwise';

comment on column GAS_CONTACT.PERSONID is
'Unique person identifier';

comment on column GAS_CONTACT.TITLE is
'Title of person';

comment on column GAS_CONTACT.FIRSTNAME is
'First Name of person';

comment on column GAS_CONTACT.LASTNAME is
'Last Name of person';

comment on column GAS_CONTACT.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_CONTACT
   add constraint GAS_CONTACT_PK primary key (PERSONID);

/*==============================================================*/
/* Table: GAS_CONTACT_DETAIL                                    */
/*==============================================================*/
create table GAS_CONTACT_DETAIL (
   PERSONID             NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   "POSITION"           VARCHAR(40)          null,
   PHONE                VARCHAR(30)          null,
   FAX                  VARCHAR(30)          null,
   MOBILE               VARCHAR(30)          null,
   EMAIL                VARCHAR(255)         null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   AUTHORISEDBY         VARCHAR(30)          null
);

comment on table GAS_CONTACT_DETAIL is
'The registered user list details of responsible contact persons for each BB Participant organisation as identified at registration or otherwise';

comment on column GAS_CONTACT_DETAIL.PERSONID is
'Unique person identifier';

comment on column GAS_CONTACT_DETAIL.EFFECTIVEDATE is
'Effective date for this operating state';

comment on column GAS_CONTACT_DETAIL.VERSIONDATETIME is
'Version date time for this operating state';

comment on column GAS_CONTACT_DETAIL."POSITION" is
'Position of person';

comment on column GAS_CONTACT_DETAIL.PHONE is
'Landline phone of person';

comment on column GAS_CONTACT_DETAIL.FAX is
'Facsimile of person';

comment on column GAS_CONTACT_DETAIL.MOBILE is
'Mobile phone of person';

comment on column GAS_CONTACT_DETAIL.EMAIL is
'Email address of person';

comment on column GAS_CONTACT_DETAIL.AUTHORISEDDATE is
'Date and time this operating state was authorised';

comment on column GAS_CONTACT_DETAIL.AUTHORISEDBY is
'User which authorised the change';

alter table GAS_CONTACT_DETAIL
   add constraint GAS_CONTACT_DETAIL_PK primary key (PERSONID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_DAILY_FLOW_ACTUAL                                 */
/*==============================================================*/
create table GAS_DAILY_FLOW_ACTUAL (
   GASDATE              TIMESTAMP(0)         not null,
   FACILITYID           NUMERIC(10,0)        not null,
   CONNECTIONPOINTID    NUMERIC(10,0)        not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   FLOWDIRECTION        VARCHAR(20)          not null,
   ACTUALQUANTITY       NUMERIC(18,3)        null,
   QUALITY              VARCHAR(5)           null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_DAILY_FLOW_ACTUAL is
'Provides on each gas day D, the BB facility operators daily gas flow data for injections and withdrawals at each connection point for gas day D-1.';

comment on column GAS_DAILY_FLOW_ACTUAL.GASDATE is
'Date of gas day';

comment on column GAS_DAILY_FLOW_ACTUAL.FACILITYID is
'Unique Facility Identifier';

comment on column GAS_DAILY_FLOW_ACTUAL.CONNECTIONPOINTID is
'Unique connection point identifier.';

comment on column GAS_DAILY_FLOW_ACTUAL.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_DAILY_FLOW_ACTUAL.FLOWDIRECTION is
'Direction of flow for the submission record. Must be one of RECEIPT or DELIVERY';

comment on column GAS_DAILY_FLOW_ACTUAL.ACTUALQUANTITY is
'The actual flow quantity reported in TJ to the nearest gigajoule.';

comment on column GAS_DAILY_FLOW_ACTUAL.QUALITY is
'Indicates whether meter data for the submission date is available';

comment on column GAS_DAILY_FLOW_ACTUAL.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_DAILY_FLOW_ACTUAL
   add constraint GAS_DAILY_FLOW_ACTUAL_PK primary key (GASDATE, FACILITYID, CONNECTIONPOINTID, VERSIONDATETIME, FLOWDIRECTION);

/*==============================================================*/
/* Table: GAS_DAILY_STORAGE_ACTUAL                              */
/*==============================================================*/
create table GAS_DAILY_STORAGE_ACTUAL (
   GASDATE              TIMESTAMP(0)         not null,
   FACILITYID           NUMERIC(10,0)        not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   ACTUALQUANTITY       NUMERIC(18,3)        null,
   LASTCHANGED          TIMESTAMP(0)         null,
   CUSHIONGASQUANTITY   NUMERIC(18,3)        null
);

comment on table GAS_DAILY_STORAGE_ACTUAL is
'Provide on each gas day D, the actual quantity of natural gas held in each storage facility at the end of the gas day D-1.';

comment on column GAS_DAILY_STORAGE_ACTUAL.GASDATE is
'Date of gas day';

comment on column GAS_DAILY_STORAGE_ACTUAL.FACILITYID is
'Unique Facility Identifier';

comment on column GAS_DAILY_STORAGE_ACTUAL.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_DAILY_STORAGE_ACTUAL.ACTUALQUANTITY is
'The actual flow quantity reported in TJ to the nearest gigajoule.';

comment on column GAS_DAILY_STORAGE_ACTUAL.LASTCHANGED is
'Date and time record was last modified';

comment on column GAS_DAILY_STORAGE_ACTUAL.CUSHIONGASQUANTITY is
'The quantity of natural gas that must be retained in the Storage or LNGImport facility in order to maintain the required pressure and deliverability rates';

alter table GAS_DAILY_STORAGE_ACTUAL
   add constraint GAS_DAILY_STORAGE_ACTUAL_PK primary key (GASDATE, FACILITYID, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_DATA_MODEL_AUDIT                                  */
/*==============================================================*/
create table GAS_DATA_MODEL_AUDIT (
   INSTALLATION_DATE    TIMESTAMP(0)         not null,
   GASDM_VERSION        VARCHAR(20)          not null,
   INSTALL_TYPE         VARCHAR(10)          not null,
   SCRIPT_VERSION       VARCHAR(20)          null,
   GAS_CHANGE_NOTICE    VARCHAR(20)          null,
   PROJECT_TITLE        VARCHAR(200)         null,
   USERNAME             VARCHAR(40)          null,
   STATUS               VARCHAR(10)          null
);

comment on table GAS_DATA_MODEL_AUDIT is
'GAS_DATA_MODEL_AUDIT shows the audit trail of scripts applied to this installation of Gas Data Model. Participants should ensure that if a database is cloned the content of this table is copied to the target database';

comment on column GAS_DATA_MODEL_AUDIT.INSTALLATION_DATE is
'The date in which the changes to the Gas Data Model were installed';

comment on column GAS_DATA_MODEL_AUDIT.GASDM_VERSION is
'The version of Gas Data Model after the script has been applied';

comment on column GAS_DATA_MODEL_AUDIT.INSTALL_TYPE is
'The type of the patch applied. Valid entries are: FULL, UPGRADE, DML';

comment on column GAS_DATA_MODEL_AUDIT.SCRIPT_VERSION is
'The  version of the patch set to the Gas Data Model';

comment on column GAS_DATA_MODEL_AUDIT.GAS_CHANGE_NOTICE is
'The Gas Change notice for which this Gas Data Model applies';

comment on column GAS_DATA_MODEL_AUDIT.PROJECT_TITLE is
'The name of the business project for which these changes to the Gas Data Model apply';

comment on column GAS_DATA_MODEL_AUDIT.USERNAME is
'The USER applying this script';

comment on column GAS_DATA_MODEL_AUDIT.STATUS is
'The status of the upgrade. Valid entries are STARTED, FAILED, SUCCESS';

alter table GAS_DATA_MODEL_AUDIT
   add constraint GAS_DATA_MODEL_AUDIT_PK primary key (INSTALLATION_DATE, GASDM_VERSION, INSTALL_TYPE);

/*==============================================================*/
/* Table: GAS_DEVFACILITY                                       */
/*==============================================================*/
create table GAS_DEVFACILITY (
   DEVFACILITYID        NUMERIC(20,0)        not null,
   DESCRIPTION          VARCHAR(100)         null,
   FACILITYTYPE         VARCHAR(40)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_DEVFACILITY is
'Details of all Facility Developments';

comment on column GAS_DEVFACILITY.DEVFACILITYID is
'A unique AEMO defined Development Facility Identifier';

comment on column GAS_DEVFACILITY.DESCRIPTION is
'Description of development facility';

comment on column GAS_DEVFACILITY.FACILITYTYPE is
'The facility development type';

comment on column GAS_DEVFACILITY.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_DEVFACILITY
   add constraint GAS_DEVFACILITY_PK primary key (DEVFACILITYID);

/*==============================================================*/
/* Table: GAS_DEVFACILITY_DETAIL                                */
/*==============================================================*/
create table GAS_DEVFACILITY_DETAIL (
   DEVFACILITYID        NUMERIC(20,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   PROPOSEDNAME         VARCHAR(100)         null,
   MINNAMEPLATE         NUMERIC(18,3)        null,
   MAXNAMEPLATE         NUMERIC(18,3)        null,
   LOCATION             VARCHAR(200)         null,
   PLANNEDCOMMISSIONFROM VARCHAR(7)           null,
   PLANNEDCOMMISSIONTO  VARCHAR(7)           null,
   DEVELOPMENTSTAGE     VARCHAR(200)         null,
   COMMENTS             VARCHAR(400)         null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   AUTHORISEDBY         VARCHAR(30)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_DEVFACILITY_DETAIL is
'Details of all Facility Developments over time';

comment on column GAS_DEVFACILITY_DETAIL.DEVFACILITYID is
'A unique AEMO defined Development Facility Identifier';

comment on column GAS_DEVFACILITY_DETAIL.EFFECTIVEDATE is
'The effective date of the submission';

comment on column GAS_DEVFACILITY_DETAIL.VERSIONDATETIME is
'Time a successful submission is accepted by AEMO systems';

comment on column GAS_DEVFACILITY_DETAIL.PROPOSEDNAME is
'The name of the Facility development';

comment on column GAS_DEVFACILITY_DETAIL.MINNAMEPLATE is
'The lower estimate of nameplate rating capacity';

comment on column GAS_DEVFACILITY_DETAIL.MAXNAMEPLATE is
'The upper estimate of nameplate rating capacity';

comment on column GAS_DEVFACILITY_DETAIL.LOCATION is
'The location of the development facility';

comment on column GAS_DEVFACILITY_DETAIL.PLANNEDCOMMISSIONFROM is
'The planned start date of commissioning';

comment on column GAS_DEVFACILITY_DETAIL.PLANNEDCOMMISSIONTO is
'The planned end date of commissioning';

comment on column GAS_DEVFACILITY_DETAIL.DEVELOPMENTSTAGE is
'The current stage of the development facility being, PROPOSED, COMMITTED, CANCELLED, ENDED';

comment on column GAS_DEVFACILITY_DETAIL.COMMENTS is
'Any additional comments included in the submission';

comment on column GAS_DEVFACILITY_DETAIL.AUTHORISEDDATE is
'Date and time this operating state was authorised';

comment on column GAS_DEVFACILITY_DETAIL.AUTHORISEDBY is
'Reporting entity for the facility';

comment on column GAS_DEVFACILITY_DETAIL.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_DEVFACILITY_DETAIL
   add constraint GAS_DEVFACILITY_DETAIL_PK primary key (DEVFACILITYID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_DEVFACILITY_FACILITY                              */
/*==============================================================*/
create table GAS_DEVFACILITY_FACILITY (
   DEVFACILITYID        NUMERIC(20,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   FACILITYID           NUMERIC(20,0)        not null
);

comment on table GAS_DEVFACILITY_FACILITY is
'Any facility that is related to dev facility';

comment on column GAS_DEVFACILITY_FACILITY.DEVFACILITYID is
'A unique AEMO defined Development Facility Identifier';

comment on column GAS_DEVFACILITY_FACILITY.EFFECTIVEDATE is
'The effective date of the submission';

comment on column GAS_DEVFACILITY_FACILITY.VERSIONDATETIME is
'Time a successful submission is accepted by AEMO systems';

comment on column GAS_DEVFACILITY_FACILITY.FACILITYID is
'Unique facility identifier';

alter table GAS_DEVFACILITY_FACILITY
   add constraint GAS_DEVFACILITY_FACILITY_PK primary key (DEVFACILITYID, EFFECTIVEDATE, VERSIONDATETIME, FACILITYID);

/*==============================================================*/
/* Table: GAS_FACILITY                                          */
/*==============================================================*/
create table GAS_FACILITY (
   FACILITYID           NUMERIC(10,0)        not null,
   FACILITYNAME         VARCHAR(100)         null,
   FACILITYTYPE         VARCHAR(40)          null,
   LASTCHANGED          TIMESTAMP(0)         null,
   FACILITYSHORTNAME    VARCHAR(30)          null
);

comment on table GAS_FACILITY is
'Provides details of all facilities';

comment on column GAS_FACILITY.FACILITYID is
'Unique Facility Identifier';

comment on column GAS_FACILITY.FACILITYNAME is
'Name of the Facility';

comment on column GAS_FACILITY.FACILITYTYPE is
'Describes type of Facility.Valid entries are: PIPE, PROD or STOR';

comment on column GAS_FACILITY.LASTCHANGED is
'Date and time record was last modified';

comment on column GAS_FACILITY.FACILITYSHORTNAME is
'An abbreviation of the facility name';

alter table GAS_FACILITY
   add constraint GAS_FACILITY_PK primary key (FACILITYID);

/*==============================================================*/
/* Table: GAS_FACILITY_DETAIL                                   */
/*==============================================================*/
create table GAS_FACILITY_DETAIL (
   FACILITYID           NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   GASDAYSTARTHOUR      NUMERIC(2,0)         null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   AUTHORISEDBY         VARCHAR(30)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_FACILITY_DETAIL is
'The details of all gas plant facilities over time';

comment on column GAS_FACILITY_DETAIL.FACILITYID is
'Unique Facility Identifier';

comment on column GAS_FACILITY_DETAIL.EFFECTIVEDATE is
'Effective date of the record';

comment on column GAS_FACILITY_DETAIL.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_FACILITY_DETAIL.GASDAYSTARTHOUR is
'Column no longer populated';

comment on column GAS_FACILITY_DETAIL.AUTHORISEDDATE is
'Date and time this operating state was authorised';

comment on column GAS_FACILITY_DETAIL.AUTHORISEDBY is
'User that authorised the change';

comment on column GAS_FACILITY_DETAIL.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_FACILITY_DETAIL
   add constraint GAS_FACILITY_DETAIL_PK primary key (FACILITYID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_FACILITY_DETAIL_CAPACITY                          */
/*==============================================================*/
create table GAS_FACILITY_DETAIL_CAPACITY (
   FACILITYID           NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   CAPACITYTYPE         VARCHAR(20)          not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   LOWRANGE             NUMERIC(3,0)         null,
   HIRANGE              NUMERIC(3,0)         null,
   UNITS                VARCHAR(5)           null,
   COMMENTS             VARCHAR(255)         null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   AUTHORISEDBY         VARCHAR(30)          null
);

comment on table GAS_FACILITY_DETAIL_CAPACITY is
'The standing details on capacity of all gas plant facilities over time';

comment on column GAS_FACILITY_DETAIL_CAPACITY.FACILITYID is
'Unique Facility identifier';

comment on column GAS_FACILITY_DETAIL_CAPACITY.EFFECTIVEDATE is
'Date capacity comes into effect';

comment on column GAS_FACILITY_DETAIL_CAPACITY.CAPACITYTYPE is
'Capacity Type - Must be either STORAGE or MDQ.';

comment on column GAS_FACILITY_DETAIL_CAPACITY.VERSIONDATETIME is
'Version timestamp of the record.';

comment on column GAS_FACILITY_DETAIL_CAPACITY.LOWRANGE is
'Percentage value used for (low) validation of capacity outlook';

comment on column GAS_FACILITY_DETAIL_CAPACITY.HIRANGE is
'Percentage value used for (high) validation of capacity outlook';

comment on column GAS_FACILITY_DETAIL_CAPACITY.UNITS is
'Unit of measurement. Only TJ is used';

comment on column GAS_FACILITY_DETAIL_CAPACITY.COMMENTS is
'Free text description of record';

comment on column GAS_FACILITY_DETAIL_CAPACITY.AUTHORISEDDATE is
'Date and time this operating state was authorised';

comment on column GAS_FACILITY_DETAIL_CAPACITY.AUTHORISEDBY is
'User which authorised the change';

alter table GAS_FACILITY_DETAIL_CAPACITY
   add constraint GAS_FACILITY_DETAIL_CAP_PK primary key (FACILITYID, EFFECTIVEDATE, CAPACITYTYPE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_FACILITY_NODE                                     */
/*==============================================================*/
create table GAS_FACILITY_NODE (
   FACILITYID           NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   NODEID               NUMERIC(10,0)        not null
);

comment on table GAS_FACILITY_NODE is
'Identifies the set of nodes linked to facilities. A facility can have any number of nodes.';

comment on column GAS_FACILITY_NODE.FACILITYID is
'Unique Facility Identifier';

comment on column GAS_FACILITY_NODE.EFFECTIVEDATE is
'Effective date of the record';

comment on column GAS_FACILITY_NODE.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_FACILITY_NODE.NODEID is
'Unique Node Identifier';

alter table GAS_FACILITY_NODE
   add constraint GAS_FACILITY_NODE_PK primary key (FACILITYID, EFFECTIVEDATE, VERSIONDATETIME, NODEID);

/*==============================================================*/
/* Table: GAS_FACILITY_NODE_CONNECTION                          */
/*==============================================================*/
create table GAS_FACILITY_NODE_CONNECTION (
   FACILITYID           NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   NODEID               NUMERIC(10,0)        not null,
   CONNECTIONPOINTID    NUMERIC(10,0)        not null,
   FLOWDIRECTION        VARCHAR(20)          not null,
   ISEXEMPT             NUMERIC(1,0)         null,
   HASAGGREGATIONPRIORITY NUMERIC(1,0)         null,
   CONNECTIONPOINTNAME  VARCHAR(200)         null,
   MIRN                 VARCHAR(20)          null,
   HASFORECAST          NUMERIC(1,0)         null,
   CONNECTIONPOINTSHORTNAME VARCHAR(30)          null,
   EXEMPTIONDESCRIPTION VARCHAR(200)         null
);

comment on table GAS_FACILITY_NODE_CONNECTION is
'Identifies the set of connections associated with a node. There are up to four. Up to two for each facility/node combination.
For example, a pipe might be connected to a store via a node. Two of the connections in the node are connected to the pipe and two to the store. Each pair of connections is a delivery or a receipt type of connection.';

comment on column GAS_FACILITY_NODE_CONNECTION.FACILITYID is
'Unique Facility Identifier';

comment on column GAS_FACILITY_NODE_CONNECTION.EFFECTIVEDATE is
'Effective date of the record';

comment on column GAS_FACILITY_NODE_CONNECTION.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_FACILITY_NODE_CONNECTION.NODEID is
'Unique Node Identifier';

comment on column GAS_FACILITY_NODE_CONNECTION.CONNECTIONPOINTID is
'Unique connection point identifier.';

comment on column GAS_FACILITY_NODE_CONNECTION.FLOWDIRECTION is
'Direction of flow for the submission record. Must be one of RECEIPT or DELIVERY';

comment on column GAS_FACILITY_NODE_CONNECTION.ISEXEMPT is
'Submissions are not required if this is set to 1.';

comment on column GAS_FACILITY_NODE_CONNECTION.HASAGGREGATIONPRIORITY is
'If 1, this will be used for aggregating.';

comment on column GAS_FACILITY_NODE_CONNECTION.CONNECTIONPOINTNAME is
'Name of the Connection point';

comment on column GAS_FACILITY_NODE_CONNECTION.MIRN is
'Meter Installation Registration Number  (MIRN). Unique identifier for the Victorian DTS delivery or supply point, used to retrieve the Daily Actual Flow and Nomination and Forecast from DWGM.';

comment on column GAS_FACILITY_NODE_CONNECTION.HASFORECAST is
'Used to determine the MIRN required for Nomination and Forecast from DWGM.';

comment on column GAS_FACILITY_NODE_CONNECTION.CONNECTIONPOINTSHORTNAME is
'An abbreviation of the connection point name';

comment on column GAS_FACILITY_NODE_CONNECTION.EXEMPTIONDESCRIPTION is
'Description of the exemption';

alter table GAS_FACILITY_NODE_CONNECTION
   add constraint GAS_FACILITY_NODE_CONNECT_PK primary key (FACILITYID, EFFECTIVEDATE, VERSIONDATETIME, NODEID, CONNECTIONPOINTID, FLOWDIRECTION);

/*==============================================================*/
/* Table: GAS_FACILITY_NODE_TRK                                 */
/*==============================================================*/
create table GAS_FACILITY_NODE_TRK (
   FACILITYID           NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   AUTHORISEDBY         VARCHAR(30)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_FACILITY_NODE_TRK is
'A facility tracking record. A parent record relating to sets of child nodes linked to a facility. The set can be found in GAS_FACILITY_NODE.';

comment on column GAS_FACILITY_NODE_TRK.FACILITYID is
'Unique Facility Identifier';

comment on column GAS_FACILITY_NODE_TRK.EFFECTIVEDATE is
'Effective date of the record';

comment on column GAS_FACILITY_NODE_TRK.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_FACILITY_NODE_TRK.AUTHORISEDDATE is
'Any records without a date are ignored by AEMO systems.';

comment on column GAS_FACILITY_NODE_TRK.AUTHORISEDBY is
'User that authorised the change.';

comment on column GAS_FACILITY_NODE_TRK.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_FACILITY_NODE_TRK
   add constraint GAS_FACILITY_NODE_TRK_PK primary key (FACILITYID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_FACILITY_OPSTATE                                  */
/*==============================================================*/
create table GAS_FACILITY_OPSTATE (
   FACILITYID           NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   OPERATINGSTATE       VARCHAR(20)          null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   AUTHORISEDBY         VARCHAR(30)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_FACILITY_OPSTATE is
'The operating status of all facilities over time';

comment on column GAS_FACILITY_OPSTATE.FACILITYID is
'Unique Facility identifier';

comment on column GAS_FACILITY_OPSTATE.EFFECTIVEDATE is
'Effective date for this operating state';

comment on column GAS_FACILITY_OPSTATE.VERSIONDATETIME is
'Version date time for this operating state';

comment on column GAS_FACILITY_OPSTATE.OPERATINGSTATE is
'Operating status - active, inactive';

comment on column GAS_FACILITY_OPSTATE.AUTHORISEDDATE is
'Date and time this operating state was authorised';

comment on column GAS_FACILITY_OPSTATE.AUTHORISEDBY is
'User which authorised the change';

comment on column GAS_FACILITY_OPSTATE.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_FACILITY_OPSTATE
   add constraint GAS_FACILITY_OPSTATE_PK primary key (FACILITYID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_FACILITY_OWNER                                    */
/*==============================================================*/
create table GAS_FACILITY_OWNER (
   COMPANYID            NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   FACILITYID           NUMERIC(10,0)        not null
);

comment on table GAS_FACILITY_OWNER is
'The detail of facility ownership over time';

comment on column GAS_FACILITY_OWNER.COMPANYID is
'Unique company identifier';

comment on column GAS_FACILITY_OWNER.EFFECTIVEDATE is
'Effective date for this operating state';

comment on column GAS_FACILITY_OWNER.VERSIONDATETIME is
'Version date time for this operating state';

comment on column GAS_FACILITY_OWNER.FACILITYID is
'Unique Facility identifier';

alter table GAS_FACILITY_OWNER
   add constraint GAS_FACILITY_OWNER_PK primary key (COMPANYID, EFFECTIVEDATE, VERSIONDATETIME, FACILITYID);

/*==============================================================*/
/* Table: GAS_FACILITY_OWNER_TRK                                */
/*==============================================================*/
create table GAS_FACILITY_OWNER_TRK (
   COMPANYID            NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   AUTHORISEDBY         VARCHAR(30)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_FACILITY_OWNER_TRK is
'The tracking of Facility ownership over time';

comment on column GAS_FACILITY_OWNER_TRK.COMPANYID is
'Unique company identifier';

comment on column GAS_FACILITY_OWNER_TRK.EFFECTIVEDATE is
'Effective date for this operating state';

comment on column GAS_FACILITY_OWNER_TRK.VERSIONDATETIME is
'Version date time for this operating state';

comment on column GAS_FACILITY_OWNER_TRK.AUTHORISEDDATE is
'Date and time this operating state was authorised';

comment on column GAS_FACILITY_OWNER_TRK.AUTHORISEDBY is
'User which authorised the change';

comment on column GAS_FACILITY_OWNER_TRK.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_FACILITY_OWNER_TRK
   add constraint GAS_FACILITY_OWNER_TRK_PK primary key (COMPANYID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_FACILITY_SUMMARY                                  */
/*==============================================================*/
create table GAS_FACILITY_SUMMARY (
   FACILITYID           NUMERIC(10,0)        not null,
   STARTDATE            TIMESTAMP(0)         not null,
   ENDDATE              TIMESTAMP(0)         null,
   NODEID               NUMERIC(10,0)        null,
   CONNECTIONPOINTID    NUMERIC(10,0)        not null,
   FLOWDIRECTION        VARCHAR(20)          not null,
   CONNECTIONPOINTNAME  VARCHAR(200)         null,
   STATEID              NUMERIC(10,0)        null,
   STATE                VARCHAR(5)           null,
   FACILITYNAME         VARCHAR(100)         null,
   FACILITYTYPE         VARCHAR(40)          null,
   FACILITYTYPEDESCRIPTION VARCHAR(800)         null,
   GASDAYSTARTHOUR      NUMERIC(2,0)         null,
   COMPANYID            NUMERIC(10,0)        null,
   COMPANYNAME          VARCHAR(50)          null,
   LOCATIONID           NUMERIC(10,0)        null,
   LOCATIONNAME         VARCHAR(100)         null,
   LOCATIONTYPE         VARCHAR(40)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_FACILITY_SUMMARY is
'A summary of gas FACILITY, capacity and ownership through time';

comment on column GAS_FACILITY_SUMMARY.FACILITYID is
'Unique Facility Identifier';

comment on column GAS_FACILITY_SUMMARY.STARTDATE is
'Effective date of the record';

comment on column GAS_FACILITY_SUMMARY.ENDDATE is
'End date of the record';

comment on column GAS_FACILITY_SUMMARY.NODEID is
'Unique Node Identifier';

comment on column GAS_FACILITY_SUMMARY.CONNECTIONPOINTID is
'Unique connection point identifier.';

comment on column GAS_FACILITY_SUMMARY.FLOWDIRECTION is
'Direction of flow for the submission record. Must be one of RECEIPT or DELIVERY';

comment on column GAS_FACILITY_SUMMARY.CONNECTIONPOINTNAME is
'Name of the Connection point';

comment on column GAS_FACILITY_SUMMARY.STATEID is
'Unique state identifier';

comment on column GAS_FACILITY_SUMMARY.STATE is
'Name of the state';

comment on column GAS_FACILITY_SUMMARY.FACILITYNAME is
'Name of the Facility';

comment on column GAS_FACILITY_SUMMARY.FACILITYTYPE is
'Describes type of Facility.Valid entries are: PIPE, PROD or STOR';

comment on column GAS_FACILITY_SUMMARY.FACILITYTYPEDESCRIPTION is
'Description of the facility type';

comment on column GAS_FACILITY_SUMMARY.GASDAYSTARTHOUR is
'Column No longer populated';

comment on column GAS_FACILITY_SUMMARY.COMPANYID is
'Unique company identifier';

comment on column GAS_FACILITY_SUMMARY.COMPANYNAME is
'Name of company';

comment on column GAS_FACILITY_SUMMARY.LOCATIONID is
'Unique Node Location Identifier';

comment on column GAS_FACILITY_SUMMARY.LOCATIONNAME is
'Name of the location.';

comment on column GAS_FACILITY_SUMMARY.LOCATIONTYPE is
'Type of location';

comment on column GAS_FACILITY_SUMMARY.LASTCHANGED is
'Last changed date for the record';

alter table GAS_FACILITY_SUMMARY
   add constraint GAS_FACILITY_SUMMARY_PK primary key (FACILITYID, STARTDATE, CONNECTIONPOINTID, FLOWDIRECTION);

/*==============================================================*/
/* Table: GAS_FACILITY_TYPE                                     */
/*==============================================================*/
create table GAS_FACILITY_TYPE (
   FACILITYTYPE         VARCHAR(40)          not null,
   DESCRIPTION          VARCHAR(800)         null,
   ACTIONTYPE           VARCHAR(40)          null,
   ISREPORTABLE         NUMERIC(1,0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_FACILITY_TYPE is
'A lookup table identifying the type of facilities stored in AEMOs GBB system';

comment on column GAS_FACILITY_TYPE.FACILITYTYPE is
'Identifier for a facility type';

comment on column GAS_FACILITY_TYPE.DESCRIPTION is
'Description of the facility type';

comment on column GAS_FACILITY_TYPE.ACTIONTYPE is
'Used in aggregation';

comment on column GAS_FACILITY_TYPE.ISREPORTABLE is
'1 if reportable, otherwise 0';

comment on column GAS_FACILITY_TYPE.LASTCHANGED is
'Last time the record was changed';

alter table GAS_FACILITY_TYPE
   add constraint GAS_FACILITY_TYPE_PK primary key (FACILITYTYPE);

/*==============================================================*/
/* Table: GAS_FIELD                                             */
/*==============================================================*/
create table GAS_FIELD (
   FIELDID              NUMERIC(20,0)        not null,
   LASTCHANGED          TIMESTAMP(0)         null,
   STATEID              NUMERIC(10,0)        null,
   STATE                VARCHAR(5)           null
);

comment on table GAS_FIELD is
'Field location';

comment on column GAS_FIELD.FIELDID is
'A unique AEMO defined Field Identifier';

comment on column GAS_FIELD.LASTCHANGED is
'Date and time record was last modified';

comment on column GAS_FIELD.STATEID is
'The state ID the field interest is in';

comment on column GAS_FIELD.STATE is
'The state the field interest is in';

alter table GAS_FIELD
   add constraint GAS_FIELD_PK primary key (FIELDID);

/*==============================================================*/
/* Table: GAS_FIELDINTEREST                                     */
/*==============================================================*/
create table GAS_FIELDINTEREST (
   FIELDINTERESTID      NUMERIC(20,0)        not null,
   FIELDID              NUMERIC(20,0)        null,
   ANNUALREPORTINGDATE  VARCHAR(30)          null
);

comment on table GAS_FIELDINTEREST is
'Field Interest annual reporting date';

comment on column GAS_FIELDINTEREST.FIELDINTERESTID is
'A unique AEMO defined Field Interest Identifier';

comment on column GAS_FIELDINTEREST.FIELDID is
'A unique AEMO defined Field Identifier';

comment on column GAS_FIELDINTEREST.ANNUALREPORTINGDATE is
'The annual reporting date of the field interest';

alter table GAS_FIELDINTEREST
   add constraint GAS_FIELDINTEREST_PK primary key (FIELDINTERESTID);

/*==============================================================*/
/* Table: GAS_FIELDINTEREST_DETAIL                              */
/*==============================================================*/
create table GAS_FIELDINTEREST_DETAIL (
   FIELDINTERESTID      NUMERIC(20,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   PETROLEUMTENEMENTS   VARCHAR(300)         null,
   PROCESSINGFACILITIES VARCHAR(300)         null,
   RESOURCECLASSIFICATION VARCHAR(100)         null,
   RESOURCESUBCLASSIFICATION VARCHAR(100)         null,
   NATUREOFGAS          VARCHAR(100)         null,
   TENEMENTSHARE        NUMERIC(10,3)        null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_FIELDINTEREST_DETAIL is
'This report displays information about Field Interests';

comment on column GAS_FIELDINTEREST_DETAIL.FIELDINTERESTID is
'A unique AEMO defined Field Interest Identifier';

comment on column GAS_FIELDINTEREST_DETAIL.EFFECTIVEDATE is
'The date on which the record takes effect';

comment on column GAS_FIELDINTEREST_DETAIL.VERSIONDATETIME is
'Time a successful submission is accepted by AEMO systems';

comment on column GAS_FIELDINTEREST_DETAIL.PETROLEUMTENEMENTS is
'The petroleum tenements which are the subject of the BB field interest';

comment on column GAS_FIELDINTEREST_DETAIL.PROCESSINGFACILITIES is
'The processing facility used to process gas from the field';

comment on column GAS_FIELDINTEREST_DETAIL.RESOURCECLASSIFICATION is
'Classification of the resources in the field as conventional or unconventional';

comment on column GAS_FIELDINTEREST_DETAIL.RESOURCESUBCLASSIFICATION is
'Any further sub-classification of the resources';

comment on column GAS_FIELDINTEREST_DETAIL.NATUREOFGAS is
'The nature of the gas in the field using classifications in the BB Procedures';

comment on column GAS_FIELDINTEREST_DETAIL.TENEMENTSHARE is
'The field interest share of the petroleum tenements';

comment on column GAS_FIELDINTEREST_DETAIL.AUTHORISEDDATE is
'Date and time this operating state was authorised';

comment on column GAS_FIELDINTEREST_DETAIL.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_FIELDINTEREST_DETAIL
   add constraint GAS_FIELDINTEREST_DETAIL_PK primary key (FIELDINTERESTID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_FIELDINTEREST_PARTICIPANT                         */
/*==============================================================*/
create table GAS_FIELDINTEREST_PARTICIPANT (
   FIELDINTERESTID      NUMERIC(20,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   COMPANYID            NUMERIC(10,0)        not null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_FIELDINTEREST_PARTICIPANT is
'The responsible participant of the field interest';

comment on column GAS_FIELDINTEREST_PARTICIPANT.FIELDINTERESTID is
'A unique AEMO defined Field Interest Identifier';

comment on column GAS_FIELDINTEREST_PARTICIPANT.EFFECTIVEDATE is
'The date on which the record takes effect';

comment on column GAS_FIELDINTEREST_PARTICIPANT.VERSIONDATETIME is
'Time a successful submission is accepted by AEMO systems';

comment on column GAS_FIELDINTEREST_PARTICIPANT.COMPANYID is
'The company ID of the responsible field owner';

comment on column GAS_FIELDINTEREST_PARTICIPANT.AUTHORISEDDATE is
'Date and time this operating state was authorised';

comment on column GAS_FIELDINTEREST_PARTICIPANT.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_FIELDINTEREST_PARTICIPANT
   add constraint GAS_FIELDINT_PARTICIPANT_PK primary key (FIELDINTERESTID, EFFECTIVEDATE, VERSIONDATETIME, COMPANYID);

/*==============================================================*/
/* Table: GAS_FIELDINTEREST_SHARE                               */
/*==============================================================*/
create table GAS_FIELDINTEREST_SHARE (
   FIELDINTERESTID      NUMERIC(20,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   REPORTEDCOMPANY      VARCHAR(100)         not null,
   PERCENTAGESHARE      NUMERIC(18,8)        null
);

comment on table GAS_FIELDINTEREST_SHARE is
'Field owner group members';

comment on column GAS_FIELDINTEREST_SHARE.FIELDINTERESTID is
'A unique AEMO defined Field Interest Identifier';

comment on column GAS_FIELDINTEREST_SHARE.EFFECTIVEDATE is
'The date on which the record takes effect';

comment on column GAS_FIELDINTEREST_SHARE.VERSIONDATETIME is
'Time a successful submission is accepted by AEMO systems';

comment on column GAS_FIELDINTEREST_SHARE.REPORTEDCOMPANY is
'Member of the field owner group';

comment on column GAS_FIELDINTEREST_SHARE.PERCENTAGESHARE is
'The percentage share of the group member';

alter table GAS_FIELDINTEREST_SHARE
   add constraint GAS_FIELDINTEREST_SHARE_PK primary key (FIELDINTERESTID, EFFECTIVEDATE, VERSIONDATETIME, REPORTEDCOMPANY);

/*==============================================================*/
/* Table: GAS_FIELDINTEREST_SHARETRK                            */
/*==============================================================*/
create table GAS_FIELDINTEREST_SHARETRK (
   FIELDINTERESTID      NUMERIC(20,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_FIELDINTEREST_SHARETRK is
'Filed Share tracking details over time';

comment on column GAS_FIELDINTEREST_SHARETRK.FIELDINTERESTID is
'A unique AEMO defined Field Interest Identifier';

comment on column GAS_FIELDINTEREST_SHARETRK.EFFECTIVEDATE is
'The date on which the record takes effect';

comment on column GAS_FIELDINTEREST_SHARETRK.VERSIONDATETIME is
'Time a successful submission is accepted by AEMO systems';

comment on column GAS_FIELDINTEREST_SHARETRK.AUTHORISEDDATE is
'Date and time this operating state was authorised';

comment on column GAS_FIELDINTEREST_SHARETRK.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_FIELDINTEREST_SHARETRK
   add constraint GAS_FIELDINTEREST_SHARETRK_PK primary key (FIELDINTERESTID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_FIELD_DETAIL                                      */
/*==============================================================*/
create table GAS_FIELD_DETAIL (
   FIELDID              NUMERIC(20,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   FIELDNAME            VARCHAR(100)         null,
   FIELDSHORTNAME       VARCHAR(30)          null,
   DESCRIPTION          VARCHAR(400)         null,
   LOCATIONDESCRIPTION  VARCHAR(200)         null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_FIELD_DETAIL is
'Field details';

comment on column GAS_FIELD_DETAIL.FIELDID is
'A unique AEMO defined Field Identifier';

comment on column GAS_FIELD_DETAIL.EFFECTIVEDATE is
'The date on which the record takes effect';

comment on column GAS_FIELD_DETAIL.VERSIONDATETIME is
'Time a successful submission is accepted by AEMO systems';

comment on column GAS_FIELD_DETAIL.FIELDNAME is
'The name of the Field in which the Field Interest is located';

comment on column GAS_FIELD_DETAIL.FIELDSHORTNAME is
'The short name of the field';

comment on column GAS_FIELD_DETAIL.DESCRIPTION is
'Additional information relating to the field';

comment on column GAS_FIELD_DETAIL.LOCATIONDESCRIPTION is
'Additional information relating to the location of the field';

comment on column GAS_FIELD_DETAIL.AUTHORISEDDATE is
'Date and time this operating state was authorised';

comment on column GAS_FIELD_DETAIL.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_FIELD_DETAIL
   add constraint GAS_FIELD_DETAIL_PK primary key (FIELDID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_GSH_TRADES                                        */
/*==============================================================*/
create table GAS_GSH_TRADES (
   TRADE_DATE           TIMESTAMP(0)         not null,
   TRANSACTION_TYPE     VARCHAR(20)          not null,
   PRODUCT              VARCHAR(80)          not null,
   LOCATION             VARCHAR(100)         not null,
   NO_OF_TRADES         NUMERIC(10)          null,
   TRADE_PRICE          NUMERIC(18,8)        not null,
   DAILY_QTY_GJ         NUMERIC(18,8)        null,
   START_DATE           TIMESTAMP(0)         null,
   END_DATE             TIMESTAMP(0)         null,
   MANUAL_TRADE         VARCHAR(50)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_GSH_TRADES is
'Provides participants (buyer and seller) a confirmation whenever a trade is executed through auto matching of bids and offers or by off-market trades. Also provides participants with a list of all their future trades when triggered daily after the end of the trades for the current day';

comment on column GAS_GSH_TRADES.TRADE_DATE is
'Date and time the trade is executed';

comment on column GAS_GSH_TRADES.TRANSACTION_TYPE is
'The type of the trade. Valid values are AUTO_MATCHED, OFF_MARKET_TRADE';

comment on column GAS_GSH_TRADES.PRODUCT is
'The product delivery period for the Gas Supply Hub market';

comment on column GAS_GSH_TRADES.LOCATION is
'Product location for the Gas Supply Hub market';

comment on column GAS_GSH_TRADES.NO_OF_TRADES is
'Count of number of trades';

comment on column GAS_GSH_TRADES.TRADE_PRICE is
'Price value of the trade in $/GJ';

comment on column GAS_GSH_TRADES.DAILY_QTY_GJ is
'Volume of gas transacted on the Trading Platform in GJ/day';

comment on column GAS_GSH_TRADES.START_DATE is
'The start gas day for the order delivery period';

comment on column GAS_GSH_TRADES.END_DATE is
'The end gas day for the order delivery period';

comment on column GAS_GSH_TRADES.MANUAL_TRADE is
'Whether the trade is a manual trade';

comment on column GAS_GSH_TRADES.LASTCHANGED is
'The date the record was last updated';

alter table GAS_GSH_TRADES
   add constraint GAS_GSH_TRADES_PK primary key (TRADE_DATE, TRANSACTION_TYPE, PRODUCT, LOCATION, TRADE_PRICE);

/*==============================================================*/
/* Table: GAS_LINEPACK_ADEQUACY                                 */
/*==============================================================*/
create table GAS_LINEPACK_ADEQUACY (
   GASDATE              TIMESTAMP(0)         not null,
   FACILITYID           NUMERIC(10,0)        not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   FLAG                 VARCHAR(5)           null,
   DESCRIPTION          VARCHAR(800)         null
);

comment on table GAS_LINEPACK_ADEQUACY is
'The linepack adequacy (flag) for each of the (non-exempt) BB pipelines';

comment on column GAS_LINEPACK_ADEQUACY.GASDATE is
'Date of gas day applicable under either the pipeline contract or market rules';

comment on column GAS_LINEPACK_ADEQUACY.FACILITYID is
'Unique Facility identifier';

comment on column GAS_LINEPACK_ADEQUACY.VERSIONDATETIME is
'Version date time for this operating state';

comment on column GAS_LINEPACK_ADEQUACY.FLAG is
'Expected adequacy for the gas day. Valid entries are GREEN, AMBER or RED';

comment on column GAS_LINEPACK_ADEQUACY.DESCRIPTION is
'Free text description of record';

alter table GAS_LINEPACK_ADEQUACY
   add constraint GAS_LINEPACK_ADEQUACY_PK primary key (GASDATE, FACILITYID, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_LNG_SHIPMENT                                      */
/*==============================================================*/
create table GAS_LNG_SHIPMENT (
   TRANSACTIONID        VARCHAR(40)          not null,
   FACILITYID           NUMERIC(20,0)        not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   SHIPMENTDATE         TIMESTAMP(0)         null,
   VOLUMEPJ             NUMERIC(18,8)        null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_LNG_SHIPMENT is
'This table displays a list of all LNG shipments';

comment on column GAS_LNG_SHIPMENT.TRANSACTIONID is
'Unique shipment identifier';

comment on column GAS_LNG_SHIPMENT.FACILITYID is
'Unique facility identifier';

comment on column GAS_LNG_SHIPMENT.VERSIONDATETIME is
'Time a successful submission is accepted by AEMO systems';

comment on column GAS_LNG_SHIPMENT.SHIPMENTDATE is
'For LNG export facility, the departure date. For LNG import facility, the date unloading commences at the LNG import facility';

comment on column GAS_LNG_SHIPMENT.VOLUMEPJ is
'Volume of the shipment in PJ';

comment on column GAS_LNG_SHIPMENT.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_LNG_SHIPMENT
   add constraint GAS_LNG_SHIPMENT_PK primary key (TRANSACTIONID, FACILITYID, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_LNG_TRANSACTION                                   */
/*==============================================================*/
create table GAS_LNG_TRANSACTION (
   TRANSACTIONSTARTDATE TIMESTAMP(0)         not null,
   TRANSACTIONENDDATE   TIMESTAMP(0)         null,
   VOLWEIGHTPRICE       NUMERIC(18,8)        null,
   VOLUMEPJ             NUMERIC(18,8)        null,
   SUPPLYPERIODSTART    TIMESTAMP(0)         null,
   SUPPLYPERIODEND      TIMESTAMP(0)         null
);

comment on table GAS_LNG_TRANSACTION is
'This table displays LNG transaction aggregated data';

comment on column GAS_LNG_TRANSACTION.TRANSACTIONSTARTDATE is
'Transaction start date';

comment on column GAS_LNG_TRANSACTION.TRANSACTIONENDDATE is
'Transaction end date';

comment on column GAS_LNG_TRANSACTION.VOLWEIGHTPRICE is
'The volume weighted price for the reporting period';

comment on column GAS_LNG_TRANSACTION.VOLUMEPJ is
'The total volume of the transactions';

comment on column GAS_LNG_TRANSACTION.SUPPLYPERIODSTART is
'The earliest start date of all transactions captured in the reporting period';

comment on column GAS_LNG_TRANSACTION.SUPPLYPERIODEND is
'The latest end date of all transactions captured in the reporting period';

alter table GAS_LNG_TRANSACTION
   add constraint GAS_LNG_TRANSACTION_PK primary key (TRANSACTIONSTARTDATE);

/*==============================================================*/
/* Table: GAS_LOCATION                                          */
/*==============================================================*/
create table GAS_LOCATION (
   LOCATIONID           NUMERIC(10,0)        not null,
   LOCATIONNAME         VARCHAR(100)         null,
   LOCATIONTYPE         VARCHAR(40)          null,
   STATEID              NUMERIC(10,0)        null,
   DESCRIPTION          VARCHAR(800)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_LOCATION is
'Lists all production and demand locations within the Bulletin Board system.';

comment on column GAS_LOCATION.LOCATIONID is
'Unique Node Location Identifier';

comment on column GAS_LOCATION.LOCATIONNAME is
'Name of the location.';

comment on column GAS_LOCATION.LOCATIONTYPE is
'Type of location';

comment on column GAS_LOCATION.STATEID is
'Unique state identifier';

comment on column GAS_LOCATION.DESCRIPTION is
'Location description';

comment on column GAS_LOCATION.LASTCHANGED is
'Last time the record was changed.';

alter table GAS_LOCATION
   add constraint GAS_LOCATION_PK primary key (LOCATIONID);

/*==============================================================*/
/* Table: GAS_LOCATION_NODE                                     */
/*==============================================================*/
create table GAS_LOCATION_NODE (
   LOCATIONID           NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   NODEID               NUMERIC(10,0)        not null
);

comment on table GAS_LOCATION_NODE is
'Provides a list of nodes within a location.';

comment on column GAS_LOCATION_NODE.LOCATIONID is
'Unique Location Identifier';

comment on column GAS_LOCATION_NODE.EFFECTIVEDATE is
'Effective date of the record';

comment on column GAS_LOCATION_NODE.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_LOCATION_NODE.NODEID is
'Unique Node Identifier';

alter table GAS_LOCATION_NODE
   add constraint GAS_LOCATION_NODE_PK primary key (LOCATIONID, EFFECTIVEDATE, VERSIONDATETIME, NODEID);

/*==============================================================*/
/* Table: GAS_LOCATION_NODE_TRK                                 */
/*==============================================================*/
create table GAS_LOCATION_NODE_TRK (
   LOCATIONID           NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   AUTHORISEDBY         VARCHAR(30)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_LOCATION_NODE_TRK is
'Provides a record of node tracking.';

comment on column GAS_LOCATION_NODE_TRK.LOCATIONID is
'Unique Location identifier.';

comment on column GAS_LOCATION_NODE_TRK.EFFECTIVEDATE is
'Effective date of the record';

comment on column GAS_LOCATION_NODE_TRK.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_LOCATION_NODE_TRK.AUTHORISEDDATE is
'Date and time this operating state was authorised';

comment on column GAS_LOCATION_NODE_TRK.AUTHORISEDBY is
'User which authorised the change';

comment on column GAS_LOCATION_NODE_TRK.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_LOCATION_NODE_TRK
   add constraint GAS_LOCATION_NODE_TRK_PK primary key (LOCATIONID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_LOCATION_OPSTATE                                  */
/*==============================================================*/
create table GAS_LOCATION_OPSTATE (
   LOCATIONID           NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   OPERATINGSTATE       VARCHAR(20)          null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   AUTHORISEDBY         VARCHAR(30)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_LOCATION_OPSTATE is
'The operating status of all locations over time.';

comment on column GAS_LOCATION_OPSTATE.LOCATIONID is
'Unique Location identifier.';

comment on column GAS_LOCATION_OPSTATE.EFFECTIVEDATE is
'Effective date of the record';

comment on column GAS_LOCATION_OPSTATE.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_LOCATION_OPSTATE.OPERATINGSTATE is
'Operating status - active, inactive.';

comment on column GAS_LOCATION_OPSTATE.AUTHORISEDDATE is
'Date and time this operating state was authorised.';

comment on column GAS_LOCATION_OPSTATE.AUTHORISEDBY is
'User that authorised the change';

comment on column GAS_LOCATION_OPSTATE.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_LOCATION_OPSTATE
   add constraint GAS_LOCATION_OPSTATE_PK primary key (LOCATIONID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_MARKET                                            */
/*==============================================================*/
create table GAS_MARKET (
   MARKETID             VARCHAR(20)          not null,
   DESCRIPTION          VARCHAR(255)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_MARKET is
'The list of defined Gas Markets against which the Participant Operating Status is activated';

comment on column GAS_MARKET.MARKETID is
'Unique market identifier';

comment on column GAS_MARKET.DESCRIPTION is
'Market description';

comment on column GAS_MARKET.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_MARKET
   add constraint GAS_MARKET_PK primary key (MARKETID);

/*==============================================================*/
/* Table: GAS_MEDIUM_TERM_CAP_OUTLOOK                           */
/*==============================================================*/
create table GAS_MEDIUM_TERM_CAP_OUTLOOK (
   FACILITYID           NUMERIC(10,0)        not null,
   FROMGASDATE          TIMESTAMP(0)         not null,
   TOGASDATE            TIMESTAMP(0)         null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   CAPACITYTYPE         VARCHAR(20)          not null,
   FLOWDIRECTION        VARCHAR(20)          not null,
   RECEIPTLOCATION      NUMERIC(10,0)        not null,
   DELIVERYLOCATION     NUMERIC(10,0)        not null,
   OUTLOOKQUANTITY      NUMERIC(18,3)        null,
   CAPACITYDESCRIPTION  VARCHAR(1000)        null,
   DESCRIPTION          VARCHAR(1000)        null,
   ACTIVEFLAG           NUMERIC(1,0)         null,
   DISABLEDDATETIME     TIMESTAMP(0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_MEDIUM_TERM_CAP_OUTLOOK is
'Provide the details of activities that may affect the daily capacity of the facility over the next 12 months.';

comment on column GAS_MEDIUM_TERM_CAP_OUTLOOK.FACILITYID is
'Unique Facility Identifier';

comment on column GAS_MEDIUM_TERM_CAP_OUTLOOK.FROMGASDATE is
'Start date (inclusive) of the record';

comment on column GAS_MEDIUM_TERM_CAP_OUTLOOK.TOGASDATE is
'End date (exclusive) of the record';

comment on column GAS_MEDIUM_TERM_CAP_OUTLOOK.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_MEDIUM_TERM_CAP_OUTLOOK.CAPACITYTYPE is
'Capacity Type - Must be either STORAGE or MDQ';

comment on column GAS_MEDIUM_TERM_CAP_OUTLOOK.FLOWDIRECTION is
'Direction of flow for the submission record. Must be one of RECEIPT or DELIVERY';

comment on column GAS_MEDIUM_TERM_CAP_OUTLOOK.RECEIPTLOCATION is
'Connection Point ID that best represents the receipt location';

comment on column GAS_MEDIUM_TERM_CAP_OUTLOOK.DELIVERYLOCATION is
'Connection Point ID that best represents the Delivery Location';

comment on column GAS_MEDIUM_TERM_CAP_OUTLOOK.OUTLOOKQUANTITY is
'Quantity of available capacity during the defined period';

comment on column GAS_MEDIUM_TERM_CAP_OUTLOOK.CAPACITYDESCRIPTION is
'Describes the meaning of the capacity Number Provided';

comment on column GAS_MEDIUM_TERM_CAP_OUTLOOK.DESCRIPTION is
'Description for reasons or comments directly related to the quantity or change in Outlook quantity in relation to the Facility';

comment on column GAS_MEDIUM_TERM_CAP_OUTLOOK.ACTIVEFLAG is
'Active flag for the record.  A value of 1 indicates the record is currently active';

comment on column GAS_MEDIUM_TERM_CAP_OUTLOOK.DISABLEDDATETIME is
'Audit timestamp of the record';

comment on column GAS_MEDIUM_TERM_CAP_OUTLOOK.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_MEDIUM_TERM_CAP_OUTLOOK
   add constraint GAS_MEDIUM_TERM_CAP_OUTLOOK_PK primary key (FACILITYID, FROMGASDATE, VERSIONDATETIME, CAPACITYTYPE, FLOWDIRECTION, RECEIPTLOCATION, DELIVERYLOCATION);

/*==============================================================*/
/* Table: GAS_NAMEPLATE_RATING                                  */
/*==============================================================*/
create table GAS_NAMEPLATE_RATING (
   FACILITYID           NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   CAPACITYTYPE         VARCHAR(20)          not null,
   FLOWDIRECTION        VARCHAR(20)          not null,
   RECEIPTLOCATION      NUMERIC(10,0)        not null,
   DELIVERYLOCATION     NUMERIC(10,0)        not null,
   CAPACITYQUANTITY     NUMERIC(18,3)        null,
   CAPACITYDESCRIPTION  VARCHAR(1000)        null,
   DESCRIPTION          VARCHAR(1000)        null
);

comment on table GAS_NAMEPLATE_RATING is
'Provide the nameplate rating of each facility or information about any planned permanent capacity reduction or expansion due to modification of the facility.';

comment on column GAS_NAMEPLATE_RATING.FACILITYID is
'Unique Facility Identifier';

comment on column GAS_NAMEPLATE_RATING.EFFECTIVEDATE is
'Effective date of the record';

comment on column GAS_NAMEPLATE_RATING.VERSIONDATETIME is
'Version timestamp of the record.';

comment on column GAS_NAMEPLATE_RATING.CAPACITYTYPE is
'Capacity Type - Must be either STORAGE or MDQ';

comment on column GAS_NAMEPLATE_RATING.FLOWDIRECTION is
'Direction of flow for the submission record. Must be either RECEIPT or DELIVERY.';

comment on column GAS_NAMEPLATE_RATING.RECEIPTLOCATION is
'Connection Point ID that best represents the Receipt location';

comment on column GAS_NAMEPLATE_RATING.DELIVERYLOCATION is
'Connection Point ID that best represents the Delivery Location';

comment on column GAS_NAMEPLATE_RATING.CAPACITYQUANTITY is
'Quantity of available capacity during the defined period.';

comment on column GAS_NAMEPLATE_RATING.CAPACITYDESCRIPTION is
'Describes the meaning of the capacity Number Provided';

comment on column GAS_NAMEPLATE_RATING.DESCRIPTION is
'Description for reasons or comments directly related to the quantity or change in Outlook quantity in relation to the Facility.';

alter table GAS_NAMEPLATE_RATING
   add constraint GAS_NAMEPLATE_RATING_PK primary key (FACILITYID, EFFECTIVEDATE, VERSIONDATETIME, CAPACITYTYPE, FLOWDIRECTION, RECEIPTLOCATION, DELIVERYLOCATION);

/*==============================================================*/
/* Table: GAS_NODE                                              */
/*==============================================================*/
create table GAS_NODE (
   NODEID               NUMERIC(10,0)        not null,
   STATEID              NUMERIC(10,0)        null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_NODE is
'A container for up to four connections';

comment on column GAS_NODE.NODEID is
'Unique Node Identifier';

comment on column GAS_NODE.STATEID is
'Unique StateIdentifier';

comment on column GAS_NODE.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_NODE
   add constraint GAS_NODE_PK primary key (NODEID);

/*==============================================================*/
/* Table: GAS_NODE_OPSTATE                                      */
/*==============================================================*/
create table GAS_NODE_OPSTATE (
   NODEID               NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   OPERATINGSTATE       VARCHAR(20)          null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   AUTHORISEDBY         VARCHAR(30)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_NODE_OPSTATE is
'The operating status of all facility nodes over time';

comment on column GAS_NODE_OPSTATE.NODEID is
'Unique Node Identifier';

comment on column GAS_NODE_OPSTATE.EFFECTIVEDATE is
'Effective date of the record';

comment on column GAS_NODE_OPSTATE.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_NODE_OPSTATE.OPERATINGSTATE is
'Operating status - active, inactive.';

comment on column GAS_NODE_OPSTATE.AUTHORISEDDATE is
'Date and time this operating state was authorised.';

comment on column GAS_NODE_OPSTATE.AUTHORISEDBY is
'User that authorised the change.';

comment on column GAS_NODE_OPSTATE.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_NODE_OPSTATE
   add constraint GAS_NODE_OPSTATE_PK primary key (NODEID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_NOMINATIONS_FORECAST                              */
/*==============================================================*/
create table GAS_NOMINATIONS_FORECAST (
   GASDATE              TIMESTAMP(0)         not null,
   FACILITYID           NUMERIC(10,0)        not null,
   CONNECTIONPOINTID    NUMERIC(10,0)        not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   FLOWDIRECTION        VARCHAR(20)          not null,
   NOMINATIONQUANTITY   NUMERIC(18,3)        null
);

comment on table GAS_NOMINATIONS_FORECAST is
'For BB pipelines forming part of a Declared Transmission System, provide on each gas day the aggregated scheduled injections and aggregated scheduled withdrawals at each controllable system point for gas day D to D+2.
For all other BB facility operators, provide on each gas day D the aggregated nominated and forecast injections and aggregated nominated and forecast withdrawals for gas day D to D+6.';

comment on column GAS_NOMINATIONS_FORECAST.GASDATE is
'Date of gas day';

comment on column GAS_NOMINATIONS_FORECAST.FACILITYID is
'Unique Facility Identifier';

comment on column GAS_NOMINATIONS_FORECAST.CONNECTIONPOINTID is
'Unique connection point identifier.';

comment on column GAS_NOMINATIONS_FORECAST.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_NOMINATIONS_FORECAST.FLOWDIRECTION is
'Direction of flow for the submission record. Must be one of RECEIPT or DELIVERY';

comment on column GAS_NOMINATIONS_FORECAST.NOMINATIONQUANTITY is
'Receipt or delivery nomination quantity in TJ';

alter table GAS_NOMINATIONS_FORECAST
   add constraint GAS_NOMINATIONS_FORECAST_PK primary key (GASDATE, FACILITYID, CONNECTIONPOINTID, VERSIONDATETIME, FLOWDIRECTION);

/*==============================================================*/
/* Table: GAS_NOMINATIONS_FORECAST_AGG                          */
/*==============================================================*/
create table GAS_NOMINATIONS_FORECAST_AGG (
   GASDATE              TIMESTAMP(0)         not null,
   FACILITYID           NUMERIC(10,0)        not null,
   LOCATIONID           NUMERIC(10,0)        not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   DEMAND               NUMERIC(18,3)        null,
   SUPPLY               NUMERIC(18,3)        null,
   TRANSFERIN           NUMERIC(18,3)        null,
   TRANSFEROUT          NUMERIC(18,3)        null,
   COMPLETENESS         NUMERIC(3,0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_NOMINATIONS_FORECAST_AGG is
'Provides the aggregate of forecast and nominations by location.';

comment on column GAS_NOMINATIONS_FORECAST_AGG.GASDATE is
'Date of gas day';

comment on column GAS_NOMINATIONS_FORECAST_AGG.FACILITYID is
'Unique Facility Identifier';

comment on column GAS_NOMINATIONS_FORECAST_AGG.LOCATIONID is
'Unique Node Location Identifier';

comment on column GAS_NOMINATIONS_FORECAST_AGG.VERSIONDATETIME is
'Version date time of the record';

comment on column GAS_NOMINATIONS_FORECAST_AGG.DEMAND is
'Usage type expressed in TJ. Three decimal places is not shown if the value has trailing zeros after the decimal place.';

comment on column GAS_NOMINATIONS_FORECAST_AGG.SUPPLY is
'Usage type expressed in TJ. Three decimal places is not shown if the value has trailing zeros after the decimal place';

comment on column GAS_NOMINATIONS_FORECAST_AGG.TRANSFERIN is
'Usage type expressed in TJ. Only applicable to BB pipelines. Three decimal places is not shown if the value has trailing zeros after the decimal place.';

comment on column GAS_NOMINATIONS_FORECAST_AGG.TRANSFEROUT is
'Usage type expressed in TJ. Only applicable to BB pipelines. Three decimal places is not shown if the value has trailing zeros after the decimal place.';

comment on column GAS_NOMINATIONS_FORECAST_AGG.COMPLETENESS is
'Completeness of aggregated figure in percentage';

comment on column GAS_NOMINATIONS_FORECAST_AGG.LASTCHANGED is
'Last changed date for the record';

alter table GAS_NOMINATIONS_FORECAST_AGG
   add constraint GAS_NOMINATIONS_FCST_AGG_PK primary key (GASDATE, FACILITYID, LOCATIONID, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_PARTICIPANT                                       */
/*==============================================================*/
create table GAS_PARTICIPANT (
   COMPANYID            NUMERIC(10,0)        not null,
   DESCRIPTION          VARCHAR(800)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_PARTICIPANT is
'The list of all registered organisations i.e  the BB Participants of the Bulletin Board';

comment on column GAS_PARTICIPANT.COMPANYID is
'Unique company identifier';

comment on column GAS_PARTICIPANT.DESCRIPTION is
'Name of company';

comment on column GAS_PARTICIPANT.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_PARTICIPANT
   add constraint GAS_PARTICIPANT_PK primary key (COMPANYID);

/*==============================================================*/
/* Table: GAS_PARTICIPANT_CONTACT                               */
/*==============================================================*/
create table GAS_PARTICIPANT_CONTACT (
   COMPANYID            NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   PERSONID             NUMERIC(10,0)        not null
);

comment on table GAS_PARTICIPANT_CONTACT is
'The detail of contacts against a gas participant over time';

comment on column GAS_PARTICIPANT_CONTACT.COMPANYID is
'Unique company identifier';

comment on column GAS_PARTICIPANT_CONTACT.EFFECTIVEDATE is
'Effective date for this operating state';

comment on column GAS_PARTICIPANT_CONTACT.VERSIONDATETIME is
'Version date time for this operating state';

comment on column GAS_PARTICIPANT_CONTACT.PERSONID is
'Unique person identifier';

alter table GAS_PARTICIPANT_CONTACT
   add constraint GAS_PARTICIPANT_CONTACT_PK primary key (COMPANYID, EFFECTIVEDATE, VERSIONDATETIME, PERSONID);

/*==============================================================*/
/* Table: GAS_PARTICIPANT_CONTACT_TRK                           */
/*==============================================================*/
create table GAS_PARTICIPANT_CONTACT_TRK (
   COMPANYID            NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   AUTHORISEDBY         VARCHAR(30)          null
);

comment on table GAS_PARTICIPANT_CONTACT_TRK is
'The tracking of sets of contacts against a gas participant over time';

comment on column GAS_PARTICIPANT_CONTACT_TRK.COMPANYID is
'Unique company identifier';

comment on column GAS_PARTICIPANT_CONTACT_TRK.EFFECTIVEDATE is
'Effective date for this operating state';

comment on column GAS_PARTICIPANT_CONTACT_TRK.VERSIONDATETIME is
'Version date time for this operating state';

comment on column GAS_PARTICIPANT_CONTACT_TRK.AUTHORISEDDATE is
'Date and time this operating state was authorised';

comment on column GAS_PARTICIPANT_CONTACT_TRK.AUTHORISEDBY is
'User which authorised the change';

alter table GAS_PARTICIPANT_CONTACT_TRK
   add constraint GAS_PARTICIPANT_CONTACT_TRK_PK primary key (COMPANYID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_PARTICIPANT_DETAIL                                */
/*==============================================================*/
create table GAS_PARTICIPANT_DETAIL (
   COMPANYID            NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   COMPANYNAME          VARCHAR(50)          null,
   ABN                  VARCHAR(30)          null,
   ADDRESSTYPE          VARCHAR(40)          null,
   ADDRESS              VARCHAR(120)         null,
   LOCALE               VARCHAR(40)          null,
   JURISDICTION         VARCHAR(5)           null,
   POSTCODE             VARCHAR(4)           null,
   COMPANYPHONE         VARCHAR(30)          null,
   COMPANYFAX           VARCHAR(30)          null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   AUTHORISEDBY         VARCHAR(30)          null,
   ORGANISATIONTYPENAME VARCHAR(40)          null
);

comment on table GAS_PARTICIPANT_DETAIL is
'The details of all registered organisations i.e  the BB Participants of the Bulletin Board over time';

comment on column GAS_PARTICIPANT_DETAIL.COMPANYID is
'Unique company identifier';

comment on column GAS_PARTICIPANT_DETAIL.EFFECTIVEDATE is
'Effective date for this operating state';

comment on column GAS_PARTICIPANT_DETAIL.VERSIONDATETIME is
'Version date time for this operating state';

comment on column GAS_PARTICIPANT_DETAIL.COMPANYNAME is
'Name of company';

comment on column GAS_PARTICIPANT_DETAIL.ABN is
'Australian Business Number of company';

comment on column GAS_PARTICIPANT_DETAIL.ADDRESSTYPE is
'Type of address eg. Head Office, Postal Address etc.';

comment on column GAS_PARTICIPANT_DETAIL.ADDRESS is
'Address of company based on AddressType';

comment on column GAS_PARTICIPANT_DETAIL.LOCALE is
'City location of company';

comment on column GAS_PARTICIPANT_DETAIL.JURISDICTION is
'State location of company';

comment on column GAS_PARTICIPANT_DETAIL.POSTCODE is
'Postcode of company';

comment on column GAS_PARTICIPANT_DETAIL.COMPANYPHONE is
'Main (reception) phone of company';

comment on column GAS_PARTICIPANT_DETAIL.COMPANYFAX is
'Main (reception) facsimile of company';

comment on column GAS_PARTICIPANT_DETAIL.AUTHORISEDDATE is
'Date and time this operating state was authorised';

comment on column GAS_PARTICIPANT_DETAIL.AUTHORISEDBY is
'User which authorised the change';

comment on column GAS_PARTICIPANT_DETAIL.ORGANISATIONTYPENAME is
'The type of organisation';

alter table GAS_PARTICIPANT_DETAIL
   add constraint GAS_PARTICIPANT_DETAIL_PK primary key (COMPANYID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_PARTICIPANT_OPSTATE                               */
/*==============================================================*/
create table GAS_PARTICIPANT_OPSTATE (
   MARKETID             VARCHAR(20)          not null,
   COMPANYID            NUMERIC(10,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   OPERATINGSTATE       VARCHAR(20)          null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   AUTHORISEDBY         VARCHAR(30)          null
);

comment on table GAS_PARTICIPANT_OPSTATE is
'The operating status of all registered organisations i.e  the BB Participants of the Bulletin Board over time';

comment on column GAS_PARTICIPANT_OPSTATE.MARKETID is
'Unique Market identifier';

comment on column GAS_PARTICIPANT_OPSTATE.COMPANYID is
'Unique company identifier';

comment on column GAS_PARTICIPANT_OPSTATE.EFFECTIVEDATE is
'Effective date for this operating state';

comment on column GAS_PARTICIPANT_OPSTATE.VERSIONDATETIME is
'Version date time for this operating state';

comment on column GAS_PARTICIPANT_OPSTATE.OPERATINGSTATE is
'Operating status - active, inactive';

comment on column GAS_PARTICIPANT_OPSTATE.AUTHORISEDDATE is
'Date and time this operating state was authorised';

comment on column GAS_PARTICIPANT_OPSTATE.AUTHORISEDBY is
'User which authorised the change';

alter table GAS_PARTICIPANT_OPSTATE
   add constraint GAS_PARTICIPANT_OPSTATE_PK primary key (MARKETID, COMPANYID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_RESERVES_AND_RESOURCES                            */
/*==============================================================*/
create table GAS_RESERVES_AND_RESOURCES (
   FIELDINTERESTID      NUMERIC(20,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   DEVELOPEDRESERVE1P   NUMERIC(18,3)        null,
   DEVELOPEDRESERVE2P   NUMERIC(18,3)        null,
   DEVELOPEDRESERVE3P   NUMERIC(18,3)        null,
   UNDEVELOPEDRESERVE1P NUMERIC(18,3)        null,
   UNDEVELOPEDRESERVE2P NUMERIC(18,3)        null,
   UNDEVELOPEDRESERVE3P NUMERIC(18,3)        null,
   RESOURCES2C          NUMERIC(18,3)        null,
   PRODUCTIONCHANGERESERVE2P NUMERIC(18,3)        null,
   PROVEDAREAEXTENSIONRESERVE2P NUMERIC(18,3)        null,
   PERCENTAGECHANGERESERVE2P NUMERIC(18,3)        null,
   UPWARDREVISION3PRESERVETO2P NUMERIC(18,3)        null,
   DOWNWARDREVISION2PRESERVETO3P NUMERIC(18,3)        null,
   OTHERREVISIONSRESERVE2P NUMERIC(18,3)        null,
   MATURITYSUBCLASS2P   VARCHAR(100)         null,
   MATURITYSUBCLASS2C   VARCHAR(100)         null,
   MINDATE2P            TIMESTAMP(0)         null,
   MAXDATE2P            TIMESTAMP(0)         null,
   MINDATE2C            TIMESTAMP(0)         null,
   MAXDATE2C            TIMESTAMP(0)         null,
   EXPECTEDBARRIERS2C   VARCHAR(400)         null,
   RESOURCESESTIMATEMETHOD VARCHAR(200)         null,
   CONVERSIONFACTORQTYTCFTOPJ NUMERIC(18,3)        null,
   ECONOMICASSUMPTION   VARCHAR(400)         null,
   UPDATEREASON         VARCHAR(400)         null,
   PREPAREDBY           VARCHAR(100)         null,
   INDEPENDENTESTIMATE  VARCHAR(3)           null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_RESERVES_AND_RESOURCES is
'This table displays information about Field Reserves and Resources';

comment on column GAS_RESERVES_AND_RESOURCES.FIELDINTERESTID is
'A unique AEMO defined Field Interest Identifier';

comment on column GAS_RESERVES_AND_RESOURCES.EFFECTIVEDATE is
'The date on which the record takes effect';

comment on column GAS_RESERVES_AND_RESOURCES.VERSIONDATETIME is
'Time a successful submission is accepted by AEMO systems';

comment on column GAS_RESERVES_AND_RESOURCES.DEVELOPEDRESERVE1P is
'An estimate of the BB field interest 1P developed reserves';

comment on column GAS_RESERVES_AND_RESOURCES.DEVELOPEDRESERVE2P is
'An estimate of the BB field interest 2P developed reserves';

comment on column GAS_RESERVES_AND_RESOURCES.DEVELOPEDRESERVE3P is
'An estimate of the BB field interest 3P developed reserves';

comment on column GAS_RESERVES_AND_RESOURCES.UNDEVELOPEDRESERVE1P is
'An estimate of the BB field interest 1P undeveloped reserves';

comment on column GAS_RESERVES_AND_RESOURCES.UNDEVELOPEDRESERVE2P is
'An estimate of the BB field interest 2P undeveloped reserves';

comment on column GAS_RESERVES_AND_RESOURCES.UNDEVELOPEDRESERVE3P is
'An estimate of the BB field interest 3P undeveloped reserves';

comment on column GAS_RESERVES_AND_RESOURCES.RESOURCES2C is
'An estimate of the BB field interest 2C resources';

comment on column GAS_RESERVES_AND_RESOURCES.PRODUCTIONCHANGERESERVE2P is
'An estimate of the total movement in the BB field interest 2P reserves since the end of prior reporting year due to the production of gas';

comment on column GAS_RESERVES_AND_RESOURCES.PROVEDAREAEXTENSIONRESERVE2P is
'An estimate of the total movement in the BB field interest 2P reserves since the end of prior reporting year due to the extension of a fields proved area';

comment on column GAS_RESERVES_AND_RESOURCES.PERCENTAGECHANGERESERVE2P is
'An estimate of the total movement in the BB field interest 2P reserves since the end of prior reporting year due to a percentage change in the BB field interest';

comment on column GAS_RESERVES_AND_RESOURCES.UPWARDREVISION3PRESERVETO2P is
'An estimate of the total movement in the BB field interest 2P reserves since the end of prior reporting year due to an upward revision of 2P reserves arising from the reclassification of 3P reserves or resources to 2P reserves';

comment on column GAS_RESERVES_AND_RESOURCES.DOWNWARDREVISION2PRESERVETO3P is
'An estimate of the total movement in the BB field interest 2P reserves since the end of prior reporting year due to a downward revision of 2P reserves arising from the reclassification of 2P reserves to 3P reserves or resources';

comment on column GAS_RESERVES_AND_RESOURCES.OTHERREVISIONSRESERVE2P is
'An estimate of the total movement in the BB field interest 2P reserves since the end of prior reporting year due to other revisions';

comment on column GAS_RESERVES_AND_RESOURCES.MATURITYSUBCLASS2P is
'The project maturity sub-class for the 2P reserves';

comment on column GAS_RESERVES_AND_RESOURCES.MATURITYSUBCLASS2C is
'The project maturity sub-class for the 2C resources';

comment on column GAS_RESERVES_AND_RESOURCES.MINDATE2P is
'The earliest estimated date for the production of the 2P reserves';

comment on column GAS_RESERVES_AND_RESOURCES.MAXDATE2P is
'The latest estimated date for the production of the 2P reserves';

comment on column GAS_RESERVES_AND_RESOURCES.MINDATE2C is
'The earliest estimated date for the production of the 2C resources';

comment on column GAS_RESERVES_AND_RESOURCES.MAXDATE2C is
'The latest estimated date for the production of the 2C resources';

comment on column GAS_RESERVES_AND_RESOURCES.EXPECTEDBARRIERS2C is
'A list of any barriers to the commercial recovery of the 2C resources';

comment on column GAS_RESERVES_AND_RESOURCES.RESOURCESESTIMATEMETHOD is
'The resources assessment method used to prepare the reserves and resources estimates';

comment on column GAS_RESERVES_AND_RESOURCES.CONVERSIONFACTORQTYTCFTOPJ is
'The conversion factor used to convert quantities measured in trillions of cubic feet to PJ';

comment on column GAS_RESERVES_AND_RESOURCES.ECONOMICASSUMPTION is
'The key economic assumptions in the forecast case used to prepare the reserves and resources estimates and the source of the assumptions';

comment on column GAS_RESERVES_AND_RESOURCES.UPDATEREASON is
'The reason for the update';

comment on column GAS_RESERVES_AND_RESOURCES.PREPAREDBY is
'The name of the person who prepared the estimates';

comment on column GAS_RESERVES_AND_RESOURCES.INDEPENDENTESTIMATE is
'Whether the qualified gas industry professional who prepared, or supervised the preparation of, the reserves and resources estimates is independent of the BB reporting entity';

comment on column GAS_RESERVES_AND_RESOURCES.AUTHORISEDDATE is
'Date and time this operating state was authorised';

comment on column GAS_RESERVES_AND_RESOURCES.LASTCHANGED is
'Date and time record was last modified';

alter table GAS_RESERVES_AND_RESOURCES
   add constraint GAS_RESERVES_AND_RESOURCES_PK primary key (FIELDINTERESTID, EFFECTIVEDATE, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_SEC_CAP_ORDERS                                    */
/*==============================================================*/
create table GAS_SEC_CAP_ORDERS (
   ORDERID              VARCHAR(40)          not null,
   SUBMITTINGCOMPANYID  NUMERIC(10,0)        null,
   VERSIONDATETIME      TIMESTAMP(0)         null,
   FACILITYID           NUMERIC(10,0)        null,
   BUYSELL              VARCHAR(5)           null,
   AVAILABLEQUANTITY    NUMERIC(18,3)        null,
   PRICE                NUMERIC(18,2)        null,
   FROMGASDATE          TIMESTAMP(0)         null,
   TOGASDATE            TIMESTAMP(0)         null,
   RECEIPTLOCATION      NUMERIC(10,0)        null,
   DELIVERYLOCATION     NUMERIC(10,0)        null,
   CONTACTDETAILS       VARCHAR(1000)        null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_SEC_CAP_ORDERS is
'Information related to the bids and offers of secondary pipeline capacity, which is derived from a secondary pipeline capacity trading platform';

comment on column GAS_SEC_CAP_ORDERS.ORDERID is
'Unique identifier for the order';

comment on column GAS_SEC_CAP_ORDERS.SUBMITTINGCOMPANYID is
'Name of the company submitting the data to AEMO';

comment on column GAS_SEC_CAP_ORDERS.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_SEC_CAP_ORDERS.FACILITYID is
'Unique facility Identifier';

comment on column GAS_SEC_CAP_ORDERS.BUYSELL is
'Indicates whether the shipper is either looking to buy or sell spare capacity';

comment on column GAS_SEC_CAP_ORDERS.AVAILABLEQUANTITY is
'Quantity of actual flow for the gas date';

comment on column GAS_SEC_CAP_ORDERS.PRICE is
'Spare capacity price expressed in $ per GJ';

comment on column GAS_SEC_CAP_ORDERS.FROMGASDATE is
'Date spare capacity comes into effect';

comment on column GAS_SEC_CAP_ORDERS.TOGASDATE is
'Date spare capacity is effective to';

comment on column GAS_SEC_CAP_ORDERS.RECEIPTLOCATION is
'Connection Point ID that best represents the receipt location';

comment on column GAS_SEC_CAP_ORDERS.DELIVERYLOCATION is
'Connection Point ID that best represents the Delivery Location';

comment on column GAS_SEC_CAP_ORDERS.CONTACTDETAILS is
'Name and number of the shippers contact person';

comment on column GAS_SEC_CAP_ORDERS.LASTCHANGED is
'Last changed date for the record';

alter table GAS_SEC_CAP_ORDERS
   add constraint GAS_SEC_CAP_ORDERS_PK primary key (ORDERID);

/*==============================================================*/
/* Index: GAS_SEC_CAP_ORDERS_IDX1                               */
/*==============================================================*/
create  index GAS_SEC_CAP_ORDERS_IDX1 on GAS_SEC_CAP_ORDERS (
SUBMITTINGCOMPANYID,
VERSIONDATETIME
);

/*==============================================================*/
/* Index: GAS_SEC_CAP_ORDERS_IDX2                               */
/*==============================================================*/
create  index GAS_SEC_CAP_ORDERS_IDX2 on GAS_SEC_CAP_ORDERS (
FROMGASDATE,
TOGASDATE
);

/*==============================================================*/
/* Table: GAS_SEC_CAP_ORDERS_TRK                                */
/*==============================================================*/
create table GAS_SEC_CAP_ORDERS_TRK (
   SUBMITTINGCOMPANYID  NUMERIC(10,0)        not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_SEC_CAP_ORDERS_TRK is
'Bids and offer tracking information of secondary pipeline capacity';

comment on column GAS_SEC_CAP_ORDERS_TRK.SUBMITTINGCOMPANYID is
'Name of the company submitting the data to AEMO';

comment on column GAS_SEC_CAP_ORDERS_TRK.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_SEC_CAP_ORDERS_TRK.LASTCHANGED is
'Last changed date for the record';

alter table GAS_SEC_CAP_ORDERS_TRK
   add constraint GAS_SEC_CAP_ORDERS_TRK_PK primary key (SUBMITTINGCOMPANYID, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_SEC_CAP_TRADES                                    */
/*==============================================================*/
create table GAS_SEC_CAP_TRADES (
   TRADEID              VARCHAR(40)          not null,
   SUBMITTINGCOMPANYID  NUMERIC(10,0)        null,
   VERSIONDATETIME      TIMESTAMP(0)         null,
   FACILITYID           NUMERIC(10,0)        null,
   GASDATE              TIMESTAMP(0)         null,
   RECEIPTLOCATION      NUMERIC(10,0)        null,
   DELIVERYLOCATION     NUMERIC(10,0)        null,
   NAMEPLATECAPACITY    NUMERIC(18,3)        null,
   DAILYNOMINATIONS     NUMERIC(18,3)        null,
   DAILYUTILISATIONPERCENT NUMERIC(18,2)        null,
   AVAILABLECAPACITY    NUMERIC(18,3)        null,
   CAPACITYONOFFER      NUMERIC(18,3)        null,
   DAILYCAPACITYTRADED  NUMERIC(18,3)        null,
   DAILYCAPACITY        NUMERIC(18,3)        null,
   CONTRACTEDCAPACITY   NUMERIC(18,3)        null,
   AVERAGEANNUALCAPACITYTRADED NUMERIC(18,3)        null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_SEC_CAP_TRADES is
'Information related to summary trading data, which is derived from a secondary pipeline capacity trading platform.';

comment on column GAS_SEC_CAP_TRADES.TRADEID is
'Unique identifier for the trade';

comment on column GAS_SEC_CAP_TRADES.SUBMITTINGCOMPANYID is
'Name of the company submitting the data to AEMO';

comment on column GAS_SEC_CAP_TRADES.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_SEC_CAP_TRADES.FACILITYID is
'Unique facility Identifier';

comment on column GAS_SEC_CAP_TRADES.GASDATE is
'Gas date';

comment on column GAS_SEC_CAP_TRADES.RECEIPTLOCATION is
'Connection Point ID that best represents the Receipt location';

comment on column GAS_SEC_CAP_TRADES.DELIVERYLOCATION is
'Connection Point ID that best represents the Delivery Location';

comment on column GAS_SEC_CAP_TRADES.NAMEPLATECAPACITY is
'Official pipeline capacity expressed in GJ';

comment on column GAS_SEC_CAP_TRADES.DAILYNOMINATIONS is
'Daily aggregate quantity of gas (expressed in GJ) nominated for delivery from the pipeline';

comment on column GAS_SEC_CAP_TRADES.DAILYUTILISATIONPERCENT is
'Percentage of the pipeline capacity that is utilised per day';

comment on column GAS_SEC_CAP_TRADES.AVAILABLECAPACITY is
'Operational capacity minus nominations each day. Expressed in GJ';

comment on column GAS_SEC_CAP_TRADES.CAPACITYONOFFER is
'Sum of total capacity offered for sale expressed in GJ';

comment on column GAS_SEC_CAP_TRADES.DAILYCAPACITYTRADED is
'Sum of total daily sold capacity expressed in GJ';

comment on column GAS_SEC_CAP_TRADES.DAILYCAPACITY is
'Operational capacity';

comment on column GAS_SEC_CAP_TRADES.CONTRACTEDCAPACITY is
'Firm-forward, contracted pipeline capacity expressed in GJ.';

comment on column GAS_SEC_CAP_TRADES.AVERAGEANNUALCAPACITYTRADED is
'Sum of total annual sold capacity divided by number of days, year to date. Expressed in GJ.';

comment on column GAS_SEC_CAP_TRADES.LASTCHANGED is
'Last changed date for the record';

alter table GAS_SEC_CAP_TRADES
   add constraint GAS_SEC_CAP_TRADES_PK primary key (TRADEID);

/*==============================================================*/
/* Index: GAS_SEC_CAP_TRADES_IDX1                               */
/*==============================================================*/
create  index GAS_SEC_CAP_TRADES_IDX1 on GAS_SEC_CAP_TRADES (
SUBMITTINGCOMPANYID,
VERSIONDATETIME
);

/*==============================================================*/
/* Index: GAS_SEC_CAP_TRADES_IDX2                               */
/*==============================================================*/
create  index GAS_SEC_CAP_TRADES_IDX2 on GAS_SEC_CAP_TRADES (
GASDATE
);

/*==============================================================*/
/* Table: GAS_SEC_CAP_TRADES_TRK                                */
/*==============================================================*/
create table GAS_SEC_CAP_TRADES_TRK (
   SUBMITTINGCOMPANYID  NUMERIC(10,0)        not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_SEC_CAP_TRADES_TRK is
'Information related to summary trading data, which is derived from a secondary pipeline capacity trading platform.';

comment on column GAS_SEC_CAP_TRADES_TRK.SUBMITTINGCOMPANYID is
'Name of the company submitting the data to AEMO';

comment on column GAS_SEC_CAP_TRADES_TRK.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_SEC_CAP_TRADES_TRK.LASTCHANGED is
'Last changed date for the record';

alter table GAS_SEC_CAP_TRADES_TRK
   add constraint GAS_SEC_CAP_TRADES_TRK_PK primary key (SUBMITTINGCOMPANYID, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_SHIPPER_LIST                                      */
/*==============================================================*/
create table GAS_SHIPPER_LIST (
   FACILITYID           NUMERIC(20,0)        not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   SHIPPERNAME          VARCHAR(200)         not null
);

comment on table GAS_SHIPPER_LIST is
'A list of shippers who have contracted primary Storage, Compression or Pipeline capacity';

comment on column GAS_SHIPPER_LIST.FACILITYID is
'A unique AEMO defined Facility Identifier';

comment on column GAS_SHIPPER_LIST.EFFECTIVEDATE is
'Gas date that corresponding record takes effect';

comment on column GAS_SHIPPER_LIST.VERSIONDATETIME is
'Time a successful submission is accepted by AEMO systems';

comment on column GAS_SHIPPER_LIST.SHIPPERNAME is
'The name of the shipper who holds the capacity';

alter table GAS_SHIPPER_LIST
   add constraint GAS_SHIPPER_LIST_PK primary key (FACILITYID, EFFECTIVEDATE, VERSIONDATETIME, SHIPPERNAME);

/*==============================================================*/
/* Table: GAS_SHORT_TERM_CAP_OUTLOOK                            */
/*==============================================================*/
create table GAS_SHORT_TERM_CAP_OUTLOOK (
   GASDATE              TIMESTAMP(0)         not null,
   FACILITYID           NUMERIC(10,0)        not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   CAPACITYTYPE         VARCHAR(20)          not null,
   FLOWDIRECTION        VARCHAR(20)          not null,
   RECEIPTLOCATION      NUMERIC(10,0)        not null,
   DELIVERYLOCATION     NUMERIC(10,0)        not null,
   OUTLOOKQUANTITY      NUMERIC(18,3)        null,
   CAPACITYDESCRIPTION  VARCHAR(1000)        null,
   DESCRIPTION          VARCHAR(1000)        null
);

comment on table GAS_SHORT_TERM_CAP_OUTLOOK is
'Provides on each gas day D, the BB facility operators good faith estimate of the daily capacity of the BB Facility for gas days D+1 to D+7.';

comment on column GAS_SHORT_TERM_CAP_OUTLOOK.GASDATE is
'Date of gas day';

comment on column GAS_SHORT_TERM_CAP_OUTLOOK.FACILITYID is
'Unique Facility Identifier';

comment on column GAS_SHORT_TERM_CAP_OUTLOOK.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_SHORT_TERM_CAP_OUTLOOK.CAPACITYTYPE is
'Capacity Type - Must be either STORAGE or MDQ';

comment on column GAS_SHORT_TERM_CAP_OUTLOOK.FLOWDIRECTION is
'Direction of flow for the submission record. Must be one of RECEIPT or DELIVERY';

comment on column GAS_SHORT_TERM_CAP_OUTLOOK.RECEIPTLOCATION is
'Connection Point ID that best represents the receipt location';

comment on column GAS_SHORT_TERM_CAP_OUTLOOK.DELIVERYLOCATION is
'Connection Point ID that best represents the Delivery Location';

comment on column GAS_SHORT_TERM_CAP_OUTLOOK.OUTLOOKQUANTITY is
'Quantity of available capacity during the defined period';

comment on column GAS_SHORT_TERM_CAP_OUTLOOK.CAPACITYDESCRIPTION is
'Describes the meaning of the capacity Number Provided';

comment on column GAS_SHORT_TERM_CAP_OUTLOOK.DESCRIPTION is
'Description for reasons or comments directly related to the quantity or change in Outlook quantity in relation to the Facility';

alter table GAS_SHORT_TERM_CAP_OUTLOOK
   add constraint GAS_SHORT_TERM_CAP_OUTLOOK_PK primary key (GASDATE, FACILITYID, VERSIONDATETIME, CAPACITYTYPE, FLOWDIRECTION, RECEIPTLOCATION, DELIVERYLOCATION);

/*==============================================================*/
/* Table: GAS_SHORT_TERM_SWAP_TRANS                             */
/*==============================================================*/
create table GAS_SHORT_TERM_SWAP_TRANS (
   PERIODSTARTDATE      TIMESTAMP(0)         not null,
   PERIODENDDATE        TIMESTAMP(0)         null,
   STATE                VARCHAR(5)           not null,
   QUANTITY             NUMERIC(18,8)        null,
   VOLUMEWEIGHTEDPRICE  NUMERIC(18,8)        null,
   TRANSACTIONTYPE      VARCHAR(255)         null,
   SUPPLYPERIODSTART    TIMESTAMP(0)         null,
   SUPPLYPERIODEND      TIMESTAMP(0)         null
);

comment on table GAS_SHORT_TERM_SWAP_TRANS is
'These reports display information regarding short-term gas swap transactions';

comment on column GAS_SHORT_TERM_SWAP_TRANS.PERIODSTARTDATE is
'The time period start date';

comment on column GAS_SHORT_TERM_SWAP_TRANS.PERIODENDDATE is
'The time period end date';

comment on column GAS_SHORT_TERM_SWAP_TRANS.STATE is
'The state where the transaction occurred';

comment on column GAS_SHORT_TERM_SWAP_TRANS.QUANTITY is
'Total volume of the transactions where trade date is in the reporting period for the given state';

comment on column GAS_SHORT_TERM_SWAP_TRANS.VOLUMEWEIGHTEDPRICE is
'Volume weighted price of transactions where trade date is in the reporting period for the given State';

comment on column GAS_SHORT_TERM_SWAP_TRANS.TRANSACTIONTYPE is
'Whether the swap is a location swap, time swap or both location and time swap';

comment on column GAS_SHORT_TERM_SWAP_TRANS.SUPPLYPERIODSTART is
'The earliest start date of all transactions in the reporting period for the given state';

comment on column GAS_SHORT_TERM_SWAP_TRANS.SUPPLYPERIODEND is
'The latest end date of all transactions in the reporting period for the given state';

alter table GAS_SHORT_TERM_SWAP_TRANS
   add constraint GAS_SHORT_TERM_SWAP_TRANS_PK primary key (PERIODSTARTDATE, STATE);

/*==============================================================*/
/* Table: GAS_SHORT_TERM_TRANS                                  */
/*==============================================================*/
create table GAS_SHORT_TERM_TRANS (
   PERIODSTARTDATE      TIMESTAMP(0)         not null,
   PERIODENDDATE        TIMESTAMP(0)         null,
   STATE                VARCHAR(5)           not null,
   QUANTITY             NUMERIC(18,8)        null,
   VOLUMEWEIGHTEDPRICE  NUMERIC(18,8)        null,
   TRANSACTIONTYPE      VARCHAR(255)         null,
   SUPPLYPERIODSTART    TIMESTAMP(0)         null,
   SUPPLYPERIODEND      TIMESTAMP(0)         null
);

comment on table GAS_SHORT_TERM_TRANS is
'These reports display information regarding short-term gas transactions';

comment on column GAS_SHORT_TERM_TRANS.PERIODSTARTDATE is
'The time period start date';

comment on column GAS_SHORT_TERM_TRANS.PERIODENDDATE is
'The time period end date';

comment on column GAS_SHORT_TERM_TRANS.STATE is
'The state where the transaction occurred';

comment on column GAS_SHORT_TERM_TRANS.QUANTITY is
'Total volume of the transactions where trade date is in the reporting period for the given state';

comment on column GAS_SHORT_TERM_TRANS.VOLUMEWEIGHTEDPRICE is
'Volume weighted price of transactions where trade date is in the reporting period for the given State';

comment on column GAS_SHORT_TERM_TRANS.TRANSACTIONTYPE is
'Transaction Type is Supply for these short-term transactions reports';

comment on column GAS_SHORT_TERM_TRANS.SUPPLYPERIODSTART is
'The earliest start date of all transactions in the reporting period for the given state';

comment on column GAS_SHORT_TERM_TRANS.SUPPLYPERIODEND is
'The latest end date of all transactions in the reporting period for the given state';

alter table GAS_SHORT_TERM_TRANS
   add constraint GAS_SHORT_TERM_TRANS_PK primary key (PERIODSTARTDATE, STATE);

/*==============================================================*/
/* Table: GAS_STATE                                             */
/*==============================================================*/
create table GAS_STATE (
   STATEID              NUMERIC(10,0)        not null,
   STATE                VARCHAR(5)           null,
   STATENAME            VARCHAR(50)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_STATE is
'List of states.';

comment on column GAS_STATE.STATEID is
'Unique state identifier';

comment on column GAS_STATE.STATE is
'State details';

comment on column GAS_STATE.STATENAME is
'Name of the state';

comment on column GAS_STATE.LASTCHANGED is
'Last changed date for the record';

alter table GAS_STATE
   add constraint GAS_STATE_PK primary key (STATEID);

/*==============================================================*/
/* Table: GAS_STORAGE_CAPACITY_TRANS                            */
/*==============================================================*/
create table GAS_STORAGE_CAPACITY_TRANS (
   TRADEID              NUMERIC(20,0)        not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   TRADEDATE            TIMESTAMP(0)         null,
   FROMGASDATE          TIMESTAMP(0)         null,
   TOGASDATE            TIMESTAMP(0)         null,
   FACILITYID           NUMERIC(20,0)        null,
   PRIORITY             VARCHAR(255)         null,
   MAXIMUMSTORAGEQUANTITY NUMERIC(18,3)        null,
   INJECTIONCAPACITY    NUMERIC(18,3)        null,
   WITHDRAWALCAPACITY   NUMERIC(18,3)        null,
   PRICE                NUMERIC(18,3)        null,
   PRICESTRUCTURE       VARCHAR(255)         null,
   PRICEESCALATIONMECHANISM VARCHAR(255)         null,
   CANCELLED            NUMERIC(1,0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GAS_STORAGE_CAPACITY_TRANS is
'Gas storage capacity transactions';

comment on column GAS_STORAGE_CAPACITY_TRANS.TRADEID is
'A unique AEMO defined Transaction Identifier';

comment on column GAS_STORAGE_CAPACITY_TRANS.VERSIONDATETIME is
'Time a successful submission is accepted by AEMO systems';

comment on column GAS_STORAGE_CAPACITY_TRANS.TRADEDATE is
'The date the transaction was entered into';

comment on column GAS_STORAGE_CAPACITY_TRANS.FROMGASDATE is
'The start date of the transaction';

comment on column GAS_STORAGE_CAPACITY_TRANS.TOGASDATE is
'The end date of the transaction';

comment on column GAS_STORAGE_CAPACITY_TRANS.FACILITYID is
'The gas storage facility ID for the facility by means of which the service is provided';

comment on column GAS_STORAGE_CAPACITY_TRANS.PRIORITY is
'The priority given to the service to which the transaction relates';

comment on column GAS_STORAGE_CAPACITY_TRANS.MAXIMUMSTORAGEQUANTITY is
'The storage capacity the subject of the transaction (in GJ)';

comment on column GAS_STORAGE_CAPACITY_TRANS.INJECTIONCAPACITY is
'The injection capacity (in GJ/day)';

comment on column GAS_STORAGE_CAPACITY_TRANS.WITHDRAWALCAPACITY is
'The withdrawal capacity (in GJ/day)';

comment on column GAS_STORAGE_CAPACITY_TRANS.PRICE is
'The transaction price (in $/GJ/day or where relevant, in $/GJ)';

comment on column GAS_STORAGE_CAPACITY_TRANS.PRICESTRUCTURE is
'The price structure applicable to the transaction';

comment on column GAS_STORAGE_CAPACITY_TRANS.PRICEESCALATIONMECHANISM is
'Any price escalation mechanism applicable to the transaction';

comment on column GAS_STORAGE_CAPACITY_TRANS.CANCELLED is
'Whether the record has been cancelled';

comment on column GAS_STORAGE_CAPACITY_TRANS.LASTCHANGED is
'The date the record was last updated';

alter table GAS_STORAGE_CAPACITY_TRANS
   add constraint GAS_STORAGE_CAPACITY_TRANS_PK primary key (TRADEID, VERSIONDATETIME);

/*==============================================================*/
/* Table: GAS_UNCONTRACTED_CAP_OUTLOOK                          */
/*==============================================================*/
create table GAS_UNCONTRACTED_CAP_OUTLOOK (
   OUTLOOKYEAR          NUMERIC(4,0)         not null,
   OUTLOOKMONTH         NUMERIC(4,0)         not null,
   FACILITYID           NUMERIC(10,0)        not null,
   VERSIONDATETIME      TIMESTAMP(0)         not null,
   CAPACITYTYPE         VARCHAR(20)          not null,
   FLOWDIRECTION        VARCHAR(20)          not null,
   RECEIPTLOCATION      NUMERIC(10,0)        not null,
   DELIVERYLOCATION     NUMERIC(10,0)        not null,
   OUTLOOKQUANTITY      NUMERIC(18,3)        null,
   CAPACITYDESCRIPTION  VARCHAR(1000)        null,
   DESCRIPTION          VARCHAR(1000)        null
);

comment on table GAS_UNCONTRACTED_CAP_OUTLOOK is
'Provide the uncontracted capacity of the facility over the next 12 months.';

comment on column GAS_UNCONTRACTED_CAP_OUTLOOK.OUTLOOKYEAR is
'The outlook year';

comment on column GAS_UNCONTRACTED_CAP_OUTLOOK.OUTLOOKMONTH is
'The outlook month.';

comment on column GAS_UNCONTRACTED_CAP_OUTLOOK.FACILITYID is
'Unique Facility Identifier';

comment on column GAS_UNCONTRACTED_CAP_OUTLOOK.VERSIONDATETIME is
'Version timestamp of the record';

comment on column GAS_UNCONTRACTED_CAP_OUTLOOK.CAPACITYTYPE is
'Capacity Type - Must be either STORAGE or MDQ';

comment on column GAS_UNCONTRACTED_CAP_OUTLOOK.FLOWDIRECTION is
'Direction of flow for the submission record. Must be one of RECEIPT or DELIVERY';

comment on column GAS_UNCONTRACTED_CAP_OUTLOOK.RECEIPTLOCATION is
'Connection Point ID that best represents the receipt location';

comment on column GAS_UNCONTRACTED_CAP_OUTLOOK.DELIVERYLOCATION is
'Connection Point ID that best represents the Delivery Location';

comment on column GAS_UNCONTRACTED_CAP_OUTLOOK.OUTLOOKQUANTITY is
'Quantity of available capacity during the defined period';

comment on column GAS_UNCONTRACTED_CAP_OUTLOOK.CAPACITYDESCRIPTION is
'Describes the meaning of the capacity Number Provided';

comment on column GAS_UNCONTRACTED_CAP_OUTLOOK.DESCRIPTION is
'Description for reasons or comments directly related to the quantity or change in Outlook quantity in relation to the Facility';

alter table GAS_UNCONTRACTED_CAP_OUTLOOK
   add constraint GAS_UNCON_CAP_OUTLOOK_PK primary key (OUTLOOKYEAR, OUTLOOKMONTH, FACILITYID, VERSIONDATETIME, CAPACITYTYPE, FLOWDIRECTION, RECEIPTLOCATION, DELIVERYLOCATION);

/*==============================================================*/
/* Table: GSH_AUCTION_BID                                       */
/*==============================================================*/
create table GSH_AUCTION_BID (
   GAS_DATE             TIMESTAMP(0)         not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   SUBMISSION_TIME      TIMESTAMP(0)         not null,
   BID_ID               NUMERIC(8,0)         not null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_AUCTION_BID is
'Information for a single bid contained within a bid profile';

comment on column GSH_AUCTION_BID.GAS_DATE is
'Single Gas Date of Submission';

comment on column GSH_AUCTION_BID.PARTICIPANT_CODE is
'Unique Participant Code';

comment on column GSH_AUCTION_BID.SUBMISSION_TIME is
'Date and Time of the submission';

comment on column GSH_AUCTION_BID.BID_ID is
'Unique Bid ID within Bid Profile';

comment on column GSH_AUCTION_BID.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_AUCTION_BID
   add constraint GSH_AUCTION_BID_PK primary key (GAS_DATE, PARTICIPANT_CODE, SUBMISSION_TIME, BID_ID);

/*==============================================================*/
/* Table: GSH_AUCTION_BID_PRODUCT                               */
/*==============================================================*/
create table GSH_AUCTION_BID_PRODUCT (
   GAS_DATE             TIMESTAMP(0)         not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   SUBMISSION_TIME      TIMESTAMP(0)         not null,
   BID_ID               NUMERIC(8,0)         not null,
   AUCTION_PRODUCT_TYPE VARCHAR(20)          not null,
   RECEIPT_POINT_ID     NUMERIC(8,0)         not null,
   RECEIPT_POINT_NAME   VARCHAR(255)         null,
   RP_DWGM_REFERENCE    VARCHAR(255)         null,
   DELIVERY_POINT_ID    NUMERIC(8,0)         not null,
   DELIVERY_POINT_NAME  VARCHAR(255)         null,
   DP_DWGM_REFERENCE    VARCHAR(255)         null,
   SERVICE_REFERENCE    VARCHAR(255)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_AUCTION_BID_PRODUCT is
'Information for a single auction product contained within a bid';

comment on column GSH_AUCTION_BID_PRODUCT.GAS_DATE is
'Single Gas Date of Submission';

comment on column GSH_AUCTION_BID_PRODUCT.PARTICIPANT_CODE is
'Unique Participant Code';

comment on column GSH_AUCTION_BID_PRODUCT.SUBMISSION_TIME is
'Date and Time of the submission';

comment on column GSH_AUCTION_BID_PRODUCT.BID_ID is
'Unique Bid ID within Bid Profile';

comment on column GSH_AUCTION_BID_PRODUCT.AUCTION_PRODUCT_TYPE is
'The product type, for example Forward Haul, Backhaul, Compression';

comment on column GSH_AUCTION_BID_PRODUCT.RECEIPT_POINT_ID is
'The unique identifier of the Receipt Point as defined in the Transportation Service Point Register. This is the point where gas is injected into the pipeline';

comment on column GSH_AUCTION_BID_PRODUCT.RECEIPT_POINT_NAME is
'The descriptive name of the Receipt Point as defined in the Transportation Service Point Register.';

comment on column GSH_AUCTION_BID_PRODUCT.RP_DWGM_REFERENCE is
'The DWGM accreditation reference associated with the Receipt Point.';

comment on column GSH_AUCTION_BID_PRODUCT.DELIVERY_POINT_ID is
'The unique identifier of the Delivery Point as defined in the Transportation Service Point Register. This is the point where gas is withdrawn from the pipeline. Mandatory for service points registered under Part 24 of the rules.';

comment on column GSH_AUCTION_BID_PRODUCT.DELIVERY_POINT_NAME is
'The descriptive name of the Delivery Point as defined in the Transportation Service Point Register.';

comment on column GSH_AUCTION_BID_PRODUCT.DP_DWGM_REFERENCE is
'The DWGM accreditation reference associated with the Delivery Point.';

comment on column GSH_AUCTION_BID_PRODUCT.SERVICE_REFERENCE is
'The service reference issued by the facility operator.';

comment on column GSH_AUCTION_BID_PRODUCT.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_AUCTION_BID_PRODUCT
   add constraint GSH_AUCTION_BID_PRODUCT_PK primary key (GAS_DATE, PARTICIPANT_CODE, SUBMISSION_TIME, BID_ID, AUCTION_PRODUCT_TYPE, DELIVERY_POINT_ID, RECEIPT_POINT_ID);

/*==============================================================*/
/* Table: GSH_AUCTION_BID_PROFILE                               */
/*==============================================================*/
create table GSH_AUCTION_BID_PROFILE (
   GAS_DATE             TIMESTAMP(0)         not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   SUBMISSION_TIME      TIMESTAMP(0)         not null,
   LAST_UPDATED_BY      VARCHAR(20)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_AUCTION_BID_PROFILE is
'Top-level table for a profile of auction bids on a given gas date';

comment on column GSH_AUCTION_BID_PROFILE.GAS_DATE is
'Single Gas Date of Submission';

comment on column GSH_AUCTION_BID_PROFILE.PARTICIPANT_CODE is
'Unique Participant Code';

comment on column GSH_AUCTION_BID_PROFILE.SUBMISSION_TIME is
'Date and Time of the submission';

comment on column GSH_AUCTION_BID_PROFILE.LAST_UPDATED_BY is
'The user name who submitted or updated the bid.';

comment on column GSH_AUCTION_BID_PROFILE.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_AUCTION_BID_PROFILE
   add constraint GSH_AUCTION_BID_PROFILE_PK primary key (GAS_DATE, PARTICIPANT_CODE, SUBMISSION_TIME);

/*==============================================================*/
/* Table: GSH_AUCTION_BID_STEPS                                 */
/*==============================================================*/
create table GSH_AUCTION_BID_STEPS (
   GAS_DATE             TIMESTAMP(0)         not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   SUBMISSION_TIME      TIMESTAMP(0)         not null,
   BID_ID               NUMERIC(8,0)         not null,
   STEP_NUMBER          NUMERIC(6,0)         not null,
   STEP_QUANTITY        NUMERIC(18,8)        null,
   STEP_PRICE           NUMERIC(18,8)        null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_AUCTION_BID_STEPS is
'Information for a single step contained within a bid.';

comment on column GSH_AUCTION_BID_STEPS.GAS_DATE is
'Single Gas Date of Submission';

comment on column GSH_AUCTION_BID_STEPS.PARTICIPANT_CODE is
'Unique Participant Code';

comment on column GSH_AUCTION_BID_STEPS.SUBMISSION_TIME is
'Date and Time of the submission';

comment on column GSH_AUCTION_BID_STEPS.BID_ID is
'Unique Bid ID within Bid Profile';

comment on column GSH_AUCTION_BID_STEPS.STEP_NUMBER is
'Unique Step ID within the Bid';

comment on column GSH_AUCTION_BID_STEPS.STEP_QUANTITY is
'Quantity for this Bid Step';

comment on column GSH_AUCTION_BID_STEPS.STEP_PRICE is
'Price for this Bid Step';

comment on column GSH_AUCTION_BID_STEPS.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_AUCTION_BID_STEPS
   add constraint GSH_AUCTION_BID_STEPS_PK primary key (GAS_DATE, PARTICIPANT_CODE, SUBMISSION_TIME, BID_ID, STEP_NUMBER);

/*==============================================================*/
/* Table: GSH_AUCTION_BID_SUBMISSION                            */
/*==============================================================*/
create table GSH_AUCTION_BID_SUBMISSION (
   SUBMISSION_ID        NUMERIC(8,0)         not null,
   PARTICIPANT_CODE     VARCHAR(20)          null,
   SUBMISSION_TIME      TIMESTAMP(0)         null,
   FROM_GAS_DATE        TIMESTAMP(0)         null,
   TO_GAS_DATE          TIMESTAMP(0)         null,
   LAST_UPDATED         TIMESTAMP(0)         null,
   LAST_UPDATED_BY      VARCHAR(100)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_AUCTION_BID_SUBMISSION is
'Provides a confirmation to trading participants that their auction bid has been successfully submitted.';

comment on column GSH_AUCTION_BID_SUBMISSION.SUBMISSION_ID is
'Unique ID for a bid entry';

comment on column GSH_AUCTION_BID_SUBMISSION.PARTICIPANT_CODE is
'Unique Participant Code';

comment on column GSH_AUCTION_BID_SUBMISSION.SUBMISSION_TIME is
'Date and Time of the submission';

comment on column GSH_AUCTION_BID_SUBMISSION.FROM_GAS_DATE is
'The first gas day of the auction bid. Disregard the time component as this is not applicable.';

comment on column GSH_AUCTION_BID_SUBMISSION.TO_GAS_DATE is
'The last gas day of the auction bid. Disregard the time component as this is not applicable.';

comment on column GSH_AUCTION_BID_SUBMISSION.LAST_UPDATED is
'The date and time the bid was submitted or updated.';

comment on column GSH_AUCTION_BID_SUBMISSION.LAST_UPDATED_BY is
'The user name who submitted or updated the bid.';

comment on column GSH_AUCTION_BID_SUBMISSION.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_AUCTION_BID_SUBMISSION
   add constraint GSH_AUCTION_BID_SUBMISSION_PK primary key (SUBMISSION_ID);

/*==============================================================*/
/* Table: GSH_AUCTION_BID_TRK                                   */
/*==============================================================*/
create table GSH_AUCTION_BID_TRK (
   AUCTION_DATE         TIMESTAMP(0)         not null,
   VERSION_DATETIME     TIMESTAMP(0)         not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   PARTICIPANT_ID       VARCHAR(20)          not null,
   PARTICIPANT_CODE     VARCHAR(20)          null,
   BID_ID               NUMERIC(8,0)         not null,
   SUBMISSION_TIME      TIMESTAMP(0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_AUCTION_BID_TRK is
'List of bids considered for an auction';

comment on column GSH_AUCTION_BID_TRK.AUCTION_DATE is
'The date of the auction.';

comment on column GSH_AUCTION_BID_TRK.VERSION_DATETIME is
'The date and time that the auction was run.';

comment on column GSH_AUCTION_BID_TRK.GAS_DATE is
'The gas date for which capacity is assigned to the shipper';

comment on column GSH_AUCTION_BID_TRK.PARTICIPANT_ID is
'Unique GSH Participant ID';

comment on column GSH_AUCTION_BID_TRK.PARTICIPANT_CODE is
'The unique identifier of the participant';

comment on column GSH_AUCTION_BID_TRK.BID_ID is
'Unique Bid ID within Bid Profile';

comment on column GSH_AUCTION_BID_TRK.SUBMISSION_TIME is
'Date and Time of the submission';

comment on column GSH_AUCTION_BID_TRK.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_AUCTION_BID_TRK
   add constraint GSH_AUCTION_BID_TRK_PK primary key (AUCTION_DATE, GAS_DATE, VERSION_DATETIME, PARTICIPANT_ID, BID_ID);

/*==============================================================*/
/* Table: GSH_AUCTION_CURTAIL_NOTICE                            */
/*==============================================================*/
create table GSH_AUCTION_CURTAIL_NOTICE (
   NOTICE_ID            NUMERIC(8,0)         not null,
   FACILITY_ID          NUMERIC(8,0)         not null,
   CURTAILED            VARCHAR(3)           null,
   MATERIAL             VARCHAR(3)           null,
   DESCRIPTION          VARCHAR(255)         null,
   EFFECTIVE_DATE_TIME  TIMESTAMP(0)         not null,
   FACILITY_NAME        VARCHAR(100)         null,
   AUCTION_SERVICE      VARCHAR(20)          not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_AUCTION_CURTAIL_NOTICE is
'Contains details of curtailment of nominations against secondary firm rights (CTP trades) for service points (receipt and delivery points).';

comment on column GSH_AUCTION_CURTAIL_NOTICE.NOTICE_ID is
'The unique identifier of the notice.';

comment on column GSH_AUCTION_CURTAIL_NOTICE.FACILITY_ID is
'The unique facility identifier as defined in the Transportation Facility Register.';

comment on column GSH_AUCTION_CURTAIL_NOTICE.CURTAILED is
'A flag to indicate curtailment';

comment on column GSH_AUCTION_CURTAIL_NOTICE.MATERIAL is
'Flag to indicate material curtailment';

comment on column GSH_AUCTION_CURTAIL_NOTICE.DESCRIPTION is
'A description of the curtailment.';

comment on column GSH_AUCTION_CURTAIL_NOTICE.EFFECTIVE_DATE_TIME is
'The date and time the curtailment is effective from.';

comment on column GSH_AUCTION_CURTAIL_NOTICE.FACILITY_NAME is
'The descriptive name of the facility.';

comment on column GSH_AUCTION_CURTAIL_NOTICE.AUCTION_SERVICE is
'The auction service impacted by curtailment';

comment on column GSH_AUCTION_CURTAIL_NOTICE.GAS_DATE is
'The gas date for which the auction service is curtailed.';

comment on column GSH_AUCTION_CURTAIL_NOTICE.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_AUCTION_CURTAIL_NOTICE
   add constraint GSH_AUCTION_CURTAIL_NOTICE_PK primary key (NOTICE_ID, FACILITY_ID, AUCTION_SERVICE, EFFECTIVE_DATE_TIME, GAS_DATE);

/*==============================================================*/
/* Table: GSH_AUCTION_PRICE_VOLUME                              */
/*==============================================================*/
create table GSH_AUCTION_PRICE_VOLUME (
   AUCTION_DATE         TIMESTAMP(0)         not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   FACILITY_ID          NUMERIC(8,0)         not null,
   FACILITY_NAME        VARCHAR(100)         null,
   LIMIT_ID             NUMERIC(8,0)         not null,
   LIMIT_NAME           VARCHAR(100)         null,
   LIMIT_TYPE           VARCHAR(100)         null,
   CLEARED_QUANTITY     NUMERIC(12,4)        null,
   CLEARED_PRICE        NUMERIC(12,4)        null,
   HIGH_PRICE_SENSITIVITY NUMERIC(12,4)        null,
   LOW_PRICE_SENSITIVITY NUMERIC(12,4)        null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_AUCTION_PRICE_VOLUME is
'The cleared quantity, cleared price and price sensitivity for each auction product component following each auction run';

comment on column GSH_AUCTION_PRICE_VOLUME.AUCTION_DATE is
'The date of the auction run. Disregard the time component as this is not applicable.';

comment on column GSH_AUCTION_PRICE_VOLUME.GAS_DATE is
'The gas date that the auction quantities and prices are valid for. Disregard the time component as this is not applicable.';

comment on column GSH_AUCTION_PRICE_VOLUME.FACILITY_ID is
'The unique identifier of the facility.';

comment on column GSH_AUCTION_PRICE_VOLUME.FACILITY_NAME is
'The descriptive name of the facility.';

comment on column GSH_AUCTION_PRICE_VOLUME.LIMIT_ID is
'The unique identifier of the auction limit.';

comment on column GSH_AUCTION_PRICE_VOLUME.LIMIT_NAME is
'The descriptive name of the auction limit.';

comment on column GSH_AUCTION_PRICE_VOLUME.LIMIT_TYPE is
'The type of auction limit';

comment on column GSH_AUCTION_PRICE_VOLUME.CLEARED_QUANTITY is
'The capacity quantity cleared at auction (GJ/day).';

comment on column GSH_AUCTION_PRICE_VOLUME.CLEARED_PRICE is
'The cleared auction price ($/GJ).';

comment on column GSH_AUCTION_PRICE_VOLUME.HIGH_PRICE_SENSITIVITY is
'The high price sensitivity factor applied to the price.';

comment on column GSH_AUCTION_PRICE_VOLUME.LOW_PRICE_SENSITIVITY is
'The low price sensitivity factor applied to the price.';

comment on column GSH_AUCTION_PRICE_VOLUME.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_AUCTION_PRICE_VOLUME
   add constraint GSH_AUCTION_PRICE_VOLUME_PK primary key (AUCTION_DATE, GAS_DATE, FACILITY_ID, LIMIT_ID);

/*==============================================================*/
/* Table: GSH_AUCTION_QUANTITIES                                */
/*==============================================================*/
create table GSH_AUCTION_QUANTITIES (
   AUCTION_DATE         TIMESTAMP(0)         not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   FACILITY_ID          NUMERIC(8,0)         not null,
   FACILITY_NAME        VARCHAR(100)         null,
   LIMIT_ID             NUMERIC(8,0)         not null,
   LIMIT_NAME           VARCHAR(100)         null,
   LIMIT_TYPE           VARCHAR(100)         not null,
   LIMIT_QUANTITY       NUMERIC(10,0)        null,
   TRANSITIONAL_RIGHTS_QUANTITY NUMERIC(10,0)        null,
   OPERATIONAL_OVERRIDE VARCHAR(3)           null,
   VERSION_DATETIME     TIMESTAMP(0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_AUCTION_QUANTITIES is
'Auction limit quantities for each product component available for the day ahead auction';

comment on column GSH_AUCTION_QUANTITIES.AUCTION_DATE is
'The date of the auction. Disregard the time component as this is not applicable.';

comment on column GSH_AUCTION_QUANTITIES.GAS_DATE is
'The gas date that the auction quantities and transitional rights are valid for. Disregard the time component as this is not applicable.';

comment on column GSH_AUCTION_QUANTITIES.FACILITY_ID is
'The unique identifier of the facility.';

comment on column GSH_AUCTION_QUANTITIES.FACILITY_NAME is
'The descriptive name of the facility.';

comment on column GSH_AUCTION_QUANTITIES.LIMIT_ID is
'The unique identifier of the auction limit.';

comment on column GSH_AUCTION_QUANTITIES.LIMIT_NAME is
'The descriptive name of the auction limit.';

comment on column GSH_AUCTION_QUANTITIES.LIMIT_TYPE is
'The type of auction limit';

comment on column GSH_AUCTION_QUANTITIES.LIMIT_QUANTITY is
'The available auction quantity of the limit in GJ.';

comment on column GSH_AUCTION_QUANTITIES.TRANSITIONAL_RIGHTS_QUANTITY is
'The quantity of nominations against grandfathered rights in GJ, for the previous gas day.';

comment on column GSH_AUCTION_QUANTITIES.OPERATIONAL_OVERRIDE is
'Flag to indicate whether the facility operator has applied an operational override to the limit quantity, for example Yes or No.';

comment on column GSH_AUCTION_QUANTITIES.VERSION_DATETIME is
'The date and time that the auction was run.';

comment on column GSH_AUCTION_QUANTITIES.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_AUCTION_QUANTITIES
   add constraint GSH_AUCTION_QUANTITIES_PK primary key (AUCTION_DATE, GAS_DATE, FACILITY_ID, LIMIT_ID, LIMIT_TYPE);

/*==============================================================*/
/* Table: GSH_AUCTION_RESULTS                                   */
/*==============================================================*/
create table GSH_AUCTION_RESULTS (
   AUCTION_TRANSACTION_ID NUMERIC(12,0)        not null,
   AUCTION_DATE         TIMESTAMP(0)         null,
   FACILITY_ID          NUMERIC(8,0)         null,
   FACILITY_NAME        VARCHAR(100)         null,
   PARTICIPANT_ID       VARCHAR(20)          null,
   PARTICIPANT_CODE     VARCHAR(20)          null,
   PARTICIPANT_NAME     VARCHAR(80)          null,
   SERVICE_REFERENCE    VARCHAR(255)         null,
   RECEIPT_POINT_ID     NUMERIC(8,0)         null,
   RECEIPT_POINT_NAME   VARCHAR(255)         null,
   RP_DWGM_REFERENCE    VARCHAR(255)         null,
   DELIVERY_POINT_ID    NUMERIC(8,0)         null,
   DELIVERY_POINT_NAME  VARCHAR(255)         null,
   DP_DWGM_REFERENCE    VARCHAR(255)         null,
   AUCTION_QUANTITY_WON NUMERIC(8,0)         null,
   GAS_DATE             TIMESTAMP(0)         null,
   PRODUCT_PRICE        NUMERIC(18,8)        null,
   VERSION_DATETIME     TIMESTAMP(0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_AUCTION_RESULTS is
'Specifies the details of the capacity won at auction';

comment on column GSH_AUCTION_RESULTS.AUCTION_TRANSACTION_ID is
'The unique identifier of the auction transaction.';

comment on column GSH_AUCTION_RESULTS.AUCTION_DATE is
'The date of the auction. Disregard the time component as this is not applicable.';

comment on column GSH_AUCTION_RESULTS.FACILITY_ID is
'The unique facility identifier as defined in the Transportation Facility Register.';

comment on column GSH_AUCTION_RESULTS.FACILITY_NAME is
'The descriptive name of the facility as defined in the Transportation Facility Register.';

comment on column GSH_AUCTION_RESULTS.PARTICIPANT_ID is
'The unique AEMO defined GSH Participant identifier.';

comment on column GSH_AUCTION_RESULTS.PARTICIPANT_CODE is
'The unique identifier of the participant';

comment on column GSH_AUCTION_RESULTS.PARTICIPANT_NAME is
'The descriptive name of the participant.';

comment on column GSH_AUCTION_RESULTS.SERVICE_REFERENCE is
'The service reference issued by the facility operator.';

comment on column GSH_AUCTION_RESULTS.RECEIPT_POINT_ID is
'The unique identifier of the Receipt Point as defined in the Transportation Service Point Register. This is the point where gas is injected into the pipeline';

comment on column GSH_AUCTION_RESULTS.RECEIPT_POINT_NAME is
'The descriptive name of the Receipt Point as defined in the Transportation Service Point Register.';

comment on column GSH_AUCTION_RESULTS.RP_DWGM_REFERENCE is
'The DWGM accreditation reference associated with the Receipt Point.';

comment on column GSH_AUCTION_RESULTS.DELIVERY_POINT_ID is
'The unique identifier of the Delivery Point as defined in the Transportation Service Point Register. This is the point where gas is withdrawn from the pipeline. Mandatory for service points registered under Part 24 of the rules.';

comment on column GSH_AUCTION_RESULTS.DELIVERY_POINT_NAME is
'The descriptive name of the Delivery Point as defined in the Transportation Service Point Register.';

comment on column GSH_AUCTION_RESULTS.DP_DWGM_REFERENCE is
'The DWGM accreditation reference associated with the Delivery Point.';

comment on column GSH_AUCTION_RESULTS.AUCTION_QUANTITY_WON is
'The capacity quantity won at auction (GJ/day) by the participant.';

comment on column GSH_AUCTION_RESULTS.GAS_DATE is
'The gas date for which capacity is assigned to the shipper, in ISO 8601 datetime format (yyyy-mm-ddThh:mm:ss).';

comment on column GSH_AUCTION_RESULTS.PRODUCT_PRICE is
'The price of the auction product won at auction ($/GJ) by the participant.';

comment on column GSH_AUCTION_RESULTS.VERSION_DATETIME is
'The date and time that the auction was run.';

comment on column GSH_AUCTION_RESULTS.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_AUCTION_RESULTS
   add constraint GSH_AUCTION_RESULTS_PK primary key (AUCTION_TRANSACTION_ID);

/*==============================================================*/
/* Table: GSH_AUCTION_RUN                                       */
/*==============================================================*/
create table GSH_AUCTION_RUN (
   AUCTION_DATE         TIMESTAMP(0)         not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   AUCTION_RUN_DATETIME TIMESTAMP(0)         not null,
   PARTICIPANT_ID       VARCHAR(20)          not null,
   PARTICIPANT_CODE     VARCHAR(20)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_AUCTION_RUN is
'Specifies the Auction Run has completed, issued to any participants bidding in the auction (regardless of whether any auction bids are won), and facility operators who have a facility in the Auction';

comment on column GSH_AUCTION_RUN.AUCTION_DATE is
'The date of the auction.';

comment on column GSH_AUCTION_RUN.GAS_DATE is
'The gas date for which capacity is assigned to the shipper.';

comment on column GSH_AUCTION_RUN.AUCTION_RUN_DATETIME is
'The date and time that the auction was run.';

comment on column GSH_AUCTION_RUN.PARTICIPANT_ID is
'Unique GSH Participant ID';

comment on column GSH_AUCTION_RUN.PARTICIPANT_CODE is
'The unique identifier of the participant';

comment on column GSH_AUCTION_RUN.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_AUCTION_RUN
   add constraint GSH_AUCTION_RUN_PK primary key (AUCTION_DATE, GAS_DATE, AUCTION_RUN_DATETIME, PARTICIPANT_ID);

/*==============================================================*/
/* Table: GSH_BENCHMARK_PRICE                                   */
/*==============================================================*/
create table GSH_BENCHMARK_PRICE (
   GAS_DATE             TIMESTAMP(0)         not null,
   PRODUCT_LOCATION     VARCHAR(80)          not null,
   PRODUCT_TYPE         VARCHAR(80)          not null,
   BENCHMARK_PRICE      NUMERIC(18,8)        null,
   IS_FIRM              VARCHAR(1)           null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_BENCHMARK_PRICE is
'The report contains the benchmark prices for the day ahead (non-firm price) PLUS the last two years of daily benchmark prices (firm price)';

comment on column GSH_BENCHMARK_PRICE.GAS_DATE is
'The gas date that the Wallumbilla benchmark price applies to';

comment on column GSH_BENCHMARK_PRICE.PRODUCT_LOCATION is
'The product location that the benchmark price applies to. Can be one of the following: WALLUMBILLA, RBP, SWQP, QGP.';

comment on column GSH_BENCHMARK_PRICE.PRODUCT_TYPE is
'The type of the product that the Wallumbilla benchmark price is calculated for';

comment on column GSH_BENCHMARK_PRICE.BENCHMARK_PRICE is
'The calculated Wallumbilla benchmark price for the specified product type for the specified gas date';

comment on column GSH_BENCHMARK_PRICE.IS_FIRM is
'Flag to determine if the Wallumbilla benchmark price is firm or not. Can be either: Y or N';

comment on column GSH_BENCHMARK_PRICE.LASTCHANGED is
'The date and time the report was generated';

alter table GSH_BENCHMARK_PRICE
   add constraint GSH_BENCHMARK_PRICE_PK primary key (GAS_DATE, PRODUCT_LOCATION, PRODUCT_TYPE);

/*==============================================================*/
/* Table: GSH_CAPACITY_TRANSACTION                              */
/*==============================================================*/
create table GSH_CAPACITY_TRANSACTION (
   TRANSACTION_ID       VARCHAR(50)          not null,
   VERSION_DATETIME     TIMESTAMP(0)         not null,
   TRADE_DATE           TIMESTAMP(0)         null,
   FROM_GAS_DATE        TIMESTAMP(0)         null,
   TO_GAS_DATE          TIMESTAMP(0)         null,
   TRADE_TYPE           VARCHAR(30)          null,
   FACILITY_ID          NUMERIC(10,0)        null,
   FACILITY_NAME        VARCHAR(100)         null,
   TRANSPORTATION_SERVICE_TYPE VARCHAR(12)          null,
   FLOW_DIRECTION       VARCHAR(20)          null,
   RECEIPT_ZONE_ID      NUMERIC(10,0)        null,
   RECEIPT_ZONE_NAME    VARCHAR(255)         null,
   DELIVERY_ZONE_ID     NUMERIC(10,0)        null,
   DELIVERY_ZONE_NAME   VARCHAR(255)         null,
   PARK_LOAN_POINT_ID   NUMERIC(10,0)        null,
   PARK_LOAN_POINT_NAME VARCHAR(255)         null,
   STANDARD_OTSA        VARCHAR(255)         null,
   PRIORITY             VARCHAR(255)         null,
   MAX_DAILY_QUANTITY   NUMERIC(10,0)        null,
   MAX_HOURLY_QUANTITY  NUMERIC(10,0)        null,
   PRICE                NUMERIC(18,8)        null,
   PRICE_STRUCTURE      VARCHAR(255)         null,
   PRICE_ESCALATION_MECHANISM VARCHAR(255)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_CAPACITY_TRANSACTION is
'Consolidated list of all secondary capacity trades which includes GSH trades (screen traded and off-market trades) and Bilateral trades submitted into the GBB';

comment on column GSH_CAPACITY_TRANSACTION.TRANSACTION_ID is
'Unique capacity transaction id';

comment on column GSH_CAPACITY_TRANSACTION.VERSION_DATETIME is
'Updated version within transaction id';

comment on column GSH_CAPACITY_TRANSACTION.TRADE_DATE is
'The date the trade was made. Disregard the time component as this is not applicable.';

comment on column GSH_CAPACITY_TRANSACTION.FROM_GAS_DATE is
'The effective start date of the trade. Disregard the time component as this is not applicable.';

comment on column GSH_CAPACITY_TRANSACTION.TO_GAS_DATE is
'The effective end date of the trade. Disregard the time component as this is not applicable.';

comment on column GSH_CAPACITY_TRANSACTION.TRADE_TYPE is
'The type of trade';

comment on column GSH_CAPACITY_TRANSACTION.FACILITY_ID is
'The unique identifier of the Part 24 facility. Mandatory for facilities registered under Part 24 of the rules.';

comment on column GSH_CAPACITY_TRANSACTION.FACILITY_NAME is
'The descriptive name of the facility.';

comment on column GSH_CAPACITY_TRANSACTION.TRANSPORTATION_SERVICE_TYPE is
'The transportation service type';

comment on column GSH_CAPACITY_TRANSACTION.FLOW_DIRECTION is
'The direction of the flow for non-Part 24 facilities, relative to the general physical direction of the facility. Valid values are:NORTHNORTH_EASTNORTH_WESTEASTSOUTH_EASTSOUTHSOUTH_WESTWEST';

comment on column GSH_CAPACITY_TRANSACTION.RECEIPT_ZONE_ID is
'The unique identifier of the Part 24 Receipt Zone. This is the zone in which the receipt point that was reported to AEMO is specified. Mandatory for service points registered under Part 24 of the rules.';

comment on column GSH_CAPACITY_TRANSACTION.RECEIPT_ZONE_NAME is
'The descriptive name of the Receipt Zone.';

comment on column GSH_CAPACITY_TRANSACTION.DELIVERY_ZONE_ID is
'The unique identifier of the Part 24 Delivery Zone. This is the zone in which the delivery point that was reported to AEMO is specified. Mandatory for service points registered under Part 24 of the rules.';

comment on column GSH_CAPACITY_TRANSACTION.DELIVERY_ZONE_NAME is
'The descriptive name of the Delivery Zone.';

comment on column GSH_CAPACITY_TRANSACTION.PARK_LOAN_POINT_ID is
'The unique identifier of the Part 24 Park and Loan service point. Mandatory for service points registered under Part 24 of the rules.';

comment on column GSH_CAPACITY_TRANSACTION.PARK_LOAN_POINT_NAME is
'The descriptive name of the service point.';

comment on column GSH_CAPACITY_TRANSACTION.STANDARD_OTSA is
'Flag to indicate whether a standard OTSA was used';

comment on column GSH_CAPACITY_TRANSACTION.PRIORITY is
'The priority of the traded capacity. Mandatory for GBB bilateral trades. Null for GSH trades.';

comment on column GSH_CAPACITY_TRANSACTION.MAX_DAILY_QUANTITY is
'The traded maximum daily quantity (MDQ) (GJ/day).';

comment on column GSH_CAPACITY_TRANSACTION.MAX_HOURLY_QUANTITY is
'The traded maximum hourly quantity (GJ/hour). Null for GSH trades.';

comment on column GSH_CAPACITY_TRANSACTION.PRICE is
'The price of the traded capacity ($/GJ/day).';

comment on column GSH_CAPACITY_TRANSACTION.PRICE_STRUCTURE is
'The price structure applied over the term of the trade. Mandatory for GBB bilateral trades. Null for GSH trades.';

comment on column GSH_CAPACITY_TRANSACTION.PRICE_ESCALATION_MECHANISM is
'The price escalation mechanism applied over the term of the trade. Null for GSH trades.';

comment on column GSH_CAPACITY_TRANSACTION.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_CAPACITY_TRANSACTION
   add constraint GSH_CAPACITY_TRANSACTION_PK primary key (TRANSACTION_ID, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_CAP_XFR_AUCTION_NOTICE                            */
/*==============================================================*/
create table GSH_CAP_XFR_AUCTION_NOTICE (
   NOTICE_ID            NUMERIC(8,0)         not null,
   NOTICE_DATE          TIMESTAMP(0)         null,
   GAS_DATE             TIMESTAMP(0)         null,
   FACILITY_ID          NUMERIC(8,0)         null,
   FACILITY_NAME        VARCHAR(100)         null,
   MARKET               VARCHAR(20)          null,
   NOTICE_TYPE          VARCHAR(50)          null,
   DESCRIPTION          VARCHAR(1000)        null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_CAP_XFR_AUCTION_NOTICE is
'Notifies participants when the market delay or cancellation process is triggered';

comment on column GSH_CAP_XFR_AUCTION_NOTICE.NOTICE_ID is
'The unique identifier of the notice.';

comment on column GSH_CAP_XFR_AUCTION_NOTICE.NOTICE_DATE is
'The date of the notice';

comment on column GSH_CAP_XFR_AUCTION_NOTICE.GAS_DATE is
'The gas date of the notice. Disregard the time component as this is not applicable.';

comment on column GSH_CAP_XFR_AUCTION_NOTICE.FACILITY_ID is
'The unique identifier of the facility.';

comment on column GSH_CAP_XFR_AUCTION_NOTICE.FACILITY_NAME is
'The descriptive name of the facility.';

comment on column GSH_CAP_XFR_AUCTION_NOTICE.MARKET is
'The market that has triggered the notice, for example CTP or DAA.';

comment on column GSH_CAP_XFR_AUCTION_NOTICE.NOTICE_TYPE is
'The type of notice, for example Cancel or Delay for CTP, Exclude or Delay for DAA.';

comment on column GSH_CAP_XFR_AUCTION_NOTICE.DESCRIPTION is
'The description of the notice.';

comment on column GSH_CAP_XFR_AUCTION_NOTICE.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_CAP_XFR_AUCTION_NOTICE
   add constraint GSH_CAP_XFR_AUCTION_NOTICE_PK primary key (NOTICE_ID);

/*==============================================================*/
/* Table: GSH_CAP_XFR_NOTICE                                    */
/*==============================================================*/
create table GSH_CAP_XFR_NOTICE (
   CAPACITY_TRANSFER_ID VARCHAR(20)          not null,
   FACILITY_ID          NUMERIC(8,0)         null,
   FACILITY_NAME        VARCHAR(100)         null,
   PARTICIPANT_ID       VARCHAR(20)          null,
   PARTICIPANT_NAME     VARCHAR(80)          null,
   SERVICE_REFERENCE    VARCHAR(255)         null,
   STTM_CONTRACT_REFERENCE VARCHAR(255)         null,
   RECEIPT_POINT_ID     NUMERIC(8,0)         null,
   RECEIPT_POINT_NAME   VARCHAR(255)         null,
   DELIVERY_POINT_ID    NUMERIC(8,0)         null,
   DELIVERY_POINT_NAME  VARCHAR(255)         null,
   RECEIPT_ZONE_ID      NUMERIC(8,0)         null,
   RECEIPT_ZONE_NAME    VARCHAR(255)         null,
   DELIVERY_ZONE_ID     NUMERIC(8,0)         null,
   DELIVERY_ZONE_NAME   VARCHAR(255)         null,
   GAS_DATE             TIMESTAMP(0)         null,
   INCREMENTAL_QUANTITY_CHANGE NUMERIC(10,0)        null,
   OVERALL_QUANTITY_CHANGE NUMERIC(10,0)        null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_CAP_XFR_NOTICE is
'Capacity transfer notice includes a list of capacity quantities per shipper contract that facility operators use to update the capacity quantity on the relevant shipper contracts in their own systems.';

comment on column GSH_CAP_XFR_NOTICE.CAPACITY_TRANSFER_ID is
'The unique AEMO defined Capacity Transfer transaction identifier.';

comment on column GSH_CAP_XFR_NOTICE.FACILITY_ID is
'The unique facility identifier as defined in the Transportation Facility Register.';

comment on column GSH_CAP_XFR_NOTICE.FACILITY_NAME is
'The descriptive name of the facility as defined in the Transportation Facility Register.';

comment on column GSH_CAP_XFR_NOTICE.PARTICIPANT_ID is
'The unique AEMO defined GSH Participant identifier.';

comment on column GSH_CAP_XFR_NOTICE.PARTICIPANT_NAME is
'The descriptive name of the participant.';

comment on column GSH_CAP_XFR_NOTICE.SERVICE_REFERENCE is
'The service reference issued by the facility operator.';

comment on column GSH_CAP_XFR_NOTICE.STTM_CONTRACT_REFERENCE is
'The STTM registered facility service identifier (CRN) associated with the service reference.';

comment on column GSH_CAP_XFR_NOTICE.RECEIPT_POINT_ID is
'The unique identifier of the Receipt Point as defined in the Transportation Service Point Register. This is the point where gas is injected into the pipeline';

comment on column GSH_CAP_XFR_NOTICE.RECEIPT_POINT_NAME is
'The descriptive name of the Receipt Point as defined in the Transportation Service Point Register.';

comment on column GSH_CAP_XFR_NOTICE.DELIVERY_POINT_ID is
'The unique identifier of the Delivery Point as defined in the Transportation Service Point Register. This is the point where gas is withdrawn from the pipeline.';

comment on column GSH_CAP_XFR_NOTICE.DELIVERY_POINT_NAME is
'The descriptive name of the Delivery Point as defined in the Transportation Service Point Register.';

comment on column GSH_CAP_XFR_NOTICE.RECEIPT_ZONE_ID is
'The unique identifier of the Receipt Zone as defined in the Transportation Service Point Register.';

comment on column GSH_CAP_XFR_NOTICE.RECEIPT_ZONE_NAME is
'The descriptive name of the Receipt Zone as defined in the Transportation Service Point Register.';

comment on column GSH_CAP_XFR_NOTICE.DELIVERY_ZONE_ID is
'The unique identifier of the Delivery Zone as defined in the Transportation Service Point Register.';

comment on column GSH_CAP_XFR_NOTICE.DELIVERY_ZONE_NAME is
'The descriptive name of the Delivery Zone as defined in the Transportation Service Point Register.';

comment on column GSH_CAP_XFR_NOTICE.GAS_DATE is
'The gas date that the capacity change applies to. Disregard the time component as this is not applicable.';

comment on column GSH_CAP_XFR_NOTICE.INCREMENTAL_QUANTITY_CHANGE is
'The incremental capacity quantity adjustment (GJ/day) to be applied to the service reference for this transfer date and gas date. This is a signed value (+/-).';

comment on column GSH_CAP_XFR_NOTICE.OVERALL_QUANTITY_CHANGE is
'The overall capacity quantity adjustment (GJ/day) applied to the relevant contracts for this transfer date and gas date. This is a signed value (+/-).';

comment on column GSH_CAP_XFR_NOTICE.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_CAP_XFR_NOTICE
   add constraint GSH_CAP_XFR_NOTICE_PK primary key (CAPACITY_TRANSFER_ID);

/*==============================================================*/
/* Table: GSH_CONTRACT_DETAILS                                  */
/*==============================================================*/
create table GSH_CONTRACT_DETAILS (
   PARTICIPANT_ID       VARCHAR(20)          not null,
   PARTICIPANT_NAME     VARCHAR(80)          null,
   PARTICIPANT_CODE     VARCHAR(20)          null,
   FACILITY_ID          NUMERIC(8,0)         not null,
   FACILITY_NAME        VARCHAR(100)         null,
   SERVICE_REFERENCE    VARCHAR(255)         not null,
   SERVICE_REFERENCE_DESCRIPTION VARCHAR(255)         null,
   STTM_CONTRACT_REFERENCE VARCHAR(255)         null,
   CONTRACT_TYPE        VARCHAR(20)          null,
   START_DATE           TIMESTAMP(0)         null,
   END_DATE             TIMESTAMP(0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_CONTRACT_DETAILS is
'Provides participants with a list of all the facility operator service references and STTM contract references that are available to them for capacity trading';

comment on column GSH_CONTRACT_DETAILS.PARTICIPANT_ID is
'The unique AEMO defined GSH Participant identifier.';

comment on column GSH_CONTRACT_DETAILS.PARTICIPANT_NAME is
'The descriptive name of the participant.';

comment on column GSH_CONTRACT_DETAILS.PARTICIPANT_CODE is
'The participant code used in the ETS.';

comment on column GSH_CONTRACT_DETAILS.FACILITY_ID is
'The unique facility identifier as defined in the Transportation Facility Register.';

comment on column GSH_CONTRACT_DETAILS.FACILITY_NAME is
'The descriptive name of the facility as defined in the Transportation Facility Register.';

comment on column GSH_CONTRACT_DETAILS.SERVICE_REFERENCE is
'The service reference issued by the facility operator.';

comment on column GSH_CONTRACT_DETAILS.SERVICE_REFERENCE_DESCRIPTION is
'Brief description of the service reference.';

comment on column GSH_CONTRACT_DETAILS.STTM_CONTRACT_REFERENCE is
'The STTM registered facility service identifier (CRN) associated with the service reference if applicable.';

comment on column GSH_CONTRACT_DETAILS.CONTRACT_TYPE is
'Indicates if a contract can be used for capacity trades, auctions or both. For example, Trade, Auction or All.';

comment on column GSH_CONTRACT_DETAILS.START_DATE is
'Effective start date of the contract. Disregard the time component as this is not applicable.';

comment on column GSH_CONTRACT_DETAILS.END_DATE is
'Effective end date of the contract. Disregard the time component as this is not applicable.';

comment on column GSH_CONTRACT_DETAILS.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_CONTRACT_DETAILS
   add constraint GSH_CONTRACT_DETAILS_PK primary key (PARTICIPANT_ID, FACILITY_ID, SERVICE_REFERENCE);

/*==============================================================*/
/* Table: GSH_CONTRACT_REFERENCES                               */
/*==============================================================*/
create table GSH_CONTRACT_REFERENCES (
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   PRODUCT_NAME         VARCHAR(80)          not null,
   BUY_SELL_POSITION    VARCHAR(4)           not null,
   FROM_GAS_DATE        TIMESTAMP(0)         not null,
   TO_GAS_DATE          TIMESTAMP(0)         not null,
   STEP_NUMBER          NUMERIC(6,0)         not null,
   SERVICE_REFERENCE    VARCHAR(255)         null,
   SERVICE_REFERENCE_STATUS VARCHAR(20)          null,
   STTM_CONTRACT_REFERENCE VARCHAR(255)         null,
   STTM_CONTRACT_REFERENCE_STATUS VARCHAR(20)          null,
   DWGM_REFERENCE       VARCHAR(255)         null,
   DWGM_REFERENCE_STATUS VARCHAR(20)          null,
   STEP_QUANTITY        NUMERIC(10,0)        null,
   VERSION_DATETIME     TIMESTAMP(0)         not null,
   LAST_UPDATED         TIMESTAMP(0)         null,
   LAST_UPDATED_BY      VARCHAR(20)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_CONTRACT_REFERENCES is
'This report is available to participants as an audit trail of the entries in the Contract References markets portal application.';

comment on column GSH_CONTRACT_REFERENCES.PARTICIPANT_CODE is
'The participant code used in the ETS.';

comment on column GSH_CONTRACT_REFERENCES.PRODUCT_NAME is
'The capacity product name.';

comment on column GSH_CONTRACT_REFERENCES.BUY_SELL_POSITION is
'Specifies the position that the contract reference must be used for, for example Buy, Sell or Both.';

comment on column GSH_CONTRACT_REFERENCES.FROM_GAS_DATE is
'The first gas date that the contract reference takes effect from. Disregard the time component as this is not applicable.';

comment on column GSH_CONTRACT_REFERENCES.TO_GAS_DATE is
'The last gas date that the contract reference takes effect until. Disregard the time component as this is not applicable.';

comment on column GSH_CONTRACT_REFERENCES.STEP_NUMBER is
'The step number for the corresponding capacity quantity.';

comment on column GSH_CONTRACT_REFERENCES.SERVICE_REFERENCE is
'The service reference issued by the facility operator.';

comment on column GSH_CONTRACT_REFERENCES.SERVICE_REFERENCE_STATUS is
'The status of the service reference, for example Valid, Missing or Expired.';

comment on column GSH_CONTRACT_REFERENCES.STTM_CONTRACT_REFERENCE is
'The STTM registered facility service identifier (CRN) associated with the service reference.';

comment on column GSH_CONTRACT_REFERENCES.STTM_CONTRACT_REFERENCE_STATUS is
'The status of the STTM contract reference, for example Valid, Missing or Expired.';

comment on column GSH_CONTRACT_REFERENCES.DWGM_REFERENCE is
'The DWGM accreditation reference associated with the transfer.';

comment on column GSH_CONTRACT_REFERENCES.DWGM_REFERENCE_STATUS is
'The status of the DWGM accreditation reference, for example Valid, Missing or Expired.';

comment on column GSH_CONTRACT_REFERENCES.STEP_QUANTITY is
'The step quantity to applied to the contract reference (GJ/day).';

comment on column GSH_CONTRACT_REFERENCES.VERSION_DATETIME is
'The date and time the preference was updated.';

comment on column GSH_CONTRACT_REFERENCES.LAST_UPDATED is
'The date and time the preference was updated.';

comment on column GSH_CONTRACT_REFERENCES.LAST_UPDATED_BY is
'The user name who updated the preference.';

comment on column GSH_CONTRACT_REFERENCES.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_CONTRACT_REFERENCES
   add constraint GSH_CONTRACT_REFERENCES_PK primary key (PARTICIPANT_CODE, PRODUCT_NAME, BUY_SELL_POSITION, FROM_GAS_DATE, TO_GAS_DATE, STEP_NUMBER, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_DELIVERED_QUANTITY                                */
/*==============================================================*/
create table GSH_DELIVERED_QUANTITY (
   GAS_DATE             TIMESTAMP(0)         not null,
   TRANSACTION_ID       VARCHAR(20)          not null,
   VERSION_DATETIME     TIMESTAMP(0)         not null,
   BUYER_PARTICIPANT_CODE VARCHAR(20)          null,
   BUYER_PARTICIPANT_NAME VARCHAR(80)          null,
   SELLER_PARTICIPANT_CODE VARCHAR(20)          null,
   SELLER_PARTICIPANT_NAME VARCHAR(80)          null,
   PRODUCT_LOCATION     VARCHAR(80)          null,
   TRANSACTION_QUANTITY NUMERIC(18,8)        null,
   SETTLEMENTS_OFF_MARKET VARCHAR(1)           null,
   DELIVERED_QUANTITY   NUMERIC(18,8)        null,
   REASON_FOR_VARIATION VARCHAR(80)          null,
   STATUS               VARCHAR(20)          null,
   LAST_UPDATED_BY_PARTICIPANT VARCHAR(20)          null,
   LAST_UPDATED         TIMESTAMP(0)         null,
   LASTCHANGED          TIMESTAMP(0)         null,
   ETS_TRADE_ID         VARCHAR(20)          null
);

comment on table GSH_DELIVERED_QUANTITY is
'The purpose of this report is to provide the Participants (buyer and seller) an acknowledgement whenever a delivered quantity record is submitted or confirmed. The report will also provide Participants with a list of all their delivered quantity records updated within the last thirty days when triggered daily prior to the start of trades for the current day.';

comment on column GSH_DELIVERED_QUANTITY.GAS_DATE is
'The delivery gas day of the trade';

comment on column GSH_DELIVERED_QUANTITY.TRANSACTION_ID is
'Unique identifier for the transaction';

comment on column GSH_DELIVERED_QUANTITY.VERSION_DATETIME is
'The date and time of when the entry was last modified in the source table.';

comment on column GSH_DELIVERED_QUANTITY.BUYER_PARTICIPANT_CODE is
'The participant code used in the Exchange Trading System (ETS)';

comment on column GSH_DELIVERED_QUANTITY.BUYER_PARTICIPANT_NAME is
'The name for the participant on buy side of the trade';

comment on column GSH_DELIVERED_QUANTITY.SELLER_PARTICIPANT_CODE is
'The participant code used in the Exchange Trading System (ETS) for the participant on the seller side of the trade';

comment on column GSH_DELIVERED_QUANTITY.SELLER_PARTICIPANT_NAME is
'The name for the participant on seller side of the trade';

comment on column GSH_DELIVERED_QUANTITY.PRODUCT_LOCATION is
'The product location for the Gas Supply Hub market';

comment on column GSH_DELIVERED_QUANTITY.TRANSACTION_QUANTITY is
'Initial gas volume of the transaction in GJ/day';

comment on column GSH_DELIVERED_QUANTITY.SETTLEMENTS_OFF_MARKET is
'Indicates whether the buyer and seller wish to settle the delivery variance outside of the GSH market. Valid values are: Y, N';

comment on column GSH_DELIVERED_QUANTITY.DELIVERED_QUANTITY is
'The delivered quantity in GJ/day';

comment on column GSH_DELIVERED_QUANTITY.REASON_FOR_VARIATION is
'The reason for the variance in the delivered quantity';

comment on column GSH_DELIVERED_QUANTITY.STATUS is
'The status of the delivery confirmation. Valid values are: SUBMITTED, CONFIRMED';

comment on column GSH_DELIVERED_QUANTITY.LAST_UPDATED_BY_PARTICIPANT is
'The participant code used in the Exchange Trading System (ETS) for the participant who last updated the delivered quantity record';

comment on column GSH_DELIVERED_QUANTITY.LAST_UPDATED is
'The date and time the delivered quantity was updated i.e. saved into database';

comment on column GSH_DELIVERED_QUANTITY.LASTCHANGED is
'The date and time the report was generated';

comment on column GSH_DELIVERED_QUANTITY.ETS_TRADE_ID is
'The exchange transaction ID associated with the delivery obligation for transactions in non-netted products. This field can be cross referenced with the TRADE_ID field in the Trade Execution report to identify the transaction price. 
This field is empty for trades involving netted products';

alter table GSH_DELIVERED_QUANTITY
   add constraint GSH_DELIVERED_QUANTITY_PK primary key (GAS_DATE, TRANSACTION_ID, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_DELIVERY_NETT_POINTS                              */
/*==============================================================*/
create table GSH_DELIVERY_NETT_POINTS (
   MARKET_ID            VARCHAR(20)          not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   PRODUCT_LOCATION     VARCHAR(80)          not null,
   VERSION_DATETIME     TIMESTAMP(0)         not null,
   POSITION_TYPE        VARCHAR(20)          not null,
   PRIORITY             NUMERIC(3,0)         not null,
   DELIVERY_POINT       VARCHAR(40)          null,
   VOLUME               NUMERIC(18,8)        null,
   LAST_UPDATED         TIMESTAMP(0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_DELIVERY_NETT_POINTS is
'List of delivery point preferences if delivery points are overridden. Child record of GSH_DELIVERY_NETT_PREFS';

comment on column GSH_DELIVERY_NETT_POINTS.MARKET_ID is
'Unique Market ID';

comment on column GSH_DELIVERY_NETT_POINTS.PARTICIPANT_CODE is
'The participant code';

comment on column GSH_DELIVERY_NETT_POINTS.PRODUCT_LOCATION is
'Product location for the GSH market';

comment on column GSH_DELIVERY_NETT_POINTS.VERSION_DATETIME is
'Date and time preference entered';

comment on column GSH_DELIVERY_NETT_POINTS.POSITION_TYPE is
'Type of position used. Valid values are: NET, BUY, SELL';

comment on column GSH_DELIVERY_NETT_POINTS.PRIORITY is
'Order of priority for overriding delivery points';

comment on column GSH_DELIVERY_NETT_POINTS.DELIVERY_POINT is
'Name of delivery point';

comment on column GSH_DELIVERY_NETT_POINTS.VOLUME is
'Amount (Gj) to assign to delivery point';

comment on column GSH_DELIVERY_NETT_POINTS.LAST_UPDATED is
'The date and time the preference was updated';

comment on column GSH_DELIVERY_NETT_POINTS.LASTCHANGED is
'The date and time the report was generated';

alter table GSH_DELIVERY_NETT_POINTS
   add constraint GSH_DELIVERY_NETT_POINTS_PK primary key (MARKET_ID, PARTICIPANT_CODE, PRODUCT_LOCATION, VERSION_DATETIME, POSITION_TYPE, PRIORITY);

/*==============================================================*/
/* Table: GSH_DELIVERY_NETT_PREFS                               */
/*==============================================================*/
create table GSH_DELIVERY_NETT_PREFS (
   MARKET_ID            VARCHAR(20)          not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   PRODUCT_LOCATION     VARCHAR(80)          not null,
   VERSION_DATETIME     TIMESTAMP(0)         not null,
   START_DATE           TIMESTAMP(0)         null,
   END_DATE             TIMESTAMP(0)         null,
   CHOSEN_POSITION      VARCHAR(20)          null,
   USE_TRANSACTIONS     VARCHAR(1)           null,
   USE_GROUPS           VARCHAR(1)           null,
   LAST_UPDATED         TIMESTAMP(0)         null,
   LAST_UPDATED_BY      VARCHAR(20)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_DELIVERY_NETT_PREFS is
'Provide a history of Delivery Netting Preferences entered through the new Delivery Netting Web Interface.';

comment on column GSH_DELIVERY_NETT_PREFS.MARKET_ID is
'Unique Market ID';

comment on column GSH_DELIVERY_NETT_PREFS.PARTICIPANT_CODE is
'The participant code';

comment on column GSH_DELIVERY_NETT_PREFS.PRODUCT_LOCATION is
'Product location for the GSH market';

comment on column GSH_DELIVERY_NETT_PREFS.VERSION_DATETIME is
'Date and time preference entered';

comment on column GSH_DELIVERY_NETT_PREFS.START_DATE is
'Date preference takes effect from';

comment on column GSH_DELIVERY_NETT_PREFS.END_DATE is
'Date preference takes effect until';

comment on column GSH_DELIVERY_NETT_PREFS.CHOSEN_POSITION is
'Type of position used. Valid values are: NET, BUY/SELL';

comment on column GSH_DELIVERY_NETT_PREFS.USE_TRANSACTIONS is
'Whether original delivery points specified in trades will be used. Valid values are Y | N';

comment on column GSH_DELIVERY_NETT_PREFS.USE_GROUPS is
'Indicates if positions are matched using groups. Valid values are Y | N';

comment on column GSH_DELIVERY_NETT_PREFS.LAST_UPDATED is
'The date and time the preference was updated';

comment on column GSH_DELIVERY_NETT_PREFS.LAST_UPDATED_BY is
'The user name who updated the preference';

comment on column GSH_DELIVERY_NETT_PREFS.LASTCHANGED is
'The date and time the report was generated';

alter table GSH_DELIVERY_NETT_PREFS
   add constraint GSH_DELIVERY_NETT_PREFS_PK primary key (MARKET_ID, PARTICIPANT_CODE, PRODUCT_LOCATION, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_DELIVERY_OBLIGATIONS                              */
/*==============================================================*/
create table GSH_DELIVERY_OBLIGATIONS (
   NETTING_TRANSACTION_ID VARCHAR(20)          not null,
   BUYER_PARTICIPANT_CODE VARCHAR(20)          null,
   BUYER_PARTICIPANT_NAME VARCHAR(80)          null,
   SELLER_PARTICIPANT_CODE VARCHAR(20)          null,
   SELLER_PARTICIPANT_NAME VARCHAR(80)          null,
   FROM_GAS_DATE        TIMESTAMP(0)         null,
   TO_GAS_DATE          TIMESTAMP(0)         null,
   PRODUCT_LOCATION     VARCHAR(80)          null,
   DELIVERY_QUANTITY    NUMERIC(18,8)        null,
   DELIVERY_POINT       VARCHAR(40)          null,
   DELIVERY_TYPE_ALERT  VARCHAR(20)          null,
   LASTCHANGED          TIMESTAMP(0)         null,
   ETS_TRADE_ID         VARCHAR(20)          null,
   RECEIPT_POINT        VARCHAR(40)          null
);

comment on table GSH_DELIVERY_OBLIGATIONS is
'The purpose of this report is to provide the Participants with their Delivery Obligations at each location. Delivery Obligation quantities are results of the delivery netting process.';

comment on column GSH_DELIVERY_OBLIGATIONS.NETTING_TRANSACTION_ID is
'The unique identifier of the delivery transaction.';

comment on column GSH_DELIVERY_OBLIGATIONS.BUYER_PARTICIPANT_CODE is
'The unique identifier for the participant on buy side of the trade that has an obligation to receipt gas
OR
The unique code for the participant with net zero position.';

comment on column GSH_DELIVERY_OBLIGATIONS.BUYER_PARTICIPANT_NAME is
'The name for the participant on buy side of the trade that has an obligation to receipt gas
OR
The name of the participant with net zero position.';

comment on column GSH_DELIVERY_OBLIGATIONS.SELLER_PARTICIPANT_CODE is
'The unique identifier for the participant on seller side of the trade that has an obligation to deliver gas
OR
The unique code for the participant with net zero position.';

comment on column GSH_DELIVERY_OBLIGATIONS.SELLER_PARTICIPANT_NAME is
'The name for the participant on seller side of the trade that has an obligation to deliver gas
OR
The name of the participant with net zero position.';

comment on column GSH_DELIVERY_OBLIGATIONS.FROM_GAS_DATE is
'The first gas date in the netting period.';

comment on column GSH_DELIVERY_OBLIGATIONS.TO_GAS_DATE is
'The last gas date of the netting period.';

comment on column GSH_DELIVERY_OBLIGATIONS.PRODUCT_LOCATION is
'The product location, product grouping code for the Gas Supply Hub market';

comment on column GSH_DELIVERY_OBLIGATIONS.DELIVERY_QUANTITY is
'The quantity of the delivery transaction GJ/day
OR
Zero if the participant has a zero net position.';

comment on column GSH_DELIVERY_OBLIGATIONS.DELIVERY_POINT is
'Delivery point/s and associated quantities determined as part of SUC7407a  Perform Participant Location Netting.
OR
NULL if the participant has a zero net position.';

comment on column GSH_DELIVERY_OBLIGATIONS.DELIVERY_TYPE_ALERT is
'This field is empty if the transaction is an output of the delivery-netting module.
If the transaction is an original executed trade (due to a system failure), this field shows Individual Contract Delivery.';

comment on column GSH_DELIVERY_OBLIGATIONS.LASTCHANGED is
'The date and time the report was generated';

comment on column GSH_DELIVERY_OBLIGATIONS.ETS_TRADE_ID is
'The exchange transaction ID associated with the delivery obligation for transactions in non-netted products. This field can be cross referenced with the TRADE_ID field in the Trade Execution report to identify the transaction price. This field is empty for trades involving netted products.';

comment on column GSH_DELIVERY_OBLIGATIONS.RECEIPT_POINT is
'Receipt point specified by the buyer or seller in an order submission for capacity products';

alter table GSH_DELIVERY_OBLIGATIONS
   add constraint GSH_DELIVERY_OBLIGATION_PK primary key (NETTING_TRANSACTION_ID);

/*==============================================================*/
/* Table: GSH_FACILITIES                                        */
/*==============================================================*/
create table GSH_FACILITIES (
   FACILITY_ID          NUMERIC(8,0)         not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSION_DATETIME     TIMESTAMP(0)         not null,
   FACILITY_NAME        VARCHAR(100)         null,
   FACILITY_DESCRIPTION VARCHAR(400)         null,
   FACILITY_TYPE        VARCHAR(20)          null,
   OPERATOR_ID          VARCHAR(20)          null,
   OPERATOR_NAME        VARCHAR(100)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_FACILITIES is
'Facilities currently registered, for capacity trading and day ahead auction.';

comment on column GSH_FACILITIES.FACILITY_ID is
'The unique identifier of the facility.';

comment on column GSH_FACILITIES.EFFECTIVEDATE is
'Effective date of the record';

comment on column GSH_FACILITIES.VERSION_DATETIME is
'Version timestamp of the record';

comment on column GSH_FACILITIES.FACILITY_NAME is
'The name of the facility.';

comment on column GSH_FACILITIES.FACILITY_DESCRIPTION is
'The descriptive name of the facility.';

comment on column GSH_FACILITIES.FACILITY_TYPE is
'The type of facility, for example pipe or compressor.';

comment on column GSH_FACILITIES.OPERATOR_ID is
'The unique identifier of the Facility Operator that operates the facility.';

comment on column GSH_FACILITIES.OPERATOR_NAME is
'The descriptive name of the participant.';

comment on column GSH_FACILITIES.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_FACILITIES
   add constraint GSH_FACILITIES_PK primary key (FACILITY_ID, EFFECTIVEDATE, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_HISTORICAL_SUMMARY                                */
/*==============================================================*/
create table GSH_HISTORICAL_SUMMARY (
   GAS_DATE             TIMESTAMP(0)         not null,
   PRODUCT_LOCATION     VARCHAR(80)          not null,
   VOLUME_WEIGHTED_AVERAGE_PRICE NUMERIC(18,8)        null,
   TOTAL_QUANTITY       NUMERIC(18,8)        null,
   ROLLING_WEIGHTED_AVERAGE_PRICE NUMERIC(18,8)        null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_HISTORICAL_SUMMARY is
'The report provides a summary of all trades delivered in the current gas day and the last thirty gas days per product location. It provides information on the volume weighted average price, and the total traded quantities.';

comment on column GSH_HISTORICAL_SUMMARY.GAS_DATE is
'The gas date when the delivery of trades occurred';

comment on column GSH_HISTORICAL_SUMMARY.PRODUCT_LOCATION is
'Product location for the Gas Supply Hub market';

comment on column GSH_HISTORICAL_SUMMARY.VOLUME_WEIGHTED_AVERAGE_PRICE is
'The volume weighted average price for trades delivered in the specified gas day at the specified location in $/GJ';

comment on column GSH_HISTORICAL_SUMMARY.TOTAL_QUANTITY is
'Total traded quantities for the specified gas day at the specified location in GJ/day';

comment on column GSH_HISTORICAL_SUMMARY.ROLLING_WEIGHTED_AVERAGE_PRICE is
'The rolling 30 days volume weighted average price for trades delivered in the last 30 days at the specified location in $/GJ';

comment on column GSH_HISTORICAL_SUMMARY.LASTCHANGED is
'The date and time the report was generated';

alter table GSH_HISTORICAL_SUMMARY
   add constraint GSH_HISTORICAL_SUMMARY_PK primary key (GAS_DATE, PRODUCT_LOCATION);

/*==============================================================*/
/* Table: GSH_NOTIONAL_POINTS                                   */
/*==============================================================*/
create table GSH_NOTIONAL_POINTS (
   FACILITY_ID          NUMERIC(8,0)         not null,
   SERVICE_POINT_ID     NUMERIC(8,0)         not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSION_DATETIME     TIMESTAMP(0)         not null,
   SERVICE_POINT_NAME   VARCHAR(100)         null,
   PHYSICAL_POINT_NAME  VARCHAR(100)         not null,
   GBB_CONNECTION_POINT_ID NUMERIC(8,0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_NOTIONAL_POINTS is
'Notional Point currently registered under Part 24 of the Rules';

comment on column GSH_NOTIONAL_POINTS.FACILITY_ID is
'Unique Facility ID for this market';

comment on column GSH_NOTIONAL_POINTS.SERVICE_POINT_ID is
'The unique identifier of the service point.';

comment on column GSH_NOTIONAL_POINTS.EFFECTIVEDATE is
'The effective date of the notional point.';

comment on column GSH_NOTIONAL_POINTS.VERSION_DATETIME is
'The version timestamp of the record';

comment on column GSH_NOTIONAL_POINTS.SERVICE_POINT_NAME is
'The descriptive name of the service point.';

comment on column GSH_NOTIONAL_POINTS.PHYSICAL_POINT_NAME is
'The descriptive name of the physical point associated with the notional point.';

comment on column GSH_NOTIONAL_POINTS.GBB_CONNECTION_POINT_ID is
'The corresponding GBB connection point identifier.';

comment on column GSH_NOTIONAL_POINTS.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_NOTIONAL_POINTS
   add constraint GSH_NOTIONAL_POINTS_PK primary key (FACILITY_ID, EFFECTIVEDATE, VERSION_DATETIME, SERVICE_POINT_ID, PHYSICAL_POINT_NAME);

/*==============================================================*/
/* Table: GSH_ORDERS                                            */
/*==============================================================*/
create table GSH_ORDERS (
   MARKET_ID            VARCHAR(20)          not null,
   ORDER_ID             VARCHAR(20)          not null,
   PARTICIPANT_CODE     VARCHAR(20)          null,
   PARTICIPANT_NAME     VARCHAR(80)          null,
   PRODUCT_LOCATION     VARCHAR(80)          null,
   PRODUCT_TYPE         VARCHAR(80)          null,
   FROM_GAS_DATE        TIMESTAMP(0)         null,
   TO_GAS_DATE          TIMESTAMP(0)         null,
   ORDER_DATE           TIMESTAMP(0)         null,
   ORDER_STATUS         VARCHAR(20)          null,
   ORDER_SIDE           VARCHAR(5)           null,
   ORDER_QUANTITY       NUMERIC(18,8)        null,
   ORDER_PRICE          NUMERIC(18,8)        null,
   EXPIRY_TYPE          VARCHAR(80)          null,
   EXPIRY_DATE          TIMESTAMP(0)         null,
   DELIVERY_POINT       VARCHAR(40)          null,
   LAST_UPDATED         TIMESTAMP(0)         null,
   LAST_UPDATED_BY      VARCHAR(20)          null,
   LASTCHANGED          TIMESTAMP(0)         null,
   ALL_OR_NONE          NUMERIC(1,0)         null,
   DURATION             VARCHAR(20)          null,
   HIDDEN_VOLUME        NUMERIC(18,8)        null,
   RECEIPT_POINT        VARCHAR(40)          null,
   VALID_SERVICE_REFERENCE VARCHAR(3)           null,
   VALID_STTM_CONTRACT_REFERENCE VARCHAR(3)           null,
   VALID_DWGM_REFERENCE VARCHAR(3)           null
);

comment on table GSH_ORDERS is
'The purpose of this report is to provide the Participant a confirmation whenever an order (bid or offer) is submitted, amended, or cancelled.';

comment on column GSH_ORDERS.MARKET_ID is
'Unique Market ID';

comment on column GSH_ORDERS.ORDER_ID is
'Unique order id generated by the Trading Platform on submission of the order';

comment on column GSH_ORDERS.PARTICIPANT_CODE is
'The participant code used in the Exchange Trading System (ETS)';

comment on column GSH_ORDERS.PARTICIPANT_NAME is
'The name of the participant that the order belongs to';

comment on column GSH_ORDERS.PRODUCT_LOCATION is
'Product location for the Gas Supply Hub market';

comment on column GSH_ORDERS.PRODUCT_TYPE is
'The product delivery period for the Gas Supply Hub market';

comment on column GSH_ORDERS.FROM_GAS_DATE is
'The start gas day for the order delivery period';

comment on column GSH_ORDERS.TO_GAS_DATE is
'The end gas day for the order delivery period';

comment on column GSH_ORDERS.ORDER_DATE is
'The date and time that the order is submitted on';

comment on column GSH_ORDERS.ORDER_STATUS is
'The status of the order. Valid values are: NEW, UPDATE, CANCELLED';

comment on column GSH_ORDERS.ORDER_SIDE is
'This field is to indicate whether the order is a Bid or Offer';

comment on column GSH_ORDERS.ORDER_QUANTITY is
'Gas volume of the order in GJ/day';

comment on column GSH_ORDERS.ORDER_PRICE is
'Price value of the order in $/GJ';

comment on column GSH_ORDERS.EXPIRY_TYPE is
'Order expiry type, for example (Good Till Cancelled, Good For Day, Good Till Date)';

comment on column GSH_ORDERS.EXPIRY_DATE is
'Date and time the order will expire. Only applies to "Good_Till_Date" expiry type';

comment on column GSH_ORDERS.DELIVERY_POINT is
'Delivery Point specified by the seller in an order submission for non-netted commodity products, or the buyer or seller in an order submission for capacity products';

comment on column GSH_ORDERS.LAST_UPDATED is
'The date and time the bid/offer was updated i.e. saved into database';

comment on column GSH_ORDERS.LAST_UPDATED_BY is
'The user name used in the bid/offer submission';

comment on column GSH_ORDERS.LASTCHANGED is
'The date and time the report was generated';

comment on column GSH_ORDERS.ALL_OR_NONE is
'Populated if the trading participant selects the "All or None" option when submitting an order';

comment on column GSH_ORDERS.DURATION is
'Time from order submission until expiry. Data has day format (ISO 8601)';

comment on column GSH_ORDERS.HIDDEN_VOLUME is
'Populated if the trading participant specifies a "Hidden Volume" when submitting an order';

comment on column GSH_ORDERS.RECEIPT_POINT is
'Receipt Point specified by the buyer or seller in an order submission for capacity products';

comment on column GSH_ORDERS.VALID_SERVICE_REFERENCE is
'Flag to indicate whether a valid service reference exists for the capacity product. Will be NULL for orders involving commodity products';

comment on column GSH_ORDERS.VALID_STTM_CONTRACT_REFERENCE is
'Flag to indicate whether a valid STTM contract reference exists for the capacity product. Will be NULL for orders involving commodity products and non-STTM integrated capacity products';

comment on column GSH_ORDERS.VALID_DWGM_REFERENCE is
'Flag to indicate whether a valid DWGM accreditation reference exists for the capacity product where a DWGM interface point has been specified. Will be NULL for orders involving commodity products and capacity products that do not have a DWGM interface point specified';

alter table GSH_ORDERS
   add constraint GSH_ORDERS_PK primary key (MARKET_ID, ORDER_ID);

/*==============================================================*/
/* Table: GSH_PARK_SERVICES                                     */
/*==============================================================*/
create table GSH_PARK_SERVICES (
   FACILITY_ID          NUMERIC(8,0)         not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSION_DATETIME     TIMESTAMP(0)         not null,
   PARK_PRODUCT_NAME    VARCHAR(100)         not null,
   PARK_SERVICE_POINT_ID NUMERIC(8,0)         null,
   PARK_SERVICE_POINT_NAME VARCHAR(100)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_PARK_SERVICES is
'Park Services currently registered under Part 24 of the Rules';

comment on column GSH_PARK_SERVICES.FACILITY_ID is
'Unique Facility ID for this market';

comment on column GSH_PARK_SERVICES.EFFECTIVEDATE is
'The effective date of the park service.';

comment on column GSH_PARK_SERVICES.VERSION_DATETIME is
'The version timestamp of the record';

comment on column GSH_PARK_SERVICES.PARK_PRODUCT_NAME is
'The descriptive name of the park product.';

comment on column GSH_PARK_SERVICES.PARK_SERVICE_POINT_ID is
'The unique identifier of the service point at which transportation capacity is required to use this park service.';

comment on column GSH_PARK_SERVICES.PARK_SERVICE_POINT_NAME is
'The descriptive name of the park service point.';

comment on column GSH_PARK_SERVICES.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_PARK_SERVICES
   add constraint GSH_PARK_SERVICES_PK primary key (FACILITY_ID, EFFECTIVEDATE, VERSION_DATETIME, PARK_PRODUCT_NAME);

/*==============================================================*/
/* Table: GSH_PARTICIPANTS                                      */
/*==============================================================*/
create table GSH_PARTICIPANTS (
   PARTICIPANT_ID       VARCHAR(20)          not null,
   PARTICIPANT_CODE     VARCHAR(20)          null,
   ORGANISATION_NAME    VARCHAR(80)          null,
   ORGANISATION_CODE    VARCHAR(20)          null,
   TRADING_NAME         VARCHAR(80)          null,
   HEAD_OFFICE_ADDRESS_LINE_1 VARCHAR(80)          null,
   HEAD_OFFICE_ADDRESS_LINE_2 VARCHAR(80)          null,
   HEAD_OFFICE_ADDRESS_LINE_3 VARCHAR(80)          null,
   HEAD_OFFICE_ADDRESS_LINE_4 VARCHAR(80)          null,
   PHONE_NUMBER         VARCHAR(80)          null,
   FAX_NUMBER           VARCHAR(80)          null,
   ABN                  VARCHAR(20)          null,
   ARBN                 VARCHAR(20)          null,
   ACN                  VARCHAR(20)          null,
   CLASSIFICATIONS      VARCHAR(255)         null,
   LASTCHANGED          TIMESTAMP(0)         null,
   MARKET_ID            VARCHAR(20)          not null default 'Gas Supply Hub',
   AGENT_ID             VARCHAR(20)          null,
   AGENT_NAME           VARCHAR(80)          null
);

comment on table GSH_PARTICIPANTS is
'The report contains the registration details of all participants currently registered in the GSH market.';

comment on column GSH_PARTICIPANTS.PARTICIPANT_ID is
'The unique identifier of the participant';

comment on column GSH_PARTICIPANTS.PARTICIPANT_CODE is
'The participant code';

comment on column GSH_PARTICIPANTS.ORGANISATION_NAME is
'Name of the organisation who has the ABN';

comment on column GSH_PARTICIPANTS.ORGANISATION_CODE is
'Unique code of the organisation';

comment on column GSH_PARTICIPANTS.TRADING_NAME is
'Trading name of the organisation';

comment on column GSH_PARTICIPANTS.HEAD_OFFICE_ADDRESS_LINE_1 is
'The head office address of the organisation';

comment on column GSH_PARTICIPANTS.HEAD_OFFICE_ADDRESS_LINE_2 is
'The head office address of the organisation';

comment on column GSH_PARTICIPANTS.HEAD_OFFICE_ADDRESS_LINE_3 is
'The head office address of the organisation';

comment on column GSH_PARTICIPANTS.HEAD_OFFICE_ADDRESS_LINE_4 is
'The head office address of the organisation';

comment on column GSH_PARTICIPANTS.PHONE_NUMBER is
'The phone number of the organisation';

comment on column GSH_PARTICIPANTS.FAX_NUMBER is
'The fax number for the organisation';

comment on column GSH_PARTICIPANTS.ABN is
'ABN of the organisation';

comment on column GSH_PARTICIPANTS.ARBN is
'ARBN of the organisation';

comment on column GSH_PARTICIPANTS.ACN is
'ACN of the organisation';

comment on column GSH_PARTICIPANTS.CLASSIFICATIONS is
'The classification of the company in the GSH market, for example trading, view-only, etc.';

comment on column GSH_PARTICIPANTS.LASTCHANGED is
'The date and time the report is generated';

comment on column GSH_PARTICIPANTS.MARKET_ID is
'The market in which the company is a participant, for example GSH, DAA, ALL.';

comment on column GSH_PARTICIPANTS.AGENT_ID is
'The unique participant identifier of the agent appointed to trade on the participants behalf.';

comment on column GSH_PARTICIPANTS.AGENT_NAME is
'Name of the agent organisation';

alter table GSH_PARTICIPANTS
   add constraint GSH_PARTICIPANTS_PK primary key (MARKET_ID, PARTICIPANT_ID);

/*==============================================================*/
/* Table: GSH_PARTICIPANT_CONTACTS                              */
/*==============================================================*/
create table GSH_PARTICIPANT_CONTACTS (
   CONTACT_ID           VARCHAR(20)          not null,
   PARTICIPANT_CODE     VARCHAR(20)          null,
   PARTICIPANT_NAME     VARCHAR(80)          null,
   ROLE                 VARCHAR(80)          null,
   FIRST_NAME           VARCHAR(80)          null,
   LAST_NAME            VARCHAR(80)          null,
   SALUTATION           VARCHAR(20)          null,
   JOB_TITLE            VARCHAR(80)          null,
   ADDRESS_LINE_1       VARCHAR(80)          null,
   ADDRESS_LINE_2       VARCHAR(80)          null,
   ADDRESS_LINE_3       VARCHAR(80)          null,
   ADDRESS_LINE_4       VARCHAR(80)          null,
   BUSINESS_PHONE_NUMBER VARCHAR(80)          null,
   MOBILE_NUMBER        VARCHAR(80)          null,
   FAX_NUMBER           VARCHAR(80)          null,
   EMAIL_ADDRESS        VARCHAR(255)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_PARTICIPANT_CONTACTS is
'The purpose of this report is to provide the participant with a list of all their contact details.';

comment on column GSH_PARTICIPANT_CONTACTS.CONTACT_ID is
'The unique identifier for the contact';

comment on column GSH_PARTICIPANT_CONTACTS.PARTICIPANT_CODE is
'The participant code';

comment on column GSH_PARTICIPANT_CONTACTS.PARTICIPANT_NAME is
'The trading participant name';

comment on column GSH_PARTICIPANT_CONTACTS.ROLE is
'Contact type, for example: Trading Contact, Gas Delivery Contact, System contact, Regulatory contact, Head office contact, Primary settlements contact, Secondary settlements contact, Settlements manager contact, CFO, CEO, First Escalation settlements, Second escalation settlements, Third escalation settlements';

comment on column GSH_PARTICIPANT_CONTACTS.FIRST_NAME is
'First name of the contact';

comment on column GSH_PARTICIPANT_CONTACTS.LAST_NAME is
'Last name of the contact';

comment on column GSH_PARTICIPANT_CONTACTS.SALUTATION is
'Salutation of the contact';

comment on column GSH_PARTICIPANT_CONTACTS.JOB_TITLE is
'Job title for the contact';

comment on column GSH_PARTICIPANT_CONTACTS.ADDRESS_LINE_1 is
'The address of the contact';

comment on column GSH_PARTICIPANT_CONTACTS.ADDRESS_LINE_2 is
'The address of the contact';

comment on column GSH_PARTICIPANT_CONTACTS.ADDRESS_LINE_3 is
'The address of the contact';

comment on column GSH_PARTICIPANT_CONTACTS.ADDRESS_LINE_4 is
'The address of the contact';

comment on column GSH_PARTICIPANT_CONTACTS.BUSINESS_PHONE_NUMBER is
'The business phone number for the contact';

comment on column GSH_PARTICIPANT_CONTACTS.MOBILE_NUMBER is
'Mobile number of the contact';

comment on column GSH_PARTICIPANT_CONTACTS.FAX_NUMBER is
'Fax number for the contact';

comment on column GSH_PARTICIPANT_CONTACTS.EMAIL_ADDRESS is
'Email address of the contact';

comment on column GSH_PARTICIPANT_CONTACTS.LASTCHANGED is
'The date and time the report is issued';

alter table GSH_PARTICIPANT_CONTACTS
   add constraint GSH_PARTICIPANT_CONTACTS_PK primary key (CONTACT_ID);

/*==============================================================*/
/* Table: GSH_PARTICIPANT_OPSTATE                               */
/*==============================================================*/
create table GSH_PARTICIPANT_OPSTATE (
   MARKET_ID            VARCHAR(20)          not null,
   PARTICIPANT_ID       VARCHAR(20)          not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSION_DATETIME     TIMESTAMP(0)         not null,
   STATUS               VARCHAR(30)          null,
   AUTHORISEDDATE       TIMESTAMP(0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_PARTICIPANT_OPSTATE is
'Operating Status of Participant';

comment on column GSH_PARTICIPANT_OPSTATE.MARKET_ID is
'Unique Market ID';

comment on column GSH_PARTICIPANT_OPSTATE.PARTICIPANT_ID is
'Unique Participant ID for this market';

comment on column GSH_PARTICIPANT_OPSTATE.EFFECTIVEDATE is
'The Effective date of the Participant Operating State';

comment on column GSH_PARTICIPANT_OPSTATE.VERSION_DATETIME is
'Versioning with EffectiveDate';

comment on column GSH_PARTICIPANT_OPSTATE.STATUS is
'Current State of the Product (ACTIVE/INACTIVE)';

comment on column GSH_PARTICIPANT_OPSTATE.AUTHORISEDDATE is
'The date of the latest authorisation record';

comment on column GSH_PARTICIPANT_OPSTATE.LASTCHANGED is
'Last changed date for the record';

alter table GSH_PARTICIPANT_OPSTATE
   add constraint GSH_PARTICIPANT_OPSTATE_PK primary key (MARKET_ID, PARTICIPANT_ID, EFFECTIVEDATE, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_PIPELINE_SEGMENTS                                 */
/*==============================================================*/
create table GSH_PIPELINE_SEGMENTS (
   FACILITY_ID          NUMERIC(8,0)         not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSION_DATETIME     TIMESTAMP(0)         not null,
   SEGMENT_ID           NUMERIC(8,0)         not null,
   FACILITY_NAME        VARCHAR(100)         null,
   SEGMENT_NAME         VARCHAR(100)         null,
   SEGMENT_TYPE         VARCHAR(20)          null,
   FROM_ID              NUMERIC(8,0)         not null,
   FROM_NAME            VARCHAR(100)         null,
   TO_ID                NUMERIC(8,0)         not null,
   TO_NAME              VARCHAR(100)         null,
   CONNECTION_TYPE      VARCHAR(20)          not null,
   FOR_AUCTION_PUBLICATION VARCHAR(3)           null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_PIPELINE_SEGMENTS is
'Pipeline Segments currently registered under Part 24 of the Rules for the capacity trading and day ahead auction.';

comment on column GSH_PIPELINE_SEGMENTS.FACILITY_ID is
'The unique identifier of the parent facility.';

comment on column GSH_PIPELINE_SEGMENTS.EFFECTIVEDATE is
'The effective date of the segment.';

comment on column GSH_PIPELINE_SEGMENTS.VERSION_DATETIME is
'The version timestamp of the record';

comment on column GSH_PIPELINE_SEGMENTS.SEGMENT_ID is
'The unique identifier of the pipeline segment or compression service.';

comment on column GSH_PIPELINE_SEGMENTS.FACILITY_NAME is
'The descriptive name of the parent facility.';

comment on column GSH_PIPELINE_SEGMENTS.SEGMENT_NAME is
'The descriptive name of the pipeline segment or compression service.';

comment on column GSH_PIPELINE_SEGMENTS.SEGMENT_TYPE is
'The segment type, for example Forward Haul, Backhaul or Compression Service';

comment on column GSH_PIPELINE_SEGMENTS.FROM_ID is
'The unique identifier of the zone for forward haul pipeline segments and compression services or the service point for backhaul pipeline segments.';

comment on column GSH_PIPELINE_SEGMENTS.FROM_NAME is
'The descriptive name of the zone or service point.';

comment on column GSH_PIPELINE_SEGMENTS.TO_ID is
'The unique identifier of the zone for forward haul pipeline segments and compression services or the service point for backhaul pipeline segments.';

comment on column GSH_PIPELINE_SEGMENTS.TO_NAME is
'The descriptive name of the zone or service point.';

comment on column GSH_PIPELINE_SEGMENTS.CONNECTION_TYPE is
'Type of connection (eg. SERVICE_POINT, ZONE)';

comment on column GSH_PIPELINE_SEGMENTS.FOR_AUCTION_PUBLICATION is
'Flag to indicate if the product component can be published in the public auction reports, for example yes or no.';

comment on column GSH_PIPELINE_SEGMENTS.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_PIPELINE_SEGMENTS
   add constraint GSH_PIPELINE_SEGMENTS_PK primary key (FACILITY_ID, EFFECTIVEDATE, VERSION_DATETIME, SEGMENT_ID, CONNECTION_TYPE, FROM_ID, TO_ID);

/*==============================================================*/
/* Table: GSH_PRD_AUCT_FWD_EXPOSURE                             */
/*==============================================================*/
create table GSH_PRD_AUCT_FWD_EXPOSURE (
   MARKET_ID            VARCHAR(20)          null,
   ORGANISATION_CODE    VARCHAR(20)          not null,
   ORGANISATION_NAME    VARCHAR(80)          null,
   PRUDENTIAL_RUN_ID    NUMERIC(4,0)         not null,
   PRUDENTIAL_DATETIME  TIMESTAMP(0)         not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   BIDS_FORWARD_EXP_AMT_NET_GST NUMERIC(18,8)        null,
   BIDS_FORWARD_EXP_GST_AMT NUMERIC(18,8)        null,
   BID_QUANTITY         NUMERIC(18,8)        null
);

comment on table GSH_PRD_AUCT_FWD_EXPOSURE is
'The purpose of the Prudential Exposure report is to provide organisations with daily estimates of their market exposure. The report also contains details of their bank guarantees and security deposits as well as their outstanding amounts for all settlement runs included in the prudential run. The Auction Forward Exposure subreport provides a summary of auction forward exposure amounts for the organisation by gas day.';

comment on column GSH_PRD_AUCT_FWD_EXPOSURE.MARKET_ID is
'Unique Market ID';

comment on column GSH_PRD_AUCT_FWD_EXPOSURE.ORGANISATION_CODE is
'The unique code for the organisation.';

comment on column GSH_PRD_AUCT_FWD_EXPOSURE.ORGANISATION_NAME is
'The name of the organisation.';

comment on column GSH_PRD_AUCT_FWD_EXPOSURE.PRUDENTIAL_RUN_ID is
'The run number of the prudential run.';

comment on column GSH_PRD_AUCT_FWD_EXPOSURE.PRUDENTIAL_DATETIME is
'The date and time of the prudential run.';

comment on column GSH_PRD_AUCT_FWD_EXPOSURE.GAS_DATE is
'The gas day. Disregard the time component as this is not applicable.';

comment on column GSH_PRD_AUCT_FWD_EXPOSURE.BIDS_FORWARD_EXP_AMT_NET_GST is
'The sum of all day ahead auction bids forward exposure amounts allocated to the organisation on this gas dayexclusive of the bids forward exposure GST amount.';

comment on column GSH_PRD_AUCT_FWD_EXPOSURE.BIDS_FORWARD_EXP_GST_AMT is
'The total amount of GST for the day ahead auction bids forward exposure.';

comment on column GSH_PRD_AUCT_FWD_EXPOSURE.BID_QUANTITY is
'The total quantity for auction bids.';

alter table GSH_PRD_AUCT_FWD_EXPOSURE
   add constraint GSH_PRD_AUCT_FWD_EXPOSURE_PK primary key (ORGANISATION_CODE, PRUDENTIAL_RUN_ID, PRUDENTIAL_DATETIME, GAS_DATE);

/*==============================================================*/
/* Table: GSH_PRD_BANK_GUARANTEES                               */
/*==============================================================*/
create table GSH_PRD_BANK_GUARANTEES (
   ORGANISATION_CODE    VARCHAR(20)          not null,
   PRUDENTIAL_RUN_ID    NUMERIC(4,0)         not null,
   PRUDENTIAL_DATETIME  TIMESTAMP(0)         not null,
   BANK_GUARANTEE_NO    VARCHAR(20)          not null,
   ORGANISATION_NAME    VARCHAR(80)          null,
   EFFECTIVE_FROM_DATE  TIMESTAMP(0)         null,
   EXPIRY_DATE          TIMESTAMP(0)         null,
   PRUDENTIAL_EXPIRY_DATE TIMESTAMP(0)         null,
   AMOUNT               NUMERIC(18,8)        null,
   BANK_GUARANTEE_ALERT VARCHAR(255)         null
);

comment on table GSH_PRD_BANK_GUARANTEES is
'The purpose of the Prudential Exposure report is to provide organisations with daily estimates of their market exposure. The report also contains details of their bank guarantees and security deposits as well as their outstanding amounts for all settlement runs included in the prudential run. The Bank Guarantees subreport provides a list of all bank guarantees used in the prudential run.';

comment on column GSH_PRD_BANK_GUARANTEES.ORGANISATION_CODE is
'The unique code for the organisation';

comment on column GSH_PRD_BANK_GUARANTEES.PRUDENTIAL_RUN_ID is
'The run number of the prudential run.';

comment on column GSH_PRD_BANK_GUARANTEES.PRUDENTIAL_DATETIME is
'The date and time of the prudential run';

comment on column GSH_PRD_BANK_GUARANTEES.BANK_GUARANTEE_NO is
'The reference number for the bank guarantee included in the prudential run';

comment on column GSH_PRD_BANK_GUARANTEES.ORGANISATION_NAME is
'The name of the organisation';

comment on column GSH_PRD_BANK_GUARANTEES.EFFECTIVE_FROM_DATE is
'The start date of the bank guarantee';

comment on column GSH_PRD_BANK_GUARANTEES.EXPIRY_DATE is
'Expiry date for the bank guarantee';

comment on column GSH_PRD_BANK_GUARANTEES.PRUDENTIAL_EXPIRY_DATE is
'The prudential expiry date of the guarantee';

comment on column GSH_PRD_BANK_GUARANTEES.AMOUNT is
'The dollar amount of the guarantee applicable to the prudential run';

comment on column GSH_PRD_BANK_GUARANTEES.BANK_GUARANTEE_ALERT is
'The field will show the warning Renew Bank Guarantee Warning if the prudential expiry date of the bank guarantee is within the next 30 days (PRUDENTIAL_EXPIRY_DATE is less than or equal to PRUDENTIAL_DATETIME + 30 days), otherwise it will have no value';

alter table GSH_PRD_BANK_GUARANTEES
   add constraint GSH_PRD_BANK_GUARANTEES_PK primary key (ORGANISATION_CODE, PRUDENTIAL_RUN_ID, PRUDENTIAL_DATETIME, BANK_GUARANTEE_NO);

/*==============================================================*/
/* Table: GSH_PRD_EST_MARKET_EXPOSURE                           */
/*==============================================================*/
create table GSH_PRD_EST_MARKET_EXPOSURE (
   ORGANISATION_CODE    VARCHAR(20)          not null,
   PRUDENTIAL_RUN_ID    NUMERIC(4,0)         not null,
   PRUDENTIAL_DATETIME  TIMESTAMP(0)         not null,
   ORGANISATION_NAME    VARCHAR(80)          null,
   TOTAL_BANK_GUARANTEES NUMERIC(18,8)        null,
   TRADING_LIMIT        NUMERIC(18,8)        null,
   SECURITY_DEPOSITS    NUMERIC(18,8)        null,
   EARLY_PAYMENTS       NUMERIC(18,8)        null,
   OUTSTANDING_AMOUNT   NUMERIC(18,8)        null,
   REALLOC_ADJ_AMT_TO_OUTSTANDING NUMERIC(18,8)        null,
   SETTLEMENT_EXPOSURE_AMOUNT NUMERIC(18,8)        null,
   TRADES_FORWARD_EXPOSURE_AMOUNT NUMERIC(18,8)        null,
   FORWARD_REALLOCATIONS_AMOUNT NUMERIC(18,8)        null,
   FORWARD_EXPOSURE_AMOUNT NUMERIC(18,8)        null,
   PRUDENTIAL_EXPOSURE  NUMERIC(18,8)        null,
   PERCENTAGE_EXPOSURE  NUMERIC(18,8)        null,
   TRADING_MARGIN       NUMERIC(18,8)        null,
   BANK_GUARANTEE_RATIO NUMERIC(18,8)        null,
   EXPOSURE_NOTIFICATION VARCHAR(80)          null,
   LASTCHANGED          TIMESTAMP(0)         null,
   DAA_SETTLEMENT_EXPOSURE_AMOUNT NUMERIC(18,8)        null,
   DAA_FORWARD_EXPOSURE_AMOUNT NUMERIC(18,8)        null
);

comment on table GSH_PRD_EST_MARKET_EXPOSURE is
'The purpose of the Prudential Exposure report is to provide organisations with daily estimates of their market exposure. The report also contains details of their bank guarantees and security deposits as well as their outstanding amounts for all settlement runs included in the prudential run. The Estimated Market Exposure subreport provides summary information for the Organisations estimated market exposure.';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.ORGANISATION_CODE is
'The unique code for the organisation';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.PRUDENTIAL_RUN_ID is
'The run number of the prudential run';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.PRUDENTIAL_DATETIME is
'The date and time of the prudential run';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.ORGANISATION_NAME is
'The name of the organisation';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.TOTAL_BANK_GUARANTEES is
'Total amount of all valid bank guarantees included in the prudential run';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.TRADING_LIMIT is
'The calculated trading limit for the organisation
(organisations trading limit is calculated by multiplying the total valid bank guarantee amounts with trading limit parameter)';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.SECURITY_DEPOSITS is
'Total amount of security deposits included in the prudential run';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.EARLY_PAYMENTS is
'The sum of all applicable early payment amounts for the organisation';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.OUTSTANDING_AMOUNT is
'The total outstanding amounts for the organisation
(The organisations oustandings amount equals to Settlement exposure amount, plus reallocation amounts for gas days less than prudential run date where the reallocation amounts were not included in settlement run, minus Security deposits minus Early payments)';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.REALLOC_ADJ_AMT_TO_OUTSTANDING is
'The total reallocations amounts used as adjustment to outstandings in the prudential run, this include all reallocations amounts not included in settlement runs and are associated with gas days covered by the prudential run where the reallocation gas day is less than the prudential run date';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.SETTLEMENT_EXPOSURE_AMOUNT is
'The sum of all commodity and capacity settlement amounts owing up to the current date for the organisation, including estimates of non-invoiced amounts';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.TRADES_FORWARD_EXPOSURE_AMOUNT is
'The sum of all commodity and capacity trades forward exposure amounts allocated to the organisationinclusive of the trades forward exposure GST amount.';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.FORWARD_REALLOCATIONS_AMOUNT is
'The aggregated value of all debit and credit authorised reallocations which apply to the prudential forward exposure period for reallocation';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.FORWARD_EXPOSURE_AMOUNT is
'The sum of all forward transaction exposure amounts and all authorised reallocations amounts which apply to gas days in the prudential forward exposure period for reallocation for the organisation  inclusive of the trades forward exposure GST amount
(The forward exposure is the organisations total exposure amount attributable to all forward dates submitted/executed orders which the delivery dates are in the future, and all authorised reallocations amounts which apply to gas days in the prudential forward exposure period for reallocation)';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.PRUDENTIAL_EXPOSURE is
'The organisations prudential exposure is calculated by aggregating all associated participants outstanding amounts, settlement exposure, the organisations forward exposure and the forward exposure GST amount.';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.PERCENTAGE_EXPOSURE is
'The percentage of prudential exposure as compared to the trading limit
(organisations percentage exposure is calculated by dividing the organisation prudential exposure amount by the trading limit)';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.TRADING_MARGIN is
'The trading margin of the organisation
(organisations trading margin is calculated by subtracting the organisation prudential exposure amount from the trading limit)';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.BANK_GUARANTEE_RATIO is
'The ratio between the organisations total bank guarantees and the sum of the organisations total bank guarantees and the organisations total security deposits in this prudential run';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.EXPOSURE_NOTIFICATION is
'The field contains exposure notifications for organisation, it can have one of the following notifications or both depending on the conditions met as the following:
Trading Limit Breach: if the organisation has gone over the trading limit (trading margin value < 0)
Bank Guarantee Ratio Breach: If the bank guarantee ratio is deemed insufficient (refer to FRDG7603 for more details) 
If none of the above conditions are met then the field will contain no value';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.LASTCHANGED is
'The date and time the report was generated';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.DAA_SETTLEMENT_EXPOSURE_AMOUNT is
'The sum of all day ahead auction settlement amounts owing up to the current date for the organisation, including estimates of non-invoiced amounts';

comment on column GSH_PRD_EST_MARKET_EXPOSURE.DAA_FORWARD_EXPOSURE_AMOUNT is
'The sum of all day ahead auction bids forward exposure amounts allocated to the organisationinclusive of the day ahead auction bids forward exposure GST amount';

alter table GSH_PRD_EST_MARKET_EXPOSURE
   add constraint GSH_PRD_EST_MARKET_EXPOSURE_PK primary key (ORGANISATION_CODE, PRUDENTIAL_RUN_ID, PRUDENTIAL_DATETIME);

/*==============================================================*/
/* Table: GSH_PRD_REALLOCATIONS                                 */
/*==============================================================*/
create table GSH_PRD_REALLOCATIONS (
   ORGANISATION_CODE    VARCHAR(20)          not null,
   PRUDENTIAL_RUN_ID    NUMERIC(4,0)         not null,
   PRUDENTIAL_DATETIME  TIMESTAMP(0)         not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   REALLOCATION_ID      VARCHAR(20)          not null,
   ORGANISATION_NAME    VARCHAR(80)          null,
   REALLOCATION_ROLE    VARCHAR(80)          null,
   COUNTERPARTY_PARTICIPANT_CODE VARCHAR(20)          null,
   START_DATE           TIMESTAMP(0)         null,
   END_DATE             TIMESTAMP(0)         null,
   AGREEMENT_TYPE       VARCHAR(80)          null,
   REALLOC_ADJ_AMT_TO_OUTSTANDING NUMERIC(18,8)        null,
   FORWARD_REALLOCATION_AMOUNT NUMERIC(18,8)        null
);

comment on table GSH_PRD_REALLOCATIONS is
'The purpose of the Prudential Exposure report is to provide organisations with daily estimates of their market exposure. The report also contains details of their bank guarantees and security deposits as well as their outstanding amounts for all settlement runs included in the prudential run. The Reallocations subreport provides a breakdown of all reallocation amounts used in the prudential run at the participant level.';

comment on column GSH_PRD_REALLOCATIONS.ORGANISATION_CODE is
'The unique code for the organisation';

comment on column GSH_PRD_REALLOCATIONS.PRUDENTIAL_RUN_ID is
'The run number of the prudential run.';

comment on column GSH_PRD_REALLOCATIONS.PRUDENTIAL_DATETIME is
'The date and time of the prudential run';

comment on column GSH_PRD_REALLOCATIONS.PARTICIPANT_CODE is
'The participant code for the participant who lodged the security deposit';

comment on column GSH_PRD_REALLOCATIONS.REALLOCATION_ID is
'The unique identifier for the reallocation';

comment on column GSH_PRD_REALLOCATIONS.ORGANISATION_NAME is
'The name of the organisation';

comment on column GSH_PRD_REALLOCATIONS.REALLOCATION_ROLE is
'The role of the participant in the reallocation';

comment on column GSH_PRD_REALLOCATIONS.COUNTERPARTY_PARTICIPANT_CODE is
'The participant code for the counterparty of the reallocation';

comment on column GSH_PRD_REALLOCATIONS.START_DATE is
'The date on which the reallocation becomes into effect';

comment on column GSH_PRD_REALLOCATIONS.END_DATE is
'The date on which the reallocation ends';

comment on column GSH_PRD_REALLOCATIONS.AGREEMENT_TYPE is
'The agreement type of the reallocation';

comment on column GSH_PRD_REALLOCATIONS.REALLOC_ADJ_AMT_TO_OUTSTANDING is
'The total dollar amount of the reallocation, which is used in this prudential run as an adjustment to outstandings';

comment on column GSH_PRD_REALLOCATIONS.FORWARD_REALLOCATION_AMOUNT is
'The total dollar amount of the reallocation, which is used in this prudential run in calculating the forward exposure amount';

alter table GSH_PRD_REALLOCATIONS
   add constraint GSH_PRD_REALLOCATIONS_PK primary key (ORGANISATION_CODE, PRUDENTIAL_RUN_ID, PRUDENTIAL_DATETIME, PARTICIPANT_CODE, REALLOCATION_ID);

/*==============================================================*/
/* Table: GSH_PRD_SECURITY_DEPOSITS                             */
/*==============================================================*/
create table GSH_PRD_SECURITY_DEPOSITS (
   ORGANISATION_CODE    VARCHAR(20)          not null,
   PRUDENTIAL_RUN_ID    NUMERIC(4,0)         not null,
   PRUDENTIAL_DATETIME  TIMESTAMP(0)         not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   SECURITY_DEPOSIT_ID  VARCHAR(20)          not null,
   ORGANISATION_NAME    VARCHAR(80)          null,
   EFFECTIVE_FROM_DATE  TIMESTAMP(0)         null,
   MATURITY_DATE        TIMESTAMP(0)         null,
   TOTAL_SECURITY_DEPOSIT_AMOUNT NUMERIC(18,8)        null,
   NON_APPLIED_SEC_DEPOSIT_AMT NUMERIC(18,8)        null,
   INTEREST_ACCT_ID     VARCHAR(20)          null,
   INTEREST_CALC_TYPE   VARCHAR(20)          null
);

comment on table GSH_PRD_SECURITY_DEPOSITS is
'The purpose of the Prudential Exposure report is to provide organisations with daily estimates of their market exposure. The report also contains details of their bank guarantees and security deposits as well as their outstanding amounts for all settlement runs included in the prudential run. The Security Deposits subreport provides a list of all security deposits used in the prudential run (Note: only security deposits that are maturing on a day later than the prudential date time are included in this report).';

comment on column GSH_PRD_SECURITY_DEPOSITS.ORGANISATION_CODE is
'The unique code for the organisation';

comment on column GSH_PRD_SECURITY_DEPOSITS.PRUDENTIAL_RUN_ID is
'The run number of the prudential run.';

comment on column GSH_PRD_SECURITY_DEPOSITS.PRUDENTIAL_DATETIME is
'The date and time of the prudential run';

comment on column GSH_PRD_SECURITY_DEPOSITS.PARTICIPANT_CODE is
'The participant code for the participant who lodged the security deposit';

comment on column GSH_PRD_SECURITY_DEPOSITS.SECURITY_DEPOSIT_ID is
'The unique identifier for a security deposit which is included in the prudential run (Note: only security deposits that are maturing on a day later than the prudential date time are included in this report)';

comment on column GSH_PRD_SECURITY_DEPOSITS.ORGANISATION_NAME is
'The name of the organisation';

comment on column GSH_PRD_SECURITY_DEPOSITS.EFFECTIVE_FROM_DATE is
'The start date of the security deposit';

comment on column GSH_PRD_SECURITY_DEPOSITS.MATURITY_DATE is
'The maturity date of the security deposit';

comment on column GSH_PRD_SECURITY_DEPOSITS.TOTAL_SECURITY_DEPOSIT_AMOUNT is
'All cash being held and not yet paid back via settlement';

comment on column GSH_PRD_SECURITY_DEPOSITS.NON_APPLIED_SEC_DEPOSIT_AMT is
'All cash that is used to directly reduce outstandings (not in a settlement run)';

comment on column GSH_PRD_SECURITY_DEPOSITS.INTEREST_ACCT_ID is
'The Interest Account Id for calculating the Interest Payment. This is NULL if the INTEREST_CALC_TYPE = FIXED';

comment on column GSH_PRD_SECURITY_DEPOSITS.INTEREST_CALC_TYPE is
'Fixed or daily';

alter table GSH_PRD_SECURITY_DEPOSITS
   add constraint GSH_PRD_SECURITY_DEPOSITS_PK primary key (ORGANISATION_CODE, PRUDENTIAL_RUN_ID, PRUDENTIAL_DATETIME, PARTICIPANT_CODE, SECURITY_DEPOSIT_ID);

/*==============================================================*/
/* Table: GSH_PRD_SETTLEMENT_AMOUNTS                            */
/*==============================================================*/
create table GSH_PRD_SETTLEMENT_AMOUNTS (
   ORGANISATION_CODE    VARCHAR(20)          not null,
   PRUDENTIAL_RUN_ID    NUMERIC(4,0)         not null,
   PRUDENTIAL_DATETIME  TIMESTAMP(0)         not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   BILLING_PERIOD       VARCHAR(20)          not null,
   ORGANISATION_NAME    VARCHAR(80)          null,
   SETTLEMENT_TYPE      VARCHAR(20)          null,
   SETTLEMENT_VERSION_ID NUMERIC(18,0)        null,
   SETTLEMENT_EXPOSURE_AMOUNT NUMERIC(18,8)        null,
   PUBLICATION_DATE     TIMESTAMP(0)         null,
   PAYMENT_DUE_DATE     TIMESTAMP(0)         null
);

comment on table GSH_PRD_SETTLEMENT_AMOUNTS is
'The purpose of the Prudential Exposure report is to provide organisations with daily estimates of their market exposure. The report also contains details of their bank guarantees and security deposits as well as their outstanding amounts for all settlement runs included in the prudential run. The Settlement Amounts subreport provides a breakdown of all settlement amounts at the participant level for all settlement runs included in the prudential run';

comment on column GSH_PRD_SETTLEMENT_AMOUNTS.ORGANISATION_CODE is
'The unique code for the organisation';

comment on column GSH_PRD_SETTLEMENT_AMOUNTS.PRUDENTIAL_RUN_ID is
'The run number of the prudential run.';

comment on column GSH_PRD_SETTLEMENT_AMOUNTS.PRUDENTIAL_DATETIME is
'The date and time of the prudential run';

comment on column GSH_PRD_SETTLEMENT_AMOUNTS.PARTICIPANT_CODE is
'The participant code for the participant which the outstanding amount belongs to';

comment on column GSH_PRD_SETTLEMENT_AMOUNTS.BILLING_PERIOD is
'The billing period';

comment on column GSH_PRD_SETTLEMENT_AMOUNTS.ORGANISATION_NAME is
'The name of the organisation';

comment on column GSH_PRD_SETTLEMENT_AMOUNTS.SETTLEMENT_TYPE is
'The run type of the settlement run (i.e. Final, Revision, etc)';

comment on column GSH_PRD_SETTLEMENT_AMOUNTS.SETTLEMENT_VERSION_ID is
'The version number of the settlement run used in this billing period';

comment on column GSH_PRD_SETTLEMENT_AMOUNTS.SETTLEMENT_EXPOSURE_AMOUNT is
'The outstanding amount for the participant as per the settlement run';

comment on column GSH_PRD_SETTLEMENT_AMOUNTS.PUBLICATION_DATE is
'The publication date of the settlement';

comment on column GSH_PRD_SETTLEMENT_AMOUNTS.PAYMENT_DUE_DATE is
'The due date of the payment for the outstanding amount';

alter table GSH_PRD_SETTLEMENT_AMOUNTS
   add constraint GSH_PRD_SETTLEMENT_AMOUNTS_PK primary key (ORGANISATION_CODE, PRUDENTIAL_RUN_ID, PRUDENTIAL_DATETIME, PARTICIPANT_CODE, BILLING_PERIOD);

/*==============================================================*/
/* Table: GSH_PRD_TRADE_FWD_EXPOSURE                            */
/*==============================================================*/
create table GSH_PRD_TRADE_FWD_EXPOSURE (
   ORGANISATION_CODE    VARCHAR(20)          not null,
   PRUDENTIAL_RUN_ID    NUMERIC(4,0)         not null,
   PRUDENTIAL_DATETIME  TIMESTAMP(0)         not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   PRODUCT_LOCATION     VARCHAR(80)          not null,
   ORGANISATION_NAME    VARCHAR(80)          null,
   TRADES_FORWARD_EXP_AMT_NET_GST NUMERIC(18,8)        null,
   TRADES_FORWARD_EXP_GST_AMT NUMERIC(18,8)        null,
   BUY_ORDER_QUANTITY   NUMERIC(18,8)        null,
   BUY_TRADE_QUANTITY   NUMERIC(18,8)        null,
   SELL_ORDER_QUANTITY  NUMERIC(18,8)        null,
   SELL_TRADE_QUANTITY  NUMERIC(18,8)        null,
   BUY_LOAD_WEIGHTED_AVG_PRICE NUMERIC(18,8)        null,
   SELL_LOAD_WEIGHTED_AVG_PRICE NUMERIC(18,8)        null
);

comment on table GSH_PRD_TRADE_FWD_EXPOSURE is
'The purpose of the Prudential Exposure report is to provide organisations with daily estimates of their market exposure. The report also contains details of their bank guarantees and security deposits as well as their outstanding amounts for all settlement runs included in the prudential run. The Trade Forward Exposure subreport provides a summary of forward exposure amounts for the organisation by product location by gas day.';

comment on column GSH_PRD_TRADE_FWD_EXPOSURE.ORGANISATION_CODE is
'The unique code for the organisation';

comment on column GSH_PRD_TRADE_FWD_EXPOSURE.PRUDENTIAL_RUN_ID is
'The run number of the prudential run.';

comment on column GSH_PRD_TRADE_FWD_EXPOSURE.PRUDENTIAL_DATETIME is
'The date and time of the prudential run';

comment on column GSH_PRD_TRADE_FWD_EXPOSURE.GAS_DATE is
'The gas day';

comment on column GSH_PRD_TRADE_FWD_EXPOSURE.PRODUCT_LOCATION is
'The product location for the Gas Supply Hub market';

comment on column GSH_PRD_TRADE_FWD_EXPOSURE.ORGANISATION_NAME is
'The name of the organisation';

comment on column GSH_PRD_TRADE_FWD_EXPOSURE.TRADES_FORWARD_EXP_AMT_NET_GST is
'The sum of all trades forward exposure amounts allocated to the organisation on this gas day  exclusive of the trades forward exposure GST amount';

comment on column GSH_PRD_TRADE_FWD_EXPOSURE.TRADES_FORWARD_EXP_GST_AMT is
'The total amount of GST for the trades forward exposure';

comment on column GSH_PRD_TRADE_FWD_EXPOSURE.BUY_ORDER_QUANTITY is
'The total quantity for positively-valued orders';

comment on column GSH_PRD_TRADE_FWD_EXPOSURE.BUY_TRADE_QUANTITY is
'The total quantity for positively-valued trades';

comment on column GSH_PRD_TRADE_FWD_EXPOSURE.SELL_ORDER_QUANTITY is
'The total quantity for negatively-valued orders';

comment on column GSH_PRD_TRADE_FWD_EXPOSURE.SELL_TRADE_QUANTITY is
'The total quantity for negatively-valued trades';

comment on column GSH_PRD_TRADE_FWD_EXPOSURE.BUY_LOAD_WEIGHTED_AVG_PRICE is
'The load weighted average price for all positively-valued orders and trades';

comment on column GSH_PRD_TRADE_FWD_EXPOSURE.SELL_LOAD_WEIGHTED_AVG_PRICE is
'The load weighted average price for all negatively-valued trades';

alter table GSH_PRD_TRADE_FWD_EXPOSURE
   add constraint GSH_PRD_TRADE_FWD_EXPOSURE_PK primary key (ORGANISATION_CODE, PRUDENTIAL_RUN_ID, PRUDENTIAL_DATETIME, GAS_DATE, PRODUCT_LOCATION);

/*==============================================================*/
/* Table: GSH_RCPT_DELIV_POINT_PREFS                            */
/*==============================================================*/
create table GSH_RCPT_DELIV_POINT_PREFS (
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   PRODUCT_NAME         VARCHAR(80)          not null,
   BUY_SELL_POSITION    VARCHAR(4)           not null,
   STEP_NUMBER          NUMERIC(6)           not null,
   VERSION_DATETIME     TIMESTAMP(0)         not null,
   FROM_GAS_DATE        TIMESTAMP(0)         null,
   TO_GAS_DATE          TIMESTAMP(0)         null,
   OVERRIDE_FLAG        VARCHAR(3)           null,
   PRODUCT_RECEIPT_POINT VARCHAR(40)          null,
   RECEIPT_POINT_ID     NUMERIC(8,0)         null,
   RECEIPT_POINT_NAME   VARCHAR(255)         null,
   PRODUCT_DELIVERY_POINT VARCHAR(40)          null,
   DELIVERY_POINT_ID    NUMERIC(8,0)         null,
   DELIVERY_POINT_NAME  VARCHAR(255)         null,
   STEP_QUANTITY        NUMERIC(10,0)        null,
   LAST_UPDATED         TIMESTAMP(0)         null,
   LAST_UPDATED_BY      VARCHAR(20)          null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_RCPT_DELIV_POINT_PREFS is
'This report is available to participants as an audit trail of the entries in the CTP Receipt and Delivery Point Preferences markets portal application.';

comment on column GSH_RCPT_DELIV_POINT_PREFS.PARTICIPANT_CODE is
'The participant code used in the ETS.';

comment on column GSH_RCPT_DELIV_POINT_PREFS.PRODUCT_NAME is
'The capacity product name.';

comment on column GSH_RCPT_DELIV_POINT_PREFS.BUY_SELL_POSITION is
'Specifies the position that the contract reference must be used for, for example Buy, Sell or Both.';

comment on column GSH_RCPT_DELIV_POINT_PREFS.STEP_NUMBER is
'The step number for the corresponding capacity quantity.';

comment on column GSH_RCPT_DELIV_POINT_PREFS.VERSION_DATETIME is
'The date and time the preference was updated.';

comment on column GSH_RCPT_DELIV_POINT_PREFS.FROM_GAS_DATE is
'The first gas date that the contract reference takes effect from. Disregard the time component as this is not applicable.';

comment on column GSH_RCPT_DELIV_POINT_PREFS.TO_GAS_DATE is
'The last gas date that the contract reference takes effect until. Disregard the time component as this is not applicable.';

comment on column GSH_RCPT_DELIV_POINT_PREFS.OVERRIDE_FLAG is
'A flag to indicate whether the receipt and delivery points specified in the original order should override this preference, for example Yes or No.';

comment on column GSH_RCPT_DELIV_POINT_PREFS.PRODUCT_RECEIPT_POINT is
'The default Receipt Point for Spread Product';

comment on column GSH_RCPT_DELIV_POINT_PREFS.RECEIPT_POINT_ID is
'The unique identifier of the Receipt Point as defined in the Transportation Service Point Register. This is the point where gas is injected into the pipeline';

comment on column GSH_RCPT_DELIV_POINT_PREFS.RECEIPT_POINT_NAME is
'The descriptive name of the Receipt Point as defined in the Transportation Service Point Register.';

comment on column GSH_RCPT_DELIV_POINT_PREFS.PRODUCT_DELIVERY_POINT is
'The default Delivery Point for Spread Product';

comment on column GSH_RCPT_DELIV_POINT_PREFS.DELIVERY_POINT_ID is
'The unique identifier of the Delivery Point as defined in the Transportation Service Point Register. This is the point where gas is withdrawn from the pipeline. Mandatory for service points registered under Part 24 of the rules.';

comment on column GSH_RCPT_DELIV_POINT_PREFS.DELIVERY_POINT_NAME is
'The descriptive name of the Delivery Point as defined in the Transportation Service Point Register.';

comment on column GSH_RCPT_DELIV_POINT_PREFS.STEP_QUANTITY is
'The step quantity to applied to the contract reference (GJ/day).';

comment on column GSH_RCPT_DELIV_POINT_PREFS.LAST_UPDATED is
'The date and time the preference was updated.';

comment on column GSH_RCPT_DELIV_POINT_PREFS.LAST_UPDATED_BY is
'The user name who updated the preference.';

comment on column GSH_RCPT_DELIV_POINT_PREFS.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_RCPT_DELIV_POINT_PREFS
   add constraint GSH_RCPT_DELIV_POINT_PREFS_PK primary key (PARTICIPANT_CODE, PRODUCT_NAME, BUY_SELL_POSITION, STEP_NUMBER, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_REALLOC_DAILY_AMOUNT                              */
/*==============================================================*/
create table GSH_REALLOC_DAILY_AMOUNT (
   REALLOCATION_ID      VARCHAR(20)          not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   AMOUNT               NUMERIC(18,8)        null
);

comment on table GSH_REALLOC_DAILY_AMOUNT is
'The purpose of the Reallocation Confirmation report is to provide the participants (Credit and Debit Parties) a confirmation wherever an allocation is submitted, authorised, cancelled or expired. The Daily Amount subreport contains the reallocation daily amount for each gas day in the reallocation period.';

comment on column GSH_REALLOC_DAILY_AMOUNT.REALLOCATION_ID is
'The unique identifier for the reallocation';

comment on column GSH_REALLOC_DAILY_AMOUNT.GAS_DATE is
'The gas date in the reallocation period that the daily amount applies to';

comment on column GSH_REALLOC_DAILY_AMOUNT.AMOUNT is
'The daily reallocation amount which can be dollar or quantity GJ amount';

alter table GSH_REALLOC_DAILY_AMOUNT
   add constraint GSH_REALLOC_DAILY_AMOUNT_PK primary key (REALLOCATION_ID, GAS_DATE);

/*==============================================================*/
/* Table: GSH_REALLOC_REALLOCATION                              */
/*==============================================================*/
create table GSH_REALLOC_REALLOCATION (
   REALLOCATION_ID      VARCHAR(20)          not null,
   DEBIT_PARTICIPANT_CODE VARCHAR(20)          null,
   DEBIT_PARTICIPANT_NAME VARCHAR(80)          null,
   DEBIT_PARTICIPANT_REFERENCE VARCHAR(80)          null,
   CREDIT_PARTICIPANT_CODE VARCHAR(20)          null,
   CREDIT_PARTICIPANT_NAME VARCHAR(80)          null,
   CREDIT_PARTICIPANT_REFERENCE VARCHAR(80)          null,
   START_DATE           TIMESTAMP(0)         null,
   END_DATE             TIMESTAMP(0)         null,
   AGREEMENT_TYPE       VARCHAR(80)          null,
   PRODUCT_LOCATION     VARCHAR(80)          null,
   REALLOCATION_STATUS  VARCHAR(80)          null,
   REASON               VARCHAR(255)         null,
   LAST_UPDATED         TIMESTAMP(0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_REALLOC_REALLOCATION is
'The purpose of the Reallocation Confirmation report is to provide the participants (Credit and Debit Parties) a confirmation wherever an allocation is submitted, authorised, cancelled or expired. The Reallocation subreport contains the main reallocation details.';

comment on column GSH_REALLOC_REALLOCATION.REALLOCATION_ID is
'The unique identifier for the reallocation';

comment on column GSH_REALLOC_REALLOCATION.DEBIT_PARTICIPANT_CODE is
'The participant code for the participant on the debit side of the reallocation';

comment on column GSH_REALLOC_REALLOCATION.DEBIT_PARTICIPANT_NAME is
'The name for the participant on the debit side of the reallocation';

comment on column GSH_REALLOC_REALLOCATION.DEBIT_PARTICIPANT_REFERENCE is
'The reference for the reallocation for the participant on the debit side of the reallocation';

comment on column GSH_REALLOC_REALLOCATION.CREDIT_PARTICIPANT_CODE is
'The participant code for the participant on the credit side of the reallocation';

comment on column GSH_REALLOC_REALLOCATION.CREDIT_PARTICIPANT_NAME is
'The name for the participant on the credit side of the reallocation';

comment on column GSH_REALLOC_REALLOCATION.CREDIT_PARTICIPANT_REFERENCE is
'The reference for the reallocation for the participant on the credit side of the reallocation';

comment on column GSH_REALLOC_REALLOCATION.START_DATE is
'The date on which the reallocation comes into effect';

comment on column GSH_REALLOC_REALLOCATION.END_DATE is
'The date on which the reallocation ends';

comment on column GSH_REALLOC_REALLOCATION.AGREEMENT_TYPE is
'The type of the reallocation agreement';

comment on column GSH_REALLOC_REALLOCATION.PRODUCT_LOCATION is
'The product location used to determine the reference price (only applicable to Fixed Quantity Amount reallocation agreement type)';

comment on column GSH_REALLOC_REALLOCATION.REALLOCATION_STATUS is
'The status of the agreement which can be submitted, authorised, cancelled, or expired';

comment on column GSH_REALLOC_REALLOCATION.REASON is
'The reason explaining why a reallocation has been cancelled or expired (only applies to Cancellation and Expiration of report)
(Note: for details on what  the REASON field value should be refer to SUC7704-Cancel Reallocation in the Reallocations System Requirements Document)';

comment on column GSH_REALLOC_REALLOCATION.LAST_UPDATED is
'The date and time the reallocation was last updated i.e. saved into database';

comment on column GSH_REALLOC_REALLOCATION.LASTCHANGED is
'The date and time the report was generated';

alter table GSH_REALLOC_REALLOCATION
   add constraint GSH_RALLOC_REALLOCATION_PK primary key (REALLOCATION_ID);

/*==============================================================*/
/* Table: GSH_REVISED_AUCT_QUANTITIES                           */
/*==============================================================*/
create table GSH_REVISED_AUCT_QUANTITIES (
   AUCTION_DATE         TIMESTAMP(0)         not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   FACILITY_ID          NUMERIC(8,0)         not null,
   FACILITY_NAME        VARCHAR(100)         null,
   SERVICE_TYPE         VARCHAR(20)          not null,
   AGG_CLEARED_QUANTITY NUMERIC(10,0)        null,
   AGG_REVISED_QUANTITY NUMERIC(10,0)        null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_REVISED_AUCT_QUANTITIES is
'The report lists the following items for each facility and service type:initial cleared quantity as determined by the Auction Solver;where there has been a curtailment of the initial cleared quantity, the revised auction quantity which must be the aggregate minimum quantity made available to auction participants on the gas day; final aggregated scheduled nominations.';

comment on column GSH_REVISED_AUCT_QUANTITIES.AUCTION_DATE is
'The date of the auction run. Disregard the time component as this is not applicable.';

comment on column GSH_REVISED_AUCT_QUANTITIES.GAS_DATE is
'The effective gas date of the capacity won at auction Disregard the time component as this is not applicable.';

comment on column GSH_REVISED_AUCT_QUANTITIES.FACILITY_ID is
'The unique identifier of the facility.';

comment on column GSH_REVISED_AUCT_QUANTITIES.FACILITY_NAME is
'The descriptive name of the facility.';

comment on column GSH_REVISED_AUCT_QUANTITIES.SERVICE_TYPE is
'The type of service, for example Forward Haul, Backhaul or Compression.';

comment on column GSH_REVISED_AUCT_QUANTITIES.AGG_CLEARED_QUANTITY is
'The aggregated initial cleared capacity quantity as determined by the auction (GJ/day).';

comment on column GSH_REVISED_AUCT_QUANTITIES.AGG_REVISED_QUANTITY is
'The aggregated revised auction capacity quantity (GJ/day).';

comment on column GSH_REVISED_AUCT_QUANTITIES.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_REVISED_AUCT_QUANTITIES
   add constraint GSH_REVISED_AUCT_QUANTITIES_PK primary key (AUCTION_DATE, GAS_DATE, FACILITY_ID, SERVICE_TYPE);

/*==============================================================*/
/* Table: GSH_SECDEPOSIT_INTEREST_RATE                          */
/*==============================================================*/
create table GSH_SECDEPOSIT_INTEREST_RATE (
   INTEREST_ACCT_ID     VARCHAR(20)          not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSION_DATETIME     TIMESTAMP(0)         not null,
   INTEREST_RATE        NUMERIC(18,8)        null
);

comment on table GSH_SECDEPOSIT_INTEREST_RATE is
'The Security Deposit Interest Rate On Daily Basis. This is the public table published when a new floating daily interest rate is entered and authorised';

comment on column GSH_SECDEPOSIT_INTEREST_RATE.INTEREST_ACCT_ID is
'The Interest Account Id for calculating the Interest Payment. This is NULL if the INTEREST_CALC_TYPE = FIXED';

comment on column GSH_SECDEPOSIT_INTEREST_RATE.EFFECTIVEDATE is
'The Effective Date of the Interest Rate Change';

comment on column GSH_SECDEPOSIT_INTEREST_RATE.VERSION_DATETIME is
'The Version Date time of the Effective Date for the Interest Rate Change';

comment on column GSH_SECDEPOSIT_INTEREST_RATE.INTEREST_RATE is
'The Interest Rate for the Interest Account ID as on the Effective Date';

alter table GSH_SECDEPOSIT_INTEREST_RATE
   add constraint GSH_SECDEP_INTEREST_RATE_PK primary key (INTEREST_ACCT_ID, EFFECTIVEDATE, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_SERVICE_POINTS                                    */
/*==============================================================*/
create table GSH_SERVICE_POINTS (
   FACILITY_ID          NUMERIC(8,0)         not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSION_DATETIME     TIMESTAMP(0)         not null,
   SERVICE_POINT_ID     NUMERIC(8,0)         not null,
   FACILITY_NAME        VARCHAR(100)         null,
   SERVICE_POINT_NAME   VARCHAR(100)         null,
   SERVICE_POINT_TYPE   VARCHAR(100)         not null,
   SERVICE_POINT_DESCRIPTION VARCHAR(255)         null,
   AUCTION_PRODUCT_TYPE VARCHAR(20)          not null,
   NOTIONAL_POINT       VARCHAR(3)           null,
   ZONE_ID              NUMERIC(8,0)         null,
   ZONE_NAME            VARCHAR(100)         null,
   GBB_CONNECTION_POINT_ID NUMERIC(8,0)         null,
   GBB_CONNECTION_POINT_NAME VARCHAR(100)         null,
   FOR_AUCTION_PUBLICATION VARCHAR(3)           null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_SERVICE_POINTS is
'Service Points currently registered under Part 24 of the Rules for the capacity trading and day ahead auction.';

comment on column GSH_SERVICE_POINTS.FACILITY_ID is
'The unique identifier of the parent facility.';

comment on column GSH_SERVICE_POINTS.EFFECTIVEDATE is
'The effective date of the service point.';

comment on column GSH_SERVICE_POINTS.VERSION_DATETIME is
'The version timestamp of the record';

comment on column GSH_SERVICE_POINTS.SERVICE_POINT_ID is
'The unique identifier of the service point.';

comment on column GSH_SERVICE_POINTS.FACILITY_NAME is
'The descriptive name of the parent facility.';

comment on column GSH_SERVICE_POINTS.SERVICE_POINT_NAME is
'The descriptive name of the service point.';

comment on column GSH_SERVICE_POINTS.SERVICE_POINT_TYPE is
'The type of service point, for example forward haul receipt point, forward haul delivery point, backhaul receipt point, backhaul delivery point, compression receipt point or compression delivery point.';

comment on column GSH_SERVICE_POINTS.SERVICE_POINT_DESCRIPTION is
'The description of the service point.';

comment on column GSH_SERVICE_POINTS.AUCTION_PRODUCT_TYPE is
'Auction Product Type for Service Point';

comment on column GSH_SERVICE_POINTS.NOTIONAL_POINT is
'Flag to indicate if the service point is a notional point, for example Yes or No.';

comment on column GSH_SERVICE_POINTS.ZONE_ID is
'The unique identifier of the zone in which the service point is located.';

comment on column GSH_SERVICE_POINTS.ZONE_NAME is
'The descriptive name of the zone.';

comment on column GSH_SERVICE_POINTS.GBB_CONNECTION_POINT_ID is
'The corresponding GBB connection point identifier.';

comment on column GSH_SERVICE_POINTS.GBB_CONNECTION_POINT_NAME is
'The descriptive name of the corresponding GBB connection point.';

comment on column GSH_SERVICE_POINTS.FOR_AUCTION_PUBLICATION is
'Flag to indicate if the product component can be published in the public auction reports, for example yes or no.';

comment on column GSH_SERVICE_POINTS.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_SERVICE_POINTS
   add constraint GSH_SERVICE_POINTS_PK primary key (FACILITY_ID, EFFECTIVEDATE, VERSION_DATETIME, SERVICE_POINT_ID, SERVICE_POINT_TYPE, AUCTION_PRODUCT_TYPE);

/*==============================================================*/
/* Table: GSH_SHIPPER_CAP_XFR_NOTICE                            */
/*==============================================================*/
create table GSH_SHIPPER_CAP_XFR_NOTICE (
   CAPACITY_TRANSFER_ID VARCHAR(20)          not null,
   GAS_DATE             TIMESTAMP(0)         null,
   PARTICIPANT_CODE     VARCHAR(20)          null,
   PARTICIPANT_NAME     VARCHAR(80)          null,
   FACILITY_ID          NUMERIC(8,0)         null,
   FACILITY_NAME        VARCHAR(100)         null,
   SERVICE_REFERENCE    VARCHAR(255)         null,
   STTM_CONTRACT_REFERENCE VARCHAR(255)         null,
   DWGM_REFERENCE       VARCHAR(255)         null,
   RECEIPT_POINT_ID     NUMERIC(8,0)         null,
   RECEIPT_POINT_NAME   VARCHAR(255)         null,
   DELIVERY_POINT_ID    NUMERIC(8,0)         null,
   DELIVERY_POINT_NAME  VARCHAR(255)         null,
   RECEIPT_ZONE_ID      NUMERIC(8,0)         null,
   RECEIPT_ZONE_NAME    VARCHAR(255)         null,
   DELIVERY_ZONE_ID     NUMERIC(8,0)         null,
   DELIVERY_ZONE_NAME   VARCHAR(255)         null,
   STATUS               VARCHAR(20)          null,
   SYSTEM               VARCHAR(20)          null,
   REASON               VARCHAR(40)          null,
   TRANSACTION_COMPLETED_FLAG VARCHAR(3)           null,
   OVERALL_QUANTITY_CHANGE NUMERIC(10,0)        null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_SHIPPER_CAP_XFR_NOTICE is
'Provides the status of a trading participants capacity transfer related to one or more capacity trades. This notice is issued each time a capacity transfer record has a status change because of capacity transfer in the facility operator system or AEMO system. For non-STTM and non-DWGM capacity products, the status is reported as received from the facility operator. For STTM and DWGM capacity products, the status is reported once the capacity transfer has been completed in the relevant market system (STTM or DWGM).';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.CAPACITY_TRANSFER_ID is
'The unique AEMO defined Capacity Transfer transaction identifier.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.GAS_DATE is
'The gas date for the capacity change. Disregard the time component as this is not applicable.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.PARTICIPANT_CODE is
'The participant code used in the ETS.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.PARTICIPANT_NAME is
'The descriptive name of the participant.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.FACILITY_ID is
'The unique facility identifier as defined in the Transportation Facility Register.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.FACILITY_NAME is
'The descriptive name of the facility as defined in the Transportation Facility Register.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.SERVICE_REFERENCE is
'The service reference issued by the facility operator.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.STTM_CONTRACT_REFERENCE is
'The STTM registered facility service identifier (CRN) associated with the service reference.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.DWGM_REFERENCE is
'The DWGM accreditation reference associated with the transfer.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.RECEIPT_POINT_ID is
'The unique identifier of the Receipt Point as defined in the Transportation Service Point Register. This is the point where gas is injected into the pipeline';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.RECEIPT_POINT_NAME is
'The descriptive name of the Receipt Point as defined in the Transportation Service Point Register.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.DELIVERY_POINT_ID is
'The unique identifier of the Delivery Point as defined in the Transportation Service Point Register. This is the point where gas is withdrawn from the pipeline.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.DELIVERY_POINT_NAME is
'The descriptive name of the Delivery Point as defined in the Transportation Service Point Register.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.RECEIPT_ZONE_ID is
'The unique identifier of the Receipt Zone as defined in the Transportation Service Point Register.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.RECEIPT_ZONE_NAME is
'The descriptive name of the Receipt Zone as defined in the Transportation Service Point Register.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.DELIVERY_ZONE_ID is
'The unique identifier of the Delivery Zone as defined in the Transportation Service Point Register.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.DELIVERY_ZONE_NAME is
'The descriptive name of the Delivery Zone as defined in the Transportation Service Point Register.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.STATUS is
'The status of the capacity transfer, for example Transferred, Partial, Pending, Rejected or Cancelled.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.SYSTEM is
'The system that has processed the capacity transfer, for example Facility operator, AEMO, AEMO STTM or AEMO DWGM.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.REASON is
'The reason for a rejected or cancelled capacity transfer, for example Buyer delivery contract, Seller delivery fault, system issue or matching record not found.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.TRANSACTION_COMPLETED_FLAG is
'Flag to indicate whether the capacity transfer process is completed, for example yes or no.';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.OVERALL_QUANTITY_CHANGE is
'The overall capacity quantity adjustment (GJ/day) applied to the relevant contracts for this transfer date and gas date. This is a signed value (+/-).';

comment on column GSH_SHIPPER_CAP_XFR_NOTICE.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_SHIPPER_CAP_XFR_NOTICE
   add constraint GSH_SHIPPER_CAP_XFR_NOTICE_PK primary key (CAPACITY_TRANSFER_ID);

/*==============================================================*/
/* Table: GSH_SUPP_ADHOC                                        */
/*==============================================================*/
create table GSH_SUPP_ADHOC (
   SETTLEMENT_RUN_ID    NUMERIC(18,0)        not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   TYPE                 VARCHAR(20)          not null,
   BILLING_PERIOD       VARCHAR(20)          null,
   AMOUNT               NUMERIC(18,8)        null,
   GST_AMOUNT           NUMERIC(18,8)        null,
   ADHOC_DESCRIPTION    VARCHAR(255)         null
);

comment on table GSH_SUPP_ADHOC is
'The Settlements Supporting Data Report is generated when Final and Revision settlement statements are issued; it is also generated daily with details of the latest initial settlement run of the day. The report contains settlements summary information as well as participant specific data from that settlement run for the relevant charges and payments per gas day. It provides a breakdown of all executed trades and delivered quantities records included in this settlement run to assist participant in their settlement statements reconciliation. The Adhoc subreport provides a list of all individual ad-hoc payments and charges included in the settlement run.';

comment on column GSH_SUPP_ADHOC.SETTLEMENT_RUN_ID is
'The unique identifier for the settlement run';

comment on column GSH_SUPP_ADHOC.PARTICIPANT_CODE is
'The participant code';

comment on column GSH_SUPP_ADHOC.GAS_DATE is
'The gas date for the ad-hoc payment or charge';

comment on column GSH_SUPP_ADHOC.TYPE is
'Type of the ad-hoc amount whether it is a charge or a payment';

comment on column GSH_SUPP_ADHOC.BILLING_PERIOD is
'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run';

comment on column GSH_SUPP_ADHOC.AMOUNT is
'Total amount of ad-hoc charge or payment for the gas date excluding GST';

comment on column GSH_SUPP_ADHOC.GST_AMOUNT is
'Total amount of GST on the ad-hoc payment or charge for the gas date';

comment on column GSH_SUPP_ADHOC.ADHOC_DESCRIPTION is
'The description of the adhoc payment';

alter table GSH_SUPP_ADHOC
   add constraint GSH_SUPP_ADHOC_PK primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, GAS_DATE, TYPE);

/*==============================================================*/
/* Table: GSH_SUPP_AUCTION_QUANTITIES                           */
/*==============================================================*/
create table GSH_SUPP_AUCTION_QUANTITIES (
   MARKET_ID            VARCHAR(20)          null,
   SETTLEMENT_RUN_ID    NUMERIC(18,0)        not null,
   BILLING_PERIOD       VARCHAR(20)          null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   GAS_DATE             TIMESTAMP(0)         null,
   AUCTION_TRANSACTION_ID NUMERIC(12,0)        not null,
   AUCTION_QUANTITY     NUMERIC(10,0)        null,
   REVISED_AUCTION_QUANTITY NUMERIC(10,0)        null,
   PRODUCT_PRICE        NUMERIC(18,8)        null
);

comment on table GSH_SUPP_AUCTION_QUANTITIES is
'Details of Auction Quantities won used in Settlements';

comment on column GSH_SUPP_AUCTION_QUANTITIES.MARKET_ID is
'Unique Market ID';

comment on column GSH_SUPP_AUCTION_QUANTITIES.SETTLEMENT_RUN_ID is
'The unique identifier for the settlement run.';

comment on column GSH_SUPP_AUCTION_QUANTITIES.BILLING_PERIOD is
'The billing period for the settlement run.';

comment on column GSH_SUPP_AUCTION_QUANTITIES.PARTICIPANT_CODE is
'The participant code.';

comment on column GSH_SUPP_AUCTION_QUANTITIES.GAS_DATE is
'The gas date when the quantity is delivered. Disregard the time component as this is not applicable.';

comment on column GSH_SUPP_AUCTION_QUANTITIES.AUCTION_TRANSACTION_ID is
'The unique identifier of the auction transaction included in the settlement run.';

comment on column GSH_SUPP_AUCTION_QUANTITIES.AUCTION_QUANTITY is
'The initial capacity quantity won at auction.';

comment on column GSH_SUPP_AUCTION_QUANTITIES.REVISED_AUCTION_QUANTITY is
'The actual auction quantity in GJ. This is based on the revised quantity reported by facility operators on D+1.';

comment on column GSH_SUPP_AUCTION_QUANTITIES.PRODUCT_PRICE is
'The price of the auction product won at auction ($/GJ) by the participant.';

alter table GSH_SUPP_AUCTION_QUANTITIES
   add constraint GSH_SUPP_AUCTION_QUANTITIES_PK primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, AUCTION_TRANSACTION_ID);

/*==============================================================*/
/* Table: GSH_SUPP_CAPACITY_AUCTION                             */
/*==============================================================*/
create table GSH_SUPP_CAPACITY_AUCTION (
   MARKET_ID            VARCHAR(20)          null,
   SETTLEMENT_RUN_ID    NUMERIC(18,0)        not null,
   BILLING_PERIOD       VARCHAR(20)          null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   TYPE                 VARCHAR(20)          not null,
   AMOUNT               NUMERIC(18,8)        null,
   GST_AMOUNT           NUMERIC(18,8)        null
);

comment on table GSH_SUPP_CAPACITY_AUCTION is
'Includes Auction items won that are included in the settlement run';

comment on column GSH_SUPP_CAPACITY_AUCTION.MARKET_ID is
'Unique Market ID';

comment on column GSH_SUPP_CAPACITY_AUCTION.SETTLEMENT_RUN_ID is
'The unique identifier for the settlement run.';

comment on column GSH_SUPP_CAPACITY_AUCTION.BILLING_PERIOD is
'The billing period for the settlement run.';

comment on column GSH_SUPP_CAPACITY_AUCTION.PARTICIPANT_CODE is
'The participant code.';

comment on column GSH_SUPP_CAPACITY_AUCTION.GAS_DATE is
'The gas date for the capacity auction payment or charge. Disregard the time component as this is not applicable.';

comment on column GSH_SUPP_CAPACITY_AUCTION.TYPE is
'Type of the capacity auction amount whether it is a charge or a payment.';

comment on column GSH_SUPP_CAPACITY_AUCTION.AMOUNT is
'Total amount of trading charge or payment for the gas date excluding GST.';

comment on column GSH_SUPP_CAPACITY_AUCTION.GST_AMOUNT is
'Total amount of GST on the capacity auction payment or charge for the gas date.';

alter table GSH_SUPP_CAPACITY_AUCTION
   add constraint GSH_SUPP_CAPACITY_AUCTION_PK primary key (SETTLEMENT_RUN_ID, GAS_DATE, PARTICIPANT_CODE, TYPE);

/*==============================================================*/
/* Table: GSH_SUPP_CAPACITY_TRANSFERS                           */
/*==============================================================*/
create table GSH_SUPP_CAPACITY_TRANSFERS (
   MARKET_ID            VARCHAR(20)          null,
   SETTLEMENT_RUN_ID    NUMERIC(18,0)        not null,
   BILLING_PERIOD       VARCHAR(20)          null,
   PARTICIPANT_CODE     VARCHAR(20)          null,
   GAS_DATE             TIMESTAMP(0)         null,
   CAPACITY_TRANSFER_ID VARCHAR(20)          not null,
   TRADE_SIDE           VARCHAR(20)          null,
   CAPACITY_QUANTITY    NUMERIC(18,8)        null,
   CONFIRMED_CAPACITY_QUANTITY NUMERIC(18,8)        null,
   INVALID_QUANTITY     NUMERIC(18,8)        null
);

comment on table GSH_SUPP_CAPACITY_TRANSFERS is
'Provides a list of all individual capacity transfer records included in the settlement run.';

comment on column GSH_SUPP_CAPACITY_TRANSFERS.MARKET_ID is
'Unique Market ID';

comment on column GSH_SUPP_CAPACITY_TRANSFERS.SETTLEMENT_RUN_ID is
'The unique identifier for the settlement run.';

comment on column GSH_SUPP_CAPACITY_TRANSFERS.BILLING_PERIOD is
'The billing period for the settlement run.';

comment on column GSH_SUPP_CAPACITY_TRANSFERS.PARTICIPANT_CODE is
'The participant code.';

comment on column GSH_SUPP_CAPACITY_TRANSFERS.GAS_DATE is
'The gas date for the capacity variance payment or charge. Disregard the time component as this is not applicable.';

comment on column GSH_SUPP_CAPACITY_TRANSFERS.CAPACITY_TRANSFER_ID is
'The unique identifier of the capacity transfer record included in the settlement run.';

comment on column GSH_SUPP_CAPACITY_TRANSFERS.TRADE_SIDE is
'This field specified whether the participant was the buyer or the seller in the trade.';

comment on column GSH_SUPP_CAPACITY_TRANSFERS.CAPACITY_QUANTITY is
'The initial capacity quantity transfer in GJ';

comment on column GSH_SUPP_CAPACITY_TRANSFERS.CONFIRMED_CAPACITY_QUANTITY is
'The confirmed capacity quantity in GJ.';

comment on column GSH_SUPP_CAPACITY_TRANSFERS.INVALID_QUANTITY is
'Invalid quantity to indicate the amount of a short sell or amount that a buyer could not accept due to incorrect contract.';

alter table GSH_SUPP_CAPACITY_TRANSFERS
   add constraint GSH_SUPP_CAPACITY_TRANSFERS_PK primary key (SETTLEMENT_RUN_ID, CAPACITY_TRANSFER_ID);

/*==============================================================*/
/* Table: GSH_SUPP_CAPACITY_VARIANCE                            */
/*==============================================================*/
create table GSH_SUPP_CAPACITY_VARIANCE (
   MARKET_ID            VARCHAR(20)          null,
   SETTLEMENT_RUN_ID    NUMERIC(18,0)        not null,
   BILLING_PERIOD       VARCHAR(20)          null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   TYPE                 VARCHAR(20)          not null,
   AMOUNT               NUMERIC(18,8)        null,
   GST_AMOUNT           NUMERIC(18,8)        null
);

comment on table GSH_SUPP_CAPACITY_VARIANCE is
'The Delivery Variance subreport provides a breakdown of total capacity variance payments and charges included in the settlement run per gas day, note that total payments and total charges are listed separately for each gas day';

comment on column GSH_SUPP_CAPACITY_VARIANCE.MARKET_ID is
'Unique Market ID';

comment on column GSH_SUPP_CAPACITY_VARIANCE.SETTLEMENT_RUN_ID is
'The unique identifier for the settlement run.';

comment on column GSH_SUPP_CAPACITY_VARIANCE.BILLING_PERIOD is
'The billing period for the settlement run.';

comment on column GSH_SUPP_CAPACITY_VARIANCE.PARTICIPANT_CODE is
'The participant code.';

comment on column GSH_SUPP_CAPACITY_VARIANCE.GAS_DATE is
'The gas date for the capacity variance payment or charge. Disregard the time component as this is not applicable.';

comment on column GSH_SUPP_CAPACITY_VARIANCE.TYPE is
'Type of the capacity variance amount whether it is a charge or a payment.';

comment on column GSH_SUPP_CAPACITY_VARIANCE.AMOUNT is
'Total amount of trading charge or payment for the gas date excluding GST.';

comment on column GSH_SUPP_CAPACITY_VARIANCE.GST_AMOUNT is
'Total amount of GST on the capacity variance payment or charge for the gas date.';

alter table GSH_SUPP_CAPACITY_VARIANCE
   add constraint GSH_SUPP_CAPACITY_VARIANCE_PK primary key (SETTLEMENT_RUN_ID, GAS_DATE, PARTICIPANT_CODE, TYPE);

/*==============================================================*/
/* Table: GSH_SUPP_DELIVERED_QUANTITY                           */
/*==============================================================*/
create table GSH_SUPP_DELIVERED_QUANTITY (
   SETTLEMENT_RUN_ID    NUMERIC(18,0)        not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   TRADE_ID             VARCHAR(20)          not null,
   BILLING_PERIOD       VARCHAR(20)          null,
   TRADE_SIDE           VARCHAR(20)          null,
   TRADE_QUANTITY       NUMERIC(18,8)        null,
   DELIVERED_QUANTITY   NUMERIC(18,8)        null
);

comment on table GSH_SUPP_DELIVERED_QUANTITY is
'The Settlements Supporting Data Report is generated when Final and Revision settlement statements are issued; it is also generated daily with details of the latest initial settlement run of the day. The report contains settlements summary information as well as participant specific data from that settlement run for the relevant charges and payments per gas day. It provides a breakdown of all executed trades and delivered quantities records included in this settlement run to assist participant in their settlement statements reconciliation. The Delivered Quantity subreport provides a list of all individual delivered quantity records included in the settlement run.';

comment on column GSH_SUPP_DELIVERED_QUANTITY.SETTLEMENT_RUN_ID is
'The unique identifier for the settlement run';

comment on column GSH_SUPP_DELIVERED_QUANTITY.PARTICIPANT_CODE is
'The participant code';

comment on column GSH_SUPP_DELIVERED_QUANTITY.GAS_DATE is
'The gas date that the quantity is delivered';

comment on column GSH_SUPP_DELIVERED_QUANTITY.TRADE_ID is
'The unique identifier for the executed trade included in the settlement run';

comment on column GSH_SUPP_DELIVERED_QUANTITY.BILLING_PERIOD is
'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run';

comment on column GSH_SUPP_DELIVERED_QUANTITY.TRADE_SIDE is
'This field specified whether the participant was the buyer or the seller in the trade';

comment on column GSH_SUPP_DELIVERED_QUANTITY.TRADE_QUANTITY is
'The initial quantity of the executed trade';

comment on column GSH_SUPP_DELIVERED_QUANTITY.DELIVERED_QUANTITY is
'The delivered quantity in GJ';

alter table GSH_SUPP_DELIVERED_QUANTITY
   add constraint GSH_SUPP_DELIVERED_QUANTITY_PK primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, GAS_DATE, TRADE_ID);

/*==============================================================*/
/* Table: GSH_SUPP_DELIVERY_VARIANCE                            */
/*==============================================================*/
create table GSH_SUPP_DELIVERY_VARIANCE (
   SETTLEMENT_RUN_ID    NUMERIC(18,0)        not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   TYPE                 VARCHAR(20)          not null,
   BILLING_PERIOD       VARCHAR(20)          null,
   AMOUNT               NUMERIC(18,8)        null,
   GST_AMOUNT           NUMERIC(18,8)        null
);

comment on table GSH_SUPP_DELIVERY_VARIANCE is
'The Settlements Supporting Data Report is generated when Final and Revision settlement statements are issued; it is also generated daily with details of the latest initial settlement run of the day. The report contains settlements summary information as well as participant specific data from that settlement run for the relevant charges and payments per gas day. It provides a breakdown of all executed trades and delivered quantities records included in this settlement run to assist participant in their settlement statements reconciliation. The Delivery Variance subreport provides a breakdown of total delivery variance payments and charges included in the settlement run per gas day, note that total payments and total charges are listed separately for each gas day.';

comment on column GSH_SUPP_DELIVERY_VARIANCE.SETTLEMENT_RUN_ID is
'The unique identifier for the settlement run';

comment on column GSH_SUPP_DELIVERY_VARIANCE.PARTICIPANT_CODE is
'The participant code';

comment on column GSH_SUPP_DELIVERY_VARIANCE.GAS_DATE is
'The gas date for the delivery variance payment or charge';

comment on column GSH_SUPP_DELIVERY_VARIANCE.TYPE is
'Type of the delivery variance amount whether it is a charge or a payment';

comment on column GSH_SUPP_DELIVERY_VARIANCE.BILLING_PERIOD is
'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run';

comment on column GSH_SUPP_DELIVERY_VARIANCE.AMOUNT is
'Total amount of delivery variance charge or payment for the gas date excluding GST';

comment on column GSH_SUPP_DELIVERY_VARIANCE.GST_AMOUNT is
'Total amount of GST on the delivery variance payment or charge for the gas date';

alter table GSH_SUPP_DELIVERY_VARIANCE
   add constraint GSH_SUPP_DELIVERY_VARIANCE_PK primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, GAS_DATE, TYPE);

/*==============================================================*/
/* Table: GSH_SUPP_EXECUTED_TRADES                              */
/*==============================================================*/
create table GSH_SUPP_EXECUTED_TRADES (
   SETTLEMENT_RUN_ID    NUMERIC(18,0)        not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   TRADE_ID             VARCHAR(20)          not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   BILLING_PERIOD       VARCHAR(20)          null,
   TRADE_SIDE           VARCHAR(20)          null,
   TRADE_QUANTITY       NUMERIC(18,8)        null,
   TRADE_PRICE          NUMERIC(18,8)        null
);

comment on table GSH_SUPP_EXECUTED_TRADES is
'The Settlements Supporting Data Report is generated when Final and Revision settlement statements are issued; it is also generated daily with details of the latest initial settlement run of the day. The report contains settlements summary information as well as participant specific data from that settlement run for the relevant charges and payments per gas day. It provides a breakdown of all executed trades and delivered quantities records included in this settlement run to assist participant in their settlement statements reconciliation. The Executed Trades subreport provides a list of all individual executed trades records included in the settlement run.';

comment on column GSH_SUPP_EXECUTED_TRADES.SETTLEMENT_RUN_ID is
'The unique identifier for the settlement run';

comment on column GSH_SUPP_EXECUTED_TRADES.PARTICIPANT_CODE is
'The participant code';

comment on column GSH_SUPP_EXECUTED_TRADES.TRADE_ID is
'The unique identifier for the executed trade included in the settlement run';

comment on column GSH_SUPP_EXECUTED_TRADES.GAS_DATE is
'The gas date for the trade was executed on';

comment on column GSH_SUPP_EXECUTED_TRADES.BILLING_PERIOD is
'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run';

comment on column GSH_SUPP_EXECUTED_TRADES.TRADE_SIDE is
'This field specified whether the participant was the buyer or the seller in the trade';

comment on column GSH_SUPP_EXECUTED_TRADES.TRADE_QUANTITY is
'The quantity of the executed trade';

comment on column GSH_SUPP_EXECUTED_TRADES.TRADE_PRICE is
'The price of the executed trade';

alter table GSH_SUPP_EXECUTED_TRADES
   add constraint GSH_SUPP_EXECUTED_TRADES_PK primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, TRADE_ID, GAS_DATE);

/*==============================================================*/
/* Table: GSH_SUPP_FEE                                          */
/*==============================================================*/
create table GSH_SUPP_FEE (
   SETTLEMENT_RUN_ID    NUMERIC(18,0)        not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   FEE_TYPE             VARCHAR(20)          not null,
   BILLING_PERIOD       VARCHAR(20)          null,
   AMOUNT               NUMERIC(18,8)        null,
   GST_AMOUNT           NUMERIC(18,8)        null
);

comment on table GSH_SUPP_FEE is
'The Settlements Supporting Data Report is generated when Final and Revision settlement statements are issued; it is also generated daily with details of the latest initial settlement run of the day. The report contains settlements summary information as well as participant specific data from that settlement run for the relevant charges and payments per gas day. It provides a breakdown of all executed trades and delivered quantities records included in this settlement run to assist participant in their settlement statements reconciliation. The Fee subreport provides a list of all market and transaction fees included in the prudential run against the gas dates.';

comment on column GSH_SUPP_FEE.SETTLEMENT_RUN_ID is
'The unique identifier for the settlement run';

comment on column GSH_SUPP_FEE.PARTICIPANT_CODE is
'The participant code';

comment on column GSH_SUPP_FEE.GAS_DATE is
'The gas date for the fee';

comment on column GSH_SUPP_FEE.FEE_TYPE is
'The type of fee whether it is a market fee or a transaction fee';

comment on column GSH_SUPP_FEE.BILLING_PERIOD is
'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run';

comment on column GSH_SUPP_FEE.AMOUNT is
'The fee amount excluding GST';

comment on column GSH_SUPP_FEE.GST_AMOUNT is
'Total amount of GST on the fee for the gas date';

alter table GSH_SUPP_FEE
   add constraint GSH_SUPP_FEE_PK primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, GAS_DATE, FEE_TYPE);

/*==============================================================*/
/* Table: GSH_SUPP_PHYSICAL_GAS                                 */
/*==============================================================*/
create table GSH_SUPP_PHYSICAL_GAS (
   SETTLEMENT_RUN_ID    NUMERIC(18,0)        not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   TYPE                 VARCHAR(20)          not null,
   BILLING_PERIOD       VARCHAR(20)          null,
   AMOUNT               NUMERIC(18,8)        null,
   GST_AMOUNT           NUMERIC(18,8)        null
);

comment on table GSH_SUPP_PHYSICAL_GAS is
'The Settlements Supporting Data Report is generated when Final and Revision settlement statements are issued; it is also generated daily with details of the latest initial settlement run of the day. The report contains settlements summary information as well as participant specific data from that settlement run for the relevant charges and payments per gas day. It provides a breakdown of all executed trades and delivered quantities records included in this settlement run to assist participant in their settlement statements reconciliation. The Physical Gas subreport provides a breakdown of total physical gas payments and charges included in the settlement run per gas day, note that total payments and total charges are listed separately for each gas day.';

comment on column GSH_SUPP_PHYSICAL_GAS.SETTLEMENT_RUN_ID is
'The unique identifier for the settlement run';

comment on column GSH_SUPP_PHYSICAL_GAS.PARTICIPANT_CODE is
'The participant code';

comment on column GSH_SUPP_PHYSICAL_GAS.GAS_DATE is
'The gas date for the physical payment or charge';

comment on column GSH_SUPP_PHYSICAL_GAS.TYPE is
'Type of the physical gas amount whether it is a charge or a payment';

comment on column GSH_SUPP_PHYSICAL_GAS.BILLING_PERIOD is
'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run';

comment on column GSH_SUPP_PHYSICAL_GAS.AMOUNT is
'Total amount of trading charge or payment for the gas date excluding GST';

comment on column GSH_SUPP_PHYSICAL_GAS.GST_AMOUNT is
'Total amount of GST on the physical gas payment or charge for the gas date';

alter table GSH_SUPP_PHYSICAL_GAS
   add constraint GSH_SUPP_PHYSICAL_GAS_PK primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, GAS_DATE, TYPE);

/*==============================================================*/
/* Table: GSH_SUPP_REALLOCATIONS                                */
/*==============================================================*/
create table GSH_SUPP_REALLOCATIONS (
   SETTLEMENT_RUN_ID    NUMERIC(18,0)        not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   REALLOCATION_ID      VARCHAR(20)          not null,
   BILLING_PERIOD       VARCHAR(20)          null,
   REALLOCATION_SIDE    VARCHAR(20)          null,
   AGREEMENT_TYPE       VARCHAR(80)          null,
   REFERENCE_PRICE      NUMERIC(18,8)        null,
   AMOUNT               NUMERIC(18,8)        null
);

comment on table GSH_SUPP_REALLOCATIONS is
'The Settlements Supporting Data Report is generated when Final and Revision settlement statements are issued; it is also generated daily with details of the latest initial settlement run of the day. The report contains settlements summary information as well as participant specific data from that settlement run for the relevant charges and payments per gas day. It provides a breakdown of all executed trades and delivered quantities records included in this settlement run to assist participant in their settlement statements reconciliation. The Reallocations subreport provides a list of all individual reallocations included in the settlement run per gas day.';

comment on column GSH_SUPP_REALLOCATIONS.SETTLEMENT_RUN_ID is
'The unique identifier for the settlement run';

comment on column GSH_SUPP_REALLOCATIONS.PARTICIPANT_CODE is
'The participant code';

comment on column GSH_SUPP_REALLOCATIONS.GAS_DATE is
'The gas date for the reallocation';

comment on column GSH_SUPP_REALLOCATIONS.REALLOCATION_ID is
'The unique identifier for the reallocation';

comment on column GSH_SUPP_REALLOCATIONS.BILLING_PERIOD is
'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run';

comment on column GSH_SUPP_REALLOCATIONS.REALLOCATION_SIDE is
'This field specifies whether the participant was the credit or the debit party of the reallocation agreement';

comment on column GSH_SUPP_REALLOCATIONS.AGREEMENT_TYPE is
'The type of the reallocation agreement';

comment on column GSH_SUPP_REALLOCATIONS.REFERENCE_PRICE is
'The reference price used for Fixed Quantity Amount agreement type';

comment on column GSH_SUPP_REALLOCATIONS.AMOUNT is
'The daily dollar or quantity amount of the reallocation for the specified gas date';

alter table GSH_SUPP_REALLOCATIONS
   add constraint GSH_SUPP_REALLOCATIONS_PK primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, GAS_DATE, REALLOCATION_ID);

/*==============================================================*/
/* Table: GSH_SUPP_SERVICES                                     */
/*==============================================================*/
create table GSH_SUPP_SERVICES (
   SETTLEMENT_RUN_ID    NUMERIC(18,0)        not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   GAS_DATE             TIMESTAMP(0)         not null,
   TYPE                 VARCHAR(20)          not null,
   BILLING_PERIOD       VARCHAR(20)          null,
   AMOUNT               NUMERIC(18,8)        null,
   GST_AMOUNT           NUMERIC(18,8)        null
);

comment on table GSH_SUPP_SERVICES is
'This report adds details from the settlements run for the new Wallumbilla Compression Services product. It is produced when a settlements run is completed.';

comment on column GSH_SUPP_SERVICES.SETTLEMENT_RUN_ID is
'The unique identifier for the settlement run.';

comment on column GSH_SUPP_SERVICES.PARTICIPANT_CODE is
'The participant code.';

comment on column GSH_SUPP_SERVICES.GAS_DATE is
'The gas date for the delivery variance payment or charge. Disregard the time component as this is not applicable.';

comment on column GSH_SUPP_SERVICES.TYPE is
'Type of the delivery variance amount whether it is a charge or a payment.';

comment on column GSH_SUPP_SERVICES.BILLING_PERIOD is
'The billing period for the settlement run.';

comment on column GSH_SUPP_SERVICES.AMOUNT is
'Total amount of delivery variance charge or payment for the gas date excluding GST.';

comment on column GSH_SUPP_SERVICES.GST_AMOUNT is
'Total amount of GST on the delivery variance payment or charge for the gas date.';

alter table GSH_SUPP_SERVICES
   add constraint GSH_SUPP_SERVICES_PK primary key (GAS_DATE, SETTLEMENT_RUN_ID, PARTICIPANT_CODE, TYPE);

/*==============================================================*/
/* Table: GSH_SUPP_SETTLEMENT_RUN                               */
/*==============================================================*/
create table GSH_SUPP_SETTLEMENT_RUN (
   SETTLEMENT_RUN_ID    NUMERIC(18,0)        not null,
   BILLING_PERIOD       VARCHAR(20)          null,
   SETTLEMENT_RUN_TYPE  VARCHAR(20)          null,
   START_GAS_DATE       TIMESTAMP(0)         null,
   END_GAS_DATE         TIMESTAMP(0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_SUPP_SETTLEMENT_RUN is
'The Settlements Supporting Data Report is generated when Final and Revision settlement statements are issued; it is also generated daily with details of the latest initial settlement run of the day. The report contains settlements summary information as well as participant specific data from that settlement run for the relevant charges and payments per gas day. It provides a breakdown of all executed trades and delivered quantities records included in this settlement run to assist participant in their settlement statements reconciliation. The Settlement Run subreport provides header details of the settlement run.';

comment on column GSH_SUPP_SETTLEMENT_RUN.SETTLEMENT_RUN_ID is
'The unique identifier for the settlement run';

comment on column GSH_SUPP_SETTLEMENT_RUN.BILLING_PERIOD is
'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run';

comment on column GSH_SUPP_SETTLEMENT_RUN.SETTLEMENT_RUN_TYPE is
'The run type for the settlement, for example final, revision, initial';

comment on column GSH_SUPP_SETTLEMENT_RUN.START_GAS_DATE is
'The first gas date of the period for which settlement is run';

comment on column GSH_SUPP_SETTLEMENT_RUN.END_GAS_DATE is
'The last gas date of the period for which settlement is run';

comment on column GSH_SUPP_SETTLEMENT_RUN.LASTCHANGED is
'The date and time the report was generated';

alter table GSH_SUPP_SETTLEMENT_RUN
   add constraint GSH_SUP_DAT_SETTLEMENT_RUN_PK primary key (SETTLEMENT_RUN_ID);

/*==============================================================*/
/* Table: GSH_SUPP_SETTLEMENT_SUMMARY                           */
/*==============================================================*/
create table GSH_SUPP_SETTLEMENT_SUMMARY (
   SETTLEMENT_RUN_ID    NUMERIC(18,0)        not null,
   PARTICIPANT_CODE     VARCHAR(20)          not null,
   REVISION_SETTLEMENT_RUN_ID NUMERIC(18,0)        not null,
   BILLING_PERIOD       VARCHAR(20)          null,
   PARTICIPANT_NAME     VARCHAR(80)          null,
   NET_SETTLEMENT_AMOUNT NUMERIC(18,8)        null,
   NET_SETTLEMENT_AMOUNT_GST NUMERIC(18,8)        null,
   TOTAL_REALLOCATION_DEBIT_AMT NUMERIC(18,8)        null,
   TOTAL_REALLOCATION_CREDIT_AMT NUMERIC(18,8)        null,
   TOTAL_SECURITY_DEPOSITS NUMERIC(18,8)        null,
   SECURITY_DEPOSIT_INTEREST NUMERIC(18,8)        null,
   SETTLEMENT_ADJUSTMENT_AMOUNT NUMERIC(18,8)        null,
   INTEREST_ON_REVISION NUMERIC(18,8)        null,
   TOTAL_PAYABLE_AMOUNT NUMERIC(18,8)        null
);

comment on table GSH_SUPP_SETTLEMENT_SUMMARY is
'The Settlements Supporting Data Report is generated when Final and Revision settlement statements are issued; it is also generated daily with details of the latest initial settlement run of the day. The report contains settlements summary information as well as participant specific data from that settlement run for the relevant charges and payments per gas day. It provides a breakdown of all executed trades and delivered quantities records included in this settlement run to assist participant in their settlement statements reconciliation. The Settlement Summary subreport provides summary details of the settlement run.';

comment on column GSH_SUPP_SETTLEMENT_SUMMARY.SETTLEMENT_RUN_ID is
'The unique identifier for the settlement run';

comment on column GSH_SUPP_SETTLEMENT_SUMMARY.PARTICIPANT_CODE is
'The participant code';

comment on column GSH_SUPP_SETTLEMENT_SUMMARY.REVISION_SETTLEMENT_RUN_ID is
'Identifier of the revision included if the settlement run is of type Final.';

comment on column GSH_SUPP_SETTLEMENT_SUMMARY.BILLING_PERIOD is
'The billing period for the settlement run, note that the billing period will only have value for Final and Revision settlement run types, and it will be null for the initial settlement run';

comment on column GSH_SUPP_SETTLEMENT_SUMMARY.PARTICIPANT_NAME is
'The name of the participant';

comment on column GSH_SUPP_SETTLEMENT_SUMMARY.NET_SETTLEMENT_AMOUNT is
'Sum of all payments and charges exclusive of GST';

comment on column GSH_SUPP_SETTLEMENT_SUMMARY.NET_SETTLEMENT_AMOUNT_GST is
'Sum of all GST amounts on payments and charges';

comment on column GSH_SUPP_SETTLEMENT_SUMMARY.TOTAL_REALLOCATION_DEBIT_AMT is
'Sum of all reallocation debit amounts included in the settlement';

comment on column GSH_SUPP_SETTLEMENT_SUMMARY.TOTAL_REALLOCATION_CREDIT_AMT is
'Sum of all reallocation credit amounts included in the settlement';

comment on column GSH_SUPP_SETTLEMENT_SUMMARY.TOTAL_SECURITY_DEPOSITS is
'The total of all security deposits included in the settlement run. This value will always be a negative value for consistency with invoices (multiplied with -1)';

comment on column GSH_SUPP_SETTLEMENT_SUMMARY.SECURITY_DEPOSIT_INTEREST is
'Total interest amounts calculated on security deposits. This value will always be a negative value for consistency with invoices (multiplied with -1)';

comment on column GSH_SUPP_SETTLEMENT_SUMMARY.SETTLEMENT_ADJUSTMENT_AMOUNT is
'Adjustment amount of the settlement if applicable';

comment on column GSH_SUPP_SETTLEMENT_SUMMARY.INTEREST_ON_REVISION is
'Amount of interest calculated for revision statement if the run type is Revision';

comment on column GSH_SUPP_SETTLEMENT_SUMMARY.TOTAL_PAYABLE_AMOUNT is
'The net payable amount for this settlement run';

alter table GSH_SUPP_SETTLEMENT_SUMMARY
   add constraint GSH_SUPP_SETTLEMENT_SUMMARY_PK primary key (SETTLEMENT_RUN_ID, PARTICIPANT_CODE, REVISION_SETTLEMENT_RUN_ID);

/*==============================================================*/
/* Table: GSH_TRADES                                            */
/*==============================================================*/
create table GSH_TRADES (
   MARKET_ID            VARCHAR(20)          not null,
   TRADE_ID             VARCHAR(20)          not null,
   BUYER_PARTICIPANT_CODE VARCHAR(20)          null,
   BUYER_PARTICIPANT_NAME VARCHAR(80)          null,
   SELLER_PARTICIPANT_CODE VARCHAR(20)          null,
   SELLER_PARTICIPANT_NAME VARCHAR(80)          null,
   FROM_GAS_DATE        TIMESTAMP(0)         null,
   TO_GAS_DATE          TIMESTAMP(0)         null,
   TRADE_DATETIME       TIMESTAMP(0)         null,
   TRADE_TYPE           VARCHAR(20)          null,
   PRODUCT_LOCATION     VARCHAR(80)          null,
   PRODUCT_TYPE         VARCHAR(80)          null,
   TRADE_QUANTITY       NUMERIC(18,8)        null,
   TRADE_PRICE          NUMERIC(18,8)        null,
   DELIVERY_POINT       VARCHAR(40)          null,
   ORDER_ID             VARCHAR(20)          null,
   LAST_UPDATED         TIMESTAMP(0)         null,
   LASTCHANGED          TIMESTAMP(0)         null,
   BUYER_USER_NAME      VARCHAR(100)         null,
   SELLER_USER_NAME     VARCHAR(100)         null,
   RELATIONSHIP_ID      VARCHAR(20)          null,
   RECEIPT_POINT        VARCHAR(40)          null,
   VALID_SERVICE_REFERENCE VARCHAR(3)           null,
   VALID_STTM_CONTRACT_REFERENCE VARCHAR(3)           null,
   VALID_DWGM_REFERENCE VARCHAR(3)           null,
   TRADE_STATUS         VARCHAR(20)          null,
   GAS_FUTURE_ID        VARCHAR(20)          null
);

comment on table GSH_TRADES is
'The purpose of this report is to provide the Participants (buyer and seller) a confirmation whenever a trade is executed through auto matching of bids and offers or by off-Market trades. The report will also provide Participants with a list of all their future trades when triggered daily after the end of the trades for the current day.';

comment on column GSH_TRADES.MARKET_ID is
'Unique Market ID';

comment on column GSH_TRADES.TRADE_ID is
'Unique trade id generated by the Trading Platform at the time a transaction is executed';

comment on column GSH_TRADES.BUYER_PARTICIPANT_CODE is
'The participant code used in the Exchange Trading System (ETS) for the participant on buy side of the trade';

comment on column GSH_TRADES.BUYER_PARTICIPANT_NAME is
'The name for the participant on buy side of the trade';

comment on column GSH_TRADES.SELLER_PARTICIPANT_CODE is
'The participant code used in the Exchange Trading System (ETS) for the participant on the seller side of the trade';

comment on column GSH_TRADES.SELLER_PARTICIPANT_NAME is
'The name for the participant on seller side of the trade';

comment on column GSH_TRADES.FROM_GAS_DATE is
'The start gas day for the order delivery period';

comment on column GSH_TRADES.TO_GAS_DATE is
'The end gas day for the order delivery period';

comment on column GSH_TRADES.TRADE_DATETIME is
'Date and time the trade is executed';

comment on column GSH_TRADES.TRADE_TYPE is
'The type of the trade. Valid values are: AUTO_MATCHED, OFF_MARKET_TRADE';

comment on column GSH_TRADES.PRODUCT_LOCATION is
'Product location for the Gas Supply Hub market';

comment on column GSH_TRADES.PRODUCT_TYPE is
'The product delivery period for the Gas Supply Hub market';

comment on column GSH_TRADES.TRADE_QUANTITY is
'Volume of gas transacted on the Trading Platform in GJ/day';

comment on column GSH_TRADES.TRADE_PRICE is
'Price value of the trade in $/GJ';

comment on column GSH_TRADES.DELIVERY_POINT is
'Delivery Point specified by the seller in an order submission for non-netted commodity products, or the buyer or seller in an order submission for capacity products';

comment on column GSH_TRADES.ORDER_ID is
'The id of the order used in the trade from the buyer or the seller side depending on the participant getting the report. This field will be null if the TRADE_TYPE is off-market or rarely if the network goes down';

comment on column GSH_TRADES.LAST_UPDATED is
'The date and time the trade was updated i.e. saved into database';

comment on column GSH_TRADES.LASTCHANGED is
'The date and time the report was generated';

comment on column GSH_TRADES.BUYER_USER_NAME is
'Name of the buyer''''s account that made submission to the exchange';

comment on column GSH_TRADES.SELLER_USER_NAME is
'Name of the seller''''s account that made submission to the exchange';

comment on column GSH_TRADES.RELATIONSHIP_ID is
'Populated for transactions executed as part of a spread product. Trades triggered by the matching of orders in a spread product will have the same RELATIONSHIP_ID';

comment on column GSH_TRADES.RECEIPT_POINT is
'Receipt Point specified by the buyer or seller in an order submission for capacity products';

comment on column GSH_TRADES.VALID_SERVICE_REFERENCE is
'Flag to indicate whether a valid service reference exists for the traded capacity product. Will be NULL for trades involving commodity products';

comment on column GSH_TRADES.VALID_STTM_CONTRACT_REFERENCE is
'Flag to indicate whether a valid STTM contract reference exists for the traded capacity product. Will be NULL for trades involving commodity products and non-STTM integrated capacity products';

comment on column GSH_TRADES.VALID_DWGM_REFERENCE is
'Flag to indicate whether a valid DWGM accreditation reference exists for the traded capacity product where a DWGM interface point has been specified. Will be NULL for trades involving commodity products and capacity products that do not have a DWGM interface point specified';

comment on column GSH_TRADES.TRADE_STATUS is
'Status of the trade, for example Accepted, Rejected or Cancelled. Capacity trades are rejected if a valid service reference or STTM contract reference does not exist for the product';

comment on column GSH_TRADES.GAS_FUTURE_ID is
'The EFP Trade ID for matching trades';

alter table GSH_TRADES
   add constraint GSH_TRADES_PK primary key (MARKET_ID, TRADE_ID);

/*==============================================================*/
/* Table: GSH_TRADES_STAGING                                    */
/*==============================================================*/
create table GSH_TRADES_STAGING (
   RECORD_ID            VARCHAR(20)          not null,
   MARKET_ID            VARCHAR(20)          null,
   PARTICIPANT_ID       VARCHAR(20)          null,
   GAS_FUTURE_ID        VARCHAR(20)          null,
   PRODUCT_GROUP_CODE   VARCHAR(20)          null,
   PRODUCT_TYPE_CODE    VARCHAR(20)          null,
   DELIVERY_POINT       VARCHAR(40)          null,
   START_DATE           TIMESTAMP(0)         null,
   END_DATE             TIMESTAMP(0)         null,
   TRADE_TYPE           VARCHAR(20)          null,
   VOLUME               NUMERIC(18,8)        null,
   PRICE                NUMERIC(18,8)        null,
   STATUS               VARCHAR(20)          null,
   VALIDATION_MSG       VARCHAR(255)         null,
   SUBMITTED_TIME       TIMESTAMP(0)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_TRADES_STAGING is
'Provides trading participants a confirmation whenever an EFP trade is submitted to AEMO';

comment on column GSH_TRADES_STAGING.RECORD_ID is
'The unique identifier of the record';

comment on column GSH_TRADES_STAGING.MARKET_ID is
'The unique Market ID';

comment on column GSH_TRADES_STAGING.PARTICIPANT_ID is
'The unique identifier of the participant';

comment on column GSH_TRADES_STAGING.GAS_FUTURE_ID is
'The EFP Trade ID for matching trades';

comment on column GSH_TRADES_STAGING.PRODUCT_GROUP_CODE is
'AEMO uses a default value only: GAS-WAL. Data submitted by participants is not stored';

comment on column GSH_TRADES_STAGING.PRODUCT_TYPE_CODE is
'AEMO uses a default value only: NG Months. Data submitted by participants is not stored';

comment on column GSH_TRADES_STAGING.DELIVERY_POINT is
'AEMO uses a default value only: WAL HP Trade Point. Data submitted by participants is not stored';

comment on column GSH_TRADES_STAGING.START_DATE is
'The gas start day for the EFP trade period';

comment on column GSH_TRADES_STAGING.END_DATE is
'The gas end day for the EFP trade period';

comment on column GSH_TRADES_STAGING.TRADE_TYPE is
'Buy or Sell';

comment on column GSH_TRADES_STAGING.VOLUME is
'Trade quantities measured in GJ for each Gas Day in the Delivery Period';

comment on column GSH_TRADES_STAGING.PRICE is
'Trade price';

comment on column GSH_TRADES_STAGING.STATUS is
'Submission Status: PENDING, FAIL, MATCH, OVERRIDE';

comment on column GSH_TRADES_STAGING.VALIDATION_MSG is
'Submission validation message';

comment on column GSH_TRADES_STAGING.SUBMITTED_TIME is
'Data submission time';

comment on column GSH_TRADES_STAGING.LASTCHANGED is
'The date and time the report is generated';

alter table GSH_TRADES_STAGING
   add constraint GSH_TRADES_STAGING_PK primary key (RECORD_ID);

/*==============================================================*/
/* Table: GSH_TRADING_CONTACTS                                  */
/*==============================================================*/
create table GSH_TRADING_CONTACTS (
   CONTACT_ID           VARCHAR(20)          not null,
   PARTICIPANT_CODE     VARCHAR(20)          null,
   PARTICIPANT_NAME     VARCHAR(80)          null,
   ROLE                 VARCHAR(80)          null,
   FIRST_NAME           VARCHAR(80)          null,
   LAST_NAME            VARCHAR(80)          null,
   SALUTATION           VARCHAR(20)          null,
   JOB_TITLE            VARCHAR(80)          null,
   ADDRESS_LINE_1       VARCHAR(80)          null,
   ADDRESS_LINE_2       VARCHAR(80)          null,
   ADDRESS_LINE_3       VARCHAR(80)          null,
   ADDRESS_LINE_4       VARCHAR(80)          null,
   BUSINESS_PHONE_NUMBER VARCHAR(80)          null,
   MOBILE_NUMBER        VARCHAR(80)          null,
   FAX_NUMBER           VARCHAR(80)          null,
   EMAIL_ADDRESS        VARCHAR(255)         null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_TRADING_CONTACTS is
'The purpose of this report is to provide all GSH participants with a list of all Trading and Delivery contacts details for all registered participants in the GSH market. This report will be made available to all GSH participants.';

comment on column GSH_TRADING_CONTACTS.CONTACT_ID is
'The unique identifier for the contact';

comment on column GSH_TRADING_CONTACTS.PARTICIPANT_CODE is
'The participant code';

comment on column GSH_TRADING_CONTACTS.PARTICIPANT_NAME is
'The trading participant name';

comment on column GSH_TRADING_CONTACTS.ROLE is
'Contact type, for this report it will only be two types: Trading Contact, Gas Delivery Contact';

comment on column GSH_TRADING_CONTACTS.FIRST_NAME is
'First name of the contact';

comment on column GSH_TRADING_CONTACTS.LAST_NAME is
'Last name of the contact';

comment on column GSH_TRADING_CONTACTS.SALUTATION is
'Salutation of the contact';

comment on column GSH_TRADING_CONTACTS.JOB_TITLE is
'Job title for the contact';

comment on column GSH_TRADING_CONTACTS.ADDRESS_LINE_1 is
'The address of the contact';

comment on column GSH_TRADING_CONTACTS.ADDRESS_LINE_2 is
'The address of the contact';

comment on column GSH_TRADING_CONTACTS.ADDRESS_LINE_3 is
'The address of the contact';

comment on column GSH_TRADING_CONTACTS.ADDRESS_LINE_4 is
'The address of the contact';

comment on column GSH_TRADING_CONTACTS.BUSINESS_PHONE_NUMBER is
'The business phone number for the contact';

comment on column GSH_TRADING_CONTACTS.MOBILE_NUMBER is
'Mobile number of the contact';

comment on column GSH_TRADING_CONTACTS.FAX_NUMBER is
'Fax number for the contact';

comment on column GSH_TRADING_CONTACTS.EMAIL_ADDRESS is
'Email address of the contact';

comment on column GSH_TRADING_CONTACTS.LASTCHANGED is
'The date and time the report is issued';

alter table GSH_TRADING_CONTACTS
   add constraint GSH_TRADING_CONTACTS_PK primary key (CONTACT_ID);

/*==============================================================*/
/* Table: GSH_TRANSACTION_SUMMARY                               */
/*==============================================================*/
create table GSH_TRANSACTION_SUMMARY (
   CURRENTDATE          TIMESTAMP(0)         not null,
   PRODUCT_LOCATION     VARCHAR(80)          not null,
   PRODUCT_TYPE         VARCHAR(80)          not null,
   FROM_GAS_DATE        TIMESTAMP(0)         not null,
   TO_GAS_DATE          TIMESTAMP(0)         not null,
   HIGH_PRICE           NUMERIC(18,8)        null,
   LOW_PRICE            NUMERIC(18,8)        null,
   OPEN_PRICE           NUMERIC(18,8)        null,
   CLOSE_PRICE          NUMERIC(18,8)        null,
   VOLUME_WEIGHTED_AVERAGE_PRICE NUMERIC(18,8)        null,
   TOTAL_QUANTITY       NUMERIC(18,8)        null,
   TOTAL_NUMBER_OF_TRADES NUMERIC(18,0)        null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_TRANSACTION_SUMMARY is
'The report provides a summary of all trades executed in the current gas day per product. It provides information on the high/low open/close prices, total traded quantities and number of trades.';

comment on column GSH_TRANSACTION_SUMMARY.CURRENTDATE is
'The current gas date';

comment on column GSH_TRANSACTION_SUMMARY.PRODUCT_LOCATION is
'Product location for the Gas Supply Hub market';

comment on column GSH_TRANSACTION_SUMMARY.PRODUCT_TYPE is
'The product delivery period for the Gas Supply Hub market';

comment on column GSH_TRANSACTION_SUMMARY.FROM_GAS_DATE is
'The start gas day for the order delivery period';

comment on column GSH_TRANSACTION_SUMMARY.TO_GAS_DATE is
'The end gas day for the order delivery period';

comment on column GSH_TRANSACTION_SUMMARY.HIGH_PRICE is
'Highest price between all trades executed today per product in $/GJ';

comment on column GSH_TRANSACTION_SUMMARY.LOW_PRICE is
'Lowest price between all trades executed today per product in $/GJ';

comment on column GSH_TRANSACTION_SUMMARY.OPEN_PRICE is
'The first price of all trades executed today per product in $/GJ';

comment on column GSH_TRANSACTION_SUMMARY.CLOSE_PRICE is
'The last price of all trades executed today per product in $/GJ';

comment on column GSH_TRANSACTION_SUMMARY.VOLUME_WEIGHTED_AVERAGE_PRICE is
'The volume weighted average price for all trades executed today per product in $/GJ';

comment on column GSH_TRANSACTION_SUMMARY.TOTAL_QUANTITY is
'Total traded quantities for all trades executed today per product in GJ/day';

comment on column GSH_TRANSACTION_SUMMARY.TOTAL_NUMBER_OF_TRADES is
'Total number of all executed trades today per product';

comment on column GSH_TRANSACTION_SUMMARY.LASTCHANGED is
'The date and time the report was generated';

alter table GSH_TRANSACTION_SUMMARY
   add constraint GSH_TRANSACTION_SUMMARY_PK primary key (CURRENTDATE, PRODUCT_LOCATION, PRODUCT_TYPE, FROM_GAS_DATE, TO_GAS_DATE);

/*==============================================================*/
/* Table: GSH_ZONES                                             */
/*==============================================================*/
create table GSH_ZONES (
   ZONE_ID              NUMERIC(8,0)         not null,
   EFFECTIVEDATE        TIMESTAMP(0)         not null,
   VERSION_DATETIME     TIMESTAMP(0)         not null,
   ZONE_NAME            VARCHAR(100)         null,
   ZONE_TYPE            VARCHAR(20)          null,
   ZONE_DESCRIPTION     VARCHAR(400)         null,
   FOR_AUCTION_PUBLICATION VARCHAR(3)           null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_ZONES is
'Zones currently registered under Part 24 of the Rules for the capacity trading and day ahead auction.';

comment on column GSH_ZONES.ZONE_ID is
'The unique identifier of the zone.';

comment on column GSH_ZONES.EFFECTIVEDATE is
'Effective date of the record';

comment on column GSH_ZONES.VERSION_DATETIME is
'Version timestamp of the record';

comment on column GSH_ZONES.ZONE_NAME is
'The descriptive name of the zone.';

comment on column GSH_ZONES.ZONE_TYPE is
'The type of zone, for example pipeline receipt, pipeline delivery, compression receipt or compression delivery.';

comment on column GSH_ZONES.ZONE_DESCRIPTION is
'The description of the zone.';

comment on column GSH_ZONES.FOR_AUCTION_PUBLICATION is
'Flag to indicate if the product component can be published in the public auction reports, for example yes or no.';

comment on column GSH_ZONES.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_ZONES
   add constraint GSH_ZONES_PK primary key (ZONE_ID, EFFECTIVEDATE, VERSION_DATETIME);

/*==============================================================*/
/* Table: GSH_ZONE_CURTAILMENT                                  */
/*==============================================================*/
create table GSH_ZONE_CURTAILMENT (
   FACILITY_ID          NUMERIC(8,0)         not null,
   FACILITY_NAME        VARCHAR(100)         null,
   SERVICE_POINT_ID     NUMERIC(8,0)         not null,
   SERVICE_POINT_NAME   VARCHAR(100)         null,
   SERVICE_POINT_TYPE   VARCHAR(100)         null,
   SERVICE_POINT_DESCRIPTION VARCHAR(255)         null,
   GAS_DATE             TIMESTAMP(0)         not null,
   CURTAILED_QUANTITY   NUMERIC(10,0)        null,
   LASTCHANGED          TIMESTAMP(0)         null
);

comment on table GSH_ZONE_CURTAILMENT is
'Contains details of curtailment of nominations against secondary firm rights (CTP trades) for service points (receipt and delivery points).';

comment on column GSH_ZONE_CURTAILMENT.FACILITY_ID is
'The unique identifier of the parent facility.';

comment on column GSH_ZONE_CURTAILMENT.FACILITY_NAME is
'The descriptive name of the parent facility.';

comment on column GSH_ZONE_CURTAILMENT.SERVICE_POINT_ID is
'The unique identifier of the service point.';

comment on column GSH_ZONE_CURTAILMENT.SERVICE_POINT_NAME is
'The descriptive name of the service point.';

comment on column GSH_ZONE_CURTAILMENT.SERVICE_POINT_TYPE is
'The type of service point, for example forward haul receipt point, forward haul delivery point, backhaul receipt point, backhaul delivery point, compression receipt point or compression delivery point.';

comment on column GSH_ZONE_CURTAILMENT.SERVICE_POINT_DESCRIPTION is
'The description of the service point.';

comment on column GSH_ZONE_CURTAILMENT.GAS_DATE is
'The effective gas date of the rejected nomination quantity.';

comment on column GSH_ZONE_CURTAILMENT.CURTAILED_QUANTITY is
'The curtailed quantity of nominations by the facility operator for transportation capacity that has been transferred pursuant to an operational transfer (GJ)';

comment on column GSH_ZONE_CURTAILMENT.LASTCHANGED is
'The date and time the report was generated.';

alter table GSH_ZONE_CURTAILMENT
   add constraint GSH_ZONE_CURTAILMENT_PK primary key (FACILITY_ID, SERVICE_POINT_ID, GAS_DATE);

